# LocalStack Resource Provider Scaffolding v2
from __future__ import annotations

from pathlib import Path
from typing import TypedDict

import localstack.services.cloudformation.provider_utils as util
from localstack.services.cloudformation.resource_provider import (
    OperationStatus,
    ProgressEvent,
    ResourceProvider,
    ResourceRequest,
)


class EC2TransitGatewayProperties(TypedDict):
    AmazonSideAsn: int | None
    AssociationDefaultRouteTableId: str | None
    AutoAcceptSharedAttachments: str | None
    DefaultRouteTableAssociation: str | None
    DefaultRouteTablePropagation: str | None
    Description: str | None
    DnsSupport: str | None
    Id: str | None
    MulticastSupport: str | None
    PropagationDefaultRouteTableId: str | None
    Tags: list[Tag] | None
    TransitGatewayCidrBlocks: list[str] | None
    VpnEcmpSupport: str | None


class Tag(TypedDict):
    Key: str | None
    Value: str | None


REPEATED_INVOCATION = "repeated_invocation"


class EC2TransitGatewayProvider(ResourceProvider[EC2TransitGatewayProperties]):
    TYPE = "AWS::EC2::TransitGateway"  # Autogenerated. Don't change
    SCHEMA = util.get_schema_path(Path(__file__))  # Autogenerated. Don't change

    def create(
        self,
        request: ResourceRequest[EC2TransitGatewayProperties],
    ) -> ProgressEvent[EC2TransitGatewayProperties]:
        """
        Create a new resource.

        Primary identifier fields:
          - /properties/Id



        Create-only properties:
          - /properties/AmazonSideAsn
          - /properties/MulticastSupport

        Read-only properties:
          - /properties/Id

        IAM permissions required:
          - ec2:CreateTransitGateway
          - ec2:CreateTags

        """
        model = request.desired_state
        create_params = {
            "Options": util.select_attributes(
                model,
                [
                    "AmazonSideAsn",
                    "AssociationDefaultRouteTableId",
                    "AutoAcceptSharedAttachments",
                    "DefaultRouteTableAssociation",
                    "DefaultRouteTablePropagation",
                    "DnsSupport",
                    "MulticastSupport",
                    "PropagationDefaultRouteTableId",
                    "TransitGatewayCidrBlocks",
                    "VpnEcmpSupport",
                ],
            )
        }

        if model.get("Description"):
            create_params["Description"] = model["Description"]

        if model.get("Tags", []):
            create_params["TagSpecifications"] = [
                {"ResourceType": "transit-gateway", "Tags": model["Tags"]}
            ]

        response = request.aws_client_factory.ec2.create_transit_gateway(**create_params)
        model["Id"] = response["TransitGateway"]["TransitGatewayId"]

        return ProgressEvent(status=OperationStatus.SUCCESS, resource_model=model)

    def read(
        self,
        request: ResourceRequest[EC2TransitGatewayProperties],
    ) -> ProgressEvent[EC2TransitGatewayProperties]:
        """
        Fetch resource information

        IAM permissions required:
          - ec2:DescribeTransitGateways
          - ec2:DescribeTags
        """
        raise NotImplementedError

    def delete(
        self,
        request: ResourceRequest[EC2TransitGatewayProperties],
    ) -> ProgressEvent[EC2TransitGatewayProperties]:
        """
        Delete a resource

        IAM permissions required:
          - ec2:DeleteTransitGateway
          - ec2:DeleteTags
        """
        model = request.desired_state
        request.aws_client_factory.ec2.delete_transit_gateway(TransitGatewayId=model["Id"])
        return ProgressEvent(
            status=OperationStatus.SUCCESS,
            resource_model={},
        )

    def update(
        self,
        request: ResourceRequest[EC2TransitGatewayProperties],
    ) -> ProgressEvent[EC2TransitGatewayProperties]:
        """
        Update a resource

        IAM permissions required:
          - ec2:ModifyTransitGateway
          - ec2:DeleteTags
          - ec2:CreateTags
          - ec2:ModifyTransitGatewayOptions
        """
        raise NotImplementedError
