# LocalStack Resource Provider Scaffolding v2
from __future__ import annotations

from pathlib import Path
from typing import TypedDict

import localstack.services.cloudformation.provider_utils as util
from localstack.services.cloudformation.resource_provider import (
    OperationStatus,
    ProgressEvent,
    ResourceProvider,
    ResourceRequest,
)


class EventsConnectionProperties(TypedDict):
    AuthParameters: AuthParameters | None
    AuthorizationType: str | None
    Arn: str | None
    Description: str | None
    Name: str | None
    SecretArn: str | None


class ApiKeyAuthParameters(TypedDict):
    ApiKeyName: str | None
    ApiKeyValue: str | None


class BasicAuthParameters(TypedDict):
    Password: str | None
    Username: str | None


class ClientParameters(TypedDict):
    ClientID: str | None
    ClientSecret: str | None


class Parameter(TypedDict):
    Key: str | None
    Value: str | None
    IsValueSecret: bool | None


class ConnectionHttpParameters(TypedDict):
    BodyParameters: list[Parameter] | None
    HeaderParameters: list[Parameter] | None
    QueryStringParameters: list[Parameter] | None


class OAuthParameters(TypedDict):
    AuthorizationEndpoint: str | None
    ClientParameters: ClientParameters | None
    HttpMethod: str | None
    OAuthHttpParameters: ConnectionHttpParameters | None


class AuthParameters(TypedDict):
    ApiKeyAuthParameters: ApiKeyAuthParameters | None
    BasicAuthParameters: BasicAuthParameters | None
    InvocationHttpParameters: ConnectionHttpParameters | None
    OAuthParameters: OAuthParameters | None


REPEATED_INVOCATION = "repeated_invocation"


class EventsConnectionProvider(ResourceProvider[EventsConnectionProperties]):
    TYPE = "AWS::Events::Connection"  # Autogenerated. Don't change
    SCHEMA = util.get_schema_path(Path(__file__))  # Autogenerated. Don't change

    def create(
        self,
        request: ResourceRequest[EventsConnectionProperties],
    ) -> ProgressEvent[EventsConnectionProperties]:
        """
        Create a new resource.

        Primary identifier fields:
          - /properties/Name

        Required properties:
          - AuthorizationType
          - AuthParameters

        Create-only properties:
          - /properties/Name

        Read-only properties:
          - /properties/Arn
          - /properties/SecretArn

        IAM permissions required:
          - events:CreateConnection
          - secretsmanager:CreateSecret
          - secretsmanager:GetSecretValue
          - secretsmanager:PutSecretValue
          - iam:CreateServiceLinkedRole

        """
        model = request.desired_state
        events = request.aws_client_factory.events

        response = events.create_connection(**model)
        model["Arn"] = response["ConnectionArn"]

        return ProgressEvent(
            status=OperationStatus.SUCCESS,
            resource_model=model,
            custom_context=request.custom_context,
        )

    def read(
        self,
        request: ResourceRequest[EventsConnectionProperties],
    ) -> ProgressEvent[EventsConnectionProperties]:
        """
        Fetch resource information

        IAM permissions required:
          - events:DescribeConnection
        """
        raise NotImplementedError

    def delete(
        self,
        request: ResourceRequest[EventsConnectionProperties],
    ) -> ProgressEvent[EventsConnectionProperties]:
        """
        Delete a resource

        IAM permissions required:
          - events:DeleteConnection
        """
        model = request.desired_state
        events = request.aws_client_factory.events

        events.delete_connection(Name=model["Name"])

        return ProgressEvent(
            status=OperationStatus.SUCCESS,
            resource_model=model,
            custom_context=request.custom_context,
        )

    def update(
        self,
        request: ResourceRequest[EventsConnectionProperties],
    ) -> ProgressEvent[EventsConnectionProperties]:
        """
        Update a resource

        IAM permissions required:
          - events:UpdateConnection
          - events:DescribeConnection
          - secretsmanager:CreateSecret
          - secretsmanager:UpdateSecret
          - secretsmanager:GetSecretValue
          - secretsmanager:PutSecretValue
        """
        raise NotImplementedError
