# LocalStack Resource Provider Scaffolding v2
from __future__ import annotations

from pathlib import Path
from typing import TypedDict

import localstack.services.cloudformation.provider_utils as util
from localstack.services.cloudformation.resource_provider import (
    OperationStatus,
    ProgressEvent,
    ResourceProvider,
    ResourceRequest,
)


class IAMServerCertificateProperties(TypedDict):
    Arn: str | None
    CertificateBody: str | None
    CertificateChain: str | None
    Path: str | None
    PrivateKey: str | None
    ServerCertificateName: str | None
    Tags: list[Tag] | None


class Tag(TypedDict):
    Key: str | None
    Value: str | None


REPEATED_INVOCATION = "repeated_invocation"


class IAMServerCertificateProvider(ResourceProvider[IAMServerCertificateProperties]):
    TYPE = "AWS::IAM::ServerCertificate"  # Autogenerated. Don't change
    SCHEMA = util.get_schema_path(Path(__file__))  # Autogenerated. Don't change

    def create(
        self,
        request: ResourceRequest[IAMServerCertificateProperties],
    ) -> ProgressEvent[IAMServerCertificateProperties]:
        """
        Create a new resource.

        Primary identifier fields:
          - /properties/ServerCertificateName



        Create-only properties:
          - /properties/ServerCertificateName
          - /properties/PrivateKey
          - /properties/CertificateBody
          - /properties/CertificateChain

        Read-only properties:
          - /properties/Arn

        IAM permissions required:
          - iam:UploadServerCertificate
          - iam:GetServerCertificate

        """
        model = request.desired_state
        if not model.get("ServerCertificateName"):
            model["ServerCertificateName"] = util.generate_default_name_without_stack(
                request.logical_resource_id
            )

        create_params = util.select_attributes(
            model,
            [
                "ServerCertificateName",
                "PrivateKey",
                "CertificateBody",
                "CertificateChain",
                "Path",
                "Tags",
            ],
        )

        # Create the resource
        certificate = request.aws_client_factory.iam.upload_server_certificate(**create_params)
        model["Arn"] = certificate["ServerCertificateMetadata"]["Arn"]

        return ProgressEvent(
            status=OperationStatus.SUCCESS,
            resource_model=model,
            custom_context=request.custom_context,
        )

    def read(
        self,
        request: ResourceRequest[IAMServerCertificateProperties],
    ) -> ProgressEvent[IAMServerCertificateProperties]:
        """
        Fetch resource information

        IAM permissions required:
          - iam:GetServerCertificate
        """
        raise NotImplementedError

    def delete(
        self,
        request: ResourceRequest[IAMServerCertificateProperties],
    ) -> ProgressEvent[IAMServerCertificateProperties]:
        """
        Delete a resource

        IAM permissions required:
          - iam:DeleteServerCertificate
        """
        model = request.desired_state
        request.aws_client_factory.iam.delete_server_certificate(
            ServerCertificateName=model["ServerCertificateName"]
        )
        return ProgressEvent(
            status=OperationStatus.SUCCESS,
        )

    def update(
        self,
        request: ResourceRequest[IAMServerCertificateProperties],
    ) -> ProgressEvent[IAMServerCertificateProperties]:
        """
        Update a resource

        IAM permissions required:
          - iam:TagServerCertificate
          - iam:UntagServerCertificate
        """
        raise NotImplementedError
