# LocalStack Resource Provider Scaffolding v2
from __future__ import annotations

from pathlib import Path
from typing import TypedDict

import localstack.services.cloudformation.provider_utils as util
from localstack.services.cloudformation.resource_provider import (
    OperationStatus,
    ProgressEvent,
    ResourceProvider,
    ResourceRequest,
)


class KMSAliasProperties(TypedDict):
    AliasName: str | None
    TargetKeyId: str | None


REPEATED_INVOCATION = "repeated_invocation"


class KMSAliasProvider(ResourceProvider[KMSAliasProperties]):
    TYPE = "AWS::KMS::Alias"  # Autogenerated. Don't change
    SCHEMA = util.get_schema_path(Path(__file__))  # Autogenerated. Don't change

    def create(
        self,
        request: ResourceRequest[KMSAliasProperties],
    ) -> ProgressEvent[KMSAliasProperties]:
        """
        Create a new resource.

        Primary identifier fields:
          - /properties/AliasName

        Required properties:
          - AliasName
          - TargetKeyId

        Create-only properties:
          - /properties/AliasName



        IAM permissions required:
          - kms:CreateAlias

        """
        model = request.desired_state
        kms = request.aws_client_factory.kms

        kms.create_alias(AliasName=model["AliasName"], TargetKeyId=model["TargetKeyId"])

        return ProgressEvent(
            status=OperationStatus.SUCCESS,
            resource_model=model,
            custom_context=request.custom_context,
        )

    def read(
        self,
        request: ResourceRequest[KMSAliasProperties],
    ) -> ProgressEvent[KMSAliasProperties]:
        """
        Fetch resource information

        IAM permissions required:
          - kms:ListAliases
        """
        raise NotImplementedError

    def delete(
        self,
        request: ResourceRequest[KMSAliasProperties],
    ) -> ProgressEvent[KMSAliasProperties]:
        """
        Delete a resource

        IAM permissions required:
          - kms:DeleteAlias
        """
        model = request.desired_state
        kms = request.aws_client_factory.kms

        kms.delete_alias(AliasName=model["AliasName"])

        return ProgressEvent(
            status=OperationStatus.SUCCESS,
            resource_model=model,
            custom_context=request.custom_context,
        )

    def update(
        self,
        request: ResourceRequest[KMSAliasProperties],
    ) -> ProgressEvent[KMSAliasProperties]:
        """
        Update a resource

        IAM permissions required:
          - kms:UpdateAlias
        """
        raise NotImplementedError
