# LocalStack Resource Provider Scaffolding v2
from __future__ import annotations

import logging
from pathlib import Path
from typing import TypedDict

import localstack.services.cloudformation.provider_utils as util
from localstack.services.cloudformation.resource_provider import (
    OperationStatus,
    ProgressEvent,
    Properties,
    ResourceProvider,
    ResourceRequest,
)
from localstack.services.lambda_.api_utils import parse_layer_arn
from localstack.utils.strings import short_uid

LOG = logging.getLogger(__name__)


class LambdaLayerVersionProperties(TypedDict):
    Content: Content | None
    CompatibleArchitectures: list[str] | None
    CompatibleRuntimes: list[str] | None
    Description: str | None
    LayerName: str | None
    LayerVersionArn: str | None
    LicenseInfo: str | None


class Content(TypedDict):
    S3Bucket: str | None
    S3Key: str | None
    S3ObjectVersion: str | None


REPEATED_INVOCATION = "repeated_invocation"


class LambdaLayerVersionProvider(ResourceProvider[LambdaLayerVersionProperties]):
    TYPE = "AWS::Lambda::LayerVersion"  # Autogenerated. Don't change
    SCHEMA = util.get_schema_path(Path(__file__))  # Autogenerated. Don't change

    def create(
        self,
        request: ResourceRequest[LambdaLayerVersionProperties],
    ) -> ProgressEvent[LambdaLayerVersionProperties]:
        """
        Create a new resource.

        Primary identifier fields:
          - /properties/LayerVersionArn

        Required properties:
          - Content

        Create-only properties:
          - /properties/CompatibleRuntimes
          - /properties/LicenseInfo
          - /properties/CompatibleArchitectures
          - /properties/LayerName
          - /properties/Description
          - /properties/Content

        Read-only properties:
          - /properties/LayerVersionArn

        IAM permissions required:
          - lambda:PublishLayerVersion
          - s3:GetObject
          - s3:GetObjectVersion

        """
        model = request.desired_state
        lambda_client = request.aws_client_factory.lambda_
        if not model.get("LayerName"):
            model["LayerName"] = f"layer-{short_uid()}"
        response = lambda_client.publish_layer_version(**model)
        model["LayerVersionArn"] = response["LayerVersionArn"]

        return ProgressEvent(
            status=OperationStatus.SUCCESS,
            resource_model=model,
            custom_context=request.custom_context,
        )

    def read(
        self,
        request: ResourceRequest[LambdaLayerVersionProperties],
    ) -> ProgressEvent[LambdaLayerVersionProperties]:
        """
        Fetch resource information

        IAM permissions required:
          - lambda:GetLayerVersion
        """
        lambda_client = request.aws_client_factory.lambda_
        layer_version_arn = request.desired_state.get("LayerVersionArn")

        try:
            _, _, layer_name, version = parse_layer_arn(layer_version_arn)
        except AttributeError as e:
            LOG.info(
                "Invalid Arn: '%s', %s",
                layer_version_arn,
                e,
                exc_info=LOG.isEnabledFor(logging.DEBUG),
            )
            return ProgressEvent(
                status=OperationStatus.FAILED,
                message="Caught unexpected syntax violation. Consider using ARN.fromString().",
                error_code="InternalFailure",
            )

        if not version:
            return ProgressEvent(
                status=OperationStatus.FAILED,
                message="Invalid request provided: Layer Version ARN contains invalid layer name or version",
                error_code="InvalidRequest",
            )

        try:
            response = lambda_client.get_layer_version_by_arn(Arn=layer_version_arn)
        except lambda_client.exceptions.ResourceNotFoundException as e:
            return ProgressEvent(
                status=OperationStatus.FAILED,
                message="The resource you requested does not exist. "
                f"(Service: Lambda, Status Code: 404, Request ID: {e.response['ResponseMetadata']['RequestId']})",
                error_code="NotFound",
            )
        layer = util.select_attributes(
            response,
            [
                "CompatibleRuntimes",
                "Description",
                "LayerVersionArn",
                "CompatibleArchitectures",
            ],
        )
        layer.setdefault("CompatibleRuntimes", [])
        layer.setdefault("CompatibleArchitectures", [])
        layer.setdefault("LayerName", layer_name)

        return ProgressEvent(
            status=OperationStatus.SUCCESS,
            resource_model=layer,
            custom_context=request.custom_context,
        )

    def delete(
        self,
        request: ResourceRequest[LambdaLayerVersionProperties],
    ) -> ProgressEvent[LambdaLayerVersionProperties]:
        """
        Delete a resource

        IAM permissions required:
          - lambda:GetLayerVersion
          - lambda:DeleteLayerVersion
        """
        model = request.desired_state
        lambda_client = request.aws_client_factory.lambda_
        version = int(model["LayerVersionArn"].split(":")[-1])

        lambda_client.delete_layer_version(LayerName=model["LayerName"], VersionNumber=version)
        return ProgressEvent(
            status=OperationStatus.SUCCESS,
            resource_model=model,
            custom_context=request.custom_context,
        )

    def update(
        self,
        request: ResourceRequest[LambdaLayerVersionProperties],
    ) -> ProgressEvent[LambdaLayerVersionProperties]:
        """
        Update a resource


        """
        raise NotImplementedError

    def list(self, request: ResourceRequest[Properties]) -> ProgressEvent[Properties]:
        """
        List resources

        IAM permissions required:
          - lambda:ListLayerVersions
        """

        lambda_client = request.aws_client_factory.lambda_

        lambda_layer = request.desired_state.get("LayerName")
        if not lambda_layer:
            return ProgressEvent(
                status=OperationStatus.FAILED,
                message="Layer Name cannot be empty",
                error_code="InvalidRequest",
            )

        layer_versions = lambda_client.list_layer_versions(LayerName=lambda_layer)

        return ProgressEvent(
            status=OperationStatus.SUCCESS,
            resource_models=[
                LambdaLayerVersionProperties(LayerVersionArn=layer_version["LayerVersionArn"])
                for layer_version in layer_versions["LayerVersions"]
            ],
            custom_context=request.custom_context,
        )
