# LocalStack Resource Provider Scaffolding v2
from __future__ import annotations

import copy
from pathlib import Path
from typing import TypedDict

import localstack.services.cloudformation.provider_utils as util
from localstack.aws.api.es import CreateElasticsearchDomainRequest
from localstack.services.cloudformation.resource_provider import (
    OperationStatus,
    ProgressEvent,
    ResourceProvider,
    ResourceRequest,
)
from localstack.utils.collections import convert_to_typed_dict


class ElasticsearchDomainProperties(TypedDict):
    AccessPolicies: dict | None
    AdvancedOptions: dict | None
    AdvancedSecurityOptions: AdvancedSecurityOptionsInput | None
    Arn: str | None
    CognitoOptions: CognitoOptions | None
    DomainArn: str | None
    DomainEndpoint: str | None
    DomainEndpointOptions: DomainEndpointOptions | None
    DomainName: str | None
    EBSOptions: EBSOptions | None
    ElasticsearchClusterConfig: ElasticsearchClusterConfig | None
    ElasticsearchVersion: str | None
    EncryptionAtRestOptions: EncryptionAtRestOptions | None
    Id: str | None
    LogPublishingOptions: dict | None
    NodeToNodeEncryptionOptions: NodeToNodeEncryptionOptions | None
    SnapshotOptions: SnapshotOptions | None
    Tags: list[Tag] | None
    VPCOptions: VPCOptions | None


class ZoneAwarenessConfig(TypedDict):
    AvailabilityZoneCount: int | None


class ColdStorageOptions(TypedDict):
    Enabled: bool | None


class ElasticsearchClusterConfig(TypedDict):
    ColdStorageOptions: ColdStorageOptions | None
    DedicatedMasterCount: int | None
    DedicatedMasterEnabled: bool | None
    DedicatedMasterType: str | None
    InstanceCount: int | None
    InstanceType: str | None
    WarmCount: int | None
    WarmEnabled: bool | None
    WarmType: str | None
    ZoneAwarenessConfig: ZoneAwarenessConfig | None
    ZoneAwarenessEnabled: bool | None


class SnapshotOptions(TypedDict):
    AutomatedSnapshotStartHour: int | None


class VPCOptions(TypedDict):
    SecurityGroupIds: list[str] | None
    SubnetIds: list[str] | None


class NodeToNodeEncryptionOptions(TypedDict):
    Enabled: bool | None


class DomainEndpointOptions(TypedDict):
    CustomEndpoint: str | None
    CustomEndpointCertificateArn: str | None
    CustomEndpointEnabled: bool | None
    EnforceHTTPS: bool | None
    TLSSecurityPolicy: str | None


class CognitoOptions(TypedDict):
    Enabled: bool | None
    IdentityPoolId: str | None
    RoleArn: str | None
    UserPoolId: str | None


class MasterUserOptions(TypedDict):
    MasterUserARN: str | None
    MasterUserName: str | None
    MasterUserPassword: str | None


class AdvancedSecurityOptionsInput(TypedDict):
    AnonymousAuthEnabled: bool | None
    Enabled: bool | None
    InternalUserDatabaseEnabled: bool | None
    MasterUserOptions: MasterUserOptions | None


class EBSOptions(TypedDict):
    EBSEnabled: bool | None
    Iops: int | None
    VolumeSize: int | None
    VolumeType: str | None


class EncryptionAtRestOptions(TypedDict):
    Enabled: bool | None
    KmsKeyId: str | None


class Tag(TypedDict):
    Key: str | None
    Value: str | None


REPEATED_INVOCATION = "repeated_invocation"


class ElasticsearchDomainProvider(ResourceProvider[ElasticsearchDomainProperties]):
    TYPE = "AWS::Elasticsearch::Domain"  # Autogenerated. Don't change
    SCHEMA = util.get_schema_path(Path(__file__))  # Autogenerated. Don't change

    def create(
        self,
        request: ResourceRequest[ElasticsearchDomainProperties],
    ) -> ProgressEvent[ElasticsearchDomainProperties]:
        """
        Create a new resource.

        Primary identifier fields:
          - /properties/Id



        Create-only properties:
          - /properties/DomainName

        Read-only properties:
          - /properties/Id
          - /properties/DomainArn
          - /properties/DomainEndpoint
          - /properties/Arn



        """
        model = request.desired_state

        client = request.aws_client_factory.es
        if not request.custom_context.get(REPEATED_INVOCATION):
            # this is the first time this callback is invoked
            request.custom_context[REPEATED_INVOCATION] = True

            # defaults
            domain_name = model.get("DomainName")
            if not domain_name:
                model["DomainName"] = util.generate_default_name(
                    request.stack_name, request.logical_resource_id
                )

            params = copy.deepcopy(model)
            params = convert_to_typed_dict(CreateElasticsearchDomainRequest, params)
            params = util.remove_none_values(params)
            cluster_config = params.get("ElasticsearchClusterConfig")
            if isinstance(cluster_config, dict):
                # set defaults required for boto3 calls
                cluster_config.setdefault("DedicatedMasterType", "m3.medium.elasticsearch")
                cluster_config.setdefault("WarmType", "ultrawarm1.medium.elasticsearch")

            client.create_elasticsearch_domain(**params)

        domain = client.describe_elasticsearch_domain(DomainName=model["DomainName"])
        if domain["DomainStatus"]["Created"]:
            # set data
            model["Arn"] = domain["DomainStatus"]["ARN"]
            model["Id"] = model["DomainName"]
            model["DomainArn"] = domain["DomainStatus"]["ARN"]
            model["DomainEndpoint"] = domain["DomainStatus"].get("Endpoint")

            if tags := model.get("Tags", []):
                client.add_tags(ARN=model["Arn"], TagList=tags)

            return ProgressEvent(status=OperationStatus.SUCCESS, resource_model=model)
        else:
            return ProgressEvent(status=OperationStatus.IN_PROGRESS, resource_model=model)

    def read(
        self,
        request: ResourceRequest[ElasticsearchDomainProperties],
    ) -> ProgressEvent[ElasticsearchDomainProperties]:
        """
        Fetch resource information


        """
        raise NotImplementedError

    def delete(
        self,
        request: ResourceRequest[ElasticsearchDomainProperties],
    ) -> ProgressEvent[ElasticsearchDomainProperties]:
        """
        Delete a resource


        """
        client = request.aws_client_factory.es
        # TODO the delete is currently synchronous;
        #   if this changes, we should also reflect the OperationStatus here
        client.delete_elasticsearch_domain(DomainName=request.previous_state["DomainName"])
        return ProgressEvent(status=OperationStatus.SUCCESS, resource_model={})

    def update(
        self,
        request: ResourceRequest[ElasticsearchDomainProperties],
    ) -> ProgressEvent[ElasticsearchDomainProperties]:
        """
        Update a resource


        """
        raise NotImplementedError
