# LocalStack Resource Provider Scaffolding v2
from __future__ import annotations

from pathlib import Path
from typing import TypedDict

import localstack.services.cloudformation.provider_utils as util
from localstack.services.cloudformation.resource_provider import (
    OperationStatus,
    ProgressEvent,
    ResourceProvider,
    ResourceRequest,
)


class SchedulerScheduleGroupProperties(TypedDict):
    Arn: str | None
    CreationDate: str | None
    LastModificationDate: str | None
    Name: str | None
    State: str | None
    Tags: list[Tag] | None


class Tag(TypedDict):
    Key: str | None
    Value: str | None


REPEATED_INVOCATION = "repeated_invocation"


class SchedulerScheduleGroupProvider(ResourceProvider[SchedulerScheduleGroupProperties]):
    TYPE = "AWS::Scheduler::ScheduleGroup"  # Autogenerated. Don't change
    SCHEMA = util.get_schema_path(Path(__file__))  # Autogenerated. Don't change

    def create(
        self,
        request: ResourceRequest[SchedulerScheduleGroupProperties],
    ) -> ProgressEvent[SchedulerScheduleGroupProperties]:
        """
        Create a new resource.

        Primary identifier fields:
          - /properties/Name



        Create-only properties:
          - /properties/Name

        Read-only properties:
          - /properties/Arn
          - /properties/CreationDate
          - /properties/LastModificationDate
          - /properties/State

        IAM permissions required:
          - scheduler:CreateScheduleGroup
          - scheduler:GetScheduleGroup
          - scheduler:ListTagsForResource

        """
        model = request.desired_state

        if not model.get("Name"):
            model["Name"] = util.generate_default_name(
                request.stack_name, request.logical_resource_id
            )

        create_params = util.select_attributes(model, ("Name", "Tags"))

        result = request.aws_client_factory.scheduler.create_schedule_group(**create_params)
        model["Arn"] = result["ScheduleGroupArn"]

        return ProgressEvent(
            status=OperationStatus.SUCCESS,
            resource_model=model,
        )

    def read(
        self,
        request: ResourceRequest[SchedulerScheduleGroupProperties],
    ) -> ProgressEvent[SchedulerScheduleGroupProperties]:
        """
        Fetch resource information

        IAM permissions required:
          - scheduler:GetScheduleGroup
          - scheduler:ListTagsForResource
        """
        raise NotImplementedError

    def delete(
        self,
        request: ResourceRequest[SchedulerScheduleGroupProperties],
    ) -> ProgressEvent[SchedulerScheduleGroupProperties]:
        """
        Delete a resource

        IAM permissions required:
          - scheduler:DeleteScheduleGroup
          - scheduler:GetScheduleGroup
          - scheduler:DeleteSchedule
        """
        model = request.desired_state
        request.aws_client_factory.scheduler.delete_schedule_group(Name=model["Name"])
        return ProgressEvent(status=OperationStatus.SUCCESS, resource_model={})

    def update(
        self,
        request: ResourceRequest[SchedulerScheduleGroupProperties],
    ) -> ProgressEvent[SchedulerScheduleGroupProperties]:
        """
        Update a resource

        IAM permissions required:
          - scheduler:TagResource
          - scheduler:UntagResource
          - scheduler:ListTagsForResource
          - scheduler:GetScheduleGroup
        """
        raise NotImplementedError
