# LocalStack Resource Provider Scaffolding v2
from __future__ import annotations

import json
from pathlib import Path
from typing import TypedDict

import localstack.services.cloudformation.provider_utils as util
from localstack.services.cloudformation.resource_provider import (
    OperationStatus,
    ProgressEvent,
    ResourceProvider,
    ResourceRequest,
)


class SecretsManagerResourcePolicyProperties(TypedDict):
    ResourcePolicy: dict | None
    SecretId: str | None
    BlockPublicPolicy: bool | None
    Id: str | None


REPEATED_INVOCATION = "repeated_invocation"


class SecretsManagerResourcePolicyProvider(
    ResourceProvider[SecretsManagerResourcePolicyProperties]
):
    TYPE = "AWS::SecretsManager::ResourcePolicy"  # Autogenerated. Don't change
    SCHEMA = util.get_schema_path(Path(__file__))  # Autogenerated. Don't change

    def create(
        self,
        request: ResourceRequest[SecretsManagerResourcePolicyProperties],
    ) -> ProgressEvent[SecretsManagerResourcePolicyProperties]:
        """
        Create a new resource.

        Primary identifier fields:
          - /properties/Id

        Required properties:
          - SecretId
          - ResourcePolicy

        Create-only properties:
          - /properties/SecretId

        Read-only properties:
          - /properties/Id

        """
        model = request.desired_state
        secret_manager = request.aws_client_factory.secretsmanager

        params = {
            "SecretId": model["SecretId"],
            "ResourcePolicy": json.dumps(model["ResourcePolicy"]),
            "BlockPublicPolicy": model.get("BlockPublicPolicy") or True,
        }
        response = secret_manager.put_resource_policy(**params)

        model["Id"] = response["ARN"]
        return ProgressEvent(
            status=OperationStatus.SUCCESS,
            resource_model=model,
            custom_context=request.custom_context,
        )

    def read(
        self,
        request: ResourceRequest[SecretsManagerResourcePolicyProperties],
    ) -> ProgressEvent[SecretsManagerResourcePolicyProperties]:
        """
        Fetch resource information


        """
        raise NotImplementedError

    def delete(
        self,
        request: ResourceRequest[SecretsManagerResourcePolicyProperties],
    ) -> ProgressEvent[SecretsManagerResourcePolicyProperties]:
        """
        Delete a resource

        """
        model = request.desired_state
        secret_manager = request.aws_client_factory.secretsmanager

        response = secret_manager.delete_resource_policy(SecretId=model["SecretId"])

        model["Id"] = response["ARN"]

        return ProgressEvent(
            status=OperationStatus.SUCCESS,
            resource_model=model,
            custom_context=request.custom_context,
        )

    def update(
        self,
        request: ResourceRequest[SecretsManagerResourcePolicyProperties],
    ) -> ProgressEvent[SecretsManagerResourcePolicyProperties]:
        """
        Update a resource


        """
        raise NotImplementedError
