# LocalStack Resource Provider Scaffolding v2
from __future__ import annotations

from pathlib import Path
from typing import TypedDict

import localstack.services.cloudformation.provider_utils as util
from localstack.services.cloudformation.resource_provider import (
    OperationStatus,
    ProgressEvent,
    ResourceProvider,
    ResourceRequest,
)


class SSMMaintenanceWindowTaskProperties(TypedDict):
    Priority: int | None
    TaskArn: str | None
    TaskType: str | None
    WindowId: str | None
    CutoffBehavior: str | None
    Description: str | None
    Id: str | None
    LoggingInfo: LoggingInfo | None
    MaxConcurrency: str | None
    MaxErrors: str | None
    Name: str | None
    ServiceRoleArn: str | None
    Targets: list[Target] | None
    TaskInvocationParameters: TaskInvocationParameters | None
    TaskParameters: dict | None


class Target(TypedDict):
    Key: str | None
    Values: list[str] | None


class MaintenanceWindowStepFunctionsParameters(TypedDict):
    Input: str | None
    Name: str | None


class CloudWatchOutputConfig(TypedDict):
    CloudWatchLogGroupName: str | None
    CloudWatchOutputEnabled: bool | None


class NotificationConfig(TypedDict):
    NotificationArn: str | None
    NotificationEvents: list[str] | None
    NotificationType: str | None


class MaintenanceWindowRunCommandParameters(TypedDict):
    CloudWatchOutputConfig: CloudWatchOutputConfig | None
    Comment: str | None
    DocumentHash: str | None
    DocumentHashType: str | None
    DocumentVersion: str | None
    NotificationConfig: NotificationConfig | None
    OutputS3BucketName: str | None
    OutputS3KeyPrefix: str | None
    Parameters: dict | None
    ServiceRoleArn: str | None
    TimeoutSeconds: int | None


class MaintenanceWindowLambdaParameters(TypedDict):
    ClientContext: str | None
    Payload: str | None
    Qualifier: str | None


class MaintenanceWindowAutomationParameters(TypedDict):
    DocumentVersion: str | None
    Parameters: dict | None


class TaskInvocationParameters(TypedDict):
    MaintenanceWindowAutomationParameters: MaintenanceWindowAutomationParameters | None
    MaintenanceWindowLambdaParameters: MaintenanceWindowLambdaParameters | None
    MaintenanceWindowRunCommandParameters: MaintenanceWindowRunCommandParameters | None
    MaintenanceWindowStepFunctionsParameters: MaintenanceWindowStepFunctionsParameters | None


class LoggingInfo(TypedDict):
    Region: str | None
    S3Bucket: str | None
    S3Prefix: str | None


REPEATED_INVOCATION = "repeated_invocation"


class SSMMaintenanceWindowTaskProvider(ResourceProvider[SSMMaintenanceWindowTaskProperties]):
    TYPE = "AWS::SSM::MaintenanceWindowTask"  # Autogenerated. Don't change
    SCHEMA = util.get_schema_path(Path(__file__))  # Autogenerated. Don't change

    def create(
        self,
        request: ResourceRequest[SSMMaintenanceWindowTaskProperties],
    ) -> ProgressEvent[SSMMaintenanceWindowTaskProperties]:
        """
        Create a new resource.

        Primary identifier fields:
          - /properties/Id

        Required properties:
          - WindowId
          - Priority
          - TaskType
          - TaskArn

        Create-only properties:
          - /properties/WindowId
          - /properties/TaskType

        Read-only properties:
          - /properties/Id



        """
        model = request.desired_state
        ssm = request.aws_client_factory.ssm

        params = util.select_attributes(
            model=model,
            params=[
                "Description",
                "Name",
                "OwnerInformation",
                "Priority",
                "ServiceRoleArn",
                "Targets",
                "TaskArn",
                "TaskParameters",
                "TaskType",
                "WindowId",
            ],
        )

        if invocation_params := model.get("TaskInvocationParameters"):
            task_type_map = {
                "MaintenanceWindowAutomationParameters": "Automation",
                "MaintenanceWindowLambdaParameters": "Lambda",
                "MaintenanceWindowRunCommandParameters": "RunCommand",
                "MaintenanceWindowStepFunctionsParameters": "StepFunctions",
            }
            params["TaskInvocationParameters"] = {
                task_type_map[k]: v for k, v in invocation_params.items()
            }

        response = ssm.register_task_with_maintenance_window(**params)

        model["Id"] = response["WindowTaskId"]

        return ProgressEvent(
            status=OperationStatus.SUCCESS,
            resource_model=model,
            custom_context=request.custom_context,
        )

    def read(
        self,
        request: ResourceRequest[SSMMaintenanceWindowTaskProperties],
    ) -> ProgressEvent[SSMMaintenanceWindowTaskProperties]:
        """
        Fetch resource information


        """
        raise NotImplementedError

    def delete(
        self,
        request: ResourceRequest[SSMMaintenanceWindowTaskProperties],
    ) -> ProgressEvent[SSMMaintenanceWindowTaskProperties]:
        """
        Delete a resource


        """
        model = request.desired_state
        ssm = request.aws_client_factory.ssm

        ssm.deregister_task_from_maintenance_window(
            WindowId=model["WindowId"], WindowTaskId=model["Id"]
        )

        return ProgressEvent(
            status=OperationStatus.SUCCESS,
            resource_model=model,
            custom_context=request.custom_context,
        )

    def update(
        self,
        request: ResourceRequest[SSMMaintenanceWindowTaskProperties],
    ) -> ProgressEvent[SSMMaintenanceWindowTaskProperties]:
        """
        Update a resource


        """
        raise NotImplementedError
