# LocalStack Resource Provider Scaffolding v2
from __future__ import annotations

from pathlib import Path
from typing import TypedDict

import localstack.services.cloudformation.provider_utils as util
from localstack.services.cloudformation.resource_provider import (
    OperationStatus,
    ProgressEvent,
    ResourceProvider,
    ResourceRequest,
)


class SSMPatchBaselineProperties(TypedDict):
    Name: str | None
    ApprovalRules: RuleGroup | None
    ApprovedPatches: list[str] | None
    ApprovedPatchesComplianceLevel: str | None
    ApprovedPatchesEnableNonSecurity: bool | None
    Description: str | None
    GlobalFilters: PatchFilterGroup | None
    Id: str | None
    OperatingSystem: str | None
    PatchGroups: list[str] | None
    RejectedPatches: list[str] | None
    RejectedPatchesAction: str | None
    Sources: list[PatchSource] | None
    Tags: list[Tag] | None


class PatchFilter(TypedDict):
    Key: str | None
    Values: list[str] | None


class PatchFilterGroup(TypedDict):
    PatchFilters: list[PatchFilter] | None


class Rule(TypedDict):
    ApproveAfterDays: int | None
    ApproveUntilDate: dict | None
    ComplianceLevel: str | None
    EnableNonSecurity: bool | None
    PatchFilterGroup: PatchFilterGroup | None


class RuleGroup(TypedDict):
    PatchRules: list[Rule] | None


class PatchSource(TypedDict):
    Configuration: str | None
    Name: str | None
    Products: list[str] | None


class Tag(TypedDict):
    Key: str | None
    Value: str | None


REPEATED_INVOCATION = "repeated_invocation"


class SSMPatchBaselineProvider(ResourceProvider[SSMPatchBaselineProperties]):
    TYPE = "AWS::SSM::PatchBaseline"  # Autogenerated. Don't change
    SCHEMA = util.get_schema_path(Path(__file__))  # Autogenerated. Don't change

    def create(
        self,
        request: ResourceRequest[SSMPatchBaselineProperties],
    ) -> ProgressEvent[SSMPatchBaselineProperties]:
        """
        Create a new resource.

        Primary identifier fields:
          - /properties/Id

        Required properties:
          - Name

        Create-only properties:
          - /properties/OperatingSystem

        Read-only properties:
          - /properties/Id



        """
        model = request.desired_state
        ssm = request.aws_client_factory.ssm

        params = util.select_attributes(
            model=model,
            params=[
                "OperatingSystem",
                "Name",
                "GlobalFilters",
                "ApprovalRules",
                "ApprovedPatches",
                "ApprovedPatchesComplianceLevel",
                "ApprovedPatchesEnableNonSecurity",
                "RejectedPatches",
                "RejectedPatchesAction",
                "Description",
                "Sources",
                "ClientToken",
                "Tags",
            ],
        )

        response = ssm.create_patch_baseline(**params)
        model["Id"] = response["BaselineId"]

        return ProgressEvent(
            status=OperationStatus.SUCCESS,
            resource_model=model,
            custom_context=request.custom_context,
        )

    def read(
        self,
        request: ResourceRequest[SSMPatchBaselineProperties],
    ) -> ProgressEvent[SSMPatchBaselineProperties]:
        """
        Fetch resource information


        """
        raise NotImplementedError

    def delete(
        self,
        request: ResourceRequest[SSMPatchBaselineProperties],
    ) -> ProgressEvent[SSMPatchBaselineProperties]:
        """
        Delete a resource


        """
        model = request.desired_state
        ssm = request.aws_client_factory.ssm

        ssm.delete_patch_baseline(BaselineId=model["Id"])

        return ProgressEvent(
            status=OperationStatus.SUCCESS,
            resource_model=model,
            custom_context=request.custom_context,
        )

    def update(
        self,
        request: ResourceRequest[SSMPatchBaselineProperties],
    ) -> ProgressEvent[SSMPatchBaselineProperties]:
        """
        Update a resource


        """
        raise NotImplementedError
