# LocalStack Resource Provider Scaffolding v2
from __future__ import annotations

from pathlib import Path
from typing import TypedDict

import localstack.services.cloudformation.provider_utils as util
from localstack.services.cloudformation.resource_provider import (
    OperationStatus,
    ProgressEvent,
    ResourceProvider,
    ResourceRequest,
)


class StepFunctionsActivityProperties(TypedDict):
    Name: str | None
    Arn: str | None
    Tags: list[TagsEntry] | None


class TagsEntry(TypedDict):
    Key: str | None
    Value: str | None


REPEATED_INVOCATION = "repeated_invocation"


class StepFunctionsActivityProvider(ResourceProvider[StepFunctionsActivityProperties]):
    TYPE = "AWS::StepFunctions::Activity"  # Autogenerated. Don't change
    SCHEMA = util.get_schema_path(Path(__file__))  # Autogenerated. Don't change

    def create(
        self,
        request: ResourceRequest[StepFunctionsActivityProperties],
    ) -> ProgressEvent[StepFunctionsActivityProperties]:
        """
        Create a new resource.

        Primary identifier fields:
          - /properties/Arn

        Required properties:
          - Name

        Create-only properties:
          - /properties/Name

        Read-only properties:
          - /properties/Arn

        IAM permissions required:
          - states:CreateActivity

        """
        model = request.desired_state
        step_functions = request.aws_client_factory.stepfunctions
        response = step_functions.create_activity(**model)
        model["Arn"] = response["activityArn"]

        return ProgressEvent(
            status=OperationStatus.SUCCESS,
            resource_model=model,
            custom_context=request.custom_context,
        )

    def read(
        self,
        request: ResourceRequest[StepFunctionsActivityProperties],
    ) -> ProgressEvent[StepFunctionsActivityProperties]:
        """
        Fetch resource information

        IAM permissions required:
          - states:DescribeActivity
          - states:ListTagsForResource
        """
        raise NotImplementedError

    def delete(
        self,
        request: ResourceRequest[StepFunctionsActivityProperties],
    ) -> ProgressEvent[StepFunctionsActivityProperties]:
        """
        Delete a resource

        IAM permissions required:
          - states:DeleteActivity
        """
        model = request.desired_state
        step_functions = request.aws_client_factory.stepfunctions

        step_functions.delete_activity(activityArn=model["Arn"])

        return ProgressEvent(
            status=OperationStatus.SUCCESS,
            resource_model=model,
            custom_context=request.custom_context,
        )

    def update(
        self,
        request: ResourceRequest[StepFunctionsActivityProperties],
    ) -> ProgressEvent[StepFunctionsActivityProperties]:
        """
        Update a resource

        IAM permissions required:
          - states:ListTagsForResource
          - states:TagResource
          - states:UntagResource
        """
        raise NotImplementedError
