# SyncEvent

This is an attribute that can be put on events in NetworkBehaviour classes to allow them to be invoked on client when the event is called on the server.

SyncEvent events are called by user code on the server, and then invoked on corresponding client objects on clients connected to the server. The arguments to the Event call are serialized across the network, so that the client event is invoked with the same values as the method on the server. These events must begin with the prefix "Event".

```cs
using UnityEngine;
using Mirror;

public class DamageClass : NetworkBehaviour
{
    public delegate void TakeDamageDelegate(int amount, float dir);

    [SyncEvent]
    public event TakeDamageDelegate EventTakeDamage;

    [Command]
    public void CmdDoMe(int val)
    {
        EventTakeDamage(val, 1.0f);
    }
}

public class Other : NetworkBehaviour
{
    public DamageClass damager;
    int health = 100;

    void Start()
    {
        if (NetworkClient.active)
            damager.EventTakeDamage += TakeDamage;
    }

    public void TakeDamage(int amount, float dir)
    {
        health -=  amount;
    }
}
```

SyncEvents allow networked actions to be propagated to other scripts attached to the object. In the example above, the Other class registers for the TakeDamage event on the DamageClass. When the event happens on the DamageClass on the server, the TakeDamage() method will be invoked on the Other class on the client object. This allows modular network aware systems to be created, that can be extended by new scripts that respond to the events generated by them.
