﻿using System.Diagnostics;
using System.Net;
using System.Net.Http.Json;
using System.Text.Json;
using Microsoft.Extensions.Logging;
using Thor.Abstractions;
using Thor.Abstractions.Chats;
using Thor.Abstractions.Chats.Dtos;
using Thor.Abstractions.Exceptions;
using Thor.Abstractions.Extensions;

namespace Thor.MiniMax.Chats;

public class MiniMaxChatCompletionsService(ILogger<MiniMaxChatCompletionsService> logger) : IThorChatCompletionsService
{
    public async Task<ThorChatCompletionsResponse> ChatCompletionsAsync(ThorChatCompletionsRequest chatCompletionCreate,
        ThorPlatformOptions? options = null,
        CancellationToken cancellationToken = default)
    {
        using var openai =
            Activity.Current?.Source.StartActivity("MiniMax 对话补全");

        var response = await HttpClientFactory.GetHttpClient(options.Address).PostJsonAsync(
            options?.Address.TrimEnd('/') + "/v1/text/chatcompletion_v2",
            new
            {
                chatCompletionCreate.Model,
                chatCompletionCreate.Messages,
                chatCompletionCreate.MaxTokens,
                chatCompletionCreate.Temperature,
                chatCompletionCreate.N,
                chatCompletionCreate.PresencePenalty,
                chatCompletionCreate.FrequencyPenalty,
                stream = true,
                chatCompletionCreate.Logprobs,
                tool_choice = chatCompletionCreate.ToolChoiceCalculated,
                tools = chatCompletionCreate.Tools?.Select(x => new
                {
                    x.Type,
                    function = new
                    {
                        x.Function?.Name,
                        x.Function?.Description,
                        parameters = JsonSerializer.Serialize(x.Function, ThorJsonSerializer.DefaultOptions)
                    }
                }),
                chatCompletionCreate.User,
                chatCompletionCreate.ResponseFormat,
                chatCompletionCreate.TopLogprobs,
                stop = chatCompletionCreate.StopCalculated,
                chatCompletionCreate.ServiceTier
            }, options.ApiKey).ConfigureAwait(false);

        openai?.SetTag("Address", options?.Address.TrimEnd('/') + "/v1/text/chatcompletion_v2");
        openai?.SetTag("Model", chatCompletionCreate.Model);
        openai?.SetTag("Response", response.StatusCode.ToString());

        if (response.StatusCode == HttpStatusCode.Unauthorized)
        {
            throw new BusinessException("渠道未登录,请联系管理人员", "401");
        }

        // 如果限流则抛出限流异常
        if (response.StatusCode == HttpStatusCode.TooManyRequests)
        {
            throw new ThorRateLimitException();
        }

        // 大于等于400的状态码都认为是异常
        if (response.StatusCode >= HttpStatusCode.BadRequest)
        {
            var error = await response.Content.ReadAsStringAsync(cancellationToken).ConfigureAwait(false);
            logger.LogError("MiniMax对话异常 , StatusCode: {StatusCode} Response: {Response}", response.StatusCode, error);

            throw new BusinessException("MiniMax对话异常", response.StatusCode.ToString());
        }

        var result =
            await response.Content.ReadFromJsonAsync<ThorChatCompletionsResponse>(
                cancellationToken: cancellationToken).ConfigureAwait(false);

        return result;
    }

    public async IAsyncEnumerable<ThorChatCompletionsResponse> StreamChatCompletionsAsync(
        ThorChatCompletionsRequest chatCompletionCreate,
        ThorPlatformOptions? options = null,
        CancellationToken cancellationToken = default)
    {
        using var openai =
            Activity.Current?.Source.StartActivity("MiniMax 对话流式补全");

        var response = await HttpClientFactory.GetHttpClient(options.Address).HttpRequestRaw(
            options?.Address.TrimEnd('/') + "/v1/text/chatcompletion_v2",
            new
            {
                chatCompletionCreate.Model,
                chatCompletionCreate.Messages,
                chatCompletionCreate.MaxTokens,
                chatCompletionCreate.Temperature,
                chatCompletionCreate.N,
                chatCompletionCreate.PresencePenalty,
                chatCompletionCreate.FrequencyPenalty,
                stream = true,
                chatCompletionCreate.Logprobs,
                tool_choice = chatCompletionCreate.ToolChoiceCalculated,
                tools = chatCompletionCreate.Tools?.Select(x => new
                {
                    x.Type,
                    function = new
                    {
                        x.Function?.Name,
                        x.Function?.Description,
                        parameters = JsonSerializer.Serialize(x.Function, ThorJsonSerializer.DefaultOptions)
                    }
                }),
                chatCompletionCreate.User,
                chatCompletionCreate.ResponseFormat,
                chatCompletionCreate.TopLogprobs,
                stop = chatCompletionCreate.StopCalculated,
                chatCompletionCreate.ServiceTier
            }, options.ApiKey);

        openai?.SetTag("Address", options?.Address.TrimEnd('/') + "/v1/text/chatcompletion_v2");
        openai?.SetTag("Model", chatCompletionCreate.Model);
        openai?.SetTag("Response", response.StatusCode.ToString());

        if (response.StatusCode == HttpStatusCode.Unauthorized)
        {
            throw new UnauthorizedAccessException();
        }

        if (response.StatusCode == HttpStatusCode.PaymentRequired)
        {
            throw new PaymentRequiredException();
        }

        // 如果限流则抛出限流异常
        if (response.StatusCode == HttpStatusCode.TooManyRequests)
        {
            throw new ThorRateLimitException();
        }

        // 大于等于400的状态码都认为是异常
        if (response.StatusCode >= HttpStatusCode.BadRequest)
        {
            logger.LogError("MiniMax对话异常 , StatusCode: {StatusCode} ", response.StatusCode);

            throw new BusinessException("MiniMax对话异常", response.StatusCode.ToString());
        }

        using var stream = new StreamReader(await response.Content.ReadAsStreamAsync(cancellationToken));

        using StreamReader reader = new(await response.Content.ReadAsStreamAsync(cancellationToken));
        string? line = string.Empty;
        while ((line = await reader.ReadLineAsync().ConfigureAwait(false)) != null)
        {
            line += Environment.NewLine;

            if (line.StartsWith('{'))
            {
                logger.LogInformation("MiniMax对话异常 , StatusCode: {StatusCode} Response: {Response}",
                    response.StatusCode,
                    line);

                throw new BusinessException("MiniMax对话异常", line);
            }

            if (line.StartsWith("data:"))
                line = line["data:".Length..];

            line = line.Trim();

            if (line == "[DONE]")
            {
                break;
            }

            if (line.StartsWith(":"))
            {
                continue;
            }


            if (string.IsNullOrWhiteSpace(line)) continue;

            var result = JsonSerializer.Deserialize<ThorChatCompletionsResponse>(line,
                ThorJsonSerializer.DefaultOptions);
            yield return result;
        }
    }
}