﻿// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// See the LICENSE file in the project root for more information.

using Microsoft.Xunit.Performance;
using System.Collections.Generic;
using System.IO;
using System.Xml;

namespace System.Runtime.Serialization.Xml.Tests.Performance
{
    public class XmlUTF8TextReaderTests
    {
        // <Root>Testing</Root>
        private static byte[] trivial = new byte[] { 0x3C, 0x52, 0x6F, 0x6F, 0x74, 0x3E, 0x54, 0x65, 0x73, 0x74, 0x69, 0x6E, 0x67, 0x3C, 0x2F, 0x52, 0x6F, 0x6F, 0x74, 0x3E };

        // <Root>?????</Root>
        private static byte[] normal = new byte[] { 0x3C, 0x52, 0x6F, 0x6F, 0x74, 0x3E, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0x3C, 0x2F, 0x52, 0x6F, 0x6F, 0x74, 0x3E };
        // <Root>?????Testing</Root>
        private static byte[] normalMore = new byte[] { 0x3C, 0x52, 0x6F, 0x6F, 0x74, 0x3E, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0x54, 0x65, 0x73, 0x74, 0x69, 0x6E, 0x67, 0x3C, 0x2F, 0x52, 0x6F, 0x6F, 0x74, 0x3E };
        // <Root>???Testing???</Root>
        private static byte[] normalMore2 = new byte[] { 0x3C, 0x52, 0x6F, 0x6F, 0x74, 0x3E, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0x54, 0x65, 0x73, 0x74, 0x69, 0x6E, 0x67, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0x3C, 0x2F, 0x52, 0x6F, 0x6F, 0x74, 0x3E };

        // <Root>?????(invalid)</Root>
        private static byte[] invalidExtraEF = new byte[] { 0x3C, 0x52, 0x6F, 0x6F, 0x74, 0x3E, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0x3C, 0x2F, 0x52, 0x6F, 0x6F, 0x74, 0x3E };
        private static byte[] invalidExtraEFBF = new byte[] { 0x3C, 0x52, 0x6F, 0x6F, 0x74, 0x3E, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0xBF, 0x3C, 0x2F, 0x52, 0x6F, 0x6F, 0x74, 0x3E };
        private static byte[] invalidExtraEFBFBE = new byte[] { 0x3C, 0x52, 0x6F, 0x6F, 0x74, 0x3E, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0xBF, 0xBE, 0x3C, 0x2F, 0x52, 0x6F, 0x6F, 0x74, 0x3E };
        private static byte[] invalidExtraEFBFBF = new byte[] { 0x3C, 0x52, 0x6F, 0x6F, 0x74, 0x3E, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0xBF, 0xBF, 0x3C, 0x2F, 0x52, 0x6F, 0x6F, 0x74, 0x3E };

        // <Root>?????(invalid)Testing</Root>
        private static byte[] invalidExtraEFMore = new byte[] { 0x3C, 0x52, 0x6F, 0x6F, 0x74, 0x3E, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0x54, 0x65, 0x73, 0x74, 0x69, 0x6E, 0x67, 0x3C, 0x2F, 0x52, 0x6F, 0x6F, 0x74, 0x3E };
        private static byte[] invalidExtraEFBFMore = new byte[] { 0x3C, 0x52, 0x6F, 0x6F, 0x74, 0x3E, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0xBF, 0x54, 0x65, 0x73, 0x74, 0x69, 0x6E, 0x67, 0x3C, 0x2F, 0x52, 0x6F, 0x6F, 0x74, 0x3E };
        private static byte[] invalidExtraEFBFBFMore = new byte[] { 0x3C, 0x52, 0x6F, 0x6F, 0x74, 0x3E, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0xBF, 0xBF, 0x54, 0x65, 0x73, 0x74, 0x69, 0x6E, 0x67, 0x3C, 0x2F, 0x52, 0x6F, 0x6F, 0x74, 0x3E };

        // <Root>?????(invalid)EOF
        private static byte[] invalidExtraEFBFBF_EOF = new byte[] { 0x3C, 0x52, 0x6F, 0x6F, 0x74, 0x3E, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0x80, 0x80, 0xEF, 0xBF, 0xBF };
        private static byte[] invalidExtraEFBF_EOF = new byte[] { 0x3C, 0x52, 0x6F, 0x6F, 0x74, 0x3E, 0xEF, 0x80, 0x80, 0xEF, 0xBF };
        private static byte[] invalidExtraEF_EOF = new byte[] { 0x3C, 0x52, 0x6F, 0x6F, 0x74, 0x3E, 0xEF, 0x80, 0x80, 0xEF };

        // Dictionary<byteArrayToTest, shouldPass>
        private static Dictionary<byte[], bool> tests = new Dictionary<byte[], bool>();

        private static byte[] GenerateTestAsciiBytes(int numChars)
        {
            byte[] byteArray = new byte[numChars + 13];

            int offset = 0;

            byteArray[offset++] = 0x3C;     // < 
            byteArray[offset++] = 0x52;     // R
            byteArray[offset++] = 0x6F;     // o
            byteArray[offset++] = 0x6F;     // o
            byteArray[offset++] = 0x74;     // t
            byteArray[offset++] = 0x3E;     // >

            for (int i = 0; i < numChars; i++)
            {
                byteArray[offset++] = (byte)((int)'a' + i % 10);
            }

            byteArray[offset++] = 0x3C;     // < 
            byteArray[offset++] = 0x2F;     // /
            byteArray[offset++] = 0x52;     // R
            byteArray[offset++] = 0x6F;     // o
            byteArray[offset++] = 0x6F;     // o
            byteArray[offset++] = 0x74;     // t
            byteArray[offset++] = 0x3E;     // >

            return byteArray;
        }

        private static byte[] GenerateTestEFBytes(int numChars)
        {
            byte[] byteArray = new byte[3 * numChars + 13];

            int offset = 0;

            byteArray[offset++] = 0x3C;     // < 
            byteArray[offset++] = 0x52;     // R
            byteArray[offset++] = 0x6F;     // o
            byteArray[offset++] = 0x6F;     // o
            byteArray[offset++] = 0x74;     // t
            byteArray[offset++] = 0x3E;     // >

            for (int i = 0; i < numChars; i++)
            {
                byteArray[offset++] = 0xEF;
                byteArray[offset++] = 0xBF;
                byteArray[offset++] = (byte)(0x0A + numChars % 10);
            }

            byteArray[offset++] = 0x3C;     // < 
            byteArray[offset++] = 0x2F;     // /
            byteArray[offset++] = 0x52;     // R
            byteArray[offset++] = 0x6F;     // o
            byteArray[offset++] = 0x6F;     // o
            byteArray[offset++] = 0x74;     // t
            byteArray[offset++] = 0x3E;     // >

            return byteArray;
        }

        const int FiveMega = 5 * 1024 * 1024;

        public void RunPerfTest(byte[] testBytes, int iterations)
        {
            foreach (var iteration in Benchmark.Iterations)
            {
                using (MemoryStream testStream = new MemoryStream(testBytes))
                {
                    using (var reader = XmlDictionaryReader.CreateTextReader(testStream, XmlDictionaryReaderQuotas.Max))
                    {
                        using (var stream = new MemoryStream())
                        {
                            using (iteration.StartMeasurement())
                            {
                                for (int i = 0; i < iterations; i++)
                                {
                                    while (reader.Read())
                                    {
                                        ;
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }

        [Benchmark]
        public void TestAsciiBytes()
        {
            byte[] testBytes = GenerateTestAsciiBytes(FiveMega);
            RunPerfTest(testBytes, 1000);
        }

        [Benchmark]
        public void TestEFBytes()
        {
            byte[] testBytes = GenerateTestEFBytes(FiveMega);
            RunPerfTest(testBytes, 1000);
        }
    }
}
