"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.consoleFormat = exports.ConsoleFormat = void 0;
const safe_1 = __importDefault(require("colors/safe"));
const triple_beam_1 = require("triple-beam");
const util_1 = require("util");
class ConsoleFormat {
    constructor(opts = {}) {
        this.opts = opts;
        if (typeof this.opts.showMeta === "undefined") {
            this.opts.showMeta = true;
        }
    }
    // eslint-disable-next-line @typescript-eslint/no-explicit-any
    inspector(value, messages) {
        const inspector = util_1.inspect(value, this.opts.inspectOptions || {});
        inspector.split("\n").forEach((line) => {
            messages.push(line);
        });
    }
    message(info, chr, color) {
        const message = info.message.replace(ConsoleFormat.reSpacesOrEmpty, `$1${color}${safe_1.default.dim(chr)}${safe_1.default.reset(" ")}`);
        return `${info.level}:${message}`;
    }
    pad(message) {
        let spaces = "";
        const matches = message && message.match(ConsoleFormat.reSpaces);
        if (matches && matches.length > 0) {
            spaces = matches[0];
        }
        return spaces;
    }
    ms(info) {
        let ms = "";
        if (info.ms) {
            ms = safe_1.default.italic(safe_1.default.dim(` ${info.ms}`));
        }
        return ms;
    }
    stack(info) {
        const messages = [];
        if (info.stack) {
            const error = new Error();
            error.stack = info.stack;
            this.inspector(error, messages);
        }
        return messages;
    }
    meta(info) {
        const messages = [];
        const stripped = Object.assign({}, info);
        ConsoleFormat.defaultStrip.forEach((e) => delete stripped[e]);
        this.opts.metaStrip && this.opts.metaStrip.forEach((e) => delete stripped[e]);
        if (Object.keys(stripped).length > 0) {
            this.inspector(stripped, messages);
        }
        return messages;
    }
    getColor(info) {
        let color = "";
        const colorMatch = info.level.match(ConsoleFormat.reColor);
        if (colorMatch) {
            color = colorMatch[0];
        }
        return color;
    }
    write(info, messages, color) {
        const pad = this.pad(info.message);
        messages.forEach((line, index, arr) => {
            const lineNumber = safe_1.default.dim(`[${(index + 1).toString().padStart(arr.length.toString().length, " ")}]`);
            let chr = ConsoleFormat.chars.line;
            if (index === arr.length - 1) {
                chr = ConsoleFormat.chars.endLine;
            }
            info[triple_beam_1.MESSAGE] += `\n${safe_1.default.dim(info.level)}:${pad}${color}${safe_1.default.dim(chr)}${safe_1.default.reset(" ")}`;
            info[triple_beam_1.MESSAGE] += `${lineNumber} ${line}`;
        });
    }
    transform(info) {
        const messages = [];
        if (this.opts.showMeta) {
            messages.push(...this.stack(info));
            messages.push(...this.meta(info));
        }
        const color = this.getColor(info);
        info[triple_beam_1.MESSAGE] = this.message(info, ConsoleFormat.chars[messages.length > 0 ? "startLine" : "singleLine"], color);
        info[triple_beam_1.MESSAGE] += this.ms(info);
        this.write(info, messages, color);
        return info;
    }
}
exports.ConsoleFormat = ConsoleFormat;
ConsoleFormat.reSpaces = /^\s+/;
ConsoleFormat.reSpacesOrEmpty = /^(\s*)/;
ConsoleFormat.reColor = /\x1B\[\d+m/;
ConsoleFormat.defaultStrip = [triple_beam_1.LEVEL, triple_beam_1.MESSAGE, triple_beam_1.SPLAT, "level", "message", "ms", "stack"];
ConsoleFormat.chars = {
    singleLine: "▪",
    startLine: "┏",
    line: "┃",
    endLine: "┗",
};
exports.consoleFormat = (opts) => {
    return new ConsoleFormat(opts);
};
//# sourceMappingURL=index.js.map