'use strict';

Object.defineProperty(exports, '__esModule', { value: true });

const graphql = require('graphql');

const BigIntMock = () => BigInt(Number.MAX_SAFE_INTEGER);
const ByteMock = () => new Uint8Array([1988, 1981, 1965, 1963, 1959, 1955]);
const DateMock = () => '2007-12-03';
const Time = () => '10:15:30Z';
const DateTime = () => '2007-12-03T10:15:30Z';
const Timestamp = () => 1592577642;
const UtcOffset = () => '+03:00';
const Duration = () => 'P3Y6M4DT12H30M5S';
const LocalDate = () => '2020-07-19';
const LocalTime = () => '08:45:59';
const LocalEndTime = () => '24:00:00';
const EmailAddress = () => 'test@test.com';
const NegativeFloat = () => -123.45;
const NegativeInt = () => -123;
const NonEmptyString = () => 'string';
const NonNegativeFloat = () => 123.45;
const NonNegativeInt = () => 123;
const NonPositiveFloat = () => -123.45;
const NonPositiveInt = () => -123;
const PhoneNumber = () => '+17895551234';
const ObjectID = () => '5e5677d71bdc2ae76344968c';
const PositiveFloat = () => 123.45;
const PositiveInt = () => 123;
const PostalCode = () => '60031';
const URLMock = () => new URL('http://www.test.com/');
// https://stackoverflow.com/questions/105034/create-guid-uuid-in-javascript
const UUID = () => {
    // Public Domain/MIT
    let d = new Date().getTime();
    if (typeof performance !== 'undefined' &&
        typeof performance.now === 'function') {
        d += performance.now(); // use high-precision timer if available
    }
    return 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, function (c) {
        // tslint:disable-next-line: no-bitwise
        const r = (d + Math.random() * 16) % 16 | 0;
        d = Math.floor(d / 16);
        // tslint:disable-next-line: no-bitwise
        return (c === 'x' ? r : (r & 0x3) | 0x8).toString(16);
    });
};
const HexColorCode = () => '#' + Math.floor(Math.random() * 16777215).toString(16);
const Hexadecimal = () => Math.floor(Math.random() * 16777215).toString(16);
const randomVal = (min, max) => {
    return Math.floor(Math.random() * (max - min) + 1) + min;
};
// https://codepen.io/meowwwls/pen/jbEJRp
const HSL = () => `hsl(${randomVal(0, 360)}, ${randomVal(30, 95)}%, ${randomVal(30, 80)}%)`;
const HSLA = () => `hsla(${randomVal(0, 360)}, ${randomVal(30, 95)}%, ${randomVal(30, 80)}%, ${Math.random()})`;
// https://stackoverflow.com/questions/43464519/creating-fake-ip-address-using-javascript
const IPv4 = () => Math.floor(Math.random() * 255) +
    1 +
    '.' +
    (Math.floor(Math.random() * 255) + 0) +
    '.' +
    (Math.floor(Math.random() * 255) + 0) +
    '.' +
    (Math.floor(Math.random() * 255) + 0);
const IPv6 = () => '2001:0db8:85a3:0000:0000:8a2e:0370:7334';
// http://jsfiddle.net/guest271314/qhbC9/
const MAC = () => 'XX:XX:XX:XX:XX:XX'.replace(/X/g, () => '0123456789ABCDEF'.charAt(Math.floor(Math.random() * 16)));
const Port = () => randomVal(0, 65535);
const RGB = () => `rgb(${randomVal(0, 255)}, ${randomVal(0, 255)}, ${randomVal(0, 255)})`;
const RGBA = () => `rgba(${randomVal(0, 255)}, ${randomVal(0, 255)}, ${randomVal(0, 255)}, ${Math.random()})`;
const ISBN = () => `978-3-16-148410-0`;
const JWT = () => {
    // HEADER: {
    //   "alg": "HS256",
    //   "typ": "JWT"
    // }
    //
    // PAYLOAD: {
    //   "sub": "1234567890",
    //   "iat": 1516239022,
    //   "project": "graphql-scalars"
    // }
    //
    // SIGNATURE: {
    //   HMACSHA256(
    //     base64UrlEncode(header) + "." +
    //     base64UrlEncode(payload),
    //     password
    //   )
    // }
    return `eyJhbGciOiJIUzI1NiIsInR5cCI6IkpXVCJ9.eyJzdWIiOiIxMjM0NTY3ODkwIiwiaWF0IjoxNTE2MjM5MDIyLCJwcm9qZWN0IjoiZ3JhcGhxbC1zY2FsYXJzIn0.nYdrSfE2nNRAgpiEU1uKgn2AYYKLo28Z0nhPXvsuIww`;
};
const Latitude = () => 41.902782;
const Longitude = () => 12.496366;
const USCurrency = () => 1000;
const Currency = () => 'USD';
const JSON$1 = () => ({});
const JSONObject = () => ({});
const IBAN = () => 'NL55INGB4789170233';
const Void = () => null;
const SafeInt = () => Number.MAX_SAFE_INTEGER;

const mocks = /*#__PURE__*/Object.freeze({
  __proto__: null,
  Time: Time,
  DateTime: DateTime,
  Timestamp: Timestamp,
  UtcOffset: UtcOffset,
  Duration: Duration,
  LocalDate: LocalDate,
  LocalTime: LocalTime,
  LocalEndTime: LocalEndTime,
  EmailAddress: EmailAddress,
  NegativeFloat: NegativeFloat,
  NegativeInt: NegativeInt,
  NonEmptyString: NonEmptyString,
  NonNegativeFloat: NonNegativeFloat,
  NonNegativeInt: NonNegativeInt,
  NonPositiveFloat: NonPositiveFloat,
  NonPositiveInt: NonPositiveInt,
  PhoneNumber: PhoneNumber,
  ObjectID: ObjectID,
  PositiveFloat: PositiveFloat,
  PositiveInt: PositiveInt,
  PostalCode: PostalCode,
  UUID: UUID,
  HexColorCode: HexColorCode,
  Hexadecimal: Hexadecimal,
  HSL: HSL,
  HSLA: HSLA,
  IPv4: IPv4,
  IPv6: IPv6,
  MAC: MAC,
  Port: Port,
  RGB: RGB,
  RGBA: RGBA,
  ISBN: ISBN,
  JWT: JWT,
  Latitude: Latitude,
  Longitude: Longitude,
  USCurrency: USCurrency,
  Currency: Currency,
  JSON: JSON$1,
  JSONObject: JSONObject,
  IBAN: IBAN,
  Void: Void,
  SafeInt: SafeInt,
  Date: DateMock,
  URL: URLMock,
  UnsignedInt: NonNegativeInt,
  UnsignedFloat: NonNegativeFloat,
  GUID: UUID,
  Long: BigIntMock,
  BigInt: BigIntMock,
  Byte: ByteMock,
  ISO8601Duration: Duration
});

/**
 * Copyright (c) 2017, Dirk-Jan Rutten
 * All rights reserved.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 */
// Check whether a certain year is a leap year.
//
// Every year that is exactly divisible by four
// is a leap year, except for years that are exactly
// divisible by 100, but these centurial years are
// leap years if they are exactly divisible by 400.
// For example, the years 1700, 1800, and 1900 are not leap years,
// but the years 1600 and 2000 are.
//
const leapYear = (year) => {
    return (year % 4 === 0 && year % 100 !== 0) || year % 400 === 0;
};
// Function that checks whether a time-string is RFC 3339 compliant.
//
// It checks whether the time-string is structured in one of the
// following formats:
//
// - hh:mm:ssZ
// - hh:mm:ss±hh:mm
// - hh:mm:ss.*sZ
// - hh:mm:ss.*s±hh:mm
//
// Where *s is a fraction of seconds with at least 1 digit.
//
// Note, this validator assumes that all minutes have
// 59 seconds. This assumption does not follow RFC 3339
// which includes leap seconds (in which case it is possible that
// there are 60 seconds in a minute).
//
// Leap seconds are ignored because it adds complexity in
// the following areas:
// - The native Javascript Date ignores them; i.e. Date.parse('1972-12-31T23:59:60Z')
//   equals NaN.
// - Leap seconds cannot be known in advance.
//
const validateTime = (time) => {
    time = time === null || time === void 0 ? void 0 : time.toUpperCase();
    const TIME_REGEX = /^([01][0-9]|2[0-3]):([0-5][0-9]):([0-5][0-9])(\.\d{1,})?(([Z])|([+|-]([01][0-9]|2[0-3]):[0-5][0-9]))$/;
    return TIME_REGEX.test(time);
};
// Function that checks whether a date-string is RFC 3339 compliant.
//
// It checks whether the date-string is a valid date in the YYYY-MM-DD.
//
// Note, the number of days in each date are determined according to the
// following lookup table:
//
// Month Number  Month/Year           Maximum value of date-mday
// ------------  ----------           --------------------------
// 01            January              31
// 02            February, normal     28
// 02            February, leap year  29
// 03            March                31
// 04            April                30
// 05            May                  31
// 06            June                 30
// 07            July                 31
// 08            August               31
// 09            September            30
// 10            October              31
// 11            November             30
// 12            December             31
//
const validateDate = (datestring) => {
    const RFC_3339_REGEX = /^(\d{4}-(0[1-9]|1[012])-(0[1-9]|[12][0-9]|3[01]))$/;
    if (!RFC_3339_REGEX.test(datestring)) {
        return false;
    }
    // Verify the correct number of days for
    // the month contained in the date-string.
    const year = Number(datestring.substr(0, 4));
    const month = Number(datestring.substr(5, 2));
    const day = Number(datestring.substr(8, 2));
    switch (month) {
        case 2: // February
            if (leapYear(year) && day > 29) {
                return false;
            }
            else if (!leapYear(year) && day > 28) {
                return false;
            }
            return true;
        case 4: // April
        case 6: // June
        case 9: // September
        case 11: // November
            if (day > 30) {
                return false;
            }
            break;
    }
    return true;
};
// Function that checks whether a date-time-string is RFC 3339 compliant.
//
// It checks whether the time-string is structured in one of the
//
// - YYYY-MM-DDThh:mm:ssZ
// - YYYY-MM-DDThh:mm:ss±hh:mm
// - YYYY-MM-DDThh:mm:ss.*sZ
// - YYYY-MM-DDThh:mm:ss.*s±hh:mm
//
// Where *s is a fraction of seconds with at least 1 digit.
//
const validateDateTime = (dateTimeString) => {
    dateTimeString = dateTimeString === null || dateTimeString === void 0 ? void 0 : dateTimeString.toUpperCase();
    const RFC_3339_REGEX = /^(\d{4}-(0[1-9]|1[012])-(0[1-9]|[12][0-9]|3[01])T([01][0-9]|2[0-3]):([0-5][0-9]):([0-5][0-9]|60))(\.\d{1,})?(([Z])|([+|-]([01][0-9]|2[0-3]):[0-5][0-9]))$/;
    // Validate the structure of the date-string
    if (!RFC_3339_REGEX.test(dateTimeString)) {
        return false;
    }
    // Check if it is a correct date using the javascript Date parse() method.
    const time = Date.parse(dateTimeString);
    if (time !== time) {
        // eslint-disable-line
        return false;
    }
    // Split the date-time-string up into the string-date and time-string part.
    // and check whether these parts are RFC 3339 compliant.
    const index = dateTimeString.indexOf('T');
    const dateString = dateTimeString.substr(0, index);
    const timeString = dateTimeString.substr(index + 1);
    return validateDate(dateString) && validateTime(timeString);
};
// Function that checks whether a javascript Date instance
// is valid.
//
const validateJSDate = (date) => {
    const time = date.getTime();
    return time === time; // eslint-disable-line
};

/**
 * Copyright (c) 2017, Dirk-Jan Rutten
 * All rights reserved.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 */
// Parses an RFC 3339 compliant time-string into a Date.
// It does this by combining the current date with the time-string
// to create a new Date instance.
//
// Example:
// Suppose the current date is 2016-01-01, then
// parseTime('11:00:12Z') parses to a Date corresponding to
// 2016-01-01T11:00:12Z.
const parseTime = (time) => {
    const currentDateString = new Date().toISOString();
    return new Date(currentDateString.substr(0, currentDateString.indexOf('T') + 1) + time);
};
// Serializes a Date into an RFC 3339 compliant time-string in the
// format hh:mm:ss.sssZ.
const serializeTime = (date) => {
    const dateTimeString = date.toISOString();
    return dateTimeString.substr(dateTimeString.indexOf('T') + 1);
};
// Serializes an RFC 3339 compliant time-string by shifting
// it to UTC.
const serializeTimeString = (time) => {
    // If already formatted to UTC then return the time string
    if (time.indexOf('Z') !== -1) {
        return time;
    }
    else {
        // These are time-strings with timezone information,
        // these need to be shifted to UTC.
        // Convert to UTC time string in
        // format hh:mm:ss.sssZ.
        const date = parseTime(time);
        let timeUTC = serializeTime(date);
        // Regex to look for fractional second part in time string
        // such as 00:00:00.345+01:00
        const regexFracSec = /\.\d{1,}/;
        // Retrieve the fractional second part of the time
        // string if it exists.
        const fractionalPart = time.match(regexFracSec);
        if (fractionalPart == null) {
            // These are time-strings without the fractional
            // seconds. So we remove them from the UTC time-string.
            timeUTC = timeUTC.replace(regexFracSec, '');
            return timeUTC;
        }
        else {
            // These are time-string with fractional seconds.
            // Make sure that we inject the fractional
            // second part back in. The `timeUTC` variable
            // has millisecond precision, we may want more or less
            // depending on the string that was passed.
            timeUTC = timeUTC.replace(regexFracSec, fractionalPart[0]);
            return timeUTC;
        }
    }
};
// Parses an RFC 3339 compliant date-string into a Date.
//
// Example:
// parseDate('2016-01-01') parses to a Date corresponding to
// 2016-01-01T00:00:00.000Z.
const parseDate = (date) => {
    return new Date(date);
};
// Serializes a Date into a RFC 3339 compliant date-string
// in the format YYYY-MM-DD.
const serializeDate = (date) => {
    return date.toISOString().split('T')[0];
};
// Parses an RFC 3339 compliant date-time-string into a Date.
const parseDateTime = (dateTime) => {
    return new Date(dateTime);
};

/**
 * Copyright (c) 2017, Dirk-Jan Rutten
 * All rights reserved.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 */
const GraphQLDateConfig = {
    name: 'Date',
    description: 'A date string, such as 2007-12-03, compliant with the `full-date` ' +
        'format outlined in section 5.6 of the RFC 3339 profile of the ' +
        'ISO 8601 standard for representation of dates and times using ' +
        'the Gregorian calendar.',
    serialize(value) {
        if (value instanceof Date) {
            if (validateJSDate(value)) {
                return serializeDate(value);
            }
            throw new TypeError('Date cannot represent an invalid Date instance');
        }
        else if (typeof value === 'string') {
            if (validateDate(value)) {
                return value;
            }
            throw new TypeError(`Date cannot represent an invalid date-string ${value}.`);
        }
        else {
            throw new TypeError('Date cannot represent a non string, or non Date type ' +
                JSON.stringify(value));
        }
    },
    parseValue(value) {
        if (!(typeof value === 'string')) {
            throw new TypeError(`Date cannot represent non string type ${JSON.stringify(value)}`);
        }
        if (validateDate(value)) {
            return parseDate(value);
        }
        throw new TypeError(`Date cannot represent an invalid date-string ${value}.`);
    },
    parseLiteral(ast) {
        if (ast.kind !== graphql.Kind.STRING) {
            throw new TypeError(`Date cannot represent non string type ${'value' in ast && ast.value}`);
        }
        const { value } = ast;
        if (validateDate(value)) {
            return parseDate(value);
        }
        throw new TypeError(`Date cannot represent an invalid date-string ${String(value)}.`);
    },
};
/**
 * An RFC 3339 compliant date scalar.
 *
 * Input:
 *    This scalar takes an RFC 3339 date string as input and
 *    parses it to a javascript Date.
 *
 * Output:
 *    This scalar serializes javascript Dates and
 *    RFC 3339 date strings to RFC 3339 date strings.
 */
const GraphQLDate = /*#__PURE__*/ new graphql.GraphQLScalarType(GraphQLDateConfig);

/**
 * Copyright (c) 2017, Dirk-Jan Rutten
 * All rights reserved.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 */
/**
 * An RFC 3339 compliant time scalar.
 *
 * Input:
 *    This scalar takes an RFC 3339 time string as input and
 *    parses it to a javascript Date (with a year-month-day relative
 *    to the current day).
 *
 * Output:
 *    This scalar serializes javascript Dates and
 *    RFC 3339 time strings to RFC 3339 UTC time strings.
 */
const config = {
    name: 'Time',
    description: 'A time string at UTC, such as 10:15:30Z, compliant with ' +
        'the `full-time` format outlined in section 5.6 of the RFC 3339' +
        'profile of the ISO 8601 standard for representation of dates and ' +
        'times using the Gregorian calendar.',
    serialize(value) {
        if (value instanceof Date) {
            if (validateJSDate(value)) {
                return serializeTime(value);
            }
            throw new TypeError('Time cannot represent an invalid Date instance');
        }
        else if (typeof value === 'string') {
            if (validateTime(value)) {
                return serializeTimeString(value);
            }
            throw new TypeError(`Time cannot represent an invalid time-string ${value}.`);
        }
        else {
            throw new TypeError('Time cannot be serialized from a non string, ' +
                'or non Date type ' +
                JSON.stringify(value));
        }
    },
    parseValue(value) {
        if (!(typeof value === 'string')) {
            throw new TypeError(`Time cannot represent non string type ${JSON.stringify(value)}`);
        }
        if (validateTime(value)) {
            return parseTime(value);
        }
        throw new TypeError(`Time cannot represent an invalid time-string ${value}.`);
    },
    parseLiteral(ast) {
        if (ast.kind !== graphql.Kind.STRING) {
            throw new TypeError(`Time cannot represent non string type ${'value' in ast && ast.value}`);
        }
        const value = ast.value;
        if (validateTime(value)) {
            return parseTime(value);
        }
        throw new TypeError(`Time cannot represent an invalid time-string ${String(value)}.`);
    },
};
const GraphQLTime = /*#__PURE__*/ new graphql.GraphQLScalarType(config);

/**
 * Copyright (c) 2017, Dirk-Jan Rutten
 * All rights reserved.
 *
 * This source code is licensed under the MIT license found in the
 * LICENSE file in the root directory of this source tree.
 *
 */
const GraphQLDateTimeConfig = {
    name: 'DateTime',
    description: 'A date-time string at UTC, such as 2007-12-03T10:15:30Z, ' +
        'compliant with the `date-time` format outlined in section 5.6 of ' +
        'the RFC 3339 profile of the ISO 8601 standard for representation ' +
        'of dates and times using the Gregorian calendar.',
    serialize(value) {
        if (value instanceof Date) {
            if (validateJSDate(value)) {
                return value;
            }
            throw new TypeError('DateTime cannot represent an invalid Date instance');
        }
        else if (typeof value === 'string') {
            if (validateDateTime(value)) {
                return parseDateTime(value);
            }
            throw new TypeError(`DateTime cannot represent an invalid date-time-string ${value}.`);
        }
        else if (typeof value === 'number') {
            try {
                return new Date(value);
            }
            catch (e) {
                throw new TypeError('DateTime cannot represent an invalid Unix timestamp ' + value);
            }
        }
        else {
            throw new TypeError('DateTime cannot be serialized from a non string, ' +
                'non numeric or non Date type ' +
                JSON.stringify(value));
        }
    },
    parseValue(value) {
        if (value instanceof Date) {
            if (validateJSDate(value)) {
                return value;
            }
            throw new TypeError('DateTime cannot represent an invalid Date instance');
        }
        if (typeof value === 'string') {
            if (validateDateTime(value)) {
                return parseDateTime(value);
            }
            throw new TypeError(`DateTime cannot represent an invalid date-time-string ${value}.`);
        }
        throw new TypeError(`DateTime cannot represent non string or Date type ${JSON.stringify(value)}`);
    },
    parseLiteral(ast) {
        if (ast.kind !== graphql.Kind.STRING) {
            throw new TypeError(`DateTime cannot represent non string or Date type ${'value' in ast && ast.value}`);
        }
        const { value } = ast;
        if (validateDateTime(value)) {
            return parseDateTime(value);
        }
        throw new TypeError(`DateTime cannot represent an invalid date-time-string ${String(value)}.`);
    },
};
/**
 * An RFC 3339 compliant date-time scalar.
 *
 * Input:
 *    This scalar takes an RFC 3339 date-time string as input and
 *    parses it to a javascript Date.
 *
 * Output:
 *    This scalar serializes javascript Dates,
 *    RFC 3339 date-time strings and unix timestamps
 *    to RFC 3339 UTC date-time strings.
 */
const GraphQLDateTime = /*#__PURE__*/ new graphql.GraphQLScalarType(GraphQLDateTimeConfig);

// Taken from https://gist.github.com/langpavel/b30f3d507a47713b0c6e89016e4e9eb7
function serializeDate$1(value) {
    if (value instanceof Date) {
        return value.getTime();
    }
    else if (typeof value === 'number') {
        return Math.trunc(value);
    }
    else if (typeof value === 'string') {
        return Date.parse(value);
    }
    return null;
}
function parseDate$1(value) {
    if (value === null) {
        return null;
    }
    try {
        return new Date(value);
    }
    catch (err) {
        return null;
    }
}
function parseDateFromLiteral(ast) {
    if (ast.kind === graphql.Kind.INT) {
        const num = parseInt(ast.value, 10);
        return new Date(num);
    }
    else if (ast.kind === graphql.Kind.STRING) {
        return parseDate$1(ast.value);
    }
    return null;
}
const GraphQLTimestamp = /*#__PURE__*/ new graphql.GraphQLScalarType({
    name: 'Timestamp',
    description: 'The javascript `Date` as integer. Type represents date and time ' +
        'as number of milliseconds from start of UNIX epoch.',
    serialize: serializeDate$1,
    parseValue: parseDate$1,
    parseLiteral: parseDateFromLiteral,
});

const validate = (value) => {
    const UTC_OFFSET_REGEX = /^([+-]?)(\d{2}):(\d{2})$/;
    if (typeof value !== 'string') {
        throw new TypeError(`Value is not string: ${value}`);
    }
    if (!UTC_OFFSET_REGEX.test(value)) {
        throw new TypeError(`Value is not a valid UTC Offset: ${value}`);
    }
    return value;
};
const GraphQLUtcOffset = /*#__PURE__*/ new graphql.GraphQLScalarType({
    name: 'UtcOffset',
    description: 'A field whose value is a UTC Offset: https://en.wikipedia.org/wiki/List_of_tz_database_time_zones',
    serialize: validate,
    parseValue: validate,
    parseLiteral(ast) {
        if (ast.kind !== graphql.Kind.STRING) {
            throw new graphql.GraphQLError(`Can only validate strings as UTC Offset but got a: ${ast.kind}`);
        }
        return validate(ast.value);
    },
});

// original implementation
// https://stackoverflow.com/questions/32044846/regex-for-iso-8601-durations
// const ISO_DURATION_NEGATIVE_ALLOWED = /^-?P(?!$)(-?\d+(?:\.\d+)?Y)?(-?\d+(?:\.\d+)?M)?(-?\d+(?:\.\d+)?W)?(-?\d+(?:\.\d+)?D)?(T(?=-?\d)(-?\d+(?:\.\d+)?H)?(-?\d+(?:\.\d+)?M)?(-?\d+(?:\.\d+)?S)?)?$/
// const ISO_DURATION_WITHOUT_SIGN = /^P(?!$)(\d+(?:\.\d+)?Y)?(\d+(?:\.\d+)?M)?(\d+(?:\.\d+)?W)?(\d+(?:\.\d+)?D)?(T(?=\d)(\d+(?:\.\d+)?H)?(\d+(?:\.\d+)?M)?(\d+(?:\.\d+)?S)?)?$/
// negative and positive durations allowed, commas and decimal points valid for fractions
const ISO_DURATION = /^(-|\+)?P(?!$)((-|\+)?\d+(?:(\.|,)\d+)?Y)?((-|\+)?\d+(?:(\.|,)\d+)?M)?((-|\+)?\d+(?:(\.|,)\d+)?W)?((-|\+)?\d+(?:(\.|,)\d+)?D)?(T(?=(-|\+)?\d)((-|\+)?\d+(?:(\.|,)\d+)?H)?((-|\+)?\d+(?:(\.|,)\d+)?M)?((-|\+)?\d+(?:(\.|,)\d+)?S)?)?$/;
const GraphQLDurationConfig = {
    name: 'Duration',
    description: `
    A string representing a duration conforming to the ISO8601 standard,
    such as: P1W1DT13H23M34S
    P is the duration designator (for period) placed at the start of the duration representation.
    Y is the year designator that follows the value for the number of years.
    M is the month designator that follows the value for the number of months.
    W is the week designator that follows the value for the number of weeks.
    D is the day designator that follows the value for the number of days.
    T is the time designator that precedes the time components of the representation.
    H is the hour designator that follows the value for the number of hours.
    M is the minute designator that follows the value for the number of minutes.
    S is the second designator that follows the value for the number of seconds.

    Note the time designator, T, that precedes the time value.

    Matches moment.js, Luxon and DateFns implementations
    ,/. is valid for decimal places and +/- is a valid prefix
  `,
    serialize(value) {
        if (typeof value !== 'string') {
            throw new TypeError(`Value is not string: ${value}`);
        }
        if (!ISO_DURATION.test(value)) {
            throw new TypeError(`Value is not a valid ISO Duration: ${value}`);
        }
        return value;
    },
    parseValue(value) {
        if (typeof value !== 'string') {
            throw new TypeError(`Value is not string: ${value}`);
        }
        if (!ISO_DURATION.test(value)) {
            throw new TypeError(`Value is not a valid ISO Duration: ${value}`);
        }
        return value;
    },
    parseLiteral(ast) {
        if (ast.kind !== graphql.Kind.STRING) {
            throw new graphql.GraphQLError(`Can only validate strings as ISO Durations but got a: ${ast.kind}`);
        }
        if (!ISO_DURATION.test(ast.value)) {
            throw new TypeError(`Value is not a valid ISO Duration: ${ast.value}`);
        }
        return ast.value;
    },
};
const GraphQLISO8601Duration = /*#__PURE__*/ new graphql.GraphQLScalarType({
    ...GraphQLDurationConfig,
    name: 'ISO8601Duration',
});
const GraphQLDuration = /*#__PURE__*/ new graphql.GraphQLScalarType({
    ...GraphQLDurationConfig,
    name: 'Duration',
});

const LOCAL_DATE_FORMAT = /^\d{4}-\d{2}-\d{2}$/;
function validateLocalDate(value) {
    if (typeof value !== 'string') {
        throw new TypeError(`Value is not string: ${value}`);
    }
    // check that it's in the `yyyy-MM-dd` format
    const isValidFormat = LOCAL_DATE_FORMAT.test(value);
    if (!isValidFormat) {
        throw new TypeError(`Value is not a valid LocalDate: ${value}`);
    }
    // check that it appears to be a valid date, e.g., not something like `2020-13-46`
    const valueAsDate = new Date(value);
    const isValidDate = !isNaN(valueAsDate.getTime());
    if (!isValidDate) {
        throw new TypeError(`Value is not a valid LocalDate: ${value}`);
    }
    // some additional logic to catch invalid dates like `2020-02-30`
    // that we catch by serializing the Date object into an ISO string and checking that our serialized date matches
    // the original value
    const isCalendarDate = valueAsDate.toISOString() === `${value}T00:00:00.000Z`;
    if (!isCalendarDate) {
        throw new TypeError(`Value is not a valid LocalDate: ${value}`);
    }
    return value;
}
const GraphQLLocalDate = /*#__PURE__*/ new graphql.GraphQLScalarType({
    name: 'LocalDate',
    description: 'A local date string (i.e., with no associated timezone) in `YYYY-MM-DD` format, e.g. `2020-01-01`.',
    serialize(value) {
        // value sent to client as string
        return validateLocalDate(value);
    },
    parseValue(value) {
        // value from client as json
        return validateLocalDate(value);
    },
    parseLiteral(ast) {
        // value from client in ast
        if (ast.kind !== graphql.Kind.STRING) {
            throw new graphql.GraphQLError(`Can only validate strings as local dates but got a: ${ast.kind}`);
        }
        return validateLocalDate(ast.value);
    },
});

// 24-hour time with optional seconds and milliseconds - `HH:mm[:ss[.SSS]]`
const LOCAL_TIME_FORMAT = /^([0-1][0-9]|2[0-3]):([0-5][0-9])(:[0-5][0-9](\.\d{3})?)?$/;
function validateLocalTime(value) {
    if (typeof value !== 'string') {
        throw new TypeError(`Value is not string: ${value}`);
    }
    const isValidFormat = LOCAL_TIME_FORMAT.test(value);
    if (!isValidFormat) {
        throw new TypeError(`Value is not a valid LocalTime: ${value}`);
    }
    return value;
}
const GraphQLLocalTime = /*#__PURE__*/ new graphql.GraphQLScalarType({
    name: 'LocalTime',
    description: 'A local time string (i.e., with no associated timezone) in 24-hr `HH:mm[:ss[.SSS]]` format, e.g. `14:25` or `14:25:06` or `14:25:06.123`.',
    serialize(value) {
        // value sent to client as string
        return validateLocalTime(value);
    },
    parseValue(value) {
        // value from client as json
        return validateLocalTime(value);
    },
    parseLiteral(ast) {
        // value from client in ast
        if (ast.kind !== graphql.Kind.STRING) {
            throw new graphql.GraphQLError(`Can only validate strings as local times but got a: ${ast.kind}`);
        }
        return validateLocalTime(ast.value);
    },
});

const LOCAL_END_TIMES = ['24:00', '24:00:00', '24:00:00.000'];
function validateLocalEndTime(value) {
    // first check if it's any of the special "end time" values
    if (LOCAL_END_TIMES.indexOf(value) >= 0) {
        return value;
    }
    // otherwise, fall back on the standard LocalTime validation
    return validateLocalTime(value);
}
const GraphQLLocalEndTime = /*#__PURE__*/ new graphql.GraphQLScalarType({
    name: 'LocalEndTime',
    description: 'A local time string (i.e., with no associated timezone) in 24-hr `HH:mm[:ss[.SSS]]` format, e.g. `14:25` or `14:25:06` or `14:25:06.123`.  This scalar is very similar to the `LocalTime`, with the only difference being that `LocalEndTime` also allows `24:00` as a valid value to indicate midnight of the following day.  This is useful when using the scalar to represent the exclusive upper bound of a time block.',
    serialize(value) {
        // value sent to client as string
        return validateLocalEndTime(value);
    },
    parseValue(value) {
        // value from client as json
        return validateLocalEndTime(value);
    },
    parseLiteral(ast) {
        // value from client in ast
        if (ast.kind !== graphql.Kind.STRING) {
            throw new graphql.GraphQLError(`Can only validate strings as local times but got a: ${ast.kind}`);
        }
        return validateLocalEndTime(ast.value);
    },
});

const validate$1 = (value) => {
    const EMAIL_ADDRESS_REGEX = /^[a-zA-Z0-9.!#$%&'*+\/=?^_`{|}~-]+@[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?(?:\.[a-zA-Z0-9](?:[a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?)*$/;
    if (typeof value !== 'string') {
        throw new TypeError(`Value is not string: ${value}`);
    }
    if (!EMAIL_ADDRESS_REGEX.test(value)) {
        throw new TypeError(`Value is not a valid email address: ${value}`);
    }
    return value;
};
const GraphQLCurrencyConfig = {
    name: 'EmailAddress',
    description: 'A field whose value conforms to the standard internet email address format as specified in RFC822: https://www.w3.org/Protocols/rfc822/.',
    serialize: validate$1,
    parseValue: validate$1,
    parseLiteral(ast) {
        if (ast.kind !== graphql.Kind.STRING) {
            throw new graphql.GraphQLError(`Can only validate strings as email addresses but got a: ${ast.kind}`);
        }
        return validate$1(ast.value);
    },
    specifiedByUrl: 'https://www.w3.org/Protocols/rfc822/',
};
const GraphQLEmailAddress = /*#__PURE__*/ new graphql.GraphQLScalarType(GraphQLCurrencyConfig);

var VALUE_RANGES;
(function (VALUE_RANGES) {
    VALUE_RANGES[VALUE_RANGES["NEGATIVE"] = 0] = "NEGATIVE";
    VALUE_RANGES[VALUE_RANGES["NON_NEGATIVE"] = 1] = "NON_NEGATIVE";
    VALUE_RANGES[VALUE_RANGES["POSITIVE"] = 2] = "POSITIVE";
    VALUE_RANGES[VALUE_RANGES["NON_POSITIVE"] = 3] = "NON_POSITIVE";
})(VALUE_RANGES || (VALUE_RANGES = {}));
var VALUE_TYPES;
(function (VALUE_TYPES) {
    VALUE_TYPES[VALUE_TYPES["INT"] = 0] = "INT";
    VALUE_TYPES[VALUE_TYPES["FLOAT"] = 1] = "FLOAT";
})(VALUE_TYPES || (VALUE_TYPES = {}));
// More info about Sexagesimal: https://en.wikipedia.org/wiki/Sexagesimal
const SEXAGESIMAL_REGEX = /^([0-9]{1,3})°\s*([0-9]{1,3}(?:\.(?:[0-9]{1,}))?)['′]\s*(([0-9]{1,3}(\.([0-9]{1,}))?)["″]\s*)?([NEOSW]?)$/;
// TODO: Consider implementing coercion like this...
// See: https://github.com/graphql/graphql-js/blob/master/src/type/scalars.js#L13
// See: https://github.com/graphql/graphql-js/blob/master/src/type/scalars.js#L60
function _validateInt(value) {
    if (!Number.isFinite(value)) {
        throw new TypeError(`Value is not a finite number: ${value}`);
    }
    if (!Number.isInteger(value)) {
        throw new TypeError(`Value is not an integer: ${value}`);
    }
    if (!Number.isSafeInteger(value)) {
        throw new TypeError(`Value is not a safe integer: ${value}`);
    }
}
function _validateFloat(value) {
    if (!Number.isFinite(value)) {
        throw new TypeError(`Value is not a finite number: ${value}`);
    }
}
function processValue(value, scalarName) {
    const VALIDATIONS = {
        NonPositiveInt: {
            range: VALUE_RANGES.NON_POSITIVE,
            type: VALUE_TYPES.INT,
        },
        PositiveInt: {
            range: VALUE_RANGES.POSITIVE,
            type: VALUE_TYPES.INT,
        },
        NonNegativeInt: {
            range: VALUE_RANGES.NON_NEGATIVE,
            type: VALUE_TYPES.INT,
        },
        NegativeInt: {
            range: VALUE_RANGES.NEGATIVE,
            type: VALUE_TYPES.INT,
        },
        NonPositiveFloat: {
            range: VALUE_RANGES.NON_POSITIVE,
            type: VALUE_TYPES.FLOAT,
        },
        PositiveFloat: {
            range: VALUE_RANGES.POSITIVE,
            type: VALUE_TYPES.FLOAT,
        },
        NonNegativeFloat: {
            range: VALUE_RANGES.NON_NEGATIVE,
            type: VALUE_TYPES.FLOAT,
        },
        NegativeFloat: {
            range: VALUE_RANGES.NEGATIVE,
            type: VALUE_TYPES.FLOAT,
        },
    };
    const { range, type } = VALIDATIONS[scalarName];
    /* eslint-disable no-restricted-globals */
    if (value === null ||
        typeof value === 'undefined' ||
        isNaN(value) ||
        Number.isNaN(value) ||
        value === Number.NaN) {
        throw new TypeError(`Value is not a number: ${value}`);
    }
    /* eslint-enable */
    let parsedValue;
    switch (type) {
        case VALUE_TYPES.FLOAT:
            parsedValue = parseFloat(value);
            _validateFloat(parsedValue);
            break;
        case VALUE_TYPES.INT:
            parsedValue = parseInt(value, 10);
            _validateInt(parsedValue);
            break;
        // no -op, return undefined
    }
    if ((range === VALUE_RANGES.NEGATIVE && !(parsedValue < 0)) ||
        (range === VALUE_RANGES.NON_NEGATIVE && !(parsedValue >= 0)) ||
        (range === VALUE_RANGES.POSITIVE && !(parsedValue > 0)) ||
        (range === VALUE_RANGES.NON_POSITIVE && !(parsedValue <= 0))) {
        throw new TypeError(`Value is not a ${VALUE_RANGES[range]
            .toLowerCase()
            .replace('_', '-')} number: ${value}`);
    }
    return parsedValue;
}
/**
 * Check if the value is in decimal format.
 *
 * @param value - Value to check
 * @returns True if is decimal, false otherwise
 */
function isDecimal(value) {
    const checkedValue = value.toString().trim();
    if (Number.isNaN(Number.parseFloat(checkedValue))) {
        return false;
    }
    return Number.parseFloat(checkedValue) === Number(checkedValue);
}
/**
 * Check if the value is in sexagesimal format.
 *
 * @param value - Value to check
 * @returns True if sexagesimal, false otherwise
 */
function isSexagesimal(value) {
    if (typeof value !== 'string')
        return false;
    return SEXAGESIMAL_REGEX.test(value.toString().trim());
}
/**
 * Converts a sexagesimal coordinate to decimal format.
 *
 * @param value - Value to convert
 * @returns Decimal coordinate
 * @throws {TypeError} if the value is not in sexagesimal format
 */
function sexagesimalToDecimal(value) {
    const data = SEXAGESIMAL_REGEX.exec(value);
    if (typeof data === 'undefined' || data === null) {
        throw new TypeError(`Value is not in sexagesimal format: ${value}`);
    }
    const min = Number(data[2]) / 60 || 0;
    const sec = Number(data[4]) / 3600 || 0;
    const decimal = Number.parseFloat(data[1]) + min + sec;
    // Southern and western coordinates must be negative decimals
    return ['S', 'W'].includes(data[7]) ? -decimal : decimal;
}

const GraphQLNegativeFloat = /*#__PURE__*/ new graphql.GraphQLScalarType({
    name: 'NegativeFloat',
    description: 'Floats that will have a value less than 0.',
    serialize(value) {
        return processValue(value, 'NegativeFloat');
    },
    parseValue(value) {
        return processValue(value, 'NegativeFloat');
    },
    parseLiteral(ast) {
        if (ast.kind !== graphql.Kind.FLOAT && ast.kind !== graphql.Kind.INT) {
            throw new graphql.GraphQLError(`Can only validate floating point numbers as negative floating point numbers but got a: ${ast.kind}`);
        }
        return processValue(ast.value, 'NegativeFloat');
    },
});

const GraphQLNegativeInt = /*#__PURE__*/ new graphql.GraphQLScalarType({
    name: 'NegativeInt',
    description: 'Integers that will have a value less than 0.',
    serialize(value) {
        return processValue(value, 'NegativeInt');
    },
    parseValue(value) {
        return processValue(value, 'NegativeInt');
    },
    parseLiteral(ast) {
        if (ast.kind !== graphql.Kind.INT) {
            throw new graphql.GraphQLError(`Can only validate integers as negative integers but got a: ${ast.kind}`);
        }
        return processValue(ast.value, 'NegativeInt');
    },
});

const validate$2 = (value) => {
    if (typeof value !== 'string') {
        throw new TypeError(`Value is not a string: ${value}`);
    }
    if (!value.trim().length) {
        throw new TypeError(`Value cannot be an empty string: ${value}`);
    }
    return value;
};
const GraphQLNonEmptyString = /*#__PURE__*/ new graphql.GraphQLScalarType({
    name: 'NonEmptyString',
    description: 'A string that cannot be passed as an empty value',
    serialize: validate$2,
    parseValue: validate$2,
    parseLiteral(ast) {
        if (ast.kind !== graphql.Kind.STRING) {
            throw new graphql.GraphQLError(`Can only validate strings but got a: ${ast.kind}`);
        }
        return validate$2(ast.value);
    },
});

const GraphQLNonNegativeFloatConfig = {
    name: 'NonNegativeFloat',
    description: 'Floats that will have a value of 0 or more.',
    serialize(value) {
        return processValue(value, 'NonNegativeFloat');
    },
    parseValue(value) {
        return processValue(value, 'NonNegativeFloat');
    },
    parseLiteral(ast) {
        if (ast.kind !== graphql.Kind.FLOAT && ast.kind !== graphql.Kind.INT) {
            throw new graphql.GraphQLError(`Can only validate floating point numbers as non-negative floating point numbers but got a: ${ast.kind}`);
        }
        return processValue(ast.value, 'NonNegativeFloat');
    },
};
const GraphQLNonNegativeFloat = /*#__PURE__*/ new graphql.GraphQLScalarType(GraphQLNonNegativeFloatConfig);

const GraphQLNonNegativeIntConfig = {
    name: 'NonNegativeInt',
    description: 'Integers that will have a value of 0 or more.',
    serialize(value) {
        return processValue(value, 'NonNegativeInt');
    },
    parseValue(value) {
        return processValue(value, 'NonNegativeInt');
    },
    parseLiteral(ast) {
        if (ast.kind !== graphql.Kind.INT) {
            throw new graphql.GraphQLError(`Can only validate integers as non-negative integers but got a: ${ast.kind}`);
        }
        return processValue(ast.value, 'NonNegativeInt');
    },
};
const GraphQLNonNegativeInt = /*#__PURE__*/ new graphql.GraphQLScalarType(GraphQLNonNegativeIntConfig);

const GraphQLNonPositiveFloat = /*#__PURE__*/ new graphql.GraphQLScalarType({
    name: 'NonPositiveFloat',
    description: 'Floats that will have a value of 0 or less.',
    serialize(value) {
        return processValue(value, 'NonPositiveFloat');
    },
    parseValue(value) {
        return processValue(value, 'NonPositiveFloat');
    },
    parseLiteral(ast) {
        if (ast.kind !== graphql.Kind.FLOAT && ast.kind !== graphql.Kind.INT) {
            throw new graphql.GraphQLError(`Can only validate floating point numbers as non-positive floating point numbers but got a: ${ast.kind}`);
        }
        return processValue(ast.value, 'NonPositiveFloat');
    },
});

const GraphQLNonPositiveInt = /*#__PURE__*/ new graphql.GraphQLScalarType({
    name: 'NonPositiveInt',
    description: 'Integers that will have a value of 0 or less.',
    serialize(value) {
        return processValue(value, 'NonPositiveInt');
    },
    parseValue(value) {
        return processValue(value, 'NonPositiveInt');
    },
    parseLiteral(ast) {
        if (ast.kind !== graphql.Kind.INT) {
            throw new graphql.GraphQLError(`Can only validate integers as non-positive integers but got a: ${ast.kind}`);
        }
        return processValue(ast.value, 'NonPositiveInt');
    },
});

const PHONE_NUMBER_REGEX = /^\+[1-9]\d{1,14}$/;
const GraphQLPhoneNumber = /*#__PURE__*/ new graphql.GraphQLScalarType({
    name: 'PhoneNumber',
    description: 'A field whose value conforms to the standard E.164 format as specified in: https://en.wikipedia.org/wiki/E.164. Basically this is +17895551234.',
    serialize(value) {
        if (typeof value !== 'string') {
            throw new TypeError(`Value is not string: ${value}`);
        }
        if (!PHONE_NUMBER_REGEX.test(value)) {
            throw new TypeError(`Value is not a valid phone number of the form +17895551234 (10-15 digits): ${value}`);
        }
        return value;
    },
    parseValue(value) {
        if (typeof value !== 'string') {
            throw new TypeError(`Value is not string: ${value}`);
        }
        if (!PHONE_NUMBER_REGEX.test(value)) {
            throw new TypeError(`Value is not a valid phone number of the form +17895551234 (10-15 digits): ${value}`);
        }
        return value;
    },
    parseLiteral(ast) {
        if (ast.kind !== graphql.Kind.STRING) {
            throw new graphql.GraphQLError(`Can only validate strings as phone numbers but got a: ${ast.kind}`);
        }
        if (!PHONE_NUMBER_REGEX.test(ast.value)) {
            throw new TypeError(`Value is not a valid phone number of the form +17895551234 (10-15 digits): ${ast.value}`);
        }
        return ast.value;
    },
});

const GraphQLPositiveFloat = /*#__PURE__*/ new graphql.GraphQLScalarType({
    name: 'PositiveFloat',
    description: 'Floats that will have a value greater than 0.',
    serialize(value) {
        return processValue(value, 'PositiveFloat');
    },
    parseValue(value) {
        return processValue(value, 'PositiveFloat');
    },
    parseLiteral(ast) {
        if (ast.kind !== graphql.Kind.FLOAT && ast.kind !== graphql.Kind.INT) {
            throw new graphql.GraphQLError(`Can only validate floating point numbers as positive floating point numbers but got a: ${ast.kind}`);
        }
        return processValue(ast.value, 'PositiveFloat');
    },
});

const GraphQLPositiveInt = /*#__PURE__*/ new graphql.GraphQLScalarType({
    name: 'PositiveInt',
    description: 'Integers that will have a value greater than 0.',
    serialize(value) {
        return processValue(value, 'PositiveInt');
    },
    parseValue(value) {
        return processValue(value, 'PositiveInt');
    },
    parseLiteral(ast) {
        if (ast.kind !== graphql.Kind.INT) {
            throw new graphql.GraphQLError(`Can only validate integers as positive integers but got a: ${ast.kind}`);
        }
        return processValue(ast.value, 'PositiveInt');
    },
});

// We're going to start with a limited set as suggested here:
// http://www.pixelenvision.com/1708/zip-postal-code-validation-regex-php-code-for-12-countries/
// and here:
// https://stackoverflow.com/questions/578406/what-is-the-ultimate-postal-code-and-zip-regex
//
// Which gives us the following countries:
//
// US - United States
// UK - United Kingdom
// DE - Germany
// CA - Canada
// FR - France
// IT - Italy
// AU - Australia
// NL - Netherlands
// ES - Spain
// DK - Denmark
// SE - Sweden
// BE - Belgium
// IN - India
// AT - Austria
// PT - Portugal
// CH - Switzerland
// LU - Luxembourg
//
// This is really a practical decision of weight (of the package) vs. completeness.
//
// In the future we might expand this list and use the more comprehensive list found here:
// http://unicode.org/cldr/trac/browser/tags/release-26-0-1/common/supplemental/postalCodeData.xml
// prettier-ignore
const POSTAL_CODE_REGEXES = [
    /* US */ /*#__PURE__*/ /^\d{5}([-]?\d{4})?$/,
    /* UK */ /*#__PURE__*/ /^(GIR|[A-Z]\d[A-Z\d]??|[A-Z]{2}\d[A-Z\d]??)[ ]??(\d[A-Z]{2})$/,
    /* DE */ /*#__PURE__*/ /\b((?:0[1-46-9]\d{3})|(?:[1-357-9]\d{4})|(?:[4][0-24-9]\d{3})|(?:[6][013-9]\d{3}))\b/,
    /* CA */ /*#__PURE__*/ /^([ABCEGHJKLMNPRSTVXY]\d[ABCEGHJKLMNPRSTVWXYZ]) {0,1}(\d[ABCEGHJKLMNPRSTVWXYZ]\d)$/,
    /* FR */ /*#__PURE__*/ /^(F-)?((2[A|B])|[0-9]{2})[0-9]{3}$/,
    /* IT */ /*#__PURE__*/ /^(V-|I-)?[0-9]{5}$/,
    /* AU */ /*#__PURE__*/ /^(0[289][0-9]{2})|([1345689][0-9]{3})|(2[0-8][0-9]{2})|(290[0-9])|(291[0-4])|(7[0-4][0-9]{2})|(7[8-9][0-9]{2})$/,
    /* NL */ /*#__PURE__*/ /^[1-9][0-9]{3}\s?([a-zA-Z]{2})?$/,
    /* ES */ /*#__PURE__*/ /^([1-9]{2}|[0-9][1-9]|[1-9][0-9])[0-9]{3}$/,
    /* DK */ /*#__PURE__*/ /^([D|d][K|k]( |-))?[1-9]{1}[0-9]{3}$/,
    /* SE */ /*#__PURE__*/ /^(s-|S-){0,1}[0-9]{3}\s?[0-9]{2}$/,
    /* BE */ /*#__PURE__*/ /^[1-9]{1}[0-9]{3}$/,
    /* IN */ /*#__PURE__*/ /^\d{6}$/,
    /* AT */ /*#__PURE__*/ /^\d{4}$/,
    /* PT */ /*#__PURE__*/ /^\d{4}([\-]\d{3})?$/,
    /* CH */ /*#__PURE__*/ /^\d{4}$/,
    /* LU */ /*#__PURE__*/ /^\d{4}$/,
];
function _testPostalCode(postalCode) {
    let result = false;
    // eslint-disable-next-line no-plusplus
    for (let i = 0; i < POSTAL_CODE_REGEXES.length; i++) {
        const regex = POSTAL_CODE_REGEXES[i];
        if (regex.test(postalCode)) {
            result = true;
            break;
        }
    }
    return result;
}
const GraphQLPostalCode = /*#__PURE__*/ new graphql.GraphQLScalarType({
    name: 'PostalCode',
    description: 'A field whose value conforms to the standard postal code formats for United States, United Kingdom, Germany, Canada, France, Italy, Australia, Netherlands, Spain, Denmark, Sweden, Belgium, India, Austria, Portugal, Switzerland or Luxembourg.',
    serialize(value) {
        if (typeof value !== 'string') {
            throw new TypeError(`Value is not string: ${value}`);
        }
        if (!_testPostalCode(value)) {
            throw new TypeError(`Value is not a valid postal code: ${value}`);
        }
        return value;
    },
    parseValue(value) {
        if (typeof value !== 'string') {
            throw new TypeError(`Value is not string: ${value}`);
        }
        if (!_testPostalCode(value)) {
            throw new TypeError(`Value is not a valid postal code: ${value}`);
        }
        return value;
    },
    parseLiteral(ast) {
        if (ast.kind !== graphql.Kind.STRING) {
            throw new graphql.GraphQLError(`Can only validate strings as postal codes but got a: ${ast.kind}`);
        }
        if (!_testPostalCode(ast.value)) {
            throw new TypeError(`Value is not a valid postal code: ${ast.value}`);
        }
        return ast.value;
    },
});

const GraphQLUnsignedFloatConfig = /*#__PURE__*/ Object.assign({}, GraphQLNonNegativeFloatConfig, {
    name: 'UnsignedFloat',
});
const GraphQLUnsignedFloat = /*#__PURE__*/ new graphql.GraphQLScalarType(GraphQLUnsignedFloatConfig);

const GraphQLUnsignedIntConfig = /*#__PURE__*/ Object.assign({}, GraphQLNonNegativeIntConfig, {
    name: 'UnsignedInt',
});
const GraphQLUnsignedInt = /*#__PURE__*/ new graphql.GraphQLScalarType(GraphQLUnsignedIntConfig);

const GraphQLURL = /*#__PURE__*/ new graphql.GraphQLScalarType({
    name: 'URL',
    description: 'A field whose value conforms to the standard URL format as specified in RFC3986: https://www.ietf.org/rfc/rfc3986.txt.',
    serialize(value) {
        if (value === null) {
            return value;
        }
        return new URL(value.toString()).toString();
    },
    parseValue: (value) => (value === null ? value : new URL(value.toString())),
    parseLiteral(ast) {
        if (ast.kind !== graphql.Kind.STRING) {
            throw new graphql.GraphQLError(`Can only validate strings as URLs but got a: ${ast.kind}`);
        }
        if (ast.value === null) {
            return ast.value;
        }
        else {
            return new URL(ast.value.toString());
        }
    },
});

function isBigIntAvailable() {
    return ((typeof global === 'object' && global.BigInt) ||
        (typeof window === 'object' && window.BigInt));
}
function patchBigInt() {
    if (!BigInt.prototype.toJSON) {
        BigInt.prototype.toJSON =
            BigInt.prototype.toJSON ||
                function () {
                    return this.toString();
                };
    }
}
function coerceBigIntValue(value) {
    if (isBigIntAvailable()) {
        patchBigInt();
        return BigInt(value);
    }
    else {
        return Number(value);
    }
}
const GraphQLBigIntConfig = {
    name: 'BigInt',
    description: 'The `BigInt` scalar type represents non-fractional signed whole numeric values.',
    serialize: coerceBigIntValue,
    parseValue: coerceBigIntValue,
    parseLiteral(ast) {
        if (ast.kind === graphql.Kind.INT ||
            ast.kind === graphql.Kind.FLOAT ||
            ast.kind === graphql.Kind.STRING) {
            return coerceBigIntValue(ast.value);
        }
        return null;
    },
};
const GraphQLBigInt = /*#__PURE__*/ new graphql.GraphQLScalarType(GraphQLBigIntConfig);

const base64Validator = /^(?:[A-Za-z0-9+]{4})*(?:[A-Za-z0-9+]{2}==|[A-Za-z0-9+]{3}=)?$/;
function hexValidator(value) {
    // Ensure that any leading 0 is removed from the hex string to avoid false negatives.
    const sanitizedValue = value.charAt(0) === '0' ? value.slice(1) : value;
    // For larger strings, we run into issues with MAX_SAFE_INTEGER, so split the string
    // into smaller pieces to avoid this issue.
    if (value.length > 8) {
        let parsedString = '';
        for (let startIndex = 0, endIndex = 8; startIndex < value.length; startIndex += 8, endIndex += 8) {
            parsedString += parseInt(value.slice(startIndex, endIndex), 16).toString(16);
        }
        return parsedString === sanitizedValue;
    }
    return parseInt(value, 16).toString(16) === sanitizedValue;
}
function validate$3(value) {
    if (typeof value !== 'string' && !(value instanceof global.Buffer)) {
        throw new TypeError(`Value is not an instance of Buffer: ${JSON.stringify(value)}`);
    }
    if (typeof value === 'string') {
        const isBase64 = base64Validator.test(value);
        const isHex = hexValidator(value);
        if (!isBase64 && !isHex) {
            throw new TypeError(`Value is not a valid base64 or hex encoded string: ${JSON.stringify(value)}`);
        }
        return global.Buffer.from(value, isHex ? 'hex' : 'base64');
    }
    return value;
}
function parseObject(ast) {
    const key = ast.fields[0].value;
    const value = ast.fields[1].value;
    if (ast.fields.length === 2 &&
        key.kind === graphql.Kind.STRING &&
        key.value === 'Buffer' &&
        value.kind === graphql.Kind.LIST) {
        return global.Buffer.from(value.values.map((astValue) => parseInt(astValue.value)));
    }
    throw new TypeError(`Value is not a JSON representation of Buffer: ${graphql.print(ast)}`);
}
const GraphQLByteConfig = {
    name: 'Byte',
    description: 'The `Byte` scalar type represents byte value as a Buffer',
    serialize: validate$3,
    parseValue: validate$3,
    parseLiteral(ast) {
        switch (ast.kind) {
            case graphql.Kind.STRING:
                return validate$3(ast.value);
            case graphql.Kind.OBJECT:
                return parseObject(ast);
            default:
                throw new TypeError(`Can only parse base64 or hex encoded strings as Byte, but got a: ${ast.kind}`);
        }
    },
};
const GraphQLByte = /*#__PURE__*/ new graphql.GraphQLScalarType(GraphQLByteConfig);

const GraphQLLongConfig = /*#__PURE__*/ Object.assign({}, GraphQLBigIntConfig, {
    name: 'Long',
});
const GraphQLLong = /*#__PURE__*/ new graphql.GraphQLScalarType(GraphQLLongConfig);

// Based on https://github.com/stems/graphql-bigint/
// eslint-disable-next-line @typescript-eslint/ban-types
function isObjectLike(value) {
    return typeof value === 'object' && value !== null;
}
function serializeObject(outputValue) {
    if (isObjectLike(outputValue)) {
        if (typeof outputValue.valueOf === 'function') {
            const valueOfResult = outputValue.valueOf();
            if (!isObjectLike(valueOfResult)) {
                return valueOfResult;
            }
        }
        if (typeof outputValue.toJSON === 'function') {
            return outputValue.toJSON();
        }
    }
    return outputValue;
}
function serializeSafeIntValue(outputValue) {
    const coercedValue = serializeObject(outputValue);
    if (typeof coercedValue === 'boolean') {
        return coercedValue ? 1 : 0;
    }
    let num = coercedValue;
    if (typeof coercedValue === 'string' && coercedValue !== '') {
        num = Number(coercedValue);
    }
    if (!Number.isSafeInteger(num)) {
        throw new graphql.GraphQLError(`SafeInt cannot represent non-safe-integer value: ${num}`);
    }
    return num;
}
function parseSafeIntValue(inputValue) {
    if (!Number.isSafeInteger(inputValue)) {
        throw new graphql.GraphQLError(`SafeInt cannot represent non-safe-integer value: ${inputValue}`);
    }
    return inputValue;
}
const GraphQLSafeIntConfig = {
    name: 'SafeInt',
    description: 'The `SafeInt` scalar type represents non-fractional signed whole numeric values that are ' +
        'considered safe as defined by the ECMAScript specification.',
    specifiedByUrl: 'https://www.ecma-international.org/ecma-262/#sec-number.issafeinteger',
    serialize: serializeSafeIntValue,
    parseValue: parseSafeIntValue,
    parseLiteral(ast) {
        if (ast.kind !== graphql.Kind.INT) {
            throw new graphql.GraphQLError(`SafeInt cannot represent non-integer value: ${graphql.print(ast)}`, ast);
        }
        const num = parseInt(ast.value, 10);
        return parseSafeIntValue(num);
    },
};
const GraphQLSafeInt = /*#__PURE__*/ new graphql.GraphQLScalarType(GraphQLSafeIntConfig);

const validate$4 = (value) => {
    const UUID_REGEX = /^(\{){0,1}[0-9a-fA-F]{8}\-[0-9a-fA-F]{4}\-[0-9a-fA-F]{4}\-[0-9a-fA-F]{4}\-[0-9a-fA-F]{12}(\}){0,1}$/gi;
    if (typeof value !== 'string') {
        throw new TypeError(`Value is not string: ${value}`);
    }
    if (value.startsWith('{')) {
        value = value.substring(1, value.length - 1);
    }
    if (!UUID_REGEX.test(value)) {
        throw new TypeError(`Value is not a valid UUID: ${value}`);
    }
    return value;
};
const GraphQLUUIDConfig = /*#__PURE__*/ new graphql.GraphQLScalarType({
    name: `UUID`,
    description: `A field whose value is a generic Universally Unique Identifier: https://en.wikipedia.org/wiki/Universally_unique_identifier.`,
    serialize(value) {
        return validate$4(value);
    },
    parseValue(value) {
        return validate$4(value);
    },
    parseLiteral(ast) {
        if (ast.kind !== graphql.Kind.STRING) {
            throw new graphql.GraphQLError(`Can only validate strings as UUIDs but got a: ${ast.kind}`);
        }
        return validate$4(ast.value);
    },
});
const GraphQLUUID = /*#__PURE__*/ new graphql.GraphQLScalarType(GraphQLUUIDConfig);

const GraphQLGUIDConfig = /*#__PURE__*/ Object.assign({}, GraphQLUUIDConfig, {
    name: 'GUID',
});
const GraphQLGUID = /*#__PURE__*/ new graphql.GraphQLScalarType(GraphQLGUIDConfig);

const validate$5 = (value) => {
    const HEXADECIMAL_REGEX = /^[a-f0-9]+$/i;
    if (typeof value !== 'string') {
        throw new TypeError(`Value is not string: ${value}`);
    }
    if (!HEXADECIMAL_REGEX.test(value)) {
        throw new TypeError(`Value is not a valid hexadecimal value: ${value}`);
    }
    return value;
};
const GraphQLHexadecimalConfig = {
    name: `Hexadecimal`,
    description: `A field whose value is a hexadecimal: https://en.wikipedia.org/wiki/Hexadecimal.`,
    serialize(value) {
        return validate$5(value);
    },
    parseValue(value) {
        return validate$5(value);
    },
    parseLiteral(ast) {
        if (ast.kind !== graphql.Kind.STRING) {
            throw new graphql.GraphQLError(`Can only validate strings as a hexadecimal but got a: ${ast.kind}`);
        }
        return validate$5(ast.value);
    },
};
const GraphQLHexadecimal = /*#__PURE__*/ new graphql.GraphQLScalarType(GraphQLHexadecimalConfig);

const HEX_COLOR_CODE = /^#([A-Fa-f0-9]{6}|[A-Fa-f0-9]{3}|[A-Fa-f0-9]{8})$/;
const validate$6 = (value) => {
    if (typeof value !== 'string') {
        throw new TypeError(`Value is not string: ${value}`);
    }
    if (!HEX_COLOR_CODE.test(value)) {
        throw new TypeError(`Value is not a valid HexColorCode: ${value}`);
    }
    return value;
};
const GraphQLHexColorCodeConfig = {
    name: `HexColorCode`,
    description: `A field whose value is a hex color code: https://en.wikipedia.org/wiki/Web_colors.`,
    serialize(value) {
        return validate$6(value);
    },
    parseValue(value) {
        return validate$6(value);
    },
    parseLiteral(ast) {
        if (ast.kind !== graphql.Kind.STRING) {
            throw new graphql.GraphQLError(`Can only validate strings as hex color codes but got a: ${ast.kind}`);
        }
        return validate$6(ast.value);
    },
    specifiedByUrl: 'https://en.wikipedia.org/wiki/Web_colors',
};
const GraphQLHexColorCode = /*#__PURE__*/ new graphql.GraphQLScalarType(GraphQLHexColorCodeConfig);

const validate$7 = (value) => {
    const HSL_REGEX = /^hsl\(\s*(-?\d+|-?\d*.\d+)\s*,\s*(-?\d+|-?\d*.\d+)%\s*,\s*(-?\d+|-?\d*.\d+)%\s*\)$/;
    if (typeof value !== 'string') {
        throw new TypeError(`Value is not string: ${value}`);
    }
    if (!HSL_REGEX.test(value)) {
        throw new TypeError(`Value is not a valid HSL color: ${value}`);
    }
    return value;
};
const GraphQLHSLConfig = {
    name: `HSL`,
    description: `A field whose value is a CSS HSL color: https://developer.mozilla.org/en-US/docs/Web/CSS/color_value#hsl()_and_hsla().`,
    serialize(value) {
        return validate$7(value);
    },
    parseValue(value) {
        return validate$7(value);
    },
    parseLiteral(ast) {
        if (ast.kind !== graphql.Kind.STRING) {
            throw new graphql.GraphQLError(`Can only validate strings as HSL colors but got a: ${ast.kind}`);
        }
        return validate$7(ast.value);
    },
    specifiedByUrl: 'https://developer.mozilla.org/en-US/docs/Web/CSS/color_value#hsl()_and_hsla()',
};
const GraphQLHSL = /*#__PURE__*/ new graphql.GraphQLScalarType(GraphQLHSLConfig);

const HSLA_REGEX = /^hsla\(\s*(-?\d+|-?\d*.\d+)\s*,\s*(-?\d+|-?\d*.\d+)%\s*,\s*(-?\d+|-?\d*.\d+)%\s*,\s*(-?\d+|-?\d*.\d+)\s*\)$/;
const validate$8 = (value) => {
    if (typeof value !== 'string') {
        throw new TypeError(`Value is not string: ${value}`);
    }
    if (!HSLA_REGEX.test(value)) {
        throw new TypeError(`Value is not a valid HSLA color: ${value}`);
    }
    return value;
};
const GraphQLHSLA = /*#__PURE__*/ new graphql.GraphQLScalarType({
    name: `HSLA`,
    description: `A field whose value is a CSS HSLA color: https://developer.mozilla.org/en-US/docs/Web/CSS/color_value#hsl()_and_hsla().`,
    serialize(value) {
        return validate$8(value);
    },
    parseValue(value) {
        return validate$8(value);
    },
    parseLiteral(ast) {
        if (ast.kind !== graphql.Kind.STRING) {
            throw new graphql.GraphQLError(`Can only validate strings as HSLA colors but got a: ${ast.kind}`);
        }
        return validate$8(ast.value);
    },
});

const IPV4_REGEX = /^(?:(?:(?:0?0?[0-9]|0?[1-9][0-9]|1[0-9][0-9]|2[0-4][0-9]|25[0-5])\.){3}(?:0?0?[0-9]|0?[1-9][0-9]|1[0-9][0-9]|2[0-4][0-9]|25[0-5])(?:\/(?:[0-9]|[1-2][0-9]|3[0-2]))?)$/;
const validate$9 = (value) => {
    if (typeof value !== 'string') {
        throw new TypeError(`Value is not string: ${value}`);
    }
    if (!IPV4_REGEX.test(value)) {
        throw new TypeError(`Value is not a valid IPv4 address: ${value}`);
    }
    return value;
};
const GraphQLIPv4 = /*#__PURE__*/ new graphql.GraphQLScalarType({
    name: `IPv4`,
    description: `A field whose value is a IPv4 address: https://en.wikipedia.org/wiki/IPv4.`,
    serialize(value) {
        return validate$9(value);
    },
    parseValue(value) {
        return validate$9(value);
    },
    parseLiteral(ast) {
        if (ast.kind !== graphql.Kind.STRING) {
            throw new graphql.GraphQLError(`Can only validate strings as IPv4 addresses but got a: ${ast.kind}`);
        }
        return validate$9(ast.value);
    },
});

const IPV6_REGEX = /^(?:(?:(?:[0-9A-Fa-f]{1,4}:){6}(?:[0-9A-Fa-f]{1,4}:[0-9A-Fa-f]{1,4}|(?:(?:0?0?[0-9]|0?[1-9][0-9]|1[0-9][0-9]|2[0-4][0-9]|25[0-5])\.){3}(?:0?0?[0-9]|0?[1-9][0-9]|1[0-9][0-9]|2[0-4][0-9]|25[0-5]))|::(?:[0-9A-Fa-f]{1,4}:){5}(?:[0-9A-Fa-f]{1,4}:[0-9A-Fa-f]{1,4}|(?:(?:0?0?[0-9]|0?[1-9][0-9]|1[0-9][0-9]|2[0-4][0-9]|25[0-5])\.){3}(?:0?0?[0-9]|0?[1-9][0-9]|1[0-9][0-9]|2[0-4][0-9]|25[0-5]))|(?:[0-9A-Fa-f]{1,4})?::(?:[0-9A-Fa-f]{1,4}:){4}(?:[0-9A-Fa-f]{1,4}:[0-9A-Fa-f]{1,4}|(?:(?:0?0?[0-9]|0?[1-9][0-9]|1[0-9][0-9]|2[0-4][0-9]|25[0-5])\.){3}(?:0?0?[0-9]|0?[1-9][0-9]|1[0-9][0-9]|2[0-4][0-9]|25[0-5]))|(?:(?:[0-9A-Fa-f]{1,4}:){0,1}[0-9A-Fa-f]{1,4})?::(?:[0-9A-Fa-f]{1,4}:){3}(?:[0-9A-Fa-f]{1,4}:[0-9A-Fa-f]{1,4}|(?:(?:0?0?[0-9]|0?[1-9][0-9]|1[0-9][0-9]|2[0-4][0-9]|25[0-5])\.){3}(?:0?0?[0-9]|0?[1-9][0-9]|1[0-9][0-9]|2[0-4][0-9]|25[0-5]))|(?:(?:[0-9A-Fa-f]{1,4}:){0,2}[0-9A-Fa-f]{1,4})?::(?:[0-9A-Fa-f]{1,4}:){2}(?:[0-9A-Fa-f]{1,4}:[0-9A-Fa-f]{1,4}|(?:(?:0?0?[0-9]|0?[1-9][0-9]|1[0-9][0-9]|2[0-4][0-9]|25[0-5])\.){3}(?:0?0?[0-9]|0?[1-9][0-9]|1[0-9][0-9]|2[0-4][0-9]|25[0-5]))|(?:(?:[0-9A-Fa-f]{1,4}:){0,3}[0-9A-Fa-f]{1,4})?::[0-9A-Fa-f]{1,4}:(?:[0-9A-Fa-f]{1,4}:[0-9A-Fa-f]{1,4}|(?:(?:0?0?[0-9]|0?[1-9][0-9]|1[0-9][0-9]|2[0-4][0-9]|25[0-5])\.){3}(?:0?0?[0-9]|0?[1-9][0-9]|1[0-9][0-9]|2[0-4][0-9]|25[0-5]))|(?:(?:[0-9A-Fa-f]{1,4}:){0,4}[0-9A-Fa-f]{1,4})?::(?:[0-9A-Fa-f]{1,4}:[0-9A-Fa-f]{1,4}|(?:(?:0?0?[0-9]|0?[1-9][0-9]|1[0-9][0-9]|2[0-4][0-9]|25[0-5])\.){3}(?:0?0?[0-9]|0?[1-9][0-9]|1[0-9][0-9]|2[0-4][0-9]|25[0-5]))|(?:(?:[0-9A-Fa-f]{1,4}:){0,5}[0-9A-Fa-f]{1,4})?::[0-9A-Fa-f]{1,4}|(?:(?:[0-9A-Fa-f]{1,4}:){0,6}[0-9A-Fa-f]{1,4})?::)(?:\/(?:0?0?[0-9]|0?[1-9][0-9]|1[01][0-9]|12[0-8]))?)$/;
const validate$a = (value) => {
    if (typeof value !== 'string') {
        throw new TypeError(`Value is not string: ${value}`);
    }
    if (!IPV6_REGEX.test(value)) {
        throw new TypeError(`Value is not a valid IPv6 address: ${value}`);
    }
    return value;
};
const GraphQLIPv6 = /*#__PURE__*/ new graphql.GraphQLScalarType({
    name: `IPv6`,
    description: `A field whose value is a IPv6 address: https://en.wikipedia.org/wiki/IPv6.`,
    serialize(value) {
        return validate$a(value);
    },
    parseValue(value) {
        return validate$a(value);
    },
    parseLiteral(ast) {
        if (ast.kind !== graphql.Kind.STRING) {
            throw new graphql.GraphQLError(`Can only validate strings as IPv6 addresses but got a: ${ast.kind}`);
        }
        return validate$a(ast.value);
    },
});

const ISBN_REGEX_ARR = [
    /^(?:ISBN(?:-10)?:? *)?((?=\d{1,5}([ -]?)\d{1,7}\2?\d{1,6}\2?\d)(?:\d\2*){9}[\dX])$/i,
    /^(?:ISBN(?:-13)?:? *)?(97(?:8|9)([ -]?)(?=\d{1,5}\2?\d{1,7}\2?\d{1,6}\2?\d)(?:\d\2*){9}\d)$/i,
];
const validate$b = (value) => {
    if (typeof value !== 'string') {
        throw new TypeError(`Value is not string: ${value}`);
    }
    let valid = false;
    for (const regex of ISBN_REGEX_ARR) {
        if (regex.test(value)) {
            valid = true;
            break;
        }
    }
    if (!valid) {
        throw new TypeError(`Value is not a valid ISBN number: ${value}`);
    }
    return value;
};
const GraphQLISBN = /*#__PURE__*/ new graphql.GraphQLScalarType({
    name: `ISBN`,
    description: `A field whose value is a ISBN-10 or ISBN-13 number: https://en.wikipedia.org/wiki/International_Standard_Book_Number.`,
    serialize(value) {
        return validate$b(value);
    },
    parseValue(value) {
        return validate$b(value);
    },
    parseLiteral(ast) {
        if (ast.kind !== graphql.Kind.STRING) {
            throw new graphql.GraphQLError(`Can only validate strings as ISBN numbers but got a: ${ast.kind}`);
        }
        return validate$b(ast.value);
    },
});

// See https://github.com/auth0/node-jws/blob/master/lib/verify-stream.js#L8
const JWS_REGEX = /^[a-zA-Z0-9\-_]+?\.[a-zA-Z0-9\-_]+?\.([a-zA-Z0-9\-_]+)?$/;
const validate$c = (value) => {
    if (typeof value !== 'string') {
        throw new TypeError(`Value is not string: ${value}`);
    }
    if (!JWS_REGEX.test(value)) {
        throw new TypeError(`Value is not a valid JWT: ${value}`);
    }
    return value;
};
const GraphQLJWT = /*#__PURE__*/ new graphql.GraphQLScalarType({
    name: `JWT`,
    description: `A field whose value is a JSON Web Token (JWT): https://jwt.io/introduction.`,
    serialize(value) {
        return validate$c(value);
    },
    parseValue(value) {
        return validate$c(value);
    },
    parseLiteral(ast) {
        if (ast.kind !== graphql.Kind.STRING) {
            throw new graphql.GraphQLError(`Can only validate strings as JWT but got a: ${ast.kind}`);
        }
        return validate$c(ast.value);
    },
});

// Inspired by Geolib: https://github.com/manuelbieh/geolib
// Minimum latitude
const MIN_LAT = -90.0;
// Maximum latitude
const MAX_LAT = 90.0;
// See https://en.wikipedia.org/wiki/Decimal_degrees#Precision
const MAX_PRECISION = 8;
const validate$d = (value) => {
    // Check if value is a string or a number
    if ((typeof value !== 'string' && typeof value !== 'number') ||
        value === null ||
        typeof value === 'undefined' ||
        Number.isNaN(value)) {
        throw new TypeError(`Value is neither a number nor a string: ${value}`);
    }
    if (isDecimal(value)) {
        const decimalValue = typeof value === 'string' ? Number.parseFloat(value) : value;
        if (decimalValue < MIN_LAT || decimalValue > MAX_LAT) {
            throw new RangeError(`Value must be between ${MIN_LAT} and ${MAX_LAT}: ${value}`);
        }
        return Number.parseFloat(decimalValue.toFixed(MAX_PRECISION));
    }
    if (isSexagesimal(value)) {
        return validate$d(sexagesimalToDecimal(value));
    }
    throw new TypeError(`Value is not a valid latitude: ${value}`);
};
const GraphQLLatitude = /*#__PURE__*/ new graphql.GraphQLScalarType({
    name: `Latitude`,
    description: `A field whose value is a valid decimal degrees latitude number (53.471): https://en.wikipedia.org/wiki/Latitude`,
    serialize(value) {
        return validate$d(value);
    },
    parseValue(value) {
        return validate$d(value);
    },
    parseLiteral(ast) {
        if (ast.kind !== graphql.Kind.FLOAT && ast.kind !== graphql.Kind.STRING) {
            throw new graphql.GraphQLError(`Can only validate floats or strings as latitude but got a: ${ast.kind}`);
        }
        return validate$d(ast.value);
    },
});

// Inspired by Geolib: https://github.com/manuelbieh/geolib
// Minimum longitude
const MIN_LON = -180.0;
// Maximum longitude
const MAX_LON = 180.0;
// See https://en.wikipedia.org/wiki/Decimal_degrees#Precision
const MAX_PRECISION$1 = 8;
const validate$e = (value) => {
    // Check if value is a string or a number
    if ((typeof value !== 'string' && typeof value !== 'number') ||
        value === null ||
        typeof value === 'undefined' ||
        Number.isNaN(value)) {
        throw new TypeError(`Value is neither a number nor a string: ${value}`);
    }
    if (isDecimal(value)) {
        const decimalValue = typeof value === 'string' ? Number.parseFloat(value) : value;
        if (decimalValue < MIN_LON || decimalValue > MAX_LON) {
            throw new RangeError(`Value must be between ${MIN_LON} and ${MAX_LON}: ${value}`);
        }
        return Number.parseFloat(decimalValue.toFixed(MAX_PRECISION$1));
    }
    if (isSexagesimal(value)) {
        return validate$e(sexagesimalToDecimal(value));
    }
    throw new TypeError(`Value is not a valid longitude: ${value}`);
};
const GraphQLLongitude = /*#__PURE__*/ new graphql.GraphQLScalarType({
    name: `Longitude`,
    description: `A field whose value is a valid decimal degrees longitude number (53.471): https://en.wikipedia.org/wiki/Longitude`,
    serialize(value) {
        return validate$e(value);
    },
    parseValue(value) {
        return validate$e(value);
    },
    parseLiteral(ast) {
        if (ast.kind !== graphql.Kind.FLOAT && ast.kind !== graphql.Kind.STRING) {
            throw new graphql.GraphQLError(`Can only validate floats or strings as longitude but got a: ${ast.kind}`);
        }
        return validate$e(ast.value);
    },
});

const MAC_REGEX = /^(?:[0-9A-Fa-f]{2}([:-]?)[0-9A-Fa-f]{2})(?:(?:\1|\.)(?:[0-9A-Fa-f]{2}([:-]?)[0-9A-Fa-f]{2})){2}$/;
const validate$f = (value) => {
    if (typeof value !== 'string') {
        throw new TypeError(`Value is not string: ${value}`);
    }
    if (!MAC_REGEX.test(value)) {
        throw new TypeError(`Value is not a valid MAC address: ${value}`);
    }
    return value;
};
const GraphQLMAC = /*#__PURE__*/ new graphql.GraphQLScalarType({
    name: `MAC`,
    description: `A field whose value is a IEEE 802 48-bit MAC address: https://en.wikipedia.org/wiki/MAC_address.`,
    serialize(value) {
        return validate$f(value);
    },
    parseValue(value) {
        return validate$f(value);
    },
    parseLiteral(ast) {
        if (ast.kind !== graphql.Kind.STRING) {
            throw new graphql.GraphQLError(`Can only validate strings as MAC addresses but got a: ${ast.kind}`);
        }
        return validate$f(ast.value);
    },
});

const validate$g = (value) => {
    const parsed = typeof value === 'string' ? parseInt(value, 10) : value;
    if (typeof parsed !== 'number' || Number.isNaN(parsed)) {
        throw new TypeError(`Value is not a number: ${value}`);
    }
    if (parsed === Infinity || parsed === -Infinity) {
        throw new TypeError(`Value is not a finite number: ${value}`);
    }
    if (parsed <= 0 || parsed > 65535) {
        throw new TypeError(`Value is not a valid TCP port: ${value}`);
    }
    return parsed;
};
const GraphQLPort = /*#__PURE__*/ new graphql.GraphQLScalarType({
    name: `Port`,
    description: `A field whose value is a valid TCP port within the range of 0 to 65535: https://en.wikipedia.org/wiki/Transmission_Control_Protocol#TCP_ports`,
    serialize(value) {
        return validate$g(value);
    },
    parseValue(value) {
        return validate$g(value);
    },
    parseLiteral(ast) {
        if (ast.kind !== graphql.Kind.INT) {
            throw new graphql.GraphQLError(`Can only validate integers as TCP ports but got a: ${ast.kind}`);
        }
        return validate$g(ast.value);
    },
});

const RGB_REGEX = /^rgb\(\s*(-?\d+|-?\d*\.\d+(?=%))(%?)\s*,\s*(-?\d+|-?\d*\.\d+(?=%))(\2)\s*,\s*(-?\d+|-?\d*\.\d+(?=%))(\2)\s*\)$/;
const validate$h = (value) => {
    if (typeof value !== 'string') {
        throw new TypeError(`Value is not string: ${value}`);
    }
    if (!RGB_REGEX.test(value)) {
        throw new TypeError(`Value is not a valid RGB color: ${value}`);
    }
    return value;
};
const GraphQLRGB = /*#__PURE__*/ new graphql.GraphQLScalarType({
    name: `RGB`,
    description: `A field whose value is a CSS RGB color: https://developer.mozilla.org/en-US/docs/Web/CSS/color_value#rgb()_and_rgba().`,
    serialize(value) {
        return validate$h(value);
    },
    parseValue(value) {
        return validate$h(value);
    },
    parseLiteral(ast) {
        if (ast.kind !== graphql.Kind.STRING) {
            throw new graphql.GraphQLError(`Can only validate strings as RGB colors but got a: ${ast.kind}`);
        }
        return validate$h(ast.value);
    },
});

const RGBA_REGEX = /^rgba\(\s*(-?\d+|-?\d*\.\d+(?=%))(%?)\s*,\s*(-?\d+|-?\d*\.\d+(?=%))(\2)\s*,\s*(-?\d+|-?\d*\.\d+(?=%))(\2)\s*,\s*(-?\d+|-?\d*.\d+)\s*\)$/;
const validate$i = (value) => {
    if (typeof value !== 'string') {
        throw new TypeError(`Value is not string: ${value}`);
    }
    if (!RGBA_REGEX.test(value)) {
        throw new TypeError(`Value is not a valid RGBA color: ${value}`);
    }
    return value;
};
const GraphQLRGBA = /*#__PURE__*/ new graphql.GraphQLScalarType({
    name: `RGBA`,
    description: `A field whose value is a CSS RGBA color: https://developer.mozilla.org/en-US/docs/Web/CSS/color_value#rgb()_and_rgba().`,
    serialize(value) {
        return validate$i(value);
    },
    parseValue(value) {
        return validate$i(value);
    },
    parseLiteral(ast) {
        if (ast.kind !== graphql.Kind.STRING) {
            throw new graphql.GraphQLError(`Can only validate strings as RGBA colors but got a: ${ast.kind}`);
        }
        return validate$i(ast.value);
    },
});

// https://github.com/abhiaiyer91/graphql-currency-scalars
function generateCurrency(value) {
    if (typeof value !== 'number') {
        throw new TypeError(`Currency cannot represent non integer type ${JSON.stringify(value)}`);
    }
    const currencyInCents = parseInt(value.toString(), 10);
    return (currencyInCents / 100).toLocaleString('en-US', {
        style: 'currency',
        currency: 'USD',
    });
}
function generateCents(value) {
    const digits = value.replace('$', '').replace(',', '');
    const number = parseFloat(digits);
    return number * 100;
}
/**
 * An Currency Scalar.
 *
 * Input:
 *    This scalar takes a currency string as input and
 *    formats it to currency in cents.
 *
 * Output:
 *    This scalar serializes currency in cents to
 *    currency strings.
 */
const GraphQLUSCurrency = /*#__PURE__*/ new graphql.GraphQLScalarType({
    name: 'USCurrency',
    description: 'A currency string, such as $21.25',
    serialize: generateCurrency,
    parseValue(value) {
        if (typeof value !== 'string') {
            throw new TypeError(`Currency cannot represent non string type ${JSON.stringify(value)}`);
        }
        return generateCents(value);
    },
    parseLiteral(ast) {
        if (ast.kind === graphql.Kind.STRING) {
            if (typeof ast.value === 'string') {
                return generateCents(ast.value);
            }
        }
        throw new TypeError(`Currency cannot represent an invalid currency-string ${JSON.stringify(ast)}.`);
    },
});

const validate$j = (value) => {
    const CURRENCY_REGEX = /^(AED|AFN|ALL|AMD|ANG|AOA|ARS|AUD|AWG|AZN|BAM|BBD|BDT|BGN|BHD|BIF|BMD|BND|BOB|BOV|BRL|BSD|BTN|BWP|BYN|BZD|CAD|CDF|CHE|CHF|CHW|CLF|CLP|CNY|COP|COU|CRC|CUC|CUP|CVE|CZK|DJF|DKK|DOP|DZD|EGP|ERN|ETB|EUR|FJD|FKP|GBP|GEL|GHS|GIP|GMD|GNF|GTQ|GYD|HKD|HNL|HRK|HTG|HUF|IDR|ILS|INR|IQD|IRR|ISK|JMD|JOD|JPY|KES|KGS|KHR|KMF|KPW|KRW|KWD|KYD|KZT|LAK|LBP|LKR|LRD|LSL|LYD|MAD|MDL|MGA|MKD|MMK|MNT|MOP|MRU|MUR|MVR|MWK|MXN|MXV|MYR|MZN|NAD|NGN|NIO|NOK|NPR|NZD|OMR|PAB|PEN|PGK|PHP|PKR|PLN|PYG|QAR|RON|RSD|RUB|RWF|SAR|SBD|SCR|SDG|SEK|SGD|SHP|SLL|SOS|SRD|SSP|STN|SVC|SYP|SZL|THB|TJS|TMT|TND|TOP|TRY|TTD|TWD|TZS|UAH|UGX|USD|USN|UYI|UYU|UYW|UZS|VES|VND|VUV|WST|XAF|XAG|XAU|XBA|XBB|XBC|XBD|XCD|XDR|XOF|XPD|XPF|XPT|XSU|XTS|XUA|XXX|YER|ZAR|ZMW|ZWL)$/i;
    if (typeof value !== 'string') {
        throw new TypeError(`Value is not string: ${value}`);
    }
    if (!CURRENCY_REGEX.test(value)) {
        throw new TypeError(`Value is not a valid currency value: ${value}`);
    }
    return value;
};
const GraphQLCurrencyConfig$1 = {
    name: `Currency`,
    description: `A field whose value is a Currency: https://en.wikipedia.org/wiki/ISO_4217.`,
    serialize(value) {
        return validate$j(value);
    },
    parseValue(value) {
        return validate$j(value);
    },
    parseLiteral(ast) {
        if (ast.kind !== graphql.Kind.STRING) {
            throw new graphql.GraphQLError(`Can only validate strings as a currency but got a: ${ast.kind}`);
        }
        return validate$j(ast.value);
    },
    specifiedByUrl: 'https://en.wikipedia.org/wiki/ISO_4217',
};
const GraphQLCurrency = /*#__PURE__*/ new graphql.GraphQLScalarType(GraphQLCurrencyConfig$1);

function identity(value) {
    return value;
}
// eslint-disable-next-line @typescript-eslint/ban-types
function ensureObject(value) {
    if (typeof value !== 'object' || value === null || Array.isArray(value)) {
        throw new TypeError(`JSONObject cannot represent non-object value: ${value}`);
    }
    return value;
}
function parseObject$1(ast, variables) {
    const value = Object.create(null);
    ast.fields.forEach((field) => {
        // eslint-disable-next-line no-use-before-define
        value[field.name.value] = parseLiteral(field.value, variables);
    });
    return value;
}
function parseLiteral(ast, variables) {
    switch (ast.kind) {
        case graphql.Kind.STRING:
        case graphql.Kind.BOOLEAN:
            return ast.value;
        case graphql.Kind.INT:
        case graphql.Kind.FLOAT:
            return parseFloat(ast.value);
        case graphql.Kind.OBJECT:
            return parseObject$1(ast, variables);
        case graphql.Kind.LIST:
            return ast.values.map((n) => parseLiteral(n, variables));
        case graphql.Kind.NULL:
            return null;
        case graphql.Kind.VARIABLE: {
            const name = ast.name.value;
            return variables ? variables[name] : undefined;
        }
    }
}

// This named export is intended for users of CommonJS. Users of ES modules
const GraphQLJSONConfig = {
    name: 'JSON',
    description: 'The `JSON` scalar type represents JSON values as specified by [ECMA-404](http://www.ecma-international.org/publications/files/ECMA-ST/ECMA-404.pdf).',
    serialize: identity,
    parseValue: identity,
    parseLiteral,
    specifiedByUrl: 'http://www.ecma-international.org/publications/files/ECMA-ST/ECMA-404.pdf',
};
const GraphQLJSON = /*#__PURE__*/ new graphql.GraphQLScalarType(GraphQLJSONConfig);

/* eslint-disable @typescript-eslint/ban-types */
const GraphQLJSONObjectConfig = {
    name: 'JSONObject',
    description: 'The `JSONObject` scalar type represents JSON objects as specified by [ECMA-404](http://www.ecma-international.org/publications/files/ECMA-ST/ECMA-404.pdf).',
    serialize: ensureObject,
    parseValue: ensureObject,
    parseLiteral: parseObject$1,
    specifiedByUrl: 'http://www.ecma-international.org/publications/files/ECMA-ST/ECMA-404.pdf',
};
const GraphQLJSONObject = /*#__PURE__*/ new graphql.GraphQLScalarType(GraphQLJSONObjectConfig);

// Based on https://github.com/arhs/iban.js
/* These are IBAN the specifications for all countries using IBAN
The key is the countrycode, the second item is the length of the IBAN,
The third item is the structure of the underlying BBAN (for validation and formatting)
*/
const IBAN_SPECIFICATIONS = {
    AD: {
        length: 24,
        structure: 'F04F04A12',
        example: 'AD1200012030200359100100',
    },
    AE: { length: 23, structure: 'F03F16', example: 'AE070331234567890123456' },
    AL: {
        length: 28,
        structure: 'F08A16',
        example: 'AL47212110090000000235698741',
    },
    AO: { length: 25, structure: 'F21', example: 'AO69123456789012345678901' },
    AT: { length: 20, structure: 'F05F11', example: 'AT611904300234573201' },
    AZ: {
        length: 28,
        structure: 'U04A20',
        example: 'AZ21NABZ00000000137010001944',
    },
    BA: {
        length: 20,
        structure: 'F03F03F08F02',
        example: 'BA391290079401028494',
    },
    BE: { length: 16, structure: 'F03F07F02', example: 'BE68539007547034' },
    BF: { length: 27, structure: 'F23', example: 'BF2312345678901234567890123' },
    BG: {
        length: 22,
        structure: 'U04F04F02A08',
        example: 'BG80BNBG96611020345678',
    },
    BH: { length: 22, structure: 'U04A14', example: 'BH67BMAG00001299123456' },
    BI: { length: 16, structure: 'F12', example: 'BI41123456789012' },
    BJ: { length: 28, structure: 'F24', example: 'BJ39123456789012345678901234' },
    BR: {
        length: 29,
        structure: 'F08F05F10U01A01',
        example: 'BR9700360305000010009795493P1',
    },
    BY: {
        length: 28,
        structure: 'A04F04A16',
        example: 'BY13NBRB3600900000002Z00AB00',
    },
    CH: { length: 21, structure: 'F05A12', example: 'CH9300762011623852957' },
    CI: {
        length: 28,
        structure: 'U02F22',
        example: 'CI70CI1234567890123456789012',
    },
    CM: { length: 27, structure: 'F23', example: 'CM9012345678901234567890123' },
    CR: { length: 22, structure: 'F04F14', example: 'CR72012300000171549015' },
    CV: { length: 25, structure: 'F21', example: 'CV30123456789012345678901' },
    CY: {
        length: 28,
        structure: 'F03F05A16',
        example: 'CY17002001280000001200527600',
    },
    CZ: {
        length: 24,
        structure: 'F04F06F10',
        example: 'CZ6508000000192000145399',
    },
    DE: { length: 22, structure: 'F08F10', example: 'DE89370400440532013000' },
    DK: { length: 18, structure: 'F04F09F01', example: 'DK5000400440116243' },
    DO: {
        length: 28,
        structure: 'U04F20',
        example: 'DO28BAGR00000001212453611324',
    },
    DZ: { length: 24, structure: 'F20', example: 'DZ8612345678901234567890' },
    EE: {
        length: 20,
        structure: 'F02F02F11F01',
        example: 'EE382200221020145685',
    },
    ES: {
        length: 24,
        structure: 'F04F04F01F01F10',
        example: 'ES9121000418450200051332',
    },
    FI: { length: 18, structure: 'F06F07F01', example: 'FI2112345600000785' },
    FO: { length: 18, structure: 'F04F09F01', example: 'FO6264600001631634' },
    FR: {
        length: 27,
        structure: 'F05F05A11F02',
        example: 'FR1420041010050500013M02606',
    },
    GB: { length: 22, structure: 'U04F06F08', example: 'GB29NWBK60161331926819' },
    GE: { length: 22, structure: 'U02F16', example: 'GE29NB0000000101904917' },
    GI: { length: 23, structure: 'U04A15', example: 'GI75NWBK000000007099453' },
    GL: { length: 18, structure: 'F04F09F01', example: 'GL8964710001000206' },
    GR: {
        length: 27,
        structure: 'F03F04A16',
        example: 'GR1601101250000000012300695',
    },
    GT: {
        length: 28,
        structure: 'A04A20',
        example: 'GT82TRAJ01020000001210029690',
    },
    HR: { length: 21, structure: 'F07F10', example: 'HR1210010051863000160' },
    HU: {
        length: 28,
        structure: 'F03F04F01F15F01',
        example: 'HU42117730161111101800000000',
    },
    IE: { length: 22, structure: 'U04F06F08', example: 'IE29AIBK93115212345678' },
    IL: {
        length: 23,
        structure: 'F03F03F13',
        example: 'IL620108000000099999999',
    },
    IS: {
        length: 26,
        structure: 'F04F02F06F10',
        example: 'IS140159260076545510730339',
    },
    IT: {
        length: 27,
        structure: 'U01F05F05A12',
        example: 'IT60X0542811101000000123456',
    },
    IQ: {
        length: 23,
        structure: 'U04F03A12',
        example: 'IQ98NBIQ850123456789012',
    },
    IR: { length: 26, structure: 'F22', example: 'IR861234568790123456789012' },
    JO: {
        length: 30,
        structure: 'A04F22',
        example: 'JO15AAAA1234567890123456789012',
    },
    KW: {
        length: 30,
        structure: 'U04A22',
        example: 'KW81CBKU0000000000001234560101',
    },
    KZ: { length: 20, structure: 'F03A13', example: 'KZ86125KZT5004100100' },
    LB: {
        length: 28,
        structure: 'F04A20',
        example: 'LB62099900000001001901229114',
    },
    LC: {
        length: 32,
        structure: 'U04F24',
        example: 'LC07HEMM000100010012001200013015',
    },
    LI: { length: 21, structure: 'F05A12', example: 'LI21088100002324013AA' },
    LT: { length: 20, structure: 'F05F11', example: 'LT121000011101001000' },
    LU: { length: 20, structure: 'F03A13', example: 'LU280019400644750000' },
    LV: { length: 21, structure: 'U04A13', example: 'LV80BANK0000435195001' },
    MC: {
        length: 27,
        structure: 'F05F05A11F02',
        example: 'MC5811222000010123456789030',
    },
    MD: { length: 24, structure: 'U02A18', example: 'MD24AG000225100013104168' },
    ME: { length: 22, structure: 'F03F13F02', example: 'ME25505000012345678951' },
    MG: { length: 27, structure: 'F23', example: 'MG1812345678901234567890123' },
    MK: { length: 19, structure: 'F03A10F02', example: 'MK07250120000058984' },
    ML: {
        length: 28,
        structure: 'U01F23',
        example: 'ML15A12345678901234567890123',
    },
    MR: {
        length: 27,
        structure: 'F05F05F11F02',
        example: 'MR1300020001010000123456753',
    },
    MT: {
        length: 31,
        structure: 'U04F05A18',
        example: 'MT84MALT011000012345MTLCAST001S',
    },
    MU: {
        length: 30,
        structure: 'U04F02F02F12F03U03',
        example: 'MU17BOMM0101101030300200000MUR',
    },
    MZ: { length: 25, structure: 'F21', example: 'MZ25123456789012345678901' },
    NL: { length: 18, structure: 'U04F10', example: 'NL91ABNA0417164300' },
    NO: { length: 15, structure: 'F04F06F01', example: 'NO9386011117947' },
    PK: { length: 24, structure: 'U04A16', example: 'PK36SCBL0000001123456702' },
    PL: {
        length: 28,
        structure: 'F08F16',
        example: 'PL61109010140000071219812874',
    },
    PS: {
        length: 29,
        structure: 'U04A21',
        example: 'PS92PALS000000000400123456702',
    },
    PT: {
        length: 25,
        structure: 'F04F04F11F02',
        example: 'PT50000201231234567890154',
    },
    QA: {
        length: 29,
        structure: 'U04A21',
        example: 'QA30AAAA123456789012345678901',
    },
    RO: { length: 24, structure: 'U04A16', example: 'RO49AAAA1B31007593840000' },
    RS: { length: 22, structure: 'F03F13F02', example: 'RS35260005601001611379' },
    SA: { length: 24, structure: 'F02A18', example: 'SA0380000000608010167519' },
    SC: {
        length: 31,
        structure: 'U04F04F16U03',
        example: 'SC18SSCB11010000000000001497USD',
    },
    SE: {
        length: 24,
        structure: 'F03F16F01',
        example: 'SE4550000000058398257466',
    },
    SI: { length: 19, structure: 'F05F08F02', example: 'SI56263300012039086' },
    SK: {
        length: 24,
        structure: 'F04F06F10',
        example: 'SK3112000000198742637541',
    },
    SM: {
        length: 27,
        structure: 'U01F05F05A12',
        example: 'SM86U0322509800000000270100',
    },
    SN: {
        length: 28,
        structure: 'U01F23',
        example: 'SN52A12345678901234567890123',
    },
    ST: {
        length: 25,
        structure: 'F08F11F02',
        example: 'ST68000100010051845310112',
    },
    SV: {
        length: 28,
        structure: 'U04F20',
        example: 'SV62CENR00000000000000700025',
    },
    TL: {
        length: 23,
        structure: 'F03F14F02',
        example: 'TL380080012345678910157',
    },
    TN: {
        length: 24,
        structure: 'F02F03F13F02',
        example: 'TN5910006035183598478831',
    },
    TR: {
        length: 26,
        structure: 'F05F01A16',
        example: 'TR330006100519786457841326',
    },
    UA: {
        length: 29,
        structure: 'F25',
        example: 'UA511234567890123456789012345',
    },
    VA: { length: 22, structure: 'F18', example: 'VA59001123000012345678' },
    VG: { length: 24, structure: 'U04F16', example: 'VG96VPVG0000012345678901' },
    XK: { length: 20, structure: 'F04F10F02', example: 'XK051212012345678906' },
};
const A = 'A'.charCodeAt(0);
const Z = 'Z'.charCodeAt(0);
function parseStructure(structure) {
    // split in blocks of 3 chars
    const regex = structure.match(/(.{3})/g).map(function (block) {
        // parse each structure block (1-char + 2-digits)
        let format;
        const pattern = block.slice(0, 1);
        const repeats = parseInt(block.slice(1), 10);
        switch (pattern) {
            case 'A':
                format = '0-9A-Za-z';
                break;
            case 'B':
                format = '0-9A-Z';
                break;
            case 'C':
                format = 'A-Za-z';
                break;
            case 'F':
                format = '0-9';
                break;
            case 'L':
                format = 'a-z';
                break;
            case 'U':
                format = 'A-Z';
                break;
            case 'W':
                format = '0-9a-z';
                break;
        }
        return '([' + format + ']{' + repeats + '})';
    });
    return /*#__PURE__*/ new RegExp('^' + regex.join('') + '$');
}
/**
 * Prepare an IBAN for mod 97 computation by moving the first 4 chars to the end and transforming the letters to
 * numbers (A = 10, B = 11, ..., Z = 35), as specified in ISO13616.
 *
 */
function iso13616Prepare(iban) {
    iban = iban.toUpperCase();
    iban = iban.substr(4) + iban.substr(0, 4);
    return iban
        .split('')
        .map(function (n) {
        const code = n.charCodeAt(0);
        if (code >= A && code <= Z) {
            // A = 10, B = 11, ... Z = 35
            return code - A + 10;
        }
        else {
            return n;
        }
    })
        .join('');
}
/**
 * Calculates the MOD 97 10 of the passed IBAN as specified in ISO7064.
 *
 * @param iban
 * @returns {number}
 */
function iso7064Mod97_10(iban) {
    let remainder = iban;
    let block;
    while (remainder.length > 2) {
        block = remainder.slice(0, 9);
        remainder = (parseInt(block, 10) % 97) + remainder.slice(block.length);
    }
    return parseInt(remainder, 10) % 97;
}
function _testIBAN(iban, countryCode, structure) {
    return (structure.length === iban.length &&
        countryCode === iban.slice(0, 2) &&
        parseStructure(structure.structure).test(iban.slice(4)) &&
        iso7064Mod97_10(iso13616Prepare(iban)) === 1);
}
function validate$k(iban) {
    // Make uppercase and remove whitespace for matching
    iban = iban.toUpperCase().replace(/\s+/g, '');
    const countryCode = iban.slice(0, 2);
    const countryStructure = IBAN_SPECIFICATIONS[countryCode];
    return !!countryStructure && _testIBAN(iban, countryCode, countryStructure);
}
const GraphQLIBAN = /*#__PURE__*/ new graphql.GraphQLScalarType({
    name: `IBAN`,
    description: `A field whose value is an International Bank Account Number (IBAN): https://en.wikipedia.org/wiki/International_Bank_Account_Number.`,
    serialize(value) {
        if (typeof value !== 'string') {
            throw new TypeError(`Value is not string: ${value}`);
        }
        if (!validate$k(value)) {
            throw new TypeError(`Value is not a valid IBAN: ${value}`);
        }
        return value;
    },
    parseValue(value) {
        if (typeof value !== 'string') {
            throw new TypeError(`Value is not string: ${value}`);
        }
        if (!validate$k(value)) {
            throw new TypeError(`Value is not a valid IBAN: ${value}`);
        }
        return value;
    },
    parseLiteral(ast) {
        if (ast.kind !== graphql.Kind.STRING) {
            throw new graphql.GraphQLError(`Can only validate strings as IBANs but got a: ${ast.kind}`);
        }
        if (!validate$k(ast.value)) {
            throw new TypeError(`Value is not a valid IBAN: ${ast.value}`);
        }
        return ast.value;
    },
});

const MONGODB_OBJECTID_REGEX = /*#__PURE__*/ /^[A-Fa-f0-9]{24}$/;
const GraphQLObjectID = /*#__PURE__*/ new graphql.GraphQLScalarType({
    name: 'ObjectID',
    description: 'A field whose value conforms with the standard mongodb object ID as described here: https://docs.mongodb.com/manual/reference/method/ObjectId/#ObjectId. Example: 5e5677d71bdc2ae76344968c',
    serialize(value) {
        if (!MONGODB_OBJECTID_REGEX.test(value)) {
            throw new TypeError(`Value is not a valid mongodb object id of form: ${value}`);
        }
        return value;
    },
    parseValue(value) {
        if (!MONGODB_OBJECTID_REGEX.test(value)) {
            throw new TypeError(`Value is not a valid mongodb object id of form: ${value}`);
        }
        return value;
    },
    parseLiteral(ast) {
        if (ast.kind !== graphql.Kind.STRING) {
            throw new graphql.GraphQLError(`Can only validate strings as mongodb object id but got a: ${ast.kind}`);
        }
        if (!MONGODB_OBJECTID_REGEX.test(ast.value)) {
            throw new TypeError(`Value is not a valid mongodb object id of form: ${ast.value}`);
        }
        return ast.value;
    },
});

const GraphQLVoid = /*#__PURE__*/ new graphql.GraphQLScalarType({
    name: 'Void',
    description: 'Represents NULL values',
    serialize() {
        return null;
    },
    parseValue() {
        return null;
    },
    parseLiteral() {
        return null;
    },
});

const BigInt$1 = 'scalar BigInt';
const Byte = 'scalar Byte';
const Date$1 = 'scalar Date';
const Time$1 = 'scalar Time';
const Timestamp$1 = 'scalar Timestamp';
const DateTime$1 = 'scalar DateTime';
const UtcOffset$1 = 'scalar UtcOffset';
const Duration$1 = 'scalar Duration';
const ISO8601Duration = 'scalar ISO8601Duration';
const LocalDate$1 = 'scalar LocalDate';
const LocalTime$1 = 'scalar LocalTime';
const LocalEndTime$1 = 'scalar LocalEndTime';
const EmailAddress$1 = 'scalar EmailAddress';
const UUID$1 = `scalar UUID`;
const Hexadecimal$1 = `scalar Hexadecimal`;
const HexColorCode$1 = `scalar HexColorCode`;
const HSL$1 = `scalar HSL`;
const HSLA$1 = `scalar HSLA`;
const IBAN$1 = `scalar IBAN`;
const IPv4$1 = `scalar IPv4`;
const IPv6$1 = `scalar IPv6`;
const ISBN$1 = `scalar ISBN`;
const JWT$1 = `scalar JWT`;
const Latitude$1 = `scalar Latitude`;
const Longitude$1 = `scalar Longitude`;
const JSON$2 = `scalar JSON`;
const JSONObject$1 = `scalar JSONObject`;
const MAC$1 = `scalar MAC`;
const NegativeFloat$1 = 'scalar NegativeFloat';
const NegativeInt$1 = 'scalar NegativeInt';
const NonEmptyString$1 = 'scalar NonEmptyString';
const NonNegativeFloat$1 = 'scalar NonNegativeFloat';
const NonNegativeInt$1 = 'scalar NonNegativeInt';
const NonPositiveFloat$1 = 'scalar NonPositiveFloat';
const NonPositiveInt$1 = 'scalar NonPositiveInt';
const PhoneNumber$1 = 'scalar PhoneNumber';
const Port$1 = `scalar Port`;
const PositiveFloat$1 = 'scalar PositiveFloat';
const PositiveInt$1 = 'scalar PositiveInt';
const PostalCode$1 = 'scalar PostalCode';
const RGB$1 = `scalar RGB`;
const RGBA$1 = `scalar RGBA`;
const SafeInt$1 = `scalar SafeInt`;
const URL$1 = 'scalar URL';
const USCurrency$1 = `scalar USCurrency`;
const Currency$1 = `scalar Currency`;
const UnsignedFloat = 'scalar UnsignedFloat';
const UnsignedInt = 'scalar UnsignedInt';
const GUID = `scalar GUID`;
const Long = 'scalar Long';
const ObjectID$1 = 'scalar ObjectID';
const Void$1 = 'scalar Void';
const typeDefs = [
    Date$1,
    Time$1,
    DateTime$1,
    Timestamp$1,
    UtcOffset$1,
    Duration$1,
    ISO8601Duration,
    LocalDate$1,
    LocalTime$1,
    LocalEndTime$1,
    EmailAddress$1,
    NegativeFloat$1,
    NegativeInt$1,
    NonEmptyString$1,
    NonNegativeFloat$1,
    NonNegativeInt$1,
    NonPositiveFloat$1,
    NonPositiveInt$1,
    PhoneNumber$1,
    PositiveFloat$1,
    PositiveInt$1,
    PostalCode$1,
    UnsignedFloat,
    UnsignedInt,
    URL$1,
    BigInt$1,
    Long,
    Byte,
    UUID$1,
    GUID,
    Hexadecimal$1,
    HexColorCode$1,
    HSL$1,
    HSLA$1,
    IPv4$1,
    IPv6$1,
    ISBN$1,
    JWT$1,
    Latitude$1,
    Longitude$1,
    MAC$1,
    Port$1,
    RGB$1,
    RGBA$1,
    SafeInt$1,
    USCurrency$1,
    Currency$1,
    JSON$2,
    JSONObject$1,
    IBAN$1,
    ObjectID$1,
    Void$1,
];

class RegularExpression extends graphql.GraphQLScalarType {
    constructor(name, regex, options = {}) {
        const REGEX = /*#__PURE__*/ regex;
        const errorMessage = options.errorMessage
            ? options.errorMessage
            : (r, v) => `Value does not match the regular expression ${r}: ${v}`;
        super({
            name,
            description: options.description ||
                `A field whose value matches the provided regular expression ${regex}.`,
            serialize(value) {
                if (!REGEX.test(value === null || value === void 0 ? void 0 : value.toString())) {
                    throw new TypeError(errorMessage(regex, value));
                }
                return value;
            },
            parseValue(value) {
                if (!REGEX.test(value === null || value === void 0 ? void 0 : value.toString())) {
                    throw new TypeError(errorMessage(regex, value));
                }
                return value;
            },
            parseLiteral(ast) {
                if (!('value' in ast)) {
                    throw new graphql.GraphQLError(`Can only validate primitive values as regular expressions but got a: ${ast.kind}`);
                }
                if (!REGEX.test(ast.value.toString())) {
                    throw new TypeError(errorMessage(regex, ast.value));
                }
                return ast.value;
            },
        });
    }
}

const resolvers = {
    Date: GraphQLDate,
    Time: GraphQLTime,
    DateTime: GraphQLDateTime,
    Timestamp: GraphQLTimestamp,
    UtcOffset: GraphQLUtcOffset,
    Duration: GraphQLDuration,
    ISO8601Duration: GraphQLISO8601Duration,
    LocalDate: GraphQLLocalDate,
    LocalTime: GraphQLLocalTime,
    LocalEndTime: GraphQLLocalEndTime,
    EmailAddress: GraphQLEmailAddress,
    NegativeFloat: GraphQLNegativeFloat,
    NegativeInt: GraphQLNegativeInt,
    NonEmptyString: GraphQLNonEmptyString,
    NonNegativeFloat: GraphQLNonNegativeFloat,
    NonNegativeInt: GraphQLNonNegativeInt,
    NonPositiveFloat: GraphQLNonPositiveFloat,
    NonPositiveInt: GraphQLNonPositiveInt,
    PhoneNumber: GraphQLPhoneNumber,
    PositiveFloat: GraphQLPositiveFloat,
    PositiveInt: GraphQLPositiveInt,
    PostalCode: GraphQLPostalCode,
    UnsignedFloat: GraphQLUnsignedFloat,
    UnsignedInt: GraphQLUnsignedInt,
    URL: GraphQLURL,
    BigInt: GraphQLBigInt,
    Byte: GraphQLByte,
    Long: GraphQLLong,
    SafeInt: GraphQLSafeInt,
    UUID: GraphQLUUID,
    GUID: GraphQLGUID,
    Hexadecimal: GraphQLHexadecimal,
    HexColorCode: GraphQLHexColorCode,
    HSL: GraphQLHSL,
    HSLA: GraphQLHSLA,
    IPv4: GraphQLIPv4,
    IPv6: GraphQLIPv6,
    ISBN: GraphQLISBN,
    JWT: GraphQLJWT,
    Latitude: GraphQLLatitude,
    Longitude: GraphQLLongitude,
    MAC: GraphQLMAC,
    Port: GraphQLPort,
    RGB: GraphQLRGB,
    RGBA: GraphQLRGBA,
    USCurrency: GraphQLUSCurrency,
    Currency: GraphQLCurrency,
    JSON: GraphQLJSON,
    JSONObject: GraphQLJSONObject,
    IBAN: GraphQLIBAN,
    ObjectID: GraphQLObjectID,
    Void: GraphQLVoid,
};

exports.BigIntMock = BigIntMock;
exports.BigIntResolver = GraphQLBigInt;
exports.BigIntTypeDefinition = BigInt$1;
exports.ByteMock = ByteMock;
exports.ByteResolver = GraphQLByte;
exports.ByteTypeDefinition = Byte;
exports.CurrencyDefinition = Currency$1;
exports.CurrencyMock = Currency;
exports.CurrencyResolver = GraphQLCurrency;
exports.DateMock = DateMock;
exports.DateResolver = GraphQLDate;
exports.DateTimeMock = DateTime;
exports.DateTimeResolver = GraphQLDateTime;
exports.DateTimeTypeDefinition = DateTime$1;
exports.DateTypeDefinition = Date$1;
exports.DurationMock = Duration;
exports.DurationResolver = GraphQLDuration;
exports.EmailAddressMock = EmailAddress;
exports.EmailAddressResolver = GraphQLEmailAddress;
exports.EmailAddressTypeDefinition = EmailAddress$1;
exports.GUIDDefinition = GUID;
exports.GUIDMock = UUID;
exports.GUIDResolver = GraphQLGUID;
exports.GraphQLBigInt = GraphQLBigInt;
exports.GraphQLByte = GraphQLByte;
exports.GraphQLCurrency = GraphQLCurrency;
exports.GraphQLDate = GraphQLDate;
exports.GraphQLDateTime = GraphQLDateTime;
exports.GraphQLDuration = GraphQLDuration;
exports.GraphQLEmailAddress = GraphQLEmailAddress;
exports.GraphQLGUID = GraphQLGUID;
exports.GraphQLHSL = GraphQLHSL;
exports.GraphQLHSLA = GraphQLHSLA;
exports.GraphQLHexColorCode = GraphQLHexColorCode;
exports.GraphQLHexadecimal = GraphQLHexadecimal;
exports.GraphQLIBAN = GraphQLIBAN;
exports.GraphQLIPv4 = GraphQLIPv4;
exports.GraphQLIPv6 = GraphQLIPv6;
exports.GraphQLISBN = GraphQLISBN;
exports.GraphQLISO8601Duration = GraphQLISO8601Duration;
exports.GraphQLJSON = GraphQLJSON;
exports.GraphQLJSONObject = GraphQLJSONObject;
exports.GraphQLJWT = GraphQLJWT;
exports.GraphQLLatitude = GraphQLLatitude;
exports.GraphQLLocalDate = GraphQLLocalDate;
exports.GraphQLLocalEndTime = GraphQLLocalEndTime;
exports.GraphQLLocalTime = GraphQLLocalTime;
exports.GraphQLLong = GraphQLLong;
exports.GraphQLLongitude = GraphQLLongitude;
exports.GraphQLMAC = GraphQLMAC;
exports.GraphQLNegativeFloat = GraphQLNegativeFloat;
exports.GraphQLNegativeInt = GraphQLNegativeInt;
exports.GraphQLNonEmptyString = GraphQLNonEmptyString;
exports.GraphQLNonNegativeFloat = GraphQLNonNegativeFloat;
exports.GraphQLNonNegativeInt = GraphQLNonNegativeInt;
exports.GraphQLNonPositiveFloat = GraphQLNonPositiveFloat;
exports.GraphQLNonPositiveInt = GraphQLNonPositiveInt;
exports.GraphQLObjectID = GraphQLObjectID;
exports.GraphQLPhoneNumber = GraphQLPhoneNumber;
exports.GraphQLPort = GraphQLPort;
exports.GraphQLPositiveFloat = GraphQLPositiveFloat;
exports.GraphQLPositiveInt = GraphQLPositiveInt;
exports.GraphQLPostalCode = GraphQLPostalCode;
exports.GraphQLRGB = GraphQLRGB;
exports.GraphQLRGBA = GraphQLRGBA;
exports.GraphQLSafeInt = GraphQLSafeInt;
exports.GraphQLTime = GraphQLTime;
exports.GraphQLTimestamp = GraphQLTimestamp;
exports.GraphQLURL = GraphQLURL;
exports.GraphQLUSCurrency = GraphQLUSCurrency;
exports.GraphQLUUID = GraphQLUUID;
exports.GraphQLUnsignedFloat = GraphQLUnsignedFloat;
exports.GraphQLUnsignedInt = GraphQLUnsignedInt;
exports.GraphQLUtcOffset = GraphQLUtcOffset;
exports.GraphQLVoid = GraphQLVoid;
exports.HSLADefinition = HSLA$1;
exports.HSLAMock = HSLA;
exports.HSLAResolver = GraphQLHSLA;
exports.HSLDefinition = HSL$1;
exports.HSLMock = HSL;
exports.HSLResolver = GraphQLHSL;
exports.HexColorCodeDefinition = HexColorCode$1;
exports.HexColorCodeMock = HexColorCode;
exports.HexColorCodeResolver = GraphQLHexColorCode;
exports.HexadecimalMock = Hexadecimal;
exports.HexadecimalResolver = GraphQLHexadecimal;
exports.HexadecimalTypeDefinition = Hexadecimal$1;
exports.IBANMock = IBAN;
exports.IBANResolver = GraphQLIBAN;
exports.IBANTypeDefinition = IBAN$1;
exports.IPv4Definition = IPv4$1;
exports.IPv4Mock = IPv4;
exports.IPv4Resolver = GraphQLIPv4;
exports.IPv6Definition = IPv6$1;
exports.IPv6Mock = IPv6;
exports.IPv6Resolver = GraphQLIPv6;
exports.ISBNDefinition = ISBN$1;
exports.ISBNMock = ISBN;
exports.ISBNResolver = GraphQLISBN;
exports.ISO8601DurationMock = Duration;
exports.ISO8601DurationResolver = GraphQLISO8601Duration;
exports.JSONDefinition = JSON$2;
exports.JSONMock = JSON$1;
exports.JSONObjectDefinition = JSONObject$1;
exports.JSONObjectMock = JSONObject;
exports.JSONObjectResolver = GraphQLJSONObject;
exports.JSONResolver = GraphQLJSON;
exports.JWTDefinition = JWT$1;
exports.JWTMock = JWT;
exports.JWTResolver = GraphQLJWT;
exports.LatitudeDefinition = Latitude$1;
exports.LatitudeMock = Latitude;
exports.LatitudeResolver = GraphQLLatitude;
exports.LocalDateMock = LocalDate;
exports.LocalDateResolver = GraphQLLocalDate;
exports.LocalDateTypeDefinition = LocalDate$1;
exports.LocalEndTimeMock = LocalEndTime;
exports.LocalEndTimeResolver = GraphQLLocalEndTime;
exports.LocalEndTimeTypeDefinition = LocalEndTime$1;
exports.LocalTimeMock = LocalTime;
exports.LocalTimeResolver = GraphQLLocalTime;
exports.LocalTimeTypeDefinition = LocalTime$1;
exports.LongMock = BigIntMock;
exports.LongResolver = GraphQLLong;
exports.LongTypeDefinition = Long;
exports.LongitudeDefinition = Longitude$1;
exports.LongitudeMock = Longitude;
exports.LongitudeResolver = GraphQLLongitude;
exports.MACDefinition = MAC$1;
exports.MACMock = MAC;
exports.MACResolver = GraphQLMAC;
exports.NegativeFloatMock = NegativeFloat;
exports.NegativeFloatResolver = GraphQLNegativeFloat;
exports.NegativeFloatTypeDefinition = NegativeFloat$1;
exports.NegativeIntMock = NegativeInt;
exports.NegativeIntResolver = GraphQLNegativeInt;
exports.NegativeIntTypeDefinition = NegativeInt$1;
exports.NonEmptyStringMock = NonEmptyString;
exports.NonEmptyStringResolver = GraphQLNonEmptyString;
exports.NonEmptyStringTypeDefinition = NonEmptyString$1;
exports.NonNegativeFloatMock = NonNegativeFloat;
exports.NonNegativeFloatResolver = GraphQLNonNegativeFloat;
exports.NonNegativeFloatTypeDefinition = NonNegativeFloat$1;
exports.NonNegativeIntMock = NonNegativeInt;
exports.NonNegativeIntResolver = GraphQLNonNegativeInt;
exports.NonNegativeIntTypeDefinition = NonNegativeInt$1;
exports.NonPositiveFloatMock = NonPositiveFloat;
exports.NonPositiveFloatResolver = GraphQLNonPositiveFloat;
exports.NonPositiveFloatTypeDefinition = NonPositiveFloat$1;
exports.NonPositiveIntMock = NonPositiveInt;
exports.NonPositiveIntResolver = GraphQLNonPositiveInt;
exports.NonPositiveIntTypeDefinition = NonPositiveInt$1;
exports.ObjectIDMock = ObjectID;
exports.ObjectIDResolver = GraphQLObjectID;
exports.ObjectIDTypeDefinition = ObjectID$1;
exports.PhoneNumberMock = PhoneNumber;
exports.PhoneNumberResolver = GraphQLPhoneNumber;
exports.PhoneNumberTypeDefinition = PhoneNumber$1;
exports.PortDefinition = Port$1;
exports.PortMock = Port;
exports.PortResolver = GraphQLPort;
exports.PositiveFloatMock = PositiveFloat;
exports.PositiveFloatResolver = GraphQLPositiveFloat;
exports.PositiveFloatTypeDefinition = PositiveFloat$1;
exports.PositiveIntMock = PositiveInt;
exports.PositiveIntResolver = GraphQLPositiveInt;
exports.PositiveIntTypeDefinition = PositiveInt$1;
exports.PostalCodeMock = PostalCode;
exports.PostalCodeResolver = GraphQLPostalCode;
exports.PostalCodeTypeDefinition = PostalCode$1;
exports.RGBADefinition = RGBA$1;
exports.RGBAMock = RGBA;
exports.RGBAResolver = GraphQLRGBA;
exports.RGBDefinition = RGB$1;
exports.RGBMock = RGB;
exports.RGBResolver = GraphQLRGB;
exports.RegularExpression = RegularExpression;
exports.SafeIntDefinition = SafeInt$1;
exports.SafeIntMock = SafeInt;
exports.SafeIntResolver = GraphQLSafeInt;
exports.TimeMock = Time;
exports.TimeResolver = GraphQLTime;
exports.TimeTypeDefinition = Time$1;
exports.TimestampMock = Timestamp;
exports.TimestampResolver = GraphQLTimestamp;
exports.TimestampTypeDefinition = Timestamp$1;
exports.URLMock = URLMock;
exports.URLResolver = GraphQLURL;
exports.URLTypeDefinition = URL$1;
exports.USCurrencyDefinition = USCurrency$1;
exports.USCurrencyMock = USCurrency;
exports.USCurrencyResolver = GraphQLUSCurrency;
exports.UUIDDefinition = UUID$1;
exports.UUIDMock = UUID;
exports.UUIDResolver = GraphQLUUID;
exports.UnsignedFloatMock = NonNegativeFloat;
exports.UnsignedFloatResolver = GraphQLUnsignedFloat;
exports.UnsignedFloatTypeDefinition = UnsignedFloat;
exports.UnsignedIntMock = NonNegativeInt;
exports.UnsignedIntResolver = GraphQLUnsignedInt;
exports.UnsignedIntTypeDefinition = UnsignedInt;
exports.UtcOffsetMock = UtcOffset;
exports.UtcOffsetResolver = GraphQLUtcOffset;
exports.UtcOffsetTypeDefinition = UtcOffset$1;
exports.VoidMock = Void;
exports.VoidResolver = GraphQLVoid;
exports.VoidTypeDefinition = Void$1;
exports.mocks = mocks;
exports.resolvers = resolvers;
exports.typeDefs = typeDefs;
//# sourceMappingURL=index.js.map
