use std::{
    fs::{self, File, OpenOptions},
    io::{self, BufRead, BufReader, Write},
};

use lazy_static::lazy_static;
use risc0_zkp::core::digest::Digest;

use crate::{PROJECT_ROOT, data_layout, decode_hex_id, path_from_env};

data_layout!(Layout {
    project_root: PROJECT_ROOT.into(),
    out_dir: path_from_env("OUT_DIR").unwrap(),
} {
    (project_root / "rust/guest_wrapper/artifacts/chain_guest") => chain_artifacts_dir,
    (chain_artifacts_dir / "elf_id") => chain_guest_id,
    (chain_artifacts_dir / "elf_id_history") => chain_guest_history,
    (chain_artifacts_dir / "CHANGELOG.md") => chain_guest_changelog,
    (out_dir / "guest_id.rs") => rust_guest_id,
});

lazy_static! {
    static ref LAYOUT: Layout = Layout::new().unwrap();
}

fn current_id_hex() -> anyhow::Result<String> {
    let path = LAYOUT.chain_guest_id();
    println!("cargo::rerun-if-changed={}", path.display());
    let file = File::open(path)?;
    let reader = BufReader::new(&file);
    reader
        .lines()
        .next()
        .transpose()?
        .ok_or(anyhow::anyhow!("ID file is empty"))
}

fn id_history_hex() -> anyhow::Result<Vec<String>> {
    let path = LAYOUT.chain_guest_history();
    println!("cargo::rerun-if-changed={}", path.display());
    let file = File::open(path)?;
    let reader = BufReader::new(&file);
    Ok(reader.lines().collect::<io::Result<_>>()?)
}

/// Assert that `generated_id` is identical to the chain guest ID stored in repo.
pub fn assert(generated_id: Digest) -> anyhow::Result<()> {
    let current_id = current_id_hex()?;
    let generated_id = hex::encode(generated_id.as_bytes());
    anyhow::ensure!(
        generated_id == current_id,
        "Chain guest ELF ID mismatch ({generated_id} != {current_id}).\nRun with `UPDATE_GUEST_ELF_ID=1` to update."
    );
    Ok(())
}

/// Append current chain guest ID to the history file.
pub fn add_current_to_history() -> anyhow::Result<()> {
    let current_id = current_id_hex()?;

    if id_history_hex()?.iter().any(|id| id == &current_id) {
        return Ok(());
    }

    let history_file = OpenOptions::new()
        .create(true)
        .append(true)
        .open(LAYOUT.chain_guest_history())?;

    println!("cargo::warning=Adding current chain guest ID to history: {current_id}");
    writeln!(&history_file, "{current_id}")?;
    Ok(())
}

/// Update the chain guest ID stored in repo to `new_id`.
/// Generate a "TODO" changelog entry.
/// `add_current_to_history` should be called before when ensuring previous id is in history.
pub fn update(new_id: Digest, ensure_previous_in_history: bool) -> anyhow::Result<()> {
    let old_hex_id = current_id_hex()?;
    let new_hex_id = hex::encode(new_id.as_bytes());

    println!("cargo::warning=Updating chain guest ID. old={old_hex_id} new={new_hex_id}");

    if old_hex_id == new_hex_id {
        return Ok(());
    }
    if ensure_previous_in_history {
        anyhow::ensure!(
            id_history_hex()?.iter().any(|id| id == &old_hex_id),
            "Previous chain guest ID ({old_hex_id}) not in history"
        );
    }

    fs::write(LAYOUT.chain_guest_id(), &new_hex_id)?;

    let changelog_file = OpenOptions::new()
        .create(true)
        .append(true)
        .open(LAYOUT.chain_guest_changelog())?;
    writeln!(&changelog_file, "  * `{new_hex_id}` – TODO")?;

    Ok(())
}

/// Generate .rs file with current (optional) & historical chain guest IDs as const byte arrays.
pub fn generate_rust(include_current_id: bool) -> anyhow::Result<()> {
    let current_id = decode_hex_id(current_id_hex()?)?;
    let history = id_history_hex()?
        .into_iter()
        .map(decode_hex_id)
        .collect::<anyhow::Result<Vec<Digest>>>()?;

    let file = File::create(LAYOUT.rust_guest_id())?;
    writeln!(
        &file,
        "// Chain guest ELF IDs. Auto-generated by build script. Current ID is the last one.\n"
    )?;
    let num_ids = history.len() + if include_current_id { 1 } else { 0 };
    writeln!(&file, "pub const CHAIN_GUEST_IDS: [[u8; 32]; {}] = [", num_ids)?;
    for old_id in history {
        writeln!(&file, "    {:?},", old_id.as_bytes())?;
    }
    if include_current_id {
        writeln!(&file, "    {:?},", current_id.as_bytes())?;
    }
    writeln!(&file, "];")?;

    Ok(())
}
