use alloy_primitives::Bytes;
use alloy_sol_types::SolValue;
use call_precompiles::json::{get_bool, get_int, get_string};

const STRING_KEY: &str = "key1";
const INT_KEY: &str = "key2";
const BOOL_KEY: &str = "key3";

use crate::Benchmark;

/// embed the files generated by build.rs
macro_rules! include_generated_json {
    ($name:ident, $file:literal) => {
        pub const $name: &str = include_str!(concat!(env!("OUT_DIR"), "/", $file));
    };
}

include_generated_json!(JSON_100B, "100b.json");
include_generated_json!(JSON_1KB, "1kb.json");
include_generated_json!(JSON_10KB, "10kb.json");
include_generated_json!(JSON_100KB, "100kb.json");
include_generated_json!(JSON_1_LVL_10K, "1_level_10kb.json");
include_generated_json!(JSON_10_LVL_10K, "10_level_10kb.json");
include_generated_json!(JSON_100_LVL_10K, "100_level_10kb.json");

lazy_static::lazy_static! {
    static ref LVL_1_KEY: String = create_nested_key_path(1, STRING_KEY);
    static ref LVL_10_KEY: String = create_nested_key_path(10, STRING_KEY);
    static ref LVL_100_KEY: String = create_nested_key_path(100, STRING_KEY);
}

fn create_nested_key_path(depth: usize, key_name: &str) -> String {
    let mut path = String::new();
    for i in 1..=depth {
        path.push_str(&format!("level{i}"));
        path.push('.');
    }
    path.push_str(key_name);
    path
}

fn benchmark_get_string(json_body: &str, path: &str) {
    let calldata: Bytes = [json_body, path].abi_encode().into();
    get_string(&calldata).expect("get_string failed");
}

fn benchmark_get_int(json_body: &str, path: &str) {
    let calldata: Bytes = [json_body, path].abi_encode().into();
    get_int(&calldata).expect("get_int failed");
}

fn benchmark_get_bool(json_body: &str, path: &str) {
    let calldata: Bytes = [json_body, path].abi_encode().into();
    get_bool(&calldata).expect("get_bool failed");
}

pub fn benchmarks() -> Vec<Benchmark> {
    vec![
        Benchmark::new(
            "json_get_string_100b",
            || benchmark_get_string(JSON_100B, STRING_KEY),
            42_000,
        ),
        Benchmark::new(
            "json_get_string_1kb",
            || benchmark_get_string(JSON_1KB, STRING_KEY),
            256_000,
        ),
        Benchmark::new(
            "json_get_string_10kb",
            || benchmark_get_string(JSON_10KB, STRING_KEY),
            3_151_000,
        ),
        Benchmark::new(
            "json_get_string_100kb",
            || benchmark_get_string(JSON_100KB, STRING_KEY),
            37_502_000,
        ),
        Benchmark::new(
            "json_get_string_1_lvl_10k",
            || benchmark_get_string(JSON_1_LVL_10K, &LVL_1_KEY),
            3_153_000,
        ),
        Benchmark::new(
            "json_get_string_10_lvl_10k",
            || benchmark_get_string(JSON_10_LVL_10K, &LVL_10_KEY),
            3_150_000,
        ),
        Benchmark::new(
            "json_get_string_100_lvl_10k",
            || benchmark_get_string(JSON_100_LVL_10K, &LVL_100_KEY),
            3_134_000,
        ),
        Benchmark::new("json_get_int_10kb", || benchmark_get_int(JSON_10KB, INT_KEY), 3_151_000),
        Benchmark::new("json_get_bool_10kb", || benchmark_get_bool(JSON_10KB, BOOL_KEY), 3_151_000),
    ]
}
