use alloy_primitives::Bytes;
use alloy_sol_types::SolValue;
use call_precompiles::regex::{capture, is_match};

use crate::Benchmark;

/// embed the files generated by build.rs
macro_rules! include_generated_text {
    ($name:ident, $file:literal) => {
        pub const $name: &str = include_str!(concat!(env!("OUT_DIR"), "/", $file));
    };
}

include_generated_text!(TEXT_1KB, "1kb.txt");
include_generated_text!(TEXT_10KB, "10kb.txt");
include_generated_text!(TEXT_100KB, "100kb.txt");

// Regex patterns for benchmarking different complexity levels
const SIMPLE_PATTERN: &str = "^.*needle.*$"; // Simple word matching
const COMPLEX_PATTERN: &str = "^.*\\d{3}-\\d{2}-\\d{4}.*$"; // Complex number pattern with character classes

fn benchmark_is_match(text: &str, pattern: &str) {
    let calldata: Bytes = [text, pattern].abi_encode().into();
    is_match(&calldata).expect("is_match failed");
}

fn benchmark_capture(text: &str, pattern: &str) {
    let calldata: Bytes = [text, pattern].abi_encode().into();
    capture(&calldata).expect("capture failed");
}

pub fn benchmarks() -> Vec<Benchmark> {
    vec![
        Benchmark::new(
            "regex_is_match_simple_1kb",
            || benchmark_is_match(TEXT_1KB, SIMPLE_PATTERN),
            480_000,
        ),
        Benchmark::new(
            "regex_is_match_simple_10kb",
            || benchmark_is_match(TEXT_10KB, SIMPLE_PATTERN),
            735_000,
        ),
        Benchmark::new(
            "regex_is_match_simple_100kb",
            || benchmark_is_match(TEXT_100KB, SIMPLE_PATTERN),
            3_280_000,
        ),
        Benchmark::new(
            "regex_is_match_complex_1kb",
            || benchmark_is_match(TEXT_1KB, COMPLEX_PATTERN),
            2_631_000,
        ),
        Benchmark::new(
            "regex_is_match_complex_10kb",
            || benchmark_is_match(TEXT_10KB, COMPLEX_PATTERN),
            2_885_000,
        ),
        Benchmark::new(
            "regex_is_match_complex_100kb",
            || benchmark_is_match(TEXT_100KB, COMPLEX_PATTERN),
            5_431_000,
        ),
        Benchmark::new(
            "regex_capture_simple_1kb",
            || benchmark_capture(TEXT_1KB, SIMPLE_PATTERN),
            485_000,
        ),
        Benchmark::new(
            "regex_capture_simple_10kb",
            || benchmark_capture(TEXT_10KB, SIMPLE_PATTERN),
            753_000,
        ),
        Benchmark::new(
            "regex_capture_simple_100kb",
            || benchmark_capture(TEXT_100KB, SIMPLE_PATTERN),
            3_437_000,
        ),
        Benchmark::new(
            "regex_capture_complex_1kb",
            || benchmark_capture(TEXT_1KB, COMPLEX_PATTERN),
            2_631_000,
        ),
        Benchmark::new(
            "regex_capture_complex_10kb",
            || benchmark_capture(TEXT_10KB, COMPLEX_PATTERN),
            2_900_000,
        ),
        Benchmark::new(
            "regex_capture_complex_100kb",
            || benchmark_capture(TEXT_100KB, COMPLEX_PATTERN),
            5_584_000,
        ),
    ]
}
