// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// See the LICENSE file in the project root for more information.

using System.Collections.Generic;
using Xunit;

namespace System.Linq.Tests
{
    public class AverageTests : EnumerableTests
    {
        [Fact]
        public void SameResultsRepeatCallsIntQuery()
        {
            var q = from x in new[] { 9999, 0, 888, -1, 66, -777, 1, 2, -12345 }
                    where x > Int32.MinValue
                    select x;

            Assert.Equal(q.Average(), q.Average());
        }

        [Fact]
        public void SameResultsRepeatCallsNullableLongQuery()
        {
            var q = from x in new long?[] { Int32.MaxValue, 0, 255, 127, 128, 1, 33, 99, null, Int32.MinValue }
                    select x;

            Assert.Equal(q.Average(), q.Average());
        }

        public static IEnumerable<object[]> Average_NullableFloat_TestData()
        {
            yield return new object[] { new float?[0], null };
            yield return new object[] { new float?[] { float.MinValue }, float.MinValue };
            yield return new object[] { new float?[] { 0f, 0f, 0f, 0f, 0f }, 0f };

            yield return new object[] { new float?[] { 5.5f, 0, null, null, null, 15.5f, 40.5f, null, null, -23.5f }, 7.6f };

            yield return new object[] { new float?[] { null, null, null, null, 45f }, 45f };
            yield return new object[] { new float?[] { null, null, null, null, null }, null };
        }

        [Theory]
        [MemberData(nameof(Average_NullableFloat_TestData))]
        public void Average_NullableFoat(float?[] source, float? expected)
        {
            Assert.Equal(expected, source.Average());
        }

        [Fact]
        public void Average_NullableFloat_EmptySourceWithSelector()
        {
            float?[] source = { };
            float? expected = null;

            Assert.Equal(expected, source.Average(i => i));
        }

        [Fact]
        public void Average_NullableFloat_NullSource_ThrowsArgumentNullException()
        {
            Assert.Throws<ArgumentNullException>("source", () => ((IEnumerable<float?>)null).Average());
            Assert.Throws<ArgumentNullException>("source", () => ((IEnumerable<float?>)null).Average(i => i));
        }

        [Fact]
        public void Average_NullableFloat_NullSelector_ThrowsArgumentNullException()
        {
            Func<float?, float?> selector = null;
            Assert.Throws<ArgumentNullException>("selector", () => Enumerable.Empty<float?>().Average(selector));
        }

        [Fact]
        public void Average_NullableFloat_AllNullWithSelector()
        {
            float?[] source = { null, null, null, null, null };
            float? expected = null;

            Assert.Equal(expected, source.Average(i => i));
        }

        [Fact]
        public void Average_NullableFloat_WithSelector()
        {
            var source = new []
            {
                new { name = "Tim", num = (float?)5.5f },
                new { name = "John", num = (float?)15.5f },
                new { name = "Bob", num = default(float?) }
            };
            float? expected = 10.5f;

            Assert.Equal(expected, source.Average(e => e.num));
        }

        [Fact]
        public void Average_Int_EmptySource_ThrowsInvalidOperationException()
        {
            int[] source = new int[0];
            
            Assert.Throws<InvalidOperationException>(() => source.Average());
            Assert.Throws<InvalidOperationException>(() => source.Average(i => i));
        }

        [Fact]
        public void Average_Int_NullSource_ThrowsArgumentNullException()
        {
            Assert.Throws<ArgumentNullException>("source", () => ((IEnumerable<int>)null).Average());
            Assert.Throws<ArgumentNullException>("source", () => ((IEnumerable<int>)null).Average(i => i));
        }

        [Fact]
        public void Average_Int_NullSelector_ThrowsArgumentNullException()
        {
            Func<int, int> selector = null;
            Assert.Throws<ArgumentNullException>("selector", () => Enumerable.Empty<int>().Average(selector));
        }

        public static IEnumerable<object[]> Average_Int_TestData()
        {
            yield return new object[] { new int[] { 5 }, 5 };
            yield return new object[] { new int[] { 0, 0, 0, 0, 0 }, 0 };
            yield return new object[] { new int[] { 5, -10, 15, 40, 28 }, 15.6 };
        }

        [Theory]
        [MemberData(nameof(Average_Int_TestData))]
        public void Average_Int(int[] source, double expected)
        {
            Assert.Equal(expected, source.Average());
        }

        [Fact]
        public void Average_Int_WithSelector()
        {
            var source = new []
            {
                new { name="Tim", num = 10 },
                new { name="John", num = -10 },
                new { name="Bob", num = 15 }
            };
            double expected = 5;

            Assert.Equal(expected, source.Average(e => e.num));
        }

        public static IEnumerable<object[]> Average_NullableInt_TestData()
        {
            yield return new object[] { new int?[0], null };
            yield return new object[] { new int?[] { -5 }, -5.0 };
            yield return new object[] { new int?[] { 0, 0, 0, 0, 0 }, 0.0 };
            yield return new object[] { new int?[] { 5, -10, null, null, null, 15, 40, 28, null, null }, 15.6 };
            yield return new object[] { new int?[] { null, null, null, null, 50 }, 50.0 };
            yield return new object[] { new int?[] { null, null, null, null, null }, null };
        }

        [Theory]
        [MemberData(nameof(Average_NullableInt_TestData))]
        public void Average_NullableInt(int?[] source, double? expected)
        {
            Assert.Equal(expected, source.Average());
        }

        [Fact]
        public void Average_NullableInt_EmptySourceWithSelector()
        {
            int?[] source = { };
            double? expected = null;

            Assert.Equal(expected, source.Average(i => i));
        }

        [Fact]
        public void Average_NullableInt_NullSource_ThrowsArgumentNullException()
        {
            Assert.Throws<ArgumentNullException>("source", () => ((IEnumerable<int?>)null).Average());
            Assert.Throws<ArgumentNullException>("source", () => ((IEnumerable<int?>)null).Average(i => i));
        }

        [Fact]
        public void Average_NullableInt_NullSelector_ThrowsArgumentNullException()
        {
            Func<int?, int?> selector = null;
            Assert.Throws<ArgumentNullException>("selector", () => Enumerable.Empty<int?>().Average(selector));
        }

        [Fact]
        public void Average_NullableInt_AllNullWithSelector()
        {
            int?[] source = { null, null, null, null, null };
            double? expected = null;

            Assert.Equal(expected, source.Average(i => i));
        }

        [Fact]
        public void Average_NullableInt_WithSelector()
        {
            var source = new []
            {
                new { name = "Tim", num  = (int?)10 },
                new { name = "John", num =  default(int?) },
                new { name = "Bob", num = (int?)10 }
            };
            double? expected = 10;

            Assert.Equal(expected, source.Average(e => e.num));
        }

        [Fact]
        public void Average_Long_EmptySource_ThrowsInvalidOperationException()
        {
            long[] source = new long[0];

            Assert.Throws<InvalidOperationException>(() => source.Average());
            Assert.Throws<InvalidOperationException>(() => source.Average(i => i));
        }

        [Fact]
        public void Average_Long_NullSource_ThrowsArgumentNullException()
        {
            Assert.Throws<ArgumentNullException>("source", () => ((IEnumerable<long>)null).Average());
            Assert.Throws<ArgumentNullException>("source", () => ((IEnumerable<long>)null).Average(i => i));
        }

        [Fact]
        public void Average_Long_NullSelector_ThrowsArgumentNullException()
        {
            Func<long, long> selector = null;
            Assert.Throws<ArgumentNullException>("selector", () => Enumerable.Empty<long>().Average(selector));
        }

        public static IEnumerable<object[]> Average_Long_TestData()
        {
            yield return new object[] { new long[] { long.MaxValue }, long.MaxValue };
            yield return new object[] { new long[] { 0, 0, 0, 0, 0 }, 0 };
            yield return new object[] { new long[] { 5, -10, 15, 40, 28 }, 15.6 };
        }
        
        [Theory]
        [MemberData(nameof(Average_Long_TestData))]
        public void Average_Long(long[] source, double expected)
        {
            Assert.Equal(expected, source.Average());
        }

        [Fact]
        public void Average_Long_FromSelector()
        {
            var source = new []
            {
                new { name = "Tim", num = 40L },
                new { name = "John", num = 50L },
                new { name = "Bob", num = 60L }
            };
            double expected = 50;

            Assert.Equal(expected, source.Average(e => e.num));
        }

        [Fact]
        public void Average_Long_SumTooLarge_ThrowsOverflowException()
        {
            long[] source = new long[] { long.MaxValue, long.MaxValue };

            Assert.Throws<OverflowException>(() => source.Average());
        }

        public static IEnumerable<object[]> Average_NullableLong_TestData()
        {
            yield return new object[] { new long?[0], null };
            yield return new object[] { new long?[] { long.MaxValue }, (double)long.MaxValue };
            yield return new object[] { new long?[] { 0, 0, 0, 0, 0 }, 0.0 };
            yield return new object[] { new long?[] { 5, -10, null, null, null, 15, 40, 28, null, null }, 15.6 };
            yield return new object[] { new long?[] { null, null, null, null, 50 }, 50.0 };
            yield return new object[] { new long?[] { null, null, null, null, null }, null };
        }

        [Theory]
        [MemberData(nameof(Average_NullableLong_TestData))]
        public void Average_NullableLong(long?[] source, double? expected)
        {
            Assert.Equal(expected, source.Average());
        }

        [Fact]
        public void Average_NullableLong_EmptySourceWithSelector()
        {
            long?[] source = { };
            double? expected = null;

            Assert.Equal(expected, source.Average(i => i));
        }

        [Fact]
        public void Average_NullableLong_NullSource_ThrowsArgumentNullException()
        {
            Assert.Throws<ArgumentNullException>("source", () => ((IEnumerable<long?>)null).Average());
            Assert.Throws<ArgumentNullException>("source", () => ((IEnumerable<long?>)null).Average(i => i));
        }

        [Fact]
        public void Average_NullableLong_NullSelector_ThrowsArgumentNullException()
        {
            Func<long?, long?> selector = null;
            Assert.Throws<ArgumentNullException>("selector", () => Enumerable.Empty<long?>().Average(selector));
        }

        [Fact]
        public void Average_NullableLong_AllNullWithSelector()
        {
            long?[] source = { null, null, null, null, null };
            double? expected = null;

            Assert.Equal(expected, source.Average(i => i));
        }

        [Fact]
        public void Average_NullableLong_WithSelector()
        {
            var source = new []
            {
                new { name = "Tim", num = (long?)40L },
                new { name = "John", num = default(long?) },
                new { name = "Bob", num = (long?)30L }
            };
            double? expected = 35;

            Assert.Equal(expected, source.Average(e => e.num));
        }

        [Fact]
        public void Average_Double_EmptySource_ThrowsInvalidOperationException()
        {
            double[] source = new double[0];

            Assert.Throws<InvalidOperationException>(() => source.Average());
            Assert.Throws<InvalidOperationException>(() => source.Average(i => i));
        }

        [Fact]
        public void Average_Double_NullSource_ThrowsArgumentNullException()
        {
            Assert.Throws<ArgumentNullException>("source", () => ((IEnumerable<double>)null).Average());
            Assert.Throws<ArgumentNullException>("source", () => ((IEnumerable<double>)null).Average(i => i));
        }

        [Fact]
        public void Average_Double_NullSelector_ThrowsArgumentNullException()
        {
            Func<double, double> selector = null;
            Assert.Throws<ArgumentNullException>("selector", () => Enumerable.Empty<double>().Average(selector));
        }

        public static IEnumerable<object[]> Average_Double_TestData()
        {
            yield return new object[] { new double[] { double.MaxValue }, double.MaxValue };
            yield return new object[] { new double[] { 0.0, 0.0, 0.0, 0.0, 0.0 }, 0 };
            yield return new object[] { new double[] { 5.5, -10, 15.5, 40.5, 28.5 }, 16 };
            yield return new object[] { new double[] { 5.58, Double.NaN, 30, 4.55, 19.38 }, double.NaN };
        }

        [Theory]
        [MemberData(nameof(Average_Double_TestData))]
        public void Average_Double(double[] source, double expected)
        {
            Assert.Equal(expected, source.Average());
        }

        [Fact]
        public void Average_Double_WithSelector()
        {
            var source = new []
            {
                new { name = "Tim", num = 5.5},
                new { name = "John", num = 15.5},
                new { name = "Bob", num = 3.0}
            };
            double expected = 8.0;

            Assert.Equal(expected, source.Average(e => e.num));
        }

        public static IEnumerable<object[]> Average_NullableDouble_TestData()
        {
            yield return new object[] { new double?[0], null };
            yield return new object[] { new double?[] { double.MinValue }, double.MinValue };
            yield return new object[] { new double?[] { 0, 0, 0, 0, 0 }, 0 };
            yield return new object[] { new double?[] { 5.5, 0, null, null, null, 15.5, 40.5, null, null, -23.5 }, 7.6 };
            yield return new object[] { new double?[] { null, null, null, null, 45 }, 45 };
            yield return new object[] { new double?[] { -23.5, 0, Double.NaN, 54.3, 0.56 }, double.NaN };
            yield return new object[] { new double?[] { null, null, null, null, null }, null };
        }

        public void Average_NullableDouble(double?[] source, double? expected)
        {
            Assert.Equal(expected, source.Average());
        }

        [Fact]
        public void Average_NullableDouble_EmptySourceWithSelector()
        {
            double?[] source = { };
            double? expected = null;

            Assert.Equal(expected, source.Average(i => i));
        }

        [Fact]
        public void Average_NullableDouble_NullSource_ThrowsArgumentNullException()
        {
            Assert.Throws<ArgumentNullException>("source", () => ((IEnumerable<double?>)null).Average());
            Assert.Throws<ArgumentNullException>("source", () => ((IEnumerable<double?>)null).Average(i => i));
        }

        [Fact]
        public void Average_NullableDouble_NullSelector_ThrowsArgumentNullException()
        {
            Func<double?, double?> selector = null;
            Assert.Throws<ArgumentNullException>("selector", () => Enumerable.Empty<double?>().Average(selector));
        }

        [Fact]
        public void Average_NullableDouble_AllNullWithSelector()
        {
            double?[] source = { null, null, null, null, null };
            double? expected = null;

            Assert.Equal(expected, source.Average(i => i));
        }

        [Fact]
        public void Average_NullableDouble_WithSelector()
        {
            var source = new[]
            {
                new{ name = "Tim", num = (double?)5.5 },
                new{ name = "John", num = (double?)15.5 },
                new{ name = "Bob", num = default(double?) }
            };
            double? expected = 10.5;

            Assert.Equal(expected, source.Average(e => e.num));
        }

        [Fact]
        public void Average_Decimal_EmptySource_ThrowsInvalidOperationException()
        {
            decimal[] source = new decimal[0];

            Assert.Throws<InvalidOperationException>(() => source.Average());
            Assert.Throws<InvalidOperationException>(() => source.Average(i => i));
        }

        [Fact]
        public void Average_Decimal_NullSource_ThrowsArgumentNullException()
        {
            Assert.Throws<ArgumentNullException>("source", () => ((IEnumerable<decimal>)null).Average());
            Assert.Throws<ArgumentNullException>("source", () => ((IEnumerable<decimal>)null).Average(i => i));
        }

        [Fact]
        public void Average_Decimal_NullSelector_ThrowsArgumentNullException()
        {
            Func<decimal, decimal> selector = null;
            Assert.Throws<ArgumentNullException>("selector", () => Enumerable.Empty<decimal>().Average(selector));
        }

        public static IEnumerable<object[]> Average_Decimal_TestData()
        {
            yield return new object[] { new decimal[] { decimal.MaxValue }, decimal.MaxValue };
            yield return new object[] { new decimal[] { 0.0m, 0.0m, 0.0m, 0.0m, 0.0m }, 0 };
            yield return new object[] { new decimal[] { 5.5m, -10m, 15.5m, 40.5m, 28.5m }, 16 };
        }

        [Theory]
        [MemberData(nameof(Average_Decimal_TestData))]
        public void Average_Decimal(decimal[] source, decimal expected)
        {
            Assert.Equal(expected, source.Average());
        }

        [Fact]
        public void Average_Decimal_WithSelector()
        {
            var source = new[]
            {
                new{ name = "Tim", num = 5.5m},
                new{ name = "John", num = 15.5m},
                new{ name = "Bob", num = 3.0m}
            };
            decimal expected = 8.0m;

            Assert.Equal(expected, source.Average(e => e.num));
        }

        public static IEnumerable<object[]> Average_NullableDecimal_TestData()
        {
            yield return new object[] { new decimal?[0], null };
            yield return new object[] { new decimal?[] { decimal.MinValue }, decimal.MinValue };
            yield return new object[] { new decimal?[] { 0m, 0m, 0m, 0m, 0m }, 0m };
            yield return new object[] { new decimal?[] { 5.5m, 0, null, null, null, 15.5m, 40.5m, null, null, -23.5m }, 7.6m };
            yield return new object[] { new decimal?[] { null, null, null, null, 45m }, 45m };
            yield return new object[] { new decimal?[] { null, null, null, null, null }, null };
        }

        [Theory]
        [MemberData(nameof(Average_NullableDecimal_TestData))]
        public void Average_NullableDecimal(decimal?[] source, decimal? expected)
        {
            Assert.Equal(expected, source.Average());
        }

        [Fact]
        public void Average_NullableDecimal_EmptySourceWithSelector()
        {
            decimal?[] source = { };
            decimal? expected = null;

            Assert.Equal(expected, source.Average(i => i));
        }

        [Fact]
        public void Average_NullableDecimal_NullSource_ThrowsArgumentNullException()
        {
            Assert.Throws<ArgumentNullException>("source", () => ((IEnumerable<decimal?>)null).Average());
            Assert.Throws<ArgumentNullException>("source", () => ((IEnumerable<decimal?>)null).Average(i => i));
        }

        [Fact]
        public void Average_NullableDecimal_NullSelector_ThrowsArgumentNullException()
        {
            Func<decimal?, decimal?> selector = null;
            Assert.Throws<ArgumentNullException>("selector", () => Enumerable.Empty<decimal?>().Average(selector));
        }

        [Fact]
        public void Average_NullableDecimal_AllNullWithSelector()
        {
            decimal?[] source = { null, null, null, null, null };
            decimal? expected = null;

            Assert.Equal(expected, source.Average(i => i));
        }

        [Fact]
        public void Average_NullableDecimal_WithSelector()
        {
            var source = new[]
            {
                new{ name = "Tim", num = (decimal?)5.5m},
                new{ name = "John", num = (decimal?)15.5m},
                new{ name = "Bob", num = (decimal?)null}
            };
            decimal? expected = 10.5m;

            Assert.Equal(expected, source.Average(e => e.num));
        }

        [Fact]
        public void Average_NullableDecimal_SumTooLarge_ThrowsOverflowException()
        {
            decimal?[] source = new decimal?[] { decimal.MaxValue, decimal.MaxValue };

            Assert.Throws<OverflowException>(() => source.Average());
        }

        [Fact]
        public void Average_Float_EmptySource_ThrowsInvalidOperationException()
        {
            float[] source = new float[0];

            Assert.Throws<InvalidOperationException>(() => source.Average());
            Assert.Throws<InvalidOperationException>(() => source.Average(i => i));
        }

        [Fact]
        public void Average_Float_NullSource_ThrowsArgumentNullException()
        {
            Assert.Throws<ArgumentNullException>("source", () => ((IEnumerable<float>)null).Average());
            Assert.Throws<ArgumentNullException>("source", () => ((IEnumerable<float>)null).Average(i => i));
        }

        [Fact]
        public void Average_Float_NullSelector_ThrowsArgumentNullException()
        {
            Func<float, float> selector = null;
            Assert.Throws<ArgumentNullException>("selector", () => Enumerable.Empty<float>().Average(selector));
        }

        public static IEnumerable<object[]> Average_Float_TestData()
        {
            yield return new object[] { new float[] { float.MaxValue }, float.MaxValue };
            yield return new object[] { new float[] { 0.0f, 0.0f, 0.0f, 0.0f, 0.0f }, 0f };
            yield return new object[] { new float[] { 5.5f, -10f, 15.5f, 40.5f, 28.5f }, 16f };
        }

        [Theory]
        [MemberData(nameof(Average_Float_TestData))]
        public void Average_Float(float[] source, float expected)
        {
            Assert.Equal(expected, source.Average());
        }

        [Fact]
        public void Average_Float_WithSelector()
        {
            var source = new[]
            {
                new{ name = "Tim", num = 5.5f},
                new{ name = "John", num = 15.5f},
                new{ name = "Bob", num = 3.0f}
            };
            float expected = 8.0f;

            Assert.Equal(expected, source.Average(e => e.num));
        }
    }
}
