/*
 * Copyright 2017 PingCAP, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.pingcap.tikv.predicates;

import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;

import com.google.common.collect.ImmutableList;
import com.pingcap.tikv.expression.TiColumnRef;
import com.pingcap.tikv.expression.TiConstant;
import com.pingcap.tikv.expression.TiExpr;
import com.pingcap.tikv.expression.scalar.*;
import com.pingcap.tikv.meta.MetaUtils;
import com.pingcap.tikv.meta.TiIndexColumn;
import com.pingcap.tikv.meta.TiIndexInfo;
import com.pingcap.tikv.meta.TiTableInfo;
import com.pingcap.tikv.types.DataTypeFactory;
import com.pingcap.tikv.types.Types;
import org.junit.Test;

public class IndexMatcherTest {
  private static TiTableInfo createTable() {
    return new MetaUtils.TableBuilder()
        .name("testTable")
        .addColumn("c1", DataTypeFactory.of(Types.TYPE_LONG), true)
        .addColumn("c2", DataTypeFactory.of(Types.TYPE_STRING))
        .addColumn("c3", DataTypeFactory.of(Types.TYPE_STRING))
        .addColumn("c4", DataTypeFactory.of(Types.TYPE_TINY))
        .appendIndex("testIndex", ImmutableList.of("c1", "c2"), false)
        .build();
  }

  @Test
  public void matchOnlyEq() throws Exception {
    TiTableInfo table = createTable();
    TiIndexInfo index = table.getIndices().get(0);
    TiIndexColumn col = index.getIndexColumns().get(0);
    IndexMatcher matcher = new IndexMatcher(col, true);

    // index col = c1, long
    TiExpr cond = new Equal(TiColumnRef.create("c1", table), TiConstant.create(1));
    assertTrue(matcher.match(cond));

    cond = new Equal(TiConstant.create(1), TiColumnRef.create("c1", table));
    assertTrue(matcher.match(cond));

    cond = new Equal(TiColumnRef.create("c2", table), TiColumnRef.create("c1", table));
    assertFalse(matcher.match(cond));

    cond = new Equal(TiConstant.create(1), TiConstant.create(1));
    assertFalse(matcher.match(cond));

    cond =
        new And(
            new Equal(TiConstant.create(1), TiColumnRef.create("c1", table)),
            new Equal(TiColumnRef.create("c1", table), TiConstant.create(2)));
    assertFalse(matcher.match(cond));

    cond =
        new Or(
            new Equal(TiConstant.create(1), TiColumnRef.create("c1", table)),
            new Equal(TiColumnRef.create("c1", table), TiConstant.create(2)));
    assertTrue(matcher.match(cond));

    cond = new In(TiColumnRef.create("c1", table), TiConstant.create(1), TiConstant.create(2));
    assertTrue(matcher.match(cond));

    cond =
        new In(
            new Equal(TiColumnRef.create("c1", table), TiConstant.create(2)),
            TiConstant.create(1),
            TiConstant.create(2));
    assertFalse(matcher.match(cond));

    cond = new LessEqual(TiConstant.create(0), TiColumnRef.create("c1", table));
    assertFalse(matcher.match(cond));
  }

  @Test
  public void matchAll() throws Exception {
    TiTableInfo table = createTable();
    TiIndexInfo index = table.getIndices().get(0);
    TiIndexColumn col = index.getIndexColumns().get(0);
    IndexMatcher matcher = new IndexMatcher(col, false);

    // index col = c1, long
    TiExpr cond = new LessEqual(TiColumnRef.create("c1", table), TiConstant.create(1));
    assertTrue(matcher.match(cond));

    cond = new GreaterEqual(TiConstant.create(1), TiColumnRef.create("c1", table));
    assertTrue(matcher.match(cond));

    cond = new LessThan(TiColumnRef.create("c2", table), TiColumnRef.create("c1", table));
    assertFalse(matcher.match(cond));

    cond = new LessThan(TiConstant.create(1), TiConstant.create(1));
    assertFalse(matcher.match(cond));

    cond =
        new And(
            new LessThan(TiConstant.create(1), TiColumnRef.create("c1", table)),
            new LessThan(TiColumnRef.create("c1", table), TiConstant.create(2)));
    assertTrue(matcher.match(cond));

    cond =
        new Or(
            new LessThan(TiConstant.create(1), TiColumnRef.create("c1", table)),
            new LessThan(TiColumnRef.create("c1", table), TiConstant.create(2)));
    assertTrue(matcher.match(cond));

    cond = new In(TiColumnRef.create("c1", table), TiConstant.create(1), TiConstant.create(2));
    assertTrue(matcher.match(cond));

    cond =
        new In(
            new Equal(TiColumnRef.create("c1", table), TiConstant.create(2)),
            TiConstant.create(1),
            TiConstant.create(2));
    assertFalse(matcher.match(cond));
  }
}
