#ifndef CAPSTONE_ARM64_H
#define CAPSTONE_ARM64_H

/* Capstone Disassembly Engine */
/* By Nguyen Anh Quynh <aquynh@gmail.com>, 2013-2015 */

#ifdef __cplusplus
extern "C" {
#endif

#include "aarch64.h"
#include "cs_operand.h"
#include "platform.h"

#include <assert.h>

#ifdef _MSC_VER
#pragma warning(disable : 4201)
#endif

typedef enum {
  ARM64_SFT_INVALID = AARCH64_SFT_INVALID,
  ARM64_SFT_LSL = AARCH64_SFT_LSL,
  ARM64_SFT_MSL = AARCH64_SFT_MSL,
  ARM64_SFT_LSR = AARCH64_SFT_LSR,
  ARM64_SFT_ASR = AARCH64_SFT_ASR,
  ARM64_SFT_ROR = AARCH64_SFT_ROR,
} arm64_shifter;

typedef enum {
  ARM64_EXT_INVALID = AARCH64_EXT_INVALID,
  ARM64_EXT_UXTB = AARCH64_EXT_UXTB,
  ARM64_EXT_UXTH = AARCH64_EXT_UXTH,
  ARM64_EXT_UXTW = AARCH64_EXT_UXTW,
  ARM64_EXT_UXTX = AARCH64_EXT_UXTX,
  ARM64_EXT_SXTB = AARCH64_EXT_SXTB,
  ARM64_EXT_SXTH = AARCH64_EXT_SXTH,
  ARM64_EXT_SXTW = AARCH64_EXT_SXTW,
  ARM64_EXT_SXTX = AARCH64_EXT_SXTX,
} arm64_extender;

typedef enum {
	ARM64LAYOUT_INVALID = AARCH64LAYOUT_INVALID,
	ARM64LAYOUT_VL_B = AARCH64LAYOUT_VL_B,
	ARM64LAYOUT_VL_H = AARCH64LAYOUT_VL_H,
	ARM64LAYOUT_VL_S = AARCH64LAYOUT_VL_S,
	ARM64LAYOUT_VL_D = AARCH64LAYOUT_VL_D,
	ARM64LAYOUT_VL_Q = AARCH64LAYOUT_VL_Q,

	ARM64LAYOUT_VL_4B = AARCH64LAYOUT_VL_4B,
	ARM64LAYOUT_VL_2H = AARCH64LAYOUT_VL_2H,
	ARM64LAYOUT_VL_1S = AARCH64LAYOUT_VL_1S,

	ARM64LAYOUT_VL_8B = AARCH64LAYOUT_VL_8B,
	ARM64LAYOUT_VL_4H = AARCH64LAYOUT_VL_4H,
	ARM64LAYOUT_VL_2S = AARCH64LAYOUT_VL_2S,
	ARM64LAYOUT_VL_1D = AARCH64LAYOUT_VL_1D,

	ARM64LAYOUT_VL_16B = AARCH64LAYOUT_VL_16B,
	ARM64LAYOUT_VL_8H = AARCH64LAYOUT_VL_8H,
	ARM64LAYOUT_VL_4S = AARCH64LAYOUT_VL_4S,
	ARM64LAYOUT_VL_2D = AARCH64LAYOUT_VL_2D,
	ARM64LAYOUT_VL_1Q = AARCH64LAYOUT_VL_1Q,

	ARM64LAYOUT_VL_64B = AARCH64LAYOUT_VL_64B,
	ARM64LAYOUT_VL_32H = AARCH64LAYOUT_VL_32H,
	ARM64LAYOUT_VL_16S = AARCH64LAYOUT_VL_16S,
	ARM64LAYOUT_VL_8D = AARCH64LAYOUT_VL_8D,

	ARM64LAYOUT_VL_COMPLETE = AARCH64LAYOUT_VL_COMPLETE,
} ARM64Layout_VectorLayout;


typedef enum {
	ARM64CC_EQ = AArch64CC_EQ,
	ARM64CC_NE = AArch64CC_NE,
	ARM64CC_HS = AArch64CC_HS,
	ARM64CC_LO = AArch64CC_LO,
	ARM64CC_MI = AArch64CC_MI,
	ARM64CC_PL = AArch64CC_PL,
	ARM64CC_VS = AArch64CC_VS,
	ARM64CC_VC = AArch64CC_VC,
	ARM64CC_HI = AArch64CC_HI,
	ARM64CC_LS = AArch64CC_LS,
	ARM64CC_GE = AArch64CC_GE,
	ARM64CC_LT = AArch64CC_LT,
	ARM64CC_GT = AArch64CC_GT,
	ARM64CC_LE = AArch64CC_LE,
	ARM64CC_AL = AArch64CC_AL,
	ARM64CC_NV = AArch64CC_NV,
	ARM64CC_Invalid = AArch64CC_Invalid,

	ARM64CC_ANY_ACTIVE = AArch64CC_ANY_ACTIVE,
	ARM64CC_FIRST_ACTIVE = AArch64CC_FIRST_ACTIVE,
	ARM64CC_LAST_ACTIVE = AArch64CC_LAST_ACTIVE,
	ARM64CC_NONE_ACTIVE = AArch64CC_NONE_ACTIVE,
} ARM64CC_CondCode;

inline static const char *ARM64CC_getCondCodeName(ARM64CC_CondCode Code)
{
	switch (Code) {
	default:
		assert(0 && "Unknown condition code");
	case ARM64CC_EQ:
		return "eq";
	case ARM64CC_NE:
		return "ne";
	case ARM64CC_HS:
		return "hs";
	case ARM64CC_LO:
		return "lo";
	case ARM64CC_MI:
		return "mi";
	case ARM64CC_PL:
		return "pl";
	case ARM64CC_VS:
		return "vs";
	case ARM64CC_VC:
		return "vc";
	case ARM64CC_HI:
		return "hi";
	case ARM64CC_LS:
		return "ls";
	case ARM64CC_GE:
		return "ge";
	case ARM64CC_LT:
		return "lt";
	case ARM64CC_GT:
		return "gt";
	case ARM64CC_LE:
		return "le";
	case ARM64CC_AL:
		return "al";
	case ARM64CC_NV:
		return "nv";
	}
}

inline static ARM64CC_CondCode ARM64CC_getInvertedCondCode(ARM64CC_CondCode Code)
{

	return (ARM64CC_CondCode)((unsigned)(Code) ^ 0x1);
}

inline static unsigned ARM64CC_getNZCVToSatisfyCondCode(ARM64CC_CondCode Code)
{
	enum { N = 8, Z = 4, C = 2, V = 1 };
	switch (Code) {
	default:
		assert(0 && "Unknown condition code");
	case ARM64CC_EQ:
		return Z; // Z == 1
	case ARM64CC_NE:
		return 0; // Z == 0
	case ARM64CC_HS:
		return C; // C == 1
	case ARM64CC_LO:
		return 0; // C == 0
	case ARM64CC_MI:
		return N; // N == 1
	case ARM64CC_PL:
		return 0; // N == 0
	case ARM64CC_VS:
		return V; // V == 1
	case ARM64CC_VC:
		return 0; // V == 0
	case ARM64CC_HI:
		return C; // C == 1 && Z == 0
	case ARM64CC_LS:
		return 0; // C == 0 || Z == 1
	case ARM64CC_GE:
		return 0; // N == V
	case ARM64CC_LT:
		return N; // N != V
	case ARM64CC_GT:
		return 0; // Z == 0 && N == V
	case ARM64CC_LE:
		return Z; // Z == 1 || N != V
	}
}

inline static bool ARM64CC_isReflexive(ARM64CC_CondCode Code)
{
	switch (Code) {
	case ARM64CC_EQ:
	case ARM64CC_HS:
	case ARM64CC_PL:
	case ARM64CC_LS:
	case ARM64CC_GE:
	case ARM64CC_LE:
	case ARM64CC_AL:
	case ARM64CC_NV:
		return true;
	default:
		return false;
	}
}

inline static bool ARM64CC_isIrreflexive(ARM64CC_CondCode Code)
{
	switch (Code) {
	case ARM64CC_NE:
	case ARM64CC_LO:
	case ARM64CC_MI:
	case ARM64CC_HI:
	case ARM64CC_LT:
	case ARM64CC_GT:
		return true;
	default:
		return false;
	}
}


typedef enum {

	ARM64_TLBI_ALLE1 = AARCH64_TLBI_ALLE1,
	ARM64_TLBI_ALLE1IS = AARCH64_TLBI_ALLE1IS,
	ARM64_TLBI_ALLE1ISNXS = AARCH64_TLBI_ALLE1ISNXS,
	ARM64_TLBI_ALLE1NXS = AARCH64_TLBI_ALLE1NXS,
	ARM64_TLBI_ALLE1OS = AARCH64_TLBI_ALLE1OS,
	ARM64_TLBI_ALLE1OSNXS = AARCH64_TLBI_ALLE1OSNXS,
	ARM64_TLBI_ALLE2 = AARCH64_TLBI_ALLE2,
	ARM64_TLBI_ALLE2IS = AARCH64_TLBI_ALLE2IS,
	ARM64_TLBI_ALLE2ISNXS = AARCH64_TLBI_ALLE2ISNXS,
	ARM64_TLBI_ALLE2NXS = AARCH64_TLBI_ALLE2NXS,
	ARM64_TLBI_ALLE2OS = AARCH64_TLBI_ALLE2OS,
	ARM64_TLBI_ALLE2OSNXS = AARCH64_TLBI_ALLE2OSNXS,
	ARM64_TLBI_ALLE3 = AARCH64_TLBI_ALLE3,
	ARM64_TLBI_ALLE3IS = AARCH64_TLBI_ALLE3IS,
	ARM64_TLBI_ALLE3ISNXS = AARCH64_TLBI_ALLE3ISNXS,
	ARM64_TLBI_ALLE3NXS = AARCH64_TLBI_ALLE3NXS,
	ARM64_TLBI_ALLE3OS = AARCH64_TLBI_ALLE3OS,
	ARM64_TLBI_ALLE3OSNXS = AARCH64_TLBI_ALLE3OSNXS,
	ARM64_TLBI_ASIDE1 = AARCH64_TLBI_ASIDE1,
	ARM64_TLBI_ASIDE1IS = AARCH64_TLBI_ASIDE1IS,
	ARM64_TLBI_ASIDE1ISNXS = AARCH64_TLBI_ASIDE1ISNXS,
	ARM64_TLBI_ASIDE1NXS = AARCH64_TLBI_ASIDE1NXS,
	ARM64_TLBI_ASIDE1OS = AARCH64_TLBI_ASIDE1OS,
	ARM64_TLBI_ASIDE1OSNXS = AARCH64_TLBI_ASIDE1OSNXS,
	ARM64_TLBI_IPAS2E1 = AARCH64_TLBI_IPAS2E1,
	ARM64_TLBI_IPAS2E1IS = AARCH64_TLBI_IPAS2E1IS,
	ARM64_TLBI_IPAS2E1ISNXS = AARCH64_TLBI_IPAS2E1ISNXS,
	ARM64_TLBI_IPAS2E1NXS = AARCH64_TLBI_IPAS2E1NXS,
	ARM64_TLBI_IPAS2E1OS = AARCH64_TLBI_IPAS2E1OS,
	ARM64_TLBI_IPAS2E1OSNXS = AARCH64_TLBI_IPAS2E1OSNXS,
	ARM64_TLBI_IPAS2LE1 = AARCH64_TLBI_IPAS2LE1,
	ARM64_TLBI_IPAS2LE1IS = AARCH64_TLBI_IPAS2LE1IS,
	ARM64_TLBI_IPAS2LE1ISNXS = AARCH64_TLBI_IPAS2LE1ISNXS,
	ARM64_TLBI_IPAS2LE1NXS = AARCH64_TLBI_IPAS2LE1NXS,
	ARM64_TLBI_IPAS2LE1OS = AARCH64_TLBI_IPAS2LE1OS,
	ARM64_TLBI_IPAS2LE1OSNXS = AARCH64_TLBI_IPAS2LE1OSNXS,
	ARM64_TLBI_PAALL = AARCH64_TLBI_PAALL,
	ARM64_TLBI_PAALLNXS = AARCH64_TLBI_PAALLNXS,
	ARM64_TLBI_PAALLOS = AARCH64_TLBI_PAALLOS,
	ARM64_TLBI_PAALLOSNXS = AARCH64_TLBI_PAALLOSNXS,
	ARM64_TLBI_RIPAS2E1 = AARCH64_TLBI_RIPAS2E1,
	ARM64_TLBI_RIPAS2E1IS = AARCH64_TLBI_RIPAS2E1IS,
	ARM64_TLBI_RIPAS2E1ISNXS = AARCH64_TLBI_RIPAS2E1ISNXS,
	ARM64_TLBI_RIPAS2E1NXS = AARCH64_TLBI_RIPAS2E1NXS,
	ARM64_TLBI_RIPAS2E1OS = AARCH64_TLBI_RIPAS2E1OS,
	ARM64_TLBI_RIPAS2E1OSNXS = AARCH64_TLBI_RIPAS2E1OSNXS,
	ARM64_TLBI_RIPAS2LE1 = AARCH64_TLBI_RIPAS2LE1,
	ARM64_TLBI_RIPAS2LE1IS = AARCH64_TLBI_RIPAS2LE1IS,
	ARM64_TLBI_RIPAS2LE1ISNXS = AARCH64_TLBI_RIPAS2LE1ISNXS,
	ARM64_TLBI_RIPAS2LE1NXS = AARCH64_TLBI_RIPAS2LE1NXS,
	ARM64_TLBI_RIPAS2LE1OS = AARCH64_TLBI_RIPAS2LE1OS,
	ARM64_TLBI_RIPAS2LE1OSNXS = AARCH64_TLBI_RIPAS2LE1OSNXS,
	ARM64_TLBI_RPALOS = AARCH64_TLBI_RPALOS,
	ARM64_TLBI_RPALOSNXS = AARCH64_TLBI_RPALOSNXS,
	ARM64_TLBI_RPAOS = AARCH64_TLBI_RPAOS,
	ARM64_TLBI_RPAOSNXS = AARCH64_TLBI_RPAOSNXS,
	ARM64_TLBI_RVAAE1 = AARCH64_TLBI_RVAAE1,
	ARM64_TLBI_RVAAE1IS = AARCH64_TLBI_RVAAE1IS,
	ARM64_TLBI_RVAAE1ISNXS = AARCH64_TLBI_RVAAE1ISNXS,
	ARM64_TLBI_RVAAE1NXS = AARCH64_TLBI_RVAAE1NXS,
	ARM64_TLBI_RVAAE1OS = AARCH64_TLBI_RVAAE1OS,
	ARM64_TLBI_RVAAE1OSNXS = AARCH64_TLBI_RVAAE1OSNXS,
	ARM64_TLBI_RVAALE1 = AARCH64_TLBI_RVAALE1,
	ARM64_TLBI_RVAALE1IS = AARCH64_TLBI_RVAALE1IS,
	ARM64_TLBI_RVAALE1ISNXS = AARCH64_TLBI_RVAALE1ISNXS,
	ARM64_TLBI_RVAALE1NXS = AARCH64_TLBI_RVAALE1NXS,
	ARM64_TLBI_RVAALE1OS = AARCH64_TLBI_RVAALE1OS,
	ARM64_TLBI_RVAALE1OSNXS = AARCH64_TLBI_RVAALE1OSNXS,
	ARM64_TLBI_RVAE1 = AARCH64_TLBI_RVAE1,
	ARM64_TLBI_RVAE1IS = AARCH64_TLBI_RVAE1IS,
	ARM64_TLBI_RVAE1ISNXS = AARCH64_TLBI_RVAE1ISNXS,
	ARM64_TLBI_RVAE1NXS = AARCH64_TLBI_RVAE1NXS,
	ARM64_TLBI_RVAE1OS = AARCH64_TLBI_RVAE1OS,
	ARM64_TLBI_RVAE1OSNXS = AARCH64_TLBI_RVAE1OSNXS,
	ARM64_TLBI_RVAE2 = AARCH64_TLBI_RVAE2,
	ARM64_TLBI_RVAE2IS = AARCH64_TLBI_RVAE2IS,
	ARM64_TLBI_RVAE2ISNXS = AARCH64_TLBI_RVAE2ISNXS,
	ARM64_TLBI_RVAE2NXS = AARCH64_TLBI_RVAE2NXS,
	ARM64_TLBI_RVAE2OS = AARCH64_TLBI_RVAE2OS,
	ARM64_TLBI_RVAE2OSNXS = AARCH64_TLBI_RVAE2OSNXS,
	ARM64_TLBI_RVAE3 = AARCH64_TLBI_RVAE3,
	ARM64_TLBI_RVAE3IS = AARCH64_TLBI_RVAE3IS,
	ARM64_TLBI_RVAE3ISNXS = AARCH64_TLBI_RVAE3ISNXS,
	ARM64_TLBI_RVAE3NXS = AARCH64_TLBI_RVAE3NXS,
	ARM64_TLBI_RVAE3OS = AARCH64_TLBI_RVAE3OS,
	ARM64_TLBI_RVAE3OSNXS = AARCH64_TLBI_RVAE3OSNXS,
	ARM64_TLBI_RVALE1 = AARCH64_TLBI_RVALE1,
	ARM64_TLBI_RVALE1IS = AARCH64_TLBI_RVALE1IS,
	ARM64_TLBI_RVALE1ISNXS = AARCH64_TLBI_RVALE1ISNXS,
	ARM64_TLBI_RVALE1NXS = AARCH64_TLBI_RVALE1NXS,
	ARM64_TLBI_RVALE1OS = AARCH64_TLBI_RVALE1OS,
	ARM64_TLBI_RVALE1OSNXS = AARCH64_TLBI_RVALE1OSNXS,
	ARM64_TLBI_RVALE2 = AARCH64_TLBI_RVALE2,
	ARM64_TLBI_RVALE2IS = AARCH64_TLBI_RVALE2IS,
	ARM64_TLBI_RVALE2ISNXS = AARCH64_TLBI_RVALE2ISNXS,
	ARM64_TLBI_RVALE2NXS = AARCH64_TLBI_RVALE2NXS,
	ARM64_TLBI_RVALE2OS = AARCH64_TLBI_RVALE2OS,
	ARM64_TLBI_RVALE2OSNXS = AARCH64_TLBI_RVALE2OSNXS,
	ARM64_TLBI_RVALE3 = AARCH64_TLBI_RVALE3,
	ARM64_TLBI_RVALE3IS = AARCH64_TLBI_RVALE3IS,
	ARM64_TLBI_RVALE3ISNXS = AARCH64_TLBI_RVALE3ISNXS,
	ARM64_TLBI_RVALE3NXS = AARCH64_TLBI_RVALE3NXS,
	ARM64_TLBI_RVALE3OS = AARCH64_TLBI_RVALE3OS,
	ARM64_TLBI_RVALE3OSNXS = AARCH64_TLBI_RVALE3OSNXS,
	ARM64_TLBI_VAAE1 = AARCH64_TLBI_VAAE1,
	ARM64_TLBI_VAAE1IS = AARCH64_TLBI_VAAE1IS,
	ARM64_TLBI_VAAE1ISNXS = AARCH64_TLBI_VAAE1ISNXS,
	ARM64_TLBI_VAAE1NXS = AARCH64_TLBI_VAAE1NXS,
	ARM64_TLBI_VAAE1OS = AARCH64_TLBI_VAAE1OS,
	ARM64_TLBI_VAAE1OSNXS = AARCH64_TLBI_VAAE1OSNXS,
	ARM64_TLBI_VAALE1 = AARCH64_TLBI_VAALE1,
	ARM64_TLBI_VAALE1IS = AARCH64_TLBI_VAALE1IS,
	ARM64_TLBI_VAALE1ISNXS = AARCH64_TLBI_VAALE1ISNXS,
	ARM64_TLBI_VAALE1NXS = AARCH64_TLBI_VAALE1NXS,
	ARM64_TLBI_VAALE1OS = AARCH64_TLBI_VAALE1OS,
	ARM64_TLBI_VAALE1OSNXS = AARCH64_TLBI_VAALE1OSNXS,
	ARM64_TLBI_VAE1 = AARCH64_TLBI_VAE1,
	ARM64_TLBI_VAE1IS = AARCH64_TLBI_VAE1IS,
	ARM64_TLBI_VAE1ISNXS = AARCH64_TLBI_VAE1ISNXS,
	ARM64_TLBI_VAE1NXS = AARCH64_TLBI_VAE1NXS,
	ARM64_TLBI_VAE1OS = AARCH64_TLBI_VAE1OS,
	ARM64_TLBI_VAE1OSNXS = AARCH64_TLBI_VAE1OSNXS,
	ARM64_TLBI_VAE2 = AARCH64_TLBI_VAE2,
	ARM64_TLBI_VAE2IS = AARCH64_TLBI_VAE2IS,
	ARM64_TLBI_VAE2ISNXS = AARCH64_TLBI_VAE2ISNXS,
	ARM64_TLBI_VAE2NXS = AARCH64_TLBI_VAE2NXS,
	ARM64_TLBI_VAE2OS = AARCH64_TLBI_VAE2OS,
	ARM64_TLBI_VAE2OSNXS = AARCH64_TLBI_VAE2OSNXS,
	ARM64_TLBI_VAE3 = AARCH64_TLBI_VAE3,
	ARM64_TLBI_VAE3IS = AARCH64_TLBI_VAE3IS,
	ARM64_TLBI_VAE3ISNXS = AARCH64_TLBI_VAE3ISNXS,
	ARM64_TLBI_VAE3NXS = AARCH64_TLBI_VAE3NXS,
	ARM64_TLBI_VAE3OS = AARCH64_TLBI_VAE3OS,
	ARM64_TLBI_VAE3OSNXS = AARCH64_TLBI_VAE3OSNXS,
	ARM64_TLBI_VALE1 = AARCH64_TLBI_VALE1,
	ARM64_TLBI_VALE1IS = AARCH64_TLBI_VALE1IS,
	ARM64_TLBI_VALE1ISNXS = AARCH64_TLBI_VALE1ISNXS,
	ARM64_TLBI_VALE1NXS = AARCH64_TLBI_VALE1NXS,
	ARM64_TLBI_VALE1OS = AARCH64_TLBI_VALE1OS,
	ARM64_TLBI_VALE1OSNXS = AARCH64_TLBI_VALE1OSNXS,
	ARM64_TLBI_VALE2 = AARCH64_TLBI_VALE2,
	ARM64_TLBI_VALE2IS = AARCH64_TLBI_VALE2IS,
	ARM64_TLBI_VALE2ISNXS = AARCH64_TLBI_VALE2ISNXS,
	ARM64_TLBI_VALE2NXS = AARCH64_TLBI_VALE2NXS,
	ARM64_TLBI_VALE2OS = AARCH64_TLBI_VALE2OS,
	ARM64_TLBI_VALE2OSNXS = AARCH64_TLBI_VALE2OSNXS,
	ARM64_TLBI_VALE3 = AARCH64_TLBI_VALE3,
	ARM64_TLBI_VALE3IS = AARCH64_TLBI_VALE3IS,
	ARM64_TLBI_VALE3ISNXS = AARCH64_TLBI_VALE3ISNXS,
	ARM64_TLBI_VALE3NXS = AARCH64_TLBI_VALE3NXS,
	ARM64_TLBI_VALE3OS = AARCH64_TLBI_VALE3OS,
	ARM64_TLBI_VALE3OSNXS = AARCH64_TLBI_VALE3OSNXS,
	ARM64_TLBI_VMALLE1 = AARCH64_TLBI_VMALLE1,
	ARM64_TLBI_VMALLE1IS = AARCH64_TLBI_VMALLE1IS,
	ARM64_TLBI_VMALLE1ISNXS = AARCH64_TLBI_VMALLE1ISNXS,
	ARM64_TLBI_VMALLE1NXS = AARCH64_TLBI_VMALLE1NXS,
	ARM64_TLBI_VMALLE1OS = AARCH64_TLBI_VMALLE1OS,
	ARM64_TLBI_VMALLE1OSNXS = AARCH64_TLBI_VMALLE1OSNXS,
	ARM64_TLBI_VMALLS12E1 = AARCH64_TLBI_VMALLS12E1,
	ARM64_TLBI_VMALLS12E1IS = AARCH64_TLBI_VMALLS12E1IS,
	ARM64_TLBI_VMALLS12E1ISNXS = AARCH64_TLBI_VMALLS12E1ISNXS,
	ARM64_TLBI_VMALLS12E1NXS = AARCH64_TLBI_VMALLS12E1NXS,
	ARM64_TLBI_VMALLS12E1OS = AARCH64_TLBI_VMALLS12E1OS,
	ARM64_TLBI_VMALLS12E1OSNXS = AARCH64_TLBI_VMALLS12E1OSNXS,
	ARM64_TLBI_VMALLWS2E1 = AARCH64_TLBI_VMALLWS2E1,
	ARM64_TLBI_VMALLWS2E1IS = AARCH64_TLBI_VMALLWS2E1IS,
	ARM64_TLBI_VMALLWS2E1ISNXS = AARCH64_TLBI_VMALLWS2E1ISNXS,
	ARM64_TLBI_VMALLWS2E1NXS = AARCH64_TLBI_VMALLWS2E1NXS,
	ARM64_TLBI_VMALLWS2E1OS = AARCH64_TLBI_VMALLWS2E1OS,
	ARM64_TLBI_VMALLWS2E1OSNXS = AARCH64_TLBI_VMALLWS2E1OSNXS,

	ARM64_TLBI_ENDING = AARCH64_TLBI_ENDING,
} arm64_tlbi;

typedef enum {

	ARM64_AT_S12E0R = AARCH64_AT_S12E0R,
	ARM64_AT_S12E0W = AARCH64_AT_S12E0W,
	ARM64_AT_S12E1R = AARCH64_AT_S12E1R,
	ARM64_AT_S12E1W = AARCH64_AT_S12E1W,
	ARM64_AT_S1E0R = AARCH64_AT_S1E0R,
	ARM64_AT_S1E0W = AARCH64_AT_S1E0W,
	ARM64_AT_S1E1A = AARCH64_AT_S1E1A,
	ARM64_AT_S1E1R = AARCH64_AT_S1E1R,
	ARM64_AT_S1E1RP = AARCH64_AT_S1E1RP,
	ARM64_AT_S1E1W = AARCH64_AT_S1E1W,
	ARM64_AT_S1E1WP = AARCH64_AT_S1E1WP,
	ARM64_AT_S1E2A = AARCH64_AT_S1E2A,
	ARM64_AT_S1E2R = AARCH64_AT_S1E2R,
	ARM64_AT_S1E2W = AARCH64_AT_S1E2W,
	ARM64_AT_S1E3A = AARCH64_AT_S1E3A,
	ARM64_AT_S1E3R = AARCH64_AT_S1E3R,
	ARM64_AT_S1E3W = AARCH64_AT_S1E3W,

	ARM64_AT_ENDING = AARCH64_AT_ENDING,
} arm64_at;

typedef enum {

	ARM64_BTI_C = AARCH64_BTI_C,
	ARM64_BTI_J = AARCH64_BTI_J,
	ARM64_BTI_JC = AARCH64_BTI_JC,

	ARM64_BTI_ENDING = AARCH64_BTI_ENDING,
} arm64_bti;

typedef enum {

	ARM64_DB_ISH = AARCH64_DB_ISH,
	ARM64_DB_ISHLD = AARCH64_DB_ISHLD,
	ARM64_DB_ISHST = AARCH64_DB_ISHST,
	ARM64_DB_LD = AARCH64_DB_LD,
	ARM64_DB_NSH = AARCH64_DB_NSH,
	ARM64_DB_NSHLD = AARCH64_DB_NSHLD,
	ARM64_DB_NSHST = AARCH64_DB_NSHST,
	ARM64_DB_OSH = AARCH64_DB_OSH,
	ARM64_DB_OSHLD = AARCH64_DB_OSHLD,
	ARM64_DB_OSHST = AARCH64_DB_OSHST,
	ARM64_DB_ST = AARCH64_DB_ST,
	ARM64_DB_SY = AARCH64_DB_SY,

	ARM64_DB_ENDING = AARCH64_DB_ENDING,
} arm64_db;

typedef enum {

	ARM64_DBNXS_ISHNXS = AARCH64_DBNXS_ISHNXS,
	ARM64_DBNXS_NSHNXS = AARCH64_DBNXS_NSHNXS,
	ARM64_DBNXS_OSHNXS = AARCH64_DBNXS_OSHNXS,
	ARM64_DBNXS_SYNXS = AARCH64_DBNXS_SYNXS,

	ARM64_DBNXS_ENDING = AARCH64_DBNXS_ENDING,
} arm64_dbnxs;

typedef enum {

	ARM64_DC_CGDSW = AARCH64_DC_CGDSW,
	ARM64_DC_CGDVAC = AARCH64_DC_CGDVAC,
	ARM64_DC_CGDVADP = AARCH64_DC_CGDVADP,
	ARM64_DC_CGDVAP = AARCH64_DC_CGDVAP,
	ARM64_DC_CGSW = AARCH64_DC_CGSW,
	ARM64_DC_CGVAC = AARCH64_DC_CGVAC,
	ARM64_DC_CGVADP = AARCH64_DC_CGVADP,
	ARM64_DC_CGVAP = AARCH64_DC_CGVAP,
	ARM64_DC_CIGDPAE = AARCH64_DC_CIGDPAE,
	ARM64_DC_CIGDSW = AARCH64_DC_CIGDSW,
	ARM64_DC_CIGDVAC = AARCH64_DC_CIGDVAC,
	ARM64_DC_CIGSW = AARCH64_DC_CIGSW,
	ARM64_DC_CIGVAC = AARCH64_DC_CIGVAC,
	ARM64_DC_CIPAE = AARCH64_DC_CIPAE,
	ARM64_DC_CISW = AARCH64_DC_CISW,
	ARM64_DC_CIVAC = AARCH64_DC_CIVAC,
	ARM64_DC_CSW = AARCH64_DC_CSW,
	ARM64_DC_CVAC = AARCH64_DC_CVAC,
	ARM64_DC_CVADP = AARCH64_DC_CVADP,
	ARM64_DC_CVAP = AARCH64_DC_CVAP,
	ARM64_DC_CVAU = AARCH64_DC_CVAU,
	ARM64_DC_GVA = AARCH64_DC_GVA,
	ARM64_DC_GZVA = AARCH64_DC_GZVA,
	ARM64_DC_IGDSW = AARCH64_DC_IGDSW,
	ARM64_DC_IGDVAC = AARCH64_DC_IGDVAC,
	ARM64_DC_IGSW = AARCH64_DC_IGSW,
	ARM64_DC_IGVAC = AARCH64_DC_IGVAC,
	ARM64_DC_ISW = AARCH64_DC_ISW,
	ARM64_DC_IVAC = AARCH64_DC_IVAC,
	ARM64_DC_ZVA = AARCH64_DC_ZVA,

	ARM64_DC_ENDING = AARCH64_DC_ENDING,
} arm64_dc;

typedef enum {

	ARM64_EXACTFPIMM_HALF = AARCH64_EXACTFPIMM_HALF,
	ARM64_EXACTFPIMM_ONE = AARCH64_EXACTFPIMM_ONE,
	ARM64_EXACTFPIMM_TWO = AARCH64_EXACTFPIMM_TWO,
	ARM64_EXACTFPIMM_ZERO = AARCH64_EXACTFPIMM_ZERO,

	ARM64_EXACTFPIMM_INVALID = AARCH64_EXACTFPIMM_INVALID,

	ARM64_EXACTFPIMM_ENDING = AARCH64_EXACTFPIMM_ENDING,
} arm64_exactfpimm;

typedef enum {

	ARM64_IC_IALLU = AARCH64_IC_IALLU,
	ARM64_IC_IALLUIS = AARCH64_IC_IALLUIS,
	ARM64_IC_IVAU = AARCH64_IC_IVAU,

	ARM64_IC_ENDING = AARCH64_IC_ENDING,
} arm64_ic;

typedef enum {

	ARM64_ISB_SY = AARCH64_ISB_SY,

	ARM64_ISB_ENDING = AARCH64_ISB_ENDING,
} arm64_isb;

typedef enum {

	ARM64_PRFM_PLDL1KEEP = AARCH64_PRFM_PLDL1KEEP,
	ARM64_PRFM_PLDL1STRM = AARCH64_PRFM_PLDL1STRM,
	ARM64_PRFM_PLDL2KEEP = AARCH64_PRFM_PLDL2KEEP,
	ARM64_PRFM_PLDL2STRM = AARCH64_PRFM_PLDL2STRM,
	ARM64_PRFM_PLDL3KEEP = AARCH64_PRFM_PLDL3KEEP,
	ARM64_PRFM_PLDL3STRM = AARCH64_PRFM_PLDL3STRM,
	ARM64_PRFM_PLDSLCKEEP = AARCH64_PRFM_PLDSLCKEEP,
	ARM64_PRFM_PLDSLCSTRM = AARCH64_PRFM_PLDSLCSTRM,
	ARM64_PRFM_PLIL1KEEP = AARCH64_PRFM_PLIL1KEEP,
	ARM64_PRFM_PLIL1STRM = AARCH64_PRFM_PLIL1STRM,
	ARM64_PRFM_PLIL2KEEP = AARCH64_PRFM_PLIL2KEEP,
	ARM64_PRFM_PLIL2STRM = AARCH64_PRFM_PLIL2STRM,
	ARM64_PRFM_PLIL3KEEP = AARCH64_PRFM_PLIL3KEEP,
	ARM64_PRFM_PLIL3STRM = AARCH64_PRFM_PLIL3STRM,
	ARM64_PRFM_PLISLCKEEP = AARCH64_PRFM_PLISLCKEEP,
	ARM64_PRFM_PLISLCSTRM = AARCH64_PRFM_PLISLCSTRM,
	ARM64_PRFM_PSTL1KEEP = AARCH64_PRFM_PSTL1KEEP,
	ARM64_PRFM_PSTL1STRM = AARCH64_PRFM_PSTL1STRM,
	ARM64_PRFM_PSTL2KEEP = AARCH64_PRFM_PSTL2KEEP,
	ARM64_PRFM_PSTL2STRM = AARCH64_PRFM_PSTL2STRM,
	ARM64_PRFM_PSTL3KEEP = AARCH64_PRFM_PSTL3KEEP,
	ARM64_PRFM_PSTL3STRM = AARCH64_PRFM_PSTL3STRM,
	ARM64_PRFM_PSTSLCKEEP = AARCH64_PRFM_PSTSLCKEEP,
	ARM64_PRFM_PSTSLCSTRM = AARCH64_PRFM_PSTSLCSTRM,

	ARM64_PRFM_ENDING = AARCH64_PRFM_ENDING,
} arm64_prfm;

typedef enum {

	ARM64_PSB_CSYNC = AARCH64_PSB_CSYNC,

	ARM64_PSB_ENDING = AARCH64_PSB_ENDING,
} arm64_psb;

typedef enum {

	ARM64_PSTATEIMM0_1_ALLINT = AARCH64_PSTATEIMM0_1_ALLINT,
	ARM64_PSTATEIMM0_1_PM = AARCH64_PSTATEIMM0_1_PM,

	ARM64_PSTATEIMM0_1_ENDING = AARCH64_PSTATEIMM0_1_ENDING,
} arm64_pstateimm0_1;

typedef enum {

	ARM64_PSTATEIMM0_15_DAIFCLR = AARCH64_PSTATEIMM0_15_DAIFCLR,
	ARM64_PSTATEIMM0_15_DAIFSET = AARCH64_PSTATEIMM0_15_DAIFSET,
	ARM64_PSTATEIMM0_15_DIT = AARCH64_PSTATEIMM0_15_DIT,
	ARM64_PSTATEIMM0_15_PAN = AARCH64_PSTATEIMM0_15_PAN,
	ARM64_PSTATEIMM0_15_SPSEL = AARCH64_PSTATEIMM0_15_SPSEL,
	ARM64_PSTATEIMM0_15_SSBS = AARCH64_PSTATEIMM0_15_SSBS,
	ARM64_PSTATEIMM0_15_TCO = AARCH64_PSTATEIMM0_15_TCO,
	ARM64_PSTATEIMM0_15_UAO = AARCH64_PSTATEIMM0_15_UAO,

	ARM64_PSTATEIMM0_15_ENDING = AARCH64_PSTATEIMM0_15_ENDING,
} arm64_pstateimm0_15;

typedef enum {

	ARM64_RPRFM_PLDKEEP = AARCH64_RPRFM_PLDKEEP,
	ARM64_RPRFM_PLDSTRM = AARCH64_RPRFM_PLDSTRM,
	ARM64_RPRFM_PSTKEEP = AARCH64_RPRFM_PSTKEEP,
	ARM64_RPRFM_PSTSTRM = AARCH64_RPRFM_PSTSTRM,

	ARM64_RPRFM_ENDING = AARCH64_RPRFM_ENDING,
} arm64_rprfm;

typedef enum {

	ARM64_SVCR_SVCRSM = AARCH64_SVCR_SVCRSM,
	ARM64_SVCR_SVCRSMZA = AARCH64_SVCR_SVCRSMZA,
	ARM64_SVCR_SVCRZA = AARCH64_SVCR_SVCRZA,

	ARM64_SVCR_ENDING = AARCH64_SVCR_ENDING,
} arm64_svcr;

typedef enum {

	ARM64_SVEPREDPAT_ALL = AARCH64_SVEPREDPAT_ALL,
	ARM64_SVEPREDPAT_MUL3 = AARCH64_SVEPREDPAT_MUL3,
	ARM64_SVEPREDPAT_MUL4 = AARCH64_SVEPREDPAT_MUL4,
	ARM64_SVEPREDPAT_POW2 = AARCH64_SVEPREDPAT_POW2,
	ARM64_SVEPREDPAT_VL1 = AARCH64_SVEPREDPAT_VL1,
	ARM64_SVEPREDPAT_VL128 = AARCH64_SVEPREDPAT_VL128,
	ARM64_SVEPREDPAT_VL16 = AARCH64_SVEPREDPAT_VL16,
	ARM64_SVEPREDPAT_VL2 = AARCH64_SVEPREDPAT_VL2,
	ARM64_SVEPREDPAT_VL256 = AARCH64_SVEPREDPAT_VL256,
	ARM64_SVEPREDPAT_VL3 = AARCH64_SVEPREDPAT_VL3,
	ARM64_SVEPREDPAT_VL32 = AARCH64_SVEPREDPAT_VL32,
	ARM64_SVEPREDPAT_VL4 = AARCH64_SVEPREDPAT_VL4,
	ARM64_SVEPREDPAT_VL5 = AARCH64_SVEPREDPAT_VL5,
	ARM64_SVEPREDPAT_VL6 = AARCH64_SVEPREDPAT_VL6,
	ARM64_SVEPREDPAT_VL64 = AARCH64_SVEPREDPAT_VL64,
	ARM64_SVEPREDPAT_VL7 = AARCH64_SVEPREDPAT_VL7,
	ARM64_SVEPREDPAT_VL8 = AARCH64_SVEPREDPAT_VL8,

	ARM64_SVEPREDPAT_ENDING = AARCH64_SVEPREDPAT_ENDING,
} arm64_svepredpat;

typedef enum {

	ARM64_SVEPRFM_PLDL1KEEP = AARCH64_SVEPRFM_PLDL1KEEP,
	ARM64_SVEPRFM_PLDL1STRM = AARCH64_SVEPRFM_PLDL1STRM,
	ARM64_SVEPRFM_PLDL2KEEP = AARCH64_SVEPRFM_PLDL2KEEP,
	ARM64_SVEPRFM_PLDL2STRM = AARCH64_SVEPRFM_PLDL2STRM,
	ARM64_SVEPRFM_PLDL3KEEP = AARCH64_SVEPRFM_PLDL3KEEP,
	ARM64_SVEPRFM_PLDL3STRM = AARCH64_SVEPRFM_PLDL3STRM,
	ARM64_SVEPRFM_PSTL1KEEP = AARCH64_SVEPRFM_PSTL1KEEP,
	ARM64_SVEPRFM_PSTL1STRM = AARCH64_SVEPRFM_PSTL1STRM,
	ARM64_SVEPRFM_PSTL2KEEP = AARCH64_SVEPRFM_PSTL2KEEP,
	ARM64_SVEPRFM_PSTL2STRM = AARCH64_SVEPRFM_PSTL2STRM,
	ARM64_SVEPRFM_PSTL3KEEP = AARCH64_SVEPRFM_PSTL3KEEP,
	ARM64_SVEPRFM_PSTL3STRM = AARCH64_SVEPRFM_PSTL3STRM,

	ARM64_SVEPRFM_ENDING = AARCH64_SVEPRFM_ENDING,
} arm64_sveprfm;

typedef enum {

	ARM64_SVEVECLENSPECIFIER_VLX2 = AARCH64_SVEVECLENSPECIFIER_VLX2,
	ARM64_SVEVECLENSPECIFIER_VLX4 = AARCH64_SVEVECLENSPECIFIER_VLX4,

	ARM64_SVEVECLENSPECIFIER_ENDING = AARCH64_SVEVECLENSPECIFIER_ENDING,
} arm64_sveveclenspecifier;

typedef enum {
	ARM64_SYSREG_INVALID = AARCH64_SYSREG_INVALID,

	ARM64_SYSREG_ACCDATA_EL1 = AARCH64_SYSREG_ACCDATA_EL1,
	ARM64_SYSREG_ACTLR_EL1 = AARCH64_SYSREG_ACTLR_EL1,
	ARM64_SYSREG_ACTLR_EL2 = AARCH64_SYSREG_ACTLR_EL2,
	ARM64_SYSREG_ACTLR_EL3 = AARCH64_SYSREG_ACTLR_EL3,
	ARM64_SYSREG_AFSR0_EL1 = AARCH64_SYSREG_AFSR0_EL1,
	ARM64_SYSREG_AFSR0_EL12 = AARCH64_SYSREG_AFSR0_EL12,
	ARM64_SYSREG_AFSR0_EL2 = AARCH64_SYSREG_AFSR0_EL2,
	ARM64_SYSREG_AFSR0_EL3 = AARCH64_SYSREG_AFSR0_EL3,
	ARM64_SYSREG_AFSR1_EL1 = AARCH64_SYSREG_AFSR1_EL1,
	ARM64_SYSREG_AFSR1_EL12 = AARCH64_SYSREG_AFSR1_EL12,
	ARM64_SYSREG_AFSR1_EL2 = AARCH64_SYSREG_AFSR1_EL2,
	ARM64_SYSREG_AFSR1_EL3 = AARCH64_SYSREG_AFSR1_EL3,
	ARM64_SYSREG_AIDR_EL1 = AARCH64_SYSREG_AIDR_EL1,
	ARM64_SYSREG_ALLINT = AARCH64_SYSREG_ALLINT,
	ARM64_SYSREG_AMAIR2_EL1 = AARCH64_SYSREG_AMAIR2_EL1,
	ARM64_SYSREG_AMAIR2_EL12 = AARCH64_SYSREG_AMAIR2_EL12,
	ARM64_SYSREG_AMAIR2_EL2 = AARCH64_SYSREG_AMAIR2_EL2,
	ARM64_SYSREG_AMAIR2_EL3 = AARCH64_SYSREG_AMAIR2_EL3,
	ARM64_SYSREG_AMAIR_EL1 = AARCH64_SYSREG_AMAIR_EL1,
	ARM64_SYSREG_AMAIR_EL12 = AARCH64_SYSREG_AMAIR_EL12,
	ARM64_SYSREG_AMAIR_EL2 = AARCH64_SYSREG_AMAIR_EL2,
	ARM64_SYSREG_AMAIR_EL3 = AARCH64_SYSREG_AMAIR_EL3,
	ARM64_SYSREG_AMCFGR_EL0 = AARCH64_SYSREG_AMCFGR_EL0,
	ARM64_SYSREG_AMCG1IDR_EL0 = AARCH64_SYSREG_AMCG1IDR_EL0,
	ARM64_SYSREG_AMCGCR_EL0 = AARCH64_SYSREG_AMCGCR_EL0,
	ARM64_SYSREG_AMCNTENCLR0_EL0 = AARCH64_SYSREG_AMCNTENCLR0_EL0,
	ARM64_SYSREG_AMCNTENCLR1_EL0 = AARCH64_SYSREG_AMCNTENCLR1_EL0,
	ARM64_SYSREG_AMCNTENSET0_EL0 = AARCH64_SYSREG_AMCNTENSET0_EL0,
	ARM64_SYSREG_AMCNTENSET1_EL0 = AARCH64_SYSREG_AMCNTENSET1_EL0,
	ARM64_SYSREG_AMCR_EL0 = AARCH64_SYSREG_AMCR_EL0,
	ARM64_SYSREG_AMEVCNTR00_EL0 = AARCH64_SYSREG_AMEVCNTR00_EL0,
	ARM64_SYSREG_AMEVCNTR01_EL0 = AARCH64_SYSREG_AMEVCNTR01_EL0,
	ARM64_SYSREG_AMEVCNTR02_EL0 = AARCH64_SYSREG_AMEVCNTR02_EL0,
	ARM64_SYSREG_AMEVCNTR03_EL0 = AARCH64_SYSREG_AMEVCNTR03_EL0,
	ARM64_SYSREG_AMEVCNTR10_EL0 = AARCH64_SYSREG_AMEVCNTR10_EL0,
	ARM64_SYSREG_AMEVCNTR110_EL0 = AARCH64_SYSREG_AMEVCNTR110_EL0,
	ARM64_SYSREG_AMEVCNTR111_EL0 = AARCH64_SYSREG_AMEVCNTR111_EL0,
	ARM64_SYSREG_AMEVCNTR112_EL0 = AARCH64_SYSREG_AMEVCNTR112_EL0,
	ARM64_SYSREG_AMEVCNTR113_EL0 = AARCH64_SYSREG_AMEVCNTR113_EL0,
	ARM64_SYSREG_AMEVCNTR114_EL0 = AARCH64_SYSREG_AMEVCNTR114_EL0,
	ARM64_SYSREG_AMEVCNTR115_EL0 = AARCH64_SYSREG_AMEVCNTR115_EL0,
	ARM64_SYSREG_AMEVCNTR11_EL0 = AARCH64_SYSREG_AMEVCNTR11_EL0,
	ARM64_SYSREG_AMEVCNTR12_EL0 = AARCH64_SYSREG_AMEVCNTR12_EL0,
	ARM64_SYSREG_AMEVCNTR13_EL0 = AARCH64_SYSREG_AMEVCNTR13_EL0,
	ARM64_SYSREG_AMEVCNTR14_EL0 = AARCH64_SYSREG_AMEVCNTR14_EL0,
	ARM64_SYSREG_AMEVCNTR15_EL0 = AARCH64_SYSREG_AMEVCNTR15_EL0,
	ARM64_SYSREG_AMEVCNTR16_EL0 = AARCH64_SYSREG_AMEVCNTR16_EL0,
	ARM64_SYSREG_AMEVCNTR17_EL0 = AARCH64_SYSREG_AMEVCNTR17_EL0,
	ARM64_SYSREG_AMEVCNTR18_EL0 = AARCH64_SYSREG_AMEVCNTR18_EL0,
	ARM64_SYSREG_AMEVCNTR19_EL0 = AARCH64_SYSREG_AMEVCNTR19_EL0,
	ARM64_SYSREG_AMEVCNTVOFF00_EL2 = AARCH64_SYSREG_AMEVCNTVOFF00_EL2,
	ARM64_SYSREG_AMEVCNTVOFF010_EL2 = AARCH64_SYSREG_AMEVCNTVOFF010_EL2,
	ARM64_SYSREG_AMEVCNTVOFF011_EL2 = AARCH64_SYSREG_AMEVCNTVOFF011_EL2,
	ARM64_SYSREG_AMEVCNTVOFF012_EL2 = AARCH64_SYSREG_AMEVCNTVOFF012_EL2,
	ARM64_SYSREG_AMEVCNTVOFF013_EL2 = AARCH64_SYSREG_AMEVCNTVOFF013_EL2,
	ARM64_SYSREG_AMEVCNTVOFF014_EL2 = AARCH64_SYSREG_AMEVCNTVOFF014_EL2,
	ARM64_SYSREG_AMEVCNTVOFF015_EL2 = AARCH64_SYSREG_AMEVCNTVOFF015_EL2,
	ARM64_SYSREG_AMEVCNTVOFF01_EL2 = AARCH64_SYSREG_AMEVCNTVOFF01_EL2,
	ARM64_SYSREG_AMEVCNTVOFF02_EL2 = AARCH64_SYSREG_AMEVCNTVOFF02_EL2,
	ARM64_SYSREG_AMEVCNTVOFF03_EL2 = AARCH64_SYSREG_AMEVCNTVOFF03_EL2,
	ARM64_SYSREG_AMEVCNTVOFF04_EL2 = AARCH64_SYSREG_AMEVCNTVOFF04_EL2,
	ARM64_SYSREG_AMEVCNTVOFF05_EL2 = AARCH64_SYSREG_AMEVCNTVOFF05_EL2,
	ARM64_SYSREG_AMEVCNTVOFF06_EL2 = AARCH64_SYSREG_AMEVCNTVOFF06_EL2,
	ARM64_SYSREG_AMEVCNTVOFF07_EL2 = AARCH64_SYSREG_AMEVCNTVOFF07_EL2,
	ARM64_SYSREG_AMEVCNTVOFF08_EL2 = AARCH64_SYSREG_AMEVCNTVOFF08_EL2,
	ARM64_SYSREG_AMEVCNTVOFF09_EL2 = AARCH64_SYSREG_AMEVCNTVOFF09_EL2,
	ARM64_SYSREG_AMEVCNTVOFF10_EL2 = AARCH64_SYSREG_AMEVCNTVOFF10_EL2,
	ARM64_SYSREG_AMEVCNTVOFF110_EL2 = AARCH64_SYSREG_AMEVCNTVOFF110_EL2,
	ARM64_SYSREG_AMEVCNTVOFF111_EL2 = AARCH64_SYSREG_AMEVCNTVOFF111_EL2,
	ARM64_SYSREG_AMEVCNTVOFF112_EL2 = AARCH64_SYSREG_AMEVCNTVOFF112_EL2,
	ARM64_SYSREG_AMEVCNTVOFF113_EL2 = AARCH64_SYSREG_AMEVCNTVOFF113_EL2,
	ARM64_SYSREG_AMEVCNTVOFF114_EL2 = AARCH64_SYSREG_AMEVCNTVOFF114_EL2,
	ARM64_SYSREG_AMEVCNTVOFF115_EL2 = AARCH64_SYSREG_AMEVCNTVOFF115_EL2,
	ARM64_SYSREG_AMEVCNTVOFF11_EL2 = AARCH64_SYSREG_AMEVCNTVOFF11_EL2,
	ARM64_SYSREG_AMEVCNTVOFF12_EL2 = AARCH64_SYSREG_AMEVCNTVOFF12_EL2,
	ARM64_SYSREG_AMEVCNTVOFF13_EL2 = AARCH64_SYSREG_AMEVCNTVOFF13_EL2,
	ARM64_SYSREG_AMEVCNTVOFF14_EL2 = AARCH64_SYSREG_AMEVCNTVOFF14_EL2,
	ARM64_SYSREG_AMEVCNTVOFF15_EL2 = AARCH64_SYSREG_AMEVCNTVOFF15_EL2,
	ARM64_SYSREG_AMEVCNTVOFF16_EL2 = AARCH64_SYSREG_AMEVCNTVOFF16_EL2,
	ARM64_SYSREG_AMEVCNTVOFF17_EL2 = AARCH64_SYSREG_AMEVCNTVOFF17_EL2,
	ARM64_SYSREG_AMEVCNTVOFF18_EL2 = AARCH64_SYSREG_AMEVCNTVOFF18_EL2,
	ARM64_SYSREG_AMEVCNTVOFF19_EL2 = AARCH64_SYSREG_AMEVCNTVOFF19_EL2,
	ARM64_SYSREG_AMEVTYPER00_EL0 = AARCH64_SYSREG_AMEVTYPER00_EL0,
	ARM64_SYSREG_AMEVTYPER01_EL0 = AARCH64_SYSREG_AMEVTYPER01_EL0,
	ARM64_SYSREG_AMEVTYPER02_EL0 = AARCH64_SYSREG_AMEVTYPER02_EL0,
	ARM64_SYSREG_AMEVTYPER03_EL0 = AARCH64_SYSREG_AMEVTYPER03_EL0,
	ARM64_SYSREG_AMEVTYPER10_EL0 = AARCH64_SYSREG_AMEVTYPER10_EL0,
	ARM64_SYSREG_AMEVTYPER110_EL0 = AARCH64_SYSREG_AMEVTYPER110_EL0,
	ARM64_SYSREG_AMEVTYPER111_EL0 = AARCH64_SYSREG_AMEVTYPER111_EL0,
	ARM64_SYSREG_AMEVTYPER112_EL0 = AARCH64_SYSREG_AMEVTYPER112_EL0,
	ARM64_SYSREG_AMEVTYPER113_EL0 = AARCH64_SYSREG_AMEVTYPER113_EL0,
	ARM64_SYSREG_AMEVTYPER114_EL0 = AARCH64_SYSREG_AMEVTYPER114_EL0,
	ARM64_SYSREG_AMEVTYPER115_EL0 = AARCH64_SYSREG_AMEVTYPER115_EL0,
	ARM64_SYSREG_AMEVTYPER11_EL0 = AARCH64_SYSREG_AMEVTYPER11_EL0,
	ARM64_SYSREG_AMEVTYPER12_EL0 = AARCH64_SYSREG_AMEVTYPER12_EL0,
	ARM64_SYSREG_AMEVTYPER13_EL0 = AARCH64_SYSREG_AMEVTYPER13_EL0,
	ARM64_SYSREG_AMEVTYPER14_EL0 = AARCH64_SYSREG_AMEVTYPER14_EL0,
	ARM64_SYSREG_AMEVTYPER15_EL0 = AARCH64_SYSREG_AMEVTYPER15_EL0,
	ARM64_SYSREG_AMEVTYPER16_EL0 = AARCH64_SYSREG_AMEVTYPER16_EL0,
	ARM64_SYSREG_AMEVTYPER17_EL0 = AARCH64_SYSREG_AMEVTYPER17_EL0,
	ARM64_SYSREG_AMEVTYPER18_EL0 = AARCH64_SYSREG_AMEVTYPER18_EL0,
	ARM64_SYSREG_AMEVTYPER19_EL0 = AARCH64_SYSREG_AMEVTYPER19_EL0,
	ARM64_SYSREG_AMUSERENR_EL0 = AARCH64_SYSREG_AMUSERENR_EL0,
	ARM64_SYSREG_APDAKEYHI_EL1 = AARCH64_SYSREG_APDAKEYHI_EL1,
	ARM64_SYSREG_APDAKEYLO_EL1 = AARCH64_SYSREG_APDAKEYLO_EL1,
	ARM64_SYSREG_APDBKEYHI_EL1 = AARCH64_SYSREG_APDBKEYHI_EL1,
	ARM64_SYSREG_APDBKEYLO_EL1 = AARCH64_SYSREG_APDBKEYLO_EL1,
	ARM64_SYSREG_APGAKEYHI_EL1 = AARCH64_SYSREG_APGAKEYHI_EL1,
	ARM64_SYSREG_APGAKEYLO_EL1 = AARCH64_SYSREG_APGAKEYLO_EL1,
	ARM64_SYSREG_APIAKEYHI_EL1 = AARCH64_SYSREG_APIAKEYHI_EL1,
	ARM64_SYSREG_APIAKEYLO_EL1 = AARCH64_SYSREG_APIAKEYLO_EL1,
	ARM64_SYSREG_APIBKEYHI_EL1 = AARCH64_SYSREG_APIBKEYHI_EL1,
	ARM64_SYSREG_APIBKEYLO_EL1 = AARCH64_SYSREG_APIBKEYLO_EL1,
	ARM64_SYSREG_BRBCR_EL1 = AARCH64_SYSREG_BRBCR_EL1,
	ARM64_SYSREG_BRBCR_EL12 = AARCH64_SYSREG_BRBCR_EL12,
	ARM64_SYSREG_BRBCR_EL2 = AARCH64_SYSREG_BRBCR_EL2,
	ARM64_SYSREG_BRBFCR_EL1 = AARCH64_SYSREG_BRBFCR_EL1,
	ARM64_SYSREG_BRBIDR0_EL1 = AARCH64_SYSREG_BRBIDR0_EL1,
	ARM64_SYSREG_BRBINF0_EL1 = AARCH64_SYSREG_BRBINF0_EL1,
	ARM64_SYSREG_BRBINF10_EL1 = AARCH64_SYSREG_BRBINF10_EL1,
	ARM64_SYSREG_BRBINF11_EL1 = AARCH64_SYSREG_BRBINF11_EL1,
	ARM64_SYSREG_BRBINF12_EL1 = AARCH64_SYSREG_BRBINF12_EL1,
	ARM64_SYSREG_BRBINF13_EL1 = AARCH64_SYSREG_BRBINF13_EL1,
	ARM64_SYSREG_BRBINF14_EL1 = AARCH64_SYSREG_BRBINF14_EL1,
	ARM64_SYSREG_BRBINF15_EL1 = AARCH64_SYSREG_BRBINF15_EL1,
	ARM64_SYSREG_BRBINF16_EL1 = AARCH64_SYSREG_BRBINF16_EL1,
	ARM64_SYSREG_BRBINF17_EL1 = AARCH64_SYSREG_BRBINF17_EL1,
	ARM64_SYSREG_BRBINF18_EL1 = AARCH64_SYSREG_BRBINF18_EL1,
	ARM64_SYSREG_BRBINF19_EL1 = AARCH64_SYSREG_BRBINF19_EL1,
	ARM64_SYSREG_BRBINF1_EL1 = AARCH64_SYSREG_BRBINF1_EL1,
	ARM64_SYSREG_BRBINF20_EL1 = AARCH64_SYSREG_BRBINF20_EL1,
	ARM64_SYSREG_BRBINF21_EL1 = AARCH64_SYSREG_BRBINF21_EL1,
	ARM64_SYSREG_BRBINF22_EL1 = AARCH64_SYSREG_BRBINF22_EL1,
	ARM64_SYSREG_BRBINF23_EL1 = AARCH64_SYSREG_BRBINF23_EL1,
	ARM64_SYSREG_BRBINF24_EL1 = AARCH64_SYSREG_BRBINF24_EL1,
	ARM64_SYSREG_BRBINF25_EL1 = AARCH64_SYSREG_BRBINF25_EL1,
	ARM64_SYSREG_BRBINF26_EL1 = AARCH64_SYSREG_BRBINF26_EL1,
	ARM64_SYSREG_BRBINF27_EL1 = AARCH64_SYSREG_BRBINF27_EL1,
	ARM64_SYSREG_BRBINF28_EL1 = AARCH64_SYSREG_BRBINF28_EL1,
	ARM64_SYSREG_BRBINF29_EL1 = AARCH64_SYSREG_BRBINF29_EL1,
	ARM64_SYSREG_BRBINF2_EL1 = AARCH64_SYSREG_BRBINF2_EL1,
	ARM64_SYSREG_BRBINF30_EL1 = AARCH64_SYSREG_BRBINF30_EL1,
	ARM64_SYSREG_BRBINF31_EL1 = AARCH64_SYSREG_BRBINF31_EL1,
	ARM64_SYSREG_BRBINF3_EL1 = AARCH64_SYSREG_BRBINF3_EL1,
	ARM64_SYSREG_BRBINF4_EL1 = AARCH64_SYSREG_BRBINF4_EL1,
	ARM64_SYSREG_BRBINF5_EL1 = AARCH64_SYSREG_BRBINF5_EL1,
	ARM64_SYSREG_BRBINF6_EL1 = AARCH64_SYSREG_BRBINF6_EL1,
	ARM64_SYSREG_BRBINF7_EL1 = AARCH64_SYSREG_BRBINF7_EL1,
	ARM64_SYSREG_BRBINF8_EL1 = AARCH64_SYSREG_BRBINF8_EL1,
	ARM64_SYSREG_BRBINF9_EL1 = AARCH64_SYSREG_BRBINF9_EL1,
	ARM64_SYSREG_BRBINFINJ_EL1 = AARCH64_SYSREG_BRBINFINJ_EL1,
	ARM64_SYSREG_BRBSRC0_EL1 = AARCH64_SYSREG_BRBSRC0_EL1,
	ARM64_SYSREG_BRBSRC10_EL1 = AARCH64_SYSREG_BRBSRC10_EL1,
	ARM64_SYSREG_BRBSRC11_EL1 = AARCH64_SYSREG_BRBSRC11_EL1,
	ARM64_SYSREG_BRBSRC12_EL1 = AARCH64_SYSREG_BRBSRC12_EL1,
	ARM64_SYSREG_BRBSRC13_EL1 = AARCH64_SYSREG_BRBSRC13_EL1,
	ARM64_SYSREG_BRBSRC14_EL1 = AARCH64_SYSREG_BRBSRC14_EL1,
	ARM64_SYSREG_BRBSRC15_EL1 = AARCH64_SYSREG_BRBSRC15_EL1,
	ARM64_SYSREG_BRBSRC16_EL1 = AARCH64_SYSREG_BRBSRC16_EL1,
	ARM64_SYSREG_BRBSRC17_EL1 = AARCH64_SYSREG_BRBSRC17_EL1,
	ARM64_SYSREG_BRBSRC18_EL1 = AARCH64_SYSREG_BRBSRC18_EL1,
	ARM64_SYSREG_BRBSRC19_EL1 = AARCH64_SYSREG_BRBSRC19_EL1,
	ARM64_SYSREG_BRBSRC1_EL1 = AARCH64_SYSREG_BRBSRC1_EL1,
	ARM64_SYSREG_BRBSRC20_EL1 = AARCH64_SYSREG_BRBSRC20_EL1,
	ARM64_SYSREG_BRBSRC21_EL1 = AARCH64_SYSREG_BRBSRC21_EL1,
	ARM64_SYSREG_BRBSRC22_EL1 = AARCH64_SYSREG_BRBSRC22_EL1,
	ARM64_SYSREG_BRBSRC23_EL1 = AARCH64_SYSREG_BRBSRC23_EL1,
	ARM64_SYSREG_BRBSRC24_EL1 = AARCH64_SYSREG_BRBSRC24_EL1,
	ARM64_SYSREG_BRBSRC25_EL1 = AARCH64_SYSREG_BRBSRC25_EL1,
	ARM64_SYSREG_BRBSRC26_EL1 = AARCH64_SYSREG_BRBSRC26_EL1,
	ARM64_SYSREG_BRBSRC27_EL1 = AARCH64_SYSREG_BRBSRC27_EL1,
	ARM64_SYSREG_BRBSRC28_EL1 = AARCH64_SYSREG_BRBSRC28_EL1,
	ARM64_SYSREG_BRBSRC29_EL1 = AARCH64_SYSREG_BRBSRC29_EL1,
	ARM64_SYSREG_BRBSRC2_EL1 = AARCH64_SYSREG_BRBSRC2_EL1,
	ARM64_SYSREG_BRBSRC30_EL1 = AARCH64_SYSREG_BRBSRC30_EL1,
	ARM64_SYSREG_BRBSRC31_EL1 = AARCH64_SYSREG_BRBSRC31_EL1,
	ARM64_SYSREG_BRBSRC3_EL1 = AARCH64_SYSREG_BRBSRC3_EL1,
	ARM64_SYSREG_BRBSRC4_EL1 = AARCH64_SYSREG_BRBSRC4_EL1,
	ARM64_SYSREG_BRBSRC5_EL1 = AARCH64_SYSREG_BRBSRC5_EL1,
	ARM64_SYSREG_BRBSRC6_EL1 = AARCH64_SYSREG_BRBSRC6_EL1,
	ARM64_SYSREG_BRBSRC7_EL1 = AARCH64_SYSREG_BRBSRC7_EL1,
	ARM64_SYSREG_BRBSRC8_EL1 = AARCH64_SYSREG_BRBSRC8_EL1,
	ARM64_SYSREG_BRBSRC9_EL1 = AARCH64_SYSREG_BRBSRC9_EL1,
	ARM64_SYSREG_BRBSRCINJ_EL1 = AARCH64_SYSREG_BRBSRCINJ_EL1,
	ARM64_SYSREG_BRBTGT0_EL1 = AARCH64_SYSREG_BRBTGT0_EL1,
	ARM64_SYSREG_BRBTGT10_EL1 = AARCH64_SYSREG_BRBTGT10_EL1,
	ARM64_SYSREG_BRBTGT11_EL1 = AARCH64_SYSREG_BRBTGT11_EL1,
	ARM64_SYSREG_BRBTGT12_EL1 = AARCH64_SYSREG_BRBTGT12_EL1,
	ARM64_SYSREG_BRBTGT13_EL1 = AARCH64_SYSREG_BRBTGT13_EL1,
	ARM64_SYSREG_BRBTGT14_EL1 = AARCH64_SYSREG_BRBTGT14_EL1,
	ARM64_SYSREG_BRBTGT15_EL1 = AARCH64_SYSREG_BRBTGT15_EL1,
	ARM64_SYSREG_BRBTGT16_EL1 = AARCH64_SYSREG_BRBTGT16_EL1,
	ARM64_SYSREG_BRBTGT17_EL1 = AARCH64_SYSREG_BRBTGT17_EL1,
	ARM64_SYSREG_BRBTGT18_EL1 = AARCH64_SYSREG_BRBTGT18_EL1,
	ARM64_SYSREG_BRBTGT19_EL1 = AARCH64_SYSREG_BRBTGT19_EL1,
	ARM64_SYSREG_BRBTGT1_EL1 = AARCH64_SYSREG_BRBTGT1_EL1,
	ARM64_SYSREG_BRBTGT20_EL1 = AARCH64_SYSREG_BRBTGT20_EL1,
	ARM64_SYSREG_BRBTGT21_EL1 = AARCH64_SYSREG_BRBTGT21_EL1,
	ARM64_SYSREG_BRBTGT22_EL1 = AARCH64_SYSREG_BRBTGT22_EL1,
	ARM64_SYSREG_BRBTGT23_EL1 = AARCH64_SYSREG_BRBTGT23_EL1,
	ARM64_SYSREG_BRBTGT24_EL1 = AARCH64_SYSREG_BRBTGT24_EL1,
	ARM64_SYSREG_BRBTGT25_EL1 = AARCH64_SYSREG_BRBTGT25_EL1,
	ARM64_SYSREG_BRBTGT26_EL1 = AARCH64_SYSREG_BRBTGT26_EL1,
	ARM64_SYSREG_BRBTGT27_EL1 = AARCH64_SYSREG_BRBTGT27_EL1,
	ARM64_SYSREG_BRBTGT28_EL1 = AARCH64_SYSREG_BRBTGT28_EL1,
	ARM64_SYSREG_BRBTGT29_EL1 = AARCH64_SYSREG_BRBTGT29_EL1,
	ARM64_SYSREG_BRBTGT2_EL1 = AARCH64_SYSREG_BRBTGT2_EL1,
	ARM64_SYSREG_BRBTGT30_EL1 = AARCH64_SYSREG_BRBTGT30_EL1,
	ARM64_SYSREG_BRBTGT31_EL1 = AARCH64_SYSREG_BRBTGT31_EL1,
	ARM64_SYSREG_BRBTGT3_EL1 = AARCH64_SYSREG_BRBTGT3_EL1,
	ARM64_SYSREG_BRBTGT4_EL1 = AARCH64_SYSREG_BRBTGT4_EL1,
	ARM64_SYSREG_BRBTGT5_EL1 = AARCH64_SYSREG_BRBTGT5_EL1,
	ARM64_SYSREG_BRBTGT6_EL1 = AARCH64_SYSREG_BRBTGT6_EL1,
	ARM64_SYSREG_BRBTGT7_EL1 = AARCH64_SYSREG_BRBTGT7_EL1,
	ARM64_SYSREG_BRBTGT8_EL1 = AARCH64_SYSREG_BRBTGT8_EL1,
	ARM64_SYSREG_BRBTGT9_EL1 = AARCH64_SYSREG_BRBTGT9_EL1,
	ARM64_SYSREG_BRBTGTINJ_EL1 = AARCH64_SYSREG_BRBTGTINJ_EL1,
	ARM64_SYSREG_BRBTS_EL1 = AARCH64_SYSREG_BRBTS_EL1,
	ARM64_SYSREG_CCSIDR2_EL1 = AARCH64_SYSREG_CCSIDR2_EL1,
	ARM64_SYSREG_CCSIDR_EL1 = AARCH64_SYSREG_CCSIDR_EL1,
	ARM64_SYSREG_CLIDR_EL1 = AARCH64_SYSREG_CLIDR_EL1,
	ARM64_SYSREG_CNTFRQ_EL0 = AARCH64_SYSREG_CNTFRQ_EL0,
	ARM64_SYSREG_CNTHCTL_EL2 = AARCH64_SYSREG_CNTHCTL_EL2,
	ARM64_SYSREG_CNTHPS_CTL_EL2 = AARCH64_SYSREG_CNTHPS_CTL_EL2,
	ARM64_SYSREG_CNTHPS_CVAL_EL2 = AARCH64_SYSREG_CNTHPS_CVAL_EL2,
	ARM64_SYSREG_CNTHPS_TVAL_EL2 = AARCH64_SYSREG_CNTHPS_TVAL_EL2,
	ARM64_SYSREG_CNTHP_CTL_EL2 = AARCH64_SYSREG_CNTHP_CTL_EL2,
	ARM64_SYSREG_CNTHP_CVAL_EL2 = AARCH64_SYSREG_CNTHP_CVAL_EL2,
	ARM64_SYSREG_CNTHP_TVAL_EL2 = AARCH64_SYSREG_CNTHP_TVAL_EL2,
	ARM64_SYSREG_CNTHVS_CTL_EL2 = AARCH64_SYSREG_CNTHVS_CTL_EL2,
	ARM64_SYSREG_CNTHVS_CVAL_EL2 = AARCH64_SYSREG_CNTHVS_CVAL_EL2,
	ARM64_SYSREG_CNTHVS_TVAL_EL2 = AARCH64_SYSREG_CNTHVS_TVAL_EL2,
	ARM64_SYSREG_CNTHV_CTL_EL2 = AARCH64_SYSREG_CNTHV_CTL_EL2,
	ARM64_SYSREG_CNTHV_CVAL_EL2 = AARCH64_SYSREG_CNTHV_CVAL_EL2,
	ARM64_SYSREG_CNTHV_TVAL_EL2 = AARCH64_SYSREG_CNTHV_TVAL_EL2,
	ARM64_SYSREG_CNTISCALE_EL2 = AARCH64_SYSREG_CNTISCALE_EL2,
	ARM64_SYSREG_CNTKCTL_EL1 = AARCH64_SYSREG_CNTKCTL_EL1,
	ARM64_SYSREG_CNTKCTL_EL12 = AARCH64_SYSREG_CNTKCTL_EL12,
	ARM64_SYSREG_CNTPCTSS_EL0 = AARCH64_SYSREG_CNTPCTSS_EL0,
	ARM64_SYSREG_CNTPCT_EL0 = AARCH64_SYSREG_CNTPCT_EL0,
	ARM64_SYSREG_CNTPOFF_EL2 = AARCH64_SYSREG_CNTPOFF_EL2,
	ARM64_SYSREG_CNTPS_CTL_EL1 = AARCH64_SYSREG_CNTPS_CTL_EL1,
	ARM64_SYSREG_CNTPS_CVAL_EL1 = AARCH64_SYSREG_CNTPS_CVAL_EL1,
	ARM64_SYSREG_CNTPS_TVAL_EL1 = AARCH64_SYSREG_CNTPS_TVAL_EL1,
	ARM64_SYSREG_CNTP_CTL_EL0 = AARCH64_SYSREG_CNTP_CTL_EL0,
	ARM64_SYSREG_CNTP_CTL_EL02 = AARCH64_SYSREG_CNTP_CTL_EL02,
	ARM64_SYSREG_CNTP_CVAL_EL0 = AARCH64_SYSREG_CNTP_CVAL_EL0,
	ARM64_SYSREG_CNTP_CVAL_EL02 = AARCH64_SYSREG_CNTP_CVAL_EL02,
	ARM64_SYSREG_CNTP_TVAL_EL0 = AARCH64_SYSREG_CNTP_TVAL_EL0,
	ARM64_SYSREG_CNTP_TVAL_EL02 = AARCH64_SYSREG_CNTP_TVAL_EL02,
	ARM64_SYSREG_CNTSCALE_EL2 = AARCH64_SYSREG_CNTSCALE_EL2,
	ARM64_SYSREG_CNTVCTSS_EL0 = AARCH64_SYSREG_CNTVCTSS_EL0,
	ARM64_SYSREG_CNTVCT_EL0 = AARCH64_SYSREG_CNTVCT_EL0,
	ARM64_SYSREG_CNTVFRQ_EL2 = AARCH64_SYSREG_CNTVFRQ_EL2,
	ARM64_SYSREG_CNTVOFF_EL2 = AARCH64_SYSREG_CNTVOFF_EL2,
	ARM64_SYSREG_CNTV_CTL_EL0 = AARCH64_SYSREG_CNTV_CTL_EL0,
	ARM64_SYSREG_CNTV_CTL_EL02 = AARCH64_SYSREG_CNTV_CTL_EL02,
	ARM64_SYSREG_CNTV_CVAL_EL0 = AARCH64_SYSREG_CNTV_CVAL_EL0,
	ARM64_SYSREG_CNTV_CVAL_EL02 = AARCH64_SYSREG_CNTV_CVAL_EL02,
	ARM64_SYSREG_CNTV_TVAL_EL0 = AARCH64_SYSREG_CNTV_TVAL_EL0,
	ARM64_SYSREG_CNTV_TVAL_EL02 = AARCH64_SYSREG_CNTV_TVAL_EL02,
	ARM64_SYSREG_CONTEXTIDR_EL1 = AARCH64_SYSREG_CONTEXTIDR_EL1,
	ARM64_SYSREG_CONTEXTIDR_EL12 = AARCH64_SYSREG_CONTEXTIDR_EL12,
	ARM64_SYSREG_CONTEXTIDR_EL2 = AARCH64_SYSREG_CONTEXTIDR_EL2,
	ARM64_SYSREG_CPACR_EL1 = AARCH64_SYSREG_CPACR_EL1,
	ARM64_SYSREG_CPACR_EL12 = AARCH64_SYSREG_CPACR_EL12,
	ARM64_SYSREG_CPM_IOACC_CTL_EL3 = AARCH64_SYSREG_CPM_IOACC_CTL_EL3,
	ARM64_SYSREG_CPTR_EL2 = AARCH64_SYSREG_CPTR_EL2,
	ARM64_SYSREG_CPTR_EL3 = AARCH64_SYSREG_CPTR_EL3,
	ARM64_SYSREG_CSSELR_EL1 = AARCH64_SYSREG_CSSELR_EL1,
	ARM64_SYSREG_CTR_EL0 = AARCH64_SYSREG_CTR_EL0,
	ARM64_SYSREG_CURRENTEL = AARCH64_SYSREG_CURRENTEL,
	ARM64_SYSREG_DACR32_EL2 = AARCH64_SYSREG_DACR32_EL2,
	ARM64_SYSREG_DAIF = AARCH64_SYSREG_DAIF,
	ARM64_SYSREG_DBGAUTHSTATUS_EL1 = AARCH64_SYSREG_DBGAUTHSTATUS_EL1,
	ARM64_SYSREG_DBGBCR0_EL1 = AARCH64_SYSREG_DBGBCR0_EL1,
	ARM64_SYSREG_DBGBCR10_EL1 = AARCH64_SYSREG_DBGBCR10_EL1,
	ARM64_SYSREG_DBGBCR11_EL1 = AARCH64_SYSREG_DBGBCR11_EL1,
	ARM64_SYSREG_DBGBCR12_EL1 = AARCH64_SYSREG_DBGBCR12_EL1,
	ARM64_SYSREG_DBGBCR13_EL1 = AARCH64_SYSREG_DBGBCR13_EL1,
	ARM64_SYSREG_DBGBCR14_EL1 = AARCH64_SYSREG_DBGBCR14_EL1,
	ARM64_SYSREG_DBGBCR15_EL1 = AARCH64_SYSREG_DBGBCR15_EL1,
	ARM64_SYSREG_DBGBCR1_EL1 = AARCH64_SYSREG_DBGBCR1_EL1,
	ARM64_SYSREG_DBGBCR2_EL1 = AARCH64_SYSREG_DBGBCR2_EL1,
	ARM64_SYSREG_DBGBCR3_EL1 = AARCH64_SYSREG_DBGBCR3_EL1,
	ARM64_SYSREG_DBGBCR4_EL1 = AARCH64_SYSREG_DBGBCR4_EL1,
	ARM64_SYSREG_DBGBCR5_EL1 = AARCH64_SYSREG_DBGBCR5_EL1,
	ARM64_SYSREG_DBGBCR6_EL1 = AARCH64_SYSREG_DBGBCR6_EL1,
	ARM64_SYSREG_DBGBCR7_EL1 = AARCH64_SYSREG_DBGBCR7_EL1,
	ARM64_SYSREG_DBGBCR8_EL1 = AARCH64_SYSREG_DBGBCR8_EL1,
	ARM64_SYSREG_DBGBCR9_EL1 = AARCH64_SYSREG_DBGBCR9_EL1,
	ARM64_SYSREG_DBGBVR0_EL1 = AARCH64_SYSREG_DBGBVR0_EL1,
	ARM64_SYSREG_DBGBVR10_EL1 = AARCH64_SYSREG_DBGBVR10_EL1,
	ARM64_SYSREG_DBGBVR11_EL1 = AARCH64_SYSREG_DBGBVR11_EL1,
	ARM64_SYSREG_DBGBVR12_EL1 = AARCH64_SYSREG_DBGBVR12_EL1,
	ARM64_SYSREG_DBGBVR13_EL1 = AARCH64_SYSREG_DBGBVR13_EL1,
	ARM64_SYSREG_DBGBVR14_EL1 = AARCH64_SYSREG_DBGBVR14_EL1,
	ARM64_SYSREG_DBGBVR15_EL1 = AARCH64_SYSREG_DBGBVR15_EL1,
	ARM64_SYSREG_DBGBVR1_EL1 = AARCH64_SYSREG_DBGBVR1_EL1,
	ARM64_SYSREG_DBGBVR2_EL1 = AARCH64_SYSREG_DBGBVR2_EL1,
	ARM64_SYSREG_DBGBVR3_EL1 = AARCH64_SYSREG_DBGBVR3_EL1,
	ARM64_SYSREG_DBGBVR4_EL1 = AARCH64_SYSREG_DBGBVR4_EL1,
	ARM64_SYSREG_DBGBVR5_EL1 = AARCH64_SYSREG_DBGBVR5_EL1,
	ARM64_SYSREG_DBGBVR6_EL1 = AARCH64_SYSREG_DBGBVR6_EL1,
	ARM64_SYSREG_DBGBVR7_EL1 = AARCH64_SYSREG_DBGBVR7_EL1,
	ARM64_SYSREG_DBGBVR8_EL1 = AARCH64_SYSREG_DBGBVR8_EL1,
	ARM64_SYSREG_DBGBVR9_EL1 = AARCH64_SYSREG_DBGBVR9_EL1,
	ARM64_SYSREG_DBGCLAIMCLR_EL1 = AARCH64_SYSREG_DBGCLAIMCLR_EL1,
	ARM64_SYSREG_DBGCLAIMSET_EL1 = AARCH64_SYSREG_DBGCLAIMSET_EL1,
	ARM64_SYSREG_DBGDTRRX_EL0 = AARCH64_SYSREG_DBGDTRRX_EL0,
	ARM64_SYSREG_DBGDTRTX_EL0 = AARCH64_SYSREG_DBGDTRTX_EL0,
	ARM64_SYSREG_DBGDTR_EL0 = AARCH64_SYSREG_DBGDTR_EL0,
	ARM64_SYSREG_DBGPRCR_EL1 = AARCH64_SYSREG_DBGPRCR_EL1,
	ARM64_SYSREG_DBGVCR32_EL2 = AARCH64_SYSREG_DBGVCR32_EL2,
	ARM64_SYSREG_DBGWCR0_EL1 = AARCH64_SYSREG_DBGWCR0_EL1,
	ARM64_SYSREG_DBGWCR10_EL1 = AARCH64_SYSREG_DBGWCR10_EL1,
	ARM64_SYSREG_DBGWCR11_EL1 = AARCH64_SYSREG_DBGWCR11_EL1,
	ARM64_SYSREG_DBGWCR12_EL1 = AARCH64_SYSREG_DBGWCR12_EL1,
	ARM64_SYSREG_DBGWCR13_EL1 = AARCH64_SYSREG_DBGWCR13_EL1,
	ARM64_SYSREG_DBGWCR14_EL1 = AARCH64_SYSREG_DBGWCR14_EL1,
	ARM64_SYSREG_DBGWCR15_EL1 = AARCH64_SYSREG_DBGWCR15_EL1,
	ARM64_SYSREG_DBGWCR1_EL1 = AARCH64_SYSREG_DBGWCR1_EL1,
	ARM64_SYSREG_DBGWCR2_EL1 = AARCH64_SYSREG_DBGWCR2_EL1,
	ARM64_SYSREG_DBGWCR3_EL1 = AARCH64_SYSREG_DBGWCR3_EL1,
	ARM64_SYSREG_DBGWCR4_EL1 = AARCH64_SYSREG_DBGWCR4_EL1,
	ARM64_SYSREG_DBGWCR5_EL1 = AARCH64_SYSREG_DBGWCR5_EL1,
	ARM64_SYSREG_DBGWCR6_EL1 = AARCH64_SYSREG_DBGWCR6_EL1,
	ARM64_SYSREG_DBGWCR7_EL1 = AARCH64_SYSREG_DBGWCR7_EL1,
	ARM64_SYSREG_DBGWCR8_EL1 = AARCH64_SYSREG_DBGWCR8_EL1,
	ARM64_SYSREG_DBGWCR9_EL1 = AARCH64_SYSREG_DBGWCR9_EL1,
	ARM64_SYSREG_DBGWVR0_EL1 = AARCH64_SYSREG_DBGWVR0_EL1,
	ARM64_SYSREG_DBGWVR10_EL1 = AARCH64_SYSREG_DBGWVR10_EL1,
	ARM64_SYSREG_DBGWVR11_EL1 = AARCH64_SYSREG_DBGWVR11_EL1,
	ARM64_SYSREG_DBGWVR12_EL1 = AARCH64_SYSREG_DBGWVR12_EL1,
	ARM64_SYSREG_DBGWVR13_EL1 = AARCH64_SYSREG_DBGWVR13_EL1,
	ARM64_SYSREG_DBGWVR14_EL1 = AARCH64_SYSREG_DBGWVR14_EL1,
	ARM64_SYSREG_DBGWVR15_EL1 = AARCH64_SYSREG_DBGWVR15_EL1,
	ARM64_SYSREG_DBGWVR1_EL1 = AARCH64_SYSREG_DBGWVR1_EL1,
	ARM64_SYSREG_DBGWVR2_EL1 = AARCH64_SYSREG_DBGWVR2_EL1,
	ARM64_SYSREG_DBGWVR3_EL1 = AARCH64_SYSREG_DBGWVR3_EL1,
	ARM64_SYSREG_DBGWVR4_EL1 = AARCH64_SYSREG_DBGWVR4_EL1,
	ARM64_SYSREG_DBGWVR5_EL1 = AARCH64_SYSREG_DBGWVR5_EL1,
	ARM64_SYSREG_DBGWVR6_EL1 = AARCH64_SYSREG_DBGWVR6_EL1,
	ARM64_SYSREG_DBGWVR7_EL1 = AARCH64_SYSREG_DBGWVR7_EL1,
	ARM64_SYSREG_DBGWVR8_EL1 = AARCH64_SYSREG_DBGWVR8_EL1,
	ARM64_SYSREG_DBGWVR9_EL1 = AARCH64_SYSREG_DBGWVR9_EL1,
	ARM64_SYSREG_DCZID_EL0 = AARCH64_SYSREG_DCZID_EL0,
	ARM64_SYSREG_DISR_EL1 = AARCH64_SYSREG_DISR_EL1,
	ARM64_SYSREG_DIT = AARCH64_SYSREG_DIT,
	ARM64_SYSREG_DLR_EL0 = AARCH64_SYSREG_DLR_EL0,
	ARM64_SYSREG_DSPSR_EL0 = AARCH64_SYSREG_DSPSR_EL0,
	ARM64_SYSREG_ELR_EL1 = AARCH64_SYSREG_ELR_EL1,
	ARM64_SYSREG_ELR_EL12 = AARCH64_SYSREG_ELR_EL12,
	ARM64_SYSREG_ELR_EL2 = AARCH64_SYSREG_ELR_EL2,
	ARM64_SYSREG_ELR_EL3 = AARCH64_SYSREG_ELR_EL3,
	ARM64_SYSREG_ERRIDR_EL1 = AARCH64_SYSREG_ERRIDR_EL1,
	ARM64_SYSREG_ERRSELR_EL1 = AARCH64_SYSREG_ERRSELR_EL1,
	ARM64_SYSREG_ERXADDR_EL1 = AARCH64_SYSREG_ERXADDR_EL1,
	ARM64_SYSREG_ERXCTLR_EL1 = AARCH64_SYSREG_ERXCTLR_EL1,
	ARM64_SYSREG_ERXFR_EL1 = AARCH64_SYSREG_ERXFR_EL1,
	ARM64_SYSREG_ERXGSR_EL1 = AARCH64_SYSREG_ERXGSR_EL1,
	ARM64_SYSREG_ERXMISC0_EL1 = AARCH64_SYSREG_ERXMISC0_EL1,
	ARM64_SYSREG_ERXMISC1_EL1 = AARCH64_SYSREG_ERXMISC1_EL1,
	ARM64_SYSREG_ERXMISC2_EL1 = AARCH64_SYSREG_ERXMISC2_EL1,
	ARM64_SYSREG_ERXMISC3_EL1 = AARCH64_SYSREG_ERXMISC3_EL1,
	ARM64_SYSREG_ERXPFGCDN_EL1 = AARCH64_SYSREG_ERXPFGCDN_EL1,
	ARM64_SYSREG_ERXPFGCTL_EL1 = AARCH64_SYSREG_ERXPFGCTL_EL1,
	ARM64_SYSREG_ERXPFGF_EL1 = AARCH64_SYSREG_ERXPFGF_EL1,
	ARM64_SYSREG_ERXSTATUS_EL1 = AARCH64_SYSREG_ERXSTATUS_EL1,
	ARM64_SYSREG_ESR_EL1 = AARCH64_SYSREG_ESR_EL1,
	ARM64_SYSREG_ESR_EL12 = AARCH64_SYSREG_ESR_EL12,
	ARM64_SYSREG_ESR_EL2 = AARCH64_SYSREG_ESR_EL2,
	ARM64_SYSREG_ESR_EL3 = AARCH64_SYSREG_ESR_EL3,
	ARM64_SYSREG_FAR_EL1 = AARCH64_SYSREG_FAR_EL1,
	ARM64_SYSREG_FAR_EL12 = AARCH64_SYSREG_FAR_EL12,
	ARM64_SYSREG_FAR_EL2 = AARCH64_SYSREG_FAR_EL2,
	ARM64_SYSREG_FAR_EL3 = AARCH64_SYSREG_FAR_EL3,
	ARM64_SYSREG_FGWTE3_EL3 = AARCH64_SYSREG_FGWTE3_EL3,
	ARM64_SYSREG_FPCR = AARCH64_SYSREG_FPCR,
	ARM64_SYSREG_FPEXC32_EL2 = AARCH64_SYSREG_FPEXC32_EL2,
	ARM64_SYSREG_FPMR = AARCH64_SYSREG_FPMR,
	ARM64_SYSREG_FPSR = AARCH64_SYSREG_FPSR,
	ARM64_SYSREG_GCR_EL1 = AARCH64_SYSREG_GCR_EL1,
	ARM64_SYSREG_GCSCRE0_EL1 = AARCH64_SYSREG_GCSCRE0_EL1,
	ARM64_SYSREG_GCSCR_EL1 = AARCH64_SYSREG_GCSCR_EL1,
	ARM64_SYSREG_GCSCR_EL12 = AARCH64_SYSREG_GCSCR_EL12,
	ARM64_SYSREG_GCSCR_EL2 = AARCH64_SYSREG_GCSCR_EL2,
	ARM64_SYSREG_GCSCR_EL3 = AARCH64_SYSREG_GCSCR_EL3,
	ARM64_SYSREG_GCSPR_EL0 = AARCH64_SYSREG_GCSPR_EL0,
	ARM64_SYSREG_GCSPR_EL1 = AARCH64_SYSREG_GCSPR_EL1,
	ARM64_SYSREG_GCSPR_EL12 = AARCH64_SYSREG_GCSPR_EL12,
	ARM64_SYSREG_GCSPR_EL2 = AARCH64_SYSREG_GCSPR_EL2,
	ARM64_SYSREG_GCSPR_EL3 = AARCH64_SYSREG_GCSPR_EL3,
	ARM64_SYSREG_GMID_EL1 = AARCH64_SYSREG_GMID_EL1,
	ARM64_SYSREG_GPCCR_EL3 = AARCH64_SYSREG_GPCCR_EL3,
	ARM64_SYSREG_GPTBR_EL3 = AARCH64_SYSREG_GPTBR_EL3,
	ARM64_SYSREG_HACDBSBR_EL2 = AARCH64_SYSREG_HACDBSBR_EL2,
	ARM64_SYSREG_HACDBSCONS_EL2 = AARCH64_SYSREG_HACDBSCONS_EL2,
	ARM64_SYSREG_HACR_EL2 = AARCH64_SYSREG_HACR_EL2,
	ARM64_SYSREG_HAFGRTR_EL2 = AARCH64_SYSREG_HAFGRTR_EL2,
	ARM64_SYSREG_HCRX_EL2 = AARCH64_SYSREG_HCRX_EL2,
	ARM64_SYSREG_HCR_EL2 = AARCH64_SYSREG_HCR_EL2,
	ARM64_SYSREG_HDBSSBR_EL2 = AARCH64_SYSREG_HDBSSBR_EL2,
	ARM64_SYSREG_HDBSSPROD_EL2 = AARCH64_SYSREG_HDBSSPROD_EL2,
	ARM64_SYSREG_HDFGRTR2_EL2 = AARCH64_SYSREG_HDFGRTR2_EL2,
	ARM64_SYSREG_HDFGRTR_EL2 = AARCH64_SYSREG_HDFGRTR_EL2,
	ARM64_SYSREG_HDFGWTR2_EL2 = AARCH64_SYSREG_HDFGWTR2_EL2,
	ARM64_SYSREG_HDFGWTR_EL2 = AARCH64_SYSREG_HDFGWTR_EL2,
	ARM64_SYSREG_HFGITR2_EL2 = AARCH64_SYSREG_HFGITR2_EL2,
	ARM64_SYSREG_HFGITR_EL2 = AARCH64_SYSREG_HFGITR_EL2,
	ARM64_SYSREG_HFGRTR2_EL2 = AARCH64_SYSREG_HFGRTR2_EL2,
	ARM64_SYSREG_HFGRTR_EL2 = AARCH64_SYSREG_HFGRTR_EL2,
	ARM64_SYSREG_HFGWTR2_EL2 = AARCH64_SYSREG_HFGWTR2_EL2,
	ARM64_SYSREG_HFGWTR_EL2 = AARCH64_SYSREG_HFGWTR_EL2,
	ARM64_SYSREG_HPFAR_EL2 = AARCH64_SYSREG_HPFAR_EL2,
	ARM64_SYSREG_HSTR_EL2 = AARCH64_SYSREG_HSTR_EL2,
	ARM64_SYSREG_ICC_AP0R0_EL1 = AARCH64_SYSREG_ICC_AP0R0_EL1,
	ARM64_SYSREG_ICC_AP0R1_EL1 = AARCH64_SYSREG_ICC_AP0R1_EL1,
	ARM64_SYSREG_ICC_AP0R2_EL1 = AARCH64_SYSREG_ICC_AP0R2_EL1,
	ARM64_SYSREG_ICC_AP0R3_EL1 = AARCH64_SYSREG_ICC_AP0R3_EL1,
	ARM64_SYSREG_ICC_AP1R0_EL1 = AARCH64_SYSREG_ICC_AP1R0_EL1,
	ARM64_SYSREG_ICC_AP1R1_EL1 = AARCH64_SYSREG_ICC_AP1R1_EL1,
	ARM64_SYSREG_ICC_AP1R2_EL1 = AARCH64_SYSREG_ICC_AP1R2_EL1,
	ARM64_SYSREG_ICC_AP1R3_EL1 = AARCH64_SYSREG_ICC_AP1R3_EL1,
	ARM64_SYSREG_ICC_ASGI1R_EL1 = AARCH64_SYSREG_ICC_ASGI1R_EL1,
	ARM64_SYSREG_ICC_BPR0_EL1 = AARCH64_SYSREG_ICC_BPR0_EL1,
	ARM64_SYSREG_ICC_BPR1_EL1 = AARCH64_SYSREG_ICC_BPR1_EL1,
	ARM64_SYSREG_ICC_CTLR_EL1 = AARCH64_SYSREG_ICC_CTLR_EL1,
	ARM64_SYSREG_ICC_CTLR_EL3 = AARCH64_SYSREG_ICC_CTLR_EL3,
	ARM64_SYSREG_ICC_DIR_EL1 = AARCH64_SYSREG_ICC_DIR_EL1,
	ARM64_SYSREG_ICC_EOIR0_EL1 = AARCH64_SYSREG_ICC_EOIR0_EL1,
	ARM64_SYSREG_ICC_EOIR1_EL1 = AARCH64_SYSREG_ICC_EOIR1_EL1,
	ARM64_SYSREG_ICC_HPPIR0_EL1 = AARCH64_SYSREG_ICC_HPPIR0_EL1,
	ARM64_SYSREG_ICC_HPPIR1_EL1 = AARCH64_SYSREG_ICC_HPPIR1_EL1,
	ARM64_SYSREG_ICC_IAR0_EL1 = AARCH64_SYSREG_ICC_IAR0_EL1,
	ARM64_SYSREG_ICC_IAR1_EL1 = AARCH64_SYSREG_ICC_IAR1_EL1,
	ARM64_SYSREG_ICC_IGRPEN0_EL1 = AARCH64_SYSREG_ICC_IGRPEN0_EL1,
	ARM64_SYSREG_ICC_IGRPEN1_EL1 = AARCH64_SYSREG_ICC_IGRPEN1_EL1,
	ARM64_SYSREG_ICC_IGRPEN1_EL3 = AARCH64_SYSREG_ICC_IGRPEN1_EL3,
	ARM64_SYSREG_ICC_NMIAR1_EL1 = AARCH64_SYSREG_ICC_NMIAR1_EL1,
	ARM64_SYSREG_ICC_PMR_EL1 = AARCH64_SYSREG_ICC_PMR_EL1,
	ARM64_SYSREG_ICC_RPR_EL1 = AARCH64_SYSREG_ICC_RPR_EL1,
	ARM64_SYSREG_ICC_SGI0R_EL1 = AARCH64_SYSREG_ICC_SGI0R_EL1,
	ARM64_SYSREG_ICC_SGI1R_EL1 = AARCH64_SYSREG_ICC_SGI1R_EL1,
	ARM64_SYSREG_ICC_SRE_EL1 = AARCH64_SYSREG_ICC_SRE_EL1,
	ARM64_SYSREG_ICC_SRE_EL2 = AARCH64_SYSREG_ICC_SRE_EL2,
	ARM64_SYSREG_ICC_SRE_EL3 = AARCH64_SYSREG_ICC_SRE_EL3,
	ARM64_SYSREG_ICH_AP0R0_EL2 = AARCH64_SYSREG_ICH_AP0R0_EL2,
	ARM64_SYSREG_ICH_AP0R1_EL2 = AARCH64_SYSREG_ICH_AP0R1_EL2,
	ARM64_SYSREG_ICH_AP0R2_EL2 = AARCH64_SYSREG_ICH_AP0R2_EL2,
	ARM64_SYSREG_ICH_AP0R3_EL2 = AARCH64_SYSREG_ICH_AP0R3_EL2,
	ARM64_SYSREG_ICH_AP1R0_EL2 = AARCH64_SYSREG_ICH_AP1R0_EL2,
	ARM64_SYSREG_ICH_AP1R1_EL2 = AARCH64_SYSREG_ICH_AP1R1_EL2,
	ARM64_SYSREG_ICH_AP1R2_EL2 = AARCH64_SYSREG_ICH_AP1R2_EL2,
	ARM64_SYSREG_ICH_AP1R3_EL2 = AARCH64_SYSREG_ICH_AP1R3_EL2,
	ARM64_SYSREG_ICH_EISR_EL2 = AARCH64_SYSREG_ICH_EISR_EL2,
	ARM64_SYSREG_ICH_ELRSR_EL2 = AARCH64_SYSREG_ICH_ELRSR_EL2,
	ARM64_SYSREG_ICH_HCR_EL2 = AARCH64_SYSREG_ICH_HCR_EL2,
	ARM64_SYSREG_ICH_LR0_EL2 = AARCH64_SYSREG_ICH_LR0_EL2,
	ARM64_SYSREG_ICH_LR10_EL2 = AARCH64_SYSREG_ICH_LR10_EL2,
	ARM64_SYSREG_ICH_LR11_EL2 = AARCH64_SYSREG_ICH_LR11_EL2,
	ARM64_SYSREG_ICH_LR12_EL2 = AARCH64_SYSREG_ICH_LR12_EL2,
	ARM64_SYSREG_ICH_LR13_EL2 = AARCH64_SYSREG_ICH_LR13_EL2,
	ARM64_SYSREG_ICH_LR14_EL2 = AARCH64_SYSREG_ICH_LR14_EL2,
	ARM64_SYSREG_ICH_LR15_EL2 = AARCH64_SYSREG_ICH_LR15_EL2,
	ARM64_SYSREG_ICH_LR1_EL2 = AARCH64_SYSREG_ICH_LR1_EL2,
	ARM64_SYSREG_ICH_LR2_EL2 = AARCH64_SYSREG_ICH_LR2_EL2,
	ARM64_SYSREG_ICH_LR3_EL2 = AARCH64_SYSREG_ICH_LR3_EL2,
	ARM64_SYSREG_ICH_LR4_EL2 = AARCH64_SYSREG_ICH_LR4_EL2,
	ARM64_SYSREG_ICH_LR5_EL2 = AARCH64_SYSREG_ICH_LR5_EL2,
	ARM64_SYSREG_ICH_LR6_EL2 = AARCH64_SYSREG_ICH_LR6_EL2,
	ARM64_SYSREG_ICH_LR7_EL2 = AARCH64_SYSREG_ICH_LR7_EL2,
	ARM64_SYSREG_ICH_LR8_EL2 = AARCH64_SYSREG_ICH_LR8_EL2,
	ARM64_SYSREG_ICH_LR9_EL2 = AARCH64_SYSREG_ICH_LR9_EL2,
	ARM64_SYSREG_ICH_MISR_EL2 = AARCH64_SYSREG_ICH_MISR_EL2,
	ARM64_SYSREG_ICH_VMCR_EL2 = AARCH64_SYSREG_ICH_VMCR_EL2,
	ARM64_SYSREG_ICH_VTR_EL2 = AARCH64_SYSREG_ICH_VTR_EL2,
	ARM64_SYSREG_ID_AA64AFR0_EL1 = AARCH64_SYSREG_ID_AA64AFR0_EL1,
	ARM64_SYSREG_ID_AA64AFR1_EL1 = AARCH64_SYSREG_ID_AA64AFR1_EL1,
	ARM64_SYSREG_ID_AA64DFR0_EL1 = AARCH64_SYSREG_ID_AA64DFR0_EL1,
	ARM64_SYSREG_ID_AA64DFR1_EL1 = AARCH64_SYSREG_ID_AA64DFR1_EL1,
	ARM64_SYSREG_ID_AA64DFR2_EL1 = AARCH64_SYSREG_ID_AA64DFR2_EL1,
	ARM64_SYSREG_ID_AA64FPFR0_EL1 = AARCH64_SYSREG_ID_AA64FPFR0_EL1,
	ARM64_SYSREG_ID_AA64ISAR0_EL1 = AARCH64_SYSREG_ID_AA64ISAR0_EL1,
	ARM64_SYSREG_ID_AA64ISAR1_EL1 = AARCH64_SYSREG_ID_AA64ISAR1_EL1,
	ARM64_SYSREG_ID_AA64ISAR2_EL1 = AARCH64_SYSREG_ID_AA64ISAR2_EL1,
	ARM64_SYSREG_ID_AA64ISAR3_EL1 = AARCH64_SYSREG_ID_AA64ISAR3_EL1,
	ARM64_SYSREG_ID_AA64MMFR0_EL1 = AARCH64_SYSREG_ID_AA64MMFR0_EL1,
	ARM64_SYSREG_ID_AA64MMFR1_EL1 = AARCH64_SYSREG_ID_AA64MMFR1_EL1,
	ARM64_SYSREG_ID_AA64MMFR2_EL1 = AARCH64_SYSREG_ID_AA64MMFR2_EL1,
	ARM64_SYSREG_ID_AA64MMFR3_EL1 = AARCH64_SYSREG_ID_AA64MMFR3_EL1,
	ARM64_SYSREG_ID_AA64MMFR4_EL1 = AARCH64_SYSREG_ID_AA64MMFR4_EL1,
	ARM64_SYSREG_ID_AA64PFR0_EL1 = AARCH64_SYSREG_ID_AA64PFR0_EL1,
	ARM64_SYSREG_ID_AA64PFR1_EL1 = AARCH64_SYSREG_ID_AA64PFR1_EL1,
	ARM64_SYSREG_ID_AA64PFR2_EL1 = AARCH64_SYSREG_ID_AA64PFR2_EL1,
	ARM64_SYSREG_ID_AA64SMFR0_EL1 = AARCH64_SYSREG_ID_AA64SMFR0_EL1,
	ARM64_SYSREG_ID_AA64ZFR0_EL1 = AARCH64_SYSREG_ID_AA64ZFR0_EL1,
	ARM64_SYSREG_ID_AFR0_EL1 = AARCH64_SYSREG_ID_AFR0_EL1,
	ARM64_SYSREG_ID_DFR0_EL1 = AARCH64_SYSREG_ID_DFR0_EL1,
	ARM64_SYSREG_ID_DFR1_EL1 = AARCH64_SYSREG_ID_DFR1_EL1,
	ARM64_SYSREG_ID_ISAR0_EL1 = AARCH64_SYSREG_ID_ISAR0_EL1,
	ARM64_SYSREG_ID_ISAR1_EL1 = AARCH64_SYSREG_ID_ISAR1_EL1,
	ARM64_SYSREG_ID_ISAR2_EL1 = AARCH64_SYSREG_ID_ISAR2_EL1,
	ARM64_SYSREG_ID_ISAR3_EL1 = AARCH64_SYSREG_ID_ISAR3_EL1,
	ARM64_SYSREG_ID_ISAR4_EL1 = AARCH64_SYSREG_ID_ISAR4_EL1,
	ARM64_SYSREG_ID_ISAR5_EL1 = AARCH64_SYSREG_ID_ISAR5_EL1,
	ARM64_SYSREG_ID_ISAR6_EL1 = AARCH64_SYSREG_ID_ISAR6_EL1,
	ARM64_SYSREG_ID_MMFR0_EL1 = AARCH64_SYSREG_ID_MMFR0_EL1,
	ARM64_SYSREG_ID_MMFR1_EL1 = AARCH64_SYSREG_ID_MMFR1_EL1,
	ARM64_SYSREG_ID_MMFR2_EL1 = AARCH64_SYSREG_ID_MMFR2_EL1,
	ARM64_SYSREG_ID_MMFR3_EL1 = AARCH64_SYSREG_ID_MMFR3_EL1,
	ARM64_SYSREG_ID_MMFR4_EL1 = AARCH64_SYSREG_ID_MMFR4_EL1,
	ARM64_SYSREG_ID_MMFR5_EL1 = AARCH64_SYSREG_ID_MMFR5_EL1,
	ARM64_SYSREG_ID_PFR0_EL1 = AARCH64_SYSREG_ID_PFR0_EL1,
	ARM64_SYSREG_ID_PFR1_EL1 = AARCH64_SYSREG_ID_PFR1_EL1,
	ARM64_SYSREG_ID_PFR2_EL1 = AARCH64_SYSREG_ID_PFR2_EL1,
	ARM64_SYSREG_IFSR32_EL2 = AARCH64_SYSREG_IFSR32_EL2,
	ARM64_SYSREG_ISR_EL1 = AARCH64_SYSREG_ISR_EL1,
	ARM64_SYSREG_LORC_EL1 = AARCH64_SYSREG_LORC_EL1,
	ARM64_SYSREG_LOREA_EL1 = AARCH64_SYSREG_LOREA_EL1,
	ARM64_SYSREG_LORID_EL1 = AARCH64_SYSREG_LORID_EL1,
	ARM64_SYSREG_LORN_EL1 = AARCH64_SYSREG_LORN_EL1,
	ARM64_SYSREG_LORSA_EL1 = AARCH64_SYSREG_LORSA_EL1,
	ARM64_SYSREG_MAIR2_EL1 = AARCH64_SYSREG_MAIR2_EL1,
	ARM64_SYSREG_MAIR2_EL12 = AARCH64_SYSREG_MAIR2_EL12,
	ARM64_SYSREG_MAIR2_EL2 = AARCH64_SYSREG_MAIR2_EL2,
	ARM64_SYSREG_MAIR2_EL3 = AARCH64_SYSREG_MAIR2_EL3,
	ARM64_SYSREG_MAIR_EL1 = AARCH64_SYSREG_MAIR_EL1,
	ARM64_SYSREG_MAIR_EL12 = AARCH64_SYSREG_MAIR_EL12,
	ARM64_SYSREG_MAIR_EL2 = AARCH64_SYSREG_MAIR_EL2,
	ARM64_SYSREG_MAIR_EL3 = AARCH64_SYSREG_MAIR_EL3,
	ARM64_SYSREG_MDCCINT_EL1 = AARCH64_SYSREG_MDCCINT_EL1,
	ARM64_SYSREG_MDCCSR_EL0 = AARCH64_SYSREG_MDCCSR_EL0,
	ARM64_SYSREG_MDCR_EL2 = AARCH64_SYSREG_MDCR_EL2,
	ARM64_SYSREG_MDCR_EL3 = AARCH64_SYSREG_MDCR_EL3,
	ARM64_SYSREG_MDRAR_EL1 = AARCH64_SYSREG_MDRAR_EL1,
	ARM64_SYSREG_MDSCR_EL1 = AARCH64_SYSREG_MDSCR_EL1,
	ARM64_SYSREG_MDSELR_EL1 = AARCH64_SYSREG_MDSELR_EL1,
	ARM64_SYSREG_MDSTEPOP_EL1 = AARCH64_SYSREG_MDSTEPOP_EL1,
	ARM64_SYSREG_MECIDR_EL2 = AARCH64_SYSREG_MECIDR_EL2,
	ARM64_SYSREG_MECID_A0_EL2 = AARCH64_SYSREG_MECID_A0_EL2,
	ARM64_SYSREG_MECID_A1_EL2 = AARCH64_SYSREG_MECID_A1_EL2,
	ARM64_SYSREG_MECID_P0_EL2 = AARCH64_SYSREG_MECID_P0_EL2,
	ARM64_SYSREG_MECID_P1_EL2 = AARCH64_SYSREG_MECID_P1_EL2,
	ARM64_SYSREG_MECID_RL_A_EL3 = AARCH64_SYSREG_MECID_RL_A_EL3,
	ARM64_SYSREG_MFAR_EL3 = AARCH64_SYSREG_MFAR_EL3,
	ARM64_SYSREG_MIDR_EL1 = AARCH64_SYSREG_MIDR_EL1,
	ARM64_SYSREG_MPAM0_EL1 = AARCH64_SYSREG_MPAM0_EL1,
	ARM64_SYSREG_MPAM1_EL1 = AARCH64_SYSREG_MPAM1_EL1,
	ARM64_SYSREG_MPAM1_EL12 = AARCH64_SYSREG_MPAM1_EL12,
	ARM64_SYSREG_MPAM2_EL2 = AARCH64_SYSREG_MPAM2_EL2,
	ARM64_SYSREG_MPAM3_EL3 = AARCH64_SYSREG_MPAM3_EL3,
	ARM64_SYSREG_MPAMHCR_EL2 = AARCH64_SYSREG_MPAMHCR_EL2,
	ARM64_SYSREG_MPAMIDR_EL1 = AARCH64_SYSREG_MPAMIDR_EL1,
	ARM64_SYSREG_MPAMSM_EL1 = AARCH64_SYSREG_MPAMSM_EL1,
	ARM64_SYSREG_MPAMVPM0_EL2 = AARCH64_SYSREG_MPAMVPM0_EL2,
	ARM64_SYSREG_MPAMVPM1_EL2 = AARCH64_SYSREG_MPAMVPM1_EL2,
	ARM64_SYSREG_MPAMVPM2_EL2 = AARCH64_SYSREG_MPAMVPM2_EL2,
	ARM64_SYSREG_MPAMVPM3_EL2 = AARCH64_SYSREG_MPAMVPM3_EL2,
	ARM64_SYSREG_MPAMVPM4_EL2 = AARCH64_SYSREG_MPAMVPM4_EL2,
	ARM64_SYSREG_MPAMVPM5_EL2 = AARCH64_SYSREG_MPAMVPM5_EL2,
	ARM64_SYSREG_MPAMVPM6_EL2 = AARCH64_SYSREG_MPAMVPM6_EL2,
	ARM64_SYSREG_MPAMVPM7_EL2 = AARCH64_SYSREG_MPAMVPM7_EL2,
	ARM64_SYSREG_MPAMVPMV_EL2 = AARCH64_SYSREG_MPAMVPMV_EL2,
	ARM64_SYSREG_MPIDR_EL1 = AARCH64_SYSREG_MPIDR_EL1,
	ARM64_SYSREG_MPUIR_EL1 = AARCH64_SYSREG_MPUIR_EL1,
	ARM64_SYSREG_MPUIR_EL2 = AARCH64_SYSREG_MPUIR_EL2,
	ARM64_SYSREG_MVFR0_EL1 = AARCH64_SYSREG_MVFR0_EL1,
	ARM64_SYSREG_MVFR1_EL1 = AARCH64_SYSREG_MVFR1_EL1,
	ARM64_SYSREG_MVFR2_EL1 = AARCH64_SYSREG_MVFR2_EL1,
	ARM64_SYSREG_NZCV = AARCH64_SYSREG_NZCV,
	ARM64_SYSREG_OSDLR_EL1 = AARCH64_SYSREG_OSDLR_EL1,
	ARM64_SYSREG_OSDTRRX_EL1 = AARCH64_SYSREG_OSDTRRX_EL1,
	ARM64_SYSREG_OSDTRTX_EL1 = AARCH64_SYSREG_OSDTRTX_EL1,
	ARM64_SYSREG_OSECCR_EL1 = AARCH64_SYSREG_OSECCR_EL1,
	ARM64_SYSREG_OSLAR_EL1 = AARCH64_SYSREG_OSLAR_EL1,
	ARM64_SYSREG_OSLSR_EL1 = AARCH64_SYSREG_OSLSR_EL1,
	ARM64_SYSREG_PAN = AARCH64_SYSREG_PAN,
	ARM64_SYSREG_PAR_EL1 = AARCH64_SYSREG_PAR_EL1,
	ARM64_SYSREG_PFAR_EL1 = AARCH64_SYSREG_PFAR_EL1,
	ARM64_SYSREG_PFAR_EL12 = AARCH64_SYSREG_PFAR_EL12,
	ARM64_SYSREG_PFAR_EL2 = AARCH64_SYSREG_PFAR_EL2,
	ARM64_SYSREG_PIRE0_EL1 = AARCH64_SYSREG_PIRE0_EL1,
	ARM64_SYSREG_PIRE0_EL12 = AARCH64_SYSREG_PIRE0_EL12,
	ARM64_SYSREG_PIRE0_EL2 = AARCH64_SYSREG_PIRE0_EL2,
	ARM64_SYSREG_PIR_EL1 = AARCH64_SYSREG_PIR_EL1,
	ARM64_SYSREG_PIR_EL12 = AARCH64_SYSREG_PIR_EL12,
	ARM64_SYSREG_PIR_EL2 = AARCH64_SYSREG_PIR_EL2,
	ARM64_SYSREG_PIR_EL3 = AARCH64_SYSREG_PIR_EL3,
	ARM64_SYSREG_PM = AARCH64_SYSREG_PM,
	ARM64_SYSREG_PMBIDR_EL1 = AARCH64_SYSREG_PMBIDR_EL1,
	ARM64_SYSREG_PMBLIMITR_EL1 = AARCH64_SYSREG_PMBLIMITR_EL1,
	ARM64_SYSREG_PMBPTR_EL1 = AARCH64_SYSREG_PMBPTR_EL1,
	ARM64_SYSREG_PMBSR_EL1 = AARCH64_SYSREG_PMBSR_EL1,
	ARM64_SYSREG_PMCCFILTR_EL0 = AARCH64_SYSREG_PMCCFILTR_EL0,
	ARM64_SYSREG_PMCCNTR_EL0 = AARCH64_SYSREG_PMCCNTR_EL0,
	ARM64_SYSREG_PMCCNTSVR_EL1 = AARCH64_SYSREG_PMCCNTSVR_EL1,
	ARM64_SYSREG_PMCEID0_EL0 = AARCH64_SYSREG_PMCEID0_EL0,
	ARM64_SYSREG_PMCEID1_EL0 = AARCH64_SYSREG_PMCEID1_EL0,
	ARM64_SYSREG_PMCNTENCLR_EL0 = AARCH64_SYSREG_PMCNTENCLR_EL0,
	ARM64_SYSREG_PMCNTENSET_EL0 = AARCH64_SYSREG_PMCNTENSET_EL0,
	ARM64_SYSREG_PMCR_EL0 = AARCH64_SYSREG_PMCR_EL0,
	ARM64_SYSREG_PMECR_EL1 = AARCH64_SYSREG_PMECR_EL1,
	ARM64_SYSREG_PMEVCNTR0_EL0 = AARCH64_SYSREG_PMEVCNTR0_EL0,
	ARM64_SYSREG_PMEVCNTR10_EL0 = AARCH64_SYSREG_PMEVCNTR10_EL0,
	ARM64_SYSREG_PMEVCNTR11_EL0 = AARCH64_SYSREG_PMEVCNTR11_EL0,
	ARM64_SYSREG_PMEVCNTR12_EL0 = AARCH64_SYSREG_PMEVCNTR12_EL0,
	ARM64_SYSREG_PMEVCNTR13_EL0 = AARCH64_SYSREG_PMEVCNTR13_EL0,
	ARM64_SYSREG_PMEVCNTR14_EL0 = AARCH64_SYSREG_PMEVCNTR14_EL0,
	ARM64_SYSREG_PMEVCNTR15_EL0 = AARCH64_SYSREG_PMEVCNTR15_EL0,
	ARM64_SYSREG_PMEVCNTR16_EL0 = AARCH64_SYSREG_PMEVCNTR16_EL0,
	ARM64_SYSREG_PMEVCNTR17_EL0 = AARCH64_SYSREG_PMEVCNTR17_EL0,
	ARM64_SYSREG_PMEVCNTR18_EL0 = AARCH64_SYSREG_PMEVCNTR18_EL0,
	ARM64_SYSREG_PMEVCNTR19_EL0 = AARCH64_SYSREG_PMEVCNTR19_EL0,
	ARM64_SYSREG_PMEVCNTR1_EL0 = AARCH64_SYSREG_PMEVCNTR1_EL0,
	ARM64_SYSREG_PMEVCNTR20_EL0 = AARCH64_SYSREG_PMEVCNTR20_EL0,
	ARM64_SYSREG_PMEVCNTR21_EL0 = AARCH64_SYSREG_PMEVCNTR21_EL0,
	ARM64_SYSREG_PMEVCNTR22_EL0 = AARCH64_SYSREG_PMEVCNTR22_EL0,
	ARM64_SYSREG_PMEVCNTR23_EL0 = AARCH64_SYSREG_PMEVCNTR23_EL0,
	ARM64_SYSREG_PMEVCNTR24_EL0 = AARCH64_SYSREG_PMEVCNTR24_EL0,
	ARM64_SYSREG_PMEVCNTR25_EL0 = AARCH64_SYSREG_PMEVCNTR25_EL0,
	ARM64_SYSREG_PMEVCNTR26_EL0 = AARCH64_SYSREG_PMEVCNTR26_EL0,
	ARM64_SYSREG_PMEVCNTR27_EL0 = AARCH64_SYSREG_PMEVCNTR27_EL0,
	ARM64_SYSREG_PMEVCNTR28_EL0 = AARCH64_SYSREG_PMEVCNTR28_EL0,
	ARM64_SYSREG_PMEVCNTR29_EL0 = AARCH64_SYSREG_PMEVCNTR29_EL0,
	ARM64_SYSREG_PMEVCNTR2_EL0 = AARCH64_SYSREG_PMEVCNTR2_EL0,
	ARM64_SYSREG_PMEVCNTR30_EL0 = AARCH64_SYSREG_PMEVCNTR30_EL0,
	ARM64_SYSREG_PMEVCNTR3_EL0 = AARCH64_SYSREG_PMEVCNTR3_EL0,
	ARM64_SYSREG_PMEVCNTR4_EL0 = AARCH64_SYSREG_PMEVCNTR4_EL0,
	ARM64_SYSREG_PMEVCNTR5_EL0 = AARCH64_SYSREG_PMEVCNTR5_EL0,
	ARM64_SYSREG_PMEVCNTR6_EL0 = AARCH64_SYSREG_PMEVCNTR6_EL0,
	ARM64_SYSREG_PMEVCNTR7_EL0 = AARCH64_SYSREG_PMEVCNTR7_EL0,
	ARM64_SYSREG_PMEVCNTR8_EL0 = AARCH64_SYSREG_PMEVCNTR8_EL0,
	ARM64_SYSREG_PMEVCNTR9_EL0 = AARCH64_SYSREG_PMEVCNTR9_EL0,
	ARM64_SYSREG_PMEVCNTSVR0_EL1 = AARCH64_SYSREG_PMEVCNTSVR0_EL1,
	ARM64_SYSREG_PMEVCNTSVR10_EL1 = AARCH64_SYSREG_PMEVCNTSVR10_EL1,
	ARM64_SYSREG_PMEVCNTSVR11_EL1 = AARCH64_SYSREG_PMEVCNTSVR11_EL1,
	ARM64_SYSREG_PMEVCNTSVR12_EL1 = AARCH64_SYSREG_PMEVCNTSVR12_EL1,
	ARM64_SYSREG_PMEVCNTSVR13_EL1 = AARCH64_SYSREG_PMEVCNTSVR13_EL1,
	ARM64_SYSREG_PMEVCNTSVR14_EL1 = AARCH64_SYSREG_PMEVCNTSVR14_EL1,
	ARM64_SYSREG_PMEVCNTSVR15_EL1 = AARCH64_SYSREG_PMEVCNTSVR15_EL1,
	ARM64_SYSREG_PMEVCNTSVR16_EL1 = AARCH64_SYSREG_PMEVCNTSVR16_EL1,
	ARM64_SYSREG_PMEVCNTSVR17_EL1 = AARCH64_SYSREG_PMEVCNTSVR17_EL1,
	ARM64_SYSREG_PMEVCNTSVR18_EL1 = AARCH64_SYSREG_PMEVCNTSVR18_EL1,
	ARM64_SYSREG_PMEVCNTSVR19_EL1 = AARCH64_SYSREG_PMEVCNTSVR19_EL1,
	ARM64_SYSREG_PMEVCNTSVR1_EL1 = AARCH64_SYSREG_PMEVCNTSVR1_EL1,
	ARM64_SYSREG_PMEVCNTSVR20_EL1 = AARCH64_SYSREG_PMEVCNTSVR20_EL1,
	ARM64_SYSREG_PMEVCNTSVR21_EL1 = AARCH64_SYSREG_PMEVCNTSVR21_EL1,
	ARM64_SYSREG_PMEVCNTSVR22_EL1 = AARCH64_SYSREG_PMEVCNTSVR22_EL1,
	ARM64_SYSREG_PMEVCNTSVR23_EL1 = AARCH64_SYSREG_PMEVCNTSVR23_EL1,
	ARM64_SYSREG_PMEVCNTSVR24_EL1 = AARCH64_SYSREG_PMEVCNTSVR24_EL1,
	ARM64_SYSREG_PMEVCNTSVR25_EL1 = AARCH64_SYSREG_PMEVCNTSVR25_EL1,
	ARM64_SYSREG_PMEVCNTSVR26_EL1 = AARCH64_SYSREG_PMEVCNTSVR26_EL1,
	ARM64_SYSREG_PMEVCNTSVR27_EL1 = AARCH64_SYSREG_PMEVCNTSVR27_EL1,
	ARM64_SYSREG_PMEVCNTSVR28_EL1 = AARCH64_SYSREG_PMEVCNTSVR28_EL1,
	ARM64_SYSREG_PMEVCNTSVR29_EL1 = AARCH64_SYSREG_PMEVCNTSVR29_EL1,
	ARM64_SYSREG_PMEVCNTSVR2_EL1 = AARCH64_SYSREG_PMEVCNTSVR2_EL1,
	ARM64_SYSREG_PMEVCNTSVR30_EL1 = AARCH64_SYSREG_PMEVCNTSVR30_EL1,
	ARM64_SYSREG_PMEVCNTSVR3_EL1 = AARCH64_SYSREG_PMEVCNTSVR3_EL1,
	ARM64_SYSREG_PMEVCNTSVR4_EL1 = AARCH64_SYSREG_PMEVCNTSVR4_EL1,
	ARM64_SYSREG_PMEVCNTSVR5_EL1 = AARCH64_SYSREG_PMEVCNTSVR5_EL1,
	ARM64_SYSREG_PMEVCNTSVR6_EL1 = AARCH64_SYSREG_PMEVCNTSVR6_EL1,
	ARM64_SYSREG_PMEVCNTSVR7_EL1 = AARCH64_SYSREG_PMEVCNTSVR7_EL1,
	ARM64_SYSREG_PMEVCNTSVR8_EL1 = AARCH64_SYSREG_PMEVCNTSVR8_EL1,
	ARM64_SYSREG_PMEVCNTSVR9_EL1 = AARCH64_SYSREG_PMEVCNTSVR9_EL1,
	ARM64_SYSREG_PMEVTYPER0_EL0 = AARCH64_SYSREG_PMEVTYPER0_EL0,
	ARM64_SYSREG_PMEVTYPER10_EL0 = AARCH64_SYSREG_PMEVTYPER10_EL0,
	ARM64_SYSREG_PMEVTYPER11_EL0 = AARCH64_SYSREG_PMEVTYPER11_EL0,
	ARM64_SYSREG_PMEVTYPER12_EL0 = AARCH64_SYSREG_PMEVTYPER12_EL0,
	ARM64_SYSREG_PMEVTYPER13_EL0 = AARCH64_SYSREG_PMEVTYPER13_EL0,
	ARM64_SYSREG_PMEVTYPER14_EL0 = AARCH64_SYSREG_PMEVTYPER14_EL0,
	ARM64_SYSREG_PMEVTYPER15_EL0 = AARCH64_SYSREG_PMEVTYPER15_EL0,
	ARM64_SYSREG_PMEVTYPER16_EL0 = AARCH64_SYSREG_PMEVTYPER16_EL0,
	ARM64_SYSREG_PMEVTYPER17_EL0 = AARCH64_SYSREG_PMEVTYPER17_EL0,
	ARM64_SYSREG_PMEVTYPER18_EL0 = AARCH64_SYSREG_PMEVTYPER18_EL0,
	ARM64_SYSREG_PMEVTYPER19_EL0 = AARCH64_SYSREG_PMEVTYPER19_EL0,
	ARM64_SYSREG_PMEVTYPER1_EL0 = AARCH64_SYSREG_PMEVTYPER1_EL0,
	ARM64_SYSREG_PMEVTYPER20_EL0 = AARCH64_SYSREG_PMEVTYPER20_EL0,
	ARM64_SYSREG_PMEVTYPER21_EL0 = AARCH64_SYSREG_PMEVTYPER21_EL0,
	ARM64_SYSREG_PMEVTYPER22_EL0 = AARCH64_SYSREG_PMEVTYPER22_EL0,
	ARM64_SYSREG_PMEVTYPER23_EL0 = AARCH64_SYSREG_PMEVTYPER23_EL0,
	ARM64_SYSREG_PMEVTYPER24_EL0 = AARCH64_SYSREG_PMEVTYPER24_EL0,
	ARM64_SYSREG_PMEVTYPER25_EL0 = AARCH64_SYSREG_PMEVTYPER25_EL0,
	ARM64_SYSREG_PMEVTYPER26_EL0 = AARCH64_SYSREG_PMEVTYPER26_EL0,
	ARM64_SYSREG_PMEVTYPER27_EL0 = AARCH64_SYSREG_PMEVTYPER27_EL0,
	ARM64_SYSREG_PMEVTYPER28_EL0 = AARCH64_SYSREG_PMEVTYPER28_EL0,
	ARM64_SYSREG_PMEVTYPER29_EL0 = AARCH64_SYSREG_PMEVTYPER29_EL0,
	ARM64_SYSREG_PMEVTYPER2_EL0 = AARCH64_SYSREG_PMEVTYPER2_EL0,
	ARM64_SYSREG_PMEVTYPER30_EL0 = AARCH64_SYSREG_PMEVTYPER30_EL0,
	ARM64_SYSREG_PMEVTYPER3_EL0 = AARCH64_SYSREG_PMEVTYPER3_EL0,
	ARM64_SYSREG_PMEVTYPER4_EL0 = AARCH64_SYSREG_PMEVTYPER4_EL0,
	ARM64_SYSREG_PMEVTYPER5_EL0 = AARCH64_SYSREG_PMEVTYPER5_EL0,
	ARM64_SYSREG_PMEVTYPER6_EL0 = AARCH64_SYSREG_PMEVTYPER6_EL0,
	ARM64_SYSREG_PMEVTYPER7_EL0 = AARCH64_SYSREG_PMEVTYPER7_EL0,
	ARM64_SYSREG_PMEVTYPER8_EL0 = AARCH64_SYSREG_PMEVTYPER8_EL0,
	ARM64_SYSREG_PMEVTYPER9_EL0 = AARCH64_SYSREG_PMEVTYPER9_EL0,
	ARM64_SYSREG_PMIAR_EL1 = AARCH64_SYSREG_PMIAR_EL1,
	ARM64_SYSREG_PMICFILTR_EL0 = AARCH64_SYSREG_PMICFILTR_EL0,
	ARM64_SYSREG_PMICNTR_EL0 = AARCH64_SYSREG_PMICNTR_EL0,
	ARM64_SYSREG_PMICNTSVR_EL1 = AARCH64_SYSREG_PMICNTSVR_EL1,
	ARM64_SYSREG_PMINTENCLR_EL1 = AARCH64_SYSREG_PMINTENCLR_EL1,
	ARM64_SYSREG_PMINTENSET_EL1 = AARCH64_SYSREG_PMINTENSET_EL1,
	ARM64_SYSREG_PMMIR_EL1 = AARCH64_SYSREG_PMMIR_EL1,
	ARM64_SYSREG_PMOVSCLR_EL0 = AARCH64_SYSREG_PMOVSCLR_EL0,
	ARM64_SYSREG_PMOVSSET_EL0 = AARCH64_SYSREG_PMOVSSET_EL0,
	ARM64_SYSREG_PMSCR_EL1 = AARCH64_SYSREG_PMSCR_EL1,
	ARM64_SYSREG_PMSCR_EL12 = AARCH64_SYSREG_PMSCR_EL12,
	ARM64_SYSREG_PMSCR_EL2 = AARCH64_SYSREG_PMSCR_EL2,
	ARM64_SYSREG_PMSDSFR_EL1 = AARCH64_SYSREG_PMSDSFR_EL1,
	ARM64_SYSREG_PMSELR_EL0 = AARCH64_SYSREG_PMSELR_EL0,
	ARM64_SYSREG_PMSEVFR_EL1 = AARCH64_SYSREG_PMSEVFR_EL1,
	ARM64_SYSREG_PMSFCR_EL1 = AARCH64_SYSREG_PMSFCR_EL1,
	ARM64_SYSREG_PMSICR_EL1 = AARCH64_SYSREG_PMSICR_EL1,
	ARM64_SYSREG_PMSIDR_EL1 = AARCH64_SYSREG_PMSIDR_EL1,
	ARM64_SYSREG_PMSIRR_EL1 = AARCH64_SYSREG_PMSIRR_EL1,
	ARM64_SYSREG_PMSLATFR_EL1 = AARCH64_SYSREG_PMSLATFR_EL1,
	ARM64_SYSREG_PMSNEVFR_EL1 = AARCH64_SYSREG_PMSNEVFR_EL1,
	ARM64_SYSREG_PMSSCR_EL1 = AARCH64_SYSREG_PMSSCR_EL1,
	ARM64_SYSREG_PMSWINC_EL0 = AARCH64_SYSREG_PMSWINC_EL0,
	ARM64_SYSREG_PMUACR_EL1 = AARCH64_SYSREG_PMUACR_EL1,
	ARM64_SYSREG_PMUSERENR_EL0 = AARCH64_SYSREG_PMUSERENR_EL0,
	ARM64_SYSREG_PMXEVCNTR_EL0 = AARCH64_SYSREG_PMXEVCNTR_EL0,
	ARM64_SYSREG_PMXEVTYPER_EL0 = AARCH64_SYSREG_PMXEVTYPER_EL0,
	ARM64_SYSREG_PMZR_EL0 = AARCH64_SYSREG_PMZR_EL0,
	ARM64_SYSREG_POR_EL0 = AARCH64_SYSREG_POR_EL0,
	ARM64_SYSREG_POR_EL1 = AARCH64_SYSREG_POR_EL1,
	ARM64_SYSREG_POR_EL12 = AARCH64_SYSREG_POR_EL12,
	ARM64_SYSREG_POR_EL2 = AARCH64_SYSREG_POR_EL2,
	ARM64_SYSREG_POR_EL3 = AARCH64_SYSREG_POR_EL3,
	ARM64_SYSREG_PRBAR10_EL1 = AARCH64_SYSREG_PRBAR10_EL1,
	ARM64_SYSREG_PRBAR10_EL2 = AARCH64_SYSREG_PRBAR10_EL2,
	ARM64_SYSREG_PRBAR11_EL1 = AARCH64_SYSREG_PRBAR11_EL1,
	ARM64_SYSREG_PRBAR11_EL2 = AARCH64_SYSREG_PRBAR11_EL2,
	ARM64_SYSREG_PRBAR12_EL1 = AARCH64_SYSREG_PRBAR12_EL1,
	ARM64_SYSREG_PRBAR12_EL2 = AARCH64_SYSREG_PRBAR12_EL2,
	ARM64_SYSREG_PRBAR13_EL1 = AARCH64_SYSREG_PRBAR13_EL1,
	ARM64_SYSREG_PRBAR13_EL2 = AARCH64_SYSREG_PRBAR13_EL2,
	ARM64_SYSREG_PRBAR14_EL1 = AARCH64_SYSREG_PRBAR14_EL1,
	ARM64_SYSREG_PRBAR14_EL2 = AARCH64_SYSREG_PRBAR14_EL2,
	ARM64_SYSREG_PRBAR15_EL1 = AARCH64_SYSREG_PRBAR15_EL1,
	ARM64_SYSREG_PRBAR15_EL2 = AARCH64_SYSREG_PRBAR15_EL2,
	ARM64_SYSREG_PRBAR1_EL1 = AARCH64_SYSREG_PRBAR1_EL1,
	ARM64_SYSREG_PRBAR1_EL2 = AARCH64_SYSREG_PRBAR1_EL2,
	ARM64_SYSREG_PRBAR2_EL1 = AARCH64_SYSREG_PRBAR2_EL1,
	ARM64_SYSREG_PRBAR2_EL2 = AARCH64_SYSREG_PRBAR2_EL2,
	ARM64_SYSREG_PRBAR3_EL1 = AARCH64_SYSREG_PRBAR3_EL1,
	ARM64_SYSREG_PRBAR3_EL2 = AARCH64_SYSREG_PRBAR3_EL2,
	ARM64_SYSREG_PRBAR4_EL1 = AARCH64_SYSREG_PRBAR4_EL1,
	ARM64_SYSREG_PRBAR4_EL2 = AARCH64_SYSREG_PRBAR4_EL2,
	ARM64_SYSREG_PRBAR5_EL1 = AARCH64_SYSREG_PRBAR5_EL1,
	ARM64_SYSREG_PRBAR5_EL2 = AARCH64_SYSREG_PRBAR5_EL2,
	ARM64_SYSREG_PRBAR6_EL1 = AARCH64_SYSREG_PRBAR6_EL1,
	ARM64_SYSREG_PRBAR6_EL2 = AARCH64_SYSREG_PRBAR6_EL2,
	ARM64_SYSREG_PRBAR7_EL1 = AARCH64_SYSREG_PRBAR7_EL1,
	ARM64_SYSREG_PRBAR7_EL2 = AARCH64_SYSREG_PRBAR7_EL2,
	ARM64_SYSREG_PRBAR8_EL1 = AARCH64_SYSREG_PRBAR8_EL1,
	ARM64_SYSREG_PRBAR8_EL2 = AARCH64_SYSREG_PRBAR8_EL2,
	ARM64_SYSREG_PRBAR9_EL1 = AARCH64_SYSREG_PRBAR9_EL1,
	ARM64_SYSREG_PRBAR9_EL2 = AARCH64_SYSREG_PRBAR9_EL2,
	ARM64_SYSREG_PRBAR_EL1 = AARCH64_SYSREG_PRBAR_EL1,
	ARM64_SYSREG_PRBAR_EL2 = AARCH64_SYSREG_PRBAR_EL2,
	ARM64_SYSREG_PRENR_EL1 = AARCH64_SYSREG_PRENR_EL1,
	ARM64_SYSREG_PRENR_EL2 = AARCH64_SYSREG_PRENR_EL2,
	ARM64_SYSREG_PRLAR10_EL1 = AARCH64_SYSREG_PRLAR10_EL1,
	ARM64_SYSREG_PRLAR10_EL2 = AARCH64_SYSREG_PRLAR10_EL2,
	ARM64_SYSREG_PRLAR11_EL1 = AARCH64_SYSREG_PRLAR11_EL1,
	ARM64_SYSREG_PRLAR11_EL2 = AARCH64_SYSREG_PRLAR11_EL2,
	ARM64_SYSREG_PRLAR12_EL1 = AARCH64_SYSREG_PRLAR12_EL1,
	ARM64_SYSREG_PRLAR12_EL2 = AARCH64_SYSREG_PRLAR12_EL2,
	ARM64_SYSREG_PRLAR13_EL1 = AARCH64_SYSREG_PRLAR13_EL1,
	ARM64_SYSREG_PRLAR13_EL2 = AARCH64_SYSREG_PRLAR13_EL2,
	ARM64_SYSREG_PRLAR14_EL1 = AARCH64_SYSREG_PRLAR14_EL1,
	ARM64_SYSREG_PRLAR14_EL2 = AARCH64_SYSREG_PRLAR14_EL2,
	ARM64_SYSREG_PRLAR15_EL1 = AARCH64_SYSREG_PRLAR15_EL1,
	ARM64_SYSREG_PRLAR15_EL2 = AARCH64_SYSREG_PRLAR15_EL2,
	ARM64_SYSREG_PRLAR1_EL1 = AARCH64_SYSREG_PRLAR1_EL1,
	ARM64_SYSREG_PRLAR1_EL2 = AARCH64_SYSREG_PRLAR1_EL2,
	ARM64_SYSREG_PRLAR2_EL1 = AARCH64_SYSREG_PRLAR2_EL1,
	ARM64_SYSREG_PRLAR2_EL2 = AARCH64_SYSREG_PRLAR2_EL2,
	ARM64_SYSREG_PRLAR3_EL1 = AARCH64_SYSREG_PRLAR3_EL1,
	ARM64_SYSREG_PRLAR3_EL2 = AARCH64_SYSREG_PRLAR3_EL2,
	ARM64_SYSREG_PRLAR4_EL1 = AARCH64_SYSREG_PRLAR4_EL1,
	ARM64_SYSREG_PRLAR4_EL2 = AARCH64_SYSREG_PRLAR4_EL2,
	ARM64_SYSREG_PRLAR5_EL1 = AARCH64_SYSREG_PRLAR5_EL1,
	ARM64_SYSREG_PRLAR5_EL2 = AARCH64_SYSREG_PRLAR5_EL2,
	ARM64_SYSREG_PRLAR6_EL1 = AARCH64_SYSREG_PRLAR6_EL1,
	ARM64_SYSREG_PRLAR6_EL2 = AARCH64_SYSREG_PRLAR6_EL2,
	ARM64_SYSREG_PRLAR7_EL1 = AARCH64_SYSREG_PRLAR7_EL1,
	ARM64_SYSREG_PRLAR7_EL2 = AARCH64_SYSREG_PRLAR7_EL2,
	ARM64_SYSREG_PRLAR8_EL1 = AARCH64_SYSREG_PRLAR8_EL1,
	ARM64_SYSREG_PRLAR8_EL2 = AARCH64_SYSREG_PRLAR8_EL2,
	ARM64_SYSREG_PRLAR9_EL1 = AARCH64_SYSREG_PRLAR9_EL1,
	ARM64_SYSREG_PRLAR9_EL2 = AARCH64_SYSREG_PRLAR9_EL2,
	ARM64_SYSREG_PRLAR_EL1 = AARCH64_SYSREG_PRLAR_EL1,
	ARM64_SYSREG_PRLAR_EL2 = AARCH64_SYSREG_PRLAR_EL2,
	ARM64_SYSREG_PRSELR_EL1 = AARCH64_SYSREG_PRSELR_EL1,
	ARM64_SYSREG_PRSELR_EL2 = AARCH64_SYSREG_PRSELR_EL2,
	ARM64_SYSREG_RCWMASK_EL1 = AARCH64_SYSREG_RCWMASK_EL1,
	ARM64_SYSREG_RCWSMASK_EL1 = AARCH64_SYSREG_RCWSMASK_EL1,
	ARM64_SYSREG_REVIDR_EL1 = AARCH64_SYSREG_REVIDR_EL1,
	ARM64_SYSREG_RGSR_EL1 = AARCH64_SYSREG_RGSR_EL1,
	ARM64_SYSREG_RMR_EL1 = AARCH64_SYSREG_RMR_EL1,
	ARM64_SYSREG_RMR_EL2 = AARCH64_SYSREG_RMR_EL2,
	ARM64_SYSREG_RMR_EL3 = AARCH64_SYSREG_RMR_EL3,
	ARM64_SYSREG_RNDR = AARCH64_SYSREG_RNDR,
	ARM64_SYSREG_RNDRRS = AARCH64_SYSREG_RNDRRS,
	ARM64_SYSREG_RVBAR_EL1 = AARCH64_SYSREG_RVBAR_EL1,
	ARM64_SYSREG_RVBAR_EL2 = AARCH64_SYSREG_RVBAR_EL2,
	ARM64_SYSREG_RVBAR_EL3 = AARCH64_SYSREG_RVBAR_EL3,
	ARM64_SYSREG_S2PIR_EL2 = AARCH64_SYSREG_S2PIR_EL2,
	ARM64_SYSREG_S2POR_EL1 = AARCH64_SYSREG_S2POR_EL1,
	ARM64_SYSREG_SCR_EL3 = AARCH64_SYSREG_SCR_EL3,
	ARM64_SYSREG_SCTLR2_EL1 = AARCH64_SYSREG_SCTLR2_EL1,
	ARM64_SYSREG_SCTLR2_EL12 = AARCH64_SYSREG_SCTLR2_EL12,
	ARM64_SYSREG_SCTLR2_EL2 = AARCH64_SYSREG_SCTLR2_EL2,
	ARM64_SYSREG_SCTLR2_EL3 = AARCH64_SYSREG_SCTLR2_EL3,
	ARM64_SYSREG_SCTLR_EL1 = AARCH64_SYSREG_SCTLR_EL1,
	ARM64_SYSREG_SCTLR_EL12 = AARCH64_SYSREG_SCTLR_EL12,
	ARM64_SYSREG_SCTLR_EL2 = AARCH64_SYSREG_SCTLR_EL2,
	ARM64_SYSREG_SCTLR_EL3 = AARCH64_SYSREG_SCTLR_EL3,
	ARM64_SYSREG_SCXTNUM_EL0 = AARCH64_SYSREG_SCXTNUM_EL0,
	ARM64_SYSREG_SCXTNUM_EL1 = AARCH64_SYSREG_SCXTNUM_EL1,
	ARM64_SYSREG_SCXTNUM_EL12 = AARCH64_SYSREG_SCXTNUM_EL12,
	ARM64_SYSREG_SCXTNUM_EL2 = AARCH64_SYSREG_SCXTNUM_EL2,
	ARM64_SYSREG_SCXTNUM_EL3 = AARCH64_SYSREG_SCXTNUM_EL3,
	ARM64_SYSREG_SDER32_EL2 = AARCH64_SYSREG_SDER32_EL2,
	ARM64_SYSREG_SDER32_EL3 = AARCH64_SYSREG_SDER32_EL3,
	ARM64_SYSREG_SMCR_EL1 = AARCH64_SYSREG_SMCR_EL1,
	ARM64_SYSREG_SMCR_EL12 = AARCH64_SYSREG_SMCR_EL12,
	ARM64_SYSREG_SMCR_EL2 = AARCH64_SYSREG_SMCR_EL2,
	ARM64_SYSREG_SMCR_EL3 = AARCH64_SYSREG_SMCR_EL3,
	ARM64_SYSREG_SMIDR_EL1 = AARCH64_SYSREG_SMIDR_EL1,
	ARM64_SYSREG_SMPRIMAP_EL2 = AARCH64_SYSREG_SMPRIMAP_EL2,
	ARM64_SYSREG_SMPRI_EL1 = AARCH64_SYSREG_SMPRI_EL1,
	ARM64_SYSREG_SPMACCESSR_EL1 = AARCH64_SYSREG_SPMACCESSR_EL1,
	ARM64_SYSREG_SPMACCESSR_EL12 = AARCH64_SYSREG_SPMACCESSR_EL12,
	ARM64_SYSREG_SPMACCESSR_EL2 = AARCH64_SYSREG_SPMACCESSR_EL2,
	ARM64_SYSREG_SPMACCESSR_EL3 = AARCH64_SYSREG_SPMACCESSR_EL3,
	ARM64_SYSREG_SPMCFGR_EL1 = AARCH64_SYSREG_SPMCFGR_EL1,
	ARM64_SYSREG_SPMCGCR0_EL1 = AARCH64_SYSREG_SPMCGCR0_EL1,
	ARM64_SYSREG_SPMCGCR1_EL1 = AARCH64_SYSREG_SPMCGCR1_EL1,
	ARM64_SYSREG_SPMCNTENCLR_EL0 = AARCH64_SYSREG_SPMCNTENCLR_EL0,
	ARM64_SYSREG_SPMCNTENSET_EL0 = AARCH64_SYSREG_SPMCNTENSET_EL0,
	ARM64_SYSREG_SPMCR_EL0 = AARCH64_SYSREG_SPMCR_EL0,
	ARM64_SYSREG_SPMDEVAFF_EL1 = AARCH64_SYSREG_SPMDEVAFF_EL1,
	ARM64_SYSREG_SPMDEVARCH_EL1 = AARCH64_SYSREG_SPMDEVARCH_EL1,
	ARM64_SYSREG_SPMEVCNTR0_EL0 = AARCH64_SYSREG_SPMEVCNTR0_EL0,
	ARM64_SYSREG_SPMEVCNTR10_EL0 = AARCH64_SYSREG_SPMEVCNTR10_EL0,
	ARM64_SYSREG_SPMEVCNTR11_EL0 = AARCH64_SYSREG_SPMEVCNTR11_EL0,
	ARM64_SYSREG_SPMEVCNTR12_EL0 = AARCH64_SYSREG_SPMEVCNTR12_EL0,
	ARM64_SYSREG_SPMEVCNTR13_EL0 = AARCH64_SYSREG_SPMEVCNTR13_EL0,
	ARM64_SYSREG_SPMEVCNTR14_EL0 = AARCH64_SYSREG_SPMEVCNTR14_EL0,
	ARM64_SYSREG_SPMEVCNTR15_EL0 = AARCH64_SYSREG_SPMEVCNTR15_EL0,
	ARM64_SYSREG_SPMEVCNTR1_EL0 = AARCH64_SYSREG_SPMEVCNTR1_EL0,
	ARM64_SYSREG_SPMEVCNTR2_EL0 = AARCH64_SYSREG_SPMEVCNTR2_EL0,
	ARM64_SYSREG_SPMEVCNTR3_EL0 = AARCH64_SYSREG_SPMEVCNTR3_EL0,
	ARM64_SYSREG_SPMEVCNTR4_EL0 = AARCH64_SYSREG_SPMEVCNTR4_EL0,
	ARM64_SYSREG_SPMEVCNTR5_EL0 = AARCH64_SYSREG_SPMEVCNTR5_EL0,
	ARM64_SYSREG_SPMEVCNTR6_EL0 = AARCH64_SYSREG_SPMEVCNTR6_EL0,
	ARM64_SYSREG_SPMEVCNTR7_EL0 = AARCH64_SYSREG_SPMEVCNTR7_EL0,
	ARM64_SYSREG_SPMEVCNTR8_EL0 = AARCH64_SYSREG_SPMEVCNTR8_EL0,
	ARM64_SYSREG_SPMEVCNTR9_EL0 = AARCH64_SYSREG_SPMEVCNTR9_EL0,
	ARM64_SYSREG_SPMEVFILT2R0_EL0 = AARCH64_SYSREG_SPMEVFILT2R0_EL0,
	ARM64_SYSREG_SPMEVFILT2R10_EL0 = AARCH64_SYSREG_SPMEVFILT2R10_EL0,
	ARM64_SYSREG_SPMEVFILT2R11_EL0 = AARCH64_SYSREG_SPMEVFILT2R11_EL0,
	ARM64_SYSREG_SPMEVFILT2R12_EL0 = AARCH64_SYSREG_SPMEVFILT2R12_EL0,
	ARM64_SYSREG_SPMEVFILT2R13_EL0 = AARCH64_SYSREG_SPMEVFILT2R13_EL0,
	ARM64_SYSREG_SPMEVFILT2R14_EL0 = AARCH64_SYSREG_SPMEVFILT2R14_EL0,
	ARM64_SYSREG_SPMEVFILT2R15_EL0 = AARCH64_SYSREG_SPMEVFILT2R15_EL0,
	ARM64_SYSREG_SPMEVFILT2R1_EL0 = AARCH64_SYSREG_SPMEVFILT2R1_EL0,
	ARM64_SYSREG_SPMEVFILT2R2_EL0 = AARCH64_SYSREG_SPMEVFILT2R2_EL0,
	ARM64_SYSREG_SPMEVFILT2R3_EL0 = AARCH64_SYSREG_SPMEVFILT2R3_EL0,
	ARM64_SYSREG_SPMEVFILT2R4_EL0 = AARCH64_SYSREG_SPMEVFILT2R4_EL0,
	ARM64_SYSREG_SPMEVFILT2R5_EL0 = AARCH64_SYSREG_SPMEVFILT2R5_EL0,
	ARM64_SYSREG_SPMEVFILT2R6_EL0 = AARCH64_SYSREG_SPMEVFILT2R6_EL0,
	ARM64_SYSREG_SPMEVFILT2R7_EL0 = AARCH64_SYSREG_SPMEVFILT2R7_EL0,
	ARM64_SYSREG_SPMEVFILT2R8_EL0 = AARCH64_SYSREG_SPMEVFILT2R8_EL0,
	ARM64_SYSREG_SPMEVFILT2R9_EL0 = AARCH64_SYSREG_SPMEVFILT2R9_EL0,
	ARM64_SYSREG_SPMEVFILTR0_EL0 = AARCH64_SYSREG_SPMEVFILTR0_EL0,
	ARM64_SYSREG_SPMEVFILTR10_EL0 = AARCH64_SYSREG_SPMEVFILTR10_EL0,
	ARM64_SYSREG_SPMEVFILTR11_EL0 = AARCH64_SYSREG_SPMEVFILTR11_EL0,
	ARM64_SYSREG_SPMEVFILTR12_EL0 = AARCH64_SYSREG_SPMEVFILTR12_EL0,
	ARM64_SYSREG_SPMEVFILTR13_EL0 = AARCH64_SYSREG_SPMEVFILTR13_EL0,
	ARM64_SYSREG_SPMEVFILTR14_EL0 = AARCH64_SYSREG_SPMEVFILTR14_EL0,
	ARM64_SYSREG_SPMEVFILTR15_EL0 = AARCH64_SYSREG_SPMEVFILTR15_EL0,
	ARM64_SYSREG_SPMEVFILTR1_EL0 = AARCH64_SYSREG_SPMEVFILTR1_EL0,
	ARM64_SYSREG_SPMEVFILTR2_EL0 = AARCH64_SYSREG_SPMEVFILTR2_EL0,
	ARM64_SYSREG_SPMEVFILTR3_EL0 = AARCH64_SYSREG_SPMEVFILTR3_EL0,
	ARM64_SYSREG_SPMEVFILTR4_EL0 = AARCH64_SYSREG_SPMEVFILTR4_EL0,
	ARM64_SYSREG_SPMEVFILTR5_EL0 = AARCH64_SYSREG_SPMEVFILTR5_EL0,
	ARM64_SYSREG_SPMEVFILTR6_EL0 = AARCH64_SYSREG_SPMEVFILTR6_EL0,
	ARM64_SYSREG_SPMEVFILTR7_EL0 = AARCH64_SYSREG_SPMEVFILTR7_EL0,
	ARM64_SYSREG_SPMEVFILTR8_EL0 = AARCH64_SYSREG_SPMEVFILTR8_EL0,
	ARM64_SYSREG_SPMEVFILTR9_EL0 = AARCH64_SYSREG_SPMEVFILTR9_EL0,
	ARM64_SYSREG_SPMEVTYPER0_EL0 = AARCH64_SYSREG_SPMEVTYPER0_EL0,
	ARM64_SYSREG_SPMEVTYPER10_EL0 = AARCH64_SYSREG_SPMEVTYPER10_EL0,
	ARM64_SYSREG_SPMEVTYPER11_EL0 = AARCH64_SYSREG_SPMEVTYPER11_EL0,
	ARM64_SYSREG_SPMEVTYPER12_EL0 = AARCH64_SYSREG_SPMEVTYPER12_EL0,
	ARM64_SYSREG_SPMEVTYPER13_EL0 = AARCH64_SYSREG_SPMEVTYPER13_EL0,
	ARM64_SYSREG_SPMEVTYPER14_EL0 = AARCH64_SYSREG_SPMEVTYPER14_EL0,
	ARM64_SYSREG_SPMEVTYPER15_EL0 = AARCH64_SYSREG_SPMEVTYPER15_EL0,
	ARM64_SYSREG_SPMEVTYPER1_EL0 = AARCH64_SYSREG_SPMEVTYPER1_EL0,
	ARM64_SYSREG_SPMEVTYPER2_EL0 = AARCH64_SYSREG_SPMEVTYPER2_EL0,
	ARM64_SYSREG_SPMEVTYPER3_EL0 = AARCH64_SYSREG_SPMEVTYPER3_EL0,
	ARM64_SYSREG_SPMEVTYPER4_EL0 = AARCH64_SYSREG_SPMEVTYPER4_EL0,
	ARM64_SYSREG_SPMEVTYPER5_EL0 = AARCH64_SYSREG_SPMEVTYPER5_EL0,
	ARM64_SYSREG_SPMEVTYPER6_EL0 = AARCH64_SYSREG_SPMEVTYPER6_EL0,
	ARM64_SYSREG_SPMEVTYPER7_EL0 = AARCH64_SYSREG_SPMEVTYPER7_EL0,
	ARM64_SYSREG_SPMEVTYPER8_EL0 = AARCH64_SYSREG_SPMEVTYPER8_EL0,
	ARM64_SYSREG_SPMEVTYPER9_EL0 = AARCH64_SYSREG_SPMEVTYPER9_EL0,
	ARM64_SYSREG_SPMIIDR_EL1 = AARCH64_SYSREG_SPMIIDR_EL1,
	ARM64_SYSREG_SPMINTENCLR_EL1 = AARCH64_SYSREG_SPMINTENCLR_EL1,
	ARM64_SYSREG_SPMINTENSET_EL1 = AARCH64_SYSREG_SPMINTENSET_EL1,
	ARM64_SYSREG_SPMOVSCLR_EL0 = AARCH64_SYSREG_SPMOVSCLR_EL0,
	ARM64_SYSREG_SPMOVSSET_EL0 = AARCH64_SYSREG_SPMOVSSET_EL0,
	ARM64_SYSREG_SPMROOTCR_EL3 = AARCH64_SYSREG_SPMROOTCR_EL3,
	ARM64_SYSREG_SPMSCR_EL1 = AARCH64_SYSREG_SPMSCR_EL1,
	ARM64_SYSREG_SPMSELR_EL0 = AARCH64_SYSREG_SPMSELR_EL0,
	ARM64_SYSREG_SPMZR_EL0 = AARCH64_SYSREG_SPMZR_EL0,
	ARM64_SYSREG_SPSEL = AARCH64_SYSREG_SPSEL,
	ARM64_SYSREG_SPSR_ABT = AARCH64_SYSREG_SPSR_ABT,
	ARM64_SYSREG_SPSR_EL1 = AARCH64_SYSREG_SPSR_EL1,
	ARM64_SYSREG_SPSR_EL12 = AARCH64_SYSREG_SPSR_EL12,
	ARM64_SYSREG_SPSR_EL2 = AARCH64_SYSREG_SPSR_EL2,
	ARM64_SYSREG_SPSR_EL3 = AARCH64_SYSREG_SPSR_EL3,
	ARM64_SYSREG_SPSR_FIQ = AARCH64_SYSREG_SPSR_FIQ,
	ARM64_SYSREG_SPSR_IRQ = AARCH64_SYSREG_SPSR_IRQ,
	ARM64_SYSREG_SPSR_UND = AARCH64_SYSREG_SPSR_UND,
	ARM64_SYSREG_SP_EL0 = AARCH64_SYSREG_SP_EL0,
	ARM64_SYSREG_SP_EL1 = AARCH64_SYSREG_SP_EL1,
	ARM64_SYSREG_SP_EL2 = AARCH64_SYSREG_SP_EL2,
	ARM64_SYSREG_SSBS = AARCH64_SYSREG_SSBS,
	ARM64_SYSREG_SVCR = AARCH64_SYSREG_SVCR,
	ARM64_SYSREG_TCO = AARCH64_SYSREG_TCO,
	ARM64_SYSREG_TCR2_EL1 = AARCH64_SYSREG_TCR2_EL1,
	ARM64_SYSREG_TCR2_EL12 = AARCH64_SYSREG_TCR2_EL12,
	ARM64_SYSREG_TCR2_EL2 = AARCH64_SYSREG_TCR2_EL2,
	ARM64_SYSREG_TCR_EL1 = AARCH64_SYSREG_TCR_EL1,
	ARM64_SYSREG_TCR_EL12 = AARCH64_SYSREG_TCR_EL12,
	ARM64_SYSREG_TCR_EL2 = AARCH64_SYSREG_TCR_EL2,
	ARM64_SYSREG_TCR_EL3 = AARCH64_SYSREG_TCR_EL3,
	ARM64_SYSREG_TEECR32_EL1 = AARCH64_SYSREG_TEECR32_EL1,
	ARM64_SYSREG_TEEHBR32_EL1 = AARCH64_SYSREG_TEEHBR32_EL1,
	ARM64_SYSREG_TFSRE0_EL1 = AARCH64_SYSREG_TFSRE0_EL1,
	ARM64_SYSREG_TFSR_EL1 = AARCH64_SYSREG_TFSR_EL1,
	ARM64_SYSREG_TFSR_EL12 = AARCH64_SYSREG_TFSR_EL12,
	ARM64_SYSREG_TFSR_EL2 = AARCH64_SYSREG_TFSR_EL2,
	ARM64_SYSREG_TFSR_EL3 = AARCH64_SYSREG_TFSR_EL3,
	ARM64_SYSREG_TPIDR2_EL0 = AARCH64_SYSREG_TPIDR2_EL0,
	ARM64_SYSREG_TPIDRRO_EL0 = AARCH64_SYSREG_TPIDRRO_EL0,
	ARM64_SYSREG_TPIDR_EL0 = AARCH64_SYSREG_TPIDR_EL0,
	ARM64_SYSREG_TPIDR_EL1 = AARCH64_SYSREG_TPIDR_EL1,
	ARM64_SYSREG_TPIDR_EL2 = AARCH64_SYSREG_TPIDR_EL2,
	ARM64_SYSREG_TPIDR_EL3 = AARCH64_SYSREG_TPIDR_EL3,
	ARM64_SYSREG_TRBBASER_EL1 = AARCH64_SYSREG_TRBBASER_EL1,
	ARM64_SYSREG_TRBIDR_EL1 = AARCH64_SYSREG_TRBIDR_EL1,
	ARM64_SYSREG_TRBLIMITR_EL1 = AARCH64_SYSREG_TRBLIMITR_EL1,
	ARM64_SYSREG_TRBMAR_EL1 = AARCH64_SYSREG_TRBMAR_EL1,
	ARM64_SYSREG_TRBPTR_EL1 = AARCH64_SYSREG_TRBPTR_EL1,
	ARM64_SYSREG_TRBSR_EL1 = AARCH64_SYSREG_TRBSR_EL1,
	ARM64_SYSREG_TRBTRG_EL1 = AARCH64_SYSREG_TRBTRG_EL1,
	ARM64_SYSREG_TRCACATR0 = AARCH64_SYSREG_TRCACATR0,
	ARM64_SYSREG_TRCACATR1 = AARCH64_SYSREG_TRCACATR1,
	ARM64_SYSREG_TRCACATR10 = AARCH64_SYSREG_TRCACATR10,
	ARM64_SYSREG_TRCACATR11 = AARCH64_SYSREG_TRCACATR11,
	ARM64_SYSREG_TRCACATR12 = AARCH64_SYSREG_TRCACATR12,
	ARM64_SYSREG_TRCACATR13 = AARCH64_SYSREG_TRCACATR13,
	ARM64_SYSREG_TRCACATR14 = AARCH64_SYSREG_TRCACATR14,
	ARM64_SYSREG_TRCACATR15 = AARCH64_SYSREG_TRCACATR15,
	ARM64_SYSREG_TRCACATR2 = AARCH64_SYSREG_TRCACATR2,
	ARM64_SYSREG_TRCACATR3 = AARCH64_SYSREG_TRCACATR3,
	ARM64_SYSREG_TRCACATR4 = AARCH64_SYSREG_TRCACATR4,
	ARM64_SYSREG_TRCACATR5 = AARCH64_SYSREG_TRCACATR5,
	ARM64_SYSREG_TRCACATR6 = AARCH64_SYSREG_TRCACATR6,
	ARM64_SYSREG_TRCACATR7 = AARCH64_SYSREG_TRCACATR7,
	ARM64_SYSREG_TRCACATR8 = AARCH64_SYSREG_TRCACATR8,
	ARM64_SYSREG_TRCACATR9 = AARCH64_SYSREG_TRCACATR9,
	ARM64_SYSREG_TRCACVR0 = AARCH64_SYSREG_TRCACVR0,
	ARM64_SYSREG_TRCACVR1 = AARCH64_SYSREG_TRCACVR1,
	ARM64_SYSREG_TRCACVR10 = AARCH64_SYSREG_TRCACVR10,
	ARM64_SYSREG_TRCACVR11 = AARCH64_SYSREG_TRCACVR11,
	ARM64_SYSREG_TRCACVR12 = AARCH64_SYSREG_TRCACVR12,
	ARM64_SYSREG_TRCACVR13 = AARCH64_SYSREG_TRCACVR13,
	ARM64_SYSREG_TRCACVR14 = AARCH64_SYSREG_TRCACVR14,
	ARM64_SYSREG_TRCACVR15 = AARCH64_SYSREG_TRCACVR15,
	ARM64_SYSREG_TRCACVR2 = AARCH64_SYSREG_TRCACVR2,
	ARM64_SYSREG_TRCACVR3 = AARCH64_SYSREG_TRCACVR3,
	ARM64_SYSREG_TRCACVR4 = AARCH64_SYSREG_TRCACVR4,
	ARM64_SYSREG_TRCACVR5 = AARCH64_SYSREG_TRCACVR5,
	ARM64_SYSREG_TRCACVR6 = AARCH64_SYSREG_TRCACVR6,
	ARM64_SYSREG_TRCACVR7 = AARCH64_SYSREG_TRCACVR7,
	ARM64_SYSREG_TRCACVR8 = AARCH64_SYSREG_TRCACVR8,
	ARM64_SYSREG_TRCACVR9 = AARCH64_SYSREG_TRCACVR9,
	ARM64_SYSREG_TRCAUTHSTATUS = AARCH64_SYSREG_TRCAUTHSTATUS,
	ARM64_SYSREG_TRCAUXCTLR = AARCH64_SYSREG_TRCAUXCTLR,
	ARM64_SYSREG_TRCBBCTLR = AARCH64_SYSREG_TRCBBCTLR,
	ARM64_SYSREG_TRCCCCTLR = AARCH64_SYSREG_TRCCCCTLR,
	ARM64_SYSREG_TRCCIDCCTLR0 = AARCH64_SYSREG_TRCCIDCCTLR0,
	ARM64_SYSREG_TRCCIDCCTLR1 = AARCH64_SYSREG_TRCCIDCCTLR1,
	ARM64_SYSREG_TRCCIDCVR0 = AARCH64_SYSREG_TRCCIDCVR0,
	ARM64_SYSREG_TRCCIDCVR1 = AARCH64_SYSREG_TRCCIDCVR1,
	ARM64_SYSREG_TRCCIDCVR2 = AARCH64_SYSREG_TRCCIDCVR2,
	ARM64_SYSREG_TRCCIDCVR3 = AARCH64_SYSREG_TRCCIDCVR3,
	ARM64_SYSREG_TRCCIDCVR4 = AARCH64_SYSREG_TRCCIDCVR4,
	ARM64_SYSREG_TRCCIDCVR5 = AARCH64_SYSREG_TRCCIDCVR5,
	ARM64_SYSREG_TRCCIDCVR6 = AARCH64_SYSREG_TRCCIDCVR6,
	ARM64_SYSREG_TRCCIDCVR7 = AARCH64_SYSREG_TRCCIDCVR7,
	ARM64_SYSREG_TRCCIDR0 = AARCH64_SYSREG_TRCCIDR0,
	ARM64_SYSREG_TRCCIDR1 = AARCH64_SYSREG_TRCCIDR1,
	ARM64_SYSREG_TRCCIDR2 = AARCH64_SYSREG_TRCCIDR2,
	ARM64_SYSREG_TRCCIDR3 = AARCH64_SYSREG_TRCCIDR3,
	ARM64_SYSREG_TRCCLAIMCLR = AARCH64_SYSREG_TRCCLAIMCLR,
	ARM64_SYSREG_TRCCLAIMSET = AARCH64_SYSREG_TRCCLAIMSET,
	ARM64_SYSREG_TRCCNTCTLR0 = AARCH64_SYSREG_TRCCNTCTLR0,
	ARM64_SYSREG_TRCCNTCTLR1 = AARCH64_SYSREG_TRCCNTCTLR1,
	ARM64_SYSREG_TRCCNTCTLR2 = AARCH64_SYSREG_TRCCNTCTLR2,
	ARM64_SYSREG_TRCCNTCTLR3 = AARCH64_SYSREG_TRCCNTCTLR3,
	ARM64_SYSREG_TRCCNTRLDVR0 = AARCH64_SYSREG_TRCCNTRLDVR0,
	ARM64_SYSREG_TRCCNTRLDVR1 = AARCH64_SYSREG_TRCCNTRLDVR1,
	ARM64_SYSREG_TRCCNTRLDVR2 = AARCH64_SYSREG_TRCCNTRLDVR2,
	ARM64_SYSREG_TRCCNTRLDVR3 = AARCH64_SYSREG_TRCCNTRLDVR3,
	ARM64_SYSREG_TRCCNTVR0 = AARCH64_SYSREG_TRCCNTVR0,
	ARM64_SYSREG_TRCCNTVR1 = AARCH64_SYSREG_TRCCNTVR1,
	ARM64_SYSREG_TRCCNTVR2 = AARCH64_SYSREG_TRCCNTVR2,
	ARM64_SYSREG_TRCCNTVR3 = AARCH64_SYSREG_TRCCNTVR3,
	ARM64_SYSREG_TRCCONFIGR = AARCH64_SYSREG_TRCCONFIGR,
	ARM64_SYSREG_TRCDEVAFF0 = AARCH64_SYSREG_TRCDEVAFF0,
	ARM64_SYSREG_TRCDEVAFF1 = AARCH64_SYSREG_TRCDEVAFF1,
	ARM64_SYSREG_TRCDEVARCH = AARCH64_SYSREG_TRCDEVARCH,
	ARM64_SYSREG_TRCDEVID = AARCH64_SYSREG_TRCDEVID,
	ARM64_SYSREG_TRCDEVTYPE = AARCH64_SYSREG_TRCDEVTYPE,
	ARM64_SYSREG_TRCDVCMR0 = AARCH64_SYSREG_TRCDVCMR0,
	ARM64_SYSREG_TRCDVCMR1 = AARCH64_SYSREG_TRCDVCMR1,
	ARM64_SYSREG_TRCDVCMR2 = AARCH64_SYSREG_TRCDVCMR2,
	ARM64_SYSREG_TRCDVCMR3 = AARCH64_SYSREG_TRCDVCMR3,
	ARM64_SYSREG_TRCDVCMR4 = AARCH64_SYSREG_TRCDVCMR4,
	ARM64_SYSREG_TRCDVCMR5 = AARCH64_SYSREG_TRCDVCMR5,
	ARM64_SYSREG_TRCDVCMR6 = AARCH64_SYSREG_TRCDVCMR6,
	ARM64_SYSREG_TRCDVCMR7 = AARCH64_SYSREG_TRCDVCMR7,
	ARM64_SYSREG_TRCDVCVR0 = AARCH64_SYSREG_TRCDVCVR0,
	ARM64_SYSREG_TRCDVCVR1 = AARCH64_SYSREG_TRCDVCVR1,
	ARM64_SYSREG_TRCDVCVR2 = AARCH64_SYSREG_TRCDVCVR2,
	ARM64_SYSREG_TRCDVCVR3 = AARCH64_SYSREG_TRCDVCVR3,
	ARM64_SYSREG_TRCDVCVR4 = AARCH64_SYSREG_TRCDVCVR4,
	ARM64_SYSREG_TRCDVCVR5 = AARCH64_SYSREG_TRCDVCVR5,
	ARM64_SYSREG_TRCDVCVR6 = AARCH64_SYSREG_TRCDVCVR6,
	ARM64_SYSREG_TRCDVCVR7 = AARCH64_SYSREG_TRCDVCVR7,
	ARM64_SYSREG_TRCEVENTCTL0R = AARCH64_SYSREG_TRCEVENTCTL0R,
	ARM64_SYSREG_TRCEVENTCTL1R = AARCH64_SYSREG_TRCEVENTCTL1R,
	ARM64_SYSREG_TRCEXTINSELR = AARCH64_SYSREG_TRCEXTINSELR,
	ARM64_SYSREG_TRCEXTINSELR0 = AARCH64_SYSREG_TRCEXTINSELR0,
	ARM64_SYSREG_TRCEXTINSELR1 = AARCH64_SYSREG_TRCEXTINSELR1,
	ARM64_SYSREG_TRCEXTINSELR2 = AARCH64_SYSREG_TRCEXTINSELR2,
	ARM64_SYSREG_TRCEXTINSELR3 = AARCH64_SYSREG_TRCEXTINSELR3,
	ARM64_SYSREG_TRCIDR0 = AARCH64_SYSREG_TRCIDR0,
	ARM64_SYSREG_TRCIDR1 = AARCH64_SYSREG_TRCIDR1,
	ARM64_SYSREG_TRCIDR10 = AARCH64_SYSREG_TRCIDR10,
	ARM64_SYSREG_TRCIDR11 = AARCH64_SYSREG_TRCIDR11,
	ARM64_SYSREG_TRCIDR12 = AARCH64_SYSREG_TRCIDR12,
	ARM64_SYSREG_TRCIDR13 = AARCH64_SYSREG_TRCIDR13,
	ARM64_SYSREG_TRCIDR2 = AARCH64_SYSREG_TRCIDR2,
	ARM64_SYSREG_TRCIDR3 = AARCH64_SYSREG_TRCIDR3,
	ARM64_SYSREG_TRCIDR4 = AARCH64_SYSREG_TRCIDR4,
	ARM64_SYSREG_TRCIDR5 = AARCH64_SYSREG_TRCIDR5,
	ARM64_SYSREG_TRCIDR6 = AARCH64_SYSREG_TRCIDR6,
	ARM64_SYSREG_TRCIDR7 = AARCH64_SYSREG_TRCIDR7,
	ARM64_SYSREG_TRCIDR8 = AARCH64_SYSREG_TRCIDR8,
	ARM64_SYSREG_TRCIDR9 = AARCH64_SYSREG_TRCIDR9,
	ARM64_SYSREG_TRCIMSPEC0 = AARCH64_SYSREG_TRCIMSPEC0,
	ARM64_SYSREG_TRCIMSPEC1 = AARCH64_SYSREG_TRCIMSPEC1,
	ARM64_SYSREG_TRCIMSPEC2 = AARCH64_SYSREG_TRCIMSPEC2,
	ARM64_SYSREG_TRCIMSPEC3 = AARCH64_SYSREG_TRCIMSPEC3,
	ARM64_SYSREG_TRCIMSPEC4 = AARCH64_SYSREG_TRCIMSPEC4,
	ARM64_SYSREG_TRCIMSPEC5 = AARCH64_SYSREG_TRCIMSPEC5,
	ARM64_SYSREG_TRCIMSPEC6 = AARCH64_SYSREG_TRCIMSPEC6,
	ARM64_SYSREG_TRCIMSPEC7 = AARCH64_SYSREG_TRCIMSPEC7,
	ARM64_SYSREG_TRCITCTRL = AARCH64_SYSREG_TRCITCTRL,
	ARM64_SYSREG_TRCITECR_EL1 = AARCH64_SYSREG_TRCITECR_EL1,
	ARM64_SYSREG_TRCITECR_EL12 = AARCH64_SYSREG_TRCITECR_EL12,
	ARM64_SYSREG_TRCITECR_EL2 = AARCH64_SYSREG_TRCITECR_EL2,
	ARM64_SYSREG_TRCITEEDCR = AARCH64_SYSREG_TRCITEEDCR,
	ARM64_SYSREG_TRCLAR = AARCH64_SYSREG_TRCLAR,
	ARM64_SYSREG_TRCLSR = AARCH64_SYSREG_TRCLSR,
	ARM64_SYSREG_TRCOSLAR = AARCH64_SYSREG_TRCOSLAR,
	ARM64_SYSREG_TRCOSLSR = AARCH64_SYSREG_TRCOSLSR,
	ARM64_SYSREG_TRCPDCR = AARCH64_SYSREG_TRCPDCR,
	ARM64_SYSREG_TRCPDSR = AARCH64_SYSREG_TRCPDSR,
	ARM64_SYSREG_TRCPIDR0 = AARCH64_SYSREG_TRCPIDR0,
	ARM64_SYSREG_TRCPIDR1 = AARCH64_SYSREG_TRCPIDR1,
	ARM64_SYSREG_TRCPIDR2 = AARCH64_SYSREG_TRCPIDR2,
	ARM64_SYSREG_TRCPIDR3 = AARCH64_SYSREG_TRCPIDR3,
	ARM64_SYSREG_TRCPIDR4 = AARCH64_SYSREG_TRCPIDR4,
	ARM64_SYSREG_TRCPIDR5 = AARCH64_SYSREG_TRCPIDR5,
	ARM64_SYSREG_TRCPIDR6 = AARCH64_SYSREG_TRCPIDR6,
	ARM64_SYSREG_TRCPIDR7 = AARCH64_SYSREG_TRCPIDR7,
	ARM64_SYSREG_TRCPRGCTLR = AARCH64_SYSREG_TRCPRGCTLR,
	ARM64_SYSREG_TRCPROCSELR = AARCH64_SYSREG_TRCPROCSELR,
	ARM64_SYSREG_TRCQCTLR = AARCH64_SYSREG_TRCQCTLR,
	ARM64_SYSREG_TRCRSCTLR10 = AARCH64_SYSREG_TRCRSCTLR10,
	ARM64_SYSREG_TRCRSCTLR11 = AARCH64_SYSREG_TRCRSCTLR11,
	ARM64_SYSREG_TRCRSCTLR12 = AARCH64_SYSREG_TRCRSCTLR12,
	ARM64_SYSREG_TRCRSCTLR13 = AARCH64_SYSREG_TRCRSCTLR13,
	ARM64_SYSREG_TRCRSCTLR14 = AARCH64_SYSREG_TRCRSCTLR14,
	ARM64_SYSREG_TRCRSCTLR15 = AARCH64_SYSREG_TRCRSCTLR15,
	ARM64_SYSREG_TRCRSCTLR16 = AARCH64_SYSREG_TRCRSCTLR16,
	ARM64_SYSREG_TRCRSCTLR17 = AARCH64_SYSREG_TRCRSCTLR17,
	ARM64_SYSREG_TRCRSCTLR18 = AARCH64_SYSREG_TRCRSCTLR18,
	ARM64_SYSREG_TRCRSCTLR19 = AARCH64_SYSREG_TRCRSCTLR19,
	ARM64_SYSREG_TRCRSCTLR2 = AARCH64_SYSREG_TRCRSCTLR2,
	ARM64_SYSREG_TRCRSCTLR20 = AARCH64_SYSREG_TRCRSCTLR20,
	ARM64_SYSREG_TRCRSCTLR21 = AARCH64_SYSREG_TRCRSCTLR21,
	ARM64_SYSREG_TRCRSCTLR22 = AARCH64_SYSREG_TRCRSCTLR22,
	ARM64_SYSREG_TRCRSCTLR23 = AARCH64_SYSREG_TRCRSCTLR23,
	ARM64_SYSREG_TRCRSCTLR24 = AARCH64_SYSREG_TRCRSCTLR24,
	ARM64_SYSREG_TRCRSCTLR25 = AARCH64_SYSREG_TRCRSCTLR25,
	ARM64_SYSREG_TRCRSCTLR26 = AARCH64_SYSREG_TRCRSCTLR26,
	ARM64_SYSREG_TRCRSCTLR27 = AARCH64_SYSREG_TRCRSCTLR27,
	ARM64_SYSREG_TRCRSCTLR28 = AARCH64_SYSREG_TRCRSCTLR28,
	ARM64_SYSREG_TRCRSCTLR29 = AARCH64_SYSREG_TRCRSCTLR29,
	ARM64_SYSREG_TRCRSCTLR3 = AARCH64_SYSREG_TRCRSCTLR3,
	ARM64_SYSREG_TRCRSCTLR30 = AARCH64_SYSREG_TRCRSCTLR30,
	ARM64_SYSREG_TRCRSCTLR31 = AARCH64_SYSREG_TRCRSCTLR31,
	ARM64_SYSREG_TRCRSCTLR4 = AARCH64_SYSREG_TRCRSCTLR4,
	ARM64_SYSREG_TRCRSCTLR5 = AARCH64_SYSREG_TRCRSCTLR5,
	ARM64_SYSREG_TRCRSCTLR6 = AARCH64_SYSREG_TRCRSCTLR6,
	ARM64_SYSREG_TRCRSCTLR7 = AARCH64_SYSREG_TRCRSCTLR7,
	ARM64_SYSREG_TRCRSCTLR8 = AARCH64_SYSREG_TRCRSCTLR8,
	ARM64_SYSREG_TRCRSCTLR9 = AARCH64_SYSREG_TRCRSCTLR9,
	ARM64_SYSREG_TRCRSR = AARCH64_SYSREG_TRCRSR,
	ARM64_SYSREG_TRCSEQEVR0 = AARCH64_SYSREG_TRCSEQEVR0,
	ARM64_SYSREG_TRCSEQEVR1 = AARCH64_SYSREG_TRCSEQEVR1,
	ARM64_SYSREG_TRCSEQEVR2 = AARCH64_SYSREG_TRCSEQEVR2,
	ARM64_SYSREG_TRCSEQRSTEVR = AARCH64_SYSREG_TRCSEQRSTEVR,
	ARM64_SYSREG_TRCSEQSTR = AARCH64_SYSREG_TRCSEQSTR,
	ARM64_SYSREG_TRCSSCCR0 = AARCH64_SYSREG_TRCSSCCR0,
	ARM64_SYSREG_TRCSSCCR1 = AARCH64_SYSREG_TRCSSCCR1,
	ARM64_SYSREG_TRCSSCCR2 = AARCH64_SYSREG_TRCSSCCR2,
	ARM64_SYSREG_TRCSSCCR3 = AARCH64_SYSREG_TRCSSCCR3,
	ARM64_SYSREG_TRCSSCCR4 = AARCH64_SYSREG_TRCSSCCR4,
	ARM64_SYSREG_TRCSSCCR5 = AARCH64_SYSREG_TRCSSCCR5,
	ARM64_SYSREG_TRCSSCCR6 = AARCH64_SYSREG_TRCSSCCR6,
	ARM64_SYSREG_TRCSSCCR7 = AARCH64_SYSREG_TRCSSCCR7,
	ARM64_SYSREG_TRCSSCSR0 = AARCH64_SYSREG_TRCSSCSR0,
	ARM64_SYSREG_TRCSSCSR1 = AARCH64_SYSREG_TRCSSCSR1,
	ARM64_SYSREG_TRCSSCSR2 = AARCH64_SYSREG_TRCSSCSR2,
	ARM64_SYSREG_TRCSSCSR3 = AARCH64_SYSREG_TRCSSCSR3,
	ARM64_SYSREG_TRCSSCSR4 = AARCH64_SYSREG_TRCSSCSR4,
	ARM64_SYSREG_TRCSSCSR5 = AARCH64_SYSREG_TRCSSCSR5,
	ARM64_SYSREG_TRCSSCSR6 = AARCH64_SYSREG_TRCSSCSR6,
	ARM64_SYSREG_TRCSSCSR7 = AARCH64_SYSREG_TRCSSCSR7,
	ARM64_SYSREG_TRCSSPCICR0 = AARCH64_SYSREG_TRCSSPCICR0,
	ARM64_SYSREG_TRCSSPCICR1 = AARCH64_SYSREG_TRCSSPCICR1,
	ARM64_SYSREG_TRCSSPCICR2 = AARCH64_SYSREG_TRCSSPCICR2,
	ARM64_SYSREG_TRCSSPCICR3 = AARCH64_SYSREG_TRCSSPCICR3,
	ARM64_SYSREG_TRCSSPCICR4 = AARCH64_SYSREG_TRCSSPCICR4,
	ARM64_SYSREG_TRCSSPCICR5 = AARCH64_SYSREG_TRCSSPCICR5,
	ARM64_SYSREG_TRCSSPCICR6 = AARCH64_SYSREG_TRCSSPCICR6,
	ARM64_SYSREG_TRCSSPCICR7 = AARCH64_SYSREG_TRCSSPCICR7,
	ARM64_SYSREG_TRCSTALLCTLR = AARCH64_SYSREG_TRCSTALLCTLR,
	ARM64_SYSREG_TRCSTATR = AARCH64_SYSREG_TRCSTATR,
	ARM64_SYSREG_TRCSYNCPR = AARCH64_SYSREG_TRCSYNCPR,
	ARM64_SYSREG_TRCTRACEIDR = AARCH64_SYSREG_TRCTRACEIDR,
	ARM64_SYSREG_TRCTSCTLR = AARCH64_SYSREG_TRCTSCTLR,
	ARM64_SYSREG_TRCVDARCCTLR = AARCH64_SYSREG_TRCVDARCCTLR,
	ARM64_SYSREG_TRCVDCTLR = AARCH64_SYSREG_TRCVDCTLR,
	ARM64_SYSREG_TRCVDSACCTLR = AARCH64_SYSREG_TRCVDSACCTLR,
	ARM64_SYSREG_TRCVICTLR = AARCH64_SYSREG_TRCVICTLR,
	ARM64_SYSREG_TRCVIIECTLR = AARCH64_SYSREG_TRCVIIECTLR,
	ARM64_SYSREG_TRCVIPCSSCTLR = AARCH64_SYSREG_TRCVIPCSSCTLR,
	ARM64_SYSREG_TRCVISSCTLR = AARCH64_SYSREG_TRCVISSCTLR,
	ARM64_SYSREG_TRCVMIDCCTLR0 = AARCH64_SYSREG_TRCVMIDCCTLR0,
	ARM64_SYSREG_TRCVMIDCCTLR1 = AARCH64_SYSREG_TRCVMIDCCTLR1,
	ARM64_SYSREG_TRCVMIDCVR0 = AARCH64_SYSREG_TRCVMIDCVR0,
	ARM64_SYSREG_TRCVMIDCVR1 = AARCH64_SYSREG_TRCVMIDCVR1,
	ARM64_SYSREG_TRCVMIDCVR2 = AARCH64_SYSREG_TRCVMIDCVR2,
	ARM64_SYSREG_TRCVMIDCVR3 = AARCH64_SYSREG_TRCVMIDCVR3,
	ARM64_SYSREG_TRCVMIDCVR4 = AARCH64_SYSREG_TRCVMIDCVR4,
	ARM64_SYSREG_TRCVMIDCVR5 = AARCH64_SYSREG_TRCVMIDCVR5,
	ARM64_SYSREG_TRCVMIDCVR6 = AARCH64_SYSREG_TRCVMIDCVR6,
	ARM64_SYSREG_TRCVMIDCVR7 = AARCH64_SYSREG_TRCVMIDCVR7,
	ARM64_SYSREG_TRFCR_EL1 = AARCH64_SYSREG_TRFCR_EL1,
	ARM64_SYSREG_TRFCR_EL12 = AARCH64_SYSREG_TRFCR_EL12,
	ARM64_SYSREG_TRFCR_EL2 = AARCH64_SYSREG_TRFCR_EL2,
	ARM64_SYSREG_TTBR0_EL1 = AARCH64_SYSREG_TTBR0_EL1,
	ARM64_SYSREG_TTBR0_EL12 = AARCH64_SYSREG_TTBR0_EL12,
	ARM64_SYSREG_TTBR0_EL2 = AARCH64_SYSREG_TTBR0_EL2,
	ARM64_SYSREG_VSCTLR_EL2 = AARCH64_SYSREG_VSCTLR_EL2,
	ARM64_SYSREG_TTBR0_EL3 = AARCH64_SYSREG_TTBR0_EL3,
	ARM64_SYSREG_TTBR1_EL1 = AARCH64_SYSREG_TTBR1_EL1,
	ARM64_SYSREG_TTBR1_EL12 = AARCH64_SYSREG_TTBR1_EL12,
	ARM64_SYSREG_TTBR1_EL2 = AARCH64_SYSREG_TTBR1_EL2,
	ARM64_SYSREG_UAO = AARCH64_SYSREG_UAO,
	ARM64_SYSREG_VBAR_EL1 = AARCH64_SYSREG_VBAR_EL1,
	ARM64_SYSREG_VBAR_EL12 = AARCH64_SYSREG_VBAR_EL12,
	ARM64_SYSREG_VBAR_EL2 = AARCH64_SYSREG_VBAR_EL2,
	ARM64_SYSREG_VBAR_EL3 = AARCH64_SYSREG_VBAR_EL3,
	ARM64_SYSREG_VDISR_EL2 = AARCH64_SYSREG_VDISR_EL2,
	ARM64_SYSREG_VDISR_EL3 = AARCH64_SYSREG_VDISR_EL3,
	ARM64_SYSREG_VMECID_A_EL2 = AARCH64_SYSREG_VMECID_A_EL2,
	ARM64_SYSREG_VMECID_P_EL2 = AARCH64_SYSREG_VMECID_P_EL2,
	ARM64_SYSREG_VMPIDR_EL2 = AARCH64_SYSREG_VMPIDR_EL2,
	ARM64_SYSREG_VNCR_EL2 = AARCH64_SYSREG_VNCR_EL2,
	ARM64_SYSREG_VPIDR_EL2 = AARCH64_SYSREG_VPIDR_EL2,
	ARM64_SYSREG_VSESR_EL2 = AARCH64_SYSREG_VSESR_EL2,
	ARM64_SYSREG_VSESR_EL3 = AARCH64_SYSREG_VSESR_EL3,
	ARM64_SYSREG_VSTCR_EL2 = AARCH64_SYSREG_VSTCR_EL2,
	ARM64_SYSREG_VSTTBR_EL2 = AARCH64_SYSREG_VSTTBR_EL2,
	ARM64_SYSREG_VTCR_EL2 = AARCH64_SYSREG_VTCR_EL2,
	ARM64_SYSREG_VTTBR_EL2 = AARCH64_SYSREG_VTTBR_EL2,
	ARM64_SYSREG_ZCR_EL1 = AARCH64_SYSREG_ZCR_EL1,
	ARM64_SYSREG_ZCR_EL12 = AARCH64_SYSREG_ZCR_EL12,
	ARM64_SYSREG_ZCR_EL2 = AARCH64_SYSREG_ZCR_EL2,
	ARM64_SYSREG_ZCR_EL3 = AARCH64_SYSREG_ZCR_EL3,

	ARM64_SYSREG_ENDING = AARCH64_SYSREG_ENDING,
} arm64_sysreg;

typedef enum {

	ARM64_TSB_CSYNC = AARCH64_TSB_CSYNC,

	ARM64_TSB_ENDING = AARCH64_TSB_ENDING,
} arm64_tsb;

typedef aarch64_sysop_reg arm64_sysop_reg;

typedef aarch64_sysop_imm arm64_sysop_imm;

typedef aarch64_sysop_alias arm64_sysop_alias;

typedef enum {
  ARM64_OP_INVALID = AARCH64_OP_INVALID,
  ARM64_OP_REG = AARCH64_OP_REG,
  ARM64_OP_IMM = AARCH64_OP_IMM,
  ARM64_OP_MEM_REG = AARCH64_OP_MEM_REG,
  ARM64_OP_MEM_IMM = AARCH64_OP_MEM_IMM,
  ARM64_OP_MEM = AARCH64_OP_MEM,
  ARM64_OP_FP = AARCH64_OP_FP,
  ARM64_OP_CIMM = AARCH64_OP_CIMM,
  ARM64_OP_REG_MRS = AARCH64_OP_REG_MRS,
  ARM64_OP_REG_MSR = AARCH64_OP_REG_MSR,
  ARM64_OP_IMPLICIT_IMM_0 = AARCH64_OP_IMPLICIT_IMM_0,
	ARM64_OP_SVCR = AARCH64_OP_SVCR,
	ARM64_OP_AT = AARCH64_OP_AT,
	ARM64_OP_DB = AARCH64_OP_DB,
	ARM64_OP_DC = AARCH64_OP_DC,
	ARM64_OP_ISB = AARCH64_OP_ISB,
	ARM64_OP_TSB = AARCH64_OP_TSB,
	ARM64_OP_PRFM = AARCH64_OP_PRFM,
	ARM64_OP_SVEPRFM = AARCH64_OP_SVEPRFM,
	ARM64_OP_RPRFM = AARCH64_OP_RPRFM,
	ARM64_OP_PSTATEIMM0_15 = AARCH64_OP_PSTATEIMM0_15,
	ARM64_OP_PSTATEIMM0_1 = AARCH64_OP_PSTATEIMM0_1,
	ARM64_OP_PSB = AARCH64_OP_PSB,
	ARM64_OP_BTI = AARCH64_OP_BTI,
	ARM64_OP_SVEPREDPAT = AARCH64_OP_SVEPREDPAT,
	ARM64_OP_SVEVECLENSPECIFIER = AARCH64_OP_SVEVECLENSPECIFIER,
	ARM64_OP_SME = AARCH64_OP_SME,
	ARM64_OP_IMM_RANGE = AARCH64_OP_IMM_RANGE,
	ARM64_OP_TLBI = AARCH64_OP_TLBI,
	ARM64_OP_IC = AARCH64_OP_IC,
	ARM64_OP_DBNXS = AARCH64_OP_DBNXS,
	ARM64_OP_EXACTFPIMM = AARCH64_OP_EXACTFPIMM,
	ARM64_OP_SYSREG = AARCH64_OP_SYSREG,
	ARM64_OP_SYSIMM = AARCH64_OP_SYSIMM,
	ARM64_OP_SYSALIAS = AARCH64_OP_SYSALIAS,
	ARM64_OP_PRED = AARCH64_OP_PRED,
} arm64_op_type;

typedef aarch64_sysop arm64_sysop;

typedef enum {

	ARM64_REG_INVALID = AARCH64_REG_INVALID,
	ARM64_REG_FFR = AARCH64_REG_FFR,
	ARM64_REG_FP = AARCH64_REG_FP,
	ARM64_REG_FPCR = AARCH64_REG_FPCR,
	ARM64_REG_LR = AARCH64_REG_LR,
	ARM64_REG_NZCV = AARCH64_REG_NZCV,
	ARM64_REG_SP = AARCH64_REG_SP,
	ARM64_REG_VG = AARCH64_REG_VG,
	ARM64_REG_WSP = AARCH64_REG_WSP,
	ARM64_REG_WZR = AARCH64_REG_WZR,
	ARM64_REG_XZR = AARCH64_REG_XZR,
	ARM64_REG_ZA = AARCH64_REG_ZA,
	ARM64_REG_B0 = AARCH64_REG_B0,
	ARM64_REG_B1 = AARCH64_REG_B1,
	ARM64_REG_B2 = AARCH64_REG_B2,
	ARM64_REG_B3 = AARCH64_REG_B3,
	ARM64_REG_B4 = AARCH64_REG_B4,
	ARM64_REG_B5 = AARCH64_REG_B5,
	ARM64_REG_B6 = AARCH64_REG_B6,
	ARM64_REG_B7 = AARCH64_REG_B7,
	ARM64_REG_B8 = AARCH64_REG_B8,
	ARM64_REG_B9 = AARCH64_REG_B9,
	ARM64_REG_B10 = AARCH64_REG_B10,
	ARM64_REG_B11 = AARCH64_REG_B11,
	ARM64_REG_B12 = AARCH64_REG_B12,
	ARM64_REG_B13 = AARCH64_REG_B13,
	ARM64_REG_B14 = AARCH64_REG_B14,
	ARM64_REG_B15 = AARCH64_REG_B15,
	ARM64_REG_B16 = AARCH64_REG_B16,
	ARM64_REG_B17 = AARCH64_REG_B17,
	ARM64_REG_B18 = AARCH64_REG_B18,
	ARM64_REG_B19 = AARCH64_REG_B19,
	ARM64_REG_B20 = AARCH64_REG_B20,
	ARM64_REG_B21 = AARCH64_REG_B21,
	ARM64_REG_B22 = AARCH64_REG_B22,
	ARM64_REG_B23 = AARCH64_REG_B23,
	ARM64_REG_B24 = AARCH64_REG_B24,
	ARM64_REG_B25 = AARCH64_REG_B25,
	ARM64_REG_B26 = AARCH64_REG_B26,
	ARM64_REG_B27 = AARCH64_REG_B27,
	ARM64_REG_B28 = AARCH64_REG_B28,
	ARM64_REG_B29 = AARCH64_REG_B29,
	ARM64_REG_B30 = AARCH64_REG_B30,
	ARM64_REG_B31 = AARCH64_REG_B31,
	ARM64_REG_D0 = AARCH64_REG_D0,
	ARM64_REG_D1 = AARCH64_REG_D1,
	ARM64_REG_D2 = AARCH64_REG_D2,
	ARM64_REG_D3 = AARCH64_REG_D3,
	ARM64_REG_D4 = AARCH64_REG_D4,
	ARM64_REG_D5 = AARCH64_REG_D5,
	ARM64_REG_D6 = AARCH64_REG_D6,
	ARM64_REG_D7 = AARCH64_REG_D7,
	ARM64_REG_D8 = AARCH64_REG_D8,
	ARM64_REG_D9 = AARCH64_REG_D9,
	ARM64_REG_D10 = AARCH64_REG_D10,
	ARM64_REG_D11 = AARCH64_REG_D11,
	ARM64_REG_D12 = AARCH64_REG_D12,
	ARM64_REG_D13 = AARCH64_REG_D13,
	ARM64_REG_D14 = AARCH64_REG_D14,
	ARM64_REG_D15 = AARCH64_REG_D15,
	ARM64_REG_D16 = AARCH64_REG_D16,
	ARM64_REG_D17 = AARCH64_REG_D17,
	ARM64_REG_D18 = AARCH64_REG_D18,
	ARM64_REG_D19 = AARCH64_REG_D19,
	ARM64_REG_D20 = AARCH64_REG_D20,
	ARM64_REG_D21 = AARCH64_REG_D21,
	ARM64_REG_D22 = AARCH64_REG_D22,
	ARM64_REG_D23 = AARCH64_REG_D23,
	ARM64_REG_D24 = AARCH64_REG_D24,
	ARM64_REG_D25 = AARCH64_REG_D25,
	ARM64_REG_D26 = AARCH64_REG_D26,
	ARM64_REG_D27 = AARCH64_REG_D27,
	ARM64_REG_D28 = AARCH64_REG_D28,
	ARM64_REG_D29 = AARCH64_REG_D29,
	ARM64_REG_D30 = AARCH64_REG_D30,
	ARM64_REG_D31 = AARCH64_REG_D31,
	ARM64_REG_H0 = AARCH64_REG_H0,
	ARM64_REG_H1 = AARCH64_REG_H1,
	ARM64_REG_H2 = AARCH64_REG_H2,
	ARM64_REG_H3 = AARCH64_REG_H3,
	ARM64_REG_H4 = AARCH64_REG_H4,
	ARM64_REG_H5 = AARCH64_REG_H5,
	ARM64_REG_H6 = AARCH64_REG_H6,
	ARM64_REG_H7 = AARCH64_REG_H7,
	ARM64_REG_H8 = AARCH64_REG_H8,
	ARM64_REG_H9 = AARCH64_REG_H9,
	ARM64_REG_H10 = AARCH64_REG_H10,
	ARM64_REG_H11 = AARCH64_REG_H11,
	ARM64_REG_H12 = AARCH64_REG_H12,
	ARM64_REG_H13 = AARCH64_REG_H13,
	ARM64_REG_H14 = AARCH64_REG_H14,
	ARM64_REG_H15 = AARCH64_REG_H15,
	ARM64_REG_H16 = AARCH64_REG_H16,
	ARM64_REG_H17 = AARCH64_REG_H17,
	ARM64_REG_H18 = AARCH64_REG_H18,
	ARM64_REG_H19 = AARCH64_REG_H19,
	ARM64_REG_H20 = AARCH64_REG_H20,
	ARM64_REG_H21 = AARCH64_REG_H21,
	ARM64_REG_H22 = AARCH64_REG_H22,
	ARM64_REG_H23 = AARCH64_REG_H23,
	ARM64_REG_H24 = AARCH64_REG_H24,
	ARM64_REG_H25 = AARCH64_REG_H25,
	ARM64_REG_H26 = AARCH64_REG_H26,
	ARM64_REG_H27 = AARCH64_REG_H27,
	ARM64_REG_H28 = AARCH64_REG_H28,
	ARM64_REG_H29 = AARCH64_REG_H29,
	ARM64_REG_H30 = AARCH64_REG_H30,
	ARM64_REG_H31 = AARCH64_REG_H31,
	ARM64_REG_P0 = AARCH64_REG_P0,
	ARM64_REG_P1 = AARCH64_REG_P1,
	ARM64_REG_P2 = AARCH64_REG_P2,
	ARM64_REG_P3 = AARCH64_REG_P3,
	ARM64_REG_P4 = AARCH64_REG_P4,
	ARM64_REG_P5 = AARCH64_REG_P5,
	ARM64_REG_P6 = AARCH64_REG_P6,
	ARM64_REG_P7 = AARCH64_REG_P7,
	ARM64_REG_P8 = AARCH64_REG_P8,
	ARM64_REG_P9 = AARCH64_REG_P9,
	ARM64_REG_P10 = AARCH64_REG_P10,
	ARM64_REG_P11 = AARCH64_REG_P11,
	ARM64_REG_P12 = AARCH64_REG_P12,
	ARM64_REG_P13 = AARCH64_REG_P13,
	ARM64_REG_P14 = AARCH64_REG_P14,
	ARM64_REG_P15 = AARCH64_REG_P15,
	ARM64_REG_PN0 = AARCH64_REG_PN0,
	ARM64_REG_PN1 = AARCH64_REG_PN1,
	ARM64_REG_PN2 = AARCH64_REG_PN2,
	ARM64_REG_PN3 = AARCH64_REG_PN3,
	ARM64_REG_PN4 = AARCH64_REG_PN4,
	ARM64_REG_PN5 = AARCH64_REG_PN5,
	ARM64_REG_PN6 = AARCH64_REG_PN6,
	ARM64_REG_PN7 = AARCH64_REG_PN7,
	ARM64_REG_PN8 = AARCH64_REG_PN8,
	ARM64_REG_PN9 = AARCH64_REG_PN9,
	ARM64_REG_PN10 = AARCH64_REG_PN10,
	ARM64_REG_PN11 = AARCH64_REG_PN11,
	ARM64_REG_PN12 = AARCH64_REG_PN12,
	ARM64_REG_PN13 = AARCH64_REG_PN13,
	ARM64_REG_PN14 = AARCH64_REG_PN14,
	ARM64_REG_PN15 = AARCH64_REG_PN15,
	ARM64_REG_Q0 = AARCH64_REG_Q0,
	ARM64_REG_Q1 = AARCH64_REG_Q1,
	ARM64_REG_Q2 = AARCH64_REG_Q2,
	ARM64_REG_Q3 = AARCH64_REG_Q3,
	ARM64_REG_Q4 = AARCH64_REG_Q4,
	ARM64_REG_Q5 = AARCH64_REG_Q5,
	ARM64_REG_Q6 = AARCH64_REG_Q6,
	ARM64_REG_Q7 = AARCH64_REG_Q7,
	ARM64_REG_Q8 = AARCH64_REG_Q8,
	ARM64_REG_Q9 = AARCH64_REG_Q9,
	ARM64_REG_Q10 = AARCH64_REG_Q10,
	ARM64_REG_Q11 = AARCH64_REG_Q11,
	ARM64_REG_Q12 = AARCH64_REG_Q12,
	ARM64_REG_Q13 = AARCH64_REG_Q13,
	ARM64_REG_Q14 = AARCH64_REG_Q14,
	ARM64_REG_Q15 = AARCH64_REG_Q15,
	ARM64_REG_Q16 = AARCH64_REG_Q16,
	ARM64_REG_Q17 = AARCH64_REG_Q17,
	ARM64_REG_Q18 = AARCH64_REG_Q18,
	ARM64_REG_Q19 = AARCH64_REG_Q19,
	ARM64_REG_Q20 = AARCH64_REG_Q20,
	ARM64_REG_Q21 = AARCH64_REG_Q21,
	ARM64_REG_Q22 = AARCH64_REG_Q22,
	ARM64_REG_Q23 = AARCH64_REG_Q23,
	ARM64_REG_Q24 = AARCH64_REG_Q24,
	ARM64_REG_Q25 = AARCH64_REG_Q25,
	ARM64_REG_Q26 = AARCH64_REG_Q26,
	ARM64_REG_Q27 = AARCH64_REG_Q27,
	ARM64_REG_Q28 = AARCH64_REG_Q28,
	ARM64_REG_Q29 = AARCH64_REG_Q29,
	ARM64_REG_Q30 = AARCH64_REG_Q30,
	ARM64_REG_Q31 = AARCH64_REG_Q31,
	ARM64_REG_S0 = AARCH64_REG_S0,
	ARM64_REG_S1 = AARCH64_REG_S1,
	ARM64_REG_S2 = AARCH64_REG_S2,
	ARM64_REG_S3 = AARCH64_REG_S3,
	ARM64_REG_S4 = AARCH64_REG_S4,
	ARM64_REG_S5 = AARCH64_REG_S5,
	ARM64_REG_S6 = AARCH64_REG_S6,
	ARM64_REG_S7 = AARCH64_REG_S7,
	ARM64_REG_S8 = AARCH64_REG_S8,
	ARM64_REG_S9 = AARCH64_REG_S9,
	ARM64_REG_S10 = AARCH64_REG_S10,
	ARM64_REG_S11 = AARCH64_REG_S11,
	ARM64_REG_S12 = AARCH64_REG_S12,
	ARM64_REG_S13 = AARCH64_REG_S13,
	ARM64_REG_S14 = AARCH64_REG_S14,
	ARM64_REG_S15 = AARCH64_REG_S15,
	ARM64_REG_S16 = AARCH64_REG_S16,
	ARM64_REG_S17 = AARCH64_REG_S17,
	ARM64_REG_S18 = AARCH64_REG_S18,
	ARM64_REG_S19 = AARCH64_REG_S19,
	ARM64_REG_S20 = AARCH64_REG_S20,
	ARM64_REG_S21 = AARCH64_REG_S21,
	ARM64_REG_S22 = AARCH64_REG_S22,
	ARM64_REG_S23 = AARCH64_REG_S23,
	ARM64_REG_S24 = AARCH64_REG_S24,
	ARM64_REG_S25 = AARCH64_REG_S25,
	ARM64_REG_S26 = AARCH64_REG_S26,
	ARM64_REG_S27 = AARCH64_REG_S27,
	ARM64_REG_S28 = AARCH64_REG_S28,
	ARM64_REG_S29 = AARCH64_REG_S29,
	ARM64_REG_S30 = AARCH64_REG_S30,
	ARM64_REG_S31 = AARCH64_REG_S31,
	ARM64_REG_W0 = AARCH64_REG_W0,
	ARM64_REG_W1 = AARCH64_REG_W1,
	ARM64_REG_W2 = AARCH64_REG_W2,
	ARM64_REG_W3 = AARCH64_REG_W3,
	ARM64_REG_W4 = AARCH64_REG_W4,
	ARM64_REG_W5 = AARCH64_REG_W5,
	ARM64_REG_W6 = AARCH64_REG_W6,
	ARM64_REG_W7 = AARCH64_REG_W7,
	ARM64_REG_W8 = AARCH64_REG_W8,
	ARM64_REG_W9 = AARCH64_REG_W9,
	ARM64_REG_W10 = AARCH64_REG_W10,
	ARM64_REG_W11 = AARCH64_REG_W11,
	ARM64_REG_W12 = AARCH64_REG_W12,
	ARM64_REG_W13 = AARCH64_REG_W13,
	ARM64_REG_W14 = AARCH64_REG_W14,
	ARM64_REG_W15 = AARCH64_REG_W15,
	ARM64_REG_W16 = AARCH64_REG_W16,
	ARM64_REG_W17 = AARCH64_REG_W17,
	ARM64_REG_W18 = AARCH64_REG_W18,
	ARM64_REG_W19 = AARCH64_REG_W19,
	ARM64_REG_W20 = AARCH64_REG_W20,
	ARM64_REG_W21 = AARCH64_REG_W21,
	ARM64_REG_W22 = AARCH64_REG_W22,
	ARM64_REG_W23 = AARCH64_REG_W23,
	ARM64_REG_W24 = AARCH64_REG_W24,
	ARM64_REG_W25 = AARCH64_REG_W25,
	ARM64_REG_W26 = AARCH64_REG_W26,
	ARM64_REG_W27 = AARCH64_REG_W27,
	ARM64_REG_W28 = AARCH64_REG_W28,
	ARM64_REG_W29 = AARCH64_REG_W29,
	ARM64_REG_W30 = AARCH64_REG_W30,
	ARM64_REG_X0 = AARCH64_REG_X0,
	ARM64_REG_X1 = AARCH64_REG_X1,
	ARM64_REG_X2 = AARCH64_REG_X2,
	ARM64_REG_X3 = AARCH64_REG_X3,
	ARM64_REG_X4 = AARCH64_REG_X4,
	ARM64_REG_X5 = AARCH64_REG_X5,
	ARM64_REG_X6 = AARCH64_REG_X6,
	ARM64_REG_X7 = AARCH64_REG_X7,
	ARM64_REG_X8 = AARCH64_REG_X8,
	ARM64_REG_X9 = AARCH64_REG_X9,
	ARM64_REG_X10 = AARCH64_REG_X10,
	ARM64_REG_X11 = AARCH64_REG_X11,
	ARM64_REG_X12 = AARCH64_REG_X12,
	ARM64_REG_X13 = AARCH64_REG_X13,
	ARM64_REG_X14 = AARCH64_REG_X14,
	ARM64_REG_X15 = AARCH64_REG_X15,
	ARM64_REG_X16 = AARCH64_REG_X16,
	ARM64_REG_X17 = AARCH64_REG_X17,
	ARM64_REG_X18 = AARCH64_REG_X18,
	ARM64_REG_X19 = AARCH64_REG_X19,
	ARM64_REG_X20 = AARCH64_REG_X20,
	ARM64_REG_X21 = AARCH64_REG_X21,
	ARM64_REG_X22 = AARCH64_REG_X22,
	ARM64_REG_X23 = AARCH64_REG_X23,
	ARM64_REG_X24 = AARCH64_REG_X24,
	ARM64_REG_X25 = AARCH64_REG_X25,
	ARM64_REG_X26 = AARCH64_REG_X26,
	ARM64_REG_X27 = AARCH64_REG_X27,
	ARM64_REG_X28 = AARCH64_REG_X28,
	ARM64_REG_Z0 = AARCH64_REG_Z0,
	ARM64_REG_Z1 = AARCH64_REG_Z1,
	ARM64_REG_Z2 = AARCH64_REG_Z2,
	ARM64_REG_Z3 = AARCH64_REG_Z3,
	ARM64_REG_Z4 = AARCH64_REG_Z4,
	ARM64_REG_Z5 = AARCH64_REG_Z5,
	ARM64_REG_Z6 = AARCH64_REG_Z6,
	ARM64_REG_Z7 = AARCH64_REG_Z7,
	ARM64_REG_Z8 = AARCH64_REG_Z8,
	ARM64_REG_Z9 = AARCH64_REG_Z9,
	ARM64_REG_Z10 = AARCH64_REG_Z10,
	ARM64_REG_Z11 = AARCH64_REG_Z11,
	ARM64_REG_Z12 = AARCH64_REG_Z12,
	ARM64_REG_Z13 = AARCH64_REG_Z13,
	ARM64_REG_Z14 = AARCH64_REG_Z14,
	ARM64_REG_Z15 = AARCH64_REG_Z15,
	ARM64_REG_Z16 = AARCH64_REG_Z16,
	ARM64_REG_Z17 = AARCH64_REG_Z17,
	ARM64_REG_Z18 = AARCH64_REG_Z18,
	ARM64_REG_Z19 = AARCH64_REG_Z19,
	ARM64_REG_Z20 = AARCH64_REG_Z20,
	ARM64_REG_Z21 = AARCH64_REG_Z21,
	ARM64_REG_Z22 = AARCH64_REG_Z22,
	ARM64_REG_Z23 = AARCH64_REG_Z23,
	ARM64_REG_Z24 = AARCH64_REG_Z24,
	ARM64_REG_Z25 = AARCH64_REG_Z25,
	ARM64_REG_Z26 = AARCH64_REG_Z26,
	ARM64_REG_Z27 = AARCH64_REG_Z27,
	ARM64_REG_Z28 = AARCH64_REG_Z28,
	ARM64_REG_Z29 = AARCH64_REG_Z29,
	ARM64_REG_Z30 = AARCH64_REG_Z30,
	ARM64_REG_Z31 = AARCH64_REG_Z31,
	ARM64_REG_ZAB0 = AARCH64_REG_ZAB0,
	ARM64_REG_ZAD0 = AARCH64_REG_ZAD0,
	ARM64_REG_ZAD1 = AARCH64_REG_ZAD1,
	ARM64_REG_ZAD2 = AARCH64_REG_ZAD2,
	ARM64_REG_ZAD3 = AARCH64_REG_ZAD3,
	ARM64_REG_ZAD4 = AARCH64_REG_ZAD4,
	ARM64_REG_ZAD5 = AARCH64_REG_ZAD5,
	ARM64_REG_ZAD6 = AARCH64_REG_ZAD6,
	ARM64_REG_ZAD7 = AARCH64_REG_ZAD7,
	ARM64_REG_ZAH0 = AARCH64_REG_ZAH0,
	ARM64_REG_ZAH1 = AARCH64_REG_ZAH1,
	ARM64_REG_ZAQ0 = AARCH64_REG_ZAQ0,
	ARM64_REG_ZAQ1 = AARCH64_REG_ZAQ1,
	ARM64_REG_ZAQ2 = AARCH64_REG_ZAQ2,
	ARM64_REG_ZAQ3 = AARCH64_REG_ZAQ3,
	ARM64_REG_ZAQ4 = AARCH64_REG_ZAQ4,
	ARM64_REG_ZAQ5 = AARCH64_REG_ZAQ5,
	ARM64_REG_ZAQ6 = AARCH64_REG_ZAQ6,
	ARM64_REG_ZAQ7 = AARCH64_REG_ZAQ7,
	ARM64_REG_ZAQ8 = AARCH64_REG_ZAQ8,
	ARM64_REG_ZAQ9 = AARCH64_REG_ZAQ9,
	ARM64_REG_ZAQ10 = AARCH64_REG_ZAQ10,
	ARM64_REG_ZAQ11 = AARCH64_REG_ZAQ11,
	ARM64_REG_ZAQ12 = AARCH64_REG_ZAQ12,
	ARM64_REG_ZAQ13 = AARCH64_REG_ZAQ13,
	ARM64_REG_ZAQ14 = AARCH64_REG_ZAQ14,
	ARM64_REG_ZAQ15 = AARCH64_REG_ZAQ15,
	ARM64_REG_ZAS0 = AARCH64_REG_ZAS0,
	ARM64_REG_ZAS1 = AARCH64_REG_ZAS1,
	ARM64_REG_ZAS2 = AARCH64_REG_ZAS2,
	ARM64_REG_ZAS3 = AARCH64_REG_ZAS3,
	ARM64_REG_ZT0 = AARCH64_REG_ZT0,
	ARM64_REG_D0_D1 = AARCH64_REG_D0_D1,
	ARM64_REG_D1_D2 = AARCH64_REG_D1_D2,
	ARM64_REG_D2_D3 = AARCH64_REG_D2_D3,
	ARM64_REG_D3_D4 = AARCH64_REG_D3_D4,
	ARM64_REG_D4_D5 = AARCH64_REG_D4_D5,
	ARM64_REG_D5_D6 = AARCH64_REG_D5_D6,
	ARM64_REG_D6_D7 = AARCH64_REG_D6_D7,
	ARM64_REG_D7_D8 = AARCH64_REG_D7_D8,
	ARM64_REG_D8_D9 = AARCH64_REG_D8_D9,
	ARM64_REG_D9_D10 = AARCH64_REG_D9_D10,
	ARM64_REG_D10_D11 = AARCH64_REG_D10_D11,
	ARM64_REG_D11_D12 = AARCH64_REG_D11_D12,
	ARM64_REG_D12_D13 = AARCH64_REG_D12_D13,
	ARM64_REG_D13_D14 = AARCH64_REG_D13_D14,
	ARM64_REG_D14_D15 = AARCH64_REG_D14_D15,
	ARM64_REG_D15_D16 = AARCH64_REG_D15_D16,
	ARM64_REG_D16_D17 = AARCH64_REG_D16_D17,
	ARM64_REG_D17_D18 = AARCH64_REG_D17_D18,
	ARM64_REG_D18_D19 = AARCH64_REG_D18_D19,
	ARM64_REG_D19_D20 = AARCH64_REG_D19_D20,
	ARM64_REG_D20_D21 = AARCH64_REG_D20_D21,
	ARM64_REG_D21_D22 = AARCH64_REG_D21_D22,
	ARM64_REG_D22_D23 = AARCH64_REG_D22_D23,
	ARM64_REG_D23_D24 = AARCH64_REG_D23_D24,
	ARM64_REG_D24_D25 = AARCH64_REG_D24_D25,
	ARM64_REG_D25_D26 = AARCH64_REG_D25_D26,
	ARM64_REG_D26_D27 = AARCH64_REG_D26_D27,
	ARM64_REG_D27_D28 = AARCH64_REG_D27_D28,
	ARM64_REG_D28_D29 = AARCH64_REG_D28_D29,
	ARM64_REG_D29_D30 = AARCH64_REG_D29_D30,
	ARM64_REG_D30_D31 = AARCH64_REG_D30_D31,
	ARM64_REG_D31_D0 = AARCH64_REG_D31_D0,
	ARM64_REG_D0_D1_D2_D3 = AARCH64_REG_D0_D1_D2_D3,
	ARM64_REG_D1_D2_D3_D4 = AARCH64_REG_D1_D2_D3_D4,
	ARM64_REG_D2_D3_D4_D5 = AARCH64_REG_D2_D3_D4_D5,
	ARM64_REG_D3_D4_D5_D6 = AARCH64_REG_D3_D4_D5_D6,
	ARM64_REG_D4_D5_D6_D7 = AARCH64_REG_D4_D5_D6_D7,
	ARM64_REG_D5_D6_D7_D8 = AARCH64_REG_D5_D6_D7_D8,
	ARM64_REG_D6_D7_D8_D9 = AARCH64_REG_D6_D7_D8_D9,
	ARM64_REG_D7_D8_D9_D10 = AARCH64_REG_D7_D8_D9_D10,
	ARM64_REG_D8_D9_D10_D11 = AARCH64_REG_D8_D9_D10_D11,
	ARM64_REG_D9_D10_D11_D12 = AARCH64_REG_D9_D10_D11_D12,
	ARM64_REG_D10_D11_D12_D13 = AARCH64_REG_D10_D11_D12_D13,
	ARM64_REG_D11_D12_D13_D14 = AARCH64_REG_D11_D12_D13_D14,
	ARM64_REG_D12_D13_D14_D15 = AARCH64_REG_D12_D13_D14_D15,
	ARM64_REG_D13_D14_D15_D16 = AARCH64_REG_D13_D14_D15_D16,
	ARM64_REG_D14_D15_D16_D17 = AARCH64_REG_D14_D15_D16_D17,
	ARM64_REG_D15_D16_D17_D18 = AARCH64_REG_D15_D16_D17_D18,
	ARM64_REG_D16_D17_D18_D19 = AARCH64_REG_D16_D17_D18_D19,
	ARM64_REG_D17_D18_D19_D20 = AARCH64_REG_D17_D18_D19_D20,
	ARM64_REG_D18_D19_D20_D21 = AARCH64_REG_D18_D19_D20_D21,
	ARM64_REG_D19_D20_D21_D22 = AARCH64_REG_D19_D20_D21_D22,
	ARM64_REG_D20_D21_D22_D23 = AARCH64_REG_D20_D21_D22_D23,
	ARM64_REG_D21_D22_D23_D24 = AARCH64_REG_D21_D22_D23_D24,
	ARM64_REG_D22_D23_D24_D25 = AARCH64_REG_D22_D23_D24_D25,
	ARM64_REG_D23_D24_D25_D26 = AARCH64_REG_D23_D24_D25_D26,
	ARM64_REG_D24_D25_D26_D27 = AARCH64_REG_D24_D25_D26_D27,
	ARM64_REG_D25_D26_D27_D28 = AARCH64_REG_D25_D26_D27_D28,
	ARM64_REG_D26_D27_D28_D29 = AARCH64_REG_D26_D27_D28_D29,
	ARM64_REG_D27_D28_D29_D30 = AARCH64_REG_D27_D28_D29_D30,
	ARM64_REG_D28_D29_D30_D31 = AARCH64_REG_D28_D29_D30_D31,
	ARM64_REG_D29_D30_D31_D0 = AARCH64_REG_D29_D30_D31_D0,
	ARM64_REG_D30_D31_D0_D1 = AARCH64_REG_D30_D31_D0_D1,
	ARM64_REG_D31_D0_D1_D2 = AARCH64_REG_D31_D0_D1_D2,
	ARM64_REG_D0_D1_D2 = AARCH64_REG_D0_D1_D2,
	ARM64_REG_D1_D2_D3 = AARCH64_REG_D1_D2_D3,
	ARM64_REG_D2_D3_D4 = AARCH64_REG_D2_D3_D4,
	ARM64_REG_D3_D4_D5 = AARCH64_REG_D3_D4_D5,
	ARM64_REG_D4_D5_D6 = AARCH64_REG_D4_D5_D6,
	ARM64_REG_D5_D6_D7 = AARCH64_REG_D5_D6_D7,
	ARM64_REG_D6_D7_D8 = AARCH64_REG_D6_D7_D8,
	ARM64_REG_D7_D8_D9 = AARCH64_REG_D7_D8_D9,
	ARM64_REG_D8_D9_D10 = AARCH64_REG_D8_D9_D10,
	ARM64_REG_D9_D10_D11 = AARCH64_REG_D9_D10_D11,
	ARM64_REG_D10_D11_D12 = AARCH64_REG_D10_D11_D12,
	ARM64_REG_D11_D12_D13 = AARCH64_REG_D11_D12_D13,
	ARM64_REG_D12_D13_D14 = AARCH64_REG_D12_D13_D14,
	ARM64_REG_D13_D14_D15 = AARCH64_REG_D13_D14_D15,
	ARM64_REG_D14_D15_D16 = AARCH64_REG_D14_D15_D16,
	ARM64_REG_D15_D16_D17 = AARCH64_REG_D15_D16_D17,
	ARM64_REG_D16_D17_D18 = AARCH64_REG_D16_D17_D18,
	ARM64_REG_D17_D18_D19 = AARCH64_REG_D17_D18_D19,
	ARM64_REG_D18_D19_D20 = AARCH64_REG_D18_D19_D20,
	ARM64_REG_D19_D20_D21 = AARCH64_REG_D19_D20_D21,
	ARM64_REG_D20_D21_D22 = AARCH64_REG_D20_D21_D22,
	ARM64_REG_D21_D22_D23 = AARCH64_REG_D21_D22_D23,
	ARM64_REG_D22_D23_D24 = AARCH64_REG_D22_D23_D24,
	ARM64_REG_D23_D24_D25 = AARCH64_REG_D23_D24_D25,
	ARM64_REG_D24_D25_D26 = AARCH64_REG_D24_D25_D26,
	ARM64_REG_D25_D26_D27 = AARCH64_REG_D25_D26_D27,
	ARM64_REG_D26_D27_D28 = AARCH64_REG_D26_D27_D28,
	ARM64_REG_D27_D28_D29 = AARCH64_REG_D27_D28_D29,
	ARM64_REG_D28_D29_D30 = AARCH64_REG_D28_D29_D30,
	ARM64_REG_D29_D30_D31 = AARCH64_REG_D29_D30_D31,
	ARM64_REG_D30_D31_D0 = AARCH64_REG_D30_D31_D0,
	ARM64_REG_D31_D0_D1 = AARCH64_REG_D31_D0_D1,
	ARM64_REG_P0_P1 = AARCH64_REG_P0_P1,
	ARM64_REG_P1_P2 = AARCH64_REG_P1_P2,
	ARM64_REG_P2_P3 = AARCH64_REG_P2_P3,
	ARM64_REG_P3_P4 = AARCH64_REG_P3_P4,
	ARM64_REG_P4_P5 = AARCH64_REG_P4_P5,
	ARM64_REG_P5_P6 = AARCH64_REG_P5_P6,
	ARM64_REG_P6_P7 = AARCH64_REG_P6_P7,
	ARM64_REG_P7_P8 = AARCH64_REG_P7_P8,
	ARM64_REG_P8_P9 = AARCH64_REG_P8_P9,
	ARM64_REG_P9_P10 = AARCH64_REG_P9_P10,
	ARM64_REG_P10_P11 = AARCH64_REG_P10_P11,
	ARM64_REG_P11_P12 = AARCH64_REG_P11_P12,
	ARM64_REG_P12_P13 = AARCH64_REG_P12_P13,
	ARM64_REG_P13_P14 = AARCH64_REG_P13_P14,
	ARM64_REG_P14_P15 = AARCH64_REG_P14_P15,
	ARM64_REG_P15_P0 = AARCH64_REG_P15_P0,
	ARM64_REG_Q0_Q1 = AARCH64_REG_Q0_Q1,
	ARM64_REG_Q1_Q2 = AARCH64_REG_Q1_Q2,
	ARM64_REG_Q2_Q3 = AARCH64_REG_Q2_Q3,
	ARM64_REG_Q3_Q4 = AARCH64_REG_Q3_Q4,
	ARM64_REG_Q4_Q5 = AARCH64_REG_Q4_Q5,
	ARM64_REG_Q5_Q6 = AARCH64_REG_Q5_Q6,
	ARM64_REG_Q6_Q7 = AARCH64_REG_Q6_Q7,
	ARM64_REG_Q7_Q8 = AARCH64_REG_Q7_Q8,
	ARM64_REG_Q8_Q9 = AARCH64_REG_Q8_Q9,
	ARM64_REG_Q9_Q10 = AARCH64_REG_Q9_Q10,
	ARM64_REG_Q10_Q11 = AARCH64_REG_Q10_Q11,
	ARM64_REG_Q11_Q12 = AARCH64_REG_Q11_Q12,
	ARM64_REG_Q12_Q13 = AARCH64_REG_Q12_Q13,
	ARM64_REG_Q13_Q14 = AARCH64_REG_Q13_Q14,
	ARM64_REG_Q14_Q15 = AARCH64_REG_Q14_Q15,
	ARM64_REG_Q15_Q16 = AARCH64_REG_Q15_Q16,
	ARM64_REG_Q16_Q17 = AARCH64_REG_Q16_Q17,
	ARM64_REG_Q17_Q18 = AARCH64_REG_Q17_Q18,
	ARM64_REG_Q18_Q19 = AARCH64_REG_Q18_Q19,
	ARM64_REG_Q19_Q20 = AARCH64_REG_Q19_Q20,
	ARM64_REG_Q20_Q21 = AARCH64_REG_Q20_Q21,
	ARM64_REG_Q21_Q22 = AARCH64_REG_Q21_Q22,
	ARM64_REG_Q22_Q23 = AARCH64_REG_Q22_Q23,
	ARM64_REG_Q23_Q24 = AARCH64_REG_Q23_Q24,
	ARM64_REG_Q24_Q25 = AARCH64_REG_Q24_Q25,
	ARM64_REG_Q25_Q26 = AARCH64_REG_Q25_Q26,
	ARM64_REG_Q26_Q27 = AARCH64_REG_Q26_Q27,
	ARM64_REG_Q27_Q28 = AARCH64_REG_Q27_Q28,
	ARM64_REG_Q28_Q29 = AARCH64_REG_Q28_Q29,
	ARM64_REG_Q29_Q30 = AARCH64_REG_Q29_Q30,
	ARM64_REG_Q30_Q31 = AARCH64_REG_Q30_Q31,
	ARM64_REG_Q31_Q0 = AARCH64_REG_Q31_Q0,
	ARM64_REG_Q0_Q1_Q2_Q3 = AARCH64_REG_Q0_Q1_Q2_Q3,
	ARM64_REG_Q1_Q2_Q3_Q4 = AARCH64_REG_Q1_Q2_Q3_Q4,
	ARM64_REG_Q2_Q3_Q4_Q5 = AARCH64_REG_Q2_Q3_Q4_Q5,
	ARM64_REG_Q3_Q4_Q5_Q6 = AARCH64_REG_Q3_Q4_Q5_Q6,
	ARM64_REG_Q4_Q5_Q6_Q7 = AARCH64_REG_Q4_Q5_Q6_Q7,
	ARM64_REG_Q5_Q6_Q7_Q8 = AARCH64_REG_Q5_Q6_Q7_Q8,
	ARM64_REG_Q6_Q7_Q8_Q9 = AARCH64_REG_Q6_Q7_Q8_Q9,
	ARM64_REG_Q7_Q8_Q9_Q10 = AARCH64_REG_Q7_Q8_Q9_Q10,
	ARM64_REG_Q8_Q9_Q10_Q11 = AARCH64_REG_Q8_Q9_Q10_Q11,
	ARM64_REG_Q9_Q10_Q11_Q12 = AARCH64_REG_Q9_Q10_Q11_Q12,
	ARM64_REG_Q10_Q11_Q12_Q13 = AARCH64_REG_Q10_Q11_Q12_Q13,
	ARM64_REG_Q11_Q12_Q13_Q14 = AARCH64_REG_Q11_Q12_Q13_Q14,
	ARM64_REG_Q12_Q13_Q14_Q15 = AARCH64_REG_Q12_Q13_Q14_Q15,
	ARM64_REG_Q13_Q14_Q15_Q16 = AARCH64_REG_Q13_Q14_Q15_Q16,
	ARM64_REG_Q14_Q15_Q16_Q17 = AARCH64_REG_Q14_Q15_Q16_Q17,
	ARM64_REG_Q15_Q16_Q17_Q18 = AARCH64_REG_Q15_Q16_Q17_Q18,
	ARM64_REG_Q16_Q17_Q18_Q19 = AARCH64_REG_Q16_Q17_Q18_Q19,
	ARM64_REG_Q17_Q18_Q19_Q20 = AARCH64_REG_Q17_Q18_Q19_Q20,
	ARM64_REG_Q18_Q19_Q20_Q21 = AARCH64_REG_Q18_Q19_Q20_Q21,
	ARM64_REG_Q19_Q20_Q21_Q22 = AARCH64_REG_Q19_Q20_Q21_Q22,
	ARM64_REG_Q20_Q21_Q22_Q23 = AARCH64_REG_Q20_Q21_Q22_Q23,
	ARM64_REG_Q21_Q22_Q23_Q24 = AARCH64_REG_Q21_Q22_Q23_Q24,
	ARM64_REG_Q22_Q23_Q24_Q25 = AARCH64_REG_Q22_Q23_Q24_Q25,
	ARM64_REG_Q23_Q24_Q25_Q26 = AARCH64_REG_Q23_Q24_Q25_Q26,
	ARM64_REG_Q24_Q25_Q26_Q27 = AARCH64_REG_Q24_Q25_Q26_Q27,
	ARM64_REG_Q25_Q26_Q27_Q28 = AARCH64_REG_Q25_Q26_Q27_Q28,
	ARM64_REG_Q26_Q27_Q28_Q29 = AARCH64_REG_Q26_Q27_Q28_Q29,
	ARM64_REG_Q27_Q28_Q29_Q30 = AARCH64_REG_Q27_Q28_Q29_Q30,
	ARM64_REG_Q28_Q29_Q30_Q31 = AARCH64_REG_Q28_Q29_Q30_Q31,
	ARM64_REG_Q29_Q30_Q31_Q0 = AARCH64_REG_Q29_Q30_Q31_Q0,
	ARM64_REG_Q30_Q31_Q0_Q1 = AARCH64_REG_Q30_Q31_Q0_Q1,
	ARM64_REG_Q31_Q0_Q1_Q2 = AARCH64_REG_Q31_Q0_Q1_Q2,
	ARM64_REG_Q0_Q1_Q2 = AARCH64_REG_Q0_Q1_Q2,
	ARM64_REG_Q1_Q2_Q3 = AARCH64_REG_Q1_Q2_Q3,
	ARM64_REG_Q2_Q3_Q4 = AARCH64_REG_Q2_Q3_Q4,
	ARM64_REG_Q3_Q4_Q5 = AARCH64_REG_Q3_Q4_Q5,
	ARM64_REG_Q4_Q5_Q6 = AARCH64_REG_Q4_Q5_Q6,
	ARM64_REG_Q5_Q6_Q7 = AARCH64_REG_Q5_Q6_Q7,
	ARM64_REG_Q6_Q7_Q8 = AARCH64_REG_Q6_Q7_Q8,
	ARM64_REG_Q7_Q8_Q9 = AARCH64_REG_Q7_Q8_Q9,
	ARM64_REG_Q8_Q9_Q10 = AARCH64_REG_Q8_Q9_Q10,
	ARM64_REG_Q9_Q10_Q11 = AARCH64_REG_Q9_Q10_Q11,
	ARM64_REG_Q10_Q11_Q12 = AARCH64_REG_Q10_Q11_Q12,
	ARM64_REG_Q11_Q12_Q13 = AARCH64_REG_Q11_Q12_Q13,
	ARM64_REG_Q12_Q13_Q14 = AARCH64_REG_Q12_Q13_Q14,
	ARM64_REG_Q13_Q14_Q15 = AARCH64_REG_Q13_Q14_Q15,
	ARM64_REG_Q14_Q15_Q16 = AARCH64_REG_Q14_Q15_Q16,
	ARM64_REG_Q15_Q16_Q17 = AARCH64_REG_Q15_Q16_Q17,
	ARM64_REG_Q16_Q17_Q18 = AARCH64_REG_Q16_Q17_Q18,
	ARM64_REG_Q17_Q18_Q19 = AARCH64_REG_Q17_Q18_Q19,
	ARM64_REG_Q18_Q19_Q20 = AARCH64_REG_Q18_Q19_Q20,
	ARM64_REG_Q19_Q20_Q21 = AARCH64_REG_Q19_Q20_Q21,
	ARM64_REG_Q20_Q21_Q22 = AARCH64_REG_Q20_Q21_Q22,
	ARM64_REG_Q21_Q22_Q23 = AARCH64_REG_Q21_Q22_Q23,
	ARM64_REG_Q22_Q23_Q24 = AARCH64_REG_Q22_Q23_Q24,
	ARM64_REG_Q23_Q24_Q25 = AARCH64_REG_Q23_Q24_Q25,
	ARM64_REG_Q24_Q25_Q26 = AARCH64_REG_Q24_Q25_Q26,
	ARM64_REG_Q25_Q26_Q27 = AARCH64_REG_Q25_Q26_Q27,
	ARM64_REG_Q26_Q27_Q28 = AARCH64_REG_Q26_Q27_Q28,
	ARM64_REG_Q27_Q28_Q29 = AARCH64_REG_Q27_Q28_Q29,
	ARM64_REG_Q28_Q29_Q30 = AARCH64_REG_Q28_Q29_Q30,
	ARM64_REG_Q29_Q30_Q31 = AARCH64_REG_Q29_Q30_Q31,
	ARM64_REG_Q30_Q31_Q0 = AARCH64_REG_Q30_Q31_Q0,
	ARM64_REG_Q31_Q0_Q1 = AARCH64_REG_Q31_Q0_Q1,
	ARM64_REG_X22_X23_X24_X25_X26_X27_X28_FP = AARCH64_REG_X22_X23_X24_X25_X26_X27_X28_FP,
	ARM64_REG_X0_X1_X2_X3_X4_X5_X6_X7 = AARCH64_REG_X0_X1_X2_X3_X4_X5_X6_X7,
	ARM64_REG_X2_X3_X4_X5_X6_X7_X8_X9 = AARCH64_REG_X2_X3_X4_X5_X6_X7_X8_X9,
	ARM64_REG_X4_X5_X6_X7_X8_X9_X10_X11 = AARCH64_REG_X4_X5_X6_X7_X8_X9_X10_X11,
	ARM64_REG_X6_X7_X8_X9_X10_X11_X12_X13 = AARCH64_REG_X6_X7_X8_X9_X10_X11_X12_X13,
	ARM64_REG_X8_X9_X10_X11_X12_X13_X14_X15 = AARCH64_REG_X8_X9_X10_X11_X12_X13_X14_X15,
	ARM64_REG_X10_X11_X12_X13_X14_X15_X16_X17 = AARCH64_REG_X10_X11_X12_X13_X14_X15_X16_X17,
	ARM64_REG_X12_X13_X14_X15_X16_X17_X18_X19 = AARCH64_REG_X12_X13_X14_X15_X16_X17_X18_X19,
	ARM64_REG_X14_X15_X16_X17_X18_X19_X20_X21 = AARCH64_REG_X14_X15_X16_X17_X18_X19_X20_X21,
	ARM64_REG_X16_X17_X18_X19_X20_X21_X22_X23 = AARCH64_REG_X16_X17_X18_X19_X20_X21_X22_X23,
	ARM64_REG_X18_X19_X20_X21_X22_X23_X24_X25 = AARCH64_REG_X18_X19_X20_X21_X22_X23_X24_X25,
	ARM64_REG_X20_X21_X22_X23_X24_X25_X26_X27 = AARCH64_REG_X20_X21_X22_X23_X24_X25_X26_X27,
	ARM64_REG_W30_WZR = AARCH64_REG_W30_WZR,
	ARM64_REG_W0_W1 = AARCH64_REG_W0_W1,
	ARM64_REG_W2_W3 = AARCH64_REG_W2_W3,
	ARM64_REG_W4_W5 = AARCH64_REG_W4_W5,
	ARM64_REG_W6_W7 = AARCH64_REG_W6_W7,
	ARM64_REG_W8_W9 = AARCH64_REG_W8_W9,
	ARM64_REG_W10_W11 = AARCH64_REG_W10_W11,
	ARM64_REG_W12_W13 = AARCH64_REG_W12_W13,
	ARM64_REG_W14_W15 = AARCH64_REG_W14_W15,
	ARM64_REG_W16_W17 = AARCH64_REG_W16_W17,
	ARM64_REG_W18_W19 = AARCH64_REG_W18_W19,
	ARM64_REG_W20_W21 = AARCH64_REG_W20_W21,
	ARM64_REG_W22_W23 = AARCH64_REG_W22_W23,
	ARM64_REG_W24_W25 = AARCH64_REG_W24_W25,
	ARM64_REG_W26_W27 = AARCH64_REG_W26_W27,
	ARM64_REG_W28_W29 = AARCH64_REG_W28_W29,
	ARM64_REG_LR_XZR = AARCH64_REG_LR_XZR,
	ARM64_REG_X28_FP = AARCH64_REG_X28_FP,
	ARM64_REG_X0_X1 = AARCH64_REG_X0_X1,
	ARM64_REG_X2_X3 = AARCH64_REG_X2_X3,
	ARM64_REG_X4_X5 = AARCH64_REG_X4_X5,
	ARM64_REG_X6_X7 = AARCH64_REG_X6_X7,
	ARM64_REG_X8_X9 = AARCH64_REG_X8_X9,
	ARM64_REG_X10_X11 = AARCH64_REG_X10_X11,
	ARM64_REG_X12_X13 = AARCH64_REG_X12_X13,
	ARM64_REG_X14_X15 = AARCH64_REG_X14_X15,
	ARM64_REG_X16_X17 = AARCH64_REG_X16_X17,
	ARM64_REG_X18_X19 = AARCH64_REG_X18_X19,
	ARM64_REG_X20_X21 = AARCH64_REG_X20_X21,
	ARM64_REG_X22_X23 = AARCH64_REG_X22_X23,
	ARM64_REG_X24_X25 = AARCH64_REG_X24_X25,
	ARM64_REG_X26_X27 = AARCH64_REG_X26_X27,
	ARM64_REG_Z0_Z1 = AARCH64_REG_Z0_Z1,
	ARM64_REG_Z1_Z2 = AARCH64_REG_Z1_Z2,
	ARM64_REG_Z2_Z3 = AARCH64_REG_Z2_Z3,
	ARM64_REG_Z3_Z4 = AARCH64_REG_Z3_Z4,
	ARM64_REG_Z4_Z5 = AARCH64_REG_Z4_Z5,
	ARM64_REG_Z5_Z6 = AARCH64_REG_Z5_Z6,
	ARM64_REG_Z6_Z7 = AARCH64_REG_Z6_Z7,
	ARM64_REG_Z7_Z8 = AARCH64_REG_Z7_Z8,
	ARM64_REG_Z8_Z9 = AARCH64_REG_Z8_Z9,
	ARM64_REG_Z9_Z10 = AARCH64_REG_Z9_Z10,
	ARM64_REG_Z10_Z11 = AARCH64_REG_Z10_Z11,
	ARM64_REG_Z11_Z12 = AARCH64_REG_Z11_Z12,
	ARM64_REG_Z12_Z13 = AARCH64_REG_Z12_Z13,
	ARM64_REG_Z13_Z14 = AARCH64_REG_Z13_Z14,
	ARM64_REG_Z14_Z15 = AARCH64_REG_Z14_Z15,
	ARM64_REG_Z15_Z16 = AARCH64_REG_Z15_Z16,
	ARM64_REG_Z16_Z17 = AARCH64_REG_Z16_Z17,
	ARM64_REG_Z17_Z18 = AARCH64_REG_Z17_Z18,
	ARM64_REG_Z18_Z19 = AARCH64_REG_Z18_Z19,
	ARM64_REG_Z19_Z20 = AARCH64_REG_Z19_Z20,
	ARM64_REG_Z20_Z21 = AARCH64_REG_Z20_Z21,
	ARM64_REG_Z21_Z22 = AARCH64_REG_Z21_Z22,
	ARM64_REG_Z22_Z23 = AARCH64_REG_Z22_Z23,
	ARM64_REG_Z23_Z24 = AARCH64_REG_Z23_Z24,
	ARM64_REG_Z24_Z25 = AARCH64_REG_Z24_Z25,
	ARM64_REG_Z25_Z26 = AARCH64_REG_Z25_Z26,
	ARM64_REG_Z26_Z27 = AARCH64_REG_Z26_Z27,
	ARM64_REG_Z27_Z28 = AARCH64_REG_Z27_Z28,
	ARM64_REG_Z28_Z29 = AARCH64_REG_Z28_Z29,
	ARM64_REG_Z29_Z30 = AARCH64_REG_Z29_Z30,
	ARM64_REG_Z30_Z31 = AARCH64_REG_Z30_Z31,
	ARM64_REG_Z31_Z0 = AARCH64_REG_Z31_Z0,
	ARM64_REG_Z0_Z1_Z2_Z3 = AARCH64_REG_Z0_Z1_Z2_Z3,
	ARM64_REG_Z1_Z2_Z3_Z4 = AARCH64_REG_Z1_Z2_Z3_Z4,
	ARM64_REG_Z2_Z3_Z4_Z5 = AARCH64_REG_Z2_Z3_Z4_Z5,
	ARM64_REG_Z3_Z4_Z5_Z6 = AARCH64_REG_Z3_Z4_Z5_Z6,
	ARM64_REG_Z4_Z5_Z6_Z7 = AARCH64_REG_Z4_Z5_Z6_Z7,
	ARM64_REG_Z5_Z6_Z7_Z8 = AARCH64_REG_Z5_Z6_Z7_Z8,
	ARM64_REG_Z6_Z7_Z8_Z9 = AARCH64_REG_Z6_Z7_Z8_Z9,
	ARM64_REG_Z7_Z8_Z9_Z10 = AARCH64_REG_Z7_Z8_Z9_Z10,
	ARM64_REG_Z8_Z9_Z10_Z11 = AARCH64_REG_Z8_Z9_Z10_Z11,
	ARM64_REG_Z9_Z10_Z11_Z12 = AARCH64_REG_Z9_Z10_Z11_Z12,
	ARM64_REG_Z10_Z11_Z12_Z13 = AARCH64_REG_Z10_Z11_Z12_Z13,
	ARM64_REG_Z11_Z12_Z13_Z14 = AARCH64_REG_Z11_Z12_Z13_Z14,
	ARM64_REG_Z12_Z13_Z14_Z15 = AARCH64_REG_Z12_Z13_Z14_Z15,
	ARM64_REG_Z13_Z14_Z15_Z16 = AARCH64_REG_Z13_Z14_Z15_Z16,
	ARM64_REG_Z14_Z15_Z16_Z17 = AARCH64_REG_Z14_Z15_Z16_Z17,
	ARM64_REG_Z15_Z16_Z17_Z18 = AARCH64_REG_Z15_Z16_Z17_Z18,
	ARM64_REG_Z16_Z17_Z18_Z19 = AARCH64_REG_Z16_Z17_Z18_Z19,
	ARM64_REG_Z17_Z18_Z19_Z20 = AARCH64_REG_Z17_Z18_Z19_Z20,
	ARM64_REG_Z18_Z19_Z20_Z21 = AARCH64_REG_Z18_Z19_Z20_Z21,
	ARM64_REG_Z19_Z20_Z21_Z22 = AARCH64_REG_Z19_Z20_Z21_Z22,
	ARM64_REG_Z20_Z21_Z22_Z23 = AARCH64_REG_Z20_Z21_Z22_Z23,
	ARM64_REG_Z21_Z22_Z23_Z24 = AARCH64_REG_Z21_Z22_Z23_Z24,
	ARM64_REG_Z22_Z23_Z24_Z25 = AARCH64_REG_Z22_Z23_Z24_Z25,
	ARM64_REG_Z23_Z24_Z25_Z26 = AARCH64_REG_Z23_Z24_Z25_Z26,
	ARM64_REG_Z24_Z25_Z26_Z27 = AARCH64_REG_Z24_Z25_Z26_Z27,
	ARM64_REG_Z25_Z26_Z27_Z28 = AARCH64_REG_Z25_Z26_Z27_Z28,
	ARM64_REG_Z26_Z27_Z28_Z29 = AARCH64_REG_Z26_Z27_Z28_Z29,
	ARM64_REG_Z27_Z28_Z29_Z30 = AARCH64_REG_Z27_Z28_Z29_Z30,
	ARM64_REG_Z28_Z29_Z30_Z31 = AARCH64_REG_Z28_Z29_Z30_Z31,
	ARM64_REG_Z29_Z30_Z31_Z0 = AARCH64_REG_Z29_Z30_Z31_Z0,
	ARM64_REG_Z30_Z31_Z0_Z1 = AARCH64_REG_Z30_Z31_Z0_Z1,
	ARM64_REG_Z31_Z0_Z1_Z2 = AARCH64_REG_Z31_Z0_Z1_Z2,
	ARM64_REG_Z0_Z1_Z2 = AARCH64_REG_Z0_Z1_Z2,
	ARM64_REG_Z1_Z2_Z3 = AARCH64_REG_Z1_Z2_Z3,
	ARM64_REG_Z2_Z3_Z4 = AARCH64_REG_Z2_Z3_Z4,
	ARM64_REG_Z3_Z4_Z5 = AARCH64_REG_Z3_Z4_Z5,
	ARM64_REG_Z4_Z5_Z6 = AARCH64_REG_Z4_Z5_Z6,
	ARM64_REG_Z5_Z6_Z7 = AARCH64_REG_Z5_Z6_Z7,
	ARM64_REG_Z6_Z7_Z8 = AARCH64_REG_Z6_Z7_Z8,
	ARM64_REG_Z7_Z8_Z9 = AARCH64_REG_Z7_Z8_Z9,
	ARM64_REG_Z8_Z9_Z10 = AARCH64_REG_Z8_Z9_Z10,
	ARM64_REG_Z9_Z10_Z11 = AARCH64_REG_Z9_Z10_Z11,
	ARM64_REG_Z10_Z11_Z12 = AARCH64_REG_Z10_Z11_Z12,
	ARM64_REG_Z11_Z12_Z13 = AARCH64_REG_Z11_Z12_Z13,
	ARM64_REG_Z12_Z13_Z14 = AARCH64_REG_Z12_Z13_Z14,
	ARM64_REG_Z13_Z14_Z15 = AARCH64_REG_Z13_Z14_Z15,
	ARM64_REG_Z14_Z15_Z16 = AARCH64_REG_Z14_Z15_Z16,
	ARM64_REG_Z15_Z16_Z17 = AARCH64_REG_Z15_Z16_Z17,
	ARM64_REG_Z16_Z17_Z18 = AARCH64_REG_Z16_Z17_Z18,
	ARM64_REG_Z17_Z18_Z19 = AARCH64_REG_Z17_Z18_Z19,
	ARM64_REG_Z18_Z19_Z20 = AARCH64_REG_Z18_Z19_Z20,
	ARM64_REG_Z19_Z20_Z21 = AARCH64_REG_Z19_Z20_Z21,
	ARM64_REG_Z20_Z21_Z22 = AARCH64_REG_Z20_Z21_Z22,
	ARM64_REG_Z21_Z22_Z23 = AARCH64_REG_Z21_Z22_Z23,
	ARM64_REG_Z22_Z23_Z24 = AARCH64_REG_Z22_Z23_Z24,
	ARM64_REG_Z23_Z24_Z25 = AARCH64_REG_Z23_Z24_Z25,
	ARM64_REG_Z24_Z25_Z26 = AARCH64_REG_Z24_Z25_Z26,
	ARM64_REG_Z25_Z26_Z27 = AARCH64_REG_Z25_Z26_Z27,
	ARM64_REG_Z26_Z27_Z28 = AARCH64_REG_Z26_Z27_Z28,
	ARM64_REG_Z27_Z28_Z29 = AARCH64_REG_Z27_Z28_Z29,
	ARM64_REG_Z28_Z29_Z30 = AARCH64_REG_Z28_Z29_Z30,
	ARM64_REG_Z29_Z30_Z31 = AARCH64_REG_Z29_Z30_Z31,
	ARM64_REG_Z30_Z31_Z0 = AARCH64_REG_Z30_Z31_Z0,
	ARM64_REG_Z31_Z0_Z1 = AARCH64_REG_Z31_Z0_Z1,
	ARM64_REG_Z16_Z24 = AARCH64_REG_Z16_Z24,
	ARM64_REG_Z17_Z25 = AARCH64_REG_Z17_Z25,
	ARM64_REG_Z18_Z26 = AARCH64_REG_Z18_Z26,
	ARM64_REG_Z19_Z27 = AARCH64_REG_Z19_Z27,
	ARM64_REG_Z20_Z28 = AARCH64_REG_Z20_Z28,
	ARM64_REG_Z21_Z29 = AARCH64_REG_Z21_Z29,
	ARM64_REG_Z22_Z30 = AARCH64_REG_Z22_Z30,
	ARM64_REG_Z23_Z31 = AARCH64_REG_Z23_Z31,
	ARM64_REG_Z0_Z8 = AARCH64_REG_Z0_Z8,
	ARM64_REG_Z1_Z9 = AARCH64_REG_Z1_Z9,
	ARM64_REG_Z2_Z10 = AARCH64_REG_Z2_Z10,
	ARM64_REG_Z3_Z11 = AARCH64_REG_Z3_Z11,
	ARM64_REG_Z4_Z12 = AARCH64_REG_Z4_Z12,
	ARM64_REG_Z5_Z13 = AARCH64_REG_Z5_Z13,
	ARM64_REG_Z6_Z14 = AARCH64_REG_Z6_Z14,
	ARM64_REG_Z7_Z15 = AARCH64_REG_Z7_Z15,
	ARM64_REG_Z16_Z20_Z24_Z28 = AARCH64_REG_Z16_Z20_Z24_Z28,
	ARM64_REG_Z17_Z21_Z25_Z29 = AARCH64_REG_Z17_Z21_Z25_Z29,
	ARM64_REG_Z18_Z22_Z26_Z30 = AARCH64_REG_Z18_Z22_Z26_Z30,
	ARM64_REG_Z19_Z23_Z27_Z31 = AARCH64_REG_Z19_Z23_Z27_Z31,
	ARM64_REG_Z0_Z4_Z8_Z12 = AARCH64_REG_Z0_Z4_Z8_Z12,
	ARM64_REG_Z1_Z5_Z9_Z13 = AARCH64_REG_Z1_Z5_Z9_Z13,
	ARM64_REG_Z2_Z6_Z10_Z14 = AARCH64_REG_Z2_Z6_Z10_Z14,
	ARM64_REG_Z3_Z7_Z11_Z15 = AARCH64_REG_Z3_Z7_Z11_Z15,
	ARM64_REG_ENDING = AARCH64_REG_ENDING,


  ARM64_REG_IP0 = AARCH64_REG_IP0,
  ARM64_REG_IP1 = AARCH64_REG_IP1,
  ARM64_REG_X29 = AARCH64_REG_X29,
  ARM64_REG_X30 = AARCH64_REG_X30,
} arm64_reg;

typedef aarch64_op_mem arm64_op_mem;

typedef enum {
	ARM64_SME_OP_INVALID = AARCH64_SME_OP_INVALID,
	ARM64_SME_OP_TILE = AARCH64_SME_OP_TILE,
	ARM64_SME_OP_TILE_VEC = AARCH64_SME_OP_TILE_VEC,
} arm64_sme_op_type;

#define ARM64_SLICE_IMM_INVALID UINT16_MAX
#define ARM64_SLICE_IMM_RANGE_INVALID UINT8_MAX

typedef aarch64_imm_range arm64_imm_range;

typedef aarch64_op_sme arm64_op_sme;

typedef aarch64_op_pred arm64_op_pred;

typedef cs_aarch64_op cs_arm64_op;

typedef aarch64_suppl_info arm64_suppl_info;

#define NUM_ARM64_OPS 16

typedef cs_aarch64 cs_arm64;

typedef enum {

	ARM64_INS_INVALID = AARCH64_INS_INVALID,
	ARM64_INS_ABS = AARCH64_INS_ABS,
	ARM64_INS_ADCLB = AARCH64_INS_ADCLB,
	ARM64_INS_ADCLT = AARCH64_INS_ADCLT,
	ARM64_INS_ADCS = AARCH64_INS_ADCS,
	ARM64_INS_ADC = AARCH64_INS_ADC,
	ARM64_INS_ADDG = AARCH64_INS_ADDG,
	ARM64_INS_ADDHA = AARCH64_INS_ADDHA,
	ARM64_INS_ADDHNB = AARCH64_INS_ADDHNB,
	ARM64_INS_ADDHNT = AARCH64_INS_ADDHNT,
	ARM64_INS_ADDHN = AARCH64_INS_ADDHN,
	ARM64_INS_ADDHN2 = AARCH64_INS_ADDHN2,
	ARM64_INS_ADDPL = AARCH64_INS_ADDPL,
	ARM64_INS_ADDPT = AARCH64_INS_ADDPT,
	ARM64_INS_ADDP = AARCH64_INS_ADDP,
	ARM64_INS_ADDQV = AARCH64_INS_ADDQV,
	ARM64_INS_ADDSPL = AARCH64_INS_ADDSPL,
	ARM64_INS_ADDSVL = AARCH64_INS_ADDSVL,
	ARM64_INS_ADDS = AARCH64_INS_ADDS,
	ARM64_INS_ADDVA = AARCH64_INS_ADDVA,
	ARM64_INS_ADDVL = AARCH64_INS_ADDVL,
	ARM64_INS_ADDV = AARCH64_INS_ADDV,
	ARM64_INS_ADD = AARCH64_INS_ADD,
	ARM64_INS_ADR = AARCH64_INS_ADR,
	ARM64_INS_ADRP = AARCH64_INS_ADRP,
	ARM64_INS_AESD = AARCH64_INS_AESD,
	ARM64_INS_AESE = AARCH64_INS_AESE,
	ARM64_INS_AESIMC = AARCH64_INS_AESIMC,
	ARM64_INS_AESMC = AARCH64_INS_AESMC,
	ARM64_INS_ANDQV = AARCH64_INS_ANDQV,
	ARM64_INS_ANDS = AARCH64_INS_ANDS,
	ARM64_INS_ANDV = AARCH64_INS_ANDV,
	ARM64_INS_AND = AARCH64_INS_AND,
	ARM64_INS_ASRD = AARCH64_INS_ASRD,
	ARM64_INS_ASRR = AARCH64_INS_ASRR,
	ARM64_INS_ASR = AARCH64_INS_ASR,
	ARM64_INS_AUTDA = AARCH64_INS_AUTDA,
	ARM64_INS_AUTDB = AARCH64_INS_AUTDB,
	ARM64_INS_AUTDZA = AARCH64_INS_AUTDZA,
	ARM64_INS_AUTDZB = AARCH64_INS_AUTDZB,
	ARM64_INS_AUTIA = AARCH64_INS_AUTIA,
	ARM64_INS_HINT = AARCH64_INS_HINT,
	ARM64_INS_AUTIA171615 = AARCH64_INS_AUTIA171615,
	ARM64_INS_AUTIASPPC = AARCH64_INS_AUTIASPPC,
	ARM64_INS_AUTIB = AARCH64_INS_AUTIB,
	ARM64_INS_AUTIB171615 = AARCH64_INS_AUTIB171615,
	ARM64_INS_AUTIBSPPC = AARCH64_INS_AUTIBSPPC,
	ARM64_INS_AUTIZA = AARCH64_INS_AUTIZA,
	ARM64_INS_AUTIZB = AARCH64_INS_AUTIZB,
	ARM64_INS_AXFLAG = AARCH64_INS_AXFLAG,
	ARM64_INS_B = AARCH64_INS_B,
	ARM64_INS_BCAX = AARCH64_INS_BCAX,
	ARM64_INS_BC = AARCH64_INS_BC,
	ARM64_INS_BDEP = AARCH64_INS_BDEP,
	ARM64_INS_BEXT = AARCH64_INS_BEXT,
	ARM64_INS_BFDOT = AARCH64_INS_BFDOT,
	ARM64_INS_BF1CVTL2 = AARCH64_INS_BF1CVTL2,
	ARM64_INS_BF1CVTLT = AARCH64_INS_BF1CVTLT,
	ARM64_INS_BF1CVTL = AARCH64_INS_BF1CVTL,
	ARM64_INS_BF1CVT = AARCH64_INS_BF1CVT,
	ARM64_INS_BF2CVTL2 = AARCH64_INS_BF2CVTL2,
	ARM64_INS_BF2CVTLT = AARCH64_INS_BF2CVTLT,
	ARM64_INS_BF2CVTL = AARCH64_INS_BF2CVTL,
	ARM64_INS_BF2CVT = AARCH64_INS_BF2CVT,
	ARM64_INS_BFADD = AARCH64_INS_BFADD,
	ARM64_INS_BFCLAMP = AARCH64_INS_BFCLAMP,
	ARM64_INS_BFCVT = AARCH64_INS_BFCVT,
	ARM64_INS_BFCVTN = AARCH64_INS_BFCVTN,
	ARM64_INS_BFCVTN2 = AARCH64_INS_BFCVTN2,
	ARM64_INS_BFCVTNT = AARCH64_INS_BFCVTNT,
	ARM64_INS_BFMAXNM = AARCH64_INS_BFMAXNM,
	ARM64_INS_BFMAX = AARCH64_INS_BFMAX,
	ARM64_INS_BFMINNM = AARCH64_INS_BFMINNM,
	ARM64_INS_BFMIN = AARCH64_INS_BFMIN,
	ARM64_INS_BFMLALB = AARCH64_INS_BFMLALB,
	ARM64_INS_BFMLALT = AARCH64_INS_BFMLALT,
	ARM64_INS_BFMLAL = AARCH64_INS_BFMLAL,
	ARM64_INS_BFMLA = AARCH64_INS_BFMLA,
	ARM64_INS_BFMLSLB = AARCH64_INS_BFMLSLB,
	ARM64_INS_BFMLSLT = AARCH64_INS_BFMLSLT,
	ARM64_INS_BFMLSL = AARCH64_INS_BFMLSL,
	ARM64_INS_BFMLS = AARCH64_INS_BFMLS,
	ARM64_INS_BFMMLA = AARCH64_INS_BFMMLA,
	ARM64_INS_BFMOPA = AARCH64_INS_BFMOPA,
	ARM64_INS_BFMOPS = AARCH64_INS_BFMOPS,
	ARM64_INS_BFMUL = AARCH64_INS_BFMUL,
	ARM64_INS_BFM = AARCH64_INS_BFM,
	ARM64_INS_BFSUB = AARCH64_INS_BFSUB,
	ARM64_INS_BFVDOT = AARCH64_INS_BFVDOT,
	ARM64_INS_BGRP = AARCH64_INS_BGRP,
	ARM64_INS_BICS = AARCH64_INS_BICS,
	ARM64_INS_BIC = AARCH64_INS_BIC,
	ARM64_INS_BIF = AARCH64_INS_BIF,
	ARM64_INS_BIT = AARCH64_INS_BIT,
	ARM64_INS_BL = AARCH64_INS_BL,
	ARM64_INS_BLR = AARCH64_INS_BLR,
	ARM64_INS_BLRAA = AARCH64_INS_BLRAA,
	ARM64_INS_BLRAAZ = AARCH64_INS_BLRAAZ,
	ARM64_INS_BLRAB = AARCH64_INS_BLRAB,
	ARM64_INS_BLRABZ = AARCH64_INS_BLRABZ,
	ARM64_INS_BMOPA = AARCH64_INS_BMOPA,
	ARM64_INS_BMOPS = AARCH64_INS_BMOPS,
	ARM64_INS_BR = AARCH64_INS_BR,
	ARM64_INS_BRAA = AARCH64_INS_BRAA,
	ARM64_INS_BRAAZ = AARCH64_INS_BRAAZ,
	ARM64_INS_BRAB = AARCH64_INS_BRAB,
	ARM64_INS_BRABZ = AARCH64_INS_BRABZ,
	ARM64_INS_BRB = AARCH64_INS_BRB,
	ARM64_INS_BRK = AARCH64_INS_BRK,
	ARM64_INS_BRKAS = AARCH64_INS_BRKAS,
	ARM64_INS_BRKA = AARCH64_INS_BRKA,
	ARM64_INS_BRKBS = AARCH64_INS_BRKBS,
	ARM64_INS_BRKB = AARCH64_INS_BRKB,
	ARM64_INS_BRKNS = AARCH64_INS_BRKNS,
	ARM64_INS_BRKN = AARCH64_INS_BRKN,
	ARM64_INS_BRKPAS = AARCH64_INS_BRKPAS,
	ARM64_INS_BRKPA = AARCH64_INS_BRKPA,
	ARM64_INS_BRKPBS = AARCH64_INS_BRKPBS,
	ARM64_INS_BRKPB = AARCH64_INS_BRKPB,
	ARM64_INS_BSL1N = AARCH64_INS_BSL1N,
	ARM64_INS_BSL2N = AARCH64_INS_BSL2N,
	ARM64_INS_BSL = AARCH64_INS_BSL,
	ARM64_INS_CADD = AARCH64_INS_CADD,
	ARM64_INS_CASAB = AARCH64_INS_CASAB,
	ARM64_INS_CASAH = AARCH64_INS_CASAH,
	ARM64_INS_CASALB = AARCH64_INS_CASALB,
	ARM64_INS_CASALH = AARCH64_INS_CASALH,
	ARM64_INS_CASAL = AARCH64_INS_CASAL,
	ARM64_INS_CASA = AARCH64_INS_CASA,
	ARM64_INS_CASB = AARCH64_INS_CASB,
	ARM64_INS_CASH = AARCH64_INS_CASH,
	ARM64_INS_CASLB = AARCH64_INS_CASLB,
	ARM64_INS_CASLH = AARCH64_INS_CASLH,
	ARM64_INS_CASL = AARCH64_INS_CASL,
	ARM64_INS_CASPAL = AARCH64_INS_CASPAL,
	ARM64_INS_CASPA = AARCH64_INS_CASPA,
	ARM64_INS_CASPL = AARCH64_INS_CASPL,
	ARM64_INS_CASP = AARCH64_INS_CASP,
	ARM64_INS_CAS = AARCH64_INS_CAS,
	ARM64_INS_CBNZ = AARCH64_INS_CBNZ,
	ARM64_INS_CBZ = AARCH64_INS_CBZ,
	ARM64_INS_CCMN = AARCH64_INS_CCMN,
	ARM64_INS_CCMP = AARCH64_INS_CCMP,
	ARM64_INS_CDOT = AARCH64_INS_CDOT,
	ARM64_INS_CFINV = AARCH64_INS_CFINV,
	ARM64_INS_CLASTA = AARCH64_INS_CLASTA,
	ARM64_INS_CLASTB = AARCH64_INS_CLASTB,
	ARM64_INS_CLREX = AARCH64_INS_CLREX,
	ARM64_INS_CLS = AARCH64_INS_CLS,
	ARM64_INS_CLZ = AARCH64_INS_CLZ,
	ARM64_INS_CMEQ = AARCH64_INS_CMEQ,
	ARM64_INS_CMGE = AARCH64_INS_CMGE,
	ARM64_INS_CMGT = AARCH64_INS_CMGT,
	ARM64_INS_CMHI = AARCH64_INS_CMHI,
	ARM64_INS_CMHS = AARCH64_INS_CMHS,
	ARM64_INS_CMLA = AARCH64_INS_CMLA,
	ARM64_INS_CMLE = AARCH64_INS_CMLE,
	ARM64_INS_CMLT = AARCH64_INS_CMLT,
	ARM64_INS_CMPEQ = AARCH64_INS_CMPEQ,
	ARM64_INS_CMPGE = AARCH64_INS_CMPGE,
	ARM64_INS_CMPGT = AARCH64_INS_CMPGT,
	ARM64_INS_CMPHI = AARCH64_INS_CMPHI,
	ARM64_INS_CMPHS = AARCH64_INS_CMPHS,
	ARM64_INS_CMPLE = AARCH64_INS_CMPLE,
	ARM64_INS_CMPLO = AARCH64_INS_CMPLO,
	ARM64_INS_CMPLS = AARCH64_INS_CMPLS,
	ARM64_INS_CMPLT = AARCH64_INS_CMPLT,
	ARM64_INS_CMPNE = AARCH64_INS_CMPNE,
	ARM64_INS_CMTST = AARCH64_INS_CMTST,
	ARM64_INS_CNOT = AARCH64_INS_CNOT,
	ARM64_INS_CNTB = AARCH64_INS_CNTB,
	ARM64_INS_CNTD = AARCH64_INS_CNTD,
	ARM64_INS_CNTH = AARCH64_INS_CNTH,
	ARM64_INS_CNTP = AARCH64_INS_CNTP,
	ARM64_INS_CNTW = AARCH64_INS_CNTW,
	ARM64_INS_CNT = AARCH64_INS_CNT,
	ARM64_INS_COMPACT = AARCH64_INS_COMPACT,
	ARM64_INS_CPYE = AARCH64_INS_CPYE,
	ARM64_INS_CPYEN = AARCH64_INS_CPYEN,
	ARM64_INS_CPYERN = AARCH64_INS_CPYERN,
	ARM64_INS_CPYERT = AARCH64_INS_CPYERT,
	ARM64_INS_CPYERTN = AARCH64_INS_CPYERTN,
	ARM64_INS_CPYERTRN = AARCH64_INS_CPYERTRN,
	ARM64_INS_CPYERTWN = AARCH64_INS_CPYERTWN,
	ARM64_INS_CPYET = AARCH64_INS_CPYET,
	ARM64_INS_CPYETN = AARCH64_INS_CPYETN,
	ARM64_INS_CPYETRN = AARCH64_INS_CPYETRN,
	ARM64_INS_CPYETWN = AARCH64_INS_CPYETWN,
	ARM64_INS_CPYEWN = AARCH64_INS_CPYEWN,
	ARM64_INS_CPYEWT = AARCH64_INS_CPYEWT,
	ARM64_INS_CPYEWTN = AARCH64_INS_CPYEWTN,
	ARM64_INS_CPYEWTRN = AARCH64_INS_CPYEWTRN,
	ARM64_INS_CPYEWTWN = AARCH64_INS_CPYEWTWN,
	ARM64_INS_CPYFE = AARCH64_INS_CPYFE,
	ARM64_INS_CPYFEN = AARCH64_INS_CPYFEN,
	ARM64_INS_CPYFERN = AARCH64_INS_CPYFERN,
	ARM64_INS_CPYFERT = AARCH64_INS_CPYFERT,
	ARM64_INS_CPYFERTN = AARCH64_INS_CPYFERTN,
	ARM64_INS_CPYFERTRN = AARCH64_INS_CPYFERTRN,
	ARM64_INS_CPYFERTWN = AARCH64_INS_CPYFERTWN,
	ARM64_INS_CPYFET = AARCH64_INS_CPYFET,
	ARM64_INS_CPYFETN = AARCH64_INS_CPYFETN,
	ARM64_INS_CPYFETRN = AARCH64_INS_CPYFETRN,
	ARM64_INS_CPYFETWN = AARCH64_INS_CPYFETWN,
	ARM64_INS_CPYFEWN = AARCH64_INS_CPYFEWN,
	ARM64_INS_CPYFEWT = AARCH64_INS_CPYFEWT,
	ARM64_INS_CPYFEWTN = AARCH64_INS_CPYFEWTN,
	ARM64_INS_CPYFEWTRN = AARCH64_INS_CPYFEWTRN,
	ARM64_INS_CPYFEWTWN = AARCH64_INS_CPYFEWTWN,
	ARM64_INS_CPYFM = AARCH64_INS_CPYFM,
	ARM64_INS_CPYFMN = AARCH64_INS_CPYFMN,
	ARM64_INS_CPYFMRN = AARCH64_INS_CPYFMRN,
	ARM64_INS_CPYFMRT = AARCH64_INS_CPYFMRT,
	ARM64_INS_CPYFMRTN = AARCH64_INS_CPYFMRTN,
	ARM64_INS_CPYFMRTRN = AARCH64_INS_CPYFMRTRN,
	ARM64_INS_CPYFMRTWN = AARCH64_INS_CPYFMRTWN,
	ARM64_INS_CPYFMT = AARCH64_INS_CPYFMT,
	ARM64_INS_CPYFMTN = AARCH64_INS_CPYFMTN,
	ARM64_INS_CPYFMTRN = AARCH64_INS_CPYFMTRN,
	ARM64_INS_CPYFMTWN = AARCH64_INS_CPYFMTWN,
	ARM64_INS_CPYFMWN = AARCH64_INS_CPYFMWN,
	ARM64_INS_CPYFMWT = AARCH64_INS_CPYFMWT,
	ARM64_INS_CPYFMWTN = AARCH64_INS_CPYFMWTN,
	ARM64_INS_CPYFMWTRN = AARCH64_INS_CPYFMWTRN,
	ARM64_INS_CPYFMWTWN = AARCH64_INS_CPYFMWTWN,
	ARM64_INS_CPYFP = AARCH64_INS_CPYFP,
	ARM64_INS_CPYFPN = AARCH64_INS_CPYFPN,
	ARM64_INS_CPYFPRN = AARCH64_INS_CPYFPRN,
	ARM64_INS_CPYFPRT = AARCH64_INS_CPYFPRT,
	ARM64_INS_CPYFPRTN = AARCH64_INS_CPYFPRTN,
	ARM64_INS_CPYFPRTRN = AARCH64_INS_CPYFPRTRN,
	ARM64_INS_CPYFPRTWN = AARCH64_INS_CPYFPRTWN,
	ARM64_INS_CPYFPT = AARCH64_INS_CPYFPT,
	ARM64_INS_CPYFPTN = AARCH64_INS_CPYFPTN,
	ARM64_INS_CPYFPTRN = AARCH64_INS_CPYFPTRN,
	ARM64_INS_CPYFPTWN = AARCH64_INS_CPYFPTWN,
	ARM64_INS_CPYFPWN = AARCH64_INS_CPYFPWN,
	ARM64_INS_CPYFPWT = AARCH64_INS_CPYFPWT,
	ARM64_INS_CPYFPWTN = AARCH64_INS_CPYFPWTN,
	ARM64_INS_CPYFPWTRN = AARCH64_INS_CPYFPWTRN,
	ARM64_INS_CPYFPWTWN = AARCH64_INS_CPYFPWTWN,
	ARM64_INS_CPYM = AARCH64_INS_CPYM,
	ARM64_INS_CPYMN = AARCH64_INS_CPYMN,
	ARM64_INS_CPYMRN = AARCH64_INS_CPYMRN,
	ARM64_INS_CPYMRT = AARCH64_INS_CPYMRT,
	ARM64_INS_CPYMRTN = AARCH64_INS_CPYMRTN,
	ARM64_INS_CPYMRTRN = AARCH64_INS_CPYMRTRN,
	ARM64_INS_CPYMRTWN = AARCH64_INS_CPYMRTWN,
	ARM64_INS_CPYMT = AARCH64_INS_CPYMT,
	ARM64_INS_CPYMTN = AARCH64_INS_CPYMTN,
	ARM64_INS_CPYMTRN = AARCH64_INS_CPYMTRN,
	ARM64_INS_CPYMTWN = AARCH64_INS_CPYMTWN,
	ARM64_INS_CPYMWN = AARCH64_INS_CPYMWN,
	ARM64_INS_CPYMWT = AARCH64_INS_CPYMWT,
	ARM64_INS_CPYMWTN = AARCH64_INS_CPYMWTN,
	ARM64_INS_CPYMWTRN = AARCH64_INS_CPYMWTRN,
	ARM64_INS_CPYMWTWN = AARCH64_INS_CPYMWTWN,
	ARM64_INS_CPYP = AARCH64_INS_CPYP,
	ARM64_INS_CPYPN = AARCH64_INS_CPYPN,
	ARM64_INS_CPYPRN = AARCH64_INS_CPYPRN,
	ARM64_INS_CPYPRT = AARCH64_INS_CPYPRT,
	ARM64_INS_CPYPRTN = AARCH64_INS_CPYPRTN,
	ARM64_INS_CPYPRTRN = AARCH64_INS_CPYPRTRN,
	ARM64_INS_CPYPRTWN = AARCH64_INS_CPYPRTWN,
	ARM64_INS_CPYPT = AARCH64_INS_CPYPT,
	ARM64_INS_CPYPTN = AARCH64_INS_CPYPTN,
	ARM64_INS_CPYPTRN = AARCH64_INS_CPYPTRN,
	ARM64_INS_CPYPTWN = AARCH64_INS_CPYPTWN,
	ARM64_INS_CPYPWN = AARCH64_INS_CPYPWN,
	ARM64_INS_CPYPWT = AARCH64_INS_CPYPWT,
	ARM64_INS_CPYPWTN = AARCH64_INS_CPYPWTN,
	ARM64_INS_CPYPWTRN = AARCH64_INS_CPYPWTRN,
	ARM64_INS_CPYPWTWN = AARCH64_INS_CPYPWTWN,
	ARM64_INS_CPY = AARCH64_INS_CPY,
	ARM64_INS_CRC32B = AARCH64_INS_CRC32B,
	ARM64_INS_CRC32CB = AARCH64_INS_CRC32CB,
	ARM64_INS_CRC32CH = AARCH64_INS_CRC32CH,
	ARM64_INS_CRC32CW = AARCH64_INS_CRC32CW,
	ARM64_INS_CRC32CX = AARCH64_INS_CRC32CX,
	ARM64_INS_CRC32H = AARCH64_INS_CRC32H,
	ARM64_INS_CRC32W = AARCH64_INS_CRC32W,
	ARM64_INS_CRC32X = AARCH64_INS_CRC32X,
	ARM64_INS_CSEL = AARCH64_INS_CSEL,
	ARM64_INS_CSINC = AARCH64_INS_CSINC,
	ARM64_INS_CSINV = AARCH64_INS_CSINV,
	ARM64_INS_CSNEG = AARCH64_INS_CSNEG,
	ARM64_INS_CTERMEQ = AARCH64_INS_CTERMEQ,
	ARM64_INS_CTERMNE = AARCH64_INS_CTERMNE,
	ARM64_INS_CTZ = AARCH64_INS_CTZ,
	ARM64_INS_DCPS1 = AARCH64_INS_DCPS1,
	ARM64_INS_DCPS2 = AARCH64_INS_DCPS2,
	ARM64_INS_DCPS3 = AARCH64_INS_DCPS3,
	ARM64_INS_DECB = AARCH64_INS_DECB,
	ARM64_INS_DECD = AARCH64_INS_DECD,
	ARM64_INS_DECH = AARCH64_INS_DECH,
	ARM64_INS_DECP = AARCH64_INS_DECP,
	ARM64_INS_DECW = AARCH64_INS_DECW,
	ARM64_INS_DMB = AARCH64_INS_DMB,
	ARM64_INS_DRPS = AARCH64_INS_DRPS,
	ARM64_INS_DSB = AARCH64_INS_DSB,
	ARM64_INS_DUPM = AARCH64_INS_DUPM,
	ARM64_INS_DUPQ = AARCH64_INS_DUPQ,
	ARM64_INS_DUP = AARCH64_INS_DUP,
	ARM64_INS_MOV = AARCH64_INS_MOV,
	ARM64_INS_EON = AARCH64_INS_EON,
	ARM64_INS_EOR3 = AARCH64_INS_EOR3,
	ARM64_INS_EORBT = AARCH64_INS_EORBT,
	ARM64_INS_EORQV = AARCH64_INS_EORQV,
	ARM64_INS_EORS = AARCH64_INS_EORS,
	ARM64_INS_EORTB = AARCH64_INS_EORTB,
	ARM64_INS_EORV = AARCH64_INS_EORV,
	ARM64_INS_EOR = AARCH64_INS_EOR,
	ARM64_INS_ERET = AARCH64_INS_ERET,
	ARM64_INS_ERETAA = AARCH64_INS_ERETAA,
	ARM64_INS_ERETAB = AARCH64_INS_ERETAB,
	ARM64_INS_EXTQ = AARCH64_INS_EXTQ,
	ARM64_INS_MOVA = AARCH64_INS_MOVA,
	ARM64_INS_EXTR = AARCH64_INS_EXTR,
	ARM64_INS_EXT = AARCH64_INS_EXT,
	ARM64_INS_F1CVTL2 = AARCH64_INS_F1CVTL2,
	ARM64_INS_F1CVTLT = AARCH64_INS_F1CVTLT,
	ARM64_INS_F1CVTL = AARCH64_INS_F1CVTL,
	ARM64_INS_F1CVT = AARCH64_INS_F1CVT,
	ARM64_INS_F2CVTL2 = AARCH64_INS_F2CVTL2,
	ARM64_INS_F2CVTLT = AARCH64_INS_F2CVTLT,
	ARM64_INS_F2CVTL = AARCH64_INS_F2CVTL,
	ARM64_INS_F2CVT = AARCH64_INS_F2CVT,
	ARM64_INS_FABD = AARCH64_INS_FABD,
	ARM64_INS_FABS = AARCH64_INS_FABS,
	ARM64_INS_FACGE = AARCH64_INS_FACGE,
	ARM64_INS_FACGT = AARCH64_INS_FACGT,
	ARM64_INS_FADDA = AARCH64_INS_FADDA,
	ARM64_INS_FADD = AARCH64_INS_FADD,
	ARM64_INS_FADDP = AARCH64_INS_FADDP,
	ARM64_INS_FADDQV = AARCH64_INS_FADDQV,
	ARM64_INS_FADDV = AARCH64_INS_FADDV,
	ARM64_INS_FAMAX = AARCH64_INS_FAMAX,
	ARM64_INS_FAMIN = AARCH64_INS_FAMIN,
	ARM64_INS_FCADD = AARCH64_INS_FCADD,
	ARM64_INS_FCCMP = AARCH64_INS_FCCMP,
	ARM64_INS_FCCMPE = AARCH64_INS_FCCMPE,
	ARM64_INS_FCLAMP = AARCH64_INS_FCLAMP,
	ARM64_INS_FCMEQ = AARCH64_INS_FCMEQ,
	ARM64_INS_FCMGE = AARCH64_INS_FCMGE,
	ARM64_INS_FCMGT = AARCH64_INS_FCMGT,
	ARM64_INS_FCMLA = AARCH64_INS_FCMLA,
	ARM64_INS_FCMLE = AARCH64_INS_FCMLE,
	ARM64_INS_FCMLT = AARCH64_INS_FCMLT,
	ARM64_INS_FCMNE = AARCH64_INS_FCMNE,
	ARM64_INS_FCMP = AARCH64_INS_FCMP,
	ARM64_INS_FCMPE = AARCH64_INS_FCMPE,
	ARM64_INS_FCMUO = AARCH64_INS_FCMUO,
	ARM64_INS_FCPY = AARCH64_INS_FCPY,
	ARM64_INS_FCSEL = AARCH64_INS_FCSEL,
	ARM64_INS_FCVTAS = AARCH64_INS_FCVTAS,
	ARM64_INS_FCVTAU = AARCH64_INS_FCVTAU,
	ARM64_INS_FCVT = AARCH64_INS_FCVT,
	ARM64_INS_FCVTLT = AARCH64_INS_FCVTLT,
	ARM64_INS_FCVTL = AARCH64_INS_FCVTL,
	ARM64_INS_FCVTL2 = AARCH64_INS_FCVTL2,
	ARM64_INS_FCVTMS = AARCH64_INS_FCVTMS,
	ARM64_INS_FCVTMU = AARCH64_INS_FCVTMU,
	ARM64_INS_FCVTNB = AARCH64_INS_FCVTNB,
	ARM64_INS_FCVTNS = AARCH64_INS_FCVTNS,
	ARM64_INS_FCVTNT = AARCH64_INS_FCVTNT,
	ARM64_INS_FCVTNU = AARCH64_INS_FCVTNU,
	ARM64_INS_FCVTN = AARCH64_INS_FCVTN,
	ARM64_INS_FCVTN2 = AARCH64_INS_FCVTN2,
	ARM64_INS_FCVTPS = AARCH64_INS_FCVTPS,
	ARM64_INS_FCVTPU = AARCH64_INS_FCVTPU,
	ARM64_INS_FCVTXNT = AARCH64_INS_FCVTXNT,
	ARM64_INS_FCVTXN = AARCH64_INS_FCVTXN,
	ARM64_INS_FCVTXN2 = AARCH64_INS_FCVTXN2,
	ARM64_INS_FCVTX = AARCH64_INS_FCVTX,
	ARM64_INS_FCVTZS = AARCH64_INS_FCVTZS,
	ARM64_INS_FCVTZU = AARCH64_INS_FCVTZU,
	ARM64_INS_FDIV = AARCH64_INS_FDIV,
	ARM64_INS_FDIVR = AARCH64_INS_FDIVR,
	ARM64_INS_FDOT = AARCH64_INS_FDOT,
	ARM64_INS_FDUP = AARCH64_INS_FDUP,
	ARM64_INS_FEXPA = AARCH64_INS_FEXPA,
	ARM64_INS_FJCVTZS = AARCH64_INS_FJCVTZS,
	ARM64_INS_FLOGB = AARCH64_INS_FLOGB,
	ARM64_INS_FMADD = AARCH64_INS_FMADD,
	ARM64_INS_FMAD = AARCH64_INS_FMAD,
	ARM64_INS_FMAX = AARCH64_INS_FMAX,
	ARM64_INS_FMAXNM = AARCH64_INS_FMAXNM,
	ARM64_INS_FMAXNMP = AARCH64_INS_FMAXNMP,
	ARM64_INS_FMAXNMQV = AARCH64_INS_FMAXNMQV,
	ARM64_INS_FMAXNMV = AARCH64_INS_FMAXNMV,
	ARM64_INS_FMAXP = AARCH64_INS_FMAXP,
	ARM64_INS_FMAXQV = AARCH64_INS_FMAXQV,
	ARM64_INS_FMAXV = AARCH64_INS_FMAXV,
	ARM64_INS_FMIN = AARCH64_INS_FMIN,
	ARM64_INS_FMINNM = AARCH64_INS_FMINNM,
	ARM64_INS_FMINNMP = AARCH64_INS_FMINNMP,
	ARM64_INS_FMINNMQV = AARCH64_INS_FMINNMQV,
	ARM64_INS_FMINNMV = AARCH64_INS_FMINNMV,
	ARM64_INS_FMINP = AARCH64_INS_FMINP,
	ARM64_INS_FMINQV = AARCH64_INS_FMINQV,
	ARM64_INS_FMINV = AARCH64_INS_FMINV,
	ARM64_INS_FMLAL2 = AARCH64_INS_FMLAL2,
	ARM64_INS_FMLALB = AARCH64_INS_FMLALB,
	ARM64_INS_FMLALLBB = AARCH64_INS_FMLALLBB,
	ARM64_INS_FMLALLBT = AARCH64_INS_FMLALLBT,
	ARM64_INS_FMLALLTB = AARCH64_INS_FMLALLTB,
	ARM64_INS_FMLALLTT = AARCH64_INS_FMLALLTT,
	ARM64_INS_FMLALL = AARCH64_INS_FMLALL,
	ARM64_INS_FMLALT = AARCH64_INS_FMLALT,
	ARM64_INS_FMLAL = AARCH64_INS_FMLAL,
	ARM64_INS_FMLA = AARCH64_INS_FMLA,
	ARM64_INS_FMLSL2 = AARCH64_INS_FMLSL2,
	ARM64_INS_FMLSLB = AARCH64_INS_FMLSLB,
	ARM64_INS_FMLSLT = AARCH64_INS_FMLSLT,
	ARM64_INS_FMLSL = AARCH64_INS_FMLSL,
	ARM64_INS_FMLS = AARCH64_INS_FMLS,
	ARM64_INS_FMMLA = AARCH64_INS_FMMLA,
	ARM64_INS_FMOPA = AARCH64_INS_FMOPA,
	ARM64_INS_FMOPS = AARCH64_INS_FMOPS,
	ARM64_INS_FMOV = AARCH64_INS_FMOV,
	ARM64_INS_FMSB = AARCH64_INS_FMSB,
	ARM64_INS_FMSUB = AARCH64_INS_FMSUB,
	ARM64_INS_FMUL = AARCH64_INS_FMUL,
	ARM64_INS_FMULX = AARCH64_INS_FMULX,
	ARM64_INS_FNEG = AARCH64_INS_FNEG,
	ARM64_INS_FNMADD = AARCH64_INS_FNMADD,
	ARM64_INS_FNMAD = AARCH64_INS_FNMAD,
	ARM64_INS_FNMLA = AARCH64_INS_FNMLA,
	ARM64_INS_FNMLS = AARCH64_INS_FNMLS,
	ARM64_INS_FNMSB = AARCH64_INS_FNMSB,
	ARM64_INS_FNMSUB = AARCH64_INS_FNMSUB,
	ARM64_INS_FNMUL = AARCH64_INS_FNMUL,
	ARM64_INS_FRECPE = AARCH64_INS_FRECPE,
	ARM64_INS_FRECPS = AARCH64_INS_FRECPS,
	ARM64_INS_FRECPX = AARCH64_INS_FRECPX,
	ARM64_INS_FRINT32X = AARCH64_INS_FRINT32X,
	ARM64_INS_FRINT32Z = AARCH64_INS_FRINT32Z,
	ARM64_INS_FRINT64X = AARCH64_INS_FRINT64X,
	ARM64_INS_FRINT64Z = AARCH64_INS_FRINT64Z,
	ARM64_INS_FRINTA = AARCH64_INS_FRINTA,
	ARM64_INS_FRINTI = AARCH64_INS_FRINTI,
	ARM64_INS_FRINTM = AARCH64_INS_FRINTM,
	ARM64_INS_FRINTN = AARCH64_INS_FRINTN,
	ARM64_INS_FRINTP = AARCH64_INS_FRINTP,
	ARM64_INS_FRINTX = AARCH64_INS_FRINTX,
	ARM64_INS_FRINTZ = AARCH64_INS_FRINTZ,
	ARM64_INS_FRSQRTE = AARCH64_INS_FRSQRTE,
	ARM64_INS_FRSQRTS = AARCH64_INS_FRSQRTS,
	ARM64_INS_FSCALE = AARCH64_INS_FSCALE,
	ARM64_INS_FSQRT = AARCH64_INS_FSQRT,
	ARM64_INS_FSUB = AARCH64_INS_FSUB,
	ARM64_INS_FSUBR = AARCH64_INS_FSUBR,
	ARM64_INS_FTMAD = AARCH64_INS_FTMAD,
	ARM64_INS_FTSMUL = AARCH64_INS_FTSMUL,
	ARM64_INS_FTSSEL = AARCH64_INS_FTSSEL,
	ARM64_INS_FVDOTB = AARCH64_INS_FVDOTB,
	ARM64_INS_FVDOTT = AARCH64_INS_FVDOTT,
	ARM64_INS_FVDOT = AARCH64_INS_FVDOT,
	ARM64_INS_GCSPOPCX = AARCH64_INS_GCSPOPCX,
	ARM64_INS_GCSPOPM = AARCH64_INS_GCSPOPM,
	ARM64_INS_GCSPOPX = AARCH64_INS_GCSPOPX,
	ARM64_INS_GCSPUSHM = AARCH64_INS_GCSPUSHM,
	ARM64_INS_GCSPUSHX = AARCH64_INS_GCSPUSHX,
	ARM64_INS_GCSSS1 = AARCH64_INS_GCSSS1,
	ARM64_INS_GCSSS2 = AARCH64_INS_GCSSS2,
	ARM64_INS_GCSSTR = AARCH64_INS_GCSSTR,
	ARM64_INS_GCSSTTR = AARCH64_INS_GCSSTTR,
	ARM64_INS_LD1B = AARCH64_INS_LD1B,
	ARM64_INS_LD1D = AARCH64_INS_LD1D,
	ARM64_INS_LD1H = AARCH64_INS_LD1H,
	ARM64_INS_LD1Q = AARCH64_INS_LD1Q,
	ARM64_INS_LD1SB = AARCH64_INS_LD1SB,
	ARM64_INS_LD1SH = AARCH64_INS_LD1SH,
	ARM64_INS_LD1SW = AARCH64_INS_LD1SW,
	ARM64_INS_LD1W = AARCH64_INS_LD1W,
	ARM64_INS_LDFF1B = AARCH64_INS_LDFF1B,
	ARM64_INS_LDFF1D = AARCH64_INS_LDFF1D,
	ARM64_INS_LDFF1H = AARCH64_INS_LDFF1H,
	ARM64_INS_LDFF1SB = AARCH64_INS_LDFF1SB,
	ARM64_INS_LDFF1SH = AARCH64_INS_LDFF1SH,
	ARM64_INS_LDFF1SW = AARCH64_INS_LDFF1SW,
	ARM64_INS_LDFF1W = AARCH64_INS_LDFF1W,
	ARM64_INS_GMI = AARCH64_INS_GMI,
	ARM64_INS_HISTCNT = AARCH64_INS_HISTCNT,
	ARM64_INS_HISTSEG = AARCH64_INS_HISTSEG,
	ARM64_INS_HLT = AARCH64_INS_HLT,
	ARM64_INS_HVC = AARCH64_INS_HVC,
	ARM64_INS_INCB = AARCH64_INS_INCB,
	ARM64_INS_INCD = AARCH64_INS_INCD,
	ARM64_INS_INCH = AARCH64_INS_INCH,
	ARM64_INS_INCP = AARCH64_INS_INCP,
	ARM64_INS_INCW = AARCH64_INS_INCW,
	ARM64_INS_INDEX = AARCH64_INS_INDEX,
	ARM64_INS_INSR = AARCH64_INS_INSR,
	ARM64_INS_INS = AARCH64_INS_INS,
	ARM64_INS_IRG = AARCH64_INS_IRG,
	ARM64_INS_ISB = AARCH64_INS_ISB,
	ARM64_INS_LASTA = AARCH64_INS_LASTA,
	ARM64_INS_LASTB = AARCH64_INS_LASTB,
	ARM64_INS_LD1 = AARCH64_INS_LD1,
	ARM64_INS_LD1RB = AARCH64_INS_LD1RB,
	ARM64_INS_LD1RD = AARCH64_INS_LD1RD,
	ARM64_INS_LD1RH = AARCH64_INS_LD1RH,
	ARM64_INS_LD1ROB = AARCH64_INS_LD1ROB,
	ARM64_INS_LD1ROD = AARCH64_INS_LD1ROD,
	ARM64_INS_LD1ROH = AARCH64_INS_LD1ROH,
	ARM64_INS_LD1ROW = AARCH64_INS_LD1ROW,
	ARM64_INS_LD1RQB = AARCH64_INS_LD1RQB,
	ARM64_INS_LD1RQD = AARCH64_INS_LD1RQD,
	ARM64_INS_LD1RQH = AARCH64_INS_LD1RQH,
	ARM64_INS_LD1RQW = AARCH64_INS_LD1RQW,
	ARM64_INS_LD1RSB = AARCH64_INS_LD1RSB,
	ARM64_INS_LD1RSH = AARCH64_INS_LD1RSH,
	ARM64_INS_LD1RSW = AARCH64_INS_LD1RSW,
	ARM64_INS_LD1RW = AARCH64_INS_LD1RW,
	ARM64_INS_LD1R = AARCH64_INS_LD1R,
	ARM64_INS_LD2B = AARCH64_INS_LD2B,
	ARM64_INS_LD2D = AARCH64_INS_LD2D,
	ARM64_INS_LD2H = AARCH64_INS_LD2H,
	ARM64_INS_LD2Q = AARCH64_INS_LD2Q,
	ARM64_INS_LD2R = AARCH64_INS_LD2R,
	ARM64_INS_LD2 = AARCH64_INS_LD2,
	ARM64_INS_LD2W = AARCH64_INS_LD2W,
	ARM64_INS_LD3B = AARCH64_INS_LD3B,
	ARM64_INS_LD3D = AARCH64_INS_LD3D,
	ARM64_INS_LD3H = AARCH64_INS_LD3H,
	ARM64_INS_LD3Q = AARCH64_INS_LD3Q,
	ARM64_INS_LD3R = AARCH64_INS_LD3R,
	ARM64_INS_LD3 = AARCH64_INS_LD3,
	ARM64_INS_LD3W = AARCH64_INS_LD3W,
	ARM64_INS_LD4B = AARCH64_INS_LD4B,
	ARM64_INS_LD4D = AARCH64_INS_LD4D,
	ARM64_INS_LD4 = AARCH64_INS_LD4,
	ARM64_INS_LD4H = AARCH64_INS_LD4H,
	ARM64_INS_LD4Q = AARCH64_INS_LD4Q,
	ARM64_INS_LD4R = AARCH64_INS_LD4R,
	ARM64_INS_LD4W = AARCH64_INS_LD4W,
	ARM64_INS_LD64B = AARCH64_INS_LD64B,
	ARM64_INS_LDADDAB = AARCH64_INS_LDADDAB,
	ARM64_INS_LDADDAH = AARCH64_INS_LDADDAH,
	ARM64_INS_LDADDALB = AARCH64_INS_LDADDALB,
	ARM64_INS_LDADDALH = AARCH64_INS_LDADDALH,
	ARM64_INS_LDADDAL = AARCH64_INS_LDADDAL,
	ARM64_INS_LDADDA = AARCH64_INS_LDADDA,
	ARM64_INS_LDADDB = AARCH64_INS_LDADDB,
	ARM64_INS_LDADDH = AARCH64_INS_LDADDH,
	ARM64_INS_LDADDLB = AARCH64_INS_LDADDLB,
	ARM64_INS_LDADDLH = AARCH64_INS_LDADDLH,
	ARM64_INS_LDADDL = AARCH64_INS_LDADDL,
	ARM64_INS_LDADD = AARCH64_INS_LDADD,
	ARM64_INS_LDAP1 = AARCH64_INS_LDAP1,
	ARM64_INS_LDAPRB = AARCH64_INS_LDAPRB,
	ARM64_INS_LDAPRH = AARCH64_INS_LDAPRH,
	ARM64_INS_LDAPR = AARCH64_INS_LDAPR,
	ARM64_INS_LDAPURB = AARCH64_INS_LDAPURB,
	ARM64_INS_LDAPURH = AARCH64_INS_LDAPURH,
	ARM64_INS_LDAPURSB = AARCH64_INS_LDAPURSB,
	ARM64_INS_LDAPURSH = AARCH64_INS_LDAPURSH,
	ARM64_INS_LDAPURSW = AARCH64_INS_LDAPURSW,
	ARM64_INS_LDAPUR = AARCH64_INS_LDAPUR,
	ARM64_INS_LDARB = AARCH64_INS_LDARB,
	ARM64_INS_LDARH = AARCH64_INS_LDARH,
	ARM64_INS_LDAR = AARCH64_INS_LDAR,
	ARM64_INS_LDAXP = AARCH64_INS_LDAXP,
	ARM64_INS_LDAXRB = AARCH64_INS_LDAXRB,
	ARM64_INS_LDAXRH = AARCH64_INS_LDAXRH,
	ARM64_INS_LDAXR = AARCH64_INS_LDAXR,
	ARM64_INS_LDCLRAB = AARCH64_INS_LDCLRAB,
	ARM64_INS_LDCLRAH = AARCH64_INS_LDCLRAH,
	ARM64_INS_LDCLRALB = AARCH64_INS_LDCLRALB,
	ARM64_INS_LDCLRALH = AARCH64_INS_LDCLRALH,
	ARM64_INS_LDCLRAL = AARCH64_INS_LDCLRAL,
	ARM64_INS_LDCLRA = AARCH64_INS_LDCLRA,
	ARM64_INS_LDCLRB = AARCH64_INS_LDCLRB,
	ARM64_INS_LDCLRH = AARCH64_INS_LDCLRH,
	ARM64_INS_LDCLRLB = AARCH64_INS_LDCLRLB,
	ARM64_INS_LDCLRLH = AARCH64_INS_LDCLRLH,
	ARM64_INS_LDCLRL = AARCH64_INS_LDCLRL,
	ARM64_INS_LDCLRP = AARCH64_INS_LDCLRP,
	ARM64_INS_LDCLRPA = AARCH64_INS_LDCLRPA,
	ARM64_INS_LDCLRPAL = AARCH64_INS_LDCLRPAL,
	ARM64_INS_LDCLRPL = AARCH64_INS_LDCLRPL,
	ARM64_INS_LDCLR = AARCH64_INS_LDCLR,
	ARM64_INS_LDEORAB = AARCH64_INS_LDEORAB,
	ARM64_INS_LDEORAH = AARCH64_INS_LDEORAH,
	ARM64_INS_LDEORALB = AARCH64_INS_LDEORALB,
	ARM64_INS_LDEORALH = AARCH64_INS_LDEORALH,
	ARM64_INS_LDEORAL = AARCH64_INS_LDEORAL,
	ARM64_INS_LDEORA = AARCH64_INS_LDEORA,
	ARM64_INS_LDEORB = AARCH64_INS_LDEORB,
	ARM64_INS_LDEORH = AARCH64_INS_LDEORH,
	ARM64_INS_LDEORLB = AARCH64_INS_LDEORLB,
	ARM64_INS_LDEORLH = AARCH64_INS_LDEORLH,
	ARM64_INS_LDEORL = AARCH64_INS_LDEORL,
	ARM64_INS_LDEOR = AARCH64_INS_LDEOR,
	ARM64_INS_LDG = AARCH64_INS_LDG,
	ARM64_INS_LDGM = AARCH64_INS_LDGM,
	ARM64_INS_LDIAPP = AARCH64_INS_LDIAPP,
	ARM64_INS_LDLARB = AARCH64_INS_LDLARB,
	ARM64_INS_LDLARH = AARCH64_INS_LDLARH,
	ARM64_INS_LDLAR = AARCH64_INS_LDLAR,
	ARM64_INS_LDNF1B = AARCH64_INS_LDNF1B,
	ARM64_INS_LDNF1D = AARCH64_INS_LDNF1D,
	ARM64_INS_LDNF1H = AARCH64_INS_LDNF1H,
	ARM64_INS_LDNF1SB = AARCH64_INS_LDNF1SB,
	ARM64_INS_LDNF1SH = AARCH64_INS_LDNF1SH,
	ARM64_INS_LDNF1SW = AARCH64_INS_LDNF1SW,
	ARM64_INS_LDNF1W = AARCH64_INS_LDNF1W,
	ARM64_INS_LDNP = AARCH64_INS_LDNP,
	ARM64_INS_LDNT1B = AARCH64_INS_LDNT1B,
	ARM64_INS_LDNT1D = AARCH64_INS_LDNT1D,
	ARM64_INS_LDNT1H = AARCH64_INS_LDNT1H,
	ARM64_INS_LDNT1SB = AARCH64_INS_LDNT1SB,
	ARM64_INS_LDNT1SH = AARCH64_INS_LDNT1SH,
	ARM64_INS_LDNT1SW = AARCH64_INS_LDNT1SW,
	ARM64_INS_LDNT1W = AARCH64_INS_LDNT1W,
	ARM64_INS_LDP = AARCH64_INS_LDP,
	ARM64_INS_LDPSW = AARCH64_INS_LDPSW,
	ARM64_INS_LDRAA = AARCH64_INS_LDRAA,
	ARM64_INS_LDRAB = AARCH64_INS_LDRAB,
	ARM64_INS_LDRB = AARCH64_INS_LDRB,
	ARM64_INS_LDR = AARCH64_INS_LDR,
	ARM64_INS_LDRH = AARCH64_INS_LDRH,
	ARM64_INS_LDRSB = AARCH64_INS_LDRSB,
	ARM64_INS_LDRSH = AARCH64_INS_LDRSH,
	ARM64_INS_LDRSW = AARCH64_INS_LDRSW,
	ARM64_INS_LDSETAB = AARCH64_INS_LDSETAB,
	ARM64_INS_LDSETAH = AARCH64_INS_LDSETAH,
	ARM64_INS_LDSETALB = AARCH64_INS_LDSETALB,
	ARM64_INS_LDSETALH = AARCH64_INS_LDSETALH,
	ARM64_INS_LDSETAL = AARCH64_INS_LDSETAL,
	ARM64_INS_LDSETA = AARCH64_INS_LDSETA,
	ARM64_INS_LDSETB = AARCH64_INS_LDSETB,
	ARM64_INS_LDSETH = AARCH64_INS_LDSETH,
	ARM64_INS_LDSETLB = AARCH64_INS_LDSETLB,
	ARM64_INS_LDSETLH = AARCH64_INS_LDSETLH,
	ARM64_INS_LDSETL = AARCH64_INS_LDSETL,
	ARM64_INS_LDSETP = AARCH64_INS_LDSETP,
	ARM64_INS_LDSETPA = AARCH64_INS_LDSETPA,
	ARM64_INS_LDSETPAL = AARCH64_INS_LDSETPAL,
	ARM64_INS_LDSETPL = AARCH64_INS_LDSETPL,
	ARM64_INS_LDSET = AARCH64_INS_LDSET,
	ARM64_INS_LDSMAXAB = AARCH64_INS_LDSMAXAB,
	ARM64_INS_LDSMAXAH = AARCH64_INS_LDSMAXAH,
	ARM64_INS_LDSMAXALB = AARCH64_INS_LDSMAXALB,
	ARM64_INS_LDSMAXALH = AARCH64_INS_LDSMAXALH,
	ARM64_INS_LDSMAXAL = AARCH64_INS_LDSMAXAL,
	ARM64_INS_LDSMAXA = AARCH64_INS_LDSMAXA,
	ARM64_INS_LDSMAXB = AARCH64_INS_LDSMAXB,
	ARM64_INS_LDSMAXH = AARCH64_INS_LDSMAXH,
	ARM64_INS_LDSMAXLB = AARCH64_INS_LDSMAXLB,
	ARM64_INS_LDSMAXLH = AARCH64_INS_LDSMAXLH,
	ARM64_INS_LDSMAXL = AARCH64_INS_LDSMAXL,
	ARM64_INS_LDSMAX = AARCH64_INS_LDSMAX,
	ARM64_INS_LDSMINAB = AARCH64_INS_LDSMINAB,
	ARM64_INS_LDSMINAH = AARCH64_INS_LDSMINAH,
	ARM64_INS_LDSMINALB = AARCH64_INS_LDSMINALB,
	ARM64_INS_LDSMINALH = AARCH64_INS_LDSMINALH,
	ARM64_INS_LDSMINAL = AARCH64_INS_LDSMINAL,
	ARM64_INS_LDSMINA = AARCH64_INS_LDSMINA,
	ARM64_INS_LDSMINB = AARCH64_INS_LDSMINB,
	ARM64_INS_LDSMINH = AARCH64_INS_LDSMINH,
	ARM64_INS_LDSMINLB = AARCH64_INS_LDSMINLB,
	ARM64_INS_LDSMINLH = AARCH64_INS_LDSMINLH,
	ARM64_INS_LDSMINL = AARCH64_INS_LDSMINL,
	ARM64_INS_LDSMIN = AARCH64_INS_LDSMIN,
	ARM64_INS_LDTRB = AARCH64_INS_LDTRB,
	ARM64_INS_LDTRH = AARCH64_INS_LDTRH,
	ARM64_INS_LDTRSB = AARCH64_INS_LDTRSB,
	ARM64_INS_LDTRSH = AARCH64_INS_LDTRSH,
	ARM64_INS_LDTRSW = AARCH64_INS_LDTRSW,
	ARM64_INS_LDTR = AARCH64_INS_LDTR,
	ARM64_INS_LDUMAXAB = AARCH64_INS_LDUMAXAB,
	ARM64_INS_LDUMAXAH = AARCH64_INS_LDUMAXAH,
	ARM64_INS_LDUMAXALB = AARCH64_INS_LDUMAXALB,
	ARM64_INS_LDUMAXALH = AARCH64_INS_LDUMAXALH,
	ARM64_INS_LDUMAXAL = AARCH64_INS_LDUMAXAL,
	ARM64_INS_LDUMAXA = AARCH64_INS_LDUMAXA,
	ARM64_INS_LDUMAXB = AARCH64_INS_LDUMAXB,
	ARM64_INS_LDUMAXH = AARCH64_INS_LDUMAXH,
	ARM64_INS_LDUMAXLB = AARCH64_INS_LDUMAXLB,
	ARM64_INS_LDUMAXLH = AARCH64_INS_LDUMAXLH,
	ARM64_INS_LDUMAXL = AARCH64_INS_LDUMAXL,
	ARM64_INS_LDUMAX = AARCH64_INS_LDUMAX,
	ARM64_INS_LDUMINAB = AARCH64_INS_LDUMINAB,
	ARM64_INS_LDUMINAH = AARCH64_INS_LDUMINAH,
	ARM64_INS_LDUMINALB = AARCH64_INS_LDUMINALB,
	ARM64_INS_LDUMINALH = AARCH64_INS_LDUMINALH,
	ARM64_INS_LDUMINAL = AARCH64_INS_LDUMINAL,
	ARM64_INS_LDUMINA = AARCH64_INS_LDUMINA,
	ARM64_INS_LDUMINB = AARCH64_INS_LDUMINB,
	ARM64_INS_LDUMINH = AARCH64_INS_LDUMINH,
	ARM64_INS_LDUMINLB = AARCH64_INS_LDUMINLB,
	ARM64_INS_LDUMINLH = AARCH64_INS_LDUMINLH,
	ARM64_INS_LDUMINL = AARCH64_INS_LDUMINL,
	ARM64_INS_LDUMIN = AARCH64_INS_LDUMIN,
	ARM64_INS_LDURB = AARCH64_INS_LDURB,
	ARM64_INS_LDUR = AARCH64_INS_LDUR,
	ARM64_INS_LDURH = AARCH64_INS_LDURH,
	ARM64_INS_LDURSB = AARCH64_INS_LDURSB,
	ARM64_INS_LDURSH = AARCH64_INS_LDURSH,
	ARM64_INS_LDURSW = AARCH64_INS_LDURSW,
	ARM64_INS_LDXP = AARCH64_INS_LDXP,
	ARM64_INS_LDXRB = AARCH64_INS_LDXRB,
	ARM64_INS_LDXRH = AARCH64_INS_LDXRH,
	ARM64_INS_LDXR = AARCH64_INS_LDXR,
	ARM64_INS_LSLR = AARCH64_INS_LSLR,
	ARM64_INS_LSL = AARCH64_INS_LSL,
	ARM64_INS_LSRR = AARCH64_INS_LSRR,
	ARM64_INS_LSR = AARCH64_INS_LSR,
	ARM64_INS_LUTI2 = AARCH64_INS_LUTI2,
	ARM64_INS_LUTI4 = AARCH64_INS_LUTI4,
	ARM64_INS_MADDPT = AARCH64_INS_MADDPT,
	ARM64_INS_MADD = AARCH64_INS_MADD,
	ARM64_INS_MADPT = AARCH64_INS_MADPT,
	ARM64_INS_MAD = AARCH64_INS_MAD,
	ARM64_INS_MATCH = AARCH64_INS_MATCH,
	ARM64_INS_MLAPT = AARCH64_INS_MLAPT,
	ARM64_INS_MLA = AARCH64_INS_MLA,
	ARM64_INS_MLS = AARCH64_INS_MLS,
	ARM64_INS_SETGE = AARCH64_INS_SETGE,
	ARM64_INS_SETGEN = AARCH64_INS_SETGEN,
	ARM64_INS_SETGET = AARCH64_INS_SETGET,
	ARM64_INS_SETGETN = AARCH64_INS_SETGETN,
	ARM64_INS_MOVAZ = AARCH64_INS_MOVAZ,
	ARM64_INS_MOVI = AARCH64_INS_MOVI,
	ARM64_INS_MOVK = AARCH64_INS_MOVK,
	ARM64_INS_MOVN = AARCH64_INS_MOVN,
	ARM64_INS_MOVPRFX = AARCH64_INS_MOVPRFX,
	ARM64_INS_MOVT = AARCH64_INS_MOVT,
	ARM64_INS_MOVZ = AARCH64_INS_MOVZ,
	ARM64_INS_MRRS = AARCH64_INS_MRRS,
	ARM64_INS_MRS = AARCH64_INS_MRS,
	ARM64_INS_MSB = AARCH64_INS_MSB,
	ARM64_INS_MSR = AARCH64_INS_MSR,
	ARM64_INS_MSRR = AARCH64_INS_MSRR,
	ARM64_INS_MSUBPT = AARCH64_INS_MSUBPT,
	ARM64_INS_MSUB = AARCH64_INS_MSUB,
	ARM64_INS_MUL = AARCH64_INS_MUL,
	ARM64_INS_MVNI = AARCH64_INS_MVNI,
	ARM64_INS_NANDS = AARCH64_INS_NANDS,
	ARM64_INS_NAND = AARCH64_INS_NAND,
	ARM64_INS_NBSL = AARCH64_INS_NBSL,
	ARM64_INS_NEG = AARCH64_INS_NEG,
	ARM64_INS_NMATCH = AARCH64_INS_NMATCH,
	ARM64_INS_NORS = AARCH64_INS_NORS,
	ARM64_INS_NOR = AARCH64_INS_NOR,
	ARM64_INS_NOT = AARCH64_INS_NOT,
	ARM64_INS_ORNS = AARCH64_INS_ORNS,
	ARM64_INS_ORN = AARCH64_INS_ORN,
	ARM64_INS_ORQV = AARCH64_INS_ORQV,
	ARM64_INS_ORRS = AARCH64_INS_ORRS,
	ARM64_INS_ORR = AARCH64_INS_ORR,
	ARM64_INS_ORV = AARCH64_INS_ORV,
	ARM64_INS_PACDA = AARCH64_INS_PACDA,
	ARM64_INS_PACDB = AARCH64_INS_PACDB,
	ARM64_INS_PACDZA = AARCH64_INS_PACDZA,
	ARM64_INS_PACDZB = AARCH64_INS_PACDZB,
	ARM64_INS_PACGA = AARCH64_INS_PACGA,
	ARM64_INS_PACIA = AARCH64_INS_PACIA,
	ARM64_INS_PACIA171615 = AARCH64_INS_PACIA171615,
	ARM64_INS_PACIASPPC = AARCH64_INS_PACIASPPC,
	ARM64_INS_PACIB = AARCH64_INS_PACIB,
	ARM64_INS_PACIB171615 = AARCH64_INS_PACIB171615,
	ARM64_INS_PACIBSPPC = AARCH64_INS_PACIBSPPC,
	ARM64_INS_PACIZA = AARCH64_INS_PACIZA,
	ARM64_INS_PACIZB = AARCH64_INS_PACIZB,
	ARM64_INS_PACNBIASPPC = AARCH64_INS_PACNBIASPPC,
	ARM64_INS_PACNBIBSPPC = AARCH64_INS_PACNBIBSPPC,
	ARM64_INS_PEXT = AARCH64_INS_PEXT,
	ARM64_INS_PFALSE = AARCH64_INS_PFALSE,
	ARM64_INS_PFIRST = AARCH64_INS_PFIRST,
	ARM64_INS_PMOV = AARCH64_INS_PMOV,
	ARM64_INS_PMULLB = AARCH64_INS_PMULLB,
	ARM64_INS_PMULLT = AARCH64_INS_PMULLT,
	ARM64_INS_PMULL2 = AARCH64_INS_PMULL2,
	ARM64_INS_PMULL = AARCH64_INS_PMULL,
	ARM64_INS_PMUL = AARCH64_INS_PMUL,
	ARM64_INS_PNEXT = AARCH64_INS_PNEXT,
	ARM64_INS_PRFB = AARCH64_INS_PRFB,
	ARM64_INS_PRFD = AARCH64_INS_PRFD,
	ARM64_INS_PRFH = AARCH64_INS_PRFH,
	ARM64_INS_PRFM = AARCH64_INS_PRFM,
	ARM64_INS_PRFUM = AARCH64_INS_PRFUM,
	ARM64_INS_PRFW = AARCH64_INS_PRFW,
	ARM64_INS_PSEL = AARCH64_INS_PSEL,
	ARM64_INS_PTEST = AARCH64_INS_PTEST,
	ARM64_INS_PTRUES = AARCH64_INS_PTRUES,
	ARM64_INS_PTRUE = AARCH64_INS_PTRUE,
	ARM64_INS_PUNPKHI = AARCH64_INS_PUNPKHI,
	ARM64_INS_PUNPKLO = AARCH64_INS_PUNPKLO,
	ARM64_INS_RADDHNB = AARCH64_INS_RADDHNB,
	ARM64_INS_RADDHNT = AARCH64_INS_RADDHNT,
	ARM64_INS_RADDHN = AARCH64_INS_RADDHN,
	ARM64_INS_RADDHN2 = AARCH64_INS_RADDHN2,
	ARM64_INS_RAX1 = AARCH64_INS_RAX1,
	ARM64_INS_RBIT = AARCH64_INS_RBIT,
	ARM64_INS_RCWCAS = AARCH64_INS_RCWCAS,
	ARM64_INS_RCWCASA = AARCH64_INS_RCWCASA,
	ARM64_INS_RCWCASAL = AARCH64_INS_RCWCASAL,
	ARM64_INS_RCWCASL = AARCH64_INS_RCWCASL,
	ARM64_INS_RCWCASP = AARCH64_INS_RCWCASP,
	ARM64_INS_RCWCASPA = AARCH64_INS_RCWCASPA,
	ARM64_INS_RCWCASPAL = AARCH64_INS_RCWCASPAL,
	ARM64_INS_RCWCASPL = AARCH64_INS_RCWCASPL,
	ARM64_INS_RCWCLR = AARCH64_INS_RCWCLR,
	ARM64_INS_RCWCLRA = AARCH64_INS_RCWCLRA,
	ARM64_INS_RCWCLRAL = AARCH64_INS_RCWCLRAL,
	ARM64_INS_RCWCLRL = AARCH64_INS_RCWCLRL,
	ARM64_INS_RCWCLRP = AARCH64_INS_RCWCLRP,
	ARM64_INS_RCWCLRPA = AARCH64_INS_RCWCLRPA,
	ARM64_INS_RCWCLRPAL = AARCH64_INS_RCWCLRPAL,
	ARM64_INS_RCWCLRPL = AARCH64_INS_RCWCLRPL,
	ARM64_INS_RCWSCLR = AARCH64_INS_RCWSCLR,
	ARM64_INS_RCWSCLRA = AARCH64_INS_RCWSCLRA,
	ARM64_INS_RCWSCLRAL = AARCH64_INS_RCWSCLRAL,
	ARM64_INS_RCWSCLRL = AARCH64_INS_RCWSCLRL,
	ARM64_INS_RCWSCLRP = AARCH64_INS_RCWSCLRP,
	ARM64_INS_RCWSCLRPA = AARCH64_INS_RCWSCLRPA,
	ARM64_INS_RCWSCLRPAL = AARCH64_INS_RCWSCLRPAL,
	ARM64_INS_RCWSCLRPL = AARCH64_INS_RCWSCLRPL,
	ARM64_INS_RCWSCAS = AARCH64_INS_RCWSCAS,
	ARM64_INS_RCWSCASA = AARCH64_INS_RCWSCASA,
	ARM64_INS_RCWSCASAL = AARCH64_INS_RCWSCASAL,
	ARM64_INS_RCWSCASL = AARCH64_INS_RCWSCASL,
	ARM64_INS_RCWSCASP = AARCH64_INS_RCWSCASP,
	ARM64_INS_RCWSCASPA = AARCH64_INS_RCWSCASPA,
	ARM64_INS_RCWSCASPAL = AARCH64_INS_RCWSCASPAL,
	ARM64_INS_RCWSCASPL = AARCH64_INS_RCWSCASPL,
	ARM64_INS_RCWSET = AARCH64_INS_RCWSET,
	ARM64_INS_RCWSETA = AARCH64_INS_RCWSETA,
	ARM64_INS_RCWSETAL = AARCH64_INS_RCWSETAL,
	ARM64_INS_RCWSETL = AARCH64_INS_RCWSETL,
	ARM64_INS_RCWSETP = AARCH64_INS_RCWSETP,
	ARM64_INS_RCWSETPA = AARCH64_INS_RCWSETPA,
	ARM64_INS_RCWSETPAL = AARCH64_INS_RCWSETPAL,
	ARM64_INS_RCWSETPL = AARCH64_INS_RCWSETPL,
	ARM64_INS_RCWSSET = AARCH64_INS_RCWSSET,
	ARM64_INS_RCWSSETA = AARCH64_INS_RCWSSETA,
	ARM64_INS_RCWSSETAL = AARCH64_INS_RCWSSETAL,
	ARM64_INS_RCWSSETL = AARCH64_INS_RCWSSETL,
	ARM64_INS_RCWSSETP = AARCH64_INS_RCWSSETP,
	ARM64_INS_RCWSSETPA = AARCH64_INS_RCWSSETPA,
	ARM64_INS_RCWSSETPAL = AARCH64_INS_RCWSSETPAL,
	ARM64_INS_RCWSSETPL = AARCH64_INS_RCWSSETPL,
	ARM64_INS_RCWSWP = AARCH64_INS_RCWSWP,
	ARM64_INS_RCWSWPA = AARCH64_INS_RCWSWPA,
	ARM64_INS_RCWSWPAL = AARCH64_INS_RCWSWPAL,
	ARM64_INS_RCWSWPL = AARCH64_INS_RCWSWPL,
	ARM64_INS_RCWSWPP = AARCH64_INS_RCWSWPP,
	ARM64_INS_RCWSWPPA = AARCH64_INS_RCWSWPPA,
	ARM64_INS_RCWSWPPAL = AARCH64_INS_RCWSWPPAL,
	ARM64_INS_RCWSWPPL = AARCH64_INS_RCWSWPPL,
	ARM64_INS_RCWSSWP = AARCH64_INS_RCWSSWP,
	ARM64_INS_RCWSSWPA = AARCH64_INS_RCWSSWPA,
	ARM64_INS_RCWSSWPAL = AARCH64_INS_RCWSSWPAL,
	ARM64_INS_RCWSSWPL = AARCH64_INS_RCWSSWPL,
	ARM64_INS_RCWSSWPP = AARCH64_INS_RCWSSWPP,
	ARM64_INS_RCWSSWPPA = AARCH64_INS_RCWSSWPPA,
	ARM64_INS_RCWSSWPPAL = AARCH64_INS_RCWSSWPPAL,
	ARM64_INS_RCWSSWPPL = AARCH64_INS_RCWSSWPPL,
	ARM64_INS_RDFFRS = AARCH64_INS_RDFFRS,
	ARM64_INS_RDFFR = AARCH64_INS_RDFFR,
	ARM64_INS_RDSVL = AARCH64_INS_RDSVL,
	ARM64_INS_RDVL = AARCH64_INS_RDVL,
	ARM64_INS_RET = AARCH64_INS_RET,
	ARM64_INS_RETAA = AARCH64_INS_RETAA,
	ARM64_INS_RETAASPPC = AARCH64_INS_RETAASPPC,
	ARM64_INS_RETAB = AARCH64_INS_RETAB,
	ARM64_INS_RETABSPPC = AARCH64_INS_RETABSPPC,
	ARM64_INS_REV16 = AARCH64_INS_REV16,
	ARM64_INS_REV32 = AARCH64_INS_REV32,
	ARM64_INS_REV64 = AARCH64_INS_REV64,
	ARM64_INS_REVB = AARCH64_INS_REVB,
	ARM64_INS_REVD = AARCH64_INS_REVD,
	ARM64_INS_REVH = AARCH64_INS_REVH,
	ARM64_INS_REVW = AARCH64_INS_REVW,
	ARM64_INS_REV = AARCH64_INS_REV,
	ARM64_INS_RMIF = AARCH64_INS_RMIF,
	ARM64_INS_ROR = AARCH64_INS_ROR,
	ARM64_INS_RPRFM = AARCH64_INS_RPRFM,
	ARM64_INS_RSHRNB = AARCH64_INS_RSHRNB,
	ARM64_INS_RSHRNT = AARCH64_INS_RSHRNT,
	ARM64_INS_RSHRN2 = AARCH64_INS_RSHRN2,
	ARM64_INS_RSHRN = AARCH64_INS_RSHRN,
	ARM64_INS_RSUBHNB = AARCH64_INS_RSUBHNB,
	ARM64_INS_RSUBHNT = AARCH64_INS_RSUBHNT,
	ARM64_INS_RSUBHN = AARCH64_INS_RSUBHN,
	ARM64_INS_RSUBHN2 = AARCH64_INS_RSUBHN2,
	ARM64_INS_SABALB = AARCH64_INS_SABALB,
	ARM64_INS_SABALT = AARCH64_INS_SABALT,
	ARM64_INS_SABAL2 = AARCH64_INS_SABAL2,
	ARM64_INS_SABAL = AARCH64_INS_SABAL,
	ARM64_INS_SABA = AARCH64_INS_SABA,
	ARM64_INS_SABDLB = AARCH64_INS_SABDLB,
	ARM64_INS_SABDLT = AARCH64_INS_SABDLT,
	ARM64_INS_SABDL2 = AARCH64_INS_SABDL2,
	ARM64_INS_SABDL = AARCH64_INS_SABDL,
	ARM64_INS_SABD = AARCH64_INS_SABD,
	ARM64_INS_SADALP = AARCH64_INS_SADALP,
	ARM64_INS_SADDLBT = AARCH64_INS_SADDLBT,
	ARM64_INS_SADDLB = AARCH64_INS_SADDLB,
	ARM64_INS_SADDLP = AARCH64_INS_SADDLP,
	ARM64_INS_SADDLT = AARCH64_INS_SADDLT,
	ARM64_INS_SADDLV = AARCH64_INS_SADDLV,
	ARM64_INS_SADDL2 = AARCH64_INS_SADDL2,
	ARM64_INS_SADDL = AARCH64_INS_SADDL,
	ARM64_INS_SADDV = AARCH64_INS_SADDV,
	ARM64_INS_SADDWB = AARCH64_INS_SADDWB,
	ARM64_INS_SADDWT = AARCH64_INS_SADDWT,
	ARM64_INS_SADDW2 = AARCH64_INS_SADDW2,
	ARM64_INS_SADDW = AARCH64_INS_SADDW,
	ARM64_INS_SB = AARCH64_INS_SB,
	ARM64_INS_SBCLB = AARCH64_INS_SBCLB,
	ARM64_INS_SBCLT = AARCH64_INS_SBCLT,
	ARM64_INS_SBCS = AARCH64_INS_SBCS,
	ARM64_INS_SBC = AARCH64_INS_SBC,
	ARM64_INS_SBFM = AARCH64_INS_SBFM,
	ARM64_INS_SCLAMP = AARCH64_INS_SCLAMP,
	ARM64_INS_SCVTF = AARCH64_INS_SCVTF,
	ARM64_INS_SDIVR = AARCH64_INS_SDIVR,
	ARM64_INS_SDIV = AARCH64_INS_SDIV,
	ARM64_INS_SDOT = AARCH64_INS_SDOT,
	ARM64_INS_SEL = AARCH64_INS_SEL,
	ARM64_INS_SETE = AARCH64_INS_SETE,
	ARM64_INS_SETEN = AARCH64_INS_SETEN,
	ARM64_INS_SETET = AARCH64_INS_SETET,
	ARM64_INS_SETETN = AARCH64_INS_SETETN,
	ARM64_INS_SETF16 = AARCH64_INS_SETF16,
	ARM64_INS_SETF8 = AARCH64_INS_SETF8,
	ARM64_INS_SETFFR = AARCH64_INS_SETFFR,
	ARM64_INS_SETGM = AARCH64_INS_SETGM,
	ARM64_INS_SETGMN = AARCH64_INS_SETGMN,
	ARM64_INS_SETGMT = AARCH64_INS_SETGMT,
	ARM64_INS_SETGMTN = AARCH64_INS_SETGMTN,
	ARM64_INS_SETGP = AARCH64_INS_SETGP,
	ARM64_INS_SETGPN = AARCH64_INS_SETGPN,
	ARM64_INS_SETGPT = AARCH64_INS_SETGPT,
	ARM64_INS_SETGPTN = AARCH64_INS_SETGPTN,
	ARM64_INS_SETM = AARCH64_INS_SETM,
	ARM64_INS_SETMN = AARCH64_INS_SETMN,
	ARM64_INS_SETMT = AARCH64_INS_SETMT,
	ARM64_INS_SETMTN = AARCH64_INS_SETMTN,
	ARM64_INS_SETP = AARCH64_INS_SETP,
	ARM64_INS_SETPN = AARCH64_INS_SETPN,
	ARM64_INS_SETPT = AARCH64_INS_SETPT,
	ARM64_INS_SETPTN = AARCH64_INS_SETPTN,
	ARM64_INS_SHA1C = AARCH64_INS_SHA1C,
	ARM64_INS_SHA1H = AARCH64_INS_SHA1H,
	ARM64_INS_SHA1M = AARCH64_INS_SHA1M,
	ARM64_INS_SHA1P = AARCH64_INS_SHA1P,
	ARM64_INS_SHA1SU0 = AARCH64_INS_SHA1SU0,
	ARM64_INS_SHA1SU1 = AARCH64_INS_SHA1SU1,
	ARM64_INS_SHA256H2 = AARCH64_INS_SHA256H2,
	ARM64_INS_SHA256H = AARCH64_INS_SHA256H,
	ARM64_INS_SHA256SU0 = AARCH64_INS_SHA256SU0,
	ARM64_INS_SHA256SU1 = AARCH64_INS_SHA256SU1,
	ARM64_INS_SHA512H = AARCH64_INS_SHA512H,
	ARM64_INS_SHA512H2 = AARCH64_INS_SHA512H2,
	ARM64_INS_SHA512SU0 = AARCH64_INS_SHA512SU0,
	ARM64_INS_SHA512SU1 = AARCH64_INS_SHA512SU1,
	ARM64_INS_SHADD = AARCH64_INS_SHADD,
	ARM64_INS_SHLL2 = AARCH64_INS_SHLL2,
	ARM64_INS_SHLL = AARCH64_INS_SHLL,
	ARM64_INS_SHL = AARCH64_INS_SHL,
	ARM64_INS_SHRNB = AARCH64_INS_SHRNB,
	ARM64_INS_SHRNT = AARCH64_INS_SHRNT,
	ARM64_INS_SHRN2 = AARCH64_INS_SHRN2,
	ARM64_INS_SHRN = AARCH64_INS_SHRN,
	ARM64_INS_SHSUBR = AARCH64_INS_SHSUBR,
	ARM64_INS_SHSUB = AARCH64_INS_SHSUB,
	ARM64_INS_SLI = AARCH64_INS_SLI,
	ARM64_INS_SM3PARTW1 = AARCH64_INS_SM3PARTW1,
	ARM64_INS_SM3PARTW2 = AARCH64_INS_SM3PARTW2,
	ARM64_INS_SM3SS1 = AARCH64_INS_SM3SS1,
	ARM64_INS_SM3TT1A = AARCH64_INS_SM3TT1A,
	ARM64_INS_SM3TT1B = AARCH64_INS_SM3TT1B,
	ARM64_INS_SM3TT2A = AARCH64_INS_SM3TT2A,
	ARM64_INS_SM3TT2B = AARCH64_INS_SM3TT2B,
	ARM64_INS_SM4E = AARCH64_INS_SM4E,
	ARM64_INS_SM4EKEY = AARCH64_INS_SM4EKEY,
	ARM64_INS_SMADDL = AARCH64_INS_SMADDL,
	ARM64_INS_SMAXP = AARCH64_INS_SMAXP,
	ARM64_INS_SMAXQV = AARCH64_INS_SMAXQV,
	ARM64_INS_SMAXV = AARCH64_INS_SMAXV,
	ARM64_INS_SMAX = AARCH64_INS_SMAX,
	ARM64_INS_SMC = AARCH64_INS_SMC,
	ARM64_INS_SMINP = AARCH64_INS_SMINP,
	ARM64_INS_SMINQV = AARCH64_INS_SMINQV,
	ARM64_INS_SMINV = AARCH64_INS_SMINV,
	ARM64_INS_SMIN = AARCH64_INS_SMIN,
	ARM64_INS_SMLALB = AARCH64_INS_SMLALB,
	ARM64_INS_SMLALL = AARCH64_INS_SMLALL,
	ARM64_INS_SMLALT = AARCH64_INS_SMLALT,
	ARM64_INS_SMLAL = AARCH64_INS_SMLAL,
	ARM64_INS_SMLAL2 = AARCH64_INS_SMLAL2,
	ARM64_INS_SMLSLB = AARCH64_INS_SMLSLB,
	ARM64_INS_SMLSLL = AARCH64_INS_SMLSLL,
	ARM64_INS_SMLSLT = AARCH64_INS_SMLSLT,
	ARM64_INS_SMLSL = AARCH64_INS_SMLSL,
	ARM64_INS_SMLSL2 = AARCH64_INS_SMLSL2,
	ARM64_INS_SMMLA = AARCH64_INS_SMMLA,
	ARM64_INS_SMOPA = AARCH64_INS_SMOPA,
	ARM64_INS_SMOPS = AARCH64_INS_SMOPS,
	ARM64_INS_SMOV = AARCH64_INS_SMOV,
	ARM64_INS_SMSUBL = AARCH64_INS_SMSUBL,
	ARM64_INS_SMULH = AARCH64_INS_SMULH,
	ARM64_INS_SMULLB = AARCH64_INS_SMULLB,
	ARM64_INS_SMULLT = AARCH64_INS_SMULLT,
	ARM64_INS_SMULL2 = AARCH64_INS_SMULL2,
	ARM64_INS_SMULL = AARCH64_INS_SMULL,
	ARM64_INS_SPLICE = AARCH64_INS_SPLICE,
	ARM64_INS_SQABS = AARCH64_INS_SQABS,
	ARM64_INS_SQADD = AARCH64_INS_SQADD,
	ARM64_INS_SQCADD = AARCH64_INS_SQCADD,
	ARM64_INS_SQCVTN = AARCH64_INS_SQCVTN,
	ARM64_INS_SQCVTUN = AARCH64_INS_SQCVTUN,
	ARM64_INS_SQCVTU = AARCH64_INS_SQCVTU,
	ARM64_INS_SQCVT = AARCH64_INS_SQCVT,
	ARM64_INS_SQDECB = AARCH64_INS_SQDECB,
	ARM64_INS_SQDECD = AARCH64_INS_SQDECD,
	ARM64_INS_SQDECH = AARCH64_INS_SQDECH,
	ARM64_INS_SQDECP = AARCH64_INS_SQDECP,
	ARM64_INS_SQDECW = AARCH64_INS_SQDECW,
	ARM64_INS_SQDMLALBT = AARCH64_INS_SQDMLALBT,
	ARM64_INS_SQDMLALB = AARCH64_INS_SQDMLALB,
	ARM64_INS_SQDMLALT = AARCH64_INS_SQDMLALT,
	ARM64_INS_SQDMLAL = AARCH64_INS_SQDMLAL,
	ARM64_INS_SQDMLAL2 = AARCH64_INS_SQDMLAL2,
	ARM64_INS_SQDMLSLBT = AARCH64_INS_SQDMLSLBT,
	ARM64_INS_SQDMLSLB = AARCH64_INS_SQDMLSLB,
	ARM64_INS_SQDMLSLT = AARCH64_INS_SQDMLSLT,
	ARM64_INS_SQDMLSL = AARCH64_INS_SQDMLSL,
	ARM64_INS_SQDMLSL2 = AARCH64_INS_SQDMLSL2,
	ARM64_INS_SQDMULH = AARCH64_INS_SQDMULH,
	ARM64_INS_SQDMULLB = AARCH64_INS_SQDMULLB,
	ARM64_INS_SQDMULLT = AARCH64_INS_SQDMULLT,
	ARM64_INS_SQDMULL = AARCH64_INS_SQDMULL,
	ARM64_INS_SQDMULL2 = AARCH64_INS_SQDMULL2,
	ARM64_INS_SQINCB = AARCH64_INS_SQINCB,
	ARM64_INS_SQINCD = AARCH64_INS_SQINCD,
	ARM64_INS_SQINCH = AARCH64_INS_SQINCH,
	ARM64_INS_SQINCP = AARCH64_INS_SQINCP,
	ARM64_INS_SQINCW = AARCH64_INS_SQINCW,
	ARM64_INS_SQNEG = AARCH64_INS_SQNEG,
	ARM64_INS_SQRDCMLAH = AARCH64_INS_SQRDCMLAH,
	ARM64_INS_SQRDMLAH = AARCH64_INS_SQRDMLAH,
	ARM64_INS_SQRDMLSH = AARCH64_INS_SQRDMLSH,
	ARM64_INS_SQRDMULH = AARCH64_INS_SQRDMULH,
	ARM64_INS_SQRSHLR = AARCH64_INS_SQRSHLR,
	ARM64_INS_SQRSHL = AARCH64_INS_SQRSHL,
	ARM64_INS_SQRSHRNB = AARCH64_INS_SQRSHRNB,
	ARM64_INS_SQRSHRNT = AARCH64_INS_SQRSHRNT,
	ARM64_INS_SQRSHRN = AARCH64_INS_SQRSHRN,
	ARM64_INS_SQRSHRN2 = AARCH64_INS_SQRSHRN2,
	ARM64_INS_SQRSHRUNB = AARCH64_INS_SQRSHRUNB,
	ARM64_INS_SQRSHRUNT = AARCH64_INS_SQRSHRUNT,
	ARM64_INS_SQRSHRUN = AARCH64_INS_SQRSHRUN,
	ARM64_INS_SQRSHRUN2 = AARCH64_INS_SQRSHRUN2,
	ARM64_INS_SQRSHRU = AARCH64_INS_SQRSHRU,
	ARM64_INS_SQRSHR = AARCH64_INS_SQRSHR,
	ARM64_INS_SQSHLR = AARCH64_INS_SQSHLR,
	ARM64_INS_SQSHLU = AARCH64_INS_SQSHLU,
	ARM64_INS_SQSHL = AARCH64_INS_SQSHL,
	ARM64_INS_SQSHRNB = AARCH64_INS_SQSHRNB,
	ARM64_INS_SQSHRNT = AARCH64_INS_SQSHRNT,
	ARM64_INS_SQSHRN = AARCH64_INS_SQSHRN,
	ARM64_INS_SQSHRN2 = AARCH64_INS_SQSHRN2,
	ARM64_INS_SQSHRUNB = AARCH64_INS_SQSHRUNB,
	ARM64_INS_SQSHRUNT = AARCH64_INS_SQSHRUNT,
	ARM64_INS_SQSHRUN = AARCH64_INS_SQSHRUN,
	ARM64_INS_SQSHRUN2 = AARCH64_INS_SQSHRUN2,
	ARM64_INS_SQSUBR = AARCH64_INS_SQSUBR,
	ARM64_INS_SQSUB = AARCH64_INS_SQSUB,
	ARM64_INS_SQXTNB = AARCH64_INS_SQXTNB,
	ARM64_INS_SQXTNT = AARCH64_INS_SQXTNT,
	ARM64_INS_SQXTN2 = AARCH64_INS_SQXTN2,
	ARM64_INS_SQXTN = AARCH64_INS_SQXTN,
	ARM64_INS_SQXTUNB = AARCH64_INS_SQXTUNB,
	ARM64_INS_SQXTUNT = AARCH64_INS_SQXTUNT,
	ARM64_INS_SQXTUN2 = AARCH64_INS_SQXTUN2,
	ARM64_INS_SQXTUN = AARCH64_INS_SQXTUN,
	ARM64_INS_SRHADD = AARCH64_INS_SRHADD,
	ARM64_INS_SRI = AARCH64_INS_SRI,
	ARM64_INS_SRSHLR = AARCH64_INS_SRSHLR,
	ARM64_INS_SRSHL = AARCH64_INS_SRSHL,
	ARM64_INS_SRSHR = AARCH64_INS_SRSHR,
	ARM64_INS_SRSRA = AARCH64_INS_SRSRA,
	ARM64_INS_SSHLLB = AARCH64_INS_SSHLLB,
	ARM64_INS_SSHLLT = AARCH64_INS_SSHLLT,
	ARM64_INS_SSHLL2 = AARCH64_INS_SSHLL2,
	ARM64_INS_SSHLL = AARCH64_INS_SSHLL,
	ARM64_INS_SSHL = AARCH64_INS_SSHL,
	ARM64_INS_SSHR = AARCH64_INS_SSHR,
	ARM64_INS_SSRA = AARCH64_INS_SSRA,
	ARM64_INS_ST1B = AARCH64_INS_ST1B,
	ARM64_INS_ST1D = AARCH64_INS_ST1D,
	ARM64_INS_ST1H = AARCH64_INS_ST1H,
	ARM64_INS_ST1Q = AARCH64_INS_ST1Q,
	ARM64_INS_ST1W = AARCH64_INS_ST1W,
	ARM64_INS_SSUBLBT = AARCH64_INS_SSUBLBT,
	ARM64_INS_SSUBLB = AARCH64_INS_SSUBLB,
	ARM64_INS_SSUBLTB = AARCH64_INS_SSUBLTB,
	ARM64_INS_SSUBLT = AARCH64_INS_SSUBLT,
	ARM64_INS_SSUBL2 = AARCH64_INS_SSUBL2,
	ARM64_INS_SSUBL = AARCH64_INS_SSUBL,
	ARM64_INS_SSUBWB = AARCH64_INS_SSUBWB,
	ARM64_INS_SSUBWT = AARCH64_INS_SSUBWT,
	ARM64_INS_SSUBW2 = AARCH64_INS_SSUBW2,
	ARM64_INS_SSUBW = AARCH64_INS_SSUBW,
	ARM64_INS_ST1 = AARCH64_INS_ST1,
	ARM64_INS_ST2B = AARCH64_INS_ST2B,
	ARM64_INS_ST2D = AARCH64_INS_ST2D,
	ARM64_INS_ST2G = AARCH64_INS_ST2G,
	ARM64_INS_ST2H = AARCH64_INS_ST2H,
	ARM64_INS_ST2Q = AARCH64_INS_ST2Q,
	ARM64_INS_ST2 = AARCH64_INS_ST2,
	ARM64_INS_ST2W = AARCH64_INS_ST2W,
	ARM64_INS_ST3B = AARCH64_INS_ST3B,
	ARM64_INS_ST3D = AARCH64_INS_ST3D,
	ARM64_INS_ST3H = AARCH64_INS_ST3H,
	ARM64_INS_ST3Q = AARCH64_INS_ST3Q,
	ARM64_INS_ST3 = AARCH64_INS_ST3,
	ARM64_INS_ST3W = AARCH64_INS_ST3W,
	ARM64_INS_ST4B = AARCH64_INS_ST4B,
	ARM64_INS_ST4D = AARCH64_INS_ST4D,
	ARM64_INS_ST4 = AARCH64_INS_ST4,
	ARM64_INS_ST4H = AARCH64_INS_ST4H,
	ARM64_INS_ST4Q = AARCH64_INS_ST4Q,
	ARM64_INS_ST4W = AARCH64_INS_ST4W,
	ARM64_INS_ST64B = AARCH64_INS_ST64B,
	ARM64_INS_ST64BV = AARCH64_INS_ST64BV,
	ARM64_INS_ST64BV0 = AARCH64_INS_ST64BV0,
	ARM64_INS_STGM = AARCH64_INS_STGM,
	ARM64_INS_STGP = AARCH64_INS_STGP,
	ARM64_INS_STG = AARCH64_INS_STG,
	ARM64_INS_STILP = AARCH64_INS_STILP,
	ARM64_INS_STL1 = AARCH64_INS_STL1,
	ARM64_INS_STLLRB = AARCH64_INS_STLLRB,
	ARM64_INS_STLLRH = AARCH64_INS_STLLRH,
	ARM64_INS_STLLR = AARCH64_INS_STLLR,
	ARM64_INS_STLRB = AARCH64_INS_STLRB,
	ARM64_INS_STLRH = AARCH64_INS_STLRH,
	ARM64_INS_STLR = AARCH64_INS_STLR,
	ARM64_INS_STLURB = AARCH64_INS_STLURB,
	ARM64_INS_STLURH = AARCH64_INS_STLURH,
	ARM64_INS_STLUR = AARCH64_INS_STLUR,
	ARM64_INS_STLXP = AARCH64_INS_STLXP,
	ARM64_INS_STLXRB = AARCH64_INS_STLXRB,
	ARM64_INS_STLXRH = AARCH64_INS_STLXRH,
	ARM64_INS_STLXR = AARCH64_INS_STLXR,
	ARM64_INS_STNP = AARCH64_INS_STNP,
	ARM64_INS_STNT1B = AARCH64_INS_STNT1B,
	ARM64_INS_STNT1D = AARCH64_INS_STNT1D,
	ARM64_INS_STNT1H = AARCH64_INS_STNT1H,
	ARM64_INS_STNT1W = AARCH64_INS_STNT1W,
	ARM64_INS_STP = AARCH64_INS_STP,
	ARM64_INS_STRB = AARCH64_INS_STRB,
	ARM64_INS_STR = AARCH64_INS_STR,
	ARM64_INS_STRH = AARCH64_INS_STRH,
	ARM64_INS_STTRB = AARCH64_INS_STTRB,
	ARM64_INS_STTRH = AARCH64_INS_STTRH,
	ARM64_INS_STTR = AARCH64_INS_STTR,
	ARM64_INS_STURB = AARCH64_INS_STURB,
	ARM64_INS_STUR = AARCH64_INS_STUR,
	ARM64_INS_STURH = AARCH64_INS_STURH,
	ARM64_INS_STXP = AARCH64_INS_STXP,
	ARM64_INS_STXRB = AARCH64_INS_STXRB,
	ARM64_INS_STXRH = AARCH64_INS_STXRH,
	ARM64_INS_STXR = AARCH64_INS_STXR,
	ARM64_INS_STZ2G = AARCH64_INS_STZ2G,
	ARM64_INS_STZGM = AARCH64_INS_STZGM,
	ARM64_INS_STZG = AARCH64_INS_STZG,
	ARM64_INS_SUBG = AARCH64_INS_SUBG,
	ARM64_INS_SUBHNB = AARCH64_INS_SUBHNB,
	ARM64_INS_SUBHNT = AARCH64_INS_SUBHNT,
	ARM64_INS_SUBHN = AARCH64_INS_SUBHN,
	ARM64_INS_SUBHN2 = AARCH64_INS_SUBHN2,
	ARM64_INS_SUBP = AARCH64_INS_SUBP,
	ARM64_INS_SUBPS = AARCH64_INS_SUBPS,
	ARM64_INS_SUBPT = AARCH64_INS_SUBPT,
	ARM64_INS_SUBR = AARCH64_INS_SUBR,
	ARM64_INS_SUBS = AARCH64_INS_SUBS,
	ARM64_INS_SUB = AARCH64_INS_SUB,
	ARM64_INS_SUDOT = AARCH64_INS_SUDOT,
	ARM64_INS_SUMLALL = AARCH64_INS_SUMLALL,
	ARM64_INS_SUMOPA = AARCH64_INS_SUMOPA,
	ARM64_INS_SUMOPS = AARCH64_INS_SUMOPS,
	ARM64_INS_SUNPKHI = AARCH64_INS_SUNPKHI,
	ARM64_INS_SUNPKLO = AARCH64_INS_SUNPKLO,
	ARM64_INS_SUNPK = AARCH64_INS_SUNPK,
	ARM64_INS_SUQADD = AARCH64_INS_SUQADD,
	ARM64_INS_SUVDOT = AARCH64_INS_SUVDOT,
	ARM64_INS_SVC = AARCH64_INS_SVC,
	ARM64_INS_SVDOT = AARCH64_INS_SVDOT,
	ARM64_INS_SWPAB = AARCH64_INS_SWPAB,
	ARM64_INS_SWPAH = AARCH64_INS_SWPAH,
	ARM64_INS_SWPALB = AARCH64_INS_SWPALB,
	ARM64_INS_SWPALH = AARCH64_INS_SWPALH,
	ARM64_INS_SWPAL = AARCH64_INS_SWPAL,
	ARM64_INS_SWPA = AARCH64_INS_SWPA,
	ARM64_INS_SWPB = AARCH64_INS_SWPB,
	ARM64_INS_SWPH = AARCH64_INS_SWPH,
	ARM64_INS_SWPLB = AARCH64_INS_SWPLB,
	ARM64_INS_SWPLH = AARCH64_INS_SWPLH,
	ARM64_INS_SWPL = AARCH64_INS_SWPL,
	ARM64_INS_SWPP = AARCH64_INS_SWPP,
	ARM64_INS_SWPPA = AARCH64_INS_SWPPA,
	ARM64_INS_SWPPAL = AARCH64_INS_SWPPAL,
	ARM64_INS_SWPPL = AARCH64_INS_SWPPL,
	ARM64_INS_SWP = AARCH64_INS_SWP,
	ARM64_INS_SXTB = AARCH64_INS_SXTB,
	ARM64_INS_SXTH = AARCH64_INS_SXTH,
	ARM64_INS_SXTW = AARCH64_INS_SXTW,
	ARM64_INS_SYSL = AARCH64_INS_SYSL,
	ARM64_INS_SYSP = AARCH64_INS_SYSP,
	ARM64_INS_SYS = AARCH64_INS_SYS,
	ARM64_INS_TBLQ = AARCH64_INS_TBLQ,
	ARM64_INS_TBL = AARCH64_INS_TBL,
	ARM64_INS_TBNZ = AARCH64_INS_TBNZ,
	ARM64_INS_TBXQ = AARCH64_INS_TBXQ,
	ARM64_INS_TBX = AARCH64_INS_TBX,
	ARM64_INS_TBZ = AARCH64_INS_TBZ,
	ARM64_INS_TCANCEL = AARCH64_INS_TCANCEL,
	ARM64_INS_TCOMMIT = AARCH64_INS_TCOMMIT,
	ARM64_INS_TRCIT = AARCH64_INS_TRCIT,
	ARM64_INS_TRN1 = AARCH64_INS_TRN1,
	ARM64_INS_TRN2 = AARCH64_INS_TRN2,
	ARM64_INS_TSB = AARCH64_INS_TSB,
	ARM64_INS_TSTART = AARCH64_INS_TSTART,
	ARM64_INS_TTEST = AARCH64_INS_TTEST,
	ARM64_INS_UABALB = AARCH64_INS_UABALB,
	ARM64_INS_UABALT = AARCH64_INS_UABALT,
	ARM64_INS_UABAL2 = AARCH64_INS_UABAL2,
	ARM64_INS_UABAL = AARCH64_INS_UABAL,
	ARM64_INS_UABA = AARCH64_INS_UABA,
	ARM64_INS_UABDLB = AARCH64_INS_UABDLB,
	ARM64_INS_UABDLT = AARCH64_INS_UABDLT,
	ARM64_INS_UABDL2 = AARCH64_INS_UABDL2,
	ARM64_INS_UABDL = AARCH64_INS_UABDL,
	ARM64_INS_UABD = AARCH64_INS_UABD,
	ARM64_INS_UADALP = AARCH64_INS_UADALP,
	ARM64_INS_UADDLB = AARCH64_INS_UADDLB,
	ARM64_INS_UADDLP = AARCH64_INS_UADDLP,
	ARM64_INS_UADDLT = AARCH64_INS_UADDLT,
	ARM64_INS_UADDLV = AARCH64_INS_UADDLV,
	ARM64_INS_UADDL2 = AARCH64_INS_UADDL2,
	ARM64_INS_UADDL = AARCH64_INS_UADDL,
	ARM64_INS_UADDV = AARCH64_INS_UADDV,
	ARM64_INS_UADDWB = AARCH64_INS_UADDWB,
	ARM64_INS_UADDWT = AARCH64_INS_UADDWT,
	ARM64_INS_UADDW2 = AARCH64_INS_UADDW2,
	ARM64_INS_UADDW = AARCH64_INS_UADDW,
	ARM64_INS_UBFM = AARCH64_INS_UBFM,
	ARM64_INS_UCLAMP = AARCH64_INS_UCLAMP,
	ARM64_INS_UCVTF = AARCH64_INS_UCVTF,
	ARM64_INS_UDF = AARCH64_INS_UDF,
	ARM64_INS_UDIVR = AARCH64_INS_UDIVR,
	ARM64_INS_UDIV = AARCH64_INS_UDIV,
	ARM64_INS_UDOT = AARCH64_INS_UDOT,
	ARM64_INS_UHADD = AARCH64_INS_UHADD,
	ARM64_INS_UHSUBR = AARCH64_INS_UHSUBR,
	ARM64_INS_UHSUB = AARCH64_INS_UHSUB,
	ARM64_INS_UMADDL = AARCH64_INS_UMADDL,
	ARM64_INS_UMAXP = AARCH64_INS_UMAXP,
	ARM64_INS_UMAXQV = AARCH64_INS_UMAXQV,
	ARM64_INS_UMAXV = AARCH64_INS_UMAXV,
	ARM64_INS_UMAX = AARCH64_INS_UMAX,
	ARM64_INS_UMINP = AARCH64_INS_UMINP,
	ARM64_INS_UMINQV = AARCH64_INS_UMINQV,
	ARM64_INS_UMINV = AARCH64_INS_UMINV,
	ARM64_INS_UMIN = AARCH64_INS_UMIN,
	ARM64_INS_UMLALB = AARCH64_INS_UMLALB,
	ARM64_INS_UMLALL = AARCH64_INS_UMLALL,
	ARM64_INS_UMLALT = AARCH64_INS_UMLALT,
	ARM64_INS_UMLAL = AARCH64_INS_UMLAL,
	ARM64_INS_UMLAL2 = AARCH64_INS_UMLAL2,
	ARM64_INS_UMLSLB = AARCH64_INS_UMLSLB,
	ARM64_INS_UMLSLL = AARCH64_INS_UMLSLL,
	ARM64_INS_UMLSLT = AARCH64_INS_UMLSLT,
	ARM64_INS_UMLSL = AARCH64_INS_UMLSL,
	ARM64_INS_UMLSL2 = AARCH64_INS_UMLSL2,
	ARM64_INS_UMMLA = AARCH64_INS_UMMLA,
	ARM64_INS_UMOPA = AARCH64_INS_UMOPA,
	ARM64_INS_UMOPS = AARCH64_INS_UMOPS,
	ARM64_INS_UMOV = AARCH64_INS_UMOV,
	ARM64_INS_UMSUBL = AARCH64_INS_UMSUBL,
	ARM64_INS_UMULH = AARCH64_INS_UMULH,
	ARM64_INS_UMULLB = AARCH64_INS_UMULLB,
	ARM64_INS_UMULLT = AARCH64_INS_UMULLT,
	ARM64_INS_UMULL2 = AARCH64_INS_UMULL2,
	ARM64_INS_UMULL = AARCH64_INS_UMULL,
	ARM64_INS_UQADD = AARCH64_INS_UQADD,
	ARM64_INS_UQCVTN = AARCH64_INS_UQCVTN,
	ARM64_INS_UQCVT = AARCH64_INS_UQCVT,
	ARM64_INS_UQDECB = AARCH64_INS_UQDECB,
	ARM64_INS_UQDECD = AARCH64_INS_UQDECD,
	ARM64_INS_UQDECH = AARCH64_INS_UQDECH,
	ARM64_INS_UQDECP = AARCH64_INS_UQDECP,
	ARM64_INS_UQDECW = AARCH64_INS_UQDECW,
	ARM64_INS_UQINCB = AARCH64_INS_UQINCB,
	ARM64_INS_UQINCD = AARCH64_INS_UQINCD,
	ARM64_INS_UQINCH = AARCH64_INS_UQINCH,
	ARM64_INS_UQINCP = AARCH64_INS_UQINCP,
	ARM64_INS_UQINCW = AARCH64_INS_UQINCW,
	ARM64_INS_UQRSHLR = AARCH64_INS_UQRSHLR,
	ARM64_INS_UQRSHL = AARCH64_INS_UQRSHL,
	ARM64_INS_UQRSHRNB = AARCH64_INS_UQRSHRNB,
	ARM64_INS_UQRSHRNT = AARCH64_INS_UQRSHRNT,
	ARM64_INS_UQRSHRN = AARCH64_INS_UQRSHRN,
	ARM64_INS_UQRSHRN2 = AARCH64_INS_UQRSHRN2,
	ARM64_INS_UQRSHR = AARCH64_INS_UQRSHR,
	ARM64_INS_UQSHLR = AARCH64_INS_UQSHLR,
	ARM64_INS_UQSHL = AARCH64_INS_UQSHL,
	ARM64_INS_UQSHRNB = AARCH64_INS_UQSHRNB,
	ARM64_INS_UQSHRNT = AARCH64_INS_UQSHRNT,
	ARM64_INS_UQSHRN = AARCH64_INS_UQSHRN,
	ARM64_INS_UQSHRN2 = AARCH64_INS_UQSHRN2,
	ARM64_INS_UQSUBR = AARCH64_INS_UQSUBR,
	ARM64_INS_UQSUB = AARCH64_INS_UQSUB,
	ARM64_INS_UQXTNB = AARCH64_INS_UQXTNB,
	ARM64_INS_UQXTNT = AARCH64_INS_UQXTNT,
	ARM64_INS_UQXTN2 = AARCH64_INS_UQXTN2,
	ARM64_INS_UQXTN = AARCH64_INS_UQXTN,
	ARM64_INS_URECPE = AARCH64_INS_URECPE,
	ARM64_INS_URHADD = AARCH64_INS_URHADD,
	ARM64_INS_URSHLR = AARCH64_INS_URSHLR,
	ARM64_INS_URSHL = AARCH64_INS_URSHL,
	ARM64_INS_URSHR = AARCH64_INS_URSHR,
	ARM64_INS_URSQRTE = AARCH64_INS_URSQRTE,
	ARM64_INS_URSRA = AARCH64_INS_URSRA,
	ARM64_INS_USDOT = AARCH64_INS_USDOT,
	ARM64_INS_USHLLB = AARCH64_INS_USHLLB,
	ARM64_INS_USHLLT = AARCH64_INS_USHLLT,
	ARM64_INS_USHLL2 = AARCH64_INS_USHLL2,
	ARM64_INS_USHLL = AARCH64_INS_USHLL,
	ARM64_INS_USHL = AARCH64_INS_USHL,
	ARM64_INS_USHR = AARCH64_INS_USHR,
	ARM64_INS_USMLALL = AARCH64_INS_USMLALL,
	ARM64_INS_USMMLA = AARCH64_INS_USMMLA,
	ARM64_INS_USMOPA = AARCH64_INS_USMOPA,
	ARM64_INS_USMOPS = AARCH64_INS_USMOPS,
	ARM64_INS_USQADD = AARCH64_INS_USQADD,
	ARM64_INS_USRA = AARCH64_INS_USRA,
	ARM64_INS_USUBLB = AARCH64_INS_USUBLB,
	ARM64_INS_USUBLT = AARCH64_INS_USUBLT,
	ARM64_INS_USUBL2 = AARCH64_INS_USUBL2,
	ARM64_INS_USUBL = AARCH64_INS_USUBL,
	ARM64_INS_USUBWB = AARCH64_INS_USUBWB,
	ARM64_INS_USUBWT = AARCH64_INS_USUBWT,
	ARM64_INS_USUBW2 = AARCH64_INS_USUBW2,
	ARM64_INS_USUBW = AARCH64_INS_USUBW,
	ARM64_INS_USVDOT = AARCH64_INS_USVDOT,
	ARM64_INS_UUNPKHI = AARCH64_INS_UUNPKHI,
	ARM64_INS_UUNPKLO = AARCH64_INS_UUNPKLO,
	ARM64_INS_UUNPK = AARCH64_INS_UUNPK,
	ARM64_INS_UVDOT = AARCH64_INS_UVDOT,
	ARM64_INS_UXTB = AARCH64_INS_UXTB,
	ARM64_INS_UXTH = AARCH64_INS_UXTH,
	ARM64_INS_UXTW = AARCH64_INS_UXTW,
	ARM64_INS_UZP1 = AARCH64_INS_UZP1,
	ARM64_INS_UZP2 = AARCH64_INS_UZP2,
	ARM64_INS_UZPQ1 = AARCH64_INS_UZPQ1,
	ARM64_INS_UZPQ2 = AARCH64_INS_UZPQ2,
	ARM64_INS_UZP = AARCH64_INS_UZP,
	ARM64_INS_WFET = AARCH64_INS_WFET,
	ARM64_INS_WFIT = AARCH64_INS_WFIT,
	ARM64_INS_WHILEGE = AARCH64_INS_WHILEGE,
	ARM64_INS_WHILEGT = AARCH64_INS_WHILEGT,
	ARM64_INS_WHILEHI = AARCH64_INS_WHILEHI,
	ARM64_INS_WHILEHS = AARCH64_INS_WHILEHS,
	ARM64_INS_WHILELE = AARCH64_INS_WHILELE,
	ARM64_INS_WHILELO = AARCH64_INS_WHILELO,
	ARM64_INS_WHILELS = AARCH64_INS_WHILELS,
	ARM64_INS_WHILELT = AARCH64_INS_WHILELT,
	ARM64_INS_WHILERW = AARCH64_INS_WHILERW,
	ARM64_INS_WHILEWR = AARCH64_INS_WHILEWR,
	ARM64_INS_WRFFR = AARCH64_INS_WRFFR,
	ARM64_INS_XAFLAG = AARCH64_INS_XAFLAG,
	ARM64_INS_XAR = AARCH64_INS_XAR,
	ARM64_INS_XPACD = AARCH64_INS_XPACD,
	ARM64_INS_XPACI = AARCH64_INS_XPACI,
	ARM64_INS_XTN2 = AARCH64_INS_XTN2,
	ARM64_INS_XTN = AARCH64_INS_XTN,
	ARM64_INS_ZERO = AARCH64_INS_ZERO,
	ARM64_INS_ZIP1 = AARCH64_INS_ZIP1,
	ARM64_INS_ZIP2 = AARCH64_INS_ZIP2,
	ARM64_INS_ZIPQ1 = AARCH64_INS_ZIPQ1,
	ARM64_INS_ZIPQ2 = AARCH64_INS_ZIPQ2,
	ARM64_INS_ZIP = AARCH64_INS_ZIP,


  ARM64_INS_ENDING = AARCH64_INS_ENDING,

	ARM64_INS_ALIAS_BEGIN = AARCH64_INS_ALIAS_BEGIN,

	ARM64_INS_ALIAS_ADDPT = AARCH64_INS_ALIAS_ADDPT,
	ARM64_INS_ALIAS_GCSB = AARCH64_INS_ALIAS_GCSB,
	ARM64_INS_ALIAS_GCSPOPM = AARCH64_INS_ALIAS_GCSPOPM,
	ARM64_INS_ALIAS_LDAPUR = AARCH64_INS_ALIAS_LDAPUR,
	ARM64_INS_ALIAS_STLLRB = AARCH64_INS_ALIAS_STLLRB,
	ARM64_INS_ALIAS_STLLRH = AARCH64_INS_ALIAS_STLLRH,
	ARM64_INS_ALIAS_STLLR = AARCH64_INS_ALIAS_STLLR,
	ARM64_INS_ALIAS_STLRB = AARCH64_INS_ALIAS_STLRB,
	ARM64_INS_ALIAS_STLRH = AARCH64_INS_ALIAS_STLRH,
	ARM64_INS_ALIAS_STLR = AARCH64_INS_ALIAS_STLR,
	ARM64_INS_ALIAS_STLUR = AARCH64_INS_ALIAS_STLUR,
	ARM64_INS_ALIAS_SUBPT = AARCH64_INS_ALIAS_SUBPT,
	ARM64_INS_ALIAS_LDRAA = AARCH64_INS_ALIAS_LDRAA,
	ARM64_INS_ALIAS_ADD = AARCH64_INS_ALIAS_ADD,
	ARM64_INS_ALIAS_CMN = AARCH64_INS_ALIAS_CMN,
	ARM64_INS_ALIAS_ADDS = AARCH64_INS_ALIAS_ADDS,
	ARM64_INS_ALIAS_AND = AARCH64_INS_ALIAS_AND,
	ARM64_INS_ALIAS_ANDS = AARCH64_INS_ALIAS_ANDS,
	ARM64_INS_ALIAS_LDR = AARCH64_INS_ALIAS_LDR,
	ARM64_INS_ALIAS_STR = AARCH64_INS_ALIAS_STR,
	ARM64_INS_ALIAS_LDRB = AARCH64_INS_ALIAS_LDRB,
	ARM64_INS_ALIAS_STRB = AARCH64_INS_ALIAS_STRB,
	ARM64_INS_ALIAS_LDRH = AARCH64_INS_ALIAS_LDRH,
	ARM64_INS_ALIAS_STRH = AARCH64_INS_ALIAS_STRH,
	ARM64_INS_ALIAS_PRFM = AARCH64_INS_ALIAS_PRFM,
	ARM64_INS_ALIAS_LDAPURB = AARCH64_INS_ALIAS_LDAPURB,
	ARM64_INS_ALIAS_STLURB = AARCH64_INS_ALIAS_STLURB,
	ARM64_INS_ALIAS_LDUR = AARCH64_INS_ALIAS_LDUR,
	ARM64_INS_ALIAS_STUR = AARCH64_INS_ALIAS_STUR,
	ARM64_INS_ALIAS_PRFUM = AARCH64_INS_ALIAS_PRFUM,
	ARM64_INS_ALIAS_LDTR = AARCH64_INS_ALIAS_LDTR,
	ARM64_INS_ALIAS_STTR = AARCH64_INS_ALIAS_STTR,
	ARM64_INS_ALIAS_LDP = AARCH64_INS_ALIAS_LDP,
	ARM64_INS_ALIAS_STGP = AARCH64_INS_ALIAS_STGP,
	ARM64_INS_ALIAS_LDNP = AARCH64_INS_ALIAS_LDNP,
	ARM64_INS_ALIAS_STNP = AARCH64_INS_ALIAS_STNP,
	ARM64_INS_ALIAS_STG = AARCH64_INS_ALIAS_STG,
	ARM64_INS_ALIAS_MOV = AARCH64_INS_ALIAS_MOV,
	ARM64_INS_ALIAS_LD1 = AARCH64_INS_ALIAS_LD1,
	ARM64_INS_ALIAS_LD1R = AARCH64_INS_ALIAS_LD1R,
	ARM64_INS_ALIAS_STADDLB = AARCH64_INS_ALIAS_STADDLB,
	ARM64_INS_ALIAS_STADDLH = AARCH64_INS_ALIAS_STADDLH,
	ARM64_INS_ALIAS_STADDL = AARCH64_INS_ALIAS_STADDL,
	ARM64_INS_ALIAS_STADDB = AARCH64_INS_ALIAS_STADDB,
	ARM64_INS_ALIAS_STADDH = AARCH64_INS_ALIAS_STADDH,
	ARM64_INS_ALIAS_STADD = AARCH64_INS_ALIAS_STADD,
	ARM64_INS_ALIAS_PTRUE = AARCH64_INS_ALIAS_PTRUE,
	ARM64_INS_ALIAS_PTRUES = AARCH64_INS_ALIAS_PTRUES,
	ARM64_INS_ALIAS_CNTB = AARCH64_INS_ALIAS_CNTB,
	ARM64_INS_ALIAS_SQINCH = AARCH64_INS_ALIAS_SQINCH,
	ARM64_INS_ALIAS_INCB = AARCH64_INS_ALIAS_INCB,
	ARM64_INS_ALIAS_SQINCB = AARCH64_INS_ALIAS_SQINCB,
	ARM64_INS_ALIAS_UQINCB = AARCH64_INS_ALIAS_UQINCB,
	ARM64_INS_ALIAS_ORR = AARCH64_INS_ALIAS_ORR,
	ARM64_INS_ALIAS_DUPM = AARCH64_INS_ALIAS_DUPM,
	ARM64_INS_ALIAS_FMOV = AARCH64_INS_ALIAS_FMOV,
	ARM64_INS_ALIAS_EOR3 = AARCH64_INS_ALIAS_EOR3,
	ARM64_INS_ALIAS_ST1B = AARCH64_INS_ALIAS_ST1B,
	ARM64_INS_ALIAS_ST2B = AARCH64_INS_ALIAS_ST2B,
	ARM64_INS_ALIAS_ST2Q = AARCH64_INS_ALIAS_ST2Q,
	ARM64_INS_ALIAS_STNT1B = AARCH64_INS_ALIAS_STNT1B,
	ARM64_INS_ALIAS_LD1B = AARCH64_INS_ALIAS_LD1B,
	ARM64_INS_ALIAS_LDNT1B = AARCH64_INS_ALIAS_LDNT1B,
	ARM64_INS_ALIAS_LD1RQB = AARCH64_INS_ALIAS_LD1RQB,
	ARM64_INS_ALIAS_LD1RB = AARCH64_INS_ALIAS_LD1RB,
	ARM64_INS_ALIAS_LDFF1B = AARCH64_INS_ALIAS_LDFF1B,
	ARM64_INS_ALIAS_LDNF1B = AARCH64_INS_ALIAS_LDNF1B,
	ARM64_INS_ALIAS_LD2B = AARCH64_INS_ALIAS_LD2B,
	ARM64_INS_ALIAS_LD1SB = AARCH64_INS_ALIAS_LD1SB,
	ARM64_INS_ALIAS_PRFB = AARCH64_INS_ALIAS_PRFB,
	ARM64_INS_ALIAS_LDNT1SB = AARCH64_INS_ALIAS_LDNT1SB,
	ARM64_INS_ALIAS_LD1ROB = AARCH64_INS_ALIAS_LD1ROB,
	ARM64_INS_ALIAS_LD1Q = AARCH64_INS_ALIAS_LD1Q,
	ARM64_INS_ALIAS_ST1Q = AARCH64_INS_ALIAS_ST1Q,
	ARM64_INS_ALIAS_LD1W = AARCH64_INS_ALIAS_LD1W,
	ARM64_INS_ALIAS_PMOV = AARCH64_INS_ALIAS_PMOV,
	ARM64_INS_ALIAS_SMSTART = AARCH64_INS_ALIAS_SMSTART,
	ARM64_INS_ALIAS_SMSTOP = AARCH64_INS_ALIAS_SMSTOP,
	ARM64_INS_ALIAS_ZERO = AARCH64_INS_ALIAS_ZERO,
	ARM64_INS_ALIAS_MOVT = AARCH64_INS_ALIAS_MOVT,
	ARM64_INS_ALIAS_NOP = AARCH64_INS_ALIAS_NOP,
	ARM64_INS_ALIAS_YIELD = AARCH64_INS_ALIAS_YIELD,
	ARM64_INS_ALIAS_WFE = AARCH64_INS_ALIAS_WFE,
	ARM64_INS_ALIAS_WFI = AARCH64_INS_ALIAS_WFI,
	ARM64_INS_ALIAS_SEV = AARCH64_INS_ALIAS_SEV,
	ARM64_INS_ALIAS_SEVL = AARCH64_INS_ALIAS_SEVL,
	ARM64_INS_ALIAS_DGH = AARCH64_INS_ALIAS_DGH,
	ARM64_INS_ALIAS_ESB = AARCH64_INS_ALIAS_ESB,
	ARM64_INS_ALIAS_CSDB = AARCH64_INS_ALIAS_CSDB,
	ARM64_INS_ALIAS_BTI = AARCH64_INS_ALIAS_BTI,
	ARM64_INS_ALIAS_PSB = AARCH64_INS_ALIAS_PSB,
	ARM64_INS_ALIAS_CHKFEAT = AARCH64_INS_ALIAS_CHKFEAT,
	ARM64_INS_ALIAS_PACIAZ = AARCH64_INS_ALIAS_PACIAZ,
	ARM64_INS_ALIAS_PACIBZ = AARCH64_INS_ALIAS_PACIBZ,
	ARM64_INS_ALIAS_AUTIAZ = AARCH64_INS_ALIAS_AUTIAZ,
	ARM64_INS_ALIAS_AUTIBZ = AARCH64_INS_ALIAS_AUTIBZ,
	ARM64_INS_ALIAS_PACIASP = AARCH64_INS_ALIAS_PACIASP,
	ARM64_INS_ALIAS_PACIBSP = AARCH64_INS_ALIAS_PACIBSP,
	ARM64_INS_ALIAS_AUTIASP = AARCH64_INS_ALIAS_AUTIASP,
	ARM64_INS_ALIAS_AUTIBSP = AARCH64_INS_ALIAS_AUTIBSP,
	ARM64_INS_ALIAS_PACIA1716 = AARCH64_INS_ALIAS_PACIA1716,
	ARM64_INS_ALIAS_PACIB1716 = AARCH64_INS_ALIAS_PACIB1716,
	ARM64_INS_ALIAS_AUTIA1716 = AARCH64_INS_ALIAS_AUTIA1716,
	ARM64_INS_ALIAS_AUTIB1716 = AARCH64_INS_ALIAS_AUTIB1716,
	ARM64_INS_ALIAS_XPACLRI = AARCH64_INS_ALIAS_XPACLRI,
	ARM64_INS_ALIAS_LDRAB = AARCH64_INS_ALIAS_LDRAB,
	ARM64_INS_ALIAS_PACM = AARCH64_INS_ALIAS_PACM,
	ARM64_INS_ALIAS_CLREX = AARCH64_INS_ALIAS_CLREX,
	ARM64_INS_ALIAS_ISB = AARCH64_INS_ALIAS_ISB,
	ARM64_INS_ALIAS_SSBB = AARCH64_INS_ALIAS_SSBB,
	ARM64_INS_ALIAS_PSSBB = AARCH64_INS_ALIAS_PSSBB,
	ARM64_INS_ALIAS_DFB = AARCH64_INS_ALIAS_DFB,
	ARM64_INS_ALIAS_SYS = AARCH64_INS_ALIAS_SYS,
	ARM64_INS_ALIAS_MOVN = AARCH64_INS_ALIAS_MOVN,
	ARM64_INS_ALIAS_MOVZ = AARCH64_INS_ALIAS_MOVZ,
	ARM64_INS_ALIAS_NGC = AARCH64_INS_ALIAS_NGC,
	ARM64_INS_ALIAS_NGCS = AARCH64_INS_ALIAS_NGCS,
	ARM64_INS_ALIAS_SUB = AARCH64_INS_ALIAS_SUB,
	ARM64_INS_ALIAS_CMP = AARCH64_INS_ALIAS_CMP,
	ARM64_INS_ALIAS_SUBS = AARCH64_INS_ALIAS_SUBS,
	ARM64_INS_ALIAS_NEG = AARCH64_INS_ALIAS_NEG,
	ARM64_INS_ALIAS_NEGS = AARCH64_INS_ALIAS_NEGS,
	ARM64_INS_ALIAS_MUL = AARCH64_INS_ALIAS_MUL,
	ARM64_INS_ALIAS_MNEG = AARCH64_INS_ALIAS_MNEG,
	ARM64_INS_ALIAS_SMULL = AARCH64_INS_ALIAS_SMULL,
	ARM64_INS_ALIAS_SMNEGL = AARCH64_INS_ALIAS_SMNEGL,
	ARM64_INS_ALIAS_UMULL = AARCH64_INS_ALIAS_UMULL,
	ARM64_INS_ALIAS_UMNEGL = AARCH64_INS_ALIAS_UMNEGL,
	ARM64_INS_ALIAS_STCLRLB = AARCH64_INS_ALIAS_STCLRLB,
	ARM64_INS_ALIAS_STCLRLH = AARCH64_INS_ALIAS_STCLRLH,
	ARM64_INS_ALIAS_STCLRL = AARCH64_INS_ALIAS_STCLRL,
	ARM64_INS_ALIAS_STCLRB = AARCH64_INS_ALIAS_STCLRB,
	ARM64_INS_ALIAS_STCLRH = AARCH64_INS_ALIAS_STCLRH,
	ARM64_INS_ALIAS_STCLR = AARCH64_INS_ALIAS_STCLR,
	ARM64_INS_ALIAS_STEORLB = AARCH64_INS_ALIAS_STEORLB,
	ARM64_INS_ALIAS_STEORLH = AARCH64_INS_ALIAS_STEORLH,
	ARM64_INS_ALIAS_STEORL = AARCH64_INS_ALIAS_STEORL,
	ARM64_INS_ALIAS_STEORB = AARCH64_INS_ALIAS_STEORB,
	ARM64_INS_ALIAS_STEORH = AARCH64_INS_ALIAS_STEORH,
	ARM64_INS_ALIAS_STEOR = AARCH64_INS_ALIAS_STEOR,
	ARM64_INS_ALIAS_STSETLB = AARCH64_INS_ALIAS_STSETLB,
	ARM64_INS_ALIAS_STSETLH = AARCH64_INS_ALIAS_STSETLH,
	ARM64_INS_ALIAS_STSETL = AARCH64_INS_ALIAS_STSETL,
	ARM64_INS_ALIAS_STSETB = AARCH64_INS_ALIAS_STSETB,
	ARM64_INS_ALIAS_STSETH = AARCH64_INS_ALIAS_STSETH,
	ARM64_INS_ALIAS_STSET = AARCH64_INS_ALIAS_STSET,
	ARM64_INS_ALIAS_STSMAXLB = AARCH64_INS_ALIAS_STSMAXLB,
	ARM64_INS_ALIAS_STSMAXLH = AARCH64_INS_ALIAS_STSMAXLH,
	ARM64_INS_ALIAS_STSMAXL = AARCH64_INS_ALIAS_STSMAXL,
	ARM64_INS_ALIAS_STSMAXB = AARCH64_INS_ALIAS_STSMAXB,
	ARM64_INS_ALIAS_STSMAXH = AARCH64_INS_ALIAS_STSMAXH,
	ARM64_INS_ALIAS_STSMAX = AARCH64_INS_ALIAS_STSMAX,
	ARM64_INS_ALIAS_STSMINLB = AARCH64_INS_ALIAS_STSMINLB,
	ARM64_INS_ALIAS_STSMINLH = AARCH64_INS_ALIAS_STSMINLH,
	ARM64_INS_ALIAS_STSMINL = AARCH64_INS_ALIAS_STSMINL,
	ARM64_INS_ALIAS_STSMINB = AARCH64_INS_ALIAS_STSMINB,
	ARM64_INS_ALIAS_STSMINH = AARCH64_INS_ALIAS_STSMINH,
	ARM64_INS_ALIAS_STSMIN = AARCH64_INS_ALIAS_STSMIN,
	ARM64_INS_ALIAS_STUMAXLB = AARCH64_INS_ALIAS_STUMAXLB,
	ARM64_INS_ALIAS_STUMAXLH = AARCH64_INS_ALIAS_STUMAXLH,
	ARM64_INS_ALIAS_STUMAXL = AARCH64_INS_ALIAS_STUMAXL,
	ARM64_INS_ALIAS_STUMAXB = AARCH64_INS_ALIAS_STUMAXB,
	ARM64_INS_ALIAS_STUMAXH = AARCH64_INS_ALIAS_STUMAXH,
	ARM64_INS_ALIAS_STUMAX = AARCH64_INS_ALIAS_STUMAX,
	ARM64_INS_ALIAS_STUMINLB = AARCH64_INS_ALIAS_STUMINLB,
	ARM64_INS_ALIAS_STUMINLH = AARCH64_INS_ALIAS_STUMINLH,
	ARM64_INS_ALIAS_STUMINL = AARCH64_INS_ALIAS_STUMINL,
	ARM64_INS_ALIAS_STUMINB = AARCH64_INS_ALIAS_STUMINB,
	ARM64_INS_ALIAS_STUMINH = AARCH64_INS_ALIAS_STUMINH,
	ARM64_INS_ALIAS_STUMIN = AARCH64_INS_ALIAS_STUMIN,
	ARM64_INS_ALIAS_IRG = AARCH64_INS_ALIAS_IRG,
	ARM64_INS_ALIAS_LDG = AARCH64_INS_ALIAS_LDG,
	ARM64_INS_ALIAS_STZG = AARCH64_INS_ALIAS_STZG,
	ARM64_INS_ALIAS_ST2G = AARCH64_INS_ALIAS_ST2G,
	ARM64_INS_ALIAS_STZ2G = AARCH64_INS_ALIAS_STZ2G,
	ARM64_INS_ALIAS_BICS = AARCH64_INS_ALIAS_BICS,
	ARM64_INS_ALIAS_BIC = AARCH64_INS_ALIAS_BIC,
	ARM64_INS_ALIAS_EON = AARCH64_INS_ALIAS_EON,
	ARM64_INS_ALIAS_EOR = AARCH64_INS_ALIAS_EOR,
	ARM64_INS_ALIAS_ORN = AARCH64_INS_ALIAS_ORN,
	ARM64_INS_ALIAS_MVN = AARCH64_INS_ALIAS_MVN,
	ARM64_INS_ALIAS_TST = AARCH64_INS_ALIAS_TST,
	ARM64_INS_ALIAS_ROR = AARCH64_INS_ALIAS_ROR,
	ARM64_INS_ALIAS_ASR = AARCH64_INS_ALIAS_ASR,
	ARM64_INS_ALIAS_SXTB = AARCH64_INS_ALIAS_SXTB,
	ARM64_INS_ALIAS_SXTH = AARCH64_INS_ALIAS_SXTH,
	ARM64_INS_ALIAS_SXTW = AARCH64_INS_ALIAS_SXTW,
	ARM64_INS_ALIAS_LSR = AARCH64_INS_ALIAS_LSR,
	ARM64_INS_ALIAS_UXTB = AARCH64_INS_ALIAS_UXTB,
	ARM64_INS_ALIAS_UXTH = AARCH64_INS_ALIAS_UXTH,
	ARM64_INS_ALIAS_UXTW = AARCH64_INS_ALIAS_UXTW,
	ARM64_INS_ALIAS_CSET = AARCH64_INS_ALIAS_CSET,
	ARM64_INS_ALIAS_CSETM = AARCH64_INS_ALIAS_CSETM,
	ARM64_INS_ALIAS_CINC = AARCH64_INS_ALIAS_CINC,
	ARM64_INS_ALIAS_CINV = AARCH64_INS_ALIAS_CINV,
	ARM64_INS_ALIAS_CNEG = AARCH64_INS_ALIAS_CNEG,
	ARM64_INS_ALIAS_RET = AARCH64_INS_ALIAS_RET,
	ARM64_INS_ALIAS_DCPS1 = AARCH64_INS_ALIAS_DCPS1,
	ARM64_INS_ALIAS_DCPS2 = AARCH64_INS_ALIAS_DCPS2,
	ARM64_INS_ALIAS_DCPS3 = AARCH64_INS_ALIAS_DCPS3,
	ARM64_INS_ALIAS_LDPSW = AARCH64_INS_ALIAS_LDPSW,
	ARM64_INS_ALIAS_LDRSH = AARCH64_INS_ALIAS_LDRSH,
	ARM64_INS_ALIAS_LDRSB = AARCH64_INS_ALIAS_LDRSB,
	ARM64_INS_ALIAS_LDRSW = AARCH64_INS_ALIAS_LDRSW,
	ARM64_INS_ALIAS_LDURH = AARCH64_INS_ALIAS_LDURH,
	ARM64_INS_ALIAS_LDURB = AARCH64_INS_ALIAS_LDURB,
	ARM64_INS_ALIAS_LDURSH = AARCH64_INS_ALIAS_LDURSH,
	ARM64_INS_ALIAS_LDURSB = AARCH64_INS_ALIAS_LDURSB,
	ARM64_INS_ALIAS_LDURSW = AARCH64_INS_ALIAS_LDURSW,
	ARM64_INS_ALIAS_LDTRH = AARCH64_INS_ALIAS_LDTRH,
	ARM64_INS_ALIAS_LDTRB = AARCH64_INS_ALIAS_LDTRB,
	ARM64_INS_ALIAS_LDTRSH = AARCH64_INS_ALIAS_LDTRSH,
	ARM64_INS_ALIAS_LDTRSB = AARCH64_INS_ALIAS_LDTRSB,
	ARM64_INS_ALIAS_LDTRSW = AARCH64_INS_ALIAS_LDTRSW,
	ARM64_INS_ALIAS_STP = AARCH64_INS_ALIAS_STP,
	ARM64_INS_ALIAS_STURH = AARCH64_INS_ALIAS_STURH,
	ARM64_INS_ALIAS_STURB = AARCH64_INS_ALIAS_STURB,
	ARM64_INS_ALIAS_STLURH = AARCH64_INS_ALIAS_STLURH,
	ARM64_INS_ALIAS_LDAPURSB = AARCH64_INS_ALIAS_LDAPURSB,
	ARM64_INS_ALIAS_LDAPURH = AARCH64_INS_ALIAS_LDAPURH,
	ARM64_INS_ALIAS_LDAPURSH = AARCH64_INS_ALIAS_LDAPURSH,
	ARM64_INS_ALIAS_LDAPURSW = AARCH64_INS_ALIAS_LDAPURSW,
	ARM64_INS_ALIAS_STTRH = AARCH64_INS_ALIAS_STTRH,
	ARM64_INS_ALIAS_STTRB = AARCH64_INS_ALIAS_STTRB,
	ARM64_INS_ALIAS_BIC_4H = AARCH64_INS_ALIAS_BIC_4H,
	ARM64_INS_ALIAS_BIC_8H = AARCH64_INS_ALIAS_BIC_8H,
	ARM64_INS_ALIAS_BIC_2S = AARCH64_INS_ALIAS_BIC_2S,
	ARM64_INS_ALIAS_BIC_4S = AARCH64_INS_ALIAS_BIC_4S,
	ARM64_INS_ALIAS_ORR_4H = AARCH64_INS_ALIAS_ORR_4H,
	ARM64_INS_ALIAS_ORR_8H = AARCH64_INS_ALIAS_ORR_8H,
	ARM64_INS_ALIAS_ORR_2S = AARCH64_INS_ALIAS_ORR_2S,
	ARM64_INS_ALIAS_ORR_4S = AARCH64_INS_ALIAS_ORR_4S,
	ARM64_INS_ALIAS_SXTL_8H = AARCH64_INS_ALIAS_SXTL_8H,
	ARM64_INS_ALIAS_SXTL = AARCH64_INS_ALIAS_SXTL,
	ARM64_INS_ALIAS_SXTL_4S = AARCH64_INS_ALIAS_SXTL_4S,
	ARM64_INS_ALIAS_SXTL_2D = AARCH64_INS_ALIAS_SXTL_2D,
	ARM64_INS_ALIAS_SXTL2_8H = AARCH64_INS_ALIAS_SXTL2_8H,
	ARM64_INS_ALIAS_SXTL2 = AARCH64_INS_ALIAS_SXTL2,
	ARM64_INS_ALIAS_SXTL2_4S = AARCH64_INS_ALIAS_SXTL2_4S,
	ARM64_INS_ALIAS_SXTL2_2D = AARCH64_INS_ALIAS_SXTL2_2D,
	ARM64_INS_ALIAS_UXTL_8H = AARCH64_INS_ALIAS_UXTL_8H,
	ARM64_INS_ALIAS_UXTL = AARCH64_INS_ALIAS_UXTL,
	ARM64_INS_ALIAS_UXTL_4S = AARCH64_INS_ALIAS_UXTL_4S,
	ARM64_INS_ALIAS_UXTL_2D = AARCH64_INS_ALIAS_UXTL_2D,
	ARM64_INS_ALIAS_UXTL2_8H = AARCH64_INS_ALIAS_UXTL2_8H,
	ARM64_INS_ALIAS_UXTL2 = AARCH64_INS_ALIAS_UXTL2,
	ARM64_INS_ALIAS_UXTL2_4S = AARCH64_INS_ALIAS_UXTL2_4S,
	ARM64_INS_ALIAS_UXTL2_2D = AARCH64_INS_ALIAS_UXTL2_2D,
	ARM64_INS_ALIAS_LD2 = AARCH64_INS_ALIAS_LD2,
	ARM64_INS_ALIAS_LD3 = AARCH64_INS_ALIAS_LD3,
	ARM64_INS_ALIAS_LD4 = AARCH64_INS_ALIAS_LD4,
	ARM64_INS_ALIAS_ST1 = AARCH64_INS_ALIAS_ST1,
	ARM64_INS_ALIAS_ST2 = AARCH64_INS_ALIAS_ST2,
	ARM64_INS_ALIAS_ST3 = AARCH64_INS_ALIAS_ST3,
	ARM64_INS_ALIAS_ST4 = AARCH64_INS_ALIAS_ST4,
	ARM64_INS_ALIAS_LD2R = AARCH64_INS_ALIAS_LD2R,
	ARM64_INS_ALIAS_LD3R = AARCH64_INS_ALIAS_LD3R,
	ARM64_INS_ALIAS_LD4R = AARCH64_INS_ALIAS_LD4R,
	ARM64_INS_ALIAS_CLRBHB = AARCH64_INS_ALIAS_CLRBHB,
	ARM64_INS_ALIAS_STILP = AARCH64_INS_ALIAS_STILP,
	ARM64_INS_ALIAS_STL1 = AARCH64_INS_ALIAS_STL1,
	ARM64_INS_ALIAS_SYSP = AARCH64_INS_ALIAS_SYSP,
	ARM64_INS_ALIAS_LD1SW = AARCH64_INS_ALIAS_LD1SW,
	ARM64_INS_ALIAS_LD1H = AARCH64_INS_ALIAS_LD1H,
	ARM64_INS_ALIAS_LD1SH = AARCH64_INS_ALIAS_LD1SH,
	ARM64_INS_ALIAS_LD1D = AARCH64_INS_ALIAS_LD1D,
	ARM64_INS_ALIAS_LD1RSW = AARCH64_INS_ALIAS_LD1RSW,
	ARM64_INS_ALIAS_LD1RH = AARCH64_INS_ALIAS_LD1RH,
	ARM64_INS_ALIAS_LD1RSH = AARCH64_INS_ALIAS_LD1RSH,
	ARM64_INS_ALIAS_LD1RW = AARCH64_INS_ALIAS_LD1RW,
	ARM64_INS_ALIAS_LD1RSB = AARCH64_INS_ALIAS_LD1RSB,
	ARM64_INS_ALIAS_LD1RD = AARCH64_INS_ALIAS_LD1RD,
	ARM64_INS_ALIAS_LD1RQH = AARCH64_INS_ALIAS_LD1RQH,
	ARM64_INS_ALIAS_LD1RQW = AARCH64_INS_ALIAS_LD1RQW,
	ARM64_INS_ALIAS_LD1RQD = AARCH64_INS_ALIAS_LD1RQD,
	ARM64_INS_ALIAS_LDNF1SW = AARCH64_INS_ALIAS_LDNF1SW,
	ARM64_INS_ALIAS_LDNF1H = AARCH64_INS_ALIAS_LDNF1H,
	ARM64_INS_ALIAS_LDNF1SH = AARCH64_INS_ALIAS_LDNF1SH,
	ARM64_INS_ALIAS_LDNF1W = AARCH64_INS_ALIAS_LDNF1W,
	ARM64_INS_ALIAS_LDNF1SB = AARCH64_INS_ALIAS_LDNF1SB,
	ARM64_INS_ALIAS_LDNF1D = AARCH64_INS_ALIAS_LDNF1D,
	ARM64_INS_ALIAS_LDFF1SW = AARCH64_INS_ALIAS_LDFF1SW,
	ARM64_INS_ALIAS_LDFF1H = AARCH64_INS_ALIAS_LDFF1H,
	ARM64_INS_ALIAS_LDFF1SH = AARCH64_INS_ALIAS_LDFF1SH,
	ARM64_INS_ALIAS_LDFF1W = AARCH64_INS_ALIAS_LDFF1W,
	ARM64_INS_ALIAS_LDFF1SB = AARCH64_INS_ALIAS_LDFF1SB,
	ARM64_INS_ALIAS_LDFF1D = AARCH64_INS_ALIAS_LDFF1D,
	ARM64_INS_ALIAS_LD3B = AARCH64_INS_ALIAS_LD3B,
	ARM64_INS_ALIAS_LD4B = AARCH64_INS_ALIAS_LD4B,
	ARM64_INS_ALIAS_LD2H = AARCH64_INS_ALIAS_LD2H,
	ARM64_INS_ALIAS_LD3H = AARCH64_INS_ALIAS_LD3H,
	ARM64_INS_ALIAS_LD4H = AARCH64_INS_ALIAS_LD4H,
	ARM64_INS_ALIAS_LD2W = AARCH64_INS_ALIAS_LD2W,
	ARM64_INS_ALIAS_LD3W = AARCH64_INS_ALIAS_LD3W,
	ARM64_INS_ALIAS_LD4W = AARCH64_INS_ALIAS_LD4W,
	ARM64_INS_ALIAS_LD2D = AARCH64_INS_ALIAS_LD2D,
	ARM64_INS_ALIAS_LD3D = AARCH64_INS_ALIAS_LD3D,
	ARM64_INS_ALIAS_LD4D = AARCH64_INS_ALIAS_LD4D,
	ARM64_INS_ALIAS_LD2Q = AARCH64_INS_ALIAS_LD2Q,
	ARM64_INS_ALIAS_LD3Q = AARCH64_INS_ALIAS_LD3Q,
	ARM64_INS_ALIAS_LD4Q = AARCH64_INS_ALIAS_LD4Q,
	ARM64_INS_ALIAS_LDNT1H = AARCH64_INS_ALIAS_LDNT1H,
	ARM64_INS_ALIAS_LDNT1W = AARCH64_INS_ALIAS_LDNT1W,
	ARM64_INS_ALIAS_LDNT1D = AARCH64_INS_ALIAS_LDNT1D,
	ARM64_INS_ALIAS_ST1H = AARCH64_INS_ALIAS_ST1H,
	ARM64_INS_ALIAS_ST1W = AARCH64_INS_ALIAS_ST1W,
	ARM64_INS_ALIAS_ST1D = AARCH64_INS_ALIAS_ST1D,
	ARM64_INS_ALIAS_ST3B = AARCH64_INS_ALIAS_ST3B,
	ARM64_INS_ALIAS_ST4B = AARCH64_INS_ALIAS_ST4B,
	ARM64_INS_ALIAS_ST2H = AARCH64_INS_ALIAS_ST2H,
	ARM64_INS_ALIAS_ST3H = AARCH64_INS_ALIAS_ST3H,
	ARM64_INS_ALIAS_ST4H = AARCH64_INS_ALIAS_ST4H,
	ARM64_INS_ALIAS_ST2W = AARCH64_INS_ALIAS_ST2W,
	ARM64_INS_ALIAS_ST3W = AARCH64_INS_ALIAS_ST3W,
	ARM64_INS_ALIAS_ST4W = AARCH64_INS_ALIAS_ST4W,
	ARM64_INS_ALIAS_ST2D = AARCH64_INS_ALIAS_ST2D,
	ARM64_INS_ALIAS_ST3D = AARCH64_INS_ALIAS_ST3D,
	ARM64_INS_ALIAS_ST4D = AARCH64_INS_ALIAS_ST4D,
	ARM64_INS_ALIAS_ST3Q = AARCH64_INS_ALIAS_ST3Q,
	ARM64_INS_ALIAS_ST4Q = AARCH64_INS_ALIAS_ST4Q,
	ARM64_INS_ALIAS_STNT1H = AARCH64_INS_ALIAS_STNT1H,
	ARM64_INS_ALIAS_STNT1W = AARCH64_INS_ALIAS_STNT1W,
	ARM64_INS_ALIAS_STNT1D = AARCH64_INS_ALIAS_STNT1D,
	ARM64_INS_ALIAS_PRFH = AARCH64_INS_ALIAS_PRFH,
	ARM64_INS_ALIAS_PRFW = AARCH64_INS_ALIAS_PRFW,
	ARM64_INS_ALIAS_PRFD = AARCH64_INS_ALIAS_PRFD,
	ARM64_INS_ALIAS_CNTH = AARCH64_INS_ALIAS_CNTH,
	ARM64_INS_ALIAS_CNTW = AARCH64_INS_ALIAS_CNTW,
	ARM64_INS_ALIAS_CNTD = AARCH64_INS_ALIAS_CNTD,
	ARM64_INS_ALIAS_DECB = AARCH64_INS_ALIAS_DECB,
	ARM64_INS_ALIAS_INCH = AARCH64_INS_ALIAS_INCH,
	ARM64_INS_ALIAS_DECH = AARCH64_INS_ALIAS_DECH,
	ARM64_INS_ALIAS_INCW = AARCH64_INS_ALIAS_INCW,
	ARM64_INS_ALIAS_DECW = AARCH64_INS_ALIAS_DECW,
	ARM64_INS_ALIAS_INCD = AARCH64_INS_ALIAS_INCD,
	ARM64_INS_ALIAS_DECD = AARCH64_INS_ALIAS_DECD,
	ARM64_INS_ALIAS_SQDECB = AARCH64_INS_ALIAS_SQDECB,
	ARM64_INS_ALIAS_UQDECB = AARCH64_INS_ALIAS_UQDECB,
	ARM64_INS_ALIAS_UQINCH = AARCH64_INS_ALIAS_UQINCH,
	ARM64_INS_ALIAS_SQDECH = AARCH64_INS_ALIAS_SQDECH,
	ARM64_INS_ALIAS_UQDECH = AARCH64_INS_ALIAS_UQDECH,
	ARM64_INS_ALIAS_SQINCW = AARCH64_INS_ALIAS_SQINCW,
	ARM64_INS_ALIAS_UQINCW = AARCH64_INS_ALIAS_UQINCW,
	ARM64_INS_ALIAS_SQDECW = AARCH64_INS_ALIAS_SQDECW,
	ARM64_INS_ALIAS_UQDECW = AARCH64_INS_ALIAS_UQDECW,
	ARM64_INS_ALIAS_SQINCD = AARCH64_INS_ALIAS_SQINCD,
	ARM64_INS_ALIAS_UQINCD = AARCH64_INS_ALIAS_UQINCD,
	ARM64_INS_ALIAS_SQDECD = AARCH64_INS_ALIAS_SQDECD,
	ARM64_INS_ALIAS_UQDECD = AARCH64_INS_ALIAS_UQDECD,
	ARM64_INS_ALIAS_MOVS = AARCH64_INS_ALIAS_MOVS,
	ARM64_INS_ALIAS_NOT = AARCH64_INS_ALIAS_NOT,
	ARM64_INS_ALIAS_NOTS = AARCH64_INS_ALIAS_NOTS,
	ARM64_INS_ALIAS_LD1ROH = AARCH64_INS_ALIAS_LD1ROH,
	ARM64_INS_ALIAS_LD1ROW = AARCH64_INS_ALIAS_LD1ROW,
	ARM64_INS_ALIAS_LD1ROD = AARCH64_INS_ALIAS_LD1ROD,
	ARM64_INS_ALIAS_BCAX = AARCH64_INS_ALIAS_BCAX,
	ARM64_INS_ALIAS_BSL = AARCH64_INS_ALIAS_BSL,
	ARM64_INS_ALIAS_BSL1N = AARCH64_INS_ALIAS_BSL1N,
	ARM64_INS_ALIAS_BSL2N = AARCH64_INS_ALIAS_BSL2N,
	ARM64_INS_ALIAS_NBSL = AARCH64_INS_ALIAS_NBSL,
	ARM64_INS_ALIAS_LDNT1SH = AARCH64_INS_ALIAS_LDNT1SH,
	ARM64_INS_ALIAS_LDNT1SW = AARCH64_INS_ALIAS_LDNT1SW,


	ARM64_INS_ALIAS_CFP = AARCH64_INS_ALIAS_CFP,
	ARM64_INS_ALIAS_DVP = AARCH64_INS_ALIAS_DVP,
	ARM64_INS_ALIAS_COSP = AARCH64_INS_ALIAS_COSP,
	ARM64_INS_ALIAS_CPP = AARCH64_INS_ALIAS_CPP,
	ARM64_INS_ALIAS_IC = AARCH64_INS_ALIAS_IC,
	ARM64_INS_ALIAS_DC = AARCH64_INS_ALIAS_DC,
	ARM64_INS_ALIAS_AT = AARCH64_INS_ALIAS_AT,
	ARM64_INS_ALIAS_TLBI = AARCH64_INS_ALIAS_TLBI,
	ARM64_INS_ALIAS_TLBIP = AARCH64_INS_ALIAS_TLBIP,
	ARM64_INS_ALIAS_RPRFM = AARCH64_INS_ALIAS_RPRFM,
	ARM64_INS_ALIAS_LSL = AARCH64_INS_ALIAS_LSL,
	ARM64_INS_ALIAS_SBFX = AARCH64_INS_ALIAS_SBFX,
	ARM64_INS_ALIAS_UBFX = AARCH64_INS_ALIAS_UBFX,
	ARM64_INS_ALIAS_SBFIZ = AARCH64_INS_ALIAS_SBFIZ,
	ARM64_INS_ALIAS_UBFIZ = AARCH64_INS_ALIAS_UBFIZ,
	ARM64_INS_ALIAS_BFC = AARCH64_INS_ALIAS_BFC,
	ARM64_INS_ALIAS_BFI = AARCH64_INS_ALIAS_BFI,
	ARM64_INS_ALIAS_BFXIL = AARCH64_INS_ALIAS_BFXIL,

	ARM64_INS_ALIAS_END = AARCH64_INS_ALIAS_END,
} arm64_insn;

typedef enum {
  ARM64_GRP_INVALID = AARCH64_GRP_INVALID,

  ARM64_GRP_JUMP = AARCH64_GRP_JUMP,
  ARM64_GRP_CALL = AARCH64_GRP_CALL,
  ARM64_GRP_RET = AARCH64_GRP_RET,
  ARM64_GRP_INT = AARCH64_GRP_INT,
  ARM64_GRP_PRIVILEGE = AARCH64_GRP_PRIVILEGE,
  ARM64_GRP_BRANCH_RELATIVE = AARCH64_GRP_BRANCH_RELATIVE,

	ARM64_FEATURE_HASV8_0A = AARCH64_FEATURE_HASV8_0A,
	ARM64_FEATURE_HASV8_1A = AARCH64_FEATURE_HASV8_1A,
	ARM64_FEATURE_HASV8_2A = AARCH64_FEATURE_HASV8_2A,
	ARM64_FEATURE_HASV8_3A = AARCH64_FEATURE_HASV8_3A,
	ARM64_FEATURE_HASV8_4A = AARCH64_FEATURE_HASV8_4A,
	ARM64_FEATURE_HASV8_5A = AARCH64_FEATURE_HASV8_5A,
	ARM64_FEATURE_HASV8_6A = AARCH64_FEATURE_HASV8_6A,
	ARM64_FEATURE_HASV8_7A = AARCH64_FEATURE_HASV8_7A,
	ARM64_FEATURE_HASV8_8A = AARCH64_FEATURE_HASV8_8A,
	ARM64_FEATURE_HASV8_9A = AARCH64_FEATURE_HASV8_9A,
	ARM64_FEATURE_HASV9_0A = AARCH64_FEATURE_HASV9_0A,
	ARM64_FEATURE_HASV9_1A = AARCH64_FEATURE_HASV9_1A,
	ARM64_FEATURE_HASV9_2A = AARCH64_FEATURE_HASV9_2A,
	ARM64_FEATURE_HASV9_3A = AARCH64_FEATURE_HASV9_3A,
	ARM64_FEATURE_HASV9_4A = AARCH64_FEATURE_HASV9_4A,
	ARM64_FEATURE_HASV8_0R = AARCH64_FEATURE_HASV8_0R,
	ARM64_FEATURE_HASEL2VMSA = AARCH64_FEATURE_HASEL2VMSA,
	ARM64_FEATURE_HASEL3 = AARCH64_FEATURE_HASEL3,
	ARM64_FEATURE_HASVH = AARCH64_FEATURE_HASVH,
	ARM64_FEATURE_HASLOR = AARCH64_FEATURE_HASLOR,
	ARM64_FEATURE_HASPAUTH = AARCH64_FEATURE_HASPAUTH,
	ARM64_FEATURE_HASPAUTHLR = AARCH64_FEATURE_HASPAUTHLR,
	ARM64_FEATURE_HASJS = AARCH64_FEATURE_HASJS,
	ARM64_FEATURE_HASCCIDX = AARCH64_FEATURE_HASCCIDX,
	ARM64_FEATURE_HASCOMPLXNUM = AARCH64_FEATURE_HASCOMPLXNUM,
	ARM64_FEATURE_HASNV = AARCH64_FEATURE_HASNV,
	ARM64_FEATURE_HASMPAM = AARCH64_FEATURE_HASMPAM,
	ARM64_FEATURE_HASDIT = AARCH64_FEATURE_HASDIT,
	ARM64_FEATURE_HASTRACEV8_4 = AARCH64_FEATURE_HASTRACEV8_4,
	ARM64_FEATURE_HASAM = AARCH64_FEATURE_HASAM,
	ARM64_FEATURE_HASSEL2 = AARCH64_FEATURE_HASSEL2,
	ARM64_FEATURE_HASTLB_RMI = AARCH64_FEATURE_HASTLB_RMI,
	ARM64_FEATURE_HASFLAGM = AARCH64_FEATURE_HASFLAGM,
	ARM64_FEATURE_HASRCPC_IMMO = AARCH64_FEATURE_HASRCPC_IMMO,
	ARM64_FEATURE_HASFPARMV8 = AARCH64_FEATURE_HASFPARMV8,
	ARM64_FEATURE_HASNEON = AARCH64_FEATURE_HASNEON,
	ARM64_FEATURE_HASSM4 = AARCH64_FEATURE_HASSM4,
	ARM64_FEATURE_HASSHA3 = AARCH64_FEATURE_HASSHA3,
	ARM64_FEATURE_HASSHA2 = AARCH64_FEATURE_HASSHA2,
	ARM64_FEATURE_HASAES = AARCH64_FEATURE_HASAES,
	ARM64_FEATURE_HASDOTPROD = AARCH64_FEATURE_HASDOTPROD,
	ARM64_FEATURE_HASCRC = AARCH64_FEATURE_HASCRC,
	ARM64_FEATURE_HASCSSC = AARCH64_FEATURE_HASCSSC,
	ARM64_FEATURE_HASLSE = AARCH64_FEATURE_HASLSE,
	ARM64_FEATURE_HASRAS = AARCH64_FEATURE_HASRAS,
	ARM64_FEATURE_HASRDM = AARCH64_FEATURE_HASRDM,
	ARM64_FEATURE_HASFULLFP16 = AARCH64_FEATURE_HASFULLFP16,
	ARM64_FEATURE_HASFP16FML = AARCH64_FEATURE_HASFP16FML,
	ARM64_FEATURE_HASSPE = AARCH64_FEATURE_HASSPE,
	ARM64_FEATURE_HASFUSEAES = AARCH64_FEATURE_HASFUSEAES,
	ARM64_FEATURE_HASSVE = AARCH64_FEATURE_HASSVE,
	ARM64_FEATURE_HASSVE2 = AARCH64_FEATURE_HASSVE2,
	ARM64_FEATURE_HASSVE2P1 = AARCH64_FEATURE_HASSVE2P1,
	ARM64_FEATURE_HASSVE2AES = AARCH64_FEATURE_HASSVE2AES,
	ARM64_FEATURE_HASSVE2SM4 = AARCH64_FEATURE_HASSVE2SM4,
	ARM64_FEATURE_HASSVE2SHA3 = AARCH64_FEATURE_HASSVE2SHA3,
	ARM64_FEATURE_HASSVE2BITPERM = AARCH64_FEATURE_HASSVE2BITPERM,
	ARM64_FEATURE_HASB16B16 = AARCH64_FEATURE_HASB16B16,
	ARM64_FEATURE_HASSME = AARCH64_FEATURE_HASSME,
	ARM64_FEATURE_HASSMEF64F64 = AARCH64_FEATURE_HASSMEF64F64,
	ARM64_FEATURE_HASSMEF16F16 = AARCH64_FEATURE_HASSMEF16F16,
	ARM64_FEATURE_HASSMEFA64 = AARCH64_FEATURE_HASSMEFA64,
	ARM64_FEATURE_HASSMEI16I64 = AARCH64_FEATURE_HASSMEI16I64,
	ARM64_FEATURE_HASSME2 = AARCH64_FEATURE_HASSME2,
	ARM64_FEATURE_HASSME2P1 = AARCH64_FEATURE_HASSME2P1,
	ARM64_FEATURE_HASFPMR = AARCH64_FEATURE_HASFPMR,
	ARM64_FEATURE_HASFP8 = AARCH64_FEATURE_HASFP8,
	ARM64_FEATURE_HASFAMINMAX = AARCH64_FEATURE_HASFAMINMAX,
	ARM64_FEATURE_HASFP8FMA = AARCH64_FEATURE_HASFP8FMA,
	ARM64_FEATURE_HASSSVE_FP8FMA = AARCH64_FEATURE_HASSSVE_FP8FMA,
	ARM64_FEATURE_HASFP8DOT2 = AARCH64_FEATURE_HASFP8DOT2,
	ARM64_FEATURE_HASSSVE_FP8DOT2 = AARCH64_FEATURE_HASSSVE_FP8DOT2,
	ARM64_FEATURE_HASFP8DOT4 = AARCH64_FEATURE_HASFP8DOT4,
	ARM64_FEATURE_HASSSVE_FP8DOT4 = AARCH64_FEATURE_HASSSVE_FP8DOT4,
	ARM64_FEATURE_HASLUT = AARCH64_FEATURE_HASLUT,
	ARM64_FEATURE_HASSME_LUTV2 = AARCH64_FEATURE_HASSME_LUTV2,
	ARM64_FEATURE_HASSMEF8F16 = AARCH64_FEATURE_HASSMEF8F16,
	ARM64_FEATURE_HASSMEF8F32 = AARCH64_FEATURE_HASSMEF8F32,
	ARM64_FEATURE_HASSVEORSME = AARCH64_FEATURE_HASSVEORSME,
	ARM64_FEATURE_HASSVE2ORSME = AARCH64_FEATURE_HASSVE2ORSME,
	ARM64_FEATURE_HASSVE2ORSME2 = AARCH64_FEATURE_HASSVE2ORSME2,
	ARM64_FEATURE_HASSVE2P1_OR_HASSME = AARCH64_FEATURE_HASSVE2P1_OR_HASSME,
	ARM64_FEATURE_HASSVE2P1_OR_HASSME2 = AARCH64_FEATURE_HASSVE2P1_OR_HASSME2,
	ARM64_FEATURE_HASSVE2P1_OR_HASSME2P1 = AARCH64_FEATURE_HASSVE2P1_OR_HASSME2P1,
	ARM64_FEATURE_HASNEONORSME = AARCH64_FEATURE_HASNEONORSME,
	ARM64_FEATURE_HASRCPC = AARCH64_FEATURE_HASRCPC,
	ARM64_FEATURE_HASALTNZCV = AARCH64_FEATURE_HASALTNZCV,
	ARM64_FEATURE_HASFRINT3264 = AARCH64_FEATURE_HASFRINT3264,
	ARM64_FEATURE_HASSB = AARCH64_FEATURE_HASSB,
	ARM64_FEATURE_HASPREDRES = AARCH64_FEATURE_HASPREDRES,
	ARM64_FEATURE_HASCCDP = AARCH64_FEATURE_HASCCDP,
	ARM64_FEATURE_HASBTI = AARCH64_FEATURE_HASBTI,
	ARM64_FEATURE_HASMTE = AARCH64_FEATURE_HASMTE,
	ARM64_FEATURE_HASTME = AARCH64_FEATURE_HASTME,
	ARM64_FEATURE_HASETE = AARCH64_FEATURE_HASETE,
	ARM64_FEATURE_HASTRBE = AARCH64_FEATURE_HASTRBE,
	ARM64_FEATURE_HASBF16 = AARCH64_FEATURE_HASBF16,
	ARM64_FEATURE_HASMATMULINT8 = AARCH64_FEATURE_HASMATMULINT8,
	ARM64_FEATURE_HASMATMULFP32 = AARCH64_FEATURE_HASMATMULFP32,
	ARM64_FEATURE_HASMATMULFP64 = AARCH64_FEATURE_HASMATMULFP64,
	ARM64_FEATURE_HASXS = AARCH64_FEATURE_HASXS,
	ARM64_FEATURE_HASWFXT = AARCH64_FEATURE_HASWFXT,
	ARM64_FEATURE_HASLS64 = AARCH64_FEATURE_HASLS64,
	ARM64_FEATURE_HASBRBE = AARCH64_FEATURE_HASBRBE,
	ARM64_FEATURE_HASSPE_EEF = AARCH64_FEATURE_HASSPE_EEF,
	ARM64_FEATURE_HASHBC = AARCH64_FEATURE_HASHBC,
	ARM64_FEATURE_HASMOPS = AARCH64_FEATURE_HASMOPS,
	ARM64_FEATURE_HASCLRBHB = AARCH64_FEATURE_HASCLRBHB,
	ARM64_FEATURE_HASSPECRES2 = AARCH64_FEATURE_HASSPECRES2,
	ARM64_FEATURE_HASITE = AARCH64_FEATURE_HASITE,
	ARM64_FEATURE_HASTHE = AARCH64_FEATURE_HASTHE,
	ARM64_FEATURE_HASRCPC3 = AARCH64_FEATURE_HASRCPC3,
	ARM64_FEATURE_HASLSE128 = AARCH64_FEATURE_HASLSE128,
	ARM64_FEATURE_HASD128 = AARCH64_FEATURE_HASD128,
	ARM64_FEATURE_HASCHK = AARCH64_FEATURE_HASCHK,
	ARM64_FEATURE_HASGCS = AARCH64_FEATURE_HASGCS,
	ARM64_FEATURE_HASCPA = AARCH64_FEATURE_HASCPA,
	ARM64_FEATURE_USENEGATIVEIMMEDIATES = AARCH64_FEATURE_USENEGATIVEIMMEDIATES,
	ARM64_FEATURE_HASCCPP = AARCH64_FEATURE_HASCCPP,
	ARM64_FEATURE_HASPAN = AARCH64_FEATURE_HASPAN,
	ARM64_FEATURE_HASPSUAO = AARCH64_FEATURE_HASPSUAO,
	ARM64_FEATURE_HASPAN_RWV = AARCH64_FEATURE_HASPAN_RWV,
	ARM64_FEATURE_HASCONTEXTIDREL2 = AARCH64_FEATURE_HASCONTEXTIDREL2,


  ARM64_GRP_ENDING = AARCH64_GRP_ENDING,
} arm64_insn_group;

#ifdef __cplusplus
}
#endif

#endif

#define arm64_cc AArch64CC_CondCode
#define ARM64_CC_EQ AArch64CC_EQ
#define ARM64_CC_NE AArch64CC_NE
#define ARM64_CC_HS AArch64CC_HS
#define ARM64_CC_LO AArch64CC_LO
#define ARM64_CC_MI AArch64CC_MI
#define ARM64_CC_PL AArch64CC_PL
#define ARM64_CC_VS AArch64CC_VS
#define ARM64_CC_VC AArch64CC_VC
#define ARM64_CC_HI AArch64CC_HI
#define ARM64_CC_LS AArch64CC_LS
#define ARM64_CC_GE AArch64CC_GE
#define ARM64_CC_LT AArch64CC_LT
#define ARM64_CC_GT AArch64CC_GT
#define ARM64_CC_LE AArch64CC_LE
#define ARM64_CC_AL AArch64CC_AL
#define ARM64_CC_INVALID AArch64CC_Invalid
#define ARM64_VAS_INVALID AARCH64LAYOUT_VL_INVALID
#define ARM64_VAS_16B AARCH64LAYOUT_VL_16B
#define ARM64_VAS_8B AARCH64LAYOUT_VL_8B
#define ARM64_VAS_4B AARCH64LAYOUT_VL_4B
#define ARM64_VAS_1B AARCH64LAYOUT_VL_1B
#define ARM64_VAS_8H AARCH64LAYOUT_VL_8H
#define ARM64_VAS_4H AARCH64LAYOUT_VL_4H
#define ARM64_VAS_2H AARCH64LAYOUT_VL_2H
#define ARM64_VAS_1H AARCH64LAYOUT_VL_1H
#define ARM64_VAS_4S AARCH64LAYOUT_VL_4S
#define ARM64_VAS_2S AARCH64LAYOUT_VL_2S
#define ARM64_VAS_1S AARCH64LAYOUT_VL_1S
#define ARM64_VAS_2D AARCH64LAYOUT_VL_2D
#define ARM64_VAS_1D AARCH64LAYOUT_VL_1D
#define ARM64_VAS_1Q AARCH64LAYOUT_VL_1Q
#define arm64_vas AArch64Layout_VectorLayout
