#ifndef CAPSTONE_PPC_H
#define CAPSTONE_PPC_H

/* Capstone Disassembly Engine */
/* By Nguyen Anh Quynh <aquynh@gmail.com>, 2013-2015 */

#ifdef __cplusplus
extern "C" {
#endif

#include <assert.h>
#include "cs_operand.h"
#include "platform.h"

#ifdef _MSC_VER
#pragma warning(disable : 4201)
#endif

/// Enum was moved from PPCPredicates.h so we do not have duplicates.
///
/// Branch predicate enum. It contains the CR predicates and CTR predicates.
///
/// Enum values are "((BI % 4) << 5) | BO"  for various predicates.
///
/// CR field encoding:
///
/// Bit:    |   0   |    1    |   2   |     3    |
///         |-------|---------|-------|----------|
/// Meaning | Less  | Greater | Zero  | Summary  |
///         | Then  | Then    |       | Overflow |
///
/// BO encoding
///
/// Bit     |   0    |     1       |   2   |      3     |     4      |
///         |--------|-------------|-------|------------|------------|
/// If      | Test   | Test        | Decr. | test       |            |
/// unset:  | CR(BI) | CR(BI) == 0 | CTR   | CTR != 0   |            |
///         |--------|-------------|-------|------------|------------|
/// If      | Don't  | Test        | Don't | test       |            |
/// set:    | Test   | CR(BI) == 1 | decr. | CTR == 0   |            |
///         | CR(BI) |             | CTR   |            |            |
///         |--------|-------------|-------|------------|------------|
/// Alter-  |        | Hint bit:   |       | Hint bit:  | Hint bit:  |
/// native  | None   |   a         | None  |    a       |    t       |
/// meaning |        | or ignored  |       | or ignored | or ignored |
///
/// NOTE: If we do not decrement the counter, it is not used for the condition.
///
/// The bits "at" are both present if:
/// 		- CTR is decremented, but CR is not checked.
///     - CR is checked, but CTR is not decremented.
typedef enum ppc_pred {
	// Technically this could be read as a valid predicate
	// But the ISA recommends to set the z bits to 0,
	// so it shouldn't come to conflicts.
	PPC_PRED_INVALID = 0xffff,

	// Name       | BI     | BO
	PPC_PRED_LT = (0 << 5) | 12,
	PPC_PRED_LE = (1 << 5) | 4,
	PPC_PRED_EQ = (2 << 5) | 12,
	PPC_PRED_GE = (0 << 5) | 4,
	PPC_PRED_GT = (1 << 5) | 12,
	PPC_PRED_NE = (2 << 5) | 4,
	PPC_PRED_UN = (3 << 5) | 12, ///< Unordered (after fp comparison)
	PPC_PRED_NU = (3 << 5) | 4,  ///< Not Unordered (after fp comparison)
	PPC_PRED_SO = (3 << 5) | 12, ///< summary overflow
	PPC_PRED_NS = (3 << 5) | 4,  ///< not summary overflow

	/// CTR predicates
	PPC_PRED_NZ = (0 << 5) | 16,
	PPC_PRED_Z = (0 << 5) | 18,
	// Likely not taken
	PPC_PRED_LT_MINUS = (0 << 5) | 14,
	PPC_PRED_LE_MINUS = (1 << 5) | 6,
	PPC_PRED_EQ_MINUS = (2 << 5) | 14,
	PPC_PRED_GE_MINUS = (0 << 5) | 6,
	PPC_PRED_GT_MINUS = (1 << 5) | 14,
	PPC_PRED_NE_MINUS = (2 << 5) | 6,
	PPC_PRED_UN_MINUS = (3 << 5) | 14,
	PPC_PRED_NU_MINUS = (3 << 5) | 6,
	PPC_PRED_NZ_MINUS = (0 << 5) | 24,
	PPC_PRED_Z_MINUS = (0 << 5) | 26,
	// Likely taken
	PPC_PRED_LT_PLUS = (0 << 5) | 15,
	PPC_PRED_LE_PLUS = (1 << 5) | 7,
	PPC_PRED_EQ_PLUS = (2 << 5) | 15,
	PPC_PRED_GE_PLUS = (0 << 5) | 7,
	PPC_PRED_GT_PLUS = (1 << 5) | 15,
	PPC_PRED_NE_PLUS = (2 << 5) | 7,
	PPC_PRED_UN_PLUS = (3 << 5) | 15,
	PPC_PRED_NU_PLUS = (3 << 5) | 7,
	PPC_PRED_NZ_PLUS = (0 << 5) | 25,
	PPC_PRED_Z_PLUS = (0 << 5) | 27,
	// Reserved
	PPC_PRED_LT_RESERVED = (0 << 5) | 13,
	PPC_PRED_LE_RESERVED = (1 << 5) | 5,
	PPC_PRED_EQ_RESERVED = (2 << 5) | 13,
	PPC_PRED_GE_RESERVED = (0 << 5) | 5,
	PPC_PRED_GT_RESERVED = (1 << 5) | 13,
	PPC_PRED_NE_RESERVED = (2 << 5) | 5,
	PPC_PRED_UN_RESERVED = (3 << 5) | 13,
	PPC_PRED_NU_RESERVED = (3 << 5) | 5,
	PPC_PRED_NZ_RESERVED = (0 << 5) | 17,
	PPC_PRED_Z_RESERVED = (0 << 5) | 19,

	// SPE scalar compare instructions always set the GT bit.
	PPC_PRED_SPE = PPC_PRED_GT,

	// When dealing with individual condition-register bits, we have simple set
	// and unset predicates.
	PPC_PRED_BIT_SET = 1024,
	PPC_PRED_BIT_UNSET = 1025
} ppc_pred;

/// CR field indices and their meaning.
typedef enum {
	PPC_BI_LT = 0,	       ///< CR bit Less Than
	PPC_BI_GT = 1,	       ///< CR bit Greater Than
	PPC_BI_Z = 2,	       ///< CR bit Zero
	PPC_BI_SO = 3,	       ///< CR bit Summary Overflow
	PPC_BI_INVALID = 0xff, ///< CR bit was not set/invalid
} ppc_cr_bit;

/// Masks of flags in the BO field.
typedef enum {
	PPC_BO_TEST_CR = (1 << 4),  ///< Flag mask: Test CR bit.
	PPC_BO_CR_CMP = (1 << 3),   ///< Flag mask: Compare CR bit to 0 or 1.
	PPC_BO_DECR_CTR = (1 << 2), ///< Flag mask: Decrement counter.
	PPC_BO_CTR_CMP = (1 << 1),  ///< Flag mask: Compare CTR to 0 or 1.
	PPC_BO_T = 1,               ///< Either ignored (z) or hint bit t
} ppc_bo_mask;

/// Bit for branch taken (plus) or not-taken (minus) hint
/// Encodes the meaning of the branch hint bits.
/// Bit:  | 0 | 1 |
/// Name: | a | t |
typedef enum {
	PPC_BR_NOT_GIVEN = 0x0,
	PPC_BR_RESERVED = 0x1,
	PPC_BR_NOT_TAKEN = 0x2, ///< Minus
	PPC_BR_TAKEN = 0x3,     ///< Plus
	PPC_BR_HINT_MASK = 0x3,
} ppc_br_hint;

/// Encodes the different meanings of the BH field.
/// The enum values does NOT match the BH field values!
typedef enum {
	PPC_BH_INVALID = 0,
	PPC_BH_SUBROUTINE_RET,
	PPC_BH_NO_SUBROUTINE_RET,
	PPC_BH_NOT_PREDICTABLE,
	PPC_BH_RESERVED,
} ppc_bh;


/// Returns the predicate without branch hint information.
inline static ppc_pred PPC_get_no_hint_pred(ppc_pred Code)
{
	switch (Code) {
	default:
		return PPC_PRED_INVALID;
	case PPC_PRED_LT:
	case PPC_PRED_LT_MINUS:
	case PPC_PRED_LT_PLUS:
	case PPC_PRED_LT_RESERVED:
		return PPC_PRED_LT;
	case PPC_PRED_LE:
	case PPC_PRED_LE_MINUS:
	case PPC_PRED_LE_PLUS:
	case PPC_PRED_LE_RESERVED:
		return PPC_PRED_LE;
	case PPC_PRED_EQ:
	case PPC_PRED_EQ_MINUS:
	case PPC_PRED_EQ_PLUS:
	case PPC_PRED_EQ_RESERVED:
		return PPC_PRED_EQ;
	case PPC_PRED_GE:
	case PPC_PRED_GE_MINUS:
	case PPC_PRED_GE_PLUS:
	case PPC_PRED_GE_RESERVED:
		return PPC_PRED_GE;
	case PPC_PRED_GT:
	case PPC_PRED_GT_MINUS:
	case PPC_PRED_GT_PLUS:
	case PPC_PRED_GT_RESERVED:
		return PPC_PRED_GT;
	case PPC_PRED_NE:
	case PPC_PRED_NE_MINUS:
	case PPC_PRED_NE_PLUS:
	case PPC_PRED_NE_RESERVED:
		return PPC_PRED_NE;
	case PPC_PRED_UN:
	case PPC_PRED_UN_MINUS:
	case PPC_PRED_UN_PLUS:
	case PPC_PRED_UN_RESERVED:
		return PPC_PRED_UN;
	case PPC_PRED_NU:
	case PPC_PRED_NU_MINUS:
	case PPC_PRED_NU_PLUS:
	case PPC_PRED_NU_RESERVED:
		return PPC_PRED_NU;
	case PPC_PRED_NZ:
	case PPC_PRED_NZ_MINUS:
	case PPC_PRED_NZ_PLUS:
	case PPC_PRED_NZ_RESERVED:
		return PPC_PRED_NZ;
	case PPC_PRED_Z:
	case PPC_PRED_Z_MINUS:
	case PPC_PRED_Z_PLUS:
	case PPC_PRED_Z_RESERVED:
		return PPC_PRED_Z;
	case PPC_PRED_BIT_SET:
		return PPC_PRED_BIT_SET;
	case PPC_PRED_BIT_UNSET:
		return PPC_PRED_BIT_UNSET;
	}
	return PPC_PRED_INVALID;
}

/// Returns the hint encoded in the BO bits a and t.
static inline ppc_br_hint PPC_get_hint(uint8_t bo)
{
	bool DecrCTR = (bo & PPC_BO_DECR_CTR) == 0;
	bool TestCR = (bo & PPC_BO_TEST_CR) == 0;
	if (!DecrCTR && !TestCR)
		return PPC_BR_NOT_GIVEN;
	else if (DecrCTR && !TestCR)
		return (ppc_br_hint)(((bo & PPC_BO_CR_CMP) >> 2) | (bo & PPC_BO_T));
	else if (!DecrCTR && TestCR)
		return (ppc_br_hint)((bo & PPC_BO_CTR_CMP) | (bo & PPC_BO_T));
	return PPC_BR_NOT_GIVEN;
}

/// Returns the branch predicate encoded in the BO and BI field.
/// If get_cr_pred = true the CR-bit predicate is returned (LE, GE, EQ...).
/// Otherwise the CTR predicate (NZ, Z)
/// The branch hint does not include the hint of the 'at' bits.
///
/// It returns PPC_PRED_INVALID if the CR predicate is requested, but no
/// CR predicate is encoded in BI and BO. Same for the CTR predicate.
static inline ppc_pred PPC_get_branch_pred(uint8_t bi, uint8_t bo,
					   bool get_cr_pred)
{
	bool TestCR = ((bo & PPC_BO_TEST_CR) == 0);
	bool DecrCTR = ((bo & PPC_BO_DECR_CTR) == 0);

	if ((get_cr_pred && !TestCR) || (!get_cr_pred && !DecrCTR))
		return PPC_PRED_INVALID;

	if (TestCR && DecrCTR) {
		// The CR-bit condition without the CTR condition.
		unsigned cr_bo_cond = (bo | PPC_BO_DECR_CTR) & ~PPC_BO_CTR_CMP;
		// The CTR condition without the CR-bit condition.
		unsigned ctr_bo_cond = (bo | PPC_BO_TEST_CR) & ~PPC_BO_CR_CMP;
		if (get_cr_pred)
			return PPC_get_no_hint_pred((ppc_pred)(((bi % 4) << 5) | cr_bo_cond));
		return PPC_get_no_hint_pred((ppc_pred)ctr_bo_cond); // BI is ignored
	}
	// BO doesn't need any separation
	return PPC_get_no_hint_pred((ppc_pred)(((bi % 4) << 5) | bo));
}

/// Operand type for instruction's operands
typedef enum ppc_op_type {
	PPC_OP_INVALID = CS_OP_INVALID, ///< Uninitialized.
	PPC_OP_REG = CS_OP_REG,		///< Register operand.
	PPC_OP_IMM = CS_OP_IMM,		///< Immediate operand.
	PPC_OP_MEM = CS_OP_MEM,		///< Memory operand.
} ppc_op_type;

/// PPC registers
typedef enum ppc_reg {
	// generated content <PPCGenCSRegEnum.inc> begin
	// clang-format off

	PPC_REG_INVALID = 0,
	PPC_REG_BP = 1,
	PPC_REG_CARRY = 2,
	PPC_REG_CTR = 3,
	PPC_REG_FP = 4,
	PPC_REG_LR = 5,
	PPC_REG_RM = 6,
	PPC_REG_SPEFSCR = 7,
	PPC_REG_VRSAVE = 8,
	PPC_REG_XER = 9,
	PPC_REG_ZERO = 10,
	PPC_REG_ACC0 = 11,
	PPC_REG_ACC1 = 12,
	PPC_REG_ACC2 = 13,
	PPC_REG_ACC3 = 14,
	PPC_REG_ACC4 = 15,
	PPC_REG_ACC5 = 16,
	PPC_REG_ACC6 = 17,
	PPC_REG_ACC7 = 18,
	PPC_REG_BP8 = 19,
	PPC_REG_CR0 = 20,
	PPC_REG_CR1 = 21,
	PPC_REG_CR2 = 22,
	PPC_REG_CR3 = 23,
	PPC_REG_CR4 = 24,
	PPC_REG_CR5 = 25,
	PPC_REG_CR6 = 26,
	PPC_REG_CR7 = 27,
	PPC_REG_CTR8 = 28,
	PPC_REG_DMR0 = 29,
	PPC_REG_DMR1 = 30,
	PPC_REG_DMR2 = 31,
	PPC_REG_DMR3 = 32,
	PPC_REG_DMR4 = 33,
	PPC_REG_DMR5 = 34,
	PPC_REG_DMR6 = 35,
	PPC_REG_DMR7 = 36,
	PPC_REG_DMRROW0 = 37,
	PPC_REG_DMRROW1 = 38,
	PPC_REG_DMRROW2 = 39,
	PPC_REG_DMRROW3 = 40,
	PPC_REG_DMRROW4 = 41,
	PPC_REG_DMRROW5 = 42,
	PPC_REG_DMRROW6 = 43,
	PPC_REG_DMRROW7 = 44,
	PPC_REG_DMRROW8 = 45,
	PPC_REG_DMRROW9 = 46,
	PPC_REG_DMRROW10 = 47,
	PPC_REG_DMRROW11 = 48,
	PPC_REG_DMRROW12 = 49,
	PPC_REG_DMRROW13 = 50,
	PPC_REG_DMRROW14 = 51,
	PPC_REG_DMRROW15 = 52,
	PPC_REG_DMRROW16 = 53,
	PPC_REG_DMRROW17 = 54,
	PPC_REG_DMRROW18 = 55,
	PPC_REG_DMRROW19 = 56,
	PPC_REG_DMRROW20 = 57,
	PPC_REG_DMRROW21 = 58,
	PPC_REG_DMRROW22 = 59,
	PPC_REG_DMRROW23 = 60,
	PPC_REG_DMRROW24 = 61,
	PPC_REG_DMRROW25 = 62,
	PPC_REG_DMRROW26 = 63,
	PPC_REG_DMRROW27 = 64,
	PPC_REG_DMRROW28 = 65,
	PPC_REG_DMRROW29 = 66,
	PPC_REG_DMRROW30 = 67,
	PPC_REG_DMRROW31 = 68,
	PPC_REG_DMRROW32 = 69,
	PPC_REG_DMRROW33 = 70,
	PPC_REG_DMRROW34 = 71,
	PPC_REG_DMRROW35 = 72,
	PPC_REG_DMRROW36 = 73,
	PPC_REG_DMRROW37 = 74,
	PPC_REG_DMRROW38 = 75,
	PPC_REG_DMRROW39 = 76,
	PPC_REG_DMRROW40 = 77,
	PPC_REG_DMRROW41 = 78,
	PPC_REG_DMRROW42 = 79,
	PPC_REG_DMRROW43 = 80,
	PPC_REG_DMRROW44 = 81,
	PPC_REG_DMRROW45 = 82,
	PPC_REG_DMRROW46 = 83,
	PPC_REG_DMRROW47 = 84,
	PPC_REG_DMRROW48 = 85,
	PPC_REG_DMRROW49 = 86,
	PPC_REG_DMRROW50 = 87,
	PPC_REG_DMRROW51 = 88,
	PPC_REG_DMRROW52 = 89,
	PPC_REG_DMRROW53 = 90,
	PPC_REG_DMRROW54 = 91,
	PPC_REG_DMRROW55 = 92,
	PPC_REG_DMRROW56 = 93,
	PPC_REG_DMRROW57 = 94,
	PPC_REG_DMRROW58 = 95,
	PPC_REG_DMRROW59 = 96,
	PPC_REG_DMRROW60 = 97,
	PPC_REG_DMRROW61 = 98,
	PPC_REG_DMRROW62 = 99,
	PPC_REG_DMRROW63 = 100,
	PPC_REG_DMRROWp0 = 101,
	PPC_REG_DMRROWp1 = 102,
	PPC_REG_DMRROWp2 = 103,
	PPC_REG_DMRROWp3 = 104,
	PPC_REG_DMRROWp4 = 105,
	PPC_REG_DMRROWp5 = 106,
	PPC_REG_DMRROWp6 = 107,
	PPC_REG_DMRROWp7 = 108,
	PPC_REG_DMRROWp8 = 109,
	PPC_REG_DMRROWp9 = 110,
	PPC_REG_DMRROWp10 = 111,
	PPC_REG_DMRROWp11 = 112,
	PPC_REG_DMRROWp12 = 113,
	PPC_REG_DMRROWp13 = 114,
	PPC_REG_DMRROWp14 = 115,
	PPC_REG_DMRROWp15 = 116,
	PPC_REG_DMRROWp16 = 117,
	PPC_REG_DMRROWp17 = 118,
	PPC_REG_DMRROWp18 = 119,
	PPC_REG_DMRROWp19 = 120,
	PPC_REG_DMRROWp20 = 121,
	PPC_REG_DMRROWp21 = 122,
	PPC_REG_DMRROWp22 = 123,
	PPC_REG_DMRROWp23 = 124,
	PPC_REG_DMRROWp24 = 125,
	PPC_REG_DMRROWp25 = 126,
	PPC_REG_DMRROWp26 = 127,
	PPC_REG_DMRROWp27 = 128,
	PPC_REG_DMRROWp28 = 129,
	PPC_REG_DMRROWp29 = 130,
	PPC_REG_DMRROWp30 = 131,
	PPC_REG_DMRROWp31 = 132,
	PPC_REG_DMRp0 = 133,
	PPC_REG_DMRp1 = 134,
	PPC_REG_DMRp2 = 135,
	PPC_REG_DMRp3 = 136,
	PPC_REG_F0 = 137,
	PPC_REG_F1 = 138,
	PPC_REG_F2 = 139,
	PPC_REG_F3 = 140,
	PPC_REG_F4 = 141,
	PPC_REG_F5 = 142,
	PPC_REG_F6 = 143,
	PPC_REG_F7 = 144,
	PPC_REG_F8 = 145,
	PPC_REG_F9 = 146,
	PPC_REG_F10 = 147,
	PPC_REG_F11 = 148,
	PPC_REG_F12 = 149,
	PPC_REG_F13 = 150,
	PPC_REG_F14 = 151,
	PPC_REG_F15 = 152,
	PPC_REG_F16 = 153,
	PPC_REG_F17 = 154,
	PPC_REG_F18 = 155,
	PPC_REG_F19 = 156,
	PPC_REG_F20 = 157,
	PPC_REG_F21 = 158,
	PPC_REG_F22 = 159,
	PPC_REG_F23 = 160,
	PPC_REG_F24 = 161,
	PPC_REG_F25 = 162,
	PPC_REG_F26 = 163,
	PPC_REG_F27 = 164,
	PPC_REG_F28 = 165,
	PPC_REG_F29 = 166,
	PPC_REG_F30 = 167,
	PPC_REG_F31 = 168,
	PPC_REG_FP8 = 169,
	PPC_REG_LR8 = 170,
	PPC_REG_QF0 = 171,
	PPC_REG_QF1 = 172,
	PPC_REG_QF2 = 173,
	PPC_REG_QF3 = 174,
	PPC_REG_QF4 = 175,
	PPC_REG_QF5 = 176,
	PPC_REG_QF6 = 177,
	PPC_REG_QF7 = 178,
	PPC_REG_QF8 = 179,
	PPC_REG_QF9 = 180,
	PPC_REG_QF10 = 181,
	PPC_REG_QF11 = 182,
	PPC_REG_QF12 = 183,
	PPC_REG_QF13 = 184,
	PPC_REG_QF14 = 185,
	PPC_REG_QF15 = 186,
	PPC_REG_QF16 = 187,
	PPC_REG_QF17 = 188,
	PPC_REG_QF18 = 189,
	PPC_REG_QF19 = 190,
	PPC_REG_QF20 = 191,
	PPC_REG_QF21 = 192,
	PPC_REG_QF22 = 193,
	PPC_REG_QF23 = 194,
	PPC_REG_QF24 = 195,
	PPC_REG_QF25 = 196,
	PPC_REG_QF26 = 197,
	PPC_REG_QF27 = 198,
	PPC_REG_QF28 = 199,
	PPC_REG_QF29 = 200,
	PPC_REG_QF30 = 201,
	PPC_REG_QF31 = 202,
	PPC_REG_R0 = 203,
	PPC_REG_R1 = 204,
	PPC_REG_R2 = 205,
	PPC_REG_R3 = 206,
	PPC_REG_R4 = 207,
	PPC_REG_R5 = 208,
	PPC_REG_R6 = 209,
	PPC_REG_R7 = 210,
	PPC_REG_R8 = 211,
	PPC_REG_R9 = 212,
	PPC_REG_R10 = 213,
	PPC_REG_R11 = 214,
	PPC_REG_R12 = 215,
	PPC_REG_R13 = 216,
	PPC_REG_R14 = 217,
	PPC_REG_R15 = 218,
	PPC_REG_R16 = 219,
	PPC_REG_R17 = 220,
	PPC_REG_R18 = 221,
	PPC_REG_R19 = 222,
	PPC_REG_R20 = 223,
	PPC_REG_R21 = 224,
	PPC_REG_R22 = 225,
	PPC_REG_R23 = 226,
	PPC_REG_R24 = 227,
	PPC_REG_R25 = 228,
	PPC_REG_R26 = 229,
	PPC_REG_R27 = 230,
	PPC_REG_R28 = 231,
	PPC_REG_R29 = 232,
	PPC_REG_R30 = 233,
	PPC_REG_R31 = 234,
	PPC_REG_S0 = 235,
	PPC_REG_S1 = 236,
	PPC_REG_S2 = 237,
	PPC_REG_S3 = 238,
	PPC_REG_S4 = 239,
	PPC_REG_S5 = 240,
	PPC_REG_S6 = 241,
	PPC_REG_S7 = 242,
	PPC_REG_S8 = 243,
	PPC_REG_S9 = 244,
	PPC_REG_S10 = 245,
	PPC_REG_S11 = 246,
	PPC_REG_S12 = 247,
	PPC_REG_S13 = 248,
	PPC_REG_S14 = 249,
	PPC_REG_S15 = 250,
	PPC_REG_S16 = 251,
	PPC_REG_S17 = 252,
	PPC_REG_S18 = 253,
	PPC_REG_S19 = 254,
	PPC_REG_S20 = 255,
	PPC_REG_S21 = 256,
	PPC_REG_S22 = 257,
	PPC_REG_S23 = 258,
	PPC_REG_S24 = 259,
	PPC_REG_S25 = 260,
	PPC_REG_S26 = 261,
	PPC_REG_S27 = 262,
	PPC_REG_S28 = 263,
	PPC_REG_S29 = 264,
	PPC_REG_S30 = 265,
	PPC_REG_S31 = 266,
	PPC_REG_UACC0 = 267,
	PPC_REG_UACC1 = 268,
	PPC_REG_UACC2 = 269,
	PPC_REG_UACC3 = 270,
	PPC_REG_UACC4 = 271,
	PPC_REG_UACC5 = 272,
	PPC_REG_UACC6 = 273,
	PPC_REG_UACC7 = 274,
	PPC_REG_V0 = 275,
	PPC_REG_V1 = 276,
	PPC_REG_V2 = 277,
	PPC_REG_V3 = 278,
	PPC_REG_V4 = 279,
	PPC_REG_V5 = 280,
	PPC_REG_V6 = 281,
	PPC_REG_V7 = 282,
	PPC_REG_V8 = 283,
	PPC_REG_V9 = 284,
	PPC_REG_V10 = 285,
	PPC_REG_V11 = 286,
	PPC_REG_V12 = 287,
	PPC_REG_V13 = 288,
	PPC_REG_V14 = 289,
	PPC_REG_V15 = 290,
	PPC_REG_V16 = 291,
	PPC_REG_V17 = 292,
	PPC_REG_V18 = 293,
	PPC_REG_V19 = 294,
	PPC_REG_V20 = 295,
	PPC_REG_V21 = 296,
	PPC_REG_V22 = 297,
	PPC_REG_V23 = 298,
	PPC_REG_V24 = 299,
	PPC_REG_V25 = 300,
	PPC_REG_V26 = 301,
	PPC_REG_V27 = 302,
	PPC_REG_V28 = 303,
	PPC_REG_V29 = 304,
	PPC_REG_V30 = 305,
	PPC_REG_V31 = 306,
	PPC_REG_VF0 = 307,
	PPC_REG_VF1 = 308,
	PPC_REG_VF2 = 309,
	PPC_REG_VF3 = 310,
	PPC_REG_VF4 = 311,
	PPC_REG_VF5 = 312,
	PPC_REG_VF6 = 313,
	PPC_REG_VF7 = 314,
	PPC_REG_VF8 = 315,
	PPC_REG_VF9 = 316,
	PPC_REG_VF10 = 317,
	PPC_REG_VF11 = 318,
	PPC_REG_VF12 = 319,
	PPC_REG_VF13 = 320,
	PPC_REG_VF14 = 321,
	PPC_REG_VF15 = 322,
	PPC_REG_VF16 = 323,
	PPC_REG_VF17 = 324,
	PPC_REG_VF18 = 325,
	PPC_REG_VF19 = 326,
	PPC_REG_VF20 = 327,
	PPC_REG_VF21 = 328,
	PPC_REG_VF22 = 329,
	PPC_REG_VF23 = 330,
	PPC_REG_VF24 = 331,
	PPC_REG_VF25 = 332,
	PPC_REG_VF26 = 333,
	PPC_REG_VF27 = 334,
	PPC_REG_VF28 = 335,
	PPC_REG_VF29 = 336,
	PPC_REG_VF30 = 337,
	PPC_REG_VF31 = 338,
	PPC_REG_VSL0 = 339,
	PPC_REG_VSL1 = 340,
	PPC_REG_VSL2 = 341,
	PPC_REG_VSL3 = 342,
	PPC_REG_VSL4 = 343,
	PPC_REG_VSL5 = 344,
	PPC_REG_VSL6 = 345,
	PPC_REG_VSL7 = 346,
	PPC_REG_VSL8 = 347,
	PPC_REG_VSL9 = 348,
	PPC_REG_VSL10 = 349,
	PPC_REG_VSL11 = 350,
	PPC_REG_VSL12 = 351,
	PPC_REG_VSL13 = 352,
	PPC_REG_VSL14 = 353,
	PPC_REG_VSL15 = 354,
	PPC_REG_VSL16 = 355,
	PPC_REG_VSL17 = 356,
	PPC_REG_VSL18 = 357,
	PPC_REG_VSL19 = 358,
	PPC_REG_VSL20 = 359,
	PPC_REG_VSL21 = 360,
	PPC_REG_VSL22 = 361,
	PPC_REG_VSL23 = 362,
	PPC_REG_VSL24 = 363,
	PPC_REG_VSL25 = 364,
	PPC_REG_VSL26 = 365,
	PPC_REG_VSL27 = 366,
	PPC_REG_VSL28 = 367,
	PPC_REG_VSL29 = 368,
	PPC_REG_VSL30 = 369,
	PPC_REG_VSL31 = 370,
	PPC_REG_VSRp0 = 371,
	PPC_REG_VSRp1 = 372,
	PPC_REG_VSRp2 = 373,
	PPC_REG_VSRp3 = 374,
	PPC_REG_VSRp4 = 375,
	PPC_REG_VSRp5 = 376,
	PPC_REG_VSRp6 = 377,
	PPC_REG_VSRp7 = 378,
	PPC_REG_VSRp8 = 379,
	PPC_REG_VSRp9 = 380,
	PPC_REG_VSRp10 = 381,
	PPC_REG_VSRp11 = 382,
	PPC_REG_VSRp12 = 383,
	PPC_REG_VSRp13 = 384,
	PPC_REG_VSRp14 = 385,
	PPC_REG_VSRp15 = 386,
	PPC_REG_VSRp16 = 387,
	PPC_REG_VSRp17 = 388,
	PPC_REG_VSRp18 = 389,
	PPC_REG_VSRp19 = 390,
	PPC_REG_VSRp20 = 391,
	PPC_REG_VSRp21 = 392,
	PPC_REG_VSRp22 = 393,
	PPC_REG_VSRp23 = 394,
	PPC_REG_VSRp24 = 395,
	PPC_REG_VSRp25 = 396,
	PPC_REG_VSRp26 = 397,
	PPC_REG_VSRp27 = 398,
	PPC_REG_VSRp28 = 399,
	PPC_REG_VSRp29 = 400,
	PPC_REG_VSRp30 = 401,
	PPC_REG_VSRp31 = 402,
	PPC_REG_VSX32 = 403,
	PPC_REG_VSX33 = 404,
	PPC_REG_VSX34 = 405,
	PPC_REG_VSX35 = 406,
	PPC_REG_VSX36 = 407,
	PPC_REG_VSX37 = 408,
	PPC_REG_VSX38 = 409,
	PPC_REG_VSX39 = 410,
	PPC_REG_VSX40 = 411,
	PPC_REG_VSX41 = 412,
	PPC_REG_VSX42 = 413,
	PPC_REG_VSX43 = 414,
	PPC_REG_VSX44 = 415,
	PPC_REG_VSX45 = 416,
	PPC_REG_VSX46 = 417,
	PPC_REG_VSX47 = 418,
	PPC_REG_VSX48 = 419,
	PPC_REG_VSX49 = 420,
	PPC_REG_VSX50 = 421,
	PPC_REG_VSX51 = 422,
	PPC_REG_VSX52 = 423,
	PPC_REG_VSX53 = 424,
	PPC_REG_VSX54 = 425,
	PPC_REG_VSX55 = 426,
	PPC_REG_VSX56 = 427,
	PPC_REG_VSX57 = 428,
	PPC_REG_VSX58 = 429,
	PPC_REG_VSX59 = 430,
	PPC_REG_VSX60 = 431,
	PPC_REG_VSX61 = 432,
	PPC_REG_VSX62 = 433,
	PPC_REG_VSX63 = 434,
	PPC_REG_WACC0 = 435,
	PPC_REG_WACC1 = 436,
	PPC_REG_WACC2 = 437,
	PPC_REG_WACC3 = 438,
	PPC_REG_WACC4 = 439,
	PPC_REG_WACC5 = 440,
	PPC_REG_WACC6 = 441,
	PPC_REG_WACC7 = 442,
	PPC_REG_WACC_HI0 = 443,
	PPC_REG_WACC_HI1 = 444,
	PPC_REG_WACC_HI2 = 445,
	PPC_REG_WACC_HI3 = 446,
	PPC_REG_WACC_HI4 = 447,
	PPC_REG_WACC_HI5 = 448,
	PPC_REG_WACC_HI6 = 449,
	PPC_REG_WACC_HI7 = 450,
	PPC_REG_X0 = 451,
	PPC_REG_X1 = 452,
	PPC_REG_X2 = 453,
	PPC_REG_X3 = 454,
	PPC_REG_X4 = 455,
	PPC_REG_X5 = 456,
	PPC_REG_X6 = 457,
	PPC_REG_X7 = 458,
	PPC_REG_X8 = 459,
	PPC_REG_X9 = 460,
	PPC_REG_X10 = 461,
	PPC_REG_X11 = 462,
	PPC_REG_X12 = 463,
	PPC_REG_X13 = 464,
	PPC_REG_X14 = 465,
	PPC_REG_X15 = 466,
	PPC_REG_X16 = 467,
	PPC_REG_X17 = 468,
	PPC_REG_X18 = 469,
	PPC_REG_X19 = 470,
	PPC_REG_X20 = 471,
	PPC_REG_X21 = 472,
	PPC_REG_X22 = 473,
	PPC_REG_X23 = 474,
	PPC_REG_X24 = 475,
	PPC_REG_X25 = 476,
	PPC_REG_X26 = 477,
	PPC_REG_X27 = 478,
	PPC_REG_X28 = 479,
	PPC_REG_X29 = 480,
	PPC_REG_X30 = 481,
	PPC_REG_X31 = 482,
	PPC_REG_ZERO8 = 483,
	PPC_REG_CR0EQ = 484,
	PPC_REG_CR1EQ = 485,
	PPC_REG_CR2EQ = 486,
	PPC_REG_CR3EQ = 487,
	PPC_REG_CR4EQ = 488,
	PPC_REG_CR5EQ = 489,
	PPC_REG_CR6EQ = 490,
	PPC_REG_CR7EQ = 491,
	PPC_REG_CR0GT = 492,
	PPC_REG_CR1GT = 493,
	PPC_REG_CR2GT = 494,
	PPC_REG_CR3GT = 495,
	PPC_REG_CR4GT = 496,
	PPC_REG_CR5GT = 497,
	PPC_REG_CR6GT = 498,
	PPC_REG_CR7GT = 499,
	PPC_REG_CR0LT = 500,
	PPC_REG_CR1LT = 501,
	PPC_REG_CR2LT = 502,
	PPC_REG_CR3LT = 503,
	PPC_REG_CR4LT = 504,
	PPC_REG_CR5LT = 505,
	PPC_REG_CR6LT = 506,
	PPC_REG_CR7LT = 507,
	PPC_REG_CR0UN = 508,
	PPC_REG_CR1UN = 509,
	PPC_REG_CR2UN = 510,
	PPC_REG_CR3UN = 511,
	PPC_REG_CR4UN = 512,
	PPC_REG_CR5UN = 513,
	PPC_REG_CR6UN = 514,
	PPC_REG_CR7UN = 515,
	PPC_REG_G8p0 = 516,
	PPC_REG_G8p1 = 517,
	PPC_REG_G8p2 = 518,
	PPC_REG_G8p3 = 519,
	PPC_REG_G8p4 = 520,
	PPC_REG_G8p5 = 521,
	PPC_REG_G8p6 = 522,
	PPC_REG_G8p7 = 523,
	PPC_REG_G8p8 = 524,
	PPC_REG_G8p9 = 525,
	PPC_REG_G8p10 = 526,
	PPC_REG_G8p11 = 527,
	PPC_REG_G8p12 = 528,
	PPC_REG_G8p13 = 529,
	PPC_REG_G8p14 = 530,
	PPC_REG_G8p15 = 531,
	PPC_REG_ENDING, // 532

	// clang-format on
	// generated content <PPCGenCSRegEnum.inc> end
} ppc_reg;

/// Instruction's operand referring to memory
/// This is associated with PPC_OP_MEM operand type above
typedef struct ppc_op_mem {
	ppc_reg base;	///< base register
	int32_t disp;	///< displacement/offset value
	ppc_reg offset; ///< Offset register
} ppc_op_mem;

/// Instruction operand
typedef struct cs_ppc_op {
	ppc_op_type type;	///< operand type
	union {
		ppc_reg reg;	///< register value for REG operand
		int64_t imm;	///< immediate value for IMM operand
		ppc_op_mem mem; ///< base/disp value for MEM operand
	};
	cs_ac_type access;
} cs_ppc_op;

typedef struct {
	uint8_t bo; ///< BO field of branch condition. UINT8_MAX if invalid.
	uint8_t bi; ///< BI field of branch condition. UINT8_MAX if invalid.
	ppc_cr_bit crX_bit; ///< CR field bit to test.
	ppc_reg crX;	    ///< The CR field accessed.
	ppc_br_hint hint;   ///< The encoded hint.
	ppc_pred pred_cr;   ///< CR-bit branch predicate
	ppc_pred pred_ctr;  ///< CTR branch predicate
	ppc_bh bh;	    ///< The BH field hint if any is present.
} ppc_bc;

/// Returns true if the CTR is decremented.
/// False otherwise.
static inline bool cs_ppc_bc_decr_ctr(uint8_t bo)
{
	if (bo != UINT8_MAX && (bo & PPC_BO_DECR_CTR) == 0)
		return true;
	return false;
}

/// Returns true if the CTR is compared to 0
/// Implies that the CTR is decremented at all.
/// False otherwise.
static inline bool cs_ppc_bc_tests_ctr_is_zero(uint8_t bo)
{
	if (bo != UINT8_MAX && (bo & PPC_BO_CTR_CMP) != 0 &&
	    cs_ppc_bc_decr_ctr(bo))
		return true;
	return false;
}

/// Returns true if a CR bit is tested.
/// False otherwise.
static inline bool cs_ppc_bc_cr_is_tested(uint8_t bo)
{
	if (bo != UINT8_MAX && (bo & PPC_BO_TEST_CR) == 0)
		return true;
	return false;
}

/// Returns true if a CR bit is compared to 1.
/// Implies that the CR field is tested at all.
/// False otherwise.
static inline bool cs_ppc_bc_cr_bit_is_one(uint8_t bo)
{
	if (bo != UINT8_MAX && (bo & PPC_BO_CR_CMP) != 0 &&
	    cs_ppc_bc_cr_is_tested(bo))
		return true;
	return false;
}

/// PPC instruction
typedef enum ppc_insn {
	// generated content <PPCGenCSInsnEnum.inc> begin
	// clang-format off

	PPC_INS_INVALID,
	PPC_INS_CLRLSLDI,
	PPC_INS_CLRLSLWI,
	PPC_INS_CLRRDI,
	PPC_INS_CLRRWI,
	PPC_INS_DCBFL,
	PPC_INS_DCBFLP,
	PPC_INS_DCBFPS,
	PPC_INS_DCBF,
	PPC_INS_DCBSTPS,
	PPC_INS_DCBTCT,
	PPC_INS_DCBTDS,
	PPC_INS_DCBTSTCT,
	PPC_INS_DCBTSTDS,
	PPC_INS_DCBTSTT,
	PPC_INS_DCBTST,
	PPC_INS_DCBTT,
	PPC_INS_DCBT,
	PPC_INS_EXTLDI,
	PPC_INS_EXTLWI,
	PPC_INS_EXTRDI,
	PPC_INS_EXTRWI,
	PPC_INS_INSLWI,
	PPC_INS_INSRDI,
	PPC_INS_INSRWI,
	PPC_INS_LA,
	PPC_INS_RLWIMI,
	PPC_INS_RLWINM,
	PPC_INS_RLWNM,
	PPC_INS_ROTRDI,
	PPC_INS_ROTRWI,
	PPC_INS_SLDI,
	PPC_INS_SLWI,
	PPC_INS_SRDI,
	PPC_INS_SRWI,
	PPC_INS_SUBI,
	PPC_INS_SUBIC,
	PPC_INS_SUBIS,
	PPC_INS_SUBPCIS,
	PPC_INS_ADD,
	PPC_INS_ADDO,
	PPC_INS_ADDC,
	PPC_INS_ADDCO,
	PPC_INS_ADDE,
	PPC_INS_ADDEO,
	PPC_INS_ADDEX,
	PPC_INS_ADDI,
	PPC_INS_ADDIC,
	PPC_INS_ADDIS,
	PPC_INS_ADDME,
	PPC_INS_ADDMEO,
	PPC_INS_ADDPCIS,
	PPC_INS_ADDZE,
	PPC_INS_ADDZEO,
	PPC_INS_AND,
	PPC_INS_ANDC,
	PPC_INS_ANDIS,
	PPC_INS_ANDI,
	PPC_INS_ATTN,
	PPC_INS_B,
	PPC_INS_BA,
	PPC_INS_BCDADD,
	PPC_INS_BCDCFN,
	PPC_INS_BCDCFSQ,
	PPC_INS_BCDCFZ,
	PPC_INS_BCDCPSGN,
	PPC_INS_BCDCTN,
	PPC_INS_BCDCTSQ,
	PPC_INS_BCDCTZ,
	PPC_INS_BCDSETSGN,
	PPC_INS_BCDSR,
	PPC_INS_BCDSUB,
	PPC_INS_BCDS,
	PPC_INS_BCDTRUNC,
	PPC_INS_BCDUS,
	PPC_INS_BCDUTRUNC,
	PPC_INS_BCTR,
	PPC_INS_BCTRL,
	PPC_INS_BL,
	PPC_INS_BLA,
	PPC_INS_BLR,
	PPC_INS_BLRL,
	PPC_INS_BPERMD,
	PPC_INS_BRD,
	PPC_INS_BRH,
	PPC_INS_BRINC,
	PPC_INS_BRW,
	PPC_INS_CFUGED,
	PPC_INS_CLRBHRB,
	PPC_INS_CMPB,
	PPC_INS_CMPD,
	PPC_INS_CMPDI,
	PPC_INS_CMPEQB,
	PPC_INS_CMPLD,
	PPC_INS_CMPLDI,
	PPC_INS_CMPLW,
	PPC_INS_CMPLWI,
	PPC_INS_CMPRB,
	PPC_INS_CMPW,
	PPC_INS_CMPWI,
	PPC_INS_CNTLZD,
	PPC_INS_CNTLZDM,
	PPC_INS_CNTLZW,
	PPC_INS_CNTTZD,
	PPC_INS_CNTTZDM,
	PPC_INS_CNTTZW,
	PPC_INS_CPABORT,
	PPC_INS_COPY,
	PPC_INS_PASTE,
	PPC_INS_CRAND,
	PPC_INS_CRANDC,
	PPC_INS_CREQV,
	PPC_INS_CRNAND,
	PPC_INS_CRNOR,
	PPC_INS_CROR,
	PPC_INS_CRORC,
	PPC_INS_CRXOR,
	PPC_INS_DARN,
	PPC_INS_DCBA,
	PPC_INS_DCBFEP,
	PPC_INS_DCBI,
	PPC_INS_DCBST,
	PPC_INS_DCBSTEP,
	PPC_INS_DCBTEP,
	PPC_INS_DCBTSTEP,
	PPC_INS_DCBZ,
	PPC_INS_DCBZEP,
	PPC_INS_DCBZL,
	PPC_INS_DCBZLEP,
	PPC_INS_DCCCI,
	PPC_INS_DIVD,
	PPC_INS_DIVDE,
	PPC_INS_DIVDEO,
	PPC_INS_DIVDEU,
	PPC_INS_DIVDEUO,
	PPC_INS_DIVDO,
	PPC_INS_DIVDU,
	PPC_INS_DIVDUO,
	PPC_INS_DIVW,
	PPC_INS_DIVWE,
	PPC_INS_DIVWEO,
	PPC_INS_DIVWEU,
	PPC_INS_DIVWEUO,
	PPC_INS_DIVWO,
	PPC_INS_DIVWU,
	PPC_INS_DIVWUO,
	PPC_INS_DMMR,
	PPC_INS_DMSETDMRZ,
	PPC_INS_DMXOR,
	PPC_INS_DMXXEXTFDMR256,
	PPC_INS_DMXXEXTFDMR512,
	PPC_INS_DMXXINSTFDMR256,
	PPC_INS_DMXXINSTFDMR512,
	PPC_INS_DSS,
	PPC_INS_DSSALL,
	PPC_INS_DST,
	PPC_INS_DSTST,
	PPC_INS_DSTSTT,
	PPC_INS_DSTT,
	PPC_INS_EFDABS,
	PPC_INS_EFDADD,
	PPC_INS_EFDCFS,
	PPC_INS_EFDCFSF,
	PPC_INS_EFDCFSI,
	PPC_INS_EFDCFSID,
	PPC_INS_EFDCFUF,
	PPC_INS_EFDCFUI,
	PPC_INS_EFDCFUID,
	PPC_INS_EFDCMPEQ,
	PPC_INS_EFDCMPGT,
	PPC_INS_EFDCMPLT,
	PPC_INS_EFDCTSF,
	PPC_INS_EFDCTSI,
	PPC_INS_EFDCTSIDZ,
	PPC_INS_EFDCTSIZ,
	PPC_INS_EFDCTUF,
	PPC_INS_EFDCTUI,
	PPC_INS_EFDCTUIDZ,
	PPC_INS_EFDCTUIZ,
	PPC_INS_EFDDIV,
	PPC_INS_EFDMUL,
	PPC_INS_EFDNABS,
	PPC_INS_EFDNEG,
	PPC_INS_EFDSUB,
	PPC_INS_EFDTSTEQ,
	PPC_INS_EFDTSTGT,
	PPC_INS_EFDTSTLT,
	PPC_INS_EFSABS,
	PPC_INS_EFSADD,
	PPC_INS_EFSCFD,
	PPC_INS_EFSCFSF,
	PPC_INS_EFSCFSI,
	PPC_INS_EFSCFUF,
	PPC_INS_EFSCFUI,
	PPC_INS_EFSCMPEQ,
	PPC_INS_EFSCMPGT,
	PPC_INS_EFSCMPLT,
	PPC_INS_EFSCTSF,
	PPC_INS_EFSCTSI,
	PPC_INS_EFSCTSIZ,
	PPC_INS_EFSCTUF,
	PPC_INS_EFSCTUI,
	PPC_INS_EFSCTUIZ,
	PPC_INS_EFSDIV,
	PPC_INS_EFSMUL,
	PPC_INS_EFSNABS,
	PPC_INS_EFSNEG,
	PPC_INS_EFSSUB,
	PPC_INS_EFSTSTEQ,
	PPC_INS_EFSTSTGT,
	PPC_INS_EFSTSTLT,
	PPC_INS_EQV,
	PPC_INS_EVABS,
	PPC_INS_EVADDIW,
	PPC_INS_EVADDSMIAAW,
	PPC_INS_EVADDSSIAAW,
	PPC_INS_EVADDUMIAAW,
	PPC_INS_EVADDUSIAAW,
	PPC_INS_EVADDW,
	PPC_INS_EVAND,
	PPC_INS_EVANDC,
	PPC_INS_EVCMPEQ,
	PPC_INS_EVCMPGTS,
	PPC_INS_EVCMPGTU,
	PPC_INS_EVCMPLTS,
	PPC_INS_EVCMPLTU,
	PPC_INS_EVCNTLSW,
	PPC_INS_EVCNTLZW,
	PPC_INS_EVDIVWS,
	PPC_INS_EVDIVWU,
	PPC_INS_EVEQV,
	PPC_INS_EVEXTSB,
	PPC_INS_EVEXTSH,
	PPC_INS_EVFSABS,
	PPC_INS_EVFSADD,
	PPC_INS_EVFSCFSF,
	PPC_INS_EVFSCFSI,
	PPC_INS_EVFSCFUF,
	PPC_INS_EVFSCFUI,
	PPC_INS_EVFSCMPEQ,
	PPC_INS_EVFSCMPGT,
	PPC_INS_EVFSCMPLT,
	PPC_INS_EVFSCTSF,
	PPC_INS_EVFSCTSI,
	PPC_INS_EVFSCTSIZ,
	PPC_INS_EVFSCTUI,
	PPC_INS_EVFSDIV,
	PPC_INS_EVFSMUL,
	PPC_INS_EVFSNABS,
	PPC_INS_EVFSNEG,
	PPC_INS_EVFSSUB,
	PPC_INS_EVFSTSTEQ,
	PPC_INS_EVFSTSTGT,
	PPC_INS_EVFSTSTLT,
	PPC_INS_EVLDD,
	PPC_INS_EVLDDX,
	PPC_INS_EVLDH,
	PPC_INS_EVLDHX,
	PPC_INS_EVLDW,
	PPC_INS_EVLDWX,
	PPC_INS_EVLHHESPLAT,
	PPC_INS_EVLHHESPLATX,
	PPC_INS_EVLHHOSSPLAT,
	PPC_INS_EVLHHOSSPLATX,
	PPC_INS_EVLHHOUSPLAT,
	PPC_INS_EVLHHOUSPLATX,
	PPC_INS_EVLWHE,
	PPC_INS_EVLWHEX,
	PPC_INS_EVLWHOS,
	PPC_INS_EVLWHOSX,
	PPC_INS_EVLWHOU,
	PPC_INS_EVLWHOUX,
	PPC_INS_EVLWHSPLAT,
	PPC_INS_EVLWHSPLATX,
	PPC_INS_EVLWWSPLAT,
	PPC_INS_EVLWWSPLATX,
	PPC_INS_EVMERGEHI,
	PPC_INS_EVMERGEHILO,
	PPC_INS_EVMERGELO,
	PPC_INS_EVMERGELOHI,
	PPC_INS_EVMHEGSMFAA,
	PPC_INS_EVMHEGSMFAN,
	PPC_INS_EVMHEGSMIAA,
	PPC_INS_EVMHEGSMIAN,
	PPC_INS_EVMHEGUMIAA,
	PPC_INS_EVMHEGUMIAN,
	PPC_INS_EVMHESMF,
	PPC_INS_EVMHESMFA,
	PPC_INS_EVMHESMFAAW,
	PPC_INS_EVMHESMFANW,
	PPC_INS_EVMHESMI,
	PPC_INS_EVMHESMIA,
	PPC_INS_EVMHESMIAAW,
	PPC_INS_EVMHESMIANW,
	PPC_INS_EVMHESSF,
	PPC_INS_EVMHESSFA,
	PPC_INS_EVMHESSFAAW,
	PPC_INS_EVMHESSFANW,
	PPC_INS_EVMHESSIAAW,
	PPC_INS_EVMHESSIANW,
	PPC_INS_EVMHEUMI,
	PPC_INS_EVMHEUMIA,
	PPC_INS_EVMHEUMIAAW,
	PPC_INS_EVMHEUMIANW,
	PPC_INS_EVMHEUSIAAW,
	PPC_INS_EVMHEUSIANW,
	PPC_INS_EVMHOGSMFAA,
	PPC_INS_EVMHOGSMFAN,
	PPC_INS_EVMHOGSMIAA,
	PPC_INS_EVMHOGSMIAN,
	PPC_INS_EVMHOGUMIAA,
	PPC_INS_EVMHOGUMIAN,
	PPC_INS_EVMHOSMF,
	PPC_INS_EVMHOSMFA,
	PPC_INS_EVMHOSMFAAW,
	PPC_INS_EVMHOSMFANW,
	PPC_INS_EVMHOSMI,
	PPC_INS_EVMHOSMIA,
	PPC_INS_EVMHOSMIAAW,
	PPC_INS_EVMHOSMIANW,
	PPC_INS_EVMHOSSF,
	PPC_INS_EVMHOSSFA,
	PPC_INS_EVMHOSSFAAW,
	PPC_INS_EVMHOSSFANW,
	PPC_INS_EVMHOSSIAAW,
	PPC_INS_EVMHOSSIANW,
	PPC_INS_EVMHOUMI,
	PPC_INS_EVMHOUMIA,
	PPC_INS_EVMHOUMIAAW,
	PPC_INS_EVMHOUMIANW,
	PPC_INS_EVMHOUSIAAW,
	PPC_INS_EVMHOUSIANW,
	PPC_INS_EVMRA,
	PPC_INS_EVMWHSMF,
	PPC_INS_EVMWHSMFA,
	PPC_INS_EVMWHSMI,
	PPC_INS_EVMWHSMIA,
	PPC_INS_EVMWHSSF,
	PPC_INS_EVMWHSSFA,
	PPC_INS_EVMWHUMI,
	PPC_INS_EVMWHUMIA,
	PPC_INS_EVMWLSMIAAW,
	PPC_INS_EVMWLSMIANW,
	PPC_INS_EVMWLSSIAAW,
	PPC_INS_EVMWLSSIANW,
	PPC_INS_EVMWLUMI,
	PPC_INS_EVMWLUMIA,
	PPC_INS_EVMWLUMIAAW,
	PPC_INS_EVMWLUMIANW,
	PPC_INS_EVMWLUSIAAW,
	PPC_INS_EVMWLUSIANW,
	PPC_INS_EVMWSMF,
	PPC_INS_EVMWSMFA,
	PPC_INS_EVMWSMFAA,
	PPC_INS_EVMWSMFAN,
	PPC_INS_EVMWSMI,
	PPC_INS_EVMWSMIA,
	PPC_INS_EVMWSMIAA,
	PPC_INS_EVMWSMIAN,
	PPC_INS_EVMWSSF,
	PPC_INS_EVMWSSFA,
	PPC_INS_EVMWSSFAA,
	PPC_INS_EVMWSSFAN,
	PPC_INS_EVMWUMI,
	PPC_INS_EVMWUMIA,
	PPC_INS_EVMWUMIAA,
	PPC_INS_EVMWUMIAN,
	PPC_INS_EVNAND,
	PPC_INS_EVNEG,
	PPC_INS_EVNOR,
	PPC_INS_EVOR,
	PPC_INS_EVORC,
	PPC_INS_EVRLW,
	PPC_INS_EVRLWI,
	PPC_INS_EVRNDW,
	PPC_INS_EVSEL,
	PPC_INS_EVSLW,
	PPC_INS_EVSLWI,
	PPC_INS_EVSPLATFI,
	PPC_INS_EVSPLATI,
	PPC_INS_EVSRWIS,
	PPC_INS_EVSRWIU,
	PPC_INS_EVSRWS,
	PPC_INS_EVSRWU,
	PPC_INS_EVSTDD,
	PPC_INS_EVSTDDX,
	PPC_INS_EVSTDH,
	PPC_INS_EVSTDHX,
	PPC_INS_EVSTDW,
	PPC_INS_EVSTDWX,
	PPC_INS_EVSTWHE,
	PPC_INS_EVSTWHEX,
	PPC_INS_EVSTWHO,
	PPC_INS_EVSTWHOX,
	PPC_INS_EVSTWWE,
	PPC_INS_EVSTWWEX,
	PPC_INS_EVSTWWO,
	PPC_INS_EVSTWWOX,
	PPC_INS_EVSUBFSMIAAW,
	PPC_INS_EVSUBFSSIAAW,
	PPC_INS_EVSUBFUMIAAW,
	PPC_INS_EVSUBFUSIAAW,
	PPC_INS_EVSUBFW,
	PPC_INS_EVSUBIFW,
	PPC_INS_EVXOR,
	PPC_INS_EXTSB,
	PPC_INS_EXTSH,
	PPC_INS_EXTSW,
	PPC_INS_EXTSWSLI,
	PPC_INS_EIEIO,
	PPC_INS_FABS,
	PPC_INS_FADD,
	PPC_INS_FADDS,
	PPC_INS_FCFID,
	PPC_INS_FCFIDS,
	PPC_INS_FCFIDU,
	PPC_INS_FCFIDUS,
	PPC_INS_FCMPO,
	PPC_INS_FCMPU,
	PPC_INS_FCPSGN,
	PPC_INS_FCTID,
	PPC_INS_FCTIDU,
	PPC_INS_FCTIDUZ,
	PPC_INS_FCTIDZ,
	PPC_INS_FCTIW,
	PPC_INS_FCTIWU,
	PPC_INS_FCTIWUZ,
	PPC_INS_FCTIWZ,
	PPC_INS_FDIV,
	PPC_INS_FDIVS,
	PPC_INS_FMADD,
	PPC_INS_FMADDS,
	PPC_INS_FMR,
	PPC_INS_FMSUB,
	PPC_INS_FMSUBS,
	PPC_INS_FMUL,
	PPC_INS_FMULS,
	PPC_INS_FNABS,
	PPC_INS_FNEG,
	PPC_INS_FNMADD,
	PPC_INS_FNMADDS,
	PPC_INS_FNMSUB,
	PPC_INS_FNMSUBS,
	PPC_INS_FRE,
	PPC_INS_FRES,
	PPC_INS_FRIM,
	PPC_INS_FRIN,
	PPC_INS_FRIP,
	PPC_INS_FRIZ,
	PPC_INS_FRSP,
	PPC_INS_FRSQRTE,
	PPC_INS_FRSQRTES,
	PPC_INS_FSEL,
	PPC_INS_FSQRT,
	PPC_INS_FSQRTS,
	PPC_INS_FSUB,
	PPC_INS_FSUBS,
	PPC_INS_FTDIV,
	PPC_INS_FTSQRT,
	PPC_INS_HASHCHK,
	PPC_INS_HASHCHKP,
	PPC_INS_HASHST,
	PPC_INS_HASHSTP,
	PPC_INS_HRFID,
	PPC_INS_ICBI,
	PPC_INS_ICBIEP,
	PPC_INS_ICBLC,
	PPC_INS_ICBLQ,
	PPC_INS_ICBT,
	PPC_INS_ICBTLS,
	PPC_INS_ICCCI,
	PPC_INS_ISEL,
	PPC_INS_ISYNC,
	PPC_INS_LBARX,
	PPC_INS_LBEPX,
	PPC_INS_LBZ,
	PPC_INS_LBZCIX,
	PPC_INS_LBZU,
	PPC_INS_LBZUX,
	PPC_INS_LBZX,
	PPC_INS_LD,
	PPC_INS_LDARX,
	PPC_INS_LDAT,
	PPC_INS_LDBRX,
	PPC_INS_LDCIX,
	PPC_INS_LDU,
	PPC_INS_LDUX,
	PPC_INS_LDX,
	PPC_INS_LFD,
	PPC_INS_LFDEPX,
	PPC_INS_LFDU,
	PPC_INS_LFDUX,
	PPC_INS_LFDX,
	PPC_INS_LFIWAX,
	PPC_INS_LFIWZX,
	PPC_INS_LFS,
	PPC_INS_LFSU,
	PPC_INS_LFSUX,
	PPC_INS_LFSX,
	PPC_INS_LHA,
	PPC_INS_LHARX,
	PPC_INS_LHAU,
	PPC_INS_LHAUX,
	PPC_INS_LHAX,
	PPC_INS_LHBRX,
	PPC_INS_LHEPX,
	PPC_INS_LHZ,
	PPC_INS_LHZCIX,
	PPC_INS_LHZU,
	PPC_INS_LHZUX,
	PPC_INS_LHZX,
	PPC_INS_LMW,
	PPC_INS_LQ,
	PPC_INS_LQARX,
	PPC_INS_LSWI,
	PPC_INS_LVEBX,
	PPC_INS_LVEHX,
	PPC_INS_LVEWX,
	PPC_INS_LVSL,
	PPC_INS_LVSR,
	PPC_INS_LVX,
	PPC_INS_LVXL,
	PPC_INS_LWA,
	PPC_INS_LWARX,
	PPC_INS_LWAT,
	PPC_INS_LWAUX,
	PPC_INS_LWAX,
	PPC_INS_LWBRX,
	PPC_INS_LWEPX,
	PPC_INS_LWZ,
	PPC_INS_LWZCIX,
	PPC_INS_LWZU,
	PPC_INS_LWZUX,
	PPC_INS_LWZX,
	PPC_INS_LXSD,
	PPC_INS_LXSDX,
	PPC_INS_LXSIBZX,
	PPC_INS_LXSIHZX,
	PPC_INS_LXSIWAX,
	PPC_INS_LXSIWZX,
	PPC_INS_LXSSP,
	PPC_INS_LXSSPX,
	PPC_INS_LXV,
	PPC_INS_LXVB16X,
	PPC_INS_LXVD2X,
	PPC_INS_LXVDSX,
	PPC_INS_LXVH8X,
	PPC_INS_LXVKQ,
	PPC_INS_LXVL,
	PPC_INS_LXVLL,
	PPC_INS_LXVP,
	PPC_INS_LXVPRL,
	PPC_INS_LXVPRLL,
	PPC_INS_LXVPX,
	PPC_INS_LXVRBX,
	PPC_INS_LXVRDX,
	PPC_INS_LXVRHX,
	PPC_INS_LXVRL,
	PPC_INS_LXVRLL,
	PPC_INS_LXVRWX,
	PPC_INS_LXVW4X,
	PPC_INS_LXVWSX,
	PPC_INS_LXVX,
	PPC_INS_MADDHD,
	PPC_INS_MADDHDU,
	PPC_INS_MADDLD,
	PPC_INS_MBAR,
	PPC_INS_MCRF,
	PPC_INS_MCRFS,
	PPC_INS_MCRXRX,
	PPC_INS_MFBHRBE,
	PPC_INS_MFCR,
	PPC_INS_MFCTR,
	PPC_INS_MFDCR,
	PPC_INS_MFFS,
	PPC_INS_MFFSCDRN,
	PPC_INS_MFFSCDRNI,
	PPC_INS_MFFSCE,
	PPC_INS_MFFSCRN,
	PPC_INS_MFFSCRNI,
	PPC_INS_MFFSL,
	PPC_INS_MFLR,
	PPC_INS_MFMSR,
	PPC_INS_MFOCRF,
	PPC_INS_MFPMR,
	PPC_INS_MFSPR,
	PPC_INS_MFSR,
	PPC_INS_MFSRIN,
	PPC_INS_MFTB,
	PPC_INS_MFVSCR,
	PPC_INS_MFVSRD,
	PPC_INS_MFVSRLD,
	PPC_INS_MFVSRWZ,
	PPC_INS_MODSD,
	PPC_INS_MODSW,
	PPC_INS_MODUD,
	PPC_INS_MODUW,
	PPC_INS_MSGSYNC,
	PPC_INS_MTCRF,
	PPC_INS_MTCTR,
	PPC_INS_MTDCR,
	PPC_INS_MTFSB0,
	PPC_INS_MTFSB1,
	PPC_INS_MTFSF,
	PPC_INS_MTFSFI,
	PPC_INS_MTLR,
	PPC_INS_MTMSR,
	PPC_INS_MTMSRD,
	PPC_INS_MTOCRF,
	PPC_INS_MTPMR,
	PPC_INS_MTSPR,
	PPC_INS_MTSR,
	PPC_INS_MTSRIN,
	PPC_INS_MTVSCR,
	PPC_INS_MTVSRBM,
	PPC_INS_MTVSRBMI,
	PPC_INS_MTVSRD,
	PPC_INS_MTVSRDD,
	PPC_INS_MTVSRDM,
	PPC_INS_MTVSRHM,
	PPC_INS_MTVSRQM,
	PPC_INS_MTVSRWA,
	PPC_INS_MTVSRWM,
	PPC_INS_MTVSRWS,
	PPC_INS_MTVSRWZ,
	PPC_INS_MULHD,
	PPC_INS_MULHDU,
	PPC_INS_MULHW,
	PPC_INS_MULHWU,
	PPC_INS_MULLD,
	PPC_INS_MULLDO,
	PPC_INS_MULLI,
	PPC_INS_MULLW,
	PPC_INS_MULLWO,
	PPC_INS_NAND,
	PPC_INS_NAP,
	PPC_INS_NEG,
	PPC_INS_NEGO,
	PPC_INS_NOP,
	PPC_INS_NOR,
	PPC_INS_OR,
	PPC_INS_ORC,
	PPC_INS_ORI,
	PPC_INS_ORIS,
	PPC_INS_PADDI,
	PPC_INS_PDEPD,
	PPC_INS_PEXTD,
	PPC_INS_PLBZ,
	PPC_INS_PLD,
	PPC_INS_PLFD,
	PPC_INS_PLFS,
	PPC_INS_PLHA,
	PPC_INS_PLHZ,
	PPC_INS_PLI,
	PPC_INS_PLWA,
	PPC_INS_PLWZ,
	PPC_INS_PLXSD,
	PPC_INS_PLXSSP,
	PPC_INS_PLXV,
	PPC_INS_PLXVP,
	PPC_INS_PMXVBF16GER2,
	PPC_INS_PMXVBF16GER2NN,
	PPC_INS_PMXVBF16GER2NP,
	PPC_INS_PMXVBF16GER2PN,
	PPC_INS_PMXVBF16GER2PP,
	PPC_INS_PMXVF16GER2,
	PPC_INS_PMXVF16GER2NN,
	PPC_INS_PMXVF16GER2NP,
	PPC_INS_PMXVF16GER2PN,
	PPC_INS_PMXVF16GER2PP,
	PPC_INS_PMXVF32GER,
	PPC_INS_PMXVF32GERNN,
	PPC_INS_PMXVF32GERNP,
	PPC_INS_PMXVF32GERPN,
	PPC_INS_PMXVF32GERPP,
	PPC_INS_PMXVF64GER,
	PPC_INS_PMXVF64GERNN,
	PPC_INS_PMXVF64GERNP,
	PPC_INS_PMXVF64GERPN,
	PPC_INS_PMXVF64GERPP,
	PPC_INS_PMXVI16GER2,
	PPC_INS_PMXVI16GER2PP,
	PPC_INS_PMXVI16GER2S,
	PPC_INS_PMXVI16GER2SPP,
	PPC_INS_PMXVI4GER8,
	PPC_INS_PMXVI4GER8PP,
	PPC_INS_PMXVI8GER4,
	PPC_INS_PMXVI8GER4PP,
	PPC_INS_PMXVI8GER4SPP,
	PPC_INS_POPCNTB,
	PPC_INS_POPCNTD,
	PPC_INS_POPCNTW,
	PPC_INS_DCBZ_L,
	PPC_INS_PSQ_L,
	PPC_INS_PSQ_LU,
	PPC_INS_PSQ_LUX,
	PPC_INS_PSQ_LX,
	PPC_INS_PSQ_ST,
	PPC_INS_PSQ_STU,
	PPC_INS_PSQ_STUX,
	PPC_INS_PSQ_STX,
	PPC_INS_PSTB,
	PPC_INS_PSTD,
	PPC_INS_PSTFD,
	PPC_INS_PSTFS,
	PPC_INS_PSTH,
	PPC_INS_PSTW,
	PPC_INS_PSTXSD,
	PPC_INS_PSTXSSP,
	PPC_INS_PSTXV,
	PPC_INS_PSTXVP,
	PPC_INS_PS_ABS,
	PPC_INS_PS_ADD,
	PPC_INS_PS_CMPO0,
	PPC_INS_PS_CMPO1,
	PPC_INS_PS_CMPU0,
	PPC_INS_PS_CMPU1,
	PPC_INS_PS_DIV,
	PPC_INS_PS_MADD,
	PPC_INS_PS_MADDS0,
	PPC_INS_PS_MADDS1,
	PPC_INS_PS_MERGE00,
	PPC_INS_PS_MERGE01,
	PPC_INS_PS_MERGE10,
	PPC_INS_PS_MERGE11,
	PPC_INS_PS_MR,
	PPC_INS_PS_MSUB,
	PPC_INS_PS_MUL,
	PPC_INS_PS_MULS0,
	PPC_INS_PS_MULS1,
	PPC_INS_PS_NABS,
	PPC_INS_PS_NEG,
	PPC_INS_PS_NMADD,
	PPC_INS_PS_NMSUB,
	PPC_INS_PS_RES,
	PPC_INS_PS_RSQRTE,
	PPC_INS_PS_SEL,
	PPC_INS_PS_SUB,
	PPC_INS_PS_SUM0,
	PPC_INS_PS_SUM1,
	PPC_INS_QVALIGNI,
	PPC_INS_QVESPLATI,
	PPC_INS_QVFABS,
	PPC_INS_QVFADD,
	PPC_INS_QVFADDS,
	PPC_INS_QVFCFID,
	PPC_INS_QVFCFIDS,
	PPC_INS_QVFCFIDU,
	PPC_INS_QVFCFIDUS,
	PPC_INS_QVFCMPEQ,
	PPC_INS_QVFCMPGT,
	PPC_INS_QVFCMPLT,
	PPC_INS_QVFCPSGN,
	PPC_INS_QVFCTID,
	PPC_INS_QVFCTIDU,
	PPC_INS_QVFCTIDUZ,
	PPC_INS_QVFCTIDZ,
	PPC_INS_QVFCTIW,
	PPC_INS_QVFCTIWU,
	PPC_INS_QVFCTIWUZ,
	PPC_INS_QVFCTIWZ,
	PPC_INS_QVFLOGICAL,
	PPC_INS_QVFMADD,
	PPC_INS_QVFMADDS,
	PPC_INS_QVFMR,
	PPC_INS_QVFMSUB,
	PPC_INS_QVFMSUBS,
	PPC_INS_QVFMUL,
	PPC_INS_QVFMULS,
	PPC_INS_QVFNABS,
	PPC_INS_QVFNEG,
	PPC_INS_QVFNMADD,
	PPC_INS_QVFNMADDS,
	PPC_INS_QVFNMSUB,
	PPC_INS_QVFNMSUBS,
	PPC_INS_QVFPERM,
	PPC_INS_QVFRE,
	PPC_INS_QVFRES,
	PPC_INS_QVFRIM,
	PPC_INS_QVFRIN,
	PPC_INS_QVFRIP,
	PPC_INS_QVFRIZ,
	PPC_INS_QVFRSP,
	PPC_INS_QVFRSQRTE,
	PPC_INS_QVFRSQRTES,
	PPC_INS_QVFSEL,
	PPC_INS_QVFSUB,
	PPC_INS_QVFSUBS,
	PPC_INS_QVFTSTNAN,
	PPC_INS_QVFXMADD,
	PPC_INS_QVFXMADDS,
	PPC_INS_QVFXMUL,
	PPC_INS_QVFXMULS,
	PPC_INS_QVFXXCPNMADD,
	PPC_INS_QVFXXCPNMADDS,
	PPC_INS_QVFXXMADD,
	PPC_INS_QVFXXMADDS,
	PPC_INS_QVFXXNPMADD,
	PPC_INS_QVFXXNPMADDS,
	PPC_INS_QVGPCI,
	PPC_INS_QVLFCDUX,
	PPC_INS_QVLFCDUXA,
	PPC_INS_QVLFCDX,
	PPC_INS_QVLFCDXA,
	PPC_INS_QVLFCSUX,
	PPC_INS_QVLFCSUXA,
	PPC_INS_QVLFCSX,
	PPC_INS_QVLFCSXA,
	PPC_INS_QVLFDUX,
	PPC_INS_QVLFDUXA,
	PPC_INS_QVLFDX,
	PPC_INS_QVLFDXA,
	PPC_INS_QVLFIWAX,
	PPC_INS_QVLFIWAXA,
	PPC_INS_QVLFIWZX,
	PPC_INS_QVLFIWZXA,
	PPC_INS_QVLFSUX,
	PPC_INS_QVLFSUXA,
	PPC_INS_QVLFSX,
	PPC_INS_QVLFSXA,
	PPC_INS_QVLPCLDX,
	PPC_INS_QVLPCLSX,
	PPC_INS_QVLPCRDX,
	PPC_INS_QVLPCRSX,
	PPC_INS_QVSTFCDUX,
	PPC_INS_QVSTFCDUXA,
	PPC_INS_QVSTFCDUXI,
	PPC_INS_QVSTFCDUXIA,
	PPC_INS_QVSTFCDX,
	PPC_INS_QVSTFCDXA,
	PPC_INS_QVSTFCDXI,
	PPC_INS_QVSTFCDXIA,
	PPC_INS_QVSTFCSUX,
	PPC_INS_QVSTFCSUXA,
	PPC_INS_QVSTFCSUXI,
	PPC_INS_QVSTFCSUXIA,
	PPC_INS_QVSTFCSX,
	PPC_INS_QVSTFCSXA,
	PPC_INS_QVSTFCSXI,
	PPC_INS_QVSTFCSXIA,
	PPC_INS_QVSTFDUX,
	PPC_INS_QVSTFDUXA,
	PPC_INS_QVSTFDUXI,
	PPC_INS_QVSTFDUXIA,
	PPC_INS_QVSTFDX,
	PPC_INS_QVSTFDXA,
	PPC_INS_QVSTFDXI,
	PPC_INS_QVSTFDXIA,
	PPC_INS_QVSTFIWX,
	PPC_INS_QVSTFIWXA,
	PPC_INS_QVSTFSUX,
	PPC_INS_QVSTFSUXA,
	PPC_INS_QVSTFSUXI,
	PPC_INS_QVSTFSUXIA,
	PPC_INS_QVSTFSX,
	PPC_INS_QVSTFSXA,
	PPC_INS_QVSTFSXI,
	PPC_INS_QVSTFSXIA,
	PPC_INS_RFCI,
	PPC_INS_RFDI,
	PPC_INS_RFEBB,
	PPC_INS_RFI,
	PPC_INS_RFID,
	PPC_INS_RFMCI,
	PPC_INS_RLDCL,
	PPC_INS_RLDCR,
	PPC_INS_RLDIC,
	PPC_INS_RLDICL,
	PPC_INS_RLDICR,
	PPC_INS_RLDIMI,
	PPC_INS_SC,
	PPC_INS_SETB,
	PPC_INS_SETBC,
	PPC_INS_SETBCR,
	PPC_INS_SETNBC,
	PPC_INS_SETNBCR,
	PPC_INS_SLBFEE,
	PPC_INS_SLBIA,
	PPC_INS_SLBIE,
	PPC_INS_SLBIEG,
	PPC_INS_SLBMFEE,
	PPC_INS_SLBMFEV,
	PPC_INS_SLBMTE,
	PPC_INS_SLBSYNC,
	PPC_INS_SLD,
	PPC_INS_SLW,
	PPC_INS_STW,
	PPC_INS_STWX,
	PPC_INS_SRAD,
	PPC_INS_SRADI,
	PPC_INS_SRAW,
	PPC_INS_SRAWI,
	PPC_INS_SRD,
	PPC_INS_SRW,
	PPC_INS_STB,
	PPC_INS_STBCIX,
	PPC_INS_STBCX,
	PPC_INS_STBEPX,
	PPC_INS_STBU,
	PPC_INS_STBUX,
	PPC_INS_STBX,
	PPC_INS_STD,
	PPC_INS_STDAT,
	PPC_INS_STDBRX,
	PPC_INS_STDCIX,
	PPC_INS_STDCX,
	PPC_INS_STDU,
	PPC_INS_STDUX,
	PPC_INS_STDX,
	PPC_INS_STFD,
	PPC_INS_STFDEPX,
	PPC_INS_STFDU,
	PPC_INS_STFDUX,
	PPC_INS_STFDX,
	PPC_INS_STFIWX,
	PPC_INS_STFS,
	PPC_INS_STFSU,
	PPC_INS_STFSUX,
	PPC_INS_STFSX,
	PPC_INS_STH,
	PPC_INS_STHBRX,
	PPC_INS_STHCIX,
	PPC_INS_STHCX,
	PPC_INS_STHEPX,
	PPC_INS_STHU,
	PPC_INS_STHUX,
	PPC_INS_STHX,
	PPC_INS_STMW,
	PPC_INS_STOP,
	PPC_INS_STQ,
	PPC_INS_STQCX,
	PPC_INS_STSWI,
	PPC_INS_STVEBX,
	PPC_INS_STVEHX,
	PPC_INS_STVEWX,
	PPC_INS_STVX,
	PPC_INS_STVXL,
	PPC_INS_STWAT,
	PPC_INS_STWBRX,
	PPC_INS_STWCIX,
	PPC_INS_STWCX,
	PPC_INS_STWEPX,
	PPC_INS_STWU,
	PPC_INS_STWUX,
	PPC_INS_STXSD,
	PPC_INS_STXSDX,
	PPC_INS_STXSIBX,
	PPC_INS_STXSIHX,
	PPC_INS_STXSIWX,
	PPC_INS_STXSSP,
	PPC_INS_STXSSPX,
	PPC_INS_STXV,
	PPC_INS_STXVB16X,
	PPC_INS_STXVD2X,
	PPC_INS_STXVH8X,
	PPC_INS_STXVL,
	PPC_INS_STXVLL,
	PPC_INS_STXVP,
	PPC_INS_STXVPRL,
	PPC_INS_STXVPRLL,
	PPC_INS_STXVPX,
	PPC_INS_STXVRBX,
	PPC_INS_STXVRDX,
	PPC_INS_STXVRHX,
	PPC_INS_STXVRL,
	PPC_INS_STXVRLL,
	PPC_INS_STXVRWX,
	PPC_INS_STXVW4X,
	PPC_INS_STXVX,
	PPC_INS_SUBF,
	PPC_INS_SUBFC,
	PPC_INS_SUBFCO,
	PPC_INS_SUBFE,
	PPC_INS_SUBFEO,
	PPC_INS_SUBFIC,
	PPC_INS_SUBFME,
	PPC_INS_SUBFMEO,
	PPC_INS_SUBFO,
	PPC_INS_SUBFUS,
	PPC_INS_SUBFZE,
	PPC_INS_SUBFZEO,
	PPC_INS_SYNC,
	PPC_INS_TABORT,
	PPC_INS_TABORTDC,
	PPC_INS_TABORTDCI,
	PPC_INS_TABORTWC,
	PPC_INS_TABORTWCI,
	PPC_INS_TBEGIN,
	PPC_INS_TCHECK,
	PPC_INS_TD,
	PPC_INS_TDI,
	PPC_INS_TEND,
	PPC_INS_TLBIA,
	PPC_INS_TLBIE,
	PPC_INS_TLBIEL,
	PPC_INS_TLBIVAX,
	PPC_INS_TLBLD,
	PPC_INS_TLBLI,
	PPC_INS_TLBRE,
	PPC_INS_TLBSX,
	PPC_INS_TLBSYNC,
	PPC_INS_TLBWE,
	PPC_INS_TRAP,
	PPC_INS_TRECHKPT,
	PPC_INS_TRECLAIM,
	PPC_INS_TSR,
	PPC_INS_TW,
	PPC_INS_TWI,
	PPC_INS_VABSDUB,
	PPC_INS_VABSDUH,
	PPC_INS_VABSDUW,
	PPC_INS_VADDCUQ,
	PPC_INS_VADDCUW,
	PPC_INS_VADDECUQ,
	PPC_INS_VADDEUQM,
	PPC_INS_VADDFP,
	PPC_INS_VADDSBS,
	PPC_INS_VADDSHS,
	PPC_INS_VADDSWS,
	PPC_INS_VADDUBM,
	PPC_INS_VADDUBS,
	PPC_INS_VADDUDM,
	PPC_INS_VADDUHM,
	PPC_INS_VADDUHS,
	PPC_INS_VADDUQM,
	PPC_INS_VADDUWM,
	PPC_INS_VADDUWS,
	PPC_INS_VAND,
	PPC_INS_VANDC,
	PPC_INS_VAVGSB,
	PPC_INS_VAVGSH,
	PPC_INS_VAVGSW,
	PPC_INS_VAVGUB,
	PPC_INS_VAVGUH,
	PPC_INS_VAVGUW,
	PPC_INS_VBPERMD,
	PPC_INS_VBPERMQ,
	PPC_INS_VCFSX,
	PPC_INS_VCFUGED,
	PPC_INS_VCFUX,
	PPC_INS_VCIPHER,
	PPC_INS_VCIPHERLAST,
	PPC_INS_VCLRLB,
	PPC_INS_VCLRRB,
	PPC_INS_VCLZB,
	PPC_INS_VCLZD,
	PPC_INS_VCLZDM,
	PPC_INS_VCLZH,
	PPC_INS_VCLZLSBB,
	PPC_INS_VCLZW,
	PPC_INS_VCMPBFP,
	PPC_INS_VCMPEQFP,
	PPC_INS_VCMPEQUB,
	PPC_INS_VCMPEQUD,
	PPC_INS_VCMPEQUH,
	PPC_INS_VCMPEQUQ,
	PPC_INS_VCMPEQUW,
	PPC_INS_VCMPGEFP,
	PPC_INS_VCMPGTFP,
	PPC_INS_VCMPGTSB,
	PPC_INS_VCMPGTSD,
	PPC_INS_VCMPGTSH,
	PPC_INS_VCMPGTSQ,
	PPC_INS_VCMPGTSW,
	PPC_INS_VCMPGTUB,
	PPC_INS_VCMPGTUD,
	PPC_INS_VCMPGTUH,
	PPC_INS_VCMPGTUQ,
	PPC_INS_VCMPGTUW,
	PPC_INS_VCMPNEB,
	PPC_INS_VCMPNEH,
	PPC_INS_VCMPNEW,
	PPC_INS_VCMPNEZB,
	PPC_INS_VCMPNEZH,
	PPC_INS_VCMPNEZW,
	PPC_INS_VCMPSQ,
	PPC_INS_VCMPUQ,
	PPC_INS_VCNTMBB,
	PPC_INS_VCNTMBD,
	PPC_INS_VCNTMBH,
	PPC_INS_VCNTMBW,
	PPC_INS_VCTSXS,
	PPC_INS_VCTUXS,
	PPC_INS_VCTZB,
	PPC_INS_VCTZD,
	PPC_INS_VCTZDM,
	PPC_INS_VCTZH,
	PPC_INS_VCTZLSBB,
	PPC_INS_VCTZW,
	PPC_INS_VDIVESD,
	PPC_INS_VDIVESQ,
	PPC_INS_VDIVESW,
	PPC_INS_VDIVEUD,
	PPC_INS_VDIVEUQ,
	PPC_INS_VDIVEUW,
	PPC_INS_VDIVSD,
	PPC_INS_VDIVSQ,
	PPC_INS_VDIVSW,
	PPC_INS_VDIVUD,
	PPC_INS_VDIVUQ,
	PPC_INS_VDIVUW,
	PPC_INS_VEQV,
	PPC_INS_VEXPANDBM,
	PPC_INS_VEXPANDDM,
	PPC_INS_VEXPANDHM,
	PPC_INS_VEXPANDQM,
	PPC_INS_VEXPANDWM,
	PPC_INS_VEXPTEFP,
	PPC_INS_VEXTDDVLX,
	PPC_INS_VEXTDDVRX,
	PPC_INS_VEXTDUBVLX,
	PPC_INS_VEXTDUBVRX,
	PPC_INS_VEXTDUHVLX,
	PPC_INS_VEXTDUHVRX,
	PPC_INS_VEXTDUWVLX,
	PPC_INS_VEXTDUWVRX,
	PPC_INS_VEXTRACTBM,
	PPC_INS_VEXTRACTD,
	PPC_INS_VEXTRACTDM,
	PPC_INS_VEXTRACTHM,
	PPC_INS_VEXTRACTQM,
	PPC_INS_VEXTRACTUB,
	PPC_INS_VEXTRACTUH,
	PPC_INS_VEXTRACTUW,
	PPC_INS_VEXTRACTWM,
	PPC_INS_VEXTSB2D,
	PPC_INS_VEXTSB2W,
	PPC_INS_VEXTSD2Q,
	PPC_INS_VEXTSH2D,
	PPC_INS_VEXTSH2W,
	PPC_INS_VEXTSW2D,
	PPC_INS_VEXTUBLX,
	PPC_INS_VEXTUBRX,
	PPC_INS_VEXTUHLX,
	PPC_INS_VEXTUHRX,
	PPC_INS_VEXTUWLX,
	PPC_INS_VEXTUWRX,
	PPC_INS_VGBBD,
	PPC_INS_VGNB,
	PPC_INS_VINSBLX,
	PPC_INS_VINSBRX,
	PPC_INS_VINSBVLX,
	PPC_INS_VINSBVRX,
	PPC_INS_VINSD,
	PPC_INS_VINSDLX,
	PPC_INS_VINSDRX,
	PPC_INS_VINSERTB,
	PPC_INS_VINSERTD,
	PPC_INS_VINSERTH,
	PPC_INS_VINSERTW,
	PPC_INS_VINSHLX,
	PPC_INS_VINSHRX,
	PPC_INS_VINSHVLX,
	PPC_INS_VINSHVRX,
	PPC_INS_VINSW,
	PPC_INS_VINSWLX,
	PPC_INS_VINSWRX,
	PPC_INS_VINSWVLX,
	PPC_INS_VINSWVRX,
	PPC_INS_VLOGEFP,
	PPC_INS_VMADDFP,
	PPC_INS_VMAXFP,
	PPC_INS_VMAXSB,
	PPC_INS_VMAXSD,
	PPC_INS_VMAXSH,
	PPC_INS_VMAXSW,
	PPC_INS_VMAXUB,
	PPC_INS_VMAXUD,
	PPC_INS_VMAXUH,
	PPC_INS_VMAXUW,
	PPC_INS_VMHADDSHS,
	PPC_INS_VMHRADDSHS,
	PPC_INS_VMINFP,
	PPC_INS_VMINSB,
	PPC_INS_VMINSD,
	PPC_INS_VMINSH,
	PPC_INS_VMINSW,
	PPC_INS_VMINUB,
	PPC_INS_VMINUD,
	PPC_INS_VMINUH,
	PPC_INS_VMINUW,
	PPC_INS_VMLADDUHM,
	PPC_INS_VMODSD,
	PPC_INS_VMODSQ,
	PPC_INS_VMODSW,
	PPC_INS_VMODUD,
	PPC_INS_VMODUQ,
	PPC_INS_VMODUW,
	PPC_INS_VMRGEW,
	PPC_INS_VMRGHB,
	PPC_INS_VMRGHH,
	PPC_INS_VMRGHW,
	PPC_INS_VMRGLB,
	PPC_INS_VMRGLH,
	PPC_INS_VMRGLW,
	PPC_INS_VMRGOW,
	PPC_INS_VMSUMCUD,
	PPC_INS_VMSUMMBM,
	PPC_INS_VMSUMSHM,
	PPC_INS_VMSUMSHS,
	PPC_INS_VMSUMUBM,
	PPC_INS_VMSUMUDM,
	PPC_INS_VMSUMUHM,
	PPC_INS_VMSUMUHS,
	PPC_INS_VMUL10CUQ,
	PPC_INS_VMUL10ECUQ,
	PPC_INS_VMUL10EUQ,
	PPC_INS_VMUL10UQ,
	PPC_INS_VMULESB,
	PPC_INS_VMULESD,
	PPC_INS_VMULESH,
	PPC_INS_VMULESW,
	PPC_INS_VMULEUB,
	PPC_INS_VMULEUD,
	PPC_INS_VMULEUH,
	PPC_INS_VMULEUW,
	PPC_INS_VMULHSD,
	PPC_INS_VMULHSW,
	PPC_INS_VMULHUD,
	PPC_INS_VMULHUW,
	PPC_INS_VMULLD,
	PPC_INS_VMULOSB,
	PPC_INS_VMULOSD,
	PPC_INS_VMULOSH,
	PPC_INS_VMULOSW,
	PPC_INS_VMULOUB,
	PPC_INS_VMULOUD,
	PPC_INS_VMULOUH,
	PPC_INS_VMULOUW,
	PPC_INS_VMULUWM,
	PPC_INS_VNAND,
	PPC_INS_VNCIPHER,
	PPC_INS_VNCIPHERLAST,
	PPC_INS_VNEGD,
	PPC_INS_VNEGW,
	PPC_INS_VNMSUBFP,
	PPC_INS_VNOR,
	PPC_INS_VOR,
	PPC_INS_VORC,
	PPC_INS_VPDEPD,
	PPC_INS_VPERM,
	PPC_INS_VPERMR,
	PPC_INS_VPERMXOR,
	PPC_INS_VPEXTD,
	PPC_INS_VPKPX,
	PPC_INS_VPKSDSS,
	PPC_INS_VPKSDUS,
	PPC_INS_VPKSHSS,
	PPC_INS_VPKSHUS,
	PPC_INS_VPKSWSS,
	PPC_INS_VPKSWUS,
	PPC_INS_VPKUDUM,
	PPC_INS_VPKUDUS,
	PPC_INS_VPKUHUM,
	PPC_INS_VPKUHUS,
	PPC_INS_VPKUWUM,
	PPC_INS_VPKUWUS,
	PPC_INS_VPMSUMB,
	PPC_INS_VPMSUMD,
	PPC_INS_VPMSUMH,
	PPC_INS_VPMSUMW,
	PPC_INS_VPOPCNTB,
	PPC_INS_VPOPCNTD,
	PPC_INS_VPOPCNTH,
	PPC_INS_VPOPCNTW,
	PPC_INS_VPRTYBD,
	PPC_INS_VPRTYBQ,
	PPC_INS_VPRTYBW,
	PPC_INS_VREFP,
	PPC_INS_VRFIM,
	PPC_INS_VRFIN,
	PPC_INS_VRFIP,
	PPC_INS_VRFIZ,
	PPC_INS_VRLB,
	PPC_INS_VRLD,
	PPC_INS_VRLDMI,
	PPC_INS_VRLDNM,
	PPC_INS_VRLH,
	PPC_INS_VRLQ,
	PPC_INS_VRLQMI,
	PPC_INS_VRLQNM,
	PPC_INS_VRLW,
	PPC_INS_VRLWMI,
	PPC_INS_VRLWNM,
	PPC_INS_VRSQRTEFP,
	PPC_INS_VSBOX,
	PPC_INS_VSEL,
	PPC_INS_VSHASIGMAD,
	PPC_INS_VSHASIGMAW,
	PPC_INS_VSL,
	PPC_INS_VSLB,
	PPC_INS_VSLD,
	PPC_INS_VSLDBI,
	PPC_INS_VSLDOI,
	PPC_INS_VSLH,
	PPC_INS_VSLO,
	PPC_INS_VSLQ,
	PPC_INS_VSLV,
	PPC_INS_VSLW,
	PPC_INS_VSPLTB,
	PPC_INS_VSPLTH,
	PPC_INS_VSPLTISB,
	PPC_INS_VSPLTISH,
	PPC_INS_VSPLTISW,
	PPC_INS_VSPLTW,
	PPC_INS_VSR,
	PPC_INS_VSRAB,
	PPC_INS_VSRAD,
	PPC_INS_VSRAH,
	PPC_INS_VSRAQ,
	PPC_INS_VSRAW,
	PPC_INS_VSRB,
	PPC_INS_VSRD,
	PPC_INS_VSRDBI,
	PPC_INS_VSRH,
	PPC_INS_VSRO,
	PPC_INS_VSRQ,
	PPC_INS_VSRV,
	PPC_INS_VSRW,
	PPC_INS_VSTRIBL,
	PPC_INS_VSTRIBR,
	PPC_INS_VSTRIHL,
	PPC_INS_VSTRIHR,
	PPC_INS_VSUBCUQ,
	PPC_INS_VSUBCUW,
	PPC_INS_VSUBECUQ,
	PPC_INS_VSUBEUQM,
	PPC_INS_VSUBFP,
	PPC_INS_VSUBSBS,
	PPC_INS_VSUBSHS,
	PPC_INS_VSUBSWS,
	PPC_INS_VSUBUBM,
	PPC_INS_VSUBUBS,
	PPC_INS_VSUBUDM,
	PPC_INS_VSUBUHM,
	PPC_INS_VSUBUHS,
	PPC_INS_VSUBUQM,
	PPC_INS_VSUBUWM,
	PPC_INS_VSUBUWS,
	PPC_INS_VSUM2SWS,
	PPC_INS_VSUM4SBS,
	PPC_INS_VSUM4SHS,
	PPC_INS_VSUM4UBS,
	PPC_INS_VSUMSWS,
	PPC_INS_VUPKHPX,
	PPC_INS_VUPKHSB,
	PPC_INS_VUPKHSH,
	PPC_INS_VUPKHSW,
	PPC_INS_VUPKLPX,
	PPC_INS_VUPKLSB,
	PPC_INS_VUPKLSH,
	PPC_INS_VUPKLSW,
	PPC_INS_VXOR,
	PPC_INS_WAIT,
	PPC_INS_WRTEE,
	PPC_INS_WRTEEI,
	PPC_INS_XOR,
	PPC_INS_XORI,
	PPC_INS_XORIS,
	PPC_INS_XSABSDP,
	PPC_INS_XSABSQP,
	PPC_INS_XSADDDP,
	PPC_INS_XSADDQP,
	PPC_INS_XSADDQPO,
	PPC_INS_XSADDSP,
	PPC_INS_XSCMPEQDP,
	PPC_INS_XSCMPEQQP,
	PPC_INS_XSCMPEXPDP,
	PPC_INS_XSCMPEXPQP,
	PPC_INS_XSCMPGEDP,
	PPC_INS_XSCMPGEQP,
	PPC_INS_XSCMPGTDP,
	PPC_INS_XSCMPGTQP,
	PPC_INS_XSCMPODP,
	PPC_INS_XSCMPOQP,
	PPC_INS_XSCMPUDP,
	PPC_INS_XSCMPUQP,
	PPC_INS_XSCPSGNDP,
	PPC_INS_XSCPSGNQP,
	PPC_INS_XSCVDPHP,
	PPC_INS_XSCVDPQP,
	PPC_INS_XSCVDPSP,
	PPC_INS_XSCVDPSPN,
	PPC_INS_XSCVDPSXDS,
	PPC_INS_XSCVDPSXWS,
	PPC_INS_XSCVDPUXDS,
	PPC_INS_XSCVDPUXWS,
	PPC_INS_XSCVHPDP,
	PPC_INS_XSCVQPDP,
	PPC_INS_XSCVQPDPO,
	PPC_INS_XSCVQPSDZ,
	PPC_INS_XSCVQPSQZ,
	PPC_INS_XSCVQPSWZ,
	PPC_INS_XSCVQPUDZ,
	PPC_INS_XSCVQPUQZ,
	PPC_INS_XSCVQPUWZ,
	PPC_INS_XSCVSDQP,
	PPC_INS_XSCVSPDP,
	PPC_INS_XSCVSPDPN,
	PPC_INS_XSCVSQQP,
	PPC_INS_XSCVSXDDP,
	PPC_INS_XSCVSXDSP,
	PPC_INS_XSCVUDQP,
	PPC_INS_XSCVUQQP,
	PPC_INS_XSCVUXDDP,
	PPC_INS_XSCVUXDSP,
	PPC_INS_XSDIVDP,
	PPC_INS_XSDIVQP,
	PPC_INS_XSDIVQPO,
	PPC_INS_XSDIVSP,
	PPC_INS_XSIEXPDP,
	PPC_INS_XSIEXPQP,
	PPC_INS_XSMADDADP,
	PPC_INS_XSMADDASP,
	PPC_INS_XSMADDMDP,
	PPC_INS_XSMADDMSP,
	PPC_INS_XSMADDQP,
	PPC_INS_XSMADDQPO,
	PPC_INS_XSMAXCDP,
	PPC_INS_XSMAXCQP,
	PPC_INS_XSMAXDP,
	PPC_INS_XSMAXJDP,
	PPC_INS_XSMINCDP,
	PPC_INS_XSMINCQP,
	PPC_INS_XSMINDP,
	PPC_INS_XSMINJDP,
	PPC_INS_XSMSUBADP,
	PPC_INS_XSMSUBASP,
	PPC_INS_XSMSUBMDP,
	PPC_INS_XSMSUBMSP,
	PPC_INS_XSMSUBQP,
	PPC_INS_XSMSUBQPO,
	PPC_INS_XSMULDP,
	PPC_INS_XSMULQP,
	PPC_INS_XSMULQPO,
	PPC_INS_XSMULSP,
	PPC_INS_XSNABSDP,
	PPC_INS_XSNABSQP,
	PPC_INS_XSNEGDP,
	PPC_INS_XSNEGQP,
	PPC_INS_XSNMADDADP,
	PPC_INS_XSNMADDASP,
	PPC_INS_XSNMADDMDP,
	PPC_INS_XSNMADDMSP,
	PPC_INS_XSNMADDQP,
	PPC_INS_XSNMADDQPO,
	PPC_INS_XSNMSUBADP,
	PPC_INS_XSNMSUBASP,
	PPC_INS_XSNMSUBMDP,
	PPC_INS_XSNMSUBMSP,
	PPC_INS_XSNMSUBQP,
	PPC_INS_XSNMSUBQPO,
	PPC_INS_XSRDPI,
	PPC_INS_XSRDPIC,
	PPC_INS_XSRDPIM,
	PPC_INS_XSRDPIP,
	PPC_INS_XSRDPIZ,
	PPC_INS_XSREDP,
	PPC_INS_XSRESP,
	PPC_INS_XSRQPI,
	PPC_INS_XSRQPIX,
	PPC_INS_XSRQPXP,
	PPC_INS_XSRSP,
	PPC_INS_XSRSQRTEDP,
	PPC_INS_XSRSQRTESP,
	PPC_INS_XSSQRTDP,
	PPC_INS_XSSQRTQP,
	PPC_INS_XSSQRTQPO,
	PPC_INS_XSSQRTSP,
	PPC_INS_XSSUBDP,
	PPC_INS_XSSUBQP,
	PPC_INS_XSSUBQPO,
	PPC_INS_XSSUBSP,
	PPC_INS_XSTDIVDP,
	PPC_INS_XSTSQRTDP,
	PPC_INS_XSTSTDCDP,
	PPC_INS_XSTSTDCQP,
	PPC_INS_XSTSTDCSP,
	PPC_INS_XSXEXPDP,
	PPC_INS_XSXEXPQP,
	PPC_INS_XSXSIGDP,
	PPC_INS_XSXSIGQP,
	PPC_INS_XVABSDP,
	PPC_INS_XVABSSP,
	PPC_INS_XVADDDP,
	PPC_INS_XVADDSP,
	PPC_INS_XVBF16GER2,
	PPC_INS_XVBF16GER2NN,
	PPC_INS_XVBF16GER2NP,
	PPC_INS_XVBF16GER2PN,
	PPC_INS_XVBF16GER2PP,
	PPC_INS_XVCMPEQDP,
	PPC_INS_XVCMPEQSP,
	PPC_INS_XVCMPGEDP,
	PPC_INS_XVCMPGESP,
	PPC_INS_XVCMPGTDP,
	PPC_INS_XVCMPGTSP,
	PPC_INS_XVCPSGNDP,
	PPC_INS_XVCPSGNSP,
	PPC_INS_XVCVBF16SPN,
	PPC_INS_XVCVDPSP,
	PPC_INS_XVCVDPSXDS,
	PPC_INS_XVCVDPSXWS,
	PPC_INS_XVCVDPUXDS,
	PPC_INS_XVCVDPUXWS,
	PPC_INS_XVCVHPSP,
	PPC_INS_XVCVSPBF16,
	PPC_INS_XVCVSPDP,
	PPC_INS_XVCVSPHP,
	PPC_INS_XVCVSPSXDS,
	PPC_INS_XVCVSPSXWS,
	PPC_INS_XVCVSPUXDS,
	PPC_INS_XVCVSPUXWS,
	PPC_INS_XVCVSXDDP,
	PPC_INS_XVCVSXDSP,
	PPC_INS_XVCVSXWDP,
	PPC_INS_XVCVSXWSP,
	PPC_INS_XVCVUXDDP,
	PPC_INS_XVCVUXDSP,
	PPC_INS_XVCVUXWDP,
	PPC_INS_XVCVUXWSP,
	PPC_INS_XVDIVDP,
	PPC_INS_XVDIVSP,
	PPC_INS_XVF16GER2,
	PPC_INS_XVF16GER2NN,
	PPC_INS_XVF16GER2NP,
	PPC_INS_XVF16GER2PN,
	PPC_INS_XVF16GER2PP,
	PPC_INS_XVF32GER,
	PPC_INS_XVF32GERNN,
	PPC_INS_XVF32GERNP,
	PPC_INS_XVF32GERPN,
	PPC_INS_XVF32GERPP,
	PPC_INS_XVF64GER,
	PPC_INS_XVF64GERNN,
	PPC_INS_XVF64GERNP,
	PPC_INS_XVF64GERPN,
	PPC_INS_XVF64GERPP,
	PPC_INS_XVI16GER2,
	PPC_INS_XVI16GER2PP,
	PPC_INS_XVI16GER2S,
	PPC_INS_XVI16GER2SPP,
	PPC_INS_XVI4GER8,
	PPC_INS_XVI4GER8PP,
	PPC_INS_XVI8GER4,
	PPC_INS_XVI8GER4PP,
	PPC_INS_XVI8GER4SPP,
	PPC_INS_XVIEXPDP,
	PPC_INS_XVIEXPSP,
	PPC_INS_XVMADDADP,
	PPC_INS_XVMADDASP,
	PPC_INS_XVMADDMDP,
	PPC_INS_XVMADDMSP,
	PPC_INS_XVMAXDP,
	PPC_INS_XVMAXSP,
	PPC_INS_XVMINDP,
	PPC_INS_XVMINSP,
	PPC_INS_XVMSUBADP,
	PPC_INS_XVMSUBASP,
	PPC_INS_XVMSUBMDP,
	PPC_INS_XVMSUBMSP,
	PPC_INS_XVMULDP,
	PPC_INS_XVMULSP,
	PPC_INS_XVNABSDP,
	PPC_INS_XVNABSSP,
	PPC_INS_XVNEGDP,
	PPC_INS_XVNEGSP,
	PPC_INS_XVNMADDADP,
	PPC_INS_XVNMADDASP,
	PPC_INS_XVNMADDMDP,
	PPC_INS_XVNMADDMSP,
	PPC_INS_XVNMSUBADP,
	PPC_INS_XVNMSUBASP,
	PPC_INS_XVNMSUBMDP,
	PPC_INS_XVNMSUBMSP,
	PPC_INS_XVRDPI,
	PPC_INS_XVRDPIC,
	PPC_INS_XVRDPIM,
	PPC_INS_XVRDPIP,
	PPC_INS_XVRDPIZ,
	PPC_INS_XVREDP,
	PPC_INS_XVRESP,
	PPC_INS_XVRSPI,
	PPC_INS_XVRSPIC,
	PPC_INS_XVRSPIM,
	PPC_INS_XVRSPIP,
	PPC_INS_XVRSPIZ,
	PPC_INS_XVRSQRTEDP,
	PPC_INS_XVRSQRTESP,
	PPC_INS_XVSQRTDP,
	PPC_INS_XVSQRTSP,
	PPC_INS_XVSUBDP,
	PPC_INS_XVSUBSP,
	PPC_INS_XVTDIVDP,
	PPC_INS_XVTDIVSP,
	PPC_INS_XVTLSBB,
	PPC_INS_XVTSQRTDP,
	PPC_INS_XVTSQRTSP,
	PPC_INS_XVTSTDCDP,
	PPC_INS_XVTSTDCSP,
	PPC_INS_XVXEXPDP,
	PPC_INS_XVXEXPSP,
	PPC_INS_XVXSIGDP,
	PPC_INS_XVXSIGSP,
	PPC_INS_XXBLENDVB,
	PPC_INS_XXBLENDVD,
	PPC_INS_XXBLENDVH,
	PPC_INS_XXBLENDVW,
	PPC_INS_XXBRD,
	PPC_INS_XXBRH,
	PPC_INS_XXBRQ,
	PPC_INS_XXBRW,
	PPC_INS_XXEVAL,
	PPC_INS_XXEXTRACTUW,
	PPC_INS_XXGENPCVBM,
	PPC_INS_XXGENPCVDM,
	PPC_INS_XXGENPCVHM,
	PPC_INS_XXGENPCVWM,
	PPC_INS_XXINSERTW,
	PPC_INS_XXLAND,
	PPC_INS_XXLANDC,
	PPC_INS_XXLEQV,
	PPC_INS_XXLNAND,
	PPC_INS_XXLNOR,
	PPC_INS_XXLOR,
	PPC_INS_XXLORC,
	PPC_INS_XXLXOR,
	PPC_INS_XXMFACC,
	PPC_INS_XXMRGHW,
	PPC_INS_XXMRGLW,
	PPC_INS_XXMTACC,
	PPC_INS_XXPERM,
	PPC_INS_XXPERMDI,
	PPC_INS_XXPERMR,
	PPC_INS_XXPERMX,
	PPC_INS_XXSEL,
	PPC_INS_XXSETACCZ,
	PPC_INS_XXSLDWI,
	PPC_INS_XXSPLTI32DX,
	PPC_INS_XXSPLTIB,
	PPC_INS_XXSPLTIDP,
	PPC_INS_XXSPLTIW,
	PPC_INS_XXSPLTW,
	PPC_INS_BC,
	PPC_INS_BCA,
	PPC_INS_BCCTR,
	PPC_INS_BCCTRL,
	PPC_INS_BCL,
	PPC_INS_BCLA,
	PPC_INS_BCLR,
	PPC_INS_BCLRL,

	// clang-format on
	// generated content <PPCGenCSInsnEnum.inc> end

	PPC_INS_ENDING,

	PPC_INS_ALIAS_BEGIN,
	// generated content <PPCGenCSAliasEnum.inc> begin
	// clang-format off

	PPC_INS_ALIAS_RFEBB, // Real instr.: PPC_RFEBB
	PPC_INS_ALIAS_LI, // Real instr.: PPC_ADDI
	PPC_INS_ALIAS_LIS, // Real instr.: PPC_ADDIS
	PPC_INS_ALIAS_MR, // Real instr.: PPC_OR
	PPC_INS_ALIAS_MR_, // Real instr.: PPC_OR_rec
	PPC_INS_ALIAS_NOT, // Real instr.: PPC_NOR
	PPC_INS_ALIAS_NOT_, // Real instr.: PPC_NOR_rec
	PPC_INS_ALIAS_NOP, // Real instr.: PPC_ORI
	PPC_INS_ALIAS_MTUDSCR, // Real instr.: PPC_MTUDSCR
	PPC_INS_ALIAS_MFUDSCR, // Real instr.: PPC_MFUDSCR
	PPC_INS_ALIAS_MTVRSAVE, // Real instr.: PPC_MTVRSAVE
	PPC_INS_ALIAS_MFVRSAVE, // Real instr.: PPC_MFVRSAVE
	PPC_INS_ALIAS_MTCR, // Real instr.: PPC_MTCRF
	PPC_INS_ALIAS_SUB, // Real instr.: PPC_SUBF
	PPC_INS_ALIAS_SUB_, // Real instr.: PPC_SUBF_rec
	PPC_INS_ALIAS_SUBC, // Real instr.: PPC_SUBFC
	PPC_INS_ALIAS_SUBC_, // Real instr.: PPC_SUBFC_rec
	PPC_INS_ALIAS_VMR, // Real instr.: PPC_VOR
	PPC_INS_ALIAS_VNOT, // Real instr.: PPC_VNOR
	PPC_INS_ALIAS_ROTLWI, // Real instr.: PPC_RLWINM8
	PPC_INS_ALIAS_ROTLWI_, // Real instr.: PPC_RLWINM8_rec
	PPC_INS_ALIAS_ROTLW, // Real instr.: PPC_RLWNM8
	PPC_INS_ALIAS_ROTLW_, // Real instr.: PPC_RLWNM8_rec
	PPC_INS_ALIAS_CLRLWI, // Real instr.: PPC_RLWINM8
	PPC_INS_ALIAS_CLRLWI_, // Real instr.: PPC_RLWINM8_rec
	PPC_INS_ALIAS_ISELLT, // Real instr.: PPC_ISEL8
	PPC_INS_ALIAS_ISELGT, // Real instr.: PPC_ISEL8
	PPC_INS_ALIAS_ISELEQ, // Real instr.: PPC_ISEL8
	PPC_INS_ALIAS_XNOP, // Real instr.: PPC_XORI8
	PPC_INS_ALIAS_CNTLZW, // Real instr.: PPC_CNTLZW8
	PPC_INS_ALIAS_CNTLZW_, // Real instr.: PPC_CNTLZW8_rec
	PPC_INS_ALIAS_MTXER, // Real instr.: PPC_MTSPR8
	PPC_INS_ALIAS_MFXER, // Real instr.: PPC_MFSPR8
	PPC_INS_ALIAS_MFRTCU, // Real instr.: PPC_MFSPR8
	PPC_INS_ALIAS_MFRTCL, // Real instr.: PPC_MFSPR8
	PPC_INS_ALIAS_MTLR, // Real instr.: PPC_MTSPR8
	PPC_INS_ALIAS_MFLR, // Real instr.: PPC_MFSPR8
	PPC_INS_ALIAS_MTCTR, // Real instr.: PPC_MTSPR8
	PPC_INS_ALIAS_MFCTR, // Real instr.: PPC_MFSPR8
	PPC_INS_ALIAS_MTUAMR, // Real instr.: PPC_MTSPR8
	PPC_INS_ALIAS_MFUAMR, // Real instr.: PPC_MFSPR8
	PPC_INS_ALIAS_MTDSCR, // Real instr.: PPC_MTSPR8
	PPC_INS_ALIAS_MFDSCR, // Real instr.: PPC_MFSPR8
	PPC_INS_ALIAS_MTDSISR, // Real instr.: PPC_MTSPR8
	PPC_INS_ALIAS_MFDSISR, // Real instr.: PPC_MFSPR8
	PPC_INS_ALIAS_MTDAR, // Real instr.: PPC_MTSPR8
	PPC_INS_ALIAS_MFDAR, // Real instr.: PPC_MFSPR8
	PPC_INS_ALIAS_MTDEC, // Real instr.: PPC_MTSPR8
	PPC_INS_ALIAS_MFDEC, // Real instr.: PPC_MFSPR8
	PPC_INS_ALIAS_MTSDR1, // Real instr.: PPC_MTSPR8
	PPC_INS_ALIAS_MFSDR1, // Real instr.: PPC_MFSPR8
	PPC_INS_ALIAS_MTSRR0, // Real instr.: PPC_MTSPR8
	PPC_INS_ALIAS_MFSRR0, // Real instr.: PPC_MFSPR8
	PPC_INS_ALIAS_MTSRR1, // Real instr.: PPC_MTSPR8
	PPC_INS_ALIAS_MFSRR1, // Real instr.: PPC_MFSPR8
	PPC_INS_ALIAS_MTCFAR, // Real instr.: PPC_MTSPR8
	PPC_INS_ALIAS_MFCFAR, // Real instr.: PPC_MFSPR8
	PPC_INS_ALIAS_MTAMR, // Real instr.: PPC_MTSPR8
	PPC_INS_ALIAS_MFAMR, // Real instr.: PPC_MFSPR8
	PPC_INS_ALIAS_MFSPRG, // Real instr.: PPC_MFSPR8
	PPC_INS_ALIAS_MFSPRG0, // Real instr.: PPC_MFSPR8
	PPC_INS_ALIAS_MTSPRG, // Real instr.: PPC_MTSPR8
	PPC_INS_ALIAS_MTSPRG0, // Real instr.: PPC_MTSPR8
	PPC_INS_ALIAS_MFSPRG1, // Real instr.: PPC_MFSPR8
	PPC_INS_ALIAS_MTSPRG1, // Real instr.: PPC_MTSPR8
	PPC_INS_ALIAS_MFSPRG2, // Real instr.: PPC_MFSPR8
	PPC_INS_ALIAS_MTSPRG2, // Real instr.: PPC_MTSPR8
	PPC_INS_ALIAS_MFSPRG3, // Real instr.: PPC_MFSPR8
	PPC_INS_ALIAS_MTSPRG3, // Real instr.: PPC_MTSPR8
	PPC_INS_ALIAS_MFASR, // Real instr.: PPC_MFSPR8
	PPC_INS_ALIAS_MTASR, // Real instr.: PPC_MTSPR8
	PPC_INS_ALIAS_MTTBL, // Real instr.: PPC_MTSPR8
	PPC_INS_ALIAS_MTTBU, // Real instr.: PPC_MTSPR8
	PPC_INS_ALIAS_MFPVR, // Real instr.: PPC_MFSPR8
	PPC_INS_ALIAS_MFSPEFSCR, // Real instr.: PPC_MFSPR8
	PPC_INS_ALIAS_MTSPEFSCR, // Real instr.: PPC_MTSPR8
	PPC_INS_ALIAS_XVMOVDP, // Real instr.: PPC_XVCPSGNDP
	PPC_INS_ALIAS_XVMOVSP, // Real instr.: PPC_XVCPSGNSP
	PPC_INS_ALIAS_XXSPLTD, // Real instr.: PPC_XXPERMDI
	PPC_INS_ALIAS_XXMRGHD, // Real instr.: PPC_XXPERMDI
	PPC_INS_ALIAS_XXMRGLD, // Real instr.: PPC_XXPERMDI
	PPC_INS_ALIAS_XXSWAPD, // Real instr.: PPC_XXPERMDI
	PPC_INS_ALIAS_MFFPRD, // Real instr.: PPC_MFVSRD
	PPC_INS_ALIAS_MTFPRD, // Real instr.: PPC_MTVSRD
	PPC_INS_ALIAS_MFFPRWZ, // Real instr.: PPC_MFVSRWZ
	PPC_INS_ALIAS_MTFPRWA, // Real instr.: PPC_MTVSRWA
	PPC_INS_ALIAS_MTFPRWZ, // Real instr.: PPC_MTVSRWZ
	PPC_INS_ALIAS_TEND_, // Real instr.: PPC_TEND
	PPC_INS_ALIAS_TENDALL_, // Real instr.: PPC_TEND
	PPC_INS_ALIAS_TSUSPEND_, // Real instr.: PPC_TSR
	PPC_INS_ALIAS_TRESUME_, // Real instr.: PPC_TSR
	PPC_INS_ALIAS_DCI, // Real instr.: PPC_DCCCI
	PPC_INS_ALIAS_DCCCI, // Real instr.: PPC_DCCCI
	PPC_INS_ALIAS_ICI, // Real instr.: PPC_ICCCI
	PPC_INS_ALIAS_ICCCI, // Real instr.: PPC_ICCCI
	PPC_INS_ALIAS_MTFSFI, // Real instr.: PPC_MTFSFI
	PPC_INS_ALIAS_MTFSFI_, // Real instr.: PPC_MTFSFI_rec
	PPC_INS_ALIAS_MTFSF, // Real instr.: PPC_MTFSF
	PPC_INS_ALIAS_MTFSF_, // Real instr.: PPC_MTFSF_rec
	PPC_INS_ALIAS_SC, // Real instr.: PPC_SC
	PPC_INS_ALIAS_SYNC, // Real instr.: PPC_SYNC
	PPC_INS_ALIAS_LWSYNC, // Real instr.: PPC_SYNC
	PPC_INS_ALIAS_PTESYNC, // Real instr.: PPC_SYNC
	PPC_INS_ALIAS_WAIT, // Real instr.: PPC_WAIT
	PPC_INS_ALIAS_WAITRSV, // Real instr.: PPC_WAIT
	PPC_INS_ALIAS_WAITIMPL, // Real instr.: PPC_WAIT
	PPC_INS_ALIAS_MBAR, // Real instr.: PPC_MBAR
	PPC_INS_ALIAS_CRSET, // Real instr.: PPC_CREQV
	PPC_INS_ALIAS_CRCLR, // Real instr.: PPC_CRXOR
	PPC_INS_ALIAS_CRMOVE, // Real instr.: PPC_CROR
	PPC_INS_ALIAS_CRNOT, // Real instr.: PPC_CRNOR
	PPC_INS_ALIAS_MFTB, // Real instr.: PPC_MFTB
	PPC_INS_ALIAS_MFTBL, // Real instr.: PPC_MFTB
	PPC_INS_ALIAS_MFTBU, // Real instr.: PPC_MFTB
	PPC_INS_ALIAS_MFBR0, // Real instr.: PPC_MFDCR
	PPC_INS_ALIAS_MTBR0, // Real instr.: PPC_MTDCR
	PPC_INS_ALIAS_MFBR1, // Real instr.: PPC_MFDCR
	PPC_INS_ALIAS_MTBR1, // Real instr.: PPC_MTDCR
	PPC_INS_ALIAS_MFBR2, // Real instr.: PPC_MFDCR
	PPC_INS_ALIAS_MTBR2, // Real instr.: PPC_MTDCR
	PPC_INS_ALIAS_MFBR3, // Real instr.: PPC_MFDCR
	PPC_INS_ALIAS_MTBR3, // Real instr.: PPC_MTDCR
	PPC_INS_ALIAS_MFBR4, // Real instr.: PPC_MFDCR
	PPC_INS_ALIAS_MTBR4, // Real instr.: PPC_MTDCR
	PPC_INS_ALIAS_MFBR5, // Real instr.: PPC_MFDCR
	PPC_INS_ALIAS_MTBR5, // Real instr.: PPC_MTDCR
	PPC_INS_ALIAS_MFBR6, // Real instr.: PPC_MFDCR
	PPC_INS_ALIAS_MTBR6, // Real instr.: PPC_MTDCR
	PPC_INS_ALIAS_MFBR7, // Real instr.: PPC_MFDCR
	PPC_INS_ALIAS_MTBR7, // Real instr.: PPC_MTDCR
	PPC_INS_ALIAS_MTMSRD, // Real instr.: PPC_MTMSRD
	PPC_INS_ALIAS_MTMSR, // Real instr.: PPC_MTMSR
	PPC_INS_ALIAS_MTPID, // Real instr.: PPC_MTSPR
	PPC_INS_ALIAS_MFPID, // Real instr.: PPC_MFSPR
	PPC_INS_ALIAS_MFSPRG4, // Real instr.: PPC_MFSPR
	PPC_INS_ALIAS_MTSPRG4, // Real instr.: PPC_MTSPR
	PPC_INS_ALIAS_MFSPRG5, // Real instr.: PPC_MFSPR
	PPC_INS_ALIAS_MTSPRG5, // Real instr.: PPC_MTSPR
	PPC_INS_ALIAS_MFSPRG6, // Real instr.: PPC_MFSPR
	PPC_INS_ALIAS_MTSPRG6, // Real instr.: PPC_MTSPR
	PPC_INS_ALIAS_MFSPRG7, // Real instr.: PPC_MFSPR
	PPC_INS_ALIAS_MTSPRG7, // Real instr.: PPC_MTSPR
	PPC_INS_ALIAS_MTDBATU, // Real instr.: PPC_MTSPR
	PPC_INS_ALIAS_MFDBATU, // Real instr.: PPC_MFSPR
	PPC_INS_ALIAS_MTDBATL, // Real instr.: PPC_MTSPR
	PPC_INS_ALIAS_MFDBATL, // Real instr.: PPC_MFSPR
	PPC_INS_ALIAS_MTIBATU, // Real instr.: PPC_MTSPR
	PPC_INS_ALIAS_MFIBATU, // Real instr.: PPC_MFSPR
	PPC_INS_ALIAS_MTIBATL, // Real instr.: PPC_MTSPR
	PPC_INS_ALIAS_MFIBATL, // Real instr.: PPC_MFSPR
	PPC_INS_ALIAS_MTPPR, // Real instr.: PPC_MTSPR
	PPC_INS_ALIAS_MFPPR, // Real instr.: PPC_MFSPR
	PPC_INS_ALIAS_MTESR, // Real instr.: PPC_MTSPR
	PPC_INS_ALIAS_MFESR, // Real instr.: PPC_MFSPR
	PPC_INS_ALIAS_MTDEAR, // Real instr.: PPC_MTSPR
	PPC_INS_ALIAS_MFDEAR, // Real instr.: PPC_MFSPR
	PPC_INS_ALIAS_MTTCR, // Real instr.: PPC_MTSPR
	PPC_INS_ALIAS_MFTCR, // Real instr.: PPC_MFSPR
	PPC_INS_ALIAS_MFTBHI, // Real instr.: PPC_MFSPR
	PPC_INS_ALIAS_MTTBHI, // Real instr.: PPC_MTSPR
	PPC_INS_ALIAS_MFTBLO, // Real instr.: PPC_MFSPR
	PPC_INS_ALIAS_MTTBLO, // Real instr.: PPC_MTSPR
	PPC_INS_ALIAS_MTSRR2, // Real instr.: PPC_MTSPR
	PPC_INS_ALIAS_MFSRR2, // Real instr.: PPC_MFSPR
	PPC_INS_ALIAS_MTSRR3, // Real instr.: PPC_MTSPR
	PPC_INS_ALIAS_MFSRR3, // Real instr.: PPC_MFSPR
	PPC_INS_ALIAS_MTDCCR, // Real instr.: PPC_MTSPR
	PPC_INS_ALIAS_MFDCCR, // Real instr.: PPC_MFSPR
	PPC_INS_ALIAS_MTICCR, // Real instr.: PPC_MTSPR
	PPC_INS_ALIAS_MFICCR, // Real instr.: PPC_MFSPR
	PPC_INS_ALIAS_TLBIE, // Real instr.: PPC_TLBIE
	PPC_INS_ALIAS_TLBREHI, // Real instr.: PPC_TLBRE2
	PPC_INS_ALIAS_TLBRELO, // Real instr.: PPC_TLBRE2
	PPC_INS_ALIAS_TLBWEHI, // Real instr.: PPC_TLBWE2
	PPC_INS_ALIAS_TLBWELO, // Real instr.: PPC_TLBWE2
	PPC_INS_ALIAS_ROTLDI, // Real instr.: PPC_RLDICL
	PPC_INS_ALIAS_ROTLDI_, // Real instr.: PPC_RLDICL_rec
	PPC_INS_ALIAS_ROTLD, // Real instr.: PPC_RLDCL
	PPC_INS_ALIAS_ROTLD_, // Real instr.: PPC_RLDCL_rec
	PPC_INS_ALIAS_CLRLDI, // Real instr.: PPC_RLDICL
	PPC_INS_ALIAS_CLRLDI_, // Real instr.: PPC_RLDICL_rec
	PPC_INS_ALIAS_LNIA, // Real instr.: PPC_ADDPCIS
	PPC_INS_ALIAS_BCp, // Real instr.: PPC_gBCat
	PPC_INS_ALIAS_BCAp, // Real instr.: PPC_gBCAat
	PPC_INS_ALIAS_BCLp, // Real instr.: PPC_gBCLat
	PPC_INS_ALIAS_BCLAp, // Real instr.: PPC_gBCLAat
	PPC_INS_ALIAS_BCm, // Real instr.: PPC_gBCat
	PPC_INS_ALIAS_BCAm, // Real instr.: PPC_gBCAat
	PPC_INS_ALIAS_BCLm, // Real instr.: PPC_gBCLat
	PPC_INS_ALIAS_BCLAm, // Real instr.: PPC_gBCLAat
	PPC_INS_ALIAS_BT, // Real instr.: PPC_gBC
	PPC_INS_ALIAS_BTA, // Real instr.: PPC_gBCA
	PPC_INS_ALIAS_BTLR, // Real instr.: PPC_gBCLR
	PPC_INS_ALIAS_BTL, // Real instr.: PPC_gBCL
	PPC_INS_ALIAS_BTLA, // Real instr.: PPC_gBCLA
	PPC_INS_ALIAS_BTLRL, // Real instr.: PPC_gBCLRL
	PPC_INS_ALIAS_BTCTR, // Real instr.: PPC_gBCCTR
	PPC_INS_ALIAS_BTCTRL, // Real instr.: PPC_gBCCTRL
	PPC_INS_ALIAS_BDZLR, // Real instr.: PPC_gBCLR
	PPC_INS_ALIAS_BDZLRL, // Real instr.: PPC_gBCLRL
	PPC_INS_ALIAS_BDZL, // Real instr.: PPC_gBCLat
	PPC_INS_ALIAS_BDZLA, // Real instr.: PPC_gBCLAat
	PPC_INS_ALIAS_BDZ, // Real instr.: PPC_gBCat
	PPC_INS_ALIAS_BDNZL, // Real instr.: PPC_gBCLat
	PPC_INS_ALIAS_BDNZLA, // Real instr.: PPC_gBCLAat
	PPC_INS_ALIAS_BDNZ, // Real instr.: PPC_gBCat
	PPC_INS_ALIAS_BDZLp, // Real instr.: PPC_gBCLat
	PPC_INS_ALIAS_BDZLAp, // Real instr.: PPC_gBCLAat
	PPC_INS_ALIAS_BDZp, // Real instr.: PPC_gBCat
	PPC_INS_ALIAS_BDNZLp, // Real instr.: PPC_gBCLat
	PPC_INS_ALIAS_BDNZLAp, // Real instr.: PPC_gBCLAat
	PPC_INS_ALIAS_BDNZp, // Real instr.: PPC_gBCat
	PPC_INS_ALIAS_BDZLm, // Real instr.: PPC_gBCLat
	PPC_INS_ALIAS_BDZLAm, // Real instr.: PPC_gBCLAat
	PPC_INS_ALIAS_BDZm, // Real instr.: PPC_gBCat
	PPC_INS_ALIAS_BDNZLm, // Real instr.: PPC_gBCLat
	PPC_INS_ALIAS_BDNZLAm, // Real instr.: PPC_gBCLAat
	PPC_INS_ALIAS_BDNZm, // Real instr.: PPC_gBCat
	PPC_INS_ALIAS_BDNZLR, // Real instr.: PPC_gBCLR
	PPC_INS_ALIAS_BDNZLRL, // Real instr.: PPC_gBCLRL
	PPC_INS_ALIAS_BDZLRp, // Real instr.: PPC_gBCLR
	PPC_INS_ALIAS_BDZLRLp, // Real instr.: PPC_gBCLRL
	PPC_INS_ALIAS_BDNZLRp, // Real instr.: PPC_gBCLR
	PPC_INS_ALIAS_BDNZLRLp, // Real instr.: PPC_gBCLRL
	PPC_INS_ALIAS_BDZLRm, // Real instr.: PPC_gBCLR
	PPC_INS_ALIAS_BDZLRLm, // Real instr.: PPC_gBCLRL
	PPC_INS_ALIAS_BDNZLRm, // Real instr.: PPC_gBCLR
	PPC_INS_ALIAS_BDNZLRLm, // Real instr.: PPC_gBCLRL
	PPC_INS_ALIAS_BF, // Real instr.: PPC_gBC
	PPC_INS_ALIAS_BFA, // Real instr.: PPC_gBCA
	PPC_INS_ALIAS_BFLR, // Real instr.: PPC_gBCLR
	PPC_INS_ALIAS_BFL, // Real instr.: PPC_gBCL
	PPC_INS_ALIAS_BFLA, // Real instr.: PPC_gBCLA
	PPC_INS_ALIAS_BFLRL, // Real instr.: PPC_gBCLRL
	PPC_INS_ALIAS_BFCTR, // Real instr.: PPC_gBCCTR
	PPC_INS_ALIAS_BFCTRL, // Real instr.: PPC_gBCCTRL
	PPC_INS_ALIAS_BTm, // Real instr.: PPC_gBC
	PPC_INS_ALIAS_BTAm, // Real instr.: PPC_gBCA
	PPC_INS_ALIAS_BTLRm, // Real instr.: PPC_gBCLR
	PPC_INS_ALIAS_BTLm, // Real instr.: PPC_gBCL
	PPC_INS_ALIAS_BTLAm, // Real instr.: PPC_gBCLA
	PPC_INS_ALIAS_BTLRLm, // Real instr.: PPC_gBCLRL
	PPC_INS_ALIAS_BTCTRm, // Real instr.: PPC_gBCCTR
	PPC_INS_ALIAS_BTCTRLm, // Real instr.: PPC_gBCCTRL
	PPC_INS_ALIAS_BFm, // Real instr.: PPC_gBC
	PPC_INS_ALIAS_BFAm, // Real instr.: PPC_gBCA
	PPC_INS_ALIAS_BFLRm, // Real instr.: PPC_gBCLR
	PPC_INS_ALIAS_BFLm, // Real instr.: PPC_gBCL
	PPC_INS_ALIAS_BFLAm, // Real instr.: PPC_gBCLA
	PPC_INS_ALIAS_BFLRLm, // Real instr.: PPC_gBCLRL
	PPC_INS_ALIAS_BFCTRm, // Real instr.: PPC_gBCCTR
	PPC_INS_ALIAS_BFCTRLm, // Real instr.: PPC_gBCCTRL
	PPC_INS_ALIAS_BTp, // Real instr.: PPC_gBC
	PPC_INS_ALIAS_BTAp, // Real instr.: PPC_gBCA
	PPC_INS_ALIAS_BTLRp, // Real instr.: PPC_gBCLR
	PPC_INS_ALIAS_BTLp, // Real instr.: PPC_gBCL
	PPC_INS_ALIAS_BTLAp, // Real instr.: PPC_gBCLA
	PPC_INS_ALIAS_BTLRLp, // Real instr.: PPC_gBCLRL
	PPC_INS_ALIAS_BTCTRp, // Real instr.: PPC_gBCCTR
	PPC_INS_ALIAS_BTCTRLp, // Real instr.: PPC_gBCCTRL
	PPC_INS_ALIAS_BFp, // Real instr.: PPC_gBC
	PPC_INS_ALIAS_BFAp, // Real instr.: PPC_gBCA
	PPC_INS_ALIAS_BFLRp, // Real instr.: PPC_gBCLR
	PPC_INS_ALIAS_BFLp, // Real instr.: PPC_gBCL
	PPC_INS_ALIAS_BFLAp, // Real instr.: PPC_gBCLA
	PPC_INS_ALIAS_BFLRLp, // Real instr.: PPC_gBCLRL
	PPC_INS_ALIAS_BFCTRp, // Real instr.: PPC_gBCCTR
	PPC_INS_ALIAS_BFCTRLp, // Real instr.: PPC_gBCCTRL
	PPC_INS_ALIAS_BDNZT, // Real instr.: PPC_gBC
	PPC_INS_ALIAS_BDNZTA, // Real instr.: PPC_gBCA
	PPC_INS_ALIAS_BDNZTLR, // Real instr.: PPC_gBCLR
	PPC_INS_ALIAS_BDNZTL, // Real instr.: PPC_gBCL
	PPC_INS_ALIAS_BDNZTLA, // Real instr.: PPC_gBCLA
	PPC_INS_ALIAS_BDNZTLRL, // Real instr.: PPC_gBCLRL
	PPC_INS_ALIAS_BDNZF, // Real instr.: PPC_gBC
	PPC_INS_ALIAS_BDNZFA, // Real instr.: PPC_gBCA
	PPC_INS_ALIAS_BDNZFLR, // Real instr.: PPC_gBCLR
	PPC_INS_ALIAS_BDNZFL, // Real instr.: PPC_gBCL
	PPC_INS_ALIAS_BDNZFLA, // Real instr.: PPC_gBCLA
	PPC_INS_ALIAS_BDNZFLRL, // Real instr.: PPC_gBCLRL
	PPC_INS_ALIAS_BDZT, // Real instr.: PPC_gBC
	PPC_INS_ALIAS_BDZTA, // Real instr.: PPC_gBCA
	PPC_INS_ALIAS_BDZTLR, // Real instr.: PPC_gBCLR
	PPC_INS_ALIAS_BDZTL, // Real instr.: PPC_gBCL
	PPC_INS_ALIAS_BDZTLA, // Real instr.: PPC_gBCLA
	PPC_INS_ALIAS_BDZTLRL, // Real instr.: PPC_gBCLRL
	PPC_INS_ALIAS_BDZF, // Real instr.: PPC_gBC
	PPC_INS_ALIAS_BDZFA, // Real instr.: PPC_gBCA
	PPC_INS_ALIAS_BDZFLR, // Real instr.: PPC_gBCLR
	PPC_INS_ALIAS_BDZFL, // Real instr.: PPC_gBCL
	PPC_INS_ALIAS_BDZFLA, // Real instr.: PPC_gBCLA
	PPC_INS_ALIAS_BDZFLRL, // Real instr.: PPC_gBCLRL
	PPC_INS_ALIAS_B, // Real instr.: PPC_gBC
	PPC_INS_ALIAS_BA, // Real instr.: PPC_gBCA
	PPC_INS_ALIAS_BL, // Real instr.: PPC_gBCL
	PPC_INS_ALIAS_BLA, // Real instr.: PPC_gBCLA
	PPC_INS_ALIAS_BLR, // Real instr.: PPC_gBCLR
	PPC_INS_ALIAS_BLRL, // Real instr.: PPC_gBCLRL
	PPC_INS_ALIAS_BCTR, // Real instr.: PPC_gBCCTR
	PPC_INS_ALIAS_BCTRL, // Real instr.: PPC_gBCCTRL
	PPC_INS_ALIAS_BLT, // Real instr.: PPC_BCC
	PPC_INS_ALIAS_BLTA, // Real instr.: PPC_BCCA
	PPC_INS_ALIAS_BLTLR, // Real instr.: PPC_BCCLR
	PPC_INS_ALIAS_BLTCTR, // Real instr.: PPC_BCCCTR
	PPC_INS_ALIAS_BLTL, // Real instr.: PPC_BCCL
	PPC_INS_ALIAS_BLTLA, // Real instr.: PPC_BCCLA
	PPC_INS_ALIAS_BLTLRL, // Real instr.: PPC_BCCLRL
	PPC_INS_ALIAS_BLTCTRL, // Real instr.: PPC_BCCCTRL
	PPC_INS_ALIAS_BLTm, // Real instr.: PPC_BCC
	PPC_INS_ALIAS_BLTAm, // Real instr.: PPC_BCCA
	PPC_INS_ALIAS_BLTLRm, // Real instr.: PPC_BCCLR
	PPC_INS_ALIAS_BLTCTRm, // Real instr.: PPC_BCCCTR
	PPC_INS_ALIAS_BLTLm, // Real instr.: PPC_BCCL
	PPC_INS_ALIAS_BLTLAm, // Real instr.: PPC_BCCLA
	PPC_INS_ALIAS_BLTLRLm, // Real instr.: PPC_BCCLRL
	PPC_INS_ALIAS_BLTCTRLm, // Real instr.: PPC_BCCCTRL
	PPC_INS_ALIAS_BLTp, // Real instr.: PPC_BCC
	PPC_INS_ALIAS_BLTAp, // Real instr.: PPC_BCCA
	PPC_INS_ALIAS_BLTLRp, // Real instr.: PPC_BCCLR
	PPC_INS_ALIAS_BLTCTRp, // Real instr.: PPC_BCCCTR
	PPC_INS_ALIAS_BLTLp, // Real instr.: PPC_BCCL
	PPC_INS_ALIAS_BLTLAp, // Real instr.: PPC_BCCLA
	PPC_INS_ALIAS_BLTLRLp, // Real instr.: PPC_BCCLRL
	PPC_INS_ALIAS_BLTCTRLp, // Real instr.: PPC_BCCCTRL
	PPC_INS_ALIAS_BGT, // Real instr.: PPC_BCC
	PPC_INS_ALIAS_BGTA, // Real instr.: PPC_BCCA
	PPC_INS_ALIAS_BGTLR, // Real instr.: PPC_BCCLR
	PPC_INS_ALIAS_BGTCTR, // Real instr.: PPC_BCCCTR
	PPC_INS_ALIAS_BGTL, // Real instr.: PPC_BCCL
	PPC_INS_ALIAS_BGTLA, // Real instr.: PPC_BCCLA
	PPC_INS_ALIAS_BGTLRL, // Real instr.: PPC_BCCLRL
	PPC_INS_ALIAS_BGTCTRL, // Real instr.: PPC_BCCCTRL
	PPC_INS_ALIAS_BGTm, // Real instr.: PPC_BCC
	PPC_INS_ALIAS_BGTAm, // Real instr.: PPC_BCCA
	PPC_INS_ALIAS_BGTLRm, // Real instr.: PPC_BCCLR
	PPC_INS_ALIAS_BGTCTRm, // Real instr.: PPC_BCCCTR
	PPC_INS_ALIAS_BGTLm, // Real instr.: PPC_BCCL
	PPC_INS_ALIAS_BGTLAm, // Real instr.: PPC_BCCLA
	PPC_INS_ALIAS_BGTLRLm, // Real instr.: PPC_BCCLRL
	PPC_INS_ALIAS_BGTCTRLm, // Real instr.: PPC_BCCCTRL
	PPC_INS_ALIAS_BGTp, // Real instr.: PPC_BCC
	PPC_INS_ALIAS_BGTAp, // Real instr.: PPC_BCCA
	PPC_INS_ALIAS_BGTLRp, // Real instr.: PPC_BCCLR
	PPC_INS_ALIAS_BGTCTRp, // Real instr.: PPC_BCCCTR
	PPC_INS_ALIAS_BGTLp, // Real instr.: PPC_BCCL
	PPC_INS_ALIAS_BGTLAp, // Real instr.: PPC_BCCLA
	PPC_INS_ALIAS_BGTLRLp, // Real instr.: PPC_BCCLRL
	PPC_INS_ALIAS_BGTCTRLp, // Real instr.: PPC_BCCCTRL
	PPC_INS_ALIAS_BEQ, // Real instr.: PPC_BCC
	PPC_INS_ALIAS_BEQA, // Real instr.: PPC_BCCA
	PPC_INS_ALIAS_BEQLR, // Real instr.: PPC_BCCLR
	PPC_INS_ALIAS_BEQCTR, // Real instr.: PPC_BCCCTR
	PPC_INS_ALIAS_BEQL, // Real instr.: PPC_BCCL
	PPC_INS_ALIAS_BEQLA, // Real instr.: PPC_BCCLA
	PPC_INS_ALIAS_BEQLRL, // Real instr.: PPC_BCCLRL
	PPC_INS_ALIAS_BEQCTRL, // Real instr.: PPC_BCCCTRL
	PPC_INS_ALIAS_BEQm, // Real instr.: PPC_BCC
	PPC_INS_ALIAS_BEQAm, // Real instr.: PPC_BCCA
	PPC_INS_ALIAS_BEQLRm, // Real instr.: PPC_BCCLR
	PPC_INS_ALIAS_BEQCTRm, // Real instr.: PPC_BCCCTR
	PPC_INS_ALIAS_BEQLm, // Real instr.: PPC_BCCL
	PPC_INS_ALIAS_BEQLAm, // Real instr.: PPC_BCCLA
	PPC_INS_ALIAS_BEQLRLm, // Real instr.: PPC_BCCLRL
	PPC_INS_ALIAS_BEQCTRLm, // Real instr.: PPC_BCCCTRL
	PPC_INS_ALIAS_BEQp, // Real instr.: PPC_BCC
	PPC_INS_ALIAS_BEQAp, // Real instr.: PPC_BCCA
	PPC_INS_ALIAS_BEQLRp, // Real instr.: PPC_BCCLR
	PPC_INS_ALIAS_BEQCTRp, // Real instr.: PPC_BCCCTR
	PPC_INS_ALIAS_BEQLp, // Real instr.: PPC_BCCL
	PPC_INS_ALIAS_BEQLAp, // Real instr.: PPC_BCCLA
	PPC_INS_ALIAS_BEQLRLp, // Real instr.: PPC_BCCLRL
	PPC_INS_ALIAS_BEQCTRLp, // Real instr.: PPC_BCCCTRL
	PPC_INS_ALIAS_BUN, // Real instr.: PPC_BCC
	PPC_INS_ALIAS_BUNA, // Real instr.: PPC_BCCA
	PPC_INS_ALIAS_BUNLR, // Real instr.: PPC_BCCLR
	PPC_INS_ALIAS_BUNCTR, // Real instr.: PPC_BCCCTR
	PPC_INS_ALIAS_BUNL, // Real instr.: PPC_BCCL
	PPC_INS_ALIAS_BUNLA, // Real instr.: PPC_BCCLA
	PPC_INS_ALIAS_BUNLRL, // Real instr.: PPC_BCCLRL
	PPC_INS_ALIAS_BUNCTRL, // Real instr.: PPC_BCCCTRL
	PPC_INS_ALIAS_BUNm, // Real instr.: PPC_BCC
	PPC_INS_ALIAS_BUNAm, // Real instr.: PPC_BCCA
	PPC_INS_ALIAS_BUNLRm, // Real instr.: PPC_BCCLR
	PPC_INS_ALIAS_BUNCTRm, // Real instr.: PPC_BCCCTR
	PPC_INS_ALIAS_BUNLm, // Real instr.: PPC_BCCL
	PPC_INS_ALIAS_BUNLAm, // Real instr.: PPC_BCCLA
	PPC_INS_ALIAS_BUNLRLm, // Real instr.: PPC_BCCLRL
	PPC_INS_ALIAS_BUNCTRLm, // Real instr.: PPC_BCCCTRL
	PPC_INS_ALIAS_BUNp, // Real instr.: PPC_BCC
	PPC_INS_ALIAS_BUNAp, // Real instr.: PPC_BCCA
	PPC_INS_ALIAS_BUNLRp, // Real instr.: PPC_BCCLR
	PPC_INS_ALIAS_BUNCTRp, // Real instr.: PPC_BCCCTR
	PPC_INS_ALIAS_BUNLp, // Real instr.: PPC_BCCL
	PPC_INS_ALIAS_BUNLAp, // Real instr.: PPC_BCCLA
	PPC_INS_ALIAS_BUNLRLp, // Real instr.: PPC_BCCLRL
	PPC_INS_ALIAS_BUNCTRLp, // Real instr.: PPC_BCCCTRL
	PPC_INS_ALIAS_BSO, // Real instr.: PPC_BCC
	PPC_INS_ALIAS_BSOA, // Real instr.: PPC_BCCA
	PPC_INS_ALIAS_BSOLR, // Real instr.: PPC_BCCLR
	PPC_INS_ALIAS_BSOCTR, // Real instr.: PPC_BCCCTR
	PPC_INS_ALIAS_BSOL, // Real instr.: PPC_BCCL
	PPC_INS_ALIAS_BSOLA, // Real instr.: PPC_BCCLA
	PPC_INS_ALIAS_BSOLRL, // Real instr.: PPC_BCCLRL
	PPC_INS_ALIAS_BSOCTRL, // Real instr.: PPC_BCCCTRL
	PPC_INS_ALIAS_BSOm, // Real instr.: PPC_BCC
	PPC_INS_ALIAS_BSOAm, // Real instr.: PPC_BCCA
	PPC_INS_ALIAS_BSOLRm, // Real instr.: PPC_BCCLR
	PPC_INS_ALIAS_BSOCTRm, // Real instr.: PPC_BCCCTR
	PPC_INS_ALIAS_BSOLm, // Real instr.: PPC_BCCL
	PPC_INS_ALIAS_BSOLAm, // Real instr.: PPC_BCCLA
	PPC_INS_ALIAS_BSOLRLm, // Real instr.: PPC_BCCLRL
	PPC_INS_ALIAS_BSOCTRLm, // Real instr.: PPC_BCCCTRL
	PPC_INS_ALIAS_BSOp, // Real instr.: PPC_BCC
	PPC_INS_ALIAS_BSOAp, // Real instr.: PPC_BCCA
	PPC_INS_ALIAS_BSOLRp, // Real instr.: PPC_BCCLR
	PPC_INS_ALIAS_BSOCTRp, // Real instr.: PPC_BCCCTR
	PPC_INS_ALIAS_BSOLp, // Real instr.: PPC_BCCL
	PPC_INS_ALIAS_BSOLAp, // Real instr.: PPC_BCCLA
	PPC_INS_ALIAS_BSOLRLp, // Real instr.: PPC_BCCLRL
	PPC_INS_ALIAS_BSOCTRLp, // Real instr.: PPC_BCCCTRL
	PPC_INS_ALIAS_BGE, // Real instr.: PPC_BCC
	PPC_INS_ALIAS_BGEA, // Real instr.: PPC_BCCA
	PPC_INS_ALIAS_BGELR, // Real instr.: PPC_BCCLR
	PPC_INS_ALIAS_BGECTR, // Real instr.: PPC_BCCCTR
	PPC_INS_ALIAS_BGEL, // Real instr.: PPC_BCCL
	PPC_INS_ALIAS_BGELA, // Real instr.: PPC_BCCLA
	PPC_INS_ALIAS_BGELRL, // Real instr.: PPC_BCCLRL
	PPC_INS_ALIAS_BGECTRL, // Real instr.: PPC_BCCCTRL
	PPC_INS_ALIAS_BGEm, // Real instr.: PPC_BCC
	PPC_INS_ALIAS_BGEAm, // Real instr.: PPC_BCCA
	PPC_INS_ALIAS_BGELRm, // Real instr.: PPC_BCCLR
	PPC_INS_ALIAS_BGECTRm, // Real instr.: PPC_BCCCTR
	PPC_INS_ALIAS_BGELm, // Real instr.: PPC_BCCL
	PPC_INS_ALIAS_BGELAm, // Real instr.: PPC_BCCLA
	PPC_INS_ALIAS_BGELRLm, // Real instr.: PPC_BCCLRL
	PPC_INS_ALIAS_BGECTRLm, // Real instr.: PPC_BCCCTRL
	PPC_INS_ALIAS_BGEp, // Real instr.: PPC_BCC
	PPC_INS_ALIAS_BGEAp, // Real instr.: PPC_BCCA
	PPC_INS_ALIAS_BGELRp, // Real instr.: PPC_BCCLR
	PPC_INS_ALIAS_BGECTRp, // Real instr.: PPC_BCCCTR
	PPC_INS_ALIAS_BGELp, // Real instr.: PPC_BCCL
	PPC_INS_ALIAS_BGELAp, // Real instr.: PPC_BCCLA
	PPC_INS_ALIAS_BGELRLp, // Real instr.: PPC_BCCLRL
	PPC_INS_ALIAS_BGECTRLp, // Real instr.: PPC_BCCCTRL
	PPC_INS_ALIAS_BNL, // Real instr.: PPC_BCC
	PPC_INS_ALIAS_BNLA, // Real instr.: PPC_BCCA
	PPC_INS_ALIAS_BNLLR, // Real instr.: PPC_BCCLR
	PPC_INS_ALIAS_BNLCTR, // Real instr.: PPC_BCCCTR
	PPC_INS_ALIAS_BNLL, // Real instr.: PPC_BCCL
	PPC_INS_ALIAS_BNLLA, // Real instr.: PPC_BCCLA
	PPC_INS_ALIAS_BNLLRL, // Real instr.: PPC_BCCLRL
	PPC_INS_ALIAS_BNLCTRL, // Real instr.: PPC_BCCCTRL
	PPC_INS_ALIAS_BNLm, // Real instr.: PPC_BCC
	PPC_INS_ALIAS_BNLAm, // Real instr.: PPC_BCCA
	PPC_INS_ALIAS_BNLLRm, // Real instr.: PPC_BCCLR
	PPC_INS_ALIAS_BNLCTRm, // Real instr.: PPC_BCCCTR
	PPC_INS_ALIAS_BNLLm, // Real instr.: PPC_BCCL
	PPC_INS_ALIAS_BNLLAm, // Real instr.: PPC_BCCLA
	PPC_INS_ALIAS_BNLLRLm, // Real instr.: PPC_BCCLRL
	PPC_INS_ALIAS_BNLCTRLm, // Real instr.: PPC_BCCCTRL
	PPC_INS_ALIAS_BNLp, // Real instr.: PPC_BCC
	PPC_INS_ALIAS_BNLAp, // Real instr.: PPC_BCCA
	PPC_INS_ALIAS_BNLLRp, // Real instr.: PPC_BCCLR
	PPC_INS_ALIAS_BNLCTRp, // Real instr.: PPC_BCCCTR
	PPC_INS_ALIAS_BNLLp, // Real instr.: PPC_BCCL
	PPC_INS_ALIAS_BNLLAp, // Real instr.: PPC_BCCLA
	PPC_INS_ALIAS_BNLLRLp, // Real instr.: PPC_BCCLRL
	PPC_INS_ALIAS_BNLCTRLp, // Real instr.: PPC_BCCCTRL
	PPC_INS_ALIAS_BLE, // Real instr.: PPC_BCC
	PPC_INS_ALIAS_BLEA, // Real instr.: PPC_BCCA
	PPC_INS_ALIAS_BLELR, // Real instr.: PPC_BCCLR
	PPC_INS_ALIAS_BLECTR, // Real instr.: PPC_BCCCTR
	PPC_INS_ALIAS_BLEL, // Real instr.: PPC_BCCL
	PPC_INS_ALIAS_BLELA, // Real instr.: PPC_BCCLA
	PPC_INS_ALIAS_BLELRL, // Real instr.: PPC_BCCLRL
	PPC_INS_ALIAS_BLECTRL, // Real instr.: PPC_BCCCTRL
	PPC_INS_ALIAS_BLEm, // Real instr.: PPC_BCC
	PPC_INS_ALIAS_BLEAm, // Real instr.: PPC_BCCA
	PPC_INS_ALIAS_BLELRm, // Real instr.: PPC_BCCLR
	PPC_INS_ALIAS_BLECTRm, // Real instr.: PPC_BCCCTR
	PPC_INS_ALIAS_BLELm, // Real instr.: PPC_BCCL
	PPC_INS_ALIAS_BLELAm, // Real instr.: PPC_BCCLA
	PPC_INS_ALIAS_BLELRLm, // Real instr.: PPC_BCCLRL
	PPC_INS_ALIAS_BLECTRLm, // Real instr.: PPC_BCCCTRL
	PPC_INS_ALIAS_BLEp, // Real instr.: PPC_BCC
	PPC_INS_ALIAS_BLEAp, // Real instr.: PPC_BCCA
	PPC_INS_ALIAS_BLELRp, // Real instr.: PPC_BCCLR
	PPC_INS_ALIAS_BLECTRp, // Real instr.: PPC_BCCCTR
	PPC_INS_ALIAS_BLELp, // Real instr.: PPC_BCCL
	PPC_INS_ALIAS_BLELAp, // Real instr.: PPC_BCCLA
	PPC_INS_ALIAS_BLELRLp, // Real instr.: PPC_BCCLRL
	PPC_INS_ALIAS_BLECTRLp, // Real instr.: PPC_BCCCTRL
	PPC_INS_ALIAS_BNG, // Real instr.: PPC_BCC
	PPC_INS_ALIAS_BNGA, // Real instr.: PPC_BCCA
	PPC_INS_ALIAS_BNGLR, // Real instr.: PPC_BCCLR
	PPC_INS_ALIAS_BNGCTR, // Real instr.: PPC_BCCCTR
	PPC_INS_ALIAS_BNGL, // Real instr.: PPC_BCCL
	PPC_INS_ALIAS_BNGLA, // Real instr.: PPC_BCCLA
	PPC_INS_ALIAS_BNGLRL, // Real instr.: PPC_BCCLRL
	PPC_INS_ALIAS_BNGCTRL, // Real instr.: PPC_BCCCTRL
	PPC_INS_ALIAS_BNGm, // Real instr.: PPC_BCC
	PPC_INS_ALIAS_BNGAm, // Real instr.: PPC_BCCA
	PPC_INS_ALIAS_BNGLRm, // Real instr.: PPC_BCCLR
	PPC_INS_ALIAS_BNGCTRm, // Real instr.: PPC_BCCCTR
	PPC_INS_ALIAS_BNGLm, // Real instr.: PPC_BCCL
	PPC_INS_ALIAS_BNGLAm, // Real instr.: PPC_BCCLA
	PPC_INS_ALIAS_BNGLRLm, // Real instr.: PPC_BCCLRL
	PPC_INS_ALIAS_BNGCTRLm, // Real instr.: PPC_BCCCTRL
	PPC_INS_ALIAS_BNGp, // Real instr.: PPC_BCC
	PPC_INS_ALIAS_BNGAp, // Real instr.: PPC_BCCA
	PPC_INS_ALIAS_BNGLRp, // Real instr.: PPC_BCCLR
	PPC_INS_ALIAS_BNGCTRp, // Real instr.: PPC_BCCCTR
	PPC_INS_ALIAS_BNGLp, // Real instr.: PPC_BCCL
	PPC_INS_ALIAS_BNGLAp, // Real instr.: PPC_BCCLA
	PPC_INS_ALIAS_BNGLRLp, // Real instr.: PPC_BCCLRL
	PPC_INS_ALIAS_BNGCTRLp, // Real instr.: PPC_BCCCTRL
	PPC_INS_ALIAS_BNE, // Real instr.: PPC_BCC
	PPC_INS_ALIAS_BNEA, // Real instr.: PPC_BCCA
	PPC_INS_ALIAS_BNELR, // Real instr.: PPC_BCCLR
	PPC_INS_ALIAS_BNECTR, // Real instr.: PPC_BCCCTR
	PPC_INS_ALIAS_BNEL, // Real instr.: PPC_BCCL
	PPC_INS_ALIAS_BNELA, // Real instr.: PPC_BCCLA
	PPC_INS_ALIAS_BNELRL, // Real instr.: PPC_BCCLRL
	PPC_INS_ALIAS_BNECTRL, // Real instr.: PPC_BCCCTRL
	PPC_INS_ALIAS_BNEm, // Real instr.: PPC_BCC
	PPC_INS_ALIAS_BNEAm, // Real instr.: PPC_BCCA
	PPC_INS_ALIAS_BNELRm, // Real instr.: PPC_BCCLR
	PPC_INS_ALIAS_BNECTRm, // Real instr.: PPC_BCCCTR
	PPC_INS_ALIAS_BNELm, // Real instr.: PPC_BCCL
	PPC_INS_ALIAS_BNELAm, // Real instr.: PPC_BCCLA
	PPC_INS_ALIAS_BNELRLm, // Real instr.: PPC_BCCLRL
	PPC_INS_ALIAS_BNECTRLm, // Real instr.: PPC_BCCCTRL
	PPC_INS_ALIAS_BNEp, // Real instr.: PPC_BCC
	PPC_INS_ALIAS_BNEAp, // Real instr.: PPC_BCCA
	PPC_INS_ALIAS_BNELRp, // Real instr.: PPC_BCCLR
	PPC_INS_ALIAS_BNECTRp, // Real instr.: PPC_BCCCTR
	PPC_INS_ALIAS_BNELp, // Real instr.: PPC_BCCL
	PPC_INS_ALIAS_BNELAp, // Real instr.: PPC_BCCLA
	PPC_INS_ALIAS_BNELRLp, // Real instr.: PPC_BCCLRL
	PPC_INS_ALIAS_BNECTRLp, // Real instr.: PPC_BCCCTRL
	PPC_INS_ALIAS_BNU, // Real instr.: PPC_BCC
	PPC_INS_ALIAS_BNUA, // Real instr.: PPC_BCCA
	PPC_INS_ALIAS_BNULR, // Real instr.: PPC_BCCLR
	PPC_INS_ALIAS_BNUCTR, // Real instr.: PPC_BCCCTR
	PPC_INS_ALIAS_BNUL, // Real instr.: PPC_BCCL
	PPC_INS_ALIAS_BNULA, // Real instr.: PPC_BCCLA
	PPC_INS_ALIAS_BNULRL, // Real instr.: PPC_BCCLRL
	PPC_INS_ALIAS_BNUCTRL, // Real instr.: PPC_BCCCTRL
	PPC_INS_ALIAS_BNUm, // Real instr.: PPC_BCC
	PPC_INS_ALIAS_BNUAm, // Real instr.: PPC_BCCA
	PPC_INS_ALIAS_BNULRm, // Real instr.: PPC_BCCLR
	PPC_INS_ALIAS_BNUCTRm, // Real instr.: PPC_BCCCTR
	PPC_INS_ALIAS_BNULm, // Real instr.: PPC_BCCL
	PPC_INS_ALIAS_BNULAm, // Real instr.: PPC_BCCLA
	PPC_INS_ALIAS_BNULRLm, // Real instr.: PPC_BCCLRL
	PPC_INS_ALIAS_BNUCTRLm, // Real instr.: PPC_BCCCTRL
	PPC_INS_ALIAS_BNUp, // Real instr.: PPC_BCC
	PPC_INS_ALIAS_BNUAp, // Real instr.: PPC_BCCA
	PPC_INS_ALIAS_BNULRp, // Real instr.: PPC_BCCLR
	PPC_INS_ALIAS_BNUCTRp, // Real instr.: PPC_BCCCTR
	PPC_INS_ALIAS_BNULp, // Real instr.: PPC_BCCL
	PPC_INS_ALIAS_BNULAp, // Real instr.: PPC_BCCLA
	PPC_INS_ALIAS_BNULRLp, // Real instr.: PPC_BCCLRL
	PPC_INS_ALIAS_BNUCTRLp, // Real instr.: PPC_BCCCTRL
	PPC_INS_ALIAS_BNS, // Real instr.: PPC_BCC
	PPC_INS_ALIAS_BNSA, // Real instr.: PPC_BCCA
	PPC_INS_ALIAS_BNSLR, // Real instr.: PPC_BCCLR
	PPC_INS_ALIAS_BNSCTR, // Real instr.: PPC_BCCCTR
	PPC_INS_ALIAS_BNSL, // Real instr.: PPC_BCCL
	PPC_INS_ALIAS_BNSLA, // Real instr.: PPC_BCCLA
	PPC_INS_ALIAS_BNSLRL, // Real instr.: PPC_BCCLRL
	PPC_INS_ALIAS_BNSCTRL, // Real instr.: PPC_BCCCTRL
	PPC_INS_ALIAS_BNSm, // Real instr.: PPC_BCC
	PPC_INS_ALIAS_BNSAm, // Real instr.: PPC_BCCA
	PPC_INS_ALIAS_BNSLRm, // Real instr.: PPC_BCCLR
	PPC_INS_ALIAS_BNSCTRm, // Real instr.: PPC_BCCCTR
	PPC_INS_ALIAS_BNSLm, // Real instr.: PPC_BCCL
	PPC_INS_ALIAS_BNSLAm, // Real instr.: PPC_BCCLA
	PPC_INS_ALIAS_BNSLRLm, // Real instr.: PPC_BCCLRL
	PPC_INS_ALIAS_BNSCTRLm, // Real instr.: PPC_BCCCTRL
	PPC_INS_ALIAS_BNSp, // Real instr.: PPC_BCC
	PPC_INS_ALIAS_BNSAp, // Real instr.: PPC_BCCA
	PPC_INS_ALIAS_BNSLRp, // Real instr.: PPC_BCCLR
	PPC_INS_ALIAS_BNSCTRp, // Real instr.: PPC_BCCCTR
	PPC_INS_ALIAS_BNSLp, // Real instr.: PPC_BCCL
	PPC_INS_ALIAS_BNSLAp, // Real instr.: PPC_BCCLA
	PPC_INS_ALIAS_BNSLRLp, // Real instr.: PPC_BCCLRL
	PPC_INS_ALIAS_BNSCTRLp, // Real instr.: PPC_BCCCTRL
	PPC_INS_ALIAS_CMPWI, // Real instr.: PPC_CMPWI
	PPC_INS_ALIAS_CMPW, // Real instr.: PPC_CMPW
	PPC_INS_ALIAS_CMPLWI, // Real instr.: PPC_CMPLWI
	PPC_INS_ALIAS_CMPLW, // Real instr.: PPC_CMPLW
	PPC_INS_ALIAS_CMPDI, // Real instr.: PPC_CMPDI
	PPC_INS_ALIAS_CMPD, // Real instr.: PPC_CMPD
	PPC_INS_ALIAS_CMPLDI, // Real instr.: PPC_CMPLDI
	PPC_INS_ALIAS_CMPLD, // Real instr.: PPC_CMPLD
	PPC_INS_ALIAS_CMPI, // Real instr.: PPC_CMPWI
	PPC_INS_ALIAS_CMP, // Real instr.: PPC_CMPW
	PPC_INS_ALIAS_CMPLI, // Real instr.: PPC_CMPLWI
	PPC_INS_ALIAS_CMPL, // Real instr.: PPC_CMPLW
	PPC_INS_ALIAS_TRAP, // Real instr.: PPC_TW
	PPC_INS_ALIAS_TDLTI, // Real instr.: PPC_TDI
	PPC_INS_ALIAS_TDLT, // Real instr.: PPC_TD
	PPC_INS_ALIAS_TWLTI, // Real instr.: PPC_TWI
	PPC_INS_ALIAS_TWLT, // Real instr.: PPC_TW
	PPC_INS_ALIAS_TDLEI, // Real instr.: PPC_TDI
	PPC_INS_ALIAS_TDLE, // Real instr.: PPC_TD
	PPC_INS_ALIAS_TWLEI, // Real instr.: PPC_TWI
	PPC_INS_ALIAS_TWLE, // Real instr.: PPC_TW
	PPC_INS_ALIAS_TDEQI, // Real instr.: PPC_TDI
	PPC_INS_ALIAS_TDEQ, // Real instr.: PPC_TD
	PPC_INS_ALIAS_TWEQI, // Real instr.: PPC_TWI
	PPC_INS_ALIAS_TWEQ, // Real instr.: PPC_TW
	PPC_INS_ALIAS_TDGEI, // Real instr.: PPC_TDI
	PPC_INS_ALIAS_TDGE, // Real instr.: PPC_TD
	PPC_INS_ALIAS_TWGEI, // Real instr.: PPC_TWI
	PPC_INS_ALIAS_TWGE, // Real instr.: PPC_TW
	PPC_INS_ALIAS_TDGTI, // Real instr.: PPC_TDI
	PPC_INS_ALIAS_TDGT, // Real instr.: PPC_TD
	PPC_INS_ALIAS_TWGTI, // Real instr.: PPC_TWI
	PPC_INS_ALIAS_TWGT, // Real instr.: PPC_TW
	PPC_INS_ALIAS_TDNLI, // Real instr.: PPC_TDI
	PPC_INS_ALIAS_TDNL, // Real instr.: PPC_TD
	PPC_INS_ALIAS_TWNLI, // Real instr.: PPC_TWI
	PPC_INS_ALIAS_TWNL, // Real instr.: PPC_TW
	PPC_INS_ALIAS_TDNEI, // Real instr.: PPC_TDI
	PPC_INS_ALIAS_TDNE, // Real instr.: PPC_TD
	PPC_INS_ALIAS_TWNEI, // Real instr.: PPC_TWI
	PPC_INS_ALIAS_TWNE, // Real instr.: PPC_TW
	PPC_INS_ALIAS_TDNGI, // Real instr.: PPC_TDI
	PPC_INS_ALIAS_TDNG, // Real instr.: PPC_TD
	PPC_INS_ALIAS_TWNGI, // Real instr.: PPC_TWI
	PPC_INS_ALIAS_TWNG, // Real instr.: PPC_TW
	PPC_INS_ALIAS_TDLLTI, // Real instr.: PPC_TDI
	PPC_INS_ALIAS_TDLLT, // Real instr.: PPC_TD
	PPC_INS_ALIAS_TWLLTI, // Real instr.: PPC_TWI
	PPC_INS_ALIAS_TWLLT, // Real instr.: PPC_TW
	PPC_INS_ALIAS_TDLLEI, // Real instr.: PPC_TDI
	PPC_INS_ALIAS_TDLLE, // Real instr.: PPC_TD
	PPC_INS_ALIAS_TWLLEI, // Real instr.: PPC_TWI
	PPC_INS_ALIAS_TWLLE, // Real instr.: PPC_TW
	PPC_INS_ALIAS_TDLGEI, // Real instr.: PPC_TDI
	PPC_INS_ALIAS_TDLGE, // Real instr.: PPC_TD
	PPC_INS_ALIAS_TWLGEI, // Real instr.: PPC_TWI
	PPC_INS_ALIAS_TWLGE, // Real instr.: PPC_TW
	PPC_INS_ALIAS_TDLGTI, // Real instr.: PPC_TDI
	PPC_INS_ALIAS_TDLGT, // Real instr.: PPC_TD
	PPC_INS_ALIAS_TWLGTI, // Real instr.: PPC_TWI
	PPC_INS_ALIAS_TWLGT, // Real instr.: PPC_TW
	PPC_INS_ALIAS_TDLNLI, // Real instr.: PPC_TDI
	PPC_INS_ALIAS_TDLNL, // Real instr.: PPC_TD
	PPC_INS_ALIAS_TWLNLI, // Real instr.: PPC_TWI
	PPC_INS_ALIAS_TWLNL, // Real instr.: PPC_TW
	PPC_INS_ALIAS_TDLNGI, // Real instr.: PPC_TDI
	PPC_INS_ALIAS_TDLNG, // Real instr.: PPC_TD
	PPC_INS_ALIAS_TWLNGI, // Real instr.: PPC_TWI
	PPC_INS_ALIAS_TWLNG, // Real instr.: PPC_TW
	PPC_INS_ALIAS_TDUI, // Real instr.: PPC_TDI
	PPC_INS_ALIAS_TDU, // Real instr.: PPC_TD
	PPC_INS_ALIAS_TWUI, // Real instr.: PPC_TWI
	PPC_INS_ALIAS_TWU, // Real instr.: PPC_TW
	PPC_INS_ALIAS_PASTE_, // Real instr.: PPC_CP_PASTE_rec
	PPC_INS_ALIAS_QVFCLR, // Real instr.: PPC_QVFLOGICALb
	PPC_INS_ALIAS_QVFAND, // Real instr.: PPC_QVFLOGICALb
	PPC_INS_ALIAS_QVFANDC, // Real instr.: PPC_QVFLOGICALb
	PPC_INS_ALIAS_QVFCTFB, // Real instr.: PPC_QVFLOGICALb
	PPC_INS_ALIAS_QVFXOR, // Real instr.: PPC_QVFLOGICALb
	PPC_INS_ALIAS_QVFOR, // Real instr.: PPC_QVFLOGICALb
	PPC_INS_ALIAS_QVFNOR, // Real instr.: PPC_QVFLOGICALb
	PPC_INS_ALIAS_QVFEQU, // Real instr.: PPC_QVFLOGICALb
	PPC_INS_ALIAS_QVFNOT, // Real instr.: PPC_QVFLOGICALb
	PPC_INS_ALIAS_QVFORC, // Real instr.: PPC_QVFLOGICALb
	PPC_INS_ALIAS_QVFNAND, // Real instr.: PPC_QVFLOGICALb
	PPC_INS_ALIAS_QVFSET, // Real instr.: PPC_QVFLOGICALb

	// clang-format on
	// generated content <PPCGenCSAliasEnum.inc> end

	// Hardcoded in LLVM printer
	PPC_INS_ALIAS_SLWI, // Real instr.: PPC_RLWINM
	PPC_INS_ALIAS_SRWI, // Real instr.: PPC_RLWINM
	PPC_INS_ALIAS_SLDI, // Real instr.: PPC_RLDICR

	PPC_INS_ALIAS_END,

} ppc_insn;

/// Group of PPC instructions
typedef enum ppc_insn_group {
	PPC_GRP_INVALID = 0, ///< = CS_GRP_INVALID

	// Generic groups
	// all jump instructions (conditional+direct+indirect jumps)
	PPC_GRP_JUMP,		 ///< = CS_GRP_JUMP
	PPC_GRP_CALL,		 ///< = CS_GRP_CALL
	PPC_GRP_INT = 4,	 ///< = CS_GRP_INT
	PPC_GRP_PRIVILEGE = 6,	 ///< = CS_GRP_PRIVILEGE
	PPC_GRP_BRANCH_RELATIVE, ///< = CS_GRP_BRANCH_RELATIVE

	// Architecture-specific groups
	// generated content <PPCGenCSFeatureEnum.inc> begin
	// clang-format off

	PPC_FEATURE_IsNotISAFuture = 128,
	PPC_FEATURE_IsISA3_0,
	PPC_FEATURE_In64BitMode,
	PPC_FEATURE_In32BitMode,
	PPC_FEATURE_PCRelativeMemops,
	PPC_FEATURE_HasBPERMD,
	PPC_FEATURE_HasSPE,
	PPC_FEATURE_IsE500,
	PPC_FEATURE_IsPPC4xx,
	PPC_FEATURE_HasExtDiv,
	PPC_FEATURE_IsISAFuture,
	PPC_FEATURE_HasFPU,
	PPC_FEATURE_HasICBT,
	PPC_FEATURE_HasPartwordAtomics,
	PPC_FEATURE_IsISA2_06,
	PPC_FEATURE_IsBookE,
	PPC_FEATURE_HasPS,
	PPC_FEATURE_HasQPX,
	PPC_FEATURE_IsPPC6xx,

	// clang-format on
	// generated content <PPCGenCSFeatureEnum.inc> end

	PPC_GRP_ENDING, // <-- mark the end of the list of groups
} ppc_insn_group;

/// PPC instruction formats. To get details about them please
/// refer to `PPCInstrFormats.td` in LLVM.
typedef enum {
	PPC_INSN_FORM_INVALID = 0,

	// generated content <PPCGenCSInsnFormatsEnum.inc> begin
	// clang-format off

	PPC_INSN_FORM_XOFORM_1,
	PPC_INSN_FORM_Z23FORM_RTAB5_CY2,
	PPC_INSN_FORM_DFORM_BASE,
	PPC_INSN_FORM_DXFORM,
	PPC_INSN_FORM_XFORM_BASE_R3XO_SWAPPED,
	PPC_INSN_FORM_DFORM_4,
	PPC_INSN_FORM_XFORM_ATTN,
	PPC_INSN_FORM_IFORM,
	PPC_INSN_FORM_VX_RD5_RSP5_PS1_XO9,
	PPC_INSN_FORM_VX_RD5_EO5_RS5_PS1_XO9,
	PPC_INSN_FORM_VXFORM_1,
	PPC_INSN_FORM_XLFORM_2,
	PPC_INSN_FORM_BFORM,
	PPC_INSN_FORM_EVXFORM_1,
	PPC_INSN_FORM_XFORM_BASE_R3XO,
	PPC_INSN_FORM_XFORM_16,
	PPC_INSN_FORM_DFORM_5,
	PPC_INSN_FORM_X_BF3_RS5_RS5,
	PPC_INSN_FORM_X_BF3_L1_RS5_RS5,
	PPC_INSN_FORM_XLFORM_1,
	PPC_INSN_FORM_XFORM_45,
	PPC_INSN_FORM_DCB_FORM,
	PPC_INSN_FORM_DCB_FORM_HINT,
	PPC_INSN_FORM_XFORM_ATB3,
	PPC_INSN_FORM_XFORM_AT3,
	PPC_INSN_FORM_XX2FORM_AT3_XBP5_P2,
	PPC_INSN_FORM_XX3FORM_AT3_XABP5_P1,
	PPC_INSN_FORM_DSS_FORM,
	PPC_INSN_FORM_EFXFORM_1,
	PPC_INSN_FORM_EFXFORM_3,
	PPC_INSN_FORM_EVXFORM_3,
	PPC_INSN_FORM_EVXFORM_D,
	PPC_INSN_FORM_EVXFORM_4,
	PPC_INSN_FORM_XSFORM_1,
	PPC_INSN_FORM_XFORM_24_SYNC,
	PPC_INSN_FORM_AFORM_1,
	PPC_INSN_FORM_XFORM_17,
	PPC_INSN_FORM_XFORM_XD6_RA5_RB5,
	PPC_INSN_FORM_XFORM_ICBT,
	PPC_INSN_FORM_AFORM_4,
	PPC_INSN_FORM_DFORM_1,
	PPC_INSN_FORM_DSFORM_1,
	PPC_INSN_FORM_DFORM_2_R0,
	PPC_INSN_FORM_DQFORM_RTP5_RA17_MEM,
	PPC_INSN_FORM_XX1FORM,
	PPC_INSN_FORM_DQ_RD6_RS5_DQ12,
	PPC_INSN_FORM_XFORM_XT6_IMM5,
	PPC_INSN_FORM_DQFORM_XTP5_RA17_MEM,
	PPC_INSN_FORM_XFORMMEMOP,
	PPC_INSN_FORM_VAFORM_1A,
	PPC_INSN_FORM_XFORM_MBAR,
	PPC_INSN_FORM_XLFORM_3,
	PPC_INSN_FORM_XFXFORM_3P,
	PPC_INSN_FORM_XFXFORM_3,
	PPC_INSN_FORM_XFXFORM_1,
	PPC_INSN_FORM_XFXFORM_5A,
	PPC_INSN_FORM_XFORM_SR,
	PPC_INSN_FORM_XFORM_SRIN,
	PPC_INSN_FORM_VXFORM_4,
	PPC_INSN_FORM_XFXFORM_5,
	PPC_INSN_FORM_XFLFORM_1,
	PPC_INSN_FORM_XLFORM_4,
	PPC_INSN_FORM_XFORM_MTMSR,
	PPC_INSN_FORM_VXFORM_5,
	PPC_INSN_FORM_VXFORM_RD5_XO5_RS5,
	PPC_INSN_FORM_DCBZL_FORM,
	PPC_INSN_FORM_PSFORM_QD,
	PPC_INSN_FORM_PSFORM_QI,
	PPC_INSN_FORM_PSFORM_Y,
	PPC_INSN_FORM_PSFORM_X,
	PPC_INSN_FORM_PSFORM_C,
	PPC_INSN_FORM_Z23FORM_1,
	PPC_INSN_FORM_XFORM_18,
	PPC_INSN_FORM_XFORM_20,
	PPC_INSN_FORM_Z23FORM_3,
	PPC_INSN_FORM_XLFORM_S,
	PPC_INSN_FORM_MDSFORM_1,
	PPC_INSN_FORM_MDFORM_1,
	PPC_INSN_FORM_MFORM_1,
	PPC_INSN_FORM_SCFORM,
	PPC_INSN_FORM_XFORM_44,
	PPC_INSN_FORM_XOFORM_RTAB5_L1,
	PPC_INSN_FORM_XFORM_HTM0,
	PPC_INSN_FORM_XFORM_HTM3,
	PPC_INSN_FORM_XFORM_HTM1,
	PPC_INSN_FORM_XFORM_TLBWS,
	PPC_INSN_FORM_XFORM_24,
	PPC_INSN_FORM_XFORM_HTM2,
	PPC_INSN_FORM_VXFORM_2,
	PPC_INSN_FORM_VXRFORM_1,
	PPC_INSN_FORM_VXFORM_BF3_VAB5,
	PPC_INSN_FORM_VXFORM_RD5_MP_VB5,
	PPC_INSN_FORM_VXFORM_RD5_N3_VB5,
	PPC_INSN_FORM_VAFORM_1,
	PPC_INSN_FORM_VXFORM_BX,
	PPC_INSN_FORM_VXFORM_CR,
	PPC_INSN_FORM_VNFORM_VTAB5_SD3,
	PPC_INSN_FORM_VAFORM_2,
	PPC_INSN_FORM_VXFORM_3,
	PPC_INSN_FORM_VXFORM_VTB5_RC,
	PPC_INSN_FORM_REQUIRES,
	PPC_INSN_FORM_XX2FORM,
	PPC_INSN_FORM_XX3FORM,
	PPC_INSN_FORM_XX3FORM_1,
	PPC_INSN_FORM_XX2_RD6_XO5_RS6,
	PPC_INSN_FORM_Z23FORM_8,
	PPC_INSN_FORM_XX2FORM_1,
	PPC_INSN_FORM_XX2_BF3_DCMX7_RS6,
	PPC_INSN_FORM_X_BF3_DCMX7_RS5,
	PPC_INSN_FORM_XX2_RD5_XO5_RS6,
	PPC_INSN_FORM_XX3FORM_AT3_XAB6,
	PPC_INSN_FORM_XX3FORM_RC,
	PPC_INSN_FORM_XX2_BF3_XO5_XB6_XO9,
	PPC_INSN_FORM_XX2_RD6_DCMX7_RS6,
	PPC_INSN_FORM_XX2_RD6_UIM5_RS6,
	PPC_INSN_FORM_XFORM_XT6_IMM5_VB5,
	PPC_INSN_FORM_XX3FORM_2,
	PPC_INSN_FORM_XX4FORM,
	PPC_INSN_FORM_X_RD6_IMM8,
	PPC_INSN_FORM_XX2FORM_2,
	PPC_INSN_FORM_BFORM_3,
	PPC_INSN_FORM_BFORM_3_AT,

	// clang-format on
	// generated content <PPCGenCSInsnFormatsEnum.inc> end
} ppc_insn_form;

static inline bool ppc_is_b_form(ppc_insn_form form)
{
	switch (form) {
	default:
		return false;
	case PPC_INSN_FORM_BFORM:
	case PPC_INSN_FORM_BFORM_3:
	case PPC_INSN_FORM_BFORM_3_AT:
		return true;
	}
}

/// Masks for specific fields
/// Left most bit is bit 0 according to ISA
#define PPC_INSN_FORM_B_BO_MASK 0x03e00000
#define PPC_INSN_FORM_XL_BO_MASK 0x03e00000
#define PPC_INSN_FORM_B_BI_MASK 0x001f0000
#define PPC_INSN_FORM_XL_BI_MASK 0x001f0000
#define PPC_INSN_FORM_XL_BH_MASK 0x00001800
#define PPC_INSN_FORM_XL_XO_MASK 0x000007fe

typedef struct {
	ppc_insn_form form;
} ppc_suppl_info;

#define NUM_PPC_OPS 8

/// Instruction structure
typedef struct cs_ppc {
	/// branch code for branch instructions
	ppc_bc bc;

	/// if update_cr0 = True, then this 'dot' insn updates CR0
	bool update_cr0;

	///< The instruction format. Can be use to determine the bit encoding of the instruction.
	ppc_insn_form format;

	/// Number of operands of this instruction,
	/// or 0 when instruction has no operand.
	uint8_t op_count;
	cs_ppc_op operands[NUM_PPC_OPS]; ///< operands for this instruction.
} cs_ppc;

#ifdef __cplusplus
}
#endif

#endif
