// Copyright (c) 2025 .NET Foundation and Contributors. All rights reserved.
// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// See the LICENSE file in the project root for full license information.

using System.Runtime.Versioning;

using Android.OS;
using Android.Text;
using Android.Views;
using Android.Widget;

using Observable = System.Reactive.Linq.Observable;

namespace ReactiveUI;

/// <summary>
/// Android view objects are not Generally Observable™, so hard-code some
/// particularly useful types.
/// </summary>
[Preserve(AllMembers = true)]
public class AndroidObservableForWidgets : ICreatesObservableForProperty
{
    private static readonly Dictionary<(Type viewType, string? propertyName), Func<object, Expression, IObservable<IObservedChange<object, object?>>>> _dispatchTable;

    [ObsoletedOSPlatform("android23.0")]
    [SupportedOSPlatform("android23.0")]
    static AndroidObservableForWidgets() =>
        _dispatchTable = new[]
        {
            CreateFromWidget<TextView, TextChangedEventArgs>(static v => v.Text, static (v, h) => v.TextChanged += h, static (v, h) => v.TextChanged -= h),
            CreateFromWidget<NumberPicker, NumberPicker.ValueChangeEventArgs>(static v => v.Value, static (v, h) => v.ValueChanged += h, static (v, h) => v.ValueChanged -= h),
            CreateFromWidget<RatingBar, RatingBar.RatingBarChangeEventArgs>(static v => v.Rating, static (v, h) => v.RatingBarChange += h, static (v, h) => v.RatingBarChange -= h),
            CreateFromWidget<CompoundButton, CompoundButton.CheckedChangeEventArgs>(static v => v.Checked, static (v, h) => v.CheckedChange += h, static (v, h) => v.CheckedChange -= h),
            CreateFromWidget<CalendarView, CalendarView.DateChangeEventArgs>(static v => v.Date, static (v, h) => v.DateChange += h, static (v, h) => v.DateChange -= h),
            CreateFromWidget<TabHost, TabHost.TabChangeEventArgs>(static v => v.CurrentTab, static (v, h) => v.TabChanged += h, static (v, h) => v.TabChanged -= h),
            CreateTimePickerHourFromWidget(),
            CreateTimePickerMinuteFromWidget(),
            CreateFromAdapterView(),
        }.ToDictionary(static k => (viewType: k.Type, propertyName: k.Property), static v => v.Func);

    /// <inheritdoc/>
#if NET6_0_OR_GREATER
    [RequiresDynamicCode("GetAffinityForObject uses reflection for property access and type checking which require dynamic code generation")]
    [RequiresUnreferencedCode("GetAffinityForObject uses reflection for property access and type checking which may require unreferenced code")]
#endif
    public int GetAffinityForObject(Type type, string propertyName, bool beforeChanged = false)
    {
        if (beforeChanged)
        {
            return 0;
        }

        return _dispatchTable.Keys.Any(x => x.viewType?.IsAssignableFrom(type) == true && x.propertyName?.Equals(propertyName) == true) ? 5 : 0;
    }

    /// <inheritdoc/>
#if NET6_0_OR_GREATER
    [RequiresDynamicCode("GetNotificationForProperty uses reflection for property access and type checking which require dynamic code generation")]
    [RequiresUnreferencedCode("GetNotificationForProperty uses reflection for property access and type checking which may require unreferenced code")]
#endif
    public IObservable<IObservedChange<object, object?>> GetNotificationForProperty(object sender, Expression expression, string propertyName, bool beforeChanged = false, bool suppressWarnings = false)
    {
        var type = sender?.GetType();
        var tableItem = _dispatchTable.Keys.First(x => x.viewType?.IsAssignableFrom(type) == true && x.propertyName?.Equals(propertyName) == true);

        return !_dispatchTable.TryGetValue(tableItem, out var dispatchItem) ?
                   Observable.Never<IObservedChange<object, object?>>() :
                   dispatchItem.Invoke(sender!, expression);
    }

    private static DispatchItem CreateFromAdapterView()
    {
        // AdapterView is more complicated because there are two events - one for select and one for deselect
        // respond to both
        const string PropName = "SelectedItem";

        return new DispatchItem(
                                typeof(AdapterView),
                                PropName,
                                (x, ex) =>
                                {
                                    var adapterView = (AdapterView)x;

                                    var itemSelected =
                                        Observable
                                            .FromEvent<EventHandler<AdapterView.ItemSelectedEventArgs>, ObservedChange<object, object?>
                                            >(
                                              eventHandler =>
                                              {
                                                  void Handler(object? sender, AdapterView.ItemSelectedEventArgs e) =>
                                                      eventHandler(new ObservedChange<object, object?>(adapterView, ex, default));

                                                  return Handler;
                                              },
                                              h => adapterView.ItemSelected += h,
                                              h => adapterView.ItemSelected -= h);

                                    var nothingSelected =
                                        Observable
                                            .FromEvent<EventHandler<AdapterView.NothingSelectedEventArgs>,
                                                ObservedChange<object, object?>>(
                                             eventHandler =>
                                             {
                                                 void Handler(object? sender, AdapterView.NothingSelectedEventArgs e) =>
                                                     eventHandler(new ObservedChange<object, object?>(adapterView, ex, default));

                                                 return Handler;
                                             },
                                             h => adapterView.NothingSelected += h,
                                             h => adapterView.NothingSelected -= h);

                                    return itemSelected.Merge(nothingSelected);
                                });
    }

    [ObsoletedOSPlatform("android23.0")]
    [SupportedOSPlatform("android23.0")]
    private static DispatchItem CreateTimePickerHourFromWidget()
    {
        if ((int)Build.VERSION.SdkInt >= 23)
        {
            return CreateFromWidget<TimePicker, TimePicker.TimeChangedEventArgs>(static v => v.Hour, static (v, h) => v.TimeChanged += h, static (v, h) => v.TimeChanged -= h);
        }

        return CreateFromWidget<TimePicker, TimePicker.TimeChangedEventArgs>(static v => v.CurrentHour, static (v, h) => v.TimeChanged += h, static (v, h) => v.TimeChanged -= h);
    }

    [ObsoletedOSPlatform("android23.0")]
    [SupportedOSPlatform("android23.0")]
    private static DispatchItem CreateTimePickerMinuteFromWidget()
    {
        if ((int)Build.VERSION.SdkInt >= 23)
        {
            return CreateFromWidget<TimePicker, TimePicker.TimeChangedEventArgs>(static v => v.Minute, static (v, h) => v.TimeChanged += h, static (v, h) => v.TimeChanged -= h);
        }

        return CreateFromWidget<TimePicker, TimePicker.TimeChangedEventArgs>(static v => v.CurrentMinute, static (v, h) => v.TimeChanged += h, static (v, h) => v.TimeChanged -= h);
    }

    [SuppressMessage("Trimming", "IL2026:Members annotated with 'RequiresUnreferencedCodeAttribute' require dynamic access otherwise can break functionality when trimming application code", Justification = "Marked as Preserve")]
    [SuppressMessage("AOT", "IL3050:Calling members annotated with 'RequiresDynamicCodeAttribute' may break functionality when AOT compiling.", Justification = "Marked as Preserve")]
    private static DispatchItem CreateFromWidget<TView, TEventArgs>(Expression<Func<TView, object?>> property, Action<TView, EventHandler<TEventArgs>> addHandler, Action<TView, EventHandler<TEventArgs>> removeHandler)
        where TView : View
        where TEventArgs : EventArgs
    {
        var memberInfo = property.Body.GetMemberInfo() ?? throw new ArgumentException("Does not have a valid body member info.", nameof(property));

        // ExpressionToPropertyNames is used here as it handles boxing expressions that might
        // occur due to our use of object
        var propName = memberInfo.Name;

        return new DispatchItem(
                                typeof(TView),
                                propName,
                                (x, ex) =>
                                {
                                    var view = (TView)x;

                                    return Observable.FromEvent<EventHandler<TEventArgs>, ObservedChange<object, object?>>(
                                     eventHandler =>
                                     {
                                         void Handler(object? sender, TEventArgs e) =>
                                             eventHandler(new ObservedChange<object, object?>(view, ex, default));

                                         return Handler;
                                     },
                                     h => addHandler(view, h),
                                     h => removeHandler(view, h));
                                });
    }

    private sealed record DispatchItem
    {
        public DispatchItem(
            Type type,
            string? property,
            Func<object, Expression, IObservable<IObservedChange<object, object?>>> func) =>
            (Type, Property, Func) = (type, property, func);

        public Type Type { get; }

        public string? Property { get; }

        public Func<object, Expression, IObservable<IObservedChange<object, object?>>> Func { get; }
    }
}
