// Copyright (c) 2025 .NET Foundation and Contributors. All rights reserved.
// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// See the LICENSE file in the project root for full license information.

using System.IO;
using System.Text.Json;
using Foundation;

namespace ReactiveUI;

/// <summary>
/// Loads and saves state to persistent storage.
/// </summary>
public class AppSupportJsonSuspensionDriver : ISuspensionDriver
{
    /// <inheritdoc/>
#if NET6_0_OR_GREATER
    [RequiresDynamicCode("LoadState implementations may use serialization which requires dynamic code generation")]
    [RequiresUnreferencedCode("LoadState implementations may use serialization which may require unreferenced code")]
#endif
    public IObservable<object?> LoadState()
    {
        try
        {
            var target = Path.Combine(CreateAppDirectory(NSSearchPathDirectory.ApplicationSupportDirectory), "state.dat");

            var result = default(object);
            using (var st = File.OpenRead(target))
            {
                result = JsonSerializer.Deserialize<object>(st);
            }

            return Observable.Return(result);
        }
        catch (Exception ex)
        {
            return Observable.Throw<object>(ex);
        }
    }

    /// <inheritdoc/>
#if NET6_0_OR_GREATER
    [RequiresDynamicCode("SaveState implementations may use serialization which requires dynamic code generation")]
    [RequiresUnreferencedCode("SaveState implementations may use serialization which may require unreferenced code")]
#endif
    public IObservable<Unit> SaveState(object state)
    {
        try
        {
            var target = Path.Combine(CreateAppDirectory(NSSearchPathDirectory.ApplicationSupportDirectory), "state.dat");

            using (var st = File.Open(target, FileMode.Create))
            {
                JsonSerializer.Serialize(st, state);
            }

            return Observables.Unit;
        }
        catch (Exception ex)
        {
            return Observable.Throw<Unit>(ex);
        }
    }

    /// <inheritdoc/>
#if NET6_0_OR_GREATER
    [RequiresDynamicCode("InvalidateState uses JsonSerializer which requires dynamic code generation")]
    [RequiresUnreferencedCode("InvalidateState uses JsonSerializer which may require unreferenced code")]
#endif
    public IObservable<Unit> InvalidateState()
    {
        try
        {
            var target = Path.Combine(CreateAppDirectory(NSSearchPathDirectory.ApplicationSupportDirectory), "state.dat");
            File.Delete(target);

            return Observables.Unit;
        }
        catch (Exception ex)
        {
            return Observable.Throw<Unit>(ex);
        }
    }

    private static string CreateAppDirectory(NSSearchPathDirectory targetDir, string subDir = "Data")
    {
        var fm = new NSFileManager();
        var url = fm.GetUrl(targetDir, NSSearchPathDomain.All, null, true, out _);
        var ret = Path.Combine(url.RelativePath!, NSBundle.MainBundle.BundleIdentifier, subDir);
        if (!Directory.Exists(ret))
        {
            Directory.CreateDirectory(ret);
        }

        return ret;
    }
}
