/*
 * COPYRIGHT:       See COPYING in the top level directory
 * PROJECT:         ReactOS EventCreate Command
 * FILE:            base/applications/cmdutils/eventcreate/evtmsggen.c
 * PURPOSE:         Generator for the event message string templates file.
 *                  Creates the message string file in the current directory.
 * PROGRAMMER:      Hermes Belusca-Maito
 *
 * You can compile this generator:
 *   with GCC : $ gcc -o evtmsggen.exe evtmsggen.c
 *   with MSVC: $ cl evtmsggen.c (or: $ cl /Fe"evtmsggen.exe" evtmsggen.c)
 */

#include <stdio.h>

/*
 * Enable/disable this option to use "English" for the message table language.
 * The default behaviour when the option is disabled selects "Neutral" language.
 */
// #define ENGLISH

/* The default End-Of-Line control for the message file */
#define EOL     "\r\n"


static void usage(char* name)
{
    fprintf(stdout, "Usage: %s ID_min ID_max outfile.mc\n", name);
}

int main(int argc, char* argv[])
{
    FILE* outFile;
    unsigned int id_min, id_max;
    unsigned int i;

    /* Validate the arguments */
    if (argc != 4)
    {
        usage(argv[0]);
        return -1;
    }

    id_min = (unsigned int)atoi(argv[1]);
    id_max = (unsigned int)atoi(argv[2]);
    if (id_min > id_max)
    {
        fprintf(stderr, "ERROR: Min ID %u cannot be strictly greater than Max ID %u !\n", id_min, id_max);
        return -1;
    }

    /* Open the file */
    outFile = fopen(argv[3], "wb");
    if (!outFile)
    {
        fprintf(stderr, "ERROR: Could not create output file '%s'.\n", argv[3]);
        return -1;
    }

    /* Generate the file */

    /* Write the header */
    fprintf(outFile,
        ";/*"    EOL
        "; * %s" EOL
        "; * Contains event message string templates for the EventCreate Command." EOL
        "; *"    EOL
        "; * This file is autogenerated, do not edit." EOL
        "; * Generated with:" EOL
        "; *   %s %u %u %s"   EOL
        "; */"   EOL
        EOL
#ifdef ENGLISH
        "LanguageNames=(English=0x409:MSG00409)"  EOL
#else
        "LanguageNames=(Neutral=0x0000:MSG00000)" EOL
#endif
        "MessageIdTypedef=DWORD" EOL
        EOL,
        argv[3],
        argv[0], id_min /* argv[1] */, id_max /* argv[2] */, argv[3]);

    /* Write the message string templates */
    for (i = id_min; i <= id_max; ++i)
    {
        fprintf(outFile,
            "MessageId=0x%X"   EOL
#ifdef ENGLISH
            "Language=English" EOL
#else
            "Language=Neutral" EOL
#endif
            "%%1" EOL
            "."   EOL
            EOL,
            i);
    }

    /* Close the file */
    fclose(outFile);

    return 0;
}

/* EOF */
