/*
 * COPYRIGHT:   See COPYING in the top level directory
 * PROJECT:     ReactOS system libraries
 * FILE:        lib/sdk/crt/mbstring/hanzen.c
 * PURPOSE:     Multibyte conversion routines formerly called hantozen and zentohan
 * PROGRAMER:   Ariadne, Taiji Yamada
 * UPDATE HISTORY:
		Modified from Taiji Yamada japanese code system utilities
 *              12/04/99: Created
 */

#include <precomp.h>
#include <mbstring.h>
#include <locale.h>

/* Maps cp932 single byte character to multi byte character */
static const unsigned char mbbtombc_932[] = {
  0x40,0x49,0x68,0x94,0x90,0x93,0x95,0x66,0x69,0x6a,0x96,0x7b,0x43,0x7c,0x44,0x5e,
  0x4f,0x50,0x51,0x52,0x53,0x54,0x55,0x56,0x57,0x58,0x46,0x47,0x83,0x81,0x84,0x48,
  0x97,0x60,0x61,0x62,0x63,0x64,0x65,0x66,0x67,0x68,0x69,0x6a,0x6b,0x6c,0x6d,0x6e,
  0x6f,0x70,0x71,0x72,0x73,0x74,0x75,0x76,0x77,0x78,0x79,0x6d,0x8f,0x6e,0x4f,0x76,
  0x77,0x78,0x79,0x6d,0x8f,0x6e,0x4f,0x51,0x65,0x81,0x82,0x83,0x84,0x85,0x86,0x87,
  0x88,0x89,0x8a,0x8b,0x8c,0x8d,0x8e,0x8f,0x90,0x91,0x92,0x93,0x94,0x95,0x50,
       0x42,0x75,0x76,0x41,0x45,0x92,0x40,0x42,0x44,0x46,0x48,0x83,0x85,0x87,0x62,
  0x5b,0x41,0x43,0x45,0x47,0x49,0x4a,0x4c,0x4e,0x50,0x52,0x54,0x56,0x58,0x5a,0x5c,
  0x5e,0x60,0x63,0x65,0x67,0x69,0x6a,0x6b,0x6c,0x6d,0x6e,0x71,0x74,0x77,0x7a,0x7d,
  0x7e,0x80,0x81,0x82,0x84,0x86,0x88,0x89,0x8a,0x8b,0x8c,0x8d,0x8f,0x93,0x4a,0x4b };

/* Maps multibyte cp932 punctuation marks to single byte equivalents */
static const unsigned char mbctombb_932_punct[] = {
  0x20,0xa4,0xa1,0x2c,0x2e,0xa5,0x3a,0x3b,0x3f,0x21,0xde,0xdf,0x00,0x00,0x00,0x5e,
  0x7e,0x5f,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xb0,0x00,0x00,0x2f,0x00,
  0x00,0x00,0x7c,0x00,0x00,0x60,0x27,0x00,0x22,0x28,0x29,0x00,0x00,0x5b,0x5d,0x7b,
  0x7d,0x00,0x00,0x00,0x00,0xa2,0xa3,0x00,0x00,0x00,0x00,0x2b,0x2d,0x00,0x00,0x00,
  0x00,0x3d,0x00,0x3c,0x3e,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x5c,
  0x24,0x00,0x00,0x25,0x23,0x26,0x2a,0x40};

/* Maps multibyte cp932 hiragana/katakana to single-byte equivalents */
static const unsigned char mbctombb_932_kana[] = {
  0xa7,0xb1,0xa8,0xb2,0xa9,0xb3,0xaa,0xb4,0xab,0xb5,0xb6,0xb6,0xb7,0xb7,0xb8,0xb8,
  0xb9,0xb9,0xba,0xba,0xbb,0xbb,0xbc,0xbc,0xbd,0xbd,0xbe,0xbe,0xbf,0xbf,0xc0,0xc0,
  0xc1,0xc1,0xaf,0xc2,0xc2,0xc3,0xc3,0xc4,0xc4,0xc5,0xc6,0xc7,0xc8,0xc9,0xca,0xca,
  0xca,0xcb,0xcb,0xcb,0xcc,0xcc,0xcc,0xcd,0xcd,0xcd,0xce,0xce,0xce,0xcf,0xd0,0xd1,
  0xd2,0xd3,0xac,0xd4,0xad,0xd5,0xae,0xd6,0xd7,0xd8,0xd9,0xda,0xdb,0xdc,0xdc,0xb2,
  0xb4,0xa6,0xdd,0xb3,0xb6,0xb9};

/*********************************************************************
 *         _mbbtombc(MSVCRT.@)
 */
unsigned int __cdecl _mbbtombc(unsigned int c)
{
  if(get_mbcinfo()->mbcodepage == 932)
  {
    if(c >= 0x20 && c <= 0x7e) {
      if((c >= 0x41 && c <= 0x5a) || (c >= 0x61 && c <= 0x7a) || (c >= 0x30 && c <= 0x39))
        return mbbtombc_932[c - 0x20] | 0x8200;
      else
        return mbbtombc_932[c - 0x20] | 0x8100;
    }
    else if(c >= 0xa1 && c <= 0xdf) {
      if(c >= 0xa6 && c <= 0xdd && c != 0xb0)
        return mbbtombc_932[c - 0xa1 + 0x5f] | 0x8300;
      else
        return mbbtombc_932[c - 0xa1 + 0x5f] | 0x8100;
    }
  }
  return c;  /* not Japanese or no MB char */
}

 /*********************************************************************
 *         _mbctombb (MSVCRT.@)
 */
unsigned int CDECL _mbctombb(unsigned int c)
{
    unsigned int value;

    if(get_mbcinfo()->mbcodepage == 932)
    {
        if(c >= 0x829f && c <= 0x82f1)    /* Hiragana */
            return mbctombb_932_kana[c - 0x829f];
        if(c >= 0x8340 && c <= 0x8396 && c != 0x837f)    /* Katakana */
            return mbctombb_932_kana[c - 0x8340 - (c >= 0x837f ? 1 : 0)];
        if(c >= 0x8140 && c <= 0x8197)    /* Punctuation */
        {
            value = mbctombb_932_punct[c - 0x8140];
            return value ? value : c;
        }
        if((c >= 0x824f && c <= 0x8258) || /* Fullwidth digits */
           (c >= 0x8260 && c <= 0x8279))   /* Fullwidth capitals letters */
            return c - 0x821f;
        if(c >= 0x8281 && c <= 0x829a)     /* Fullwidth small letters */
            return c - 0x8220;
        /* all other cases return c */
    }
    return c;
}



