---
title: 'TelegraphChat'
description: ''
category: 'Models'
fullscreen: false 
position: 41
---

Chat informations are stored in database inside a `telegraph_chats` table and can be retrieved using `DefStudio\Telegraph\Models\TelegaphChat` model or using a custom Chat model.

## Custom Chat Model

To customize on your own Chat model, make sure that your custom model extends the `DefStudio\Telegraph\Models\TelegraphChat`, e.g. `App\Models\Chat`, it will looks like this:

```php
<?php

namespace App\Models;

use DefStudio\Telegraph\Models\TelegraphChat as BaseModel;

class Chat extends BaseModel
{
    
}
```

You should specify the class name of your model in the `models.chat` key of the telegraph config file.

```php
'models' => [
    'chat' => App\Models\Chat::class,
],
```

## Available methods

### `info()`

Retrieves the chat data from telegram

```php
/** @var \DefStudio\Telegraph\Models\TelegraphChat $telegraphChat */

$telegraphChat->info();

/*
id: xxxxx
type: group
title: my telegram group
...
*/
```

### `memberCount()`

Retrieves the chat member count from telegram

```php
/** @var \DefStudio\Telegraph\Models\TelegraphChat $telegraphChat */

$telegraphChat->info(); /* 42 */
```

## Optional parameters

Attachments methods only supports required parameters, optional parameters can be sent through Telegraph `->withData()` method:

```php
$telegraphChat->message('hi')->withData('caption', 'test')->send();
$telegraphChat->withData('caption', 'test')->message('hi')->send();
```

### `message()`

Starts a `Telegraph` call to send a message

```php
/** @var \DefStudio\Telegraph\Models\TelegraphChat $telegraphChat */

$telegraphChat->message('hello!')->send();
```

### `html()`

Starts a `Telegraph` call to send a message using html formatting

```php
/** @var \DefStudio\Telegraph\Models\TelegraphChat $telegraphChat */

$telegraphChat->html('<b>hello</b>')->send();
```

### `markdown()`

Starts a `Telegraph` call to send a message using markdown formatting

```php
/** @var \DefStudio\Telegraph\Models\TelegraphChat $telegraphChat */

$telegraphChat->markdown('*hello*')->send();
```

### `markdownV2()`

Starts a `Telegraph` call to send a message using markdown V2 formatting

```php
/** @var \DefStudio\Telegraph\Models\TelegraphChat $telegraphChat */

$telegraphChat->markdownV2('*hello*')->send();
```

### `edit()`

Starts a `Telegraph` call to edit a message

```php
/** @var \DefStudio\Telegraph\Models\TelegraphChat $telegraphChat */

$telegraphChat->edit($messageId)->message('new text')->send();
```

### `editCaption()`

Starts a `Telegraph` call to edit an attachment's caption

```php
/** @var \DefStudio\Telegraph\Models\TelegraphChat $telegraphChat */

$telegraphChat->editCaption($messageId)->message('new caption')->send();
```

### `editMedia()`

Starts a `Telegraph` call to edit a media messages with a new media content

```php
/** @var \DefStudio\Telegraph\Models\TelegraphChat $telegraphChat */

$telegraphChat->editMedia($messageId)->photo($path)->send();
$telegraphChat->editMedia($messageId)->document($path)->send();
$telegraphChat->editMedia($messageId)->animation($path)->send();
$telegraphChat->editMedia($messageId)->video($path)->send();
$telegraphChat->editMedia($messageId)->audio($path)->send();
```

### `replaceKeyboard()`

Starts a `Telegraph` call to replace a message keyboard (see [keyboards](features/keyboards) for details)

```php
/** @var \DefStudio\Telegraph\Models\TelegraphChat $telegraphChat */

$telegraphChat->replaceKeyboard(
    $messageId, 
    Keyboard::make()->buttons([
        Button::make('open')->url('https://test.dev')
    ])
)->send();
```

### `deleteKeyboard()`

Starts a `Telegraph` call to remove a message keyboard (see [keyboards](features/keyboards) for details)

```php
/** @var \DefStudio\Telegraph\Models\TelegraphChat $telegraphChat */

$telegraphChat->deleteKeyboard($messageId)->send();
```

<alert type="alert">Follow [installation](installation#set-up) instructions for creating the database tables</alert>

### `deleteMessage()`

Starts a `Telegraph` call to delete a message

```php
/** @var \DefStudio\Telegraph\Models\TelegraphChat $telegraphChat */

$telegraphChat->deleteMessage($messageId)->send();
```

### `forwardMessage()`

forwards a message from another chat

```php
/** @var \DefStudio\Telegraph\Models\TelegraphChat $telegraphChat */

$telegraphChat->forwardMessage($fromChat, $messageId)->send();
```

### `pinMessage()`

Starts a `Telegraph` call to pin a message

```php
/** @var \DefStudio\Telegraph\Models\TelegraphChat $telegraphChat */

$telegraphChat->pinMessage($messageId)->send();
```

### `unpinMessage()`

Starts a `Telegraph` call to unpin a message

```php
/** @var \DefStudio\Telegraph\Models\TelegraphChat $telegraphChat */

$telegraphChat->pinMessage($messageId)->send();
```

### `unpinAllMessages()`

Starts a `Telegraph` call to unpin all messages

```php
/** @var \DefStudio\Telegraph\Models\TelegraphChat $telegraphChat */

$telegraphChat->unpinAllMessages()->send();
```

### `document()`

sends a document

```php
$telegraphChat->document($documentPath)->send();
```

### `location()`

sends a location attachment

```php
$telegraphChat->location(12.345, -54.321)->send();
````

### `action()`

Tells the chat users that something is happening on the bot's side. The status is set for up to 5 seconds or when a new message is received from the bot.

<img src="screenshots/chat-action.png" />

```php
use DefStudio\Telegraph\Enums\ChatActions;
use DefStudio\Telegraph\Models\TelegraphChat;

/** @var TelegraphChat $telegraphChat */

$telegraphChat->action(ChatActions::TYPING)->send();
```

## `photo()`

sends a photo

```php
use DefStudio\Telegraph\Enums\ChatActions;
use DefStudio\Telegraph\Models\TelegraphChat;

/** @var TelegraphChat $telegraphChat */

$telegraphChat->photo(Storage::path('photo.jpg'))->send();
```


## `voice()`

sends a vocal message

```php
use DefStudio\Telegraph\Enums\ChatActions;
use DefStudio\Telegraph\Models\TelegraphChat;

/** @var TelegraphChat $telegraphChat */

$telegraphChat->voice(Storage::path('voice.ogg'))->send();
```


## `setBaseUrl()`

allows to override Telegram API url on a per-message basis:

```php
use DefStudio\Telegraph\Enums\ChatActions;
use DefStudio\Telegraph\Models\TelegraphChat;

/** @var TelegraphChat $telegraphChat */

$telegraphChat->setBaseUrl('https://my-secret-server.dev')->message('secret message')->send();
```


## `setTitle()`

sets chat title

```php
use DefStudio\Telegraph\Enums\ChatActions;
use DefStudio\Telegraph\Models\TelegraphChat;

/** @var TelegraphChat $telegraphChat */

$telegraphChat->setTitle("my chat")->send();
```



## `setDescription()`

sets chat description

```php
use DefStudio\Telegraph\Enums\ChatActions;
use DefStudio\Telegraph\Models\TelegraphChat;

/** @var TelegraphChat $telegraphChat */

$telegraphChat->setDescription("a test chat with my bot")->send();
```



## `setChatPhoto()`

sets chat profile photo

```php
use DefStudio\Telegraph\Enums\ChatActions;
use DefStudio\Telegraph\Models\TelegraphChat;

/** @var TelegraphChat $telegraphChat */

$telegraphChat->setChatPhoto(Storage::path('photo.jpg'))->send();
```


## `generatePrimaryInviteLink()`

generates a new primary invite link for a chat. Any previously generated primary link is revoked. For more info, see telegram [bot documentation](https://core.telegram.org/bots/api#exportchatinvitelink)


```php
use DefStudio\Telegraph\Models\TelegraphChat;

/** @var TelegraphChat $telegraphChat */

$telegraphChat->generatePrimaryInviteLink()->send();
```

## `createInviteLink()`

creates an additional invite link for a chat. For more info about options, see telegram [bot documentation](https://core.telegram.org/bots/api#createchatinvitelink)


```php
use DefStudio\Telegraph\Models\TelegraphChat;

/** @var TelegraphChat $telegraphChat */

$telegraphChat->createInviteLink()
    ->name('September promotional link')    //optional
    ->expire(today()->addMonth())           //optional
    ->memberLimit(42)                       //optional
    ->withJoinRequest()                     //optional
    ->send();
```

## `editInviteLink()`

edits an existing invite link for a chat. For more info about options, see telegram [bot documentation](https://core.telegram.org/bots/api#editchatinvitelink)

```php
use DefStudio\Telegraph\Models\TelegraphChat;

/** @var TelegraphChat $telegraphChat */

$telegraphChat->editInviteLink('http://t.me/123456')
    ->name('new name')               //optional
    ->expire(today()->addYear())     //optional
    ->memberLimit(12)                //optional
    ->withJoinRequest(false)         //optional
    ->send();
```

## `revokeInviteLink()`

revokes an existing invite link for a chat. For more info, see telegram [bot documentation](https://core.telegram.org/bots/api#revokechatinvitelink)

```php
use DefStudio\Telegraph\Models\TelegraphChat;

/** @var TelegraphChat $telegraphChat */

$telegraphChat->revokeInviteLink('http://t.me/123456')->send();
```

## `setPermissions()`

set users permissions for a chat. For more info, see telegram [bot documentation](https://core.telegram.org/bots/api#setchatpermissions)

```php
use DefStudio\Telegraph\Models\TelegraphChat;

/** @var TelegraphChat $telegraphChat */

$telegraphChat->setPermissions([
    ChatPermissions::CAN_INVITE_USERS,
    ChatPermissions::CAN_CHANGE_INFO,
    ChatPermissions::CAN_ADD_WEB_PAGE_PREVIEWS => true,
    ChatPermissions::CAN_SEND_MESSAGES => false,
])->send();
```

## `banMember()`

ban a user in a group, a supergroup or a channel. In the case of supergroups and channels, the user will not be able to return to the chat on their own using invite links. For more info, see telegram [bot documentation](https://core.telegram.org/bots/api#banchatmember)

```php
use DefStudio\Telegraph\Models\TelegraphChat;

/** @var TelegraphChat $telegraphChat */

$telegraphChat->banMember($userId)
    ->until(now()->addDay());      //optional, only for supergroups and channels
    ->andRevokeMessages()          //optional, always true for supergroups and channels
    ->send();
```

## `unbanMember()`

unban a user in a group, a supergroup or a channel. For more info, see telegram [bot documentation](https://core.telegram.org/bots/api#unbanchatmember)

```php
use DefStudio\Telegraph\Models\TelegraphChat;

/** @var TelegraphChat $telegraphChat */

$telegraphChat->unbanMember($userId)->send();
```

## `restrictMember()`

restrict a user in a group, a supergroup or a channel from taking the give actions. For more info, see telegram [bot documentation](https://core.telegram.org/bots/api#restrictchatmember)

```php
use DefStudio\Telegraph\Models\TelegraphChat;

/** @var TelegraphChat $telegraphChat */

$telegraphChat->restrictMember($userId[
       DefStudio\Telegraph\Enums\ChatPermissions::CAN_PIN_MESSAGES => false,
       DefStudio\Telegraph\Enums\ChatPermissions::CAN_INVITE_USERS => true,
       DefStudio\Telegraph\Enums\ChatPermissions::CAN_SEND_MESSAGES,
    ])
    ->until(now()->addDay())        //optional+
    ->send();
```

## `promoteMember()`

promotes a user in a group, a supergroup or a channel to administrator status. For more info, see telegram [bot documentation](https://core.telegram.org/bots/api#promotechatmember)

```php
use DefStudio\Telegraph\Models\TelegraphChat;

/** @var TelegraphChat $telegraphChat */

$telegraphChat->promoteMember($userid, [
       DefStudio\Telegraph\Enums\ChatAdminPermissions::CAN_PIN_MESSAGES => false,
       DefStudio\Telegraph\Enums\ChatAdminPermissions::CAN_INVITE_USERS => true,
       DefStudio\Telegraph\Enums\ChatAdminPermissions::CAN_CHANGE_INFO,
    ])
    ->send();
```

## `demoteMember()`

demote a user in a group, a supergroup or a channel from administrator status.

```php
use DefStudio\Telegraph\Models\TelegraphChat;

/** @var TelegraphChat $telegraphChat */
$telegraphChat->demoteMember($userid)->send();
```


## `poll`

creates a native poll. For more info, see telegram [bot documentation](https://core.telegram.org/bots/api#sendpoll)

```php
use DefStudio\Telegraph\Models\TelegraphChat;

/** @var TelegraphChat $telegraphChat */
$telegraphChat->poll("What's your favourite programming language?")
    ->option('php')
    ->option('typescript')
    ->option('rust')
    ->allowMultipleAnswers()
    ->validUntil(now()->addMinutes(5))
    ->send();
```

## `quiz`

creates a quiz. For more info, see telegram [bot documentation](https://core.telegram.org/bots/api#sendpoll)

```php
/** @var DefStudio\Telegraph\Models\TelegraphChat $telegraphChat */
$telegraphChat->quiz("What's your favourite programming language?")
    ->option('php', correct: true)
    ->option('typescript')
    ->option('rust')
    ->explanation('We all love php, right?')
    ->validUntil(now()->addMinutes(5))
    ->send();
```

### `dice`

An animated emoji attachment that will display a random value can be sent through Telegraph `->dice()` method:

```php
/** @var DefStudio\Telegraph\Models\TelegraphChat $telegraphChat */
$telegraphChat->dice()->send();
```

### `animation`

An animation attachment can be sent through Telegraph `->animation()` method:

```php
/** @var DefStudio\Telegraph\Models\TelegraphChat $telegraphChat */
$telegraphChat->animation()->send();
```

### `video`

A video attachment can be sent through Telegraph `->video()` method:

```php
/** @var DefStudio\Telegraph\Models\TelegraphChat $telegraphChat */
$telegraphChat->video()->send();
```

### `audio`

An audio attachment can be sent through Telegraph `->audio()` method:

```php
/** @var DefStudio\Telegraph\Models\TelegraphChat $telegraphChat */
$telegraphChat->audio()->send();
```

### `menuButton`

Retrieves chat menu button

```php
/** @var DefStudio\Telegraph\Models\TelegraphChat $telegraphChat */
$button = $telegraphChat->menuButton()->send();
```

### `setMenuButton`

set chat menu button

```php
/** @var DefStudio\Telegraph\Models\TelegraphChat $telegraphChat */
$telegraphChat->setMenuButton()->default()->send(); //restore default 
$telegraphChat->setMenuButton()->commands()->send(); //show bot commands in menu button 
$telegraphChat->setMenuButton()->webApp("Web App", "https://my-web.app")->send(); //show start web app button 
```
