# RFC: Property Animation

* **Authors**: Ib Green
* **Date**: Aug 2018 (Initial version Aug 2017)
* **Status**: **Draft**

> Note: This RFC Makes a distinction between "animations" and "transitions". This RFC is about "animations", for more information about property transitions, see the complementary RFC about that topic.

References:
* [Animation Roadmap](/dev-docs/roadmaps/animation-roadmap.md)


## Abstract

This RFC proposes a system for programmatic *animation* of deck.gl layer properties (it does not covering interpolation, or *transitions*). It is based on exposing the luma.gl v6 capability of setting uniforms to function values, together with ways to trigger animation to happen even when the layers have not been modified.


## Motivation

Adding animation to an visualization, especially an interactive visualization, can take it from having great to extraordinary impact. Well-executed animations add a deep level of polish and interest. However implementing good animations often requires considerable custom work in applications.

The goal of this RFC is making it easier for deck.gl users to achieve great animations (and almost effortless to achieve good basic animations), at least for a certain category of visualization properties.


## Marketing Pitch (deck.gl What's New)

* **Property Animation** - By setting compatible layer properties to functions, it is now possible to have layers animate a number of various attributes. It is also possible to create custom layers that react to time, mouse position etc for advanced rendering effects.
* **Control over rendering FPS** - Since the new animation system activates continuous rendering, deck.gl also provides applications with control over animation FPS to avoid excessive energy consumption, fans running hot etc.

A GIF with a layer driven with property animations should be included.


## Requirements

The following proposals are included in this RFC:

- **Time based animation** - layers respond to time, enabling "movement or pulsation" through calling a sine or similar function.
- **Mouse pointer base animation** - layers respond to the pointer
- Automatic animation as result of user events (i.e. enter/leave type animations)


## Considerations


### Driving the Animation

Animating the props on each frame is one part of the problem. The other part is ensuring that we keep animating when any animation is active.

* layers are only redrawn when the dirty flag of at least one layer is set, or when the viewport changes.
* layer dirty flags are only set when new layers are actually provided by the application (which typically happens when application state changes).


### FPS Control

Since the new animation system activates continuous rendering, deck.gl also provides applications with control over animation FPS to avoid excessive energy consumption, fans running hot etc.


### PropTypes System

Animation can benefit from the prop type system. The following Property Types suitable for animation

* Floats - The easiest to animate, just multiply with fractions
* Integers - needs to be animated in integer steps
* Colors - can be component-wise animated from a start color to and end color


## Proposals

### Proposal: Selected Layer Props can now be set to "Updater Functions"

To make time-based animation work, the idea is that individual props could be set to "updater functions" in addition to constant values. The functions would be called every time the layer is updated (whether through a render or a time-based update).

The updated functions would take a context parameter (see the luma.gl [AnimationLoop](https://visgl.github.io/luma.gl/docs/api-reference/animation-loop) for a good reference on how this would work).

```js
const layer = new Layer {
  radius: ({tick}) => Math.sin(tick * 0.1),
  color: ({tick}) => [128, 128, tick % 255, 255]
}
```


## Proposal: Support time-based animation

Animation that is not driven by property updates, but rather the passing of time.

Layers would have the ability to mark themselves as animated.

```js
new Layer({
  animated: true
});
```

The layers that are marked as animated will be updated every *browser animation frame* (i.e. 60 times per second), even if the application has not rendered (created layers with new props).


## animation parameters

Generated by animation frame. We could add a mechanism to add additional params. Maybe that could be intevgrated directly into animation frame instead of deck?


## Open Questions

* Question: Call `updateState` every frame, or provide a new function life cycle function `animateState` to allow for optimized treatment of the animation case?
* **Model matrix interpolation** - Animating the `modelMatrix` `prop` could be very powerful:
    * For time based animations the updater funciton would return updated matrices
    * For interpolations, is there a reasonable way to automatically interpolate matrices that generally give the right visual results? This could be incredibly neat if it worked, but numeric "instability" might make it impractical.
