# RFC: Jupyter Notebook/Python Integration Strategy for deck.gl and applications

* **Author**: Ib Green, Andrew Duberstein
* **Date**: Mar, 2019
* **Status**: **Draft**

## Summary

This RFC proposes adding a minimal official Python integration to deck.gl, focusing on Jupyter notebook support.


## Background

Improved support for data science and machine learning visualization is a focus area for deck.gl, and deck.gl is already very useful for geospatial visualizations in these contexts. Many data scientists work in Python and being able to effortlessly visualize data generated by libraries like numpy and pandas directly from the notebook is very valuable.

By providing a set of very simple bindings, focusing on overcoming the most basic integration issues and exposing a reasonable subset of the deck.gl API, we should be able to provide significantly increase adoption and value of deck.gl in these areas.

Offering such an integration clearly makes sense, as long as we can do it with low effort and maintenance cost.


## Leveraging the deck.gl JSON API

Normally, exposing an API in a new language requires porting that API to the new language, which can be a significant undertaking, creating a large set of class and function "wrappers" that glue to the original code, with associated maintenance costs.

However, deck.gl now supports a [JSON API](http://deck.gl/#/documentation/developer-guide/json/using-the-json-api) [API Reference](http://deck.gl/#/documentation/json-api-reference-experimental/json-layer-experimental) ([demo](http://deck.gl/json/).

The JSON API makes it possible for users to specify deck.gl layers and access a large subset of deck.gl's functionality without directly using the deck.gl JavaScript API, making it an ideal API entry point to expose in other languages.

The idea is thus to create a minimal binding in Python that allows a python application to generate a JSON payload compatible with the deck.gl API and then start an inline frame running deck.gl rendering that JSON payload.


### Stability of the JSON API

The JSON API is a pass-through of the underlying layer APIs. In fact, the JSON API allows the app to register layers imported from modules. Thus, the exposed API will change if the deck.gl layer APIs change.

This fits with the spirit that the official Python API matches the official deck.gl API.


### Completeness of the JSON API

The JSON API does not support everything. Some issues are already on the "roadmap" for the json module:
* Only a single mapbox base map
* Accessors can only be 'dotty' style strings, general expression evaluation has not been integrated yet.

Others reflect generally useful features that are not yet supported by the deck.gl core:
* No time playback
* No tooltips

One option is that it could make sense to build things like tooltip support and support for time into the core deck.gl library, in which case these would be easy to expose through JSON. This fits with the platform spirit where we solve application problems in a reusable way.



### Focus Areas

To minimize initial effort and long-term maintenance, we want to keep the scope very limited and focus on solving core problems such as:

Primary goals:
* Making the deck.gl Python module easy to install and use for Python users
* Handle proper initial sizing and resizing the JavaScript/HTML window
* Efficiently transfer large data sets between Python and JavaScript

Secondary goals:
* Make it easy to build more ambitions Python class libraries on top of the basic module.


### Constraints

deck.gl **should not** include a major Python port of its API (e.g. a big Python class library that mirrors all deck.gl layers). That would represent too big a distraction to the core deck.gl mission.

We should leave such work to the Python community, and instead encourage, support and promote such efforts as they happen.
Thus, we'd encourage independent deck.gl/Python integration libraries build on our official "minimal" integration and try to support them if they have issues.

External python deck.gl; integration libraries could  either use it as a python dependency, or alternatively, they could just use it as an springboard to get started, perhaps forking it to help them solve some of the basic Python/JavaScript integration issues.


## Integration Approaches

Starting up a JavaScript cell in Notebook can be done in (at least) two different ways:

* (Dynamically) generated JavaScript app
    * Could generate dynamical code
    * loading pre-bundled deck.gl from CDN.

* iframe linking to published app
    * python would generates minimal iframe HTML wrapper
    * I would like to URL published application (e.g. https://deck.gl/json).
    * NOTE: This setup can support non-deck.gl applications


x* We do not currently publish the JSON module as part of the standalone bundle in 6.4. In 7.0 we plan to publish a master bundle (deckgl.min.js) as well as individual bundles for each submodule.
* We could make a decision which bundles to include for the Python API.
* Alternatively, the JSON module allows additional layer modules to be registered. Perhaps the Python API could take a list of layer module URLs, and inject those requires into the HTML and register the imported layers.


## Technical Challenges


* The solution will be loading pre-bundled deck.gl from CDN.

* Controlling size of deck.gl visualization
    * height of frame controllable by python code?
    * frame auto resizes to full width screen?
    * frame height (width?) resizable by user?

* Continuous Updates?
    * Most Python/JS bindings generate HTML dynamically using Python API then create the iframe, after that Python has no more control.
    * However deck.gl JSON api can accept new JSON payloads, they go through standard deck.gl layer diffing.
    * As fallback, Python can emit new invisible HTML cells that "postmessage" to the (iframed) app.
    * Ideally for best perf we'd maintain a network connection from remote notebook to JS app so that Python can push new payloads?

* Two-way Communication?
    * It would be great if deck.gl could send mouse clicks etc back to Python.
    * So far we have not discovered and Ikernel or similar method to achieve this.

* Efficient/Binary Data Transfer
    * Many Python JS integrations serialize data into the HTML/JS payload.
    * Inefficient for large data sets.
    * Want to support binary data transfer: panda dataframe -> pyarrow -> deck
    * Ideally for best perf we'd maintain a network connection from remote notebook to JS app so that Python can push new payloads?

* WebGL Context Reuse
    * When using multiple cells the user can quickly run out of deck.gl contexts, leading to intermittent deck.gl failures
    * It should be possible to add a WebGLContext reuse system to deck.gl similar to react-map-gl
    * Investigate if the notebook provides the right callbacks to enable reuse across cells
    * (It may be hard to reuse across iframed apps...)


## Practical Concerns

* How to publish Python modules (pip, conda)?
* When to update? (every JS release)
* Versioning schemes (do Python package managers use semver)?
* Where to put Python code? (in same github repo, `bindings/python`?)
* Maintenance: deck.gl programmers don't regularly use Python
* Testing/Deployment
* ...


## Prior Art

### kepler.gl Jupyter Integration

* [RFC](https://github.com/keplergl/kepler.gl/issues/331)


### Mapbox Jupyter Integration

* [Blog Post](https://github.com/mapbox/mapboxgl-jupyter)
* [Github](https://github.com/mapbox/mapboxgl-jupyter/blob/master/docs/viz.md)


### Plotly Jupyter Integration

TBA


### D3 Integration

https://www.stefaanlippens.net/jupyter-custom-d3-visualization.html



## Future Extensions

Minimal integrations for other languages? Especially languages that run in Notebooks?

### R

See the [mapdeck](https://symbolixau.github.io/mapdeck/articles/mapdeck.html) integration for R.
