# TransitionInterpolator

Base interpolator class that provides common functionality required to interpolate between two [View States](/docs/developer-guide/views.md). This class is not intended to be used directly. See [View State Transitions](/docs/developer-guide/view-state-transitions.md) for backgrounds.

## Example

```js
import {TransitionInterpolator} from '@deck.gl/core';

// Interpolate between two values
function lerp(from, to, t) {
  return from + (to - from) * t;
}

/*
 * This interpolator moves the camera along a straight path on the Web Mercator map
 * Horizontally it moves in the direction that is shorter
 */
class SphericalLinearInterpolator extends TransitionInterpolator {
  constructor({speed = 100} = {}) {
    super(['longitude', 'latitude']);

    // degrees per second
    this.speed = speed;
  }

  getDuration(startViewState, endViewState) {
    const deltaLat = Math.abs(startViewState.latitude - endViewState.latitude);
    let deltaLng = Math.abs(startViewState.longitude - endViewState.longitude);
    // Transition to the destination longitude along the smaller half of the circle
    if (deltaLng > 180) deltaLng = 360 - deltaLng;
    return Math.max(deltaLng, deltaLat) / this.speed * 1000;
  }

  initializeProps(startViewState, endViewState) {
    const fromLng = startViewState.longitude;
    let toLng = endViewState.longitude;
    // Transition to the destination longitude along the smaller half of the latitude circle
    if (toLng > fromLng + 180) toLng -= 360;
    if (toLng < fromLng - 180) toLng += 360;

    return {
      start: {longitude: fromLng, latitude: startViewState.latitude},
      end: {longitude: toLng, latitude: endViewState.latitude}
    };
  }

  interpolateProps(start, end, t) {
    const latitude = lerp(start.latitude, end.latitude, t);
    let longitude = lerp(start.longitude, end.longitude, t);
    // Put longitude back into the [-180, 180] range
    if (longitude > 180) longitude -= 360;
    if (longitude < -180) longitude += 360;

    return {longitude, latitude};
  }
}
```

## Constructor

Parameters:

- `opts` (Object | Array)
  + If an object is provided, it should contain the following fields:
    * `compare` (Array) - prop names used in equality check. Transition is triggered if some of the `compare` props are not deeply equal.
    * `extract` (Array) - prop names needed for interpolation. See `initializeProps` below.
    * `required` (Array) - prop names that must be supplied. See `initializeProps` below.
  + If an array is provided, the same list of prop names is used for all three above.


## Interface

##### `getDuration`

This method can be optionally implemented by a subclass. If implemented, the user can specify `transitionDuration: 'auto'` when using this interpolator, and the duration will be dynamically computed using this method.

Receives the following arguments:

- `startViewState` (Object) - the view state that is transitioning from.
- `endViewState` (Object) - the view state that is transitioning to.

Returns:

- A number in milliseconds representing the target duration of the transition. If `0`, transition is disabled.

##### `initializeProps`

Called when a transition is about to be triggered. This can be used to preprocess values for use in every transition frame by `interpolateProps`.

Receives the following arguments:

- `startViewState` (Object) - the view state that is transitioning from.
- `endViewState` (Object) - the view state that is transitioning to.

Returns:

- An object in the shape of `{start, end}` that will be passed to `interpolateProps`.

The default implementation takes all values of the `extract` prop names from the start and end view states.

If some `required` prop names are missing from the view state, an error will be thrown.


##### `interpolateProps`

This method must be implemented by a subclasses.

Receives the following arguments:

- `start` (Object) - descriptor of the state that is transitioning from, generated by `initializeProps`.
- `end` (Object) - descriptor of the state that is transitioning to, generated by `initializeProps`.
- `t` (Number) - current time into the transition, between `0` and `1`.

Returns:

- An object that contains the interpolated view state fields.

## Source

[modules/core/src/transitions/transition-interpolator.js](https://github.com/visgl/deck.gl/tree/8.4-release/modules/core/src/transitions/transition-interpolator.js)
