package io.quarkus.smallrye.graphql.client.deployment;

import static io.smallrye.graphql.client.core.Argument.arg;
import static io.smallrye.graphql.client.core.Document.document;
import static io.smallrye.graphql.client.core.Field.field;
import static org.junit.jupiter.api.Assertions.*;

import java.time.Duration;
import java.util.List;
import java.util.concurrent.ExecutionException;

import jakarta.inject.Inject;

import org.jboss.shrinkwrap.api.asset.EmptyAsset;
import org.jboss.shrinkwrap.api.asset.StringAsset;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.RegisterExtension;

import io.quarkus.smallrye.graphql.client.deployment.model.Person;
import io.quarkus.smallrye.graphql.client.deployment.model.PersonDto;
import io.quarkus.smallrye.graphql.client.deployment.model.TestingGraphQLApi;
import io.quarkus.test.QuarkusUnitTest;
import io.smallrye.graphql.client.GraphQLClient;
import io.smallrye.graphql.client.core.Document;
import io.smallrye.graphql.client.core.Operation;
import io.smallrye.graphql.client.dynamic.api.DynamicGraphQLClient;

public class DynamicGraphQLClientInjectionTest {

    static String url = "http://" + System.getProperty("quarkus.http.host", "localhost") + ":" +
            System.getProperty("quarkus.http.test-port", "8081") + "/graphql";

    @RegisterExtension
    static QuarkusUnitTest test = new QuarkusUnitTest()
            .withApplicationRoot((jar) -> jar
                    .addClasses(TestingGraphQLApi.class, Person.class, PersonDto.class)
                    .addAsResource(new StringAsset("people-client/mp-graphql/url=" + url + "\n" +
                            "people-client/mp-graphql/header/My-Header=My-Value"),
                            "application.properties")
                    .addAsManifestResource(EmptyAsset.INSTANCE, "beans.xml"));

    @Inject
    @GraphQLClient("people-client")
    DynamicGraphQLClient client;

    @Test
    public void checkInjectedClient() {
        Document query = document(
                Operation.operation("PeopleQuery", field("people", field("firstName"), field("lastName"))));
        List<Person> people = client.executeAsync(query)
                .await().atMost(Duration.ofSeconds(30)).getList(Person.class, "people");
        assertEquals("John", people.get(0).getFirstName());
        assertEquals("Arthur", people.get(1).getFirstName());
    }

    @Test
    public void checkHeaders() throws ExecutionException, InterruptedException {
        Document query = document(
                Operation.operation(field("returnHeader", arg("key", "My-Header"))));
        String header = client.executeSync(query).getData().getString("returnHeader");
        assertEquals("My-Value", header);
    }

}
