// Copyright (c) Microsoft. All rights reserved.
// Licensed under the MIT license. See LICENSE file in the project root for full license information.

// =+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+=+
//
// ParallelEnumerableWrapper.cs
//
// =-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-

using System.Collections;
using System.Collections.Generic;
using System.Diagnostics.Contracts;

namespace System.Linq.Parallel
{
    /// <summary>
    /// A simple implementation of the ParallelQuery{object} interface which wraps an
    /// underlying IEnumerable, such that it can be used in parallel queries.
    /// </summary>
    internal class ParallelEnumerableWrapper : ParallelQuery<object>
    {
        private readonly IEnumerable _source; // The wrapped enumerable object.

        //-----------------------------------------------------------------------------------
        // Instantiates a new wrapper object.
        //

        internal ParallelEnumerableWrapper(Collections.IEnumerable source)
            : base(QuerySettings.Empty)
        {
            Contract.Assert(source != null);
            _source = source;
        }

        internal override IEnumerator GetEnumeratorUntyped()
        {
            return _source.GetEnumerator();
        }

        public override IEnumerator<object> GetEnumerator()
        {
            return new EnumerableWrapperWeakToStrong(_source).GetEnumerator();
        }
    }

    /// <summary>
    /// A simple implementation of the ParallelQuery{T} interface which wraps an
    /// underlying IEnumerable{T}, such that it can be used in parallel queries.
    /// </summary>
    /// <typeparam name="T"></typeparam>
    internal class ParallelEnumerableWrapper<T> : ParallelQuery<T>
    {
        private readonly IEnumerable<T> _wrappedEnumerable; // The wrapped enumerable object.

        //-----------------------------------------------------------------------------------
        // Instantiates a new wrapper object.
        //
        // Arguments:
        //     wrappedEnumerable   - the underlying enumerable object being wrapped
        //
        // Notes:
        //     The analysisOptions and degreeOfParallelism settings are optional.  Passing null
        //     indicates that the system defaults should be used instead.
        //

        internal ParallelEnumerableWrapper(IEnumerable<T> wrappedEnumerable)
            : base(QuerySettings.Empty)
        {
            Contract.Assert(wrappedEnumerable != null, "wrappedEnumerable must not be null.");

            _wrappedEnumerable = wrappedEnumerable;
        }

        //-----------------------------------------------------------------------------------
        // Retrieves the wrapped enumerable object.
        //

        internal IEnumerable<T> WrappedEnumerable
        {
            get { return _wrappedEnumerable; }
        }

        //-----------------------------------------------------------------------------------
        // Implementations of GetEnumerator that just delegate to the wrapped enumerable.
        //

        public override IEnumerator<T> GetEnumerator()
        {
            Contract.Assert(_wrappedEnumerable != null);
            return _wrappedEnumerable.GetEnumerator();
        }
    }
}