﻿// Copyright (c) Microsoft.  All Rights Reserved.  Licensed under the Apache License, Version 2.0.  See License.txt in the project root for license information.

using System;
using System.Collections.Generic;
using System.Collections.Immutable;
using System.Diagnostics;
using System.Text;

namespace Microsoft.CodeAnalysis.Text
{
    internal sealed class ChangedText : SourceText
    {
        private readonly SourceText _oldText;
        private readonly SourceText _newText;
        private readonly ImmutableArray<TextChangeRange> _changes;

        public ChangedText(SourceText oldText, ImmutableArray<TextChangeRange> changeRanges, ImmutableArray<SourceText> segments)
            : base(checksumAlgorithm: oldText.ChecksumAlgorithm)
        {
            Debug.Assert(oldText != null);
            Debug.Assert(!changeRanges.IsDefault);
            Debug.Assert(!segments.IsDefault);

            _oldText = oldText;
            _newText = segments.IsEmpty ? new StringText("", oldText.Encoding, checksumAlgorithm: oldText.ChecksumAlgorithm) : (SourceText)new CompositeText(segments);
            _changes = changeRanges;
        }

        public override Encoding Encoding
        {
            get { return _oldText.Encoding; }
        }

        public SourceText OldText
        {
            get { return _oldText; }
        }

        public SourceText NewText
        {
            get { return _newText; }
        }

        public IEnumerable<TextChangeRange> Changes
        {
            get { return _changes; }
        }

        public override int Length
        {
            get { return _newText.Length; }
        }

        public override char this[int position]
        {
            get { return _newText[position]; }
        }

        public override string ToString(TextSpan span)
        {
            return _newText.ToString(span);
        }

        public override SourceText GetSubText(TextSpan span)
        {
            return _newText.GetSubText(span);
        }

        public override void CopyTo(int sourceIndex, char[] destination, int destinationIndex, int count)
        {
            _newText.CopyTo(sourceIndex, destination, destinationIndex, count);
        }

        public override IReadOnlyList<TextChangeRange> GetChangeRanges(SourceText oldText)
        {
            if (oldText == null)
            {
                throw new ArgumentNullException("oldText");
            }

            if (ReferenceEquals(_oldText, oldText))
            {
                // check whether the bases are same one
                return _changes;
            }

            if (_oldText.GetChangeRanges(oldText).Count == 0)
            {
                // okay, the bases are different, but the contents might be same.
                return _changes;
            }

            if (this == oldText)
            {
                return TextChangeRange.NoChanges;
            }

            return ImmutableArray.Create(new TextChangeRange(new TextSpan(0, oldText.Length), _newText.Length));
        }

        protected override TextLineCollection GetLinesCore()
        {
            var oldLineInfo = _oldText.Lines;
            var lineStarts = ArrayBuilder<int>.GetInstance();

            lineStarts.Add(0);

            // position in the original document
            var position = 0;
            // delta generated by already processed changes (position in the new document = position + delta)
            var delta = 0;
            // true if last segment ends with CR and we need to check for CR+LF code below assumes that both CR and LF are also line breaks alone
            var endsWithCR = false;
            foreach (var change in _changes)
            {
                // change.Span.Start < position already ruled out by SourceText.WithChanges
                // if we've skipped a range, add
                if (change.Span.Start > position)
                {
                    if (endsWithCR && _newText[position + delta] == '\n')
                    {
                        lineStarts.RemoveLast();
                    }
                    var lps = oldLineInfo.GetLinePositionSpan(TextSpan.FromBounds(position, change.Span.Start));
                    for (int i = lps.Start.Line + 1; i <= lps.End.Line; i++)
                    {
                        lineStarts.Add(oldLineInfo[i].Start + delta);
                    }
                    endsWithCR = _oldText[change.Span.Start - 1] == '\r';
                    // in case change is inserted between CR+LF we treat CR as line break alone, but this line break might be retracted and replaced with new one in case LF is inserted
                    if (endsWithCR && change.Span.Start < _oldText.Length && _oldText[change.Span.Start] == '\n')
                    {
                        lineStarts.Add(change.Span.Start + delta);
                    }
                }

                if (change.NewLength > 0)
                {
                    var text = GetSubText(new TextSpan(change.Span.Start + delta, change.NewLength));

                    // optimizations copied from SourceText.LineInfo.ParseLineStarts
                    var index = 0;
                    while (index < text.Length)
                    {
                        char c = text[index++];

                        // Common case - ASCII & not a line break
                        // if (c > '\r' && c <= 127)
                        // if (c >= ('\r'+1) && c <= 127)
                        const uint bias = '\r' + 1;
                        if (unchecked(c - bias) <= (127 - bias))
                        {
                            continue;
                        }

                        if (endsWithCR && c == '\n')
                        {
                            lineStarts.RemoveLast();
                        }
                        else if (c == '\r' && index < text.Length && text[index] == '\n')
                        {
                            index++;
                        }
                        else if (!TextUtilities.IsAnyLineBreakCharacter(c))
                        {
                            continue;
                        }

                        lineStarts.Add(change.Span.Start + delta + index);
                    }
                    endsWithCR = text[change.NewLength - 1] == '\r';
                }

                position = change.Span.End;
                delta += (change.NewLength - change.Span.Length);
            }

            if (position < _oldText.Length)
            {
                if (endsWithCR && _newText[position + delta] == '\n')
                {
                    lineStarts.RemoveLast();
                }
                var lps = oldLineInfo.GetLinePositionSpan(TextSpan.FromBounds(position, _oldText.Length));
                for (int i = lps.Start.Line + 1; i <= lps.End.Line; i++)
                {
                    lineStarts.Add(oldLineInfo[i].Start + delta);
                }
            }

            return new LineInfo(this, lineStarts.ToArrayAndFree());
        }
    }
}
