//
// SignerInfoTest.cs - NUnit tests for SignerInfo
//
// Author:
//	Sebastien Pouliot  <sebastien@ximian.com>
//
// (C) 2003 Motus Technologies Inc. (http://www.motus.com)
// Copyright (C) 2004-2005 Novell, Inc (http://www.novell.com)
//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
// 
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//
#if !MOBILE

using NUnit.Framework;

using System;
using System.Collections;
using System.Security.Cryptography;
using System.Security.Cryptography.Pkcs;
using System.Security.Cryptography.X509Certificates;
using System.Security.Cryptography.Xml;

namespace MonoTests.System.Security.Cryptography.Pkcs {

	[TestFixture]
	public class SignerInfoTest {

		static byte[] asnNull = { 0x05, 0x00 };
		static string sha1Oid = "1.3.14.3.2.26";
		static string sha1Name = "sha1";

		static internal byte[] issuerAndSerialNumberSignature = { 0x30, 0x82, 0x03, 0x4C, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x02, 0xA0, 0x82, 0x03, 0x3D, 0x30, 0x82, 0x03, 0x39, 0x02, 0x01, 0x01, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x05, 0x2B, 0x0E, 0x03, 0x02, 0x1A, 0x05, 0x00, 0x30, 0x11, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x01, 0xA0, 0x04, 0x04, 0x02, 0x05, 0x00, 0xA0, 0x82, 0x02, 0x2E, 0x30, 0x82, 0x02, 0x2A, 0x30, 0x82, 0x01, 0x97, 0xA0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x10, 0x91, 0xC4, 0x4B, 0x0D, 0xB7, 0xD8, 0x10, 0x84, 0x42, 0x26, 0x71, 0xB3, 0x97, 0xB5, 0x00, 0x97, 0x30, 0x09, 0x06, 0x05, 0x2B, 0x0E, 0x03, 0x02, 0x1D, 0x05, 0x00, 0x30, 0x28, 0x31, 0x26, 0x30, 0x24, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x1D, 0x4D, 0x6F, 0x74, 0x75, 0x73, 0x20, 0x54, 0x65,
			0x63, 0x68, 0x6E, 0x6F, 0x6C, 0x6F, 0x67, 0x69, 0x65, 0x73, 0x20, 0x69, 0x6E, 0x63, 0x2E, 0x28, 0x74, 0x65, 0x73, 0x74, 0x29, 0x30, 0x1E, 0x17, 0x0D, 0x30, 0x33, 0x30, 0x38, 0x31, 0x33, 0x30, 0x30, 0x34, 0x33, 0x34, 0x37, 0x5A, 0x17, 0x0D, 0x33, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5A, 0x30, 0x13, 0x31, 0x11, 0x30, 0x0F, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x08, 0x46, 0x41, 0x52, 0x53, 0x43, 0x41, 0x50, 0x45, 0x30, 0x81, 0x9F, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x81, 0x8D, 0x00, 0x30, 0x81, 0x89, 0x02, 0x81, 0x81, 0x00, 0xD2, 0xCB, 0x47, 0x21, 0xF5, 0x62, 0xDD, 0x35, 0xBF, 0x1D, 0xEC, 0x9A, 0x4C, 0x07, 0x2C, 0x01, 0xF0, 0x28, 0xC2, 0x82, 0x17, 0x8E, 0x58, 0x32, 
			0xD5, 0x4C, 0xAC, 0x86, 0xB4, 0xC9, 0xEB, 0x21, 0x26, 0xF3, 0x22, 0x30, 0xC5, 0x7A, 0xA3, 0x5A, 0xDD, 0x53, 0xAB, 0x1C, 0x06, 0x3E, 0xB2, 0x13, 0xC4, 0x05, 0x1D, 0x95, 0x8B, 0x0A, 0x71, 0x71, 0x11, 0xA7, 0x47, 0x26, 0x61, 0xF1, 0x76, 0xBE, 0x35, 0x72, 0x32, 0xC5, 0xCB, 0x47, 0xA4, 0x22, 0x41, 0x1E, 0xAD, 0x29, 0x11, 0x0D, 0x39, 0x22, 0x0C, 0x79, 0x90, 0xC6, 0x52, 0xA1, 0x10, 0xF6, 0x55, 0x09, 0x4E, 0x51, 0x26, 0x47, 0x0E, 0x94, 0xE6, 0x81, 0xF5, 0x18, 0x6B, 0x99, 0xF0, 0x76, 0xF3, 0xB2, 0x4C, 0x91, 0xE9, 0xBA, 0x3B, 0x3F, 0x6E, 0x63, 0xDA, 0x12, 0xD1, 0x0B, 0x73, 0x0E, 0x12, 0xC7, 0x70, 0x77, 0x22, 0x03, 0x9D, 0x5D, 0x02, 0x03, 0x01, 0x00, 0x01, 0xA3, 0x72, 0x30, 0x70, 0x30, 0x13, 0x06, 0x03, 0x55, 0x1D, 0x25, 0x04, 0x0C, 0x30, 0x0A, 0x06, 0x08, 0x2B, 
			0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x01, 0x30, 0x59, 0x06, 0x03, 0x55, 0x1D, 0x01, 0x04, 0x52, 0x30, 0x50, 0x80, 0x10, 0xAE, 0xD7, 0x80, 0x88, 0xA6, 0x3D, 0xBA, 0x50, 0xA1, 0x7E, 0x57, 0xE5, 0x40, 0xC9, 0x6F, 0xC5, 0xA1, 0x2A, 0x30, 0x28, 0x31, 0x26, 0x30, 0x24, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x1D, 0x4D, 0x6F, 0x74, 0x75, 0x73, 0x20, 0x54, 0x65, 0x63, 0x68, 0x6E, 0x6F, 0x6C, 0x6F, 0x67, 0x69, 0x65, 0x73, 0x20, 0x69, 0x6E, 0x63, 0x2E, 0x28, 0x74, 0x65, 0x73, 0x74, 0x29, 0x82, 0x10, 0x9D, 0xAE, 0xA3, 0x39, 0x47, 0x0E, 0xD4, 0xA2, 0x49, 0x78, 0xEA, 0x6C, 0xBA, 0x0D, 0xDE, 0x9C, 0x30, 0x09, 0x06, 0x05, 0x2B, 0x0E, 0x03, 0x02, 0x1D, 0x05, 0x00, 0x03, 0x81, 0x81, 0x00, 0x32, 0x8A, 0x7E, 0xAD, 0xE7, 0x67, 0x9E, 0x5C, 0x4C, 0xD8, 0x33, 0x59, 0x68, 0xCF, 
			0x94, 0xC0, 0x36, 0x47, 0x7A, 0xA7, 0x85, 0xC2, 0xDD, 0xD8, 0xDA, 0x11, 0x3C, 0x66, 0xC1, 0x83, 0xE3, 0xAB, 0x33, 0x06, 0x7C, 0xE3, 0x6A, 0x15, 0x72, 0xB8, 0x83, 0x3D, 0x0B, 0xAB, 0x3C, 0xEE, 0x75, 0x13, 0xBD, 0x5C, 0x96, 0x25, 0x56, 0x36, 0x05, 0xFA, 0xAE, 0xD4, 0xF4, 0xCF, 0x52, 0xEC, 0x11, 0xB5, 0xEA, 0x9F, 0x20, 0xA3, 0xC8, 0x34, 0x72, 0x59, 0x09, 0x51, 0xE7, 0x36, 0x87, 0x86, 0x86, 0x98, 0xB5, 0x30, 0x7B, 0xFB, 0x3D, 0xCC, 0x5E, 0xE8, 0xC9, 0x49, 0xE0, 0xC6, 0xEA, 0x02, 0x76, 0x01, 0xE0, 0xBB, 0x8A, 0x70, 0xEB, 0x07, 0x86, 0xE8, 0x04, 0xE7, 0x48, 0xE4, 0x6C, 0x90, 0xE6, 0x16, 0x42, 0xB4, 0xBB, 0xC0, 0xC4, 0x82, 0x5F, 0xF8, 0xFB, 0x7E, 0xB2, 0x9E, 0xC2, 0x78, 0x26, 0x86, 0x31, 0x81, 0xE1, 0x30, 0x81, 0xDE, 0x02, 0x01, 0x01, 0x30, 0x3C, 0x30, 0x28, 
			0x31, 0x26, 0x30, 0x24, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x1D, 0x4D, 0x6F, 0x74, 0x75, 0x73, 0x20, 0x54, 0x65, 0x63, 0x68, 0x6E, 0x6F, 0x6C, 0x6F, 0x67, 0x69, 0x65, 0x73, 0x20, 0x69, 0x6E, 0x63, 0x2E, 0x28, 0x74, 0x65, 0x73, 0x74, 0x29, 0x02, 0x10, 0x91, 0xC4, 0x4B, 0x0D, 0xB7, 0xD8, 0x10, 0x84, 0x42, 0x26, 0x71, 0xB3, 0x97, 0xB5, 0x00, 0x97, 0x30, 0x09, 0x06, 0x05, 0x2B, 0x0E, 0x03, 0x02, 0x1A, 0x05, 0x00, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x01, 0x05, 0x00, 0x04, 0x81, 0x80, 0x45, 0x88, 0x80, 0x58, 0xC7, 0x4F, 0xE4, 0xD8, 0x88, 0xB0, 0xC0, 0x08, 0x70, 0x84, 0xCC, 0x8E, 0xA7, 0xF1, 0xA4, 0x07, 0x41, 0x14, 0x3E, 0xF5, 0xEA, 0x6E, 0x05, 0x75, 0xB8, 0x58, 0xAA, 0x5C, 0x0E, 0xFD, 0x7A, 0x07, 0x09, 0xE1, 0x80, 0x94, 
			0xBD, 0xAA, 0x45, 0xBB, 0x55, 0x9C, 0xC2, 0xD9, 0x72, 0x14, 0x4B, 0xA4, 0x64, 0xFB, 0x38, 0x9F, 0xD3, 0x22, 0xED, 0xB3, 0x0B, 0xF7, 0xAE, 0x4D, 0xE6, 0x65, 0x4D, 0x2A, 0x31, 0x18, 0xB5, 0xB4, 0x2D, 0x9E, 0x4E, 0xD7, 0xC0, 0x44, 0x5F, 0xAC, 0x43, 0xDC, 0x4F, 0x3D, 0x6D, 0x2C, 0x8C, 0xA1, 0xFE, 0x08, 0x38, 0xB7, 0xC4, 0xC4, 0x08, 0xDB, 0xF8, 0xF0, 0xC1, 0x55, 0x54, 0x49, 0x9D, 0xA4, 0x7F, 0x76, 0xDE, 0xF4, 0x29, 0x1C, 0x0B, 0x95, 0x10, 0x90, 0xB5, 0x0A, 0x9A, 0xEC, 0xCA, 0x89, 0x9A, 0x85, 0x92, 0x76, 0x78, 0x6F, 0x97, 0x67 };

		static internal byte[] subjectKeyIdentifierSignature = { 0x30, 0x82, 0x03, 0x24, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x02, 0xA0, 0x82, 0x03, 0x15, 0x30, 0x82, 0x03, 0x11, 0x02, 0x01, 0x03, 0x31, 0x0B, 0x30, 0x09, 0x06, 0x05, 0x2B, 0x0E, 0x03, 0x02, 0x1A, 0x05, 0x00, 0x30, 0x11, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x01, 0xA0, 0x04, 0x04, 0x02, 0x05, 0x00, 0xA0, 0x82, 0x02, 0x2E, 0x30, 0x82, 0x02, 0x2A, 0x30, 0x82, 0x01, 0x97, 0xA0, 0x03, 0x02, 0x01, 0x02, 0x02, 0x10, 0x91, 0xC4, 0x4B, 0x0D, 0xB7, 0xD8, 0x10, 0x84, 0x42, 0x26, 0x71, 0xB3, 0x97, 0xB5, 0x00, 0x97, 0x30, 0x09, 0x06, 0x05, 0x2B, 0x0E, 0x03, 0x02, 0x1D, 0x05, 0x00, 0x30, 0x28, 0x31, 0x26, 0x30, 0x24, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x1D, 0x4D, 0x6F, 0x74, 0x75, 0x73, 0x20, 0x54, 0x65, 
			0x63, 0x68, 0x6E, 0x6F, 0x6C, 0x6F, 0x67, 0x69, 0x65, 0x73, 0x20, 0x69, 0x6E, 0x63, 0x2E, 0x28, 0x74, 0x65, 0x73, 0x74, 0x29, 0x30, 0x1E, 0x17, 0x0D, 0x30, 0x33, 0x30, 0x38, 0x31, 0x33, 0x30, 0x30, 0x34, 0x33, 0x34, 0x37, 0x5A, 0x17, 0x0D, 0x33, 0x39, 0x31, 0x32, 0x33, 0x31, 0x32, 0x33, 0x35, 0x39, 0x35, 0x39, 0x5A, 0x30, 0x13, 0x31, 0x11, 0x30, 0x0F, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x08, 0x46, 0x41, 0x52, 0x53, 0x43, 0x41, 0x50, 0x45, 0x30, 0x81, 0x9F, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x01, 0x05, 0x00, 0x03, 0x81, 0x8D, 0x00, 0x30, 0x81, 0x89, 0x02, 0x81, 0x81, 0x00, 0xD2, 0xCB, 0x47, 0x21, 0xF5, 0x62, 0xDD, 0x35, 0xBF, 0x1D, 0xEC, 0x9A, 0x4C, 0x07, 0x2C, 0x01, 0xF0, 0x28, 0xC2, 0x82, 0x17, 0x8E, 0x58, 0x32, 
			0xD5, 0x4C, 0xAC, 0x86, 0xB4, 0xC9, 0xEB, 0x21, 0x26, 0xF3, 0x22, 0x30, 0xC5, 0x7A, 0xA3, 0x5A, 0xDD, 0x53, 0xAB, 0x1C, 0x06, 0x3E, 0xB2, 0x13, 0xC4, 0x05, 0x1D, 0x95, 0x8B, 0x0A, 0x71, 0x71, 0x11, 0xA7, 0x47, 0x26, 0x61, 0xF1, 0x76, 0xBE, 0x35, 0x72, 0x32, 0xC5, 0xCB, 0x47, 0xA4, 0x22, 0x41, 0x1E, 0xAD, 0x29, 0x11, 0x0D, 0x39, 0x22, 0x0C, 0x79, 0x90, 0xC6, 0x52, 0xA1, 0x10, 0xF6, 0x55, 0x09, 0x4E, 0x51, 0x26, 0x47, 0x0E, 0x94, 0xE6, 0x81, 0xF5, 0x18, 0x6B, 0x99, 0xF0, 0x76, 0xF3, 0xB2, 0x4C, 0x91, 0xE9, 0xBA, 0x3B, 0x3F, 0x6E, 0x63, 0xDA, 0x12, 0xD1, 0x0B, 0x73, 0x0E, 0x12, 0xC7, 0x70, 0x77, 0x22, 0x03, 0x9D, 0x5D, 0x02, 0x03, 0x01, 0x00, 0x01, 0xA3, 0x72, 0x30, 0x70, 0x30, 0x13, 0x06, 0x03, 0x55, 0x1D, 0x25, 0x04, 0x0C, 0x30, 0x0A, 0x06, 0x08, 0x2B, 
			0x06, 0x01, 0x05, 0x05, 0x07, 0x03, 0x01, 0x30, 0x59, 0x06, 0x03, 0x55, 0x1D, 0x01, 0x04, 0x52, 0x30, 0x50, 0x80, 0x10, 0xAE, 0xD7, 0x80, 0x88, 0xA6, 0x3D, 0xBA, 0x50, 0xA1, 0x7E, 0x57, 0xE5, 0x40, 0xC9, 0x6F, 0xC5, 0xA1, 0x2A, 0x30, 0x28, 0x31, 0x26, 0x30, 0x24, 0x06, 0x03, 0x55, 0x04, 0x03, 0x13, 0x1D, 0x4D, 0x6F, 0x74, 0x75, 0x73, 0x20, 0x54, 0x65, 0x63, 0x68, 0x6E, 0x6F, 0x6C, 0x6F, 0x67, 0x69, 0x65, 0x73, 0x20, 0x69, 0x6E, 0x63, 0x2E, 0x28, 0x74, 0x65, 0x73, 0x74, 0x29, 0x82, 0x10, 0x9D, 0xAE, 0xA3, 0x39, 0x47, 0x0E, 0xD4, 0xA2, 0x49, 0x78, 0xEA, 0x6C, 0xBA, 0x0D, 0xDE, 0x9C, 0x30, 0x09, 0x06, 0x05, 0x2B, 0x0E, 0x03, 0x02, 0x1D, 0x05, 0x00, 0x03, 0x81, 0x81, 0x00, 0x32, 0x8A, 0x7E, 0xAD, 0xE7, 0x67, 0x9E, 0x5C, 0x4C, 0xD8, 0x33, 0x59, 0x68, 0xCF, 
			0x94, 0xC0, 0x36, 0x47, 0x7A, 0xA7, 0x85, 0xC2, 0xDD, 0xD8, 0xDA, 0x11, 0x3C, 0x66, 0xC1, 0x83, 0xE3, 0xAB, 0x33, 0x06, 0x7C, 0xE3, 0x6A, 0x15, 0x72, 0xB8, 0x83, 0x3D, 0x0B, 0xAB, 0x3C, 0xEE, 0x75, 0x13, 0xBD, 0x5C, 0x96, 0x25, 0x56, 0x36, 0x05, 0xFA, 0xAE, 0xD4, 0xF4, 0xCF, 0x52, 0xEC, 0x11, 0xB5, 0xEA, 0x9F, 0x20, 0xA3, 0xC8, 0x34, 0x72, 0x59, 0x09, 0x51, 0xE7, 0x36, 0x87, 0x86, 0x86, 0x98, 0xB5, 0x30, 0x7B, 0xFB, 0x3D, 0xCC, 0x5E, 0xE8, 0xC9, 0x49, 0xE0, 0xC6, 0xEA, 0x02, 0x76, 0x01, 0xE0, 0xBB, 0x8A, 0x70, 0xEB, 0x07, 0x86, 0xE8, 0x04, 0xE7, 0x48, 0xE4, 0x6C, 0x90, 0xE6, 0x16, 0x42, 0xB4, 0xBB, 0xC0, 0xC4, 0x82, 0x5F, 0xF8, 0xFB, 0x7E, 0xB2, 0x9E, 0xC2, 0x78, 0x26, 0x86, 0x31, 0x81, 0xB9, 0x30, 0x81, 0xB6, 0x02, 0x01, 0x03, 0x80, 0x14, 0x02, 0xE1, 
			0xA7, 0x32, 0x54, 0xAE, 0xFD, 0xC0, 0xA4, 0x32, 0x36, 0xF6, 0xFE, 0x23, 0x6A, 0x03, 0x72, 0x28, 0xB1, 0xF7, 0x30, 0x09, 0x06, 0x05, 0x2B, 0x0E, 0x03, 0x02, 0x1A, 0x05, 0x00, 0x30, 0x0D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x01, 0x01, 0x05, 0x00, 0x04, 0x81, 0x80, 0x45, 0x88, 0x80, 0x58, 0xC7, 0x4F, 0xE4, 0xD8, 0x88, 0xB0, 0xC0, 0x08, 0x70, 0x84, 0xCC, 0x8E, 0xA7, 0xF1, 0xA4, 0x07, 0x41, 0x14, 0x3E, 0xF5, 0xEA, 0x6E, 0x05, 0x75, 0xB8, 0x58, 0xAA, 0x5C, 0x0E, 0xFD, 0x7A, 0x07, 0x09, 0xE1, 0x80, 0x94, 0xBD, 0xAA, 0x45, 0xBB, 0x55, 0x9C, 0xC2, 0xD9, 0x72, 0x14, 0x4B, 0xA4, 0x64, 0xFB, 0x38, 0x9F, 0xD3, 0x22, 0xED, 0xB3, 0x0B, 0xF7, 0xAE, 0x4D, 0xE6, 0x65, 0x4D, 0x2A, 0x31, 0x18, 0xB5, 0xB4, 0x2D, 0x9E, 0x4E, 0xD7, 0xC0, 0x44, 0x5F, 0xAC, 
			0x43, 0xDC, 0x4F, 0x3D, 0x6D, 0x2C, 0x8C, 0xA1, 0xFE, 0x08, 0x38, 0xB7, 0xC4, 0xC4, 0x08, 0xDB, 0xF8, 0xF0, 0xC1, 0x55, 0x54, 0x49, 0x9D, 0xA4, 0x7F, 0x76, 0xDE, 0xF4, 0x29, 0x1C, 0x0B, 0x95, 0x10, 0x90, 0xB5, 0x0A, 0x9A, 0xEC, 0xCA, 0x89, 0x9A, 0x85, 0x92, 0x76, 0x78, 0x6F, 0x97, 0x67 };

		static internal SignerInfo GetSignerInfo (byte[] signature) 
		{
			SignedCms sp = new SignedCms ();
			sp.Decode (signature);
			return sp.SignerInfos [0];
		}

		[Test]
		public void IssuerAndSerialNumberProperties () 
		{
			SignerInfo si = GetSignerInfo (issuerAndSerialNumberSignature);
			// default properties
			Assert.AreEqual (0, si.SignedAttributes.Count, "SignedAttributes");
			Assert.IsNotNull (si.Certificate, "Certificate");
			Assert.AreEqual (0, si.CounterSignerInfos.Count, "CounterSignerInfos");
			Assert.AreEqual (sha1Name, si.DigestAlgorithm.FriendlyName, "DigestAlgorithm.FriendlyName");
			Assert.AreEqual (sha1Oid, si.DigestAlgorithm.Value, "DigestAlgorithm.Value");
			Assert.AreEqual (SubjectIdentifierType.IssuerAndSerialNumber, si.SignerIdentifier.Type, "SignerIdentifier.Type");
			Assert.IsTrue ((si.SignerIdentifier.Value.GetType () == typeof (X509IssuerSerial)), "SignerIdentifier.Value");
			Assert.AreEqual (0, si.UnsignedAttributes.Count, "UnsignedAttributes");
			Assert.AreEqual (1, si.Version, "Version");
		}

		[Test]
		public void SubjectKeyIdentifierProperties () 
		{
			SignerInfo si = GetSignerInfo (subjectKeyIdentifierSignature);
			// default properties
			Assert.AreEqual (0, si.SignedAttributes.Count, "SignedAttributes");
			Assert.IsNotNull (si.Certificate, "Certificate");
			Assert.AreEqual (0, si.CounterSignerInfos.Count, "CounterSignerInfos");
			Assert.AreEqual (sha1Name, si.DigestAlgorithm.FriendlyName, "DigestAlgorithm.FriendlyName");
			Assert.AreEqual (sha1Oid, si.DigestAlgorithm.Value, "DigestAlgorithm.Value");
			Assert.AreEqual (SubjectIdentifierType.SubjectKeyIdentifier, si.SignerIdentifier.Type, "SignerIdentifier.Type");
			Assert.AreEqual ("02E1A73254AEFDC0A43236F6FE236A037228B1F7", (string)si.SignerIdentifier.Value, "SignerIdentifier.Value");
			Assert.AreEqual (0, si.UnsignedAttributes.Count, "UnsignedAttributes");
			Assert.AreEqual (3, si.Version, "Version");
		}

		[Test]
		public void ExplicitCryptographicObjectIdentifier ()
		{
			var data = new byte [] {
				48, 130, 36, 126, 6, 9, 42, 134, 72, 134, 247, 13, 1, 7, 2, 160, 130, 36, 111, 48, 130, 36, 107, 2, 1, 3, 49, 15, 48, 13, 6, 9, 96, 134, 72, 1, 101, 3, 4, 2, 1, 5, 0, 48, 100, 6, 9, 42, 134, 72, 134, 247, 13, 1, 7, 1, 160, 87, 4, 85, 86, 101, 114, 115, 105, 111, 110, 58, 49, 10, 10, 50, 46, 49, 54, 46, 56, 52, 48, 46, 49, 46, 49, 48, 49, 46, 51, 46, 52, 46, 50, 46, 49, 45, 72, 97, 115, 104, 58, 109, 117, 115, 49, 107, 97, 84, 89, 76, 113, 111, 81, 110, 85, 105, 104, 103, 104, 109, 102, 48, 120, 110, 104, 43, 69, 111, 65, 85, 73, 119, 104, 122, 50, 82, 117, 57, 52, 109, 69, 54, 117, 73, 61, 10, 10, 160, 130, 14, 42, 48, 130, 3, 183, 48, 130, 2, 159, 160, 3, 2, 1, 2, 2, 16, 12, 231, 224, 229, 23, 216, 70, 254, 143, 229, 96, 252, 27, 240, 48, 57, 48, 13, 6, 9, 42, 134, 72, 134, 247, 13, 1, 1, 5, 5, 0, 48, 101, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 85, 83, 49, 21, 48, 19, 6, 3, 85, 4, 10, 19, 12, 68, 105, 103, 105, 67, 101, 114, 116, 32, 73, 110, 99, 49, 25, 48, 23, 6, 3, 85, 4, 11, 19, 16, 119, 119, 119, 46, 100, 105, 103, 105, 99, 101, 114, 116, 46, 99, 111, 109, 49, 36, 48, 34, 6, 3, 85, 4, 3, 19, 27, 68, 105, 103, 105, 67, 101, 114, 116, 32, 65, 115, 115, 117, 114, 101, 100, 32, 73, 68, 32, 82, 111, 111, 116, 32, 67, 65, 48, 30, 23, 13, 48, 54, 49, 49, 49, 48, 48, 48, 48, 48, 48, 48, 90, 23, 13, 51, 49, 49, 49, 49, 48, 48, 48, 48, 48, 48, 48, 90, 48, 101, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 85, 83, 49, 21, 48, 19, 6, 3, 85, 4, 10, 19, 12, 68, 105, 103, 105, 67, 101, 114, 116, 32, 73, 110, 99, 49, 25, 48, 23, 6, 3, 85, 4, 11, 19, 16, 119, 119, 119, 46, 100, 105, 103, 105, 99, 101, 114, 116, 46, 99, 111, 109, 49, 36, 48, 34, 6, 3, 85, 4, 3, 19, 27, 68, 105, 103, 105, 67, 101, 114, 116, 32, 65, 115, 115, 117, 114, 101, 100, 32, 73, 68, 32, 82, 111, 111, 116, 32, 67, 65, 48, 130, 1, 34, 48, 13, 6, 9, 42, 134, 72, 134, 247, 13, 1, 1, 1, 5, 0, 3, 130, 1, 15, 0, 48, 130, 1, 10, 2, 130, 1, 1, 0, 173, 14, 21, 206, 228, 67, 128, 92, 177, 135, 243, 183, 96, 249, 113, 18, 165, 174, 220, 38, 148, 136, 170, 244, 206, 245, 32, 57, 40, 88, 96, 12, 248, 128, 218, 169, 21, 149, 50, 97, 60, 181, 177, 40, 132, 138, 138, 220, 159, 10, 12, 131, 23, 122, 143, 144, 172, 138, 231, 121, 83, 92, 49, 132, 42, 246, 15, 152, 50, 54, 118, 204, 222, 221, 60, 168, 162, 239, 106, 251, 33, 242, 82, 97, 223, 159, 32, 215, 31, 226, 177, 217, 254, 24, 100, 210, 18, 91, 95, 249, 88, 24, 53, 188, 71, 205, 161, 54, 249, 107, 127, 212, 176, 56, 62, 193, 27, 195, 140, 51, 217, 216, 47, 24, 254, 40, 15, 179, 167, 131, 214, 195, 110, 68, 192, 97, 53, 150, 22, 254, 89, 156, 139, 118, 109, 215, 241, 162, 75, 13, 43, 255, 11, 114, 218, 158, 96, 208, 142, 144, 53, 198, 120, 85, 135, 32, 161, 207, 229, 109, 10, 200, 73, 124, 49, 152, 51, 108, 34, 233, 135, 208, 50, 90, 162, 186, 19, 130, 17, 237, 57, 23, 157, 153, 58, 114, 161, 230, 250, 164, 217, 213, 23, 49, 117, 174, 133, 125, 34, 174, 63, 1, 70, 134, 246, 40, 121, 200, 177, 218, 228, 87, 23, 196, 126, 28, 14, 176, 180, 146, 166, 86, 179, 189, 178, 151, 237, 170, 167, 240, 183, 197, 168, 63, 149, 22, 208, 255, 161, 150, 235, 8, 95, 24, 119, 79, 2, 3, 1, 0, 1, 163, 99, 48, 97, 48, 14, 6, 3, 85, 29, 15, 1, 1, 255, 4, 4, 3, 2, 1, 134, 48, 15, 6, 3, 85, 29, 19, 1, 1, 255, 4, 5, 48, 3, 1, 1, 255, 48, 29, 6, 3, 85, 29, 14, 4, 22, 4, 20, 69, 235, 162, 175, 244, 146, 203, 130, 49, 45, 81, 139, 167, 167, 33, 157, 243, 109, 200, 15, 48, 31, 6, 3, 85, 29, 35, 4, 24, 48, 22, 128, 20, 69, 235, 162, 175, 244, 146, 203, 130, 49, 45, 81, 139, 167, 167, 33, 157, 243, 109, 200, 15, 48, 13, 6, 9, 42, 134, 72, 134, 247, 13, 1, 1, 5, 5, 0, 3, 130, 1, 1, 0, 162, 14, 188, 223, 226, 237, 240, 227, 114, 115, 122, 100, 148, 191, 247, 114, 102, 216, 50, 228, 66, 117, 98, 174, 135, 235, 242, 213, 217, 222, 86, 179, 159, 204, 206, 20, 40, 185, 13, 151, 96, 92, 18, 76, 88, 228, 211, 61, 131, 73, 69, 88, 151, 53, 105, 26, 168, 71, 234, 86, 198, 121, 171, 18, 216, 103, 129, 132, 223, 127, 9, 60, 148, 230, 184, 38, 44, 32, 189, 61, 179, 40, 137, 247, 95, 255, 34, 226, 151, 132, 31, 233, 101, 239, 135, 224, 223, 193, 103, 73, 179, 93, 235, 178, 9, 42, 235, 38, 237, 120, 190, 125, 63, 43, 243, 183, 38, 53, 109, 95, 137, 1, 182, 73, 91, 159, 1, 5, 155, 171, 61, 37, 193, 204, 182, 127, 194, 241, 111, 134, 198, 250, 100, 104, 235, 129, 45, 148, 235, 66, 183, 250, 140, 30, 221, 98, 241, 190, 80, 103, 183, 108, 189, 243, 241, 31, 107, 12, 54, 7, 22, 127, 55, 124, 169, 91, 109, 122, 241, 18, 70, 96, 131, 215, 39, 4, 190, 75, 206, 151, 190, 195, 103, 42, 104, 17, 223, 128, 231, 12, 51, 102, 191, 19, 13, 20, 110, 243, 127, 31, 99, 16, 30, 250, 141, 27, 37, 109, 108, 143, 165, 183, 97, 1, 177, 210, 163, 38, 161, 16, 113, 157, 173, 226, 195, 249, 195, 153, 81, 183, 43, 7, 8, 206, 46, 230, 80, 178, 167, 250, 10, 69, 47, 162, 240, 242, 48, 130, 5, 48, 48, 130, 4, 24, 160, 3, 2, 1, 2, 2, 16, 4, 9, 24, 27, 95, 213, 187, 102, 117, 83, 67, 181, 111, 149, 80, 8, 48, 13, 6, 9, 42, 134, 72, 134, 247, 13, 1, 1, 11, 5, 0, 48, 101, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 85, 83, 49, 21, 48, 19, 6, 3, 85, 4, 10, 19, 12, 68, 105, 103, 105, 67, 101, 114, 116, 32, 73, 110, 99, 49, 25, 48, 23, 6, 3, 85, 4, 11, 19, 16, 119, 119, 119, 46, 100, 105, 103, 105, 99, 101, 114, 116, 46, 99, 111, 109, 49, 36, 48, 34, 6, 3, 85, 4, 3, 19, 27, 68, 105, 103, 105, 67, 101, 114, 116, 32, 65, 115, 115, 117, 114, 101, 100, 32, 73, 68, 32, 82, 111, 111, 116, 32, 67, 65, 48, 30, 23, 13, 49, 51, 49, 48, 50, 50, 49, 50, 48, 48, 48, 48, 90, 23, 13, 50, 56, 49, 48, 50, 50, 49, 50, 48, 48, 48, 48, 90, 48, 114, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 85, 83, 49, 21, 48, 19, 6, 3, 85, 4, 10, 19, 12, 68, 105, 103, 105, 67, 101, 114, 116, 32, 73, 110, 99, 49, 25, 48, 23, 6, 3, 85, 4, 11, 19, 16, 119, 119, 119, 46, 100, 105, 103, 105, 99, 101, 114, 116, 46, 99, 111, 109, 49, 49, 48, 47, 6, 3, 85, 4, 3, 19, 40, 68, 105, 103, 105, 67, 101, 114, 116, 32, 83, 72, 65, 50, 32, 65, 115, 115, 117, 114, 101, 100, 32, 73, 68, 32, 67, 111, 100, 101, 32, 83, 105, 103, 110, 105, 110, 103, 32, 67, 65, 48, 130, 1, 34, 48, 13, 6, 9, 42, 134, 72, 134, 247, 13, 1, 1, 1, 5, 0, 3, 130, 1, 15, 0, 48, 130, 1, 10, 2, 130, 1, 1, 0, 248, 211, 179, 28, 127, 14, 17, 175, 103, 119, 7, 211, 11, 49, 73, 25, 207, 208, 251, 69, 153, 177, 58, 219, 68, 245, 127, 229, 168, 157, 219, 50, 215, 113, 234, 118, 157, 5, 46, 183, 143, 250, 146, 67, 192, 165, 249, 137, 212, 55, 25, 215, 182, 170, 240, 156, 134, 165, 216, 37, 172, 14, 121, 40, 58, 126, 233, 209, 103, 211, 198, 251, 41, 39, 199, 211, 123, 35, 148, 228, 145, 35, 150, 144, 119, 130, 249, 161, 132, 35, 102, 18, 84, 51, 80, 116, 177, 40, 38, 187, 36, 105, 194, 194, 82, 242, 20, 103, 138, 137, 69, 212, 45, 161, 163, 233, 136, 44, 32, 149, 174, 28, 74, 135, 8, 223, 12, 245, 226, 77, 96, 24, 190, 170, 196, 178, 174, 112, 49, 102, 51, 113, 62, 172, 112, 162, 171, 206, 127, 233, 124, 203, 146, 161, 229, 59, 49, 28, 207, 234, 242, 10, 228, 87, 187, 74, 181, 233, 116, 230, 43, 254, 108, 203, 126, 116, 57, 54, 13, 144, 239, 228, 181, 78, 164, 169, 234, 106, 10, 171, 132, 243, 172, 103, 78, 181, 196, 247, 140, 209, 32, 37, 35, 235, 8, 100, 62, 82, 150, 193, 242, 15, 18, 244, 197, 142, 15, 193, 162, 232, 44, 81, 247, 115, 188, 189, 133, 177, 98, 131, 115, 65, 130, 7, 228, 56, 139, 106, 115, 32, 208, 15, 100, 115, 60, 158, 159, 166, 51, 169, 253, 25, 223, 37, 147, 209, 2, 3, 1, 0, 1, 163, 130, 1, 205, 48, 130, 1, 201, 48, 18, 6, 3, 85, 29, 19, 1, 1, 255, 4, 8, 48, 6, 1, 1, 255, 2, 1, 0, 48, 14, 6, 3, 85, 29, 15, 1, 1, 255, 4, 4, 3, 2, 1, 134, 48, 19, 6, 3, 85, 29, 37, 4, 12, 48, 10, 6, 8, 43, 6, 1, 5, 5, 7, 3, 3, 48, 121, 6, 8, 43, 6, 1, 5, 5, 7, 1, 1, 4, 109, 48, 107, 48, 36, 6, 8, 43, 6, 1, 5, 5, 7, 48, 1, 134, 24, 104, 116, 116, 112, 58, 47, 47, 111, 99, 115, 112, 46, 100, 105, 103, 105, 99, 101, 114, 116, 46, 99, 111, 109, 48, 67, 6, 8, 43, 6, 1, 5, 5, 7, 48, 2, 134, 55, 104, 116, 116, 112, 58, 47, 47, 99, 97, 99, 101, 114, 116, 115, 46, 100, 105, 103, 105, 99, 101, 114, 116, 46, 99, 111, 109, 47, 68, 105, 103, 105, 67, 101, 114, 116, 65, 115, 115, 117, 114, 101, 100, 73, 68, 82, 111, 111, 116, 67, 65, 46, 99, 114, 116, 48, 129, 129, 6, 3, 85, 29, 31, 4, 122, 48, 120, 48, 58, 160, 56, 160, 54, 134, 52, 104, 116, 116, 112, 58, 47, 47, 99, 114, 108, 52, 46, 100, 105, 103, 105, 99, 101, 114, 116, 46, 99, 111, 109, 47, 68, 105, 103, 105, 67, 101, 114, 116, 65, 115, 115, 117, 114, 101, 100, 73, 68, 82, 111, 111, 116, 67, 65, 46, 99, 114, 108, 48, 58, 160, 56, 160, 54, 134, 52, 104, 116, 116, 112, 58, 47, 47, 99, 114, 108, 51, 46, 100, 105, 103, 105, 99, 101, 114, 116, 46, 99, 111, 109, 47, 68, 105, 103, 105, 67, 101, 114, 116, 65, 115, 115, 117, 114, 101, 100, 73, 68, 82, 111, 111, 116, 67, 65, 46, 99, 114, 108, 48, 79, 6, 3, 85, 29, 32, 4, 72, 48, 70, 48, 56, 6, 10, 96, 134, 72, 1, 134, 253, 108, 0, 2, 4, 48, 42, 48, 40, 6, 8, 43, 6, 1, 5, 5, 7, 2, 1, 22, 28, 104, 116, 116, 112, 115, 58, 47, 47, 119, 119, 119, 46, 100, 105, 103, 105, 99, 101, 114, 116, 46, 99, 111, 109, 47, 67, 80, 83, 48, 10, 6, 8, 96, 134, 72, 1, 134, 253, 108, 3, 48, 29, 6, 3, 85, 29, 14, 4, 22, 4, 20, 90, 196, 185, 123, 42, 10, 163, 165, 234, 113, 3, 192, 96, 249, 45, 246, 101, 117, 14, 88, 48, 31, 6, 3, 85, 29, 35, 4, 24, 48, 22, 128, 20, 69, 235, 162, 175, 244, 146, 203, 130, 49, 45, 81, 139, 167, 167, 33, 157, 243, 109, 200, 15, 48, 13, 6, 9, 42, 134, 72, 134, 247, 13, 1, 1, 11, 5, 0, 3, 130, 1, 1, 0, 62, 236, 13, 90, 36, 179, 243, 34, 209, 21, 200, 44, 124, 37, 41, 118, 168, 29, 93, 28, 45, 58, 26, 196, 239, 48, 97, 215, 126, 11, 96, 253, 195, 61, 15, 196, 175, 139, 253, 239, 42, 223, 32, 85, 55, 176, 225, 246, 209, 146, 117, 15, 81, 180, 110, 165, 142, 90, 226, 94, 36, 129, 78, 16, 164, 238, 63, 113, 142, 99, 14, 19, 75, 173, 215, 95, 68, 121, 243, 54, 20, 6, 138, 247, 156, 70, 78, 92, 255, 144, 177, 27, 7, 14, 145, 21, 251, 186, 175, 181, 81, 194, 141, 36, 174, 36, 198, 199, 39, 42, 161, 41, 40, 26, 58, 113, 40, 2, 60, 46, 145, 163, 192, 37, 17, 226, 156, 20, 71, 161, 122, 104, 104, 175, 155, 167, 92, 32, 92, 217, 113, 177, 12, 143, 187, 168, 248, 197, 18, 104, 159, 207, 64, 203, 64, 68, 165, 19, 240, 230, 100, 12, 37, 8, 66, 50, 178, 54, 138, 36, 2, 254, 47, 114, 126, 28, 215, 73, 69, 150, 232, 89, 29, 233, 250, 116, 100, 107, 178, 235, 102, 67, 218, 179, 176, 140, 213, 233, 13, 221, 246, 1, 32, 206, 153, 49, 99, 61, 8, 26, 24, 179, 129, 155, 79, 198, 147, 16, 6, 252, 7, 129, 250, 139, 218, 249, 130, 73, 247, 98, 110, 161, 83, 250, 18, 148, 24, 133, 46, 146, 145, 234, 104, 108, 68, 50, 178, 102, 161, 231, 24, 164, 154, 100, 81, 239, 48, 130, 5, 55, 48, 130, 4, 31, 160, 3, 2, 1, 2, 2, 16, 5, 164, 252, 121, 80, 158, 247, 106, 72, 125, 189, 177, 112, 191, 141, 16, 48, 13, 6, 9, 42, 134, 72, 134, 247, 13, 1, 1, 11, 5, 0, 48, 114, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 85, 83, 49, 21, 48, 19, 6, 3, 85, 4, 10, 19, 12, 68, 105, 103, 105, 67, 101, 114, 116, 32, 73, 110, 99, 49, 25, 48, 23, 6, 3, 85, 4, 11, 19, 16, 119, 119, 119, 46, 100, 105, 103, 105, 99, 101, 114, 116, 46, 99, 111, 109, 49, 49, 48, 47, 6, 3, 85, 4, 3, 19, 40, 68, 105, 103, 105, 67, 101, 114, 116, 32, 83, 72, 65, 50, 32, 65, 115, 115, 117, 114, 101, 100, 32, 73, 68, 32, 67, 111, 100, 101, 32, 83, 105, 103, 110, 105, 110, 103, 32, 67, 65, 48, 30, 23, 13, 49, 56, 48, 50, 50, 54, 48, 48, 48, 48, 48, 48, 90, 23, 13, 50, 49, 48, 49, 50, 55, 49, 50, 48, 48, 48, 48, 90, 48, 116, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 85, 83, 49, 19, 48, 17, 6, 3, 85, 4, 8, 19, 10, 87, 97, 115, 104, 105, 110, 103, 116, 111, 110, 49, 16, 48, 14, 6, 3, 85, 4, 7, 19, 7, 82, 101, 100, 109, 111, 110, 100, 49, 30, 48, 28, 6, 3, 85, 4, 10, 19, 21, 77, 105, 99, 114, 111, 115, 111, 102, 116, 32, 67, 111, 114, 112, 111, 114, 97, 116, 105, 111, 110, 49, 30, 48, 28, 6, 3, 85, 4, 3, 19, 21, 77, 105, 99, 114, 111, 115, 111, 102, 116, 32, 67, 111, 114, 112, 111, 114, 97, 116, 105, 111, 110, 48, 130, 1, 34, 48, 13, 6, 9, 42, 134, 72, 134, 247, 13, 1, 1, 1, 5, 0, 3, 130, 1, 15, 0, 48, 130, 1, 10, 2, 130, 1, 1, 0, 211, 124, 87, 1, 204, 10, 221, 84, 38, 73, 119, 123, 251, 127, 163, 37, 31, 96, 41, 3, 188, 149, 240, 41, 8, 226, 39, 190, 183, 183, 122, 255, 210, 152, 166, 98, 85, 191, 0, 149, 50, 221, 2, 174, 156, 244, 208, 132, 80, 32, 18, 74, 157, 44, 241, 203, 63, 112, 224, 200, 207, 157, 148, 177, 184, 94, 41, 159, 176, 217, 122, 199, 124, 119, 149, 95, 47, 213, 137, 178, 147, 221, 176, 96, 255, 254, 78, 24, 26, 224, 241, 130, 131, 87, 75, 197, 233, 80, 101, 140, 52, 183, 52, 128, 91, 151, 123, 42, 1, 191, 48, 142, 4, 26, 177, 134, 68, 161, 59, 34, 224, 27, 241, 207, 204, 60, 185, 86, 104, 239, 47, 249, 103, 104, 195, 43, 208, 71, 182, 133, 49, 110, 194, 120, 130, 81, 25, 49, 254, 22, 160, 67, 161, 206, 39, 254, 246, 134, 82, 161, 187, 125, 185, 58, 200, 100, 30, 178, 157, 172, 141, 23, 247, 141, 114, 139, 191, 65, 155, 179, 135, 214, 180, 35, 173, 50, 34, 144, 187, 2, 124, 180, 127, 140, 57, 136, 247, 154, 142, 127, 251, 164, 118, 118, 152, 198, 31, 65, 237, 173, 113, 32, 216, 181, 200, 41, 254, 114, 77, 104, 157, 145, 136, 108, 175, 1, 189, 17, 180, 28, 34, 234, 13, 238, 61, 186, 193, 201, 119, 172, 133, 171, 102, 6, 11, 147, 6, 24, 180, 219, 79, 43, 227, 152, 124, 183, 2, 3, 1, 0, 1, 163, 130, 1, 197, 48, 130, 1, 193, 48, 31, 6, 3, 85, 29, 35, 4, 24, 48, 22, 128, 20, 90, 196, 185, 123, 42, 10, 163, 165, 234, 113, 3, 192, 96, 249, 45, 246, 101, 117, 14, 88, 48, 29, 6, 3, 85, 29, 14, 4, 22, 4, 20, 241, 204, 113, 132, 235, 218, 83, 113, 64, 193, 165, 116, 22, 1, 120, 16, 223, 19, 48, 32, 48, 14, 6, 3, 85, 29, 15, 1, 1, 255, 4, 4, 3, 2, 7, 128, 48, 19, 6, 3, 85, 29, 37, 4, 12, 48, 10, 6, 8, 43, 6, 1, 5, 5, 7, 3, 3, 48, 119, 6, 3, 85, 29, 31, 4, 112, 48, 110, 48, 53, 160, 51, 160, 49, 134, 47, 104, 116, 116, 112, 58, 47, 47, 99, 114, 108, 51, 46, 100, 105, 103, 105, 99, 101, 114, 116, 46, 99, 111, 109, 47, 115, 104, 97, 50, 45, 97, 115, 115, 117, 114, 101, 100, 45, 99, 115, 45, 103, 49, 46, 99, 114, 108, 48, 53, 160, 51, 160, 49, 134, 47, 104, 116, 116, 112, 58, 47, 47, 99, 114, 108, 52, 46, 100, 105, 103, 105, 99, 101, 114, 116, 46, 99, 111, 109, 47, 115, 104, 97, 50, 45, 97, 115, 115, 117, 114, 101, 100, 45, 99, 115, 45, 103, 49, 46, 99, 114, 108, 48, 76, 6, 3, 85, 29, 32, 4, 69, 48, 67, 48, 55, 6, 9, 96, 134, 72, 1, 134, 253, 108, 3, 1, 48, 42, 48, 40, 6, 8, 43, 6, 1, 5, 5, 7, 2, 1, 22, 28, 104, 116, 116, 112, 115, 58, 47, 47, 119, 119, 119, 46, 100, 105, 103, 105, 99, 101, 114, 116, 46, 99, 111, 109, 47, 67, 80, 83, 48, 8, 6, 6, 103, 129, 12, 1, 4, 1, 48, 129, 132, 6, 8, 43, 6, 1, 5, 5, 7, 1, 1, 4, 120, 48, 118, 48, 36, 6, 8, 43, 6, 1, 5, 5, 7, 48, 1, 134, 24, 104, 116, 116, 112, 58, 47, 47, 111, 99, 115, 112, 46, 100, 105, 103, 105, 99, 101, 114, 116, 46, 99, 111, 109, 48, 78, 6, 8, 43, 6, 1, 5, 5, 7, 48, 2, 134, 66, 104, 116, 116, 112, 58, 47, 47, 99, 97, 99, 101, 114, 116, 115, 46, 100, 105, 103, 105, 99, 101, 114, 116, 46, 99, 111, 109, 47, 68, 105, 103, 105, 67, 101, 114, 116, 83, 72, 65, 50, 65, 115, 115, 117, 114, 101, 100, 73, 68, 67, 111, 100, 101, 83, 105, 103, 110, 105, 110, 103, 67, 65, 46, 99, 114, 116, 48, 12, 6, 3, 85, 29, 19, 1, 1, 255, 4, 2, 48, 0, 48, 13, 6, 9, 42, 134, 72, 134, 247, 13, 1, 1, 11, 5, 0, 3, 130, 1, 1, 0, 119, 98, 99, 31, 119, 97, 102, 178, 74, 74, 10, 201, 90, 131, 225, 201, 156, 243, 178, 159, 85, 60, 62, 142, 82, 115, 231, 164, 145, 51, 80, 187, 2, 193, 217, 241, 117, 245, 104, 46, 49, 199, 165, 190, 24, 65, 223, 129, 113, 46, 181, 195, 106, 182, 2, 116, 123, 244, 121, 251, 96, 169, 43, 98, 62, 234, 56, 128, 220, 6, 109, 116, 94, 112, 215, 242, 240, 62, 185, 70, 240, 69, 29, 191, 69, 191, 234, 211, 154, 97, 235, 125, 249, 52, 12, 151, 139, 246, 180, 213, 26, 221, 204, 242, 160, 65, 4, 161, 187, 8, 170, 248, 238, 18, 96, 1, 36, 62, 156, 100, 228, 159, 11, 108, 220, 200, 11, 5, 24, 85, 230, 111, 177, 156, 27, 99, 214, 191, 78, 118, 59, 57, 167, 217, 59, 12, 100, 95, 81, 65, 238, 245, 88, 69, 152, 62, 228, 92, 13, 28, 158, 225, 133, 51, 217, 240, 81, 44, 213, 102, 57, 237, 62, 162, 41, 221, 202, 218, 218, 116, 200, 160, 88, 31, 87, 235, 33, 139, 160, 34, 235, 38, 126, 35, 25, 19, 65, 147, 204, 116, 58, 116, 238, 2, 222, 205, 60, 209, 210, 29, 249, 186, 214, 244, 240, 104, 171, 55, 104, 151, 165, 88, 24, 198, 19, 165, 76, 116, 140, 3, 117, 75, 147, 45, 88, 104, 229, 93, 9, 48, 242, 201, 21, 144, 222, 146, 109, 34, 147, 202, 173, 242, 137, 205, 247, 39, 49, 130, 21, 191, 48, 130, 21, 187, 2, 1, 3, 128, 20, 241, 204, 113, 132, 235, 218, 83, 113, 64, 193, 165, 116, 22, 1, 120, 16, 223, 19, 48, 32, 48, 13, 6, 9, 96, 134, 72, 1, 101, 3, 4, 2, 1, 5, 0, 160, 130, 1, 97, 48, 24, 6, 9, 42, 134, 72, 134, 247, 13, 1, 9, 3, 49, 11, 6, 9, 42, 134, 72, 134, 247, 13, 1, 7, 1, 48, 28, 6, 9, 42, 134, 72, 134, 247, 13, 1, 9, 5, 49, 15, 23, 13, 49, 56, 48, 50, 50, 56, 50, 49, 53, 51, 48, 57, 90, 48, 30, 6, 11, 42, 134, 72, 134, 247, 13, 1, 9, 16, 2, 16, 49, 15, 48, 13, 6, 11, 42, 134, 72, 134, 247, 13, 1, 9, 16, 6, 1, 48, 47, 6, 9, 42, 134, 72, 134, 247, 13, 1, 9, 4, 49, 34, 4, 32, 232, 208, 227, 28, 160, 65, 227, 230, 127, 75, 122, 182, 107, 138, 34, 217, 15, 26, 167, 183, 176, 231, 24, 148, 96, 80, 254, 108, 36, 31, 240, 33, 48, 129, 213, 6, 11, 42, 134, 72, 134, 247, 13, 1, 9, 16, 2, 47, 49, 129, 197, 48, 129, 194, 48, 129, 191, 48, 129, 188, 48, 11, 6, 9, 96, 134, 72, 1, 101, 3, 4, 2, 1, 4, 32, 63, 144, 1, 234, 131, 197, 96, 215, 18, 194, 76, 242, 19, 195, 211, 18, 203, 59, 255, 81, 238, 137, 67, 93, 52, 48, 189, 6, 181, 208, 238, 206, 48, 129, 138, 48, 118, 164, 116, 48, 114, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 85, 83, 49, 21, 48, 19, 6, 3, 85, 4, 10, 19, 12, 68, 105, 103, 105, 67, 101, 114, 116, 32, 73, 110, 99, 49, 25, 48, 23, 6, 3, 85, 4, 11, 19, 16, 119, 119, 119, 46, 100, 105, 103, 105, 99, 101, 114, 116, 46, 99, 111, 109, 49, 49, 48, 47, 6, 3, 85, 4, 3, 19, 40, 68, 105, 103, 105, 67, 101, 114, 116, 32, 83, 72, 65, 50, 32, 65, 115, 115, 117, 114, 101, 100, 32, 73, 68, 32, 67, 111, 100, 101, 32, 83, 105, 103, 110, 105, 110, 103, 32, 67, 65, 2, 16, 5, 164, 252, 121, 80, 158, 247, 106, 72, 125, 189, 177, 112, 191, 141, 16, 48, 13, 6, 9, 42, 134, 72, 134, 247, 13, 1, 1, 1, 5, 0, 4, 130, 1, 0, 189, 12, 245, 218, 226, 188, 198, 52, 13, 113, 29, 234, 154, 142, 105, 162, 167, 196, 37, 192, 93, 5, 39, 90, 53, 216, 124, 178, 245, 38, 67, 236, 143, 33, 59, 226, 152, 102, 38, 239, 160, 128, 212, 148, 215, 183, 201, 135, 110, 221, 86, 171, 79, 82, 77, 104, 153, 74, 107, 22, 168, 104, 165, 218, 87, 222, 188, 111, 163, 173, 157, 131, 192, 73, 92, 13, 104, 214, 112, 147, 16, 240, 246, 12, 238, 254, 50, 57, 133, 237, 137, 25, 211, 63, 102, 1, 214, 157, 125, 211, 233, 90, 51, 225, 61, 80, 57, 173, 234, 229, 211, 1, 159, 0, 16, 231, 165, 146, 111, 172, 50, 43, 34, 40, 87, 199, 209, 41, 190, 254, 169, 85, 2, 70, 70, 60, 174, 155, 91, 201, 31, 149, 79, 244, 75, 247, 175, 26, 234, 65, 201, 16, 45, 64, 129, 109, 218, 79, 21, 188, 251, 246, 173, 151, 121, 64, 240, 78, 236, 177, 251, 223, 52, 168, 249, 158, 138, 47, 20, 209, 47, 137, 188, 75, 165, 34, 112, 228, 72, 131, 29, 43, 241, 68, 135, 48, 222, 186, 239, 82, 4, 57, 223, 17, 107, 78, 117, 192, 120, 42, 77, 60, 18, 79, 33, 57, 42, 97, 176, 14, 161, 19, 27, 57, 237, 167, 39, 22, 80, 39, 17, 37, 161, 113, 73, 88, 26, 96, 5, 164, 54, 211, 25, 250, 59, 71, 82, 60, 224, 39, 249, 185, 224, 145, 146, 172, 161, 130, 19, 23, 48, 130, 19, 19, 6, 11, 42, 134, 72, 134, 247, 13, 1, 9, 16, 2, 14, 49, 130, 19, 2, 48, 130, 18, 254, 6, 9, 42, 134, 72, 134, 247, 13, 1, 7, 2, 160, 130, 18, 239, 48, 130, 18, 235, 2, 1, 3, 49, 13, 48, 11, 6, 9, 96, 134, 72, 1, 101, 3, 4, 2, 1, 48, 130, 1, 43, 6, 11, 42, 134, 72, 134, 247, 13, 1, 9, 16, 1, 4, 160, 130, 1, 26, 4, 130, 1, 22, 48, 130, 1, 18, 2, 1, 1, 6, 11, 96, 134, 72, 1, 134, 248, 69, 1, 7, 23, 3, 48, 49, 48, 13, 6, 9, 96, 134, 72, 1, 101, 3, 4, 2, 1, 5, 0, 4, 32, 69, 124, 130, 19, 157, 209, 175, 133, 231, 103, 158, 39, 132, 2, 3, 70, 105, 83, 22, 206, 167, 181, 233, 3, 244, 28, 114, 148, 185, 25, 25, 130, 2, 20, 90, 184, 139, 103, 48, 158, 128, 176, 39, 47, 50, 149, 253, 125, 115, 81, 111, 111, 163, 17, 24, 15, 50, 48, 49, 56, 48, 50, 50, 56, 50, 49, 53, 51, 48, 57, 90, 48, 3, 2, 1, 30, 2, 24, 194, 204, 146, 137, 35, 88, 111, 10, 205, 159, 60, 26, 8, 210, 62, 229, 119, 182, 204, 15, 23, 243, 233, 211, 160, 129, 134, 164, 129, 131, 48, 129, 128, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 85, 83, 49, 29, 48, 27, 6, 3, 85, 4, 10, 19, 20, 83, 121, 109, 97, 110, 116, 101, 99, 32, 67, 111, 114, 112, 111, 114, 97, 116, 105, 111, 110, 49, 31, 48, 29, 6, 3, 85, 4, 11, 19, 22, 83, 121, 109, 97, 110, 116, 101, 99, 32, 84, 114, 117, 115, 116, 32, 78, 101, 116, 119, 111, 114, 107, 49, 49, 48, 47, 6, 3, 85, 4, 3, 19, 40, 83, 121, 109, 97, 110, 116, 101, 99, 32, 83, 72, 65, 50, 53, 54, 32, 84, 105, 109, 101, 83, 116, 97, 109, 112, 105, 110, 103, 32, 83, 105, 103, 110, 101, 114, 32, 45, 32, 71, 50, 160, 130, 15, 72, 48, 130, 4, 185, 48, 130, 3, 161, 160, 3, 2, 1, 2, 2, 16, 64, 26, 196, 100, 33, 179, 19, 33, 3, 14, 187, 228, 18, 26, 197, 29, 48, 13, 6, 9, 42, 134, 72, 134, 247, 13, 1, 1, 11, 5, 0, 48, 129, 189, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 85, 83, 49, 23, 48, 21, 6, 3, 85, 4, 10, 19, 14, 86, 101, 114, 105, 83, 105, 103, 110, 44, 32, 73, 110, 99, 46, 49, 31, 48, 29, 6, 3, 85, 4, 11, 19, 22, 86, 101, 114, 105, 83, 105, 103, 110, 32, 84, 114, 117, 115, 116, 32, 78, 101, 116, 119, 111, 114, 107, 49, 58, 48, 56, 6, 3, 85, 4, 11, 19, 49, 40, 99, 41, 32, 50, 48, 48, 56, 32, 86, 101, 114, 105, 83, 105, 103, 110, 44, 32, 73, 110, 99, 46, 32, 45, 32, 70, 111, 114, 32, 97, 117, 116, 104, 111, 114, 105, 122, 101, 100, 32, 117, 115, 101, 32, 111, 110, 108, 121, 49, 56, 48, 54, 6, 3, 85, 4, 3, 19, 47, 86, 101, 114, 105, 83, 105, 103, 110, 32, 85, 110, 105, 118, 101, 114, 115, 97, 108, 32, 82, 111, 111, 116, 32, 67, 101, 114, 116, 105, 102, 105, 99, 97, 116, 105, 111, 110, 32, 65, 117, 116, 104, 111, 114, 105, 116, 121, 48, 30, 23, 13, 48, 56, 48, 52, 48, 50, 48, 48, 48, 48, 48, 48, 90, 23, 13, 51, 55, 49, 50, 48, 49, 50, 51, 53, 57, 53, 57, 90, 48, 129, 189, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 85, 83, 49, 23, 48, 21, 6, 3, 85, 4, 10, 19, 14, 86, 101, 114, 105, 83, 105, 103, 110, 44, 32, 73, 110, 99, 46, 49, 31, 48, 29, 6, 3, 85, 4, 11, 19, 22, 86, 101, 114, 105, 83, 105, 103, 110, 32, 84, 114, 117, 115, 116, 32, 78, 101, 116, 119, 111, 114, 107, 49, 58, 48, 56, 6, 3, 85, 4, 11, 19, 49, 40, 99, 41, 32, 50, 48, 48, 56, 32, 86, 101, 114, 105, 83, 105, 103, 110, 44, 32, 73, 110, 99, 46, 32, 45, 32, 70, 111, 114, 32, 97, 117, 116, 104, 111, 114, 105, 122, 101, 100, 32, 117, 115, 101, 32, 111, 110, 108, 121, 49, 56, 48, 54, 6, 3, 85, 4, 3, 19, 47, 86, 101, 114, 105, 83, 105, 103, 110, 32, 85, 110, 105, 118, 101, 114, 115, 97, 108, 32, 82, 111, 111, 116, 32, 67, 101, 114, 116, 105, 102, 105, 99, 97, 116, 105, 111, 110, 32, 65, 117, 116, 104, 111, 114, 105, 116, 121, 48, 130, 1, 34, 48, 13, 6, 9, 42, 134, 72, 134, 247, 13, 1, 1, 1, 5, 0, 3, 130, 1, 15, 0, 48, 130, 1, 10, 2, 130, 1, 1, 0, 199, 97, 55, 94, 177, 1, 52, 219, 98, 215, 21, 155, 255, 88, 90, 140, 35, 35, 214, 96, 142, 145, 215, 144, 152, 131, 122, 230, 88, 25, 56, 140, 197, 246, 229, 100, 133, 180, 162, 113, 251, 237, 189, 185, 218, 205, 77, 0, 180, 200, 45, 115, 165, 199, 105, 113, 149, 31, 57, 60, 178, 68, 7, 156, 232, 14, 250, 77, 74, 196, 33, 223, 41, 97, 143, 50, 34, 97, 130, 197, 135, 31, 110, 140, 124, 95, 22, 32, 81, 68, 209, 112, 79, 87, 234, 227, 28, 227, 204, 121, 238, 88, 216, 14, 194, 179, 69, 147, 192, 44, 231, 154, 23, 43, 123, 0, 55, 122, 65, 51, 120, 225, 51, 226, 243, 16, 26, 127, 135, 44, 190, 246, 245, 247, 66, 226, 229, 191, 135, 98, 137, 95, 0, 75, 223, 197, 221, 228, 117, 68, 50, 65, 58, 30, 113, 110, 105, 203, 11, 117, 70, 8, 209, 202, 210, 43, 149, 208, 207, 251, 185, 64, 107, 100, 140, 87, 77, 252, 19, 17, 121, 132, 237, 94, 84, 246, 52, 159, 8, 1, 243, 16, 37, 6, 23, 74, 218, 241, 29, 122, 102, 107, 152, 96, 102, 164, 217, 239, 210, 46, 130, 241, 240, 239, 9, 234, 68, 201, 21, 106, 226, 3, 110, 51, 211, 172, 159, 85, 0, 199, 246, 8, 106, 148, 185, 95, 220, 224, 51, 241, 132, 96, 249, 91, 39, 17, 180, 252, 22, 242, 187, 86, 106, 128, 37, 141, 2, 3, 1, 0, 1, 163, 129, 178, 48, 129, 175, 48, 15, 6, 3, 85, 29, 19, 1, 1, 255, 4, 5, 48, 3, 1, 1, 255, 48, 14, 6, 3, 85, 29, 15, 1, 1, 255, 4, 4, 3, 2, 1, 6, 48, 109, 6, 8, 43, 6, 1, 5, 5, 7, 1, 12, 4, 97, 48, 95, 161, 93, 160, 91, 48, 89, 48, 87, 48, 85, 22, 9, 105, 109, 97, 103, 101, 47, 103, 105, 102, 48, 33, 48, 31, 48, 7, 6, 5, 43, 14, 3, 2, 26, 4, 20, 143, 229, 211, 26, 134, 172, 141, 142, 107, 195, 207, 128, 106, 212, 72, 24, 44, 123, 25, 46, 48, 37, 22, 35, 104, 116, 116, 112, 58, 47, 47, 108, 111, 103, 111, 46, 118, 101, 114, 105, 115, 105, 103, 110, 46, 99, 111, 109, 47, 118, 115, 108, 111, 103, 111, 46, 103, 105, 102, 48, 29, 6, 3, 85, 29, 14, 4, 22, 4, 20, 182, 119, 250, 105, 72, 71, 159, 83, 18, 213, 194, 234, 7, 50, 118, 7, 209, 151, 7, 25, 48, 13, 6, 9, 42, 134, 72, 134, 247, 13, 1, 1, 11, 5, 0, 3, 130, 1, 1, 0, 74, 248, 248, 176, 3, 230, 44, 103, 123, 228, 148, 119, 99, 204, 110, 76, 249, 125, 14, 13, 220, 200, 185, 53, 185, 112, 79, 99, 250, 36, 250, 108, 131, 140, 71, 157, 59, 99, 243, 154, 249, 118, 50, 149, 145, 177, 119, 188, 172, 154, 190, 177, 228, 49, 33, 198, 129, 149, 86, 90, 14, 177, 194, 212, 177, 166, 89, 172, 241, 99, 203, 184, 76, 29, 89, 144, 74, 239, 144, 22, 40, 31, 90, 174, 16, 251, 129, 80, 56, 12, 108, 204, 241, 61, 195, 245, 99, 227, 179, 227, 33, 201, 36, 57, 233, 253, 21, 102, 70, 244, 27, 17, 208, 77, 115, 163, 125, 70, 249, 61, 237, 168, 95, 98, 212, 241, 63, 248, 224, 116, 87, 43, 24, 157, 129, 180, 196, 40, 218, 148, 151, 165, 112, 235, 172, 29, 190, 7, 17, 240, 213, 219, 221, 229, 140, 240, 213, 50, 176, 131, 230, 87, 226, 143, 191, 190, 161, 170, 191, 61, 29, 181, 212, 56, 234, 215, 176, 92, 58, 79, 106, 63, 143, 192, 102, 108, 99, 170, 233, 217, 164, 22, 244, 129, 209, 149, 20, 14, 125, 205, 149, 52, 217, 210, 143, 112, 115, 129, 123, 156, 126, 189, 152, 97, 216, 69, 135, 152, 144, 197, 235, 134, 48, 198, 53, 191, 240, 255, 195, 85, 136, 131, 75, 239, 5, 146, 6, 113, 242, 184, 152, 147, 183, 236, 205, 130, 97, 241, 56, 230, 79, 151, 152, 42, 90, 141, 48, 130, 5, 56, 48, 130, 4, 32, 160, 3, 2, 1, 2, 2, 16, 123, 5, 177, 212, 73, 104, 81, 68, 247, 201, 137, 210, 156, 25, 157, 18, 48, 13, 6, 9, 42, 134, 72, 134, 247, 13, 1, 1, 11, 5, 0, 48, 129, 189, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 85, 83, 49, 23, 48, 21, 6, 3, 85, 4, 10, 19, 14, 86, 101, 114, 105, 83, 105, 103, 110, 44, 32, 73, 110, 99, 46, 49, 31, 48, 29, 6, 3, 85, 4, 11, 19, 22, 86, 101, 114, 105, 83, 105, 103, 110, 32, 84, 114, 117, 115, 116, 32, 78, 101, 116, 119, 111, 114, 107, 49, 58, 48, 56, 6, 3, 85, 4, 11, 19, 49, 40, 99, 41, 32, 50, 48, 48, 56, 32, 86, 101, 114, 105, 83, 105, 103, 110, 44, 32, 73, 110, 99, 46, 32, 45, 32, 70, 111, 114, 32, 97, 117, 116, 104, 111, 114, 105, 122, 101, 100, 32, 117, 115, 101, 32, 111, 110, 108, 121, 49, 56, 48, 54, 6, 3, 85, 4, 3, 19, 47, 86, 101, 114, 105, 83, 105, 103, 110, 32, 85, 110, 105, 118, 101, 114, 115, 97, 108, 32, 82, 111, 111, 116, 32, 67, 101, 114, 116, 105, 102, 105, 99, 97, 116, 105, 111, 110, 32, 65, 117, 116, 104, 111, 114, 105, 116, 121, 48, 30, 23, 13, 49, 54, 48, 49, 49, 50, 48, 48, 48, 48, 48, 48, 90, 23, 13, 51, 49, 48, 49, 49, 49, 50, 51, 53, 57, 53, 57, 90, 48, 119, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 85, 83, 49, 29, 48, 27, 6, 3, 85, 4, 10, 19, 20, 83, 121, 109, 97, 110, 116, 101, 99, 32, 67, 111, 114, 112, 111, 114, 97, 116, 105, 111, 110, 49, 31, 48, 29, 6, 3, 85, 4, 11, 19, 22, 83, 121, 109, 97, 110, 116, 101, 99, 32, 84, 114, 117, 115, 116, 32, 78, 101, 116, 119, 111, 114, 107, 49, 40, 48, 38, 6, 3, 85, 4, 3, 19, 31, 83, 121, 109, 97, 110, 116, 101, 99, 32, 83, 72, 65, 50, 53, 54, 32, 84, 105, 109, 101, 83, 116, 97, 109, 112, 105, 110, 103, 32, 67, 65, 48, 130, 1, 34, 48, 13, 6, 9, 42, 134, 72, 134, 247, 13, 1, 1, 1, 5, 0, 3, 130, 1, 15, 0, 48, 130, 1, 10, 2, 130, 1, 1, 0, 187, 89, 157, 89, 85, 79, 157, 140, 114, 93, 26, 129, 162, 235, 85, 243, 176, 1, 173, 60, 113, 172, 50, 143, 5, 107, 134, 154, 39, 0, 50, 151, 106, 77, 201, 100, 20, 75, 41, 187, 194, 217, 41, 185, 46, 236, 99, 179, 225, 207, 63, 11, 86, 144, 248, 98, 27, 126, 235, 166, 7, 226, 222, 127, 94, 109, 64, 56, 212, 145, 6, 231, 65, 124, 121, 28, 203, 203, 173, 27, 191, 216, 149, 145, 243, 240, 238, 108, 248, 173, 150, 57, 46, 127, 193, 39, 184, 120, 57, 197, 132, 165, 237, 237, 175, 135, 142, 206, 141, 199, 109, 234, 210, 152, 181, 58, 31, 30, 57, 157, 195, 244, 154, 168, 244, 132, 225, 196, 209, 124, 113, 198, 6, 41, 180, 63, 228, 131, 13, 38, 195, 123, 8, 62, 77, 249, 10, 183, 51, 73, 255, 202, 59, 212, 245, 178, 155, 75, 225, 136, 153, 26, 245, 192, 233, 51, 20, 214, 223, 199, 128, 219, 145, 238, 254, 188, 146, 87, 114, 119, 244, 205, 168, 204, 254, 9, 245, 147, 55, 190, 149, 136, 106, 197, 220, 244, 177, 75, 212, 206, 232, 9, 145, 95, 181, 132, 121, 53, 138, 120, 172, 25, 50, 143, 35, 193, 50, 65, 27, 89, 14, 169, 62, 177, 204, 249, 214, 43, 239, 183, 216, 228, 213, 29, 109, 17, 58, 146, 246, 147, 201, 156, 227, 72, 238, 187, 83, 14, 212, 54, 151, 134, 120, 197, 161, 55, 2, 3, 1, 0, 1, 163, 130, 1, 119, 48, 130, 1, 115, 48, 14, 6, 3, 85, 29, 15, 1, 1, 255, 4, 4, 3, 2, 1, 6, 48, 18, 6, 3, 85, 29, 19, 1, 1, 255, 4, 8, 48, 6, 1, 1, 255, 2, 1, 0, 48, 102, 6, 3, 85, 29, 32, 4, 95, 48, 93, 48, 91, 6, 11, 96, 134, 72, 1, 134, 248, 69, 1, 7, 23, 3, 48, 76, 48, 35, 6, 8, 43, 6, 1, 5, 5, 7, 2, 1, 22, 23, 104, 116, 116, 112, 115, 58, 47, 47, 100, 46, 115, 121, 109, 99, 98, 46, 99, 111, 109, 47, 99, 112, 115, 48, 37, 6, 8, 43, 6, 1, 5, 5, 7, 2, 2, 48, 25, 26, 23, 104, 116, 116, 112, 115, 58, 47, 47, 100, 46, 115, 121, 109, 99, 98, 46, 99, 111, 109, 47, 114, 112, 97, 48, 46, 6, 8, 43, 6, 1, 5, 5, 7, 1, 1, 4, 34, 48, 32, 48, 30, 6, 8, 43, 6, 1, 5, 5, 7, 48, 1, 134, 18, 104, 116, 116, 112, 58, 47, 47, 115, 46, 115, 121, 109, 99, 100, 46, 99, 111, 109, 48, 54, 6, 3, 85, 29, 31, 4, 47, 48, 45, 48, 43, 160, 41, 160, 39, 134, 37, 104, 116, 116, 112, 58, 47, 47, 115, 46, 115, 121, 109, 99, 98, 46, 99, 111, 109, 47, 117, 110, 105, 118, 101, 114, 115, 97, 108, 45, 114, 111, 111, 116, 46, 99, 114, 108, 48, 19, 6, 3, 85, 29, 37, 4, 12, 48, 10, 6, 8, 43, 6, 1, 5, 5, 7, 3, 8, 48, 40, 6, 3, 85, 29, 17, 4, 33, 48, 31, 164, 29, 48, 27, 49, 25, 48, 23, 6, 3, 85, 4, 3, 19, 16, 84, 105, 109, 101, 83, 116, 97, 109, 112, 45, 50, 48, 52, 56, 45, 51, 48, 29, 6, 3, 85, 29, 14, 4, 22, 4, 20, 175, 99, 214, 202, 163, 78, 133, 114, 224, 167, 188, 65, 243, 41, 162, 56, 127, 128, 117, 98, 48, 31, 6, 3, 85, 29, 35, 4, 24, 48, 22, 128, 20, 182, 119, 250, 105, 72, 71, 159, 83, 18, 213, 194, 234, 7, 50, 118, 7, 209, 151, 7, 25, 48, 13, 6, 9, 42, 134, 72, 134, 247, 13, 1, 1, 11, 5, 0, 3, 130, 1, 1, 0, 117, 234, 176, 45, 213, 52, 25, 92, 50, 69, 254, 14, 225, 212, 79, 166, 120, 193, 111, 215, 234, 221, 220, 79, 243, 161, 200, 129, 136, 247, 167, 143, 21, 230, 64, 41, 173, 230, 93, 244, 162, 217, 86, 100, 132, 113, 48, 42, 221, 30, 97, 23, 102, 32, 86, 6, 152, 25, 141, 93, 113, 242, 248, 151, 188, 9, 253, 28, 145, 71, 201, 226, 232, 141, 3, 251, 204, 144, 47, 214, 10, 108, 78, 51, 236, 214, 180, 147, 200, 76, 144, 99, 72, 57, 64, 33, 196, 221, 214, 110, 137, 152, 60, 181, 152, 151, 232, 169, 6, 183, 9, 201, 143, 83, 87, 65, 144, 47, 225, 30, 77, 78, 220, 202, 16, 120, 108, 66, 110, 240, 182, 197, 248, 97, 92, 82, 245, 78, 246, 107, 141, 247, 74, 122, 190, 243, 205, 253, 3, 215, 217, 246, 3, 168, 15, 227, 83, 247, 10, 117, 236, 198, 117, 46, 170, 102, 133, 4, 153, 183, 248, 6, 87, 225, 198, 14, 246, 232, 175, 218, 236, 155, 24, 31, 170, 185, 227, 58, 0, 191, 206, 138, 148, 203, 1, 219, 158, 199, 56, 187, 15, 82, 171, 209, 227, 148, 3, 96, 10, 77, 160, 254, 39, 109, 20, 50, 252, 63, 151, 64, 225, 191, 153, 137, 219, 228, 57, 20, 189, 218, 228, 211, 195, 234, 43, 90, 179, 149, 88, 85, 4, 125, 199, 154, 236, 35, 3, 141, 133, 42, 210, 255, 174, 169, 97, 129, 48, 130, 5, 75, 48, 130, 4, 51, 160, 3, 2, 1, 2, 2, 16, 84, 88, 242, 170, 215, 65, 214, 68, 188, 132, 169, 123, 160, 150, 82, 230, 48, 13, 6, 9, 42, 134, 72, 134, 247, 13, 1, 1, 11, 5, 0, 48, 119, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 85, 83, 49, 29, 48, 27, 6, 3, 85, 4, 10, 19, 20, 83, 121, 109, 97, 110, 116, 101, 99, 32, 67, 111, 114, 112, 111, 114, 97, 116, 105, 111, 110, 49, 31, 48, 29, 6, 3, 85, 4, 11, 19, 22, 83, 121, 109, 97, 110, 116, 101, 99, 32, 84, 114, 117, 115, 116, 32, 78, 101, 116, 119, 111, 114, 107, 49, 40, 48, 38, 6, 3, 85, 4, 3, 19, 31, 83, 121, 109, 97, 110, 116, 101, 99, 32, 83, 72, 65, 50, 53, 54, 32, 84, 105, 109, 101, 83, 116, 97, 109, 112, 105, 110, 103, 32, 67, 65, 48, 30, 23, 13, 49, 55, 48, 49, 48, 50, 48, 48, 48, 48, 48, 48, 90, 23, 13, 50, 56, 48, 52, 48, 49, 50, 51, 53, 57, 53, 57, 90, 48, 129, 128, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 85, 83, 49, 29, 48, 27, 6, 3, 85, 4, 10, 19, 20, 83, 121, 109, 97, 110, 116, 101, 99, 32, 67, 111, 114, 112, 111, 114, 97, 116, 105, 111, 110, 49, 31, 48, 29, 6, 3, 85, 4, 11, 19, 22, 83, 121, 109, 97, 110, 116, 101, 99, 32, 84, 114, 117, 115, 116, 32, 78, 101, 116, 119, 111, 114, 107, 49, 49, 48, 47, 6, 3, 85, 4, 3, 19, 40, 83, 121, 109, 97, 110, 116, 101, 99, 32, 83, 72, 65, 50, 53, 54, 32, 84, 105, 109, 101, 83, 116, 97, 109, 112, 105, 110, 103, 32, 83, 105, 103, 110, 101, 114, 32, 45, 32, 71, 50, 48, 130, 1, 34, 48, 13, 6, 9, 42, 134, 72, 134, 247, 13, 1, 1, 1, 5, 0, 3, 130, 1, 15, 0, 48, 130, 1, 10, 2, 130, 1, 1, 0, 153, 243, 252, 216, 4, 9, 3, 134, 249, 215, 92, 166, 147, 192, 66, 124, 234, 124, 99, 207, 93, 0, 226, 142, 243, 192, 144, 223, 143, 41, 245, 24, 234, 148, 183, 146, 229, 215, 176, 160, 115, 129, 232, 233, 10, 155, 74, 124, 1, 255, 157, 143, 164, 57, 167, 14, 234, 69, 244, 34, 12, 58, 112, 237, 57, 69, 139, 228, 197, 27, 92, 240, 69, 104, 70, 36, 5, 99, 118, 155, 28, 252, 158, 108, 42, 177, 86, 229, 138, 127, 82, 113, 174, 242, 53, 213, 70, 35, 6, 28, 207, 72, 45, 29, 180, 205, 184, 217, 118, 35, 142, 28, 255, 62, 191, 187, 6, 92, 105, 7, 166, 101, 14, 248, 94, 174, 125, 46, 237, 77, 174, 53, 239, 201, 215, 0, 66, 253, 40, 149, 14, 159, 93, 114, 66, 9, 188, 195, 218, 68, 210, 237, 204, 71, 132, 228, 252, 202, 45, 172, 88, 190, 174, 247, 174, 217, 68, 13, 8, 183, 194, 119, 214, 26, 67, 112, 209, 110, 3, 222, 82, 146, 196, 16, 8, 113, 217, 186, 34, 85, 242, 31, 188, 237, 155, 157, 59, 226, 94, 29, 76, 131, 255, 151, 15, 123, 11, 231, 85, 131, 78, 210, 13, 235, 190, 215, 236, 174, 110, 71, 185, 159, 223, 165, 214, 81, 188, 4, 85, 237, 255, 39, 112, 76, 201, 237, 42, 75, 19, 225, 177, 185, 76, 15, 201, 1, 238, 85, 101, 95, 105, 2, 120, 102, 203, 63, 2, 3, 1, 0, 1, 163, 130, 1, 199, 48, 130, 1, 195, 48, 12, 6, 3, 85, 29, 19, 1, 1, 255, 4, 2, 48, 0, 48, 102, 6, 3, 85, 29, 32, 4, 95, 48, 93, 48, 91, 6, 11, 96, 134, 72, 1, 134, 248, 69, 1, 7, 23, 3, 48, 76, 48, 35, 6, 8, 43, 6, 1, 5, 5, 7, 2, 1, 22, 23, 104, 116, 116, 112, 115, 58, 47, 47, 100, 46, 115, 121, 109, 99, 98, 46, 99, 111, 109, 47, 99, 112, 115, 48, 37, 6, 8, 43, 6, 1, 5, 5, 7, 2, 2, 48, 25, 26, 23, 104, 116, 116, 112, 115, 58, 47, 47, 100, 46, 115, 121, 109, 99, 98, 46, 99, 111, 109, 47, 114, 112, 97, 48, 64, 6, 3, 85, 29, 31, 4, 57, 48, 55, 48, 53, 160, 51, 160, 49, 134, 47, 104, 116, 116, 112, 58, 47, 47, 116, 115, 45, 99, 114, 108, 46, 119, 115, 46, 115, 121, 109, 97, 110, 116, 101, 99, 46, 99, 111, 109, 47, 115, 104, 97, 50, 53, 54, 45, 116, 115, 115, 45, 99, 97, 46, 99, 114, 108, 48, 22, 6, 3, 85, 29, 37, 1, 1, 255, 4, 12, 48, 10, 6, 8, 43, 6, 1, 5, 5, 7, 3, 8, 48, 14, 6, 3, 85, 29, 15, 1, 1, 255, 4, 4, 3, 2, 7, 128, 48, 119, 6, 8, 43, 6, 1, 5, 5, 7, 1, 1, 4, 107, 48, 105, 48, 42, 6, 8, 43, 6, 1, 5, 5, 7, 48, 1, 134, 30, 104, 116, 116, 112, 58, 47, 47, 116, 115, 45, 111, 99, 115, 112, 46, 119, 115, 46, 115, 121, 109, 97, 110, 116, 101, 99, 46, 99, 111, 109, 48, 59, 6, 8, 43, 6, 1, 5, 5, 7, 48, 2, 134, 47, 104, 116, 116, 112, 58, 47, 47, 116, 115, 45, 97, 105, 97, 46, 119, 115, 46, 115, 121, 109, 97, 110, 116, 101, 99, 46, 99, 111, 109, 47, 115, 104, 97, 50, 53, 54, 45, 116, 115, 115, 45, 99, 97, 46, 99, 101, 114, 48, 40, 6, 3, 85, 29, 17, 4, 33, 48, 31, 164, 29, 48, 27, 49, 25, 48, 23, 6, 3, 85, 4, 3, 19, 16, 84, 105, 109, 101, 83, 116, 97, 109, 112, 45, 50, 48, 52, 56, 45, 53, 48, 29, 6, 3, 85, 29, 14, 4, 22, 4, 20, 9, 181, 193, 254, 150, 114, 151, 41, 67, 154, 201, 224, 2, 186, 174, 248, 253, 47, 186, 246, 48, 31, 6, 3, 85, 29, 35, 4, 24, 48, 22, 128, 20, 175, 99, 214, 202, 163, 78, 133, 114, 224, 167, 188, 65, 243, 41, 162, 56, 127, 128, 117, 98, 48, 13, 6, 9, 42, 134, 72, 134, 247, 13, 1, 1, 11, 5, 0, 3, 130, 1, 1, 0, 23, 179, 10, 136, 233, 92, 90, 94, 32, 107, 59, 10, 21, 178, 108, 197, 169, 138, 50, 135, 211, 177, 244, 28, 83, 174, 133, 190, 63, 155, 255, 215, 188, 183, 148, 133, 180, 199, 82, 126, 148, 232, 189, 237, 97, 178, 212, 167, 153, 228, 195, 201, 147, 193, 53, 61, 11, 232, 104, 10, 93, 86, 152, 189, 177, 34, 59, 209, 68, 122, 215, 191, 240, 109, 81, 50, 138, 213, 35, 223, 56, 1, 55, 246, 226, 83, 43, 122, 43, 17, 143, 183, 77, 108, 122, 51, 3, 27, 124, 107, 9, 148, 23, 187, 228, 219, 88, 212, 33, 19, 101, 231, 236, 209, 37, 202, 44, 117, 154, 156, 127, 252, 201, 187, 42, 104, 171, 196, 125, 180, 207, 163, 201, 108, 167, 217, 196, 0, 156, 137, 10, 119, 145, 244, 77, 162, 251, 49, 59, 134, 110, 246, 230, 31, 80, 3, 134, 155, 191, 203, 66, 171, 230, 118, 155, 114, 90, 17, 1, 138, 198, 239, 165, 111, 149, 231, 221, 174, 186, 230, 34, 101, 240, 24, 89, 27, 17, 201, 205, 128, 183, 216, 151, 71, 31, 66, 8, 248, 172, 113, 31, 176, 70, 83, 179, 212, 178, 213, 163, 171, 80, 117, 72, 18, 23, 130, 173, 207, 224, 65, 79, 50, 126, 205, 149, 28, 191, 145, 138, 8, 61, 164, 167, 103, 2, 150, 223, 36, 76, 165, 208, 65, 192, 130, 96, 163, 138, 23, 50, 75, 211, 188, 207, 164, 180, 140, 49, 130, 2, 90, 48, 130, 2, 86, 2, 1, 1, 48, 129, 139, 48, 119, 49, 11, 48, 9, 6, 3, 85, 4, 6, 19, 2, 85, 83, 49, 29, 48, 27, 6, 3, 85, 4, 10, 19, 20, 83, 121, 109, 97, 110, 116, 101, 99, 32, 67, 111, 114, 112, 111, 114, 97, 116, 105, 111, 110, 49, 31, 48, 29, 6, 3, 85, 4, 11, 19, 22, 83, 121, 109, 97, 110, 116, 101, 99, 32, 84, 114, 117, 115, 116, 32, 78, 101, 116, 119, 111, 114, 107, 49, 40, 48, 38, 6, 3, 85, 4, 3, 19, 31, 83, 121, 109, 97, 110, 116, 101, 99, 32, 83, 72, 65, 50, 53, 54, 32, 84, 105, 109, 101, 83, 116, 97, 109, 112, 105, 110, 103, 32, 67, 65, 2, 16, 84, 88, 242, 170, 215, 65, 214, 68, 188, 132, 169, 123, 160, 150, 82, 230, 48, 11, 6, 9, 96, 134, 72, 1, 101, 3, 4, 2, 1, 160, 129, 164, 48, 26, 6, 9, 42, 134, 72, 134, 247, 13, 1, 9, 3, 49, 13, 6, 11, 42, 134, 72, 134, 247, 13, 1, 9, 16, 1, 4, 48, 28, 6, 9, 42, 134, 72, 134, 247, 13, 1, 9, 5, 49, 15, 23, 13, 49, 56, 48, 50, 50, 56, 50, 49, 53, 51, 48, 57, 90, 48, 47, 6, 9, 42, 134, 72, 134, 247, 13, 1, 9, 4, 49, 34, 4, 32, 76, 216, 185, 107, 197, 171, 110, 228, 214, 76, 53, 220, 252, 90, 45, 220, 13, 124, 42, 171, 249, 197, 175, 183, 34, 167, 215, 36, 172, 247, 134, 236, 48, 55, 6, 11, 42, 134, 72, 134, 247, 13, 1, 9, 16, 2, 47, 49, 40, 48, 38, 48, 36, 48, 34, 4, 32, 207, 122, 193, 122, 208, 71, 236, 213, 253, 195, 104, 34, 3, 27, 18, 212, 239, 7, 139, 111, 43, 76, 94, 107, 164, 31, 143, 242, 207, 75, 173, 103, 48, 11, 6, 9, 42, 134, 72, 134, 247, 13, 1, 1, 1, 4, 130, 1, 0, 38, 85, 141, 162, 207, 88, 183, 218, 160, 93, 177, 149, 166, 96, 13, 49, 97, 118, 237, 148, 243, 40, 185, 9, 181, 167, 55, 63, 3, 211, 141, 90, 21, 125, 42, 43, 167, 170, 106, 216, 229, 147, 201, 187, 226, 204, 164, 162, 208, 172, 232, 19, 144, 137, 193, 151, 177, 209, 169, 32, 92, 166, 246, 93, 245, 63, 93, 200, 219, 163, 38, 105, 89, 156, 62, 21, 102, 211, 160, 60, 200, 158, 228, 123, 48, 165, 128, 238, 101, 0, 79, 172, 22, 219, 102, 39, 95, 95, 12, 6, 136, 208, 127, 136, 45, 53, 193, 50, 16, 227, 251, 200, 76, 23, 123, 51, 142, 218, 123, 122, 220, 217, 179, 135, 202, 190, 171, 118, 219, 147, 218, 224, 21, 94, 145, 194, 112, 193, 68, 98, 12, 54, 241, 153, 74, 83, 202, 56, 82, 88, 103, 37, 179, 110, 79, 43, 187, 73, 105, 145, 237, 236, 88, 88, 219, 79, 237, 241, 74, 128, 24, 112, 45, 239, 35, 222, 148, 97, 3, 81, 169, 32, 151, 206, 244, 204, 187, 91, 89, 226, 71, 11, 2, 180, 7, 61, 167, 53, 197, 169, 251, 212, 224, 164, 212, 167, 118, 37, 250, 79, 74, 28, 58, 218, 202, 194, 198, 3, 98, 33, 55, 71, 248, 171, 161, 193, 206, 229, 208, 167, 82, 218, 138, 0, 229, 4, 144, 56, 192, 153, 222, 117, 14, 175, 152, 192, 13, 233, 153, 141, 22, 13, 129, 85, 38, 196
			};

			SignerInfo si = GetSignerInfo (data);
			// default properties
			Assert.AreEqual (0, si.SignedAttributes.Count, "SignedAttributes");
			Assert.IsNotNull (si.Certificate, "Certificate");
			Assert.AreEqual (0, si.CounterSignerInfos.Count, "CounterSignerInfos");
			Assert.AreEqual ("sha256", si.DigestAlgorithm.FriendlyName, "DigestAlgorithm.FriendlyName");
			Assert.AreEqual ("2.16.840.1.101.3.4.2.1", si.DigestAlgorithm.Value, "DigestAlgorithm.Value");
			Assert.AreEqual (SubjectIdentifierType.SubjectKeyIdentifier, si.SignerIdentifier.Type, "SignerIdentifier.Type");
			Assert.AreEqual ("F1CC7184EBDA537140C1A57416017810DF133020", (string)si.SignerIdentifier.Value, "SignerIdentifier.Value");
			Assert.AreEqual (0, si.UnsignedAttributes.Count, "UnsignedAttributes");
			Assert.AreEqual (3, si.Version, "Version");
		}
	}
}
#endif

