package com.kickstarter.ui.adapters;

import android.support.annotation.LayoutRes;
import android.support.annotation.NonNull;
import android.view.View;

import com.kickstarter.R;
import com.kickstarter.libs.utils.ListUtils;
import com.kickstarter.models.Activity;
import com.kickstarter.ui.viewholders.EmptyActivityFeedViewHolder;
import com.kickstarter.ui.viewholders.EmptyViewHolder;
import com.kickstarter.ui.viewholders.FriendBackingViewHolder;
import com.kickstarter.ui.viewholders.FriendFollowViewHolder;
import com.kickstarter.ui.viewholders.KSViewHolder;
import com.kickstarter.ui.viewholders.ProjectStateChangedPositiveViewHolder;
import com.kickstarter.ui.viewholders.ProjectStateChangedViewHolder;
import com.kickstarter.ui.viewholders.ProjectUpdateViewHolder;

import java.util.Collections;
import java.util.List;

public final class ActivityFeedAdapter extends KSAdapter {
  private static final int SECTION_LOGGED_IN_EMPTY_VIEW = 0;
  private static final int SECTION_LOGGED_OUT_EMPTY_VIEW = 1;
  private static final int SECTION_ACTIVITIES_VIEW = 2;

  private final Delegate delegate;

  public interface Delegate extends FriendBackingViewHolder.Delegate, ProjectStateChangedPositiveViewHolder.Delegate,
    ProjectStateChangedViewHolder.Delegate, ProjectUpdateViewHolder.Delegate, EmptyActivityFeedViewHolder.Delegate {}

  public ActivityFeedAdapter(final @NonNull Delegate delegate) {
    this.delegate = delegate;

    insertSection(SECTION_LOGGED_IN_EMPTY_VIEW, Collections.emptyList());
    insertSection(SECTION_LOGGED_OUT_EMPTY_VIEW, Collections.emptyList());
    insertSection(SECTION_ACTIVITIES_VIEW, Collections.emptyList());
  }

  public void takeActivities(final @NonNull List<Activity> activities) {
    setSection(SECTION_ACTIVITIES_VIEW, activities);
    notifyDataSetChanged();
  }

  public void showLoggedInEmptyState(final boolean show) {
    setSection(SECTION_LOGGED_IN_EMPTY_VIEW, show ? Collections.singletonList(true) : ListUtils.empty());
    notifyDataSetChanged();
  }

  public void showLoggedOutEmptyState(final boolean show) {
    setSection(SECTION_LOGGED_OUT_EMPTY_VIEW, show ? Collections.singletonList(false) : ListUtils.empty());
    notifyDataSetChanged();
  }

  @Override
  protected @LayoutRes int layout(final @NonNull SectionRow sectionRow) {
    if (sectionRow.section() == SECTION_LOGGED_IN_EMPTY_VIEW) {
      return R.layout.empty_activity_feed_view;
    } else if (sectionRow.section() == SECTION_LOGGED_OUT_EMPTY_VIEW) {
      return R.layout.empty_activity_feed_view;
    } else if (sectionRow.section() == SECTION_ACTIVITIES_VIEW) {
      if (objectFromSectionRow(sectionRow) instanceof Activity) {
        final Activity activity = (Activity) objectFromSectionRow(sectionRow);
        switch (activity.category()) {
          case Activity.CATEGORY_BACKING:
            return R.layout.activity_friend_backing_view;
          case Activity.CATEGORY_FOLLOW:
            return R.layout.activity_friend_follow_view;
          case Activity.CATEGORY_FAILURE:
          case Activity.CATEGORY_CANCELLATION:
          case Activity.CATEGORY_SUSPENSION:
            return R.layout.activity_project_state_changed_view;
          case Activity.CATEGORY_LAUNCH:
          case Activity.CATEGORY_SUCCESS:
            return R.layout.activity_project_state_changed_positive_view;
          case Activity.CATEGORY_UPDATE:
            return R.layout.activity_project_update_view;
          default:
            return R.layout.empty_view;
        }
      }
    }
    return R.layout.empty_view;
  }


  @Override
  protected @NonNull KSViewHolder viewHolder(final @LayoutRes int layout, final @NonNull View view) {
    switch (layout) {
      case R.layout.activity_friend_backing_view:
        return new FriendBackingViewHolder(view, delegate);
      case R.layout.activity_friend_follow_view:
        return new FriendFollowViewHolder(view);
      case R.layout.activity_project_state_changed_view:
        return new ProjectStateChangedViewHolder(view, delegate);
      case R.layout.activity_project_state_changed_positive_view:
        return new ProjectStateChangedPositiveViewHolder(view, delegate);
      case R.layout.activity_project_update_view:
        return new ProjectUpdateViewHolder(view, delegate);
      case R.layout.empty_activity_feed_view:
        return new EmptyActivityFeedViewHolder(view, delegate);
      default:
        return new EmptyViewHolder(view);
    }
  }
}
