// Unless explicitly stated otherwise all files in this repository are licensed
// under the Apache License Version 2.0.
// This product includes software developed at Datadog (https://www.datadoghq.com/).
// Copyright 2016-2020 Datadog, Inc.

// +build functionaltests

package tests

import (
	"os"
	"syscall"
	"testing"

	"github.com/DataDog/datadog-agent/pkg/security/rules"
)

func TestUnlink(t *testing.T) {
	rule := &rules.RuleDefinition{
		ID:         "test_rule",
		Expression: `unlink.filename == "{{.Root}}/test-unlink" || unlink.filename == "{{.Root}}/test-unlinkat"`,
	}

	test, err := newTestModule(nil, []*rules.RuleDefinition{rule}, testOpts{})
	if err != nil {
		t.Fatal(err)
	}
	defer test.Close()

	testFile, testFilePtr, err := test.Path("test-unlink")
	if err != nil {
		t.Fatal(err)
	}

	f, err := os.Create(testFile)
	if err != nil {
		t.Fatal(err)
	}
	f.Close()
	defer os.Remove(testFile)

	inode := getInode(t, testFile)

	t.Run("unlink", func(t *testing.T) {
		if _, _, err := syscall.Syscall(syscall.SYS_UNLINK, uintptr(testFilePtr), 0, 0); err != 0 {
			t.Fatal(err)
		}

		event, _, err := test.GetEvent()
		if err != nil {
			t.Error(err)
		} else {
			if event.GetType() != "unlink" {
				t.Errorf("expected unlink event, got %s", event.GetType())
			}

			if inode != event.Unlink.Inode {
				t.Logf("expected inode %d, got %d", event.Unlink.Inode, inode)
			}

			testContainerPath(t, event, "unlink.container_path")
		}
	})

	testatFile, testatFilePtr, err := test.Path("test-unlinkat")
	if err != nil {
		t.Fatal(err)
	}

	f, err = os.Create(testatFile)
	if err != nil {
		t.Fatal(err)
	}
	f.Close()
	defer os.Remove(testatFile)

	inode = getInode(t, testatFile)

	t.Run("unlinkat", func(t *testing.T) {
		if _, _, err := syscall.Syscall(syscall.SYS_UNLINKAT, 0, uintptr(testatFilePtr), 0); err != 0 {
			t.Fatal(err)
		}

		event, _, err := test.GetEvent()
		if err != nil {
			t.Error(err)
		} else {
			if event.GetType() != "unlink" {
				t.Errorf("expected unlink event, got %s", event.GetType())
			}

			if inode != event.Unlink.Inode {
				t.Logf("expected inode %d, got %d", event.Unlink.Inode, inode)
			}

			testContainerPath(t, event, "unlink.container_path")
		}
	})
}
