//! Objects that are created as part of a ceremony and distributed.

use bytes::{Buf, BufMut};
use commonware_codec::{EncodeSize, FixedSize as _, Read, ReadExt as _, Write, varint::UInt};
use commonware_consensus::types::Epoch;
use commonware_cryptography::bls12381::primitives::{group, poly::Public, variant::MinSig};
use commonware_utils::quorum;
use tempo_dkg_onchain_artifacts::Ack;

/// The actual message that is sent over p2p.
#[derive(Clone, Debug, PartialEq)]
pub(super) struct Message {
    pub(super) epoch: Epoch,
    pub(super) payload: Payload,
}

impl Write for Message {
    fn write(&self, buf: &mut impl BufMut) {
        UInt(self.epoch).write(buf);
        self.payload.write(buf);
    }
}

impl Read for Message {
    type Cfg = u32;

    fn read_cfg(buf: &mut impl Buf, num_players: &u32) -> Result<Self, commonware_codec::Error> {
        let epoch = UInt::read(buf)?.into();
        let payload = Payload::read_cfg(buf, num_players)?;
        Ok(Self { epoch, payload })
    }
}

impl EncodeSize for Message {
    fn encode_size(&self) -> usize {
        UInt(self.epoch).encode_size() + self.payload.encode_size()
    }
}

/// The two kinds of messages that are being exchanged during a ceremony.
#[derive(Clone, Debug, PartialEq)]
pub(super) enum Payload {
    /// Message from dealer to player.
    ///
    /// Contains the commitment / public polynomial generated by the dealer for
    /// the a ceremony and corresponding share of the private key for the
    /// receiving player.
    Share(Share),

    /// Message from player to dealer.
    ///
    /// Acknowledges the receipt and verification of a [`Share`].
    ///
    /// Contains a signature of dealer and commitment to authenticate the
    /// acknowledged.
    Ack(Box<Ack>),
}

impl From<Ack> for Payload {
    fn from(value: Ack) -> Self {
        Self::Ack(value.into())
    }
}

impl From<Share> for Payload {
    fn from(value: Share) -> Self {
        Self::Share(value)
    }
}

impl Write for Payload {
    fn write(&self, buf: &mut impl BufMut) {
        match self {
            Self::Share(inner) => {
                buf.put_u8(SHARE_TAG);
                inner.write(buf);
            }
            Self::Ack(inner) => {
                buf.put_u8(ACK_TAG);
                inner.write(buf);
            }
        }
    }
}

const SHARE_TAG: u8 = 0;
const ACK_TAG: u8 = 1;

impl Read for Payload {
    type Cfg = u32;

    fn read_cfg(buf: &mut impl Buf, p: &u32) -> Result<Self, commonware_codec::Error> {
        let tag = u8::read(buf)?;
        let result = match tag {
            SHARE_TAG => Self::Share(Share::read_cfg(buf, p)?),
            ACK_TAG => Self::Ack(Ack::read(buf)?.into()),
            _ => return Err(commonware_codec::Error::InvalidEnum(tag)),
        };
        Ok(result)
    }
}

impl EncodeSize for Payload {
    fn encode_size(&self) -> usize {
        u8::SIZE
            + match self {
                Self::Share(inner) => inner.encode_size(),
                Self::Ack(inner) => inner.encode_size(),
            }
    }
}

/// A message from a dealer to a player.
///
/// Contains the commitment and one of the shares the dealer generated for the
/// current ceremony.
///
/// The receipt of this message is acknowledged by the player returning an
/// [`Ack`] to the dealer.
#[derive(Debug, Clone, PartialEq, Eq)]
pub(super) struct Share {
    /// The dealer's generated public polyonimal.
    pub(super) commitment: Public<MinSig>,
    /// The secret share generated for the player.
    pub(super) share: group::Share,
}

impl Write for Share {
    fn write(&self, buf: &mut impl BufMut) {
        self.commitment.write(buf);
        self.share.write(buf);
    }
}

impl EncodeSize for Share {
    fn encode_size(&self) -> usize {
        self.commitment.encode_size() + self.share.encode_size()
    }
}

impl Read for Share {
    type Cfg = u32;

    fn read_cfg(buf: &mut impl Buf, t: &u32) -> Result<Self, commonware_codec::Error> {
        let q = quorum(*t);
        Ok(Self {
            commitment: Public::<MinSig>::read_cfg(buf, &(q as usize))?,
            share: group::Share::read(buf)?,
        })
    }
}

#[cfg(test)]
mod tests {
    use commonware_codec::{Encode as _, Read as _};
    use commonware_cryptography::{
        PrivateKeyExt as _, Signer as _,
        bls12381::{dkg, primitives::variant::MinSig},
        ed25519::{PrivateKey, PublicKey},
    };
    use commonware_utils::{set::Ordered, union};
    use rand::{SeedableRng as _, rngs::StdRng};

    use crate::dkg::ceremony::ACK_NAMESPACE;

    use super::Ack;

    fn three_public_keys() -> Ordered<PublicKey> {
        vec![
            PrivateKey::from_seed(0).public_key(),
            PrivateKey::from_seed(1).public_key(),
            PrivateKey::from_seed(2).public_key(),
        ]
        .into()
    }

    #[test]
    fn roundtrip_ack() {
        let (_, commitment, _) = dkg::Dealer::<_, MinSig>::new(
            &mut StdRng::from_seed([0; 32]),
            None,
            three_public_keys(),
        );
        let player = PrivateKey::from_seed(0);
        let dealer = PrivateKey::from_seed(1).public_key();
        let ack = Ack::new(
            &union(b"test", ACK_NAMESPACE),
            player.clone(),
            player.public_key(),
            42,
            &dealer,
            &commitment,
        );

        let bytes = ack.encode();

        assert_eq!(Ack::read_cfg(&mut bytes.as_ref(), &()).unwrap(), ack);
    }
}
