use alloy::primitives::U256;
use std::marker::PhantomData;

use crate::{
    error::Result,
    storage::{Storable, StorageOps},
};

/// Sealed trait for compile-time slot identification.
///
/// This trait can only be implemented by types generated by the `#[contract]` macro.
/// Each implementation represents a unique storage slot with its U256 address.
///
/// The sealed pattern prevents external crates from implementing this trait,
/// ensuring type safety and compile-time slot guarantees.
mod sealed {
    use alloy::primitives::U256;

    pub trait SlotId {
        const SLOT: U256;
    }
}

pub use sealed::SlotId;

/// Type-safe wrapper for a single EVM storage slot.
///
/// # Type Parameters
///
/// - `T`: The Rust type stored in this slot (must implement `Storable<N>`)
/// - `Id`: Zero-sized marker type identifying the slot number (implements `SlotId`)
///
/// # Compile-Time Guarantees
///
/// - Different slots have distinct types: `Slot<T, Id1>` ≠ `Slot<T, Id2>`
/// - Slot number is encoded in the type system via `Id::SLOT`
///
/// # Example
///
/// ```ignore
/// // Generated by #[contract] macro:
/// pub struct NameSlotId;
/// impl SlotId for NameSlotId {
///     const SLOT: U256 = uint!(2_U256);
/// }
/// ```
///
/// The actual storage operations are handled by generated accessor methods
/// that read/write values using the `PrecompileStorageProvider` trait.
#[derive(Debug, Clone, Copy)]
pub struct Slot<T, Id: SlotId> {
    _phantom: PhantomData<(T, Id)>,
}

impl<T, Id: SlotId> Slot<T, Id> {
    /// Creates a new `Slot` marker.
    ///
    /// This is typically not called directly; instead, slots are declared
    /// as struct fields and accessed via macro-generated methods.
    #[inline]
    pub const fn new() -> Self {
        Self {
            _phantom: PhantomData,
        }
    }

    /// Returns the U256 storage slot number.
    ///
    /// Returns the slot number from the `SlotId` associated const.
    #[inline]
    pub const fn slot() -> U256 {
        Id::SLOT
    }

    /// Reads a value from storage at this slot.
    ///
    /// This method delegates to the `Storable::load` implementation,
    /// which may read one or more consecutive slots depending on `N`.
    ///
    /// # Example
    ///
    /// ```ignore
    /// type NamedSlot = Slot<String, NameSlotId>;
    /// let name = NamedSlot::read(&mut contract)?;
    /// ```
    #[inline]
    pub fn read<S: StorageOps, const N: usize>(storage: &mut S) -> Result<T>
    where
        T: Storable<N>,
    {
        T::load(storage, Id::SLOT)
    }

    /// Writes a value to storage at this slot.
    ///
    /// This method delegates to the `Storable::store` implementation,
    /// which may write one or more consecutive slots depending on `N`.
    ///
    /// # Example
    ///
    /// ```ignore
    /// type NamedSlot = Slot<String, NameSlotId>;
    /// NamedSlot::write(&mut contract, "MyToken".to_string())?;
    /// ```
    #[inline]
    pub fn write<S: StorageOps, const N: usize>(storage: &mut S, value: T) -> Result<()>
    where
        T: Storable<N>,
    {
        value.store(storage, Id::SLOT)
    }

    /// Deletes the value at this slot (sets all slots to zero).
    ///
    /// This method delegates to the `Storable::delete` implementation,
    /// which sets `N` consecutive slots to zero.
    ///
    /// # Example
    ///
    /// ```ignore
    /// type NamedSlot = Slot<String, NameSlotId>;
    /// NamedSlot::delete(&mut contract)?;
    /// ```
    #[inline]
    pub fn delete<S: StorageOps, const N: usize>(storage: &mut S) -> Result<()>
    where
        T: Storable<N>,
    {
        T::delete(storage, Id::SLOT)
    }

    /// Reads a value from a field within a struct at a runtime base slot.
    ///
    /// This enables accessing non-packed fields within structs when you have
    /// the struct's base slot at runtime and know the field's offset.
    ///
    /// # Example
    ///
    /// ```ignore
    /// // For: mapping(bytes32 => Orderbook) books, where Orderbook.base is at field offset 0
    /// let orderbook_base = mapping_slot(pair_key, BooksSlot::SLOT);
    /// let base_address = Slot::<Address, DummySlot>::read_at_offset(
    ///     &mut storage,
    ///     orderbook_base,
    ///     0  // field offset
    /// )?;
    /// ```
    #[inline]
    pub fn read_at_offset<S: StorageOps, const N: usize>(
        storage: &mut S,
        struct_base_slot: U256,
        field_offset_slots: usize,
    ) -> Result<T>
    where
        T: Storable<N>,
    {
        let slot = struct_base_slot + U256::from(field_offset_slots);
        T::load(storage, slot)
    }

    /// Writes a value to a field within a struct at a runtime base slot.
    ///
    /// # Example
    ///
    /// ```ignore
    /// let orderbook_base = mapping_slot(pair_key, BooksSlot::SLOT);
    /// Slot::<Address, DummySlot>::write_at_offset(
    ///     &mut storage,
    ///     orderbook_base,
    ///     0,  // field offset
    ///     base_address
    /// )?;
    /// ```
    #[inline]
    pub fn write_at_offset<S: StorageOps, const N: usize>(
        storage: &mut S,
        struct_base_slot: U256,
        field_offset_slots: usize,
        value: T,
    ) -> Result<()>
    where
        T: Storable<N>,
    {
        let slot = struct_base_slot + U256::from(field_offset_slots);
        value.store(storage, slot)
    }

    /// Deletes a field within a struct at a runtime base slot (sets slots to zero).
    ///
    /// # Example
    ///
    /// ```ignore
    /// let orderbook_base = mapping_slot(pair_key, BooksSlot::SLOT);
    /// Slot::<Address, DummySlot>::delete_at_offset(
    ///     &mut storage,
    ///     orderbook_base,
    ///     0  // field offset
    /// )?;
    /// ```
    #[inline]
    pub fn delete_at_offset<S: StorageOps, const N: usize>(
        storage: &mut S,
        struct_base_slot: U256,
        field_offset_slots: usize,
    ) -> Result<()>
    where
        T: Storable<N>,
    {
        let slot = struct_base_slot + U256::from(field_offset_slots);
        T::delete(storage, slot)
    }

    /// Reads a packed field from storage at a given base slot.
    ///
    /// Use this method when the field shares a storage slot with other fields (i.e., is packed).
    /// For fields that occupy their own slot(s), use `read_at_offset` instead.
    #[inline]
    pub fn read_at_offset_packed<S: StorageOps>(
        storage: &mut S,
        struct_base_slot: U256,
        field_offset_slots: usize,
        field_offset_bytes: usize,
        field_size_bytes: usize,
    ) -> Result<T>
    where
        T: Storable<1>,
    {
        crate::storage::packing::read_packed_at(
            storage,
            struct_base_slot,
            field_offset_slots,
            field_offset_bytes,
            field_size_bytes,
        )
    }

    /// Writes a packed field to storage at a given base slot.
    ///
    /// Use this method when the field shares a storage slot with other fields (i.e., is packed).
    /// This correctly preserves other fields in the same slot.
    /// For fields that occupy their own slot(s), use `write_at_offset` instead.
    #[inline]
    pub fn write_at_offset_packed<S: StorageOps>(
        storage: &mut S,
        struct_base_slot: U256,
        field_offset_slots: usize,
        field_offset_bytes: usize,
        field_size_bytes: usize,
        value: T,
    ) -> Result<()>
    where
        T: Storable<1>,
    {
        crate::storage::packing::write_packed_at(
            storage,
            struct_base_slot,
            field_offset_slots,
            field_offset_bytes,
            field_size_bytes,
            &value,
        )
    }

    /// Deletes a packed field in storage at a given base slot (sets bytes to zero).
    ///
    /// Use this method when the field shares a storage slot with other fields (i.e., is packed).
    /// This correctly preserves other fields in the same slot.
    /// For fields that occupy their own slot(s), use `delete_at_offset` instead.
    #[inline]
    pub fn delete_at_offset_packed<S: StorageOps>(
        storage: &mut S,
        struct_base_slot: U256,
        field_offset_slots: usize,
        field_offset_bytes: usize,
        field_size_bytes: usize,
    ) -> Result<()>
    where
        T: Storable<1>,
    {
        crate::storage::packing::clear_packed_at(
            storage,
            struct_base_slot,
            field_offset_slots,
            field_offset_bytes,
            field_size_bytes,
        )
    }
}

impl<T, Id: SlotId> Default for Slot<T, Id> {
    fn default() -> Self {
        Self::new()
    }
}

#[cfg(test)]
mod tests {
    use super::*;
    use crate::storage::{
        DummySlot, PrecompileStorageProvider, hashmap::HashMapStorageProvider, mapping_slot,
    };
    use alloy::primitives::{Address, B256};
    use proptest::prelude::*;

    // Test helper that implements StorageOps
    struct TestContract<'a, S> {
        address: Address,
        storage: &'a mut S,
    }

    impl<'a, S: PrecompileStorageProvider> StorageOps for TestContract<'a, S> {
        fn sstore(&mut self, slot: U256, value: U256) -> Result<()> {
            self.storage.sstore(self.address, slot, value)
        }

        fn sload(&mut self, slot: U256) -> Result<U256> {
            self.storage.sload(self.address, slot)
        }
    }

    /// Helper to create a test contract with fresh storage.
    fn setup_test_contract<'a>(
        storage: &'a mut HashMapStorageProvider,
    ) -> TestContract<'a, HashMapStorageProvider> {
        TestContract {
            address: Address::random(),
            storage,
        }
    }

    // Test SlotId implementations
    struct TestSlot0;
    impl SlotId for TestSlot0 {
        const SLOT: U256 = U256::ZERO;
    }

    struct TestSlot1;
    impl SlotId for TestSlot1 {
        const SLOT: U256 = U256::ONE;
    }

    struct TestSlot2;
    impl SlotId for TestSlot2 {
        const SLOT: U256 = U256::from_limbs([2, 0, 0, 0]);
    }

    struct TestSlotMax;
    impl SlotId for TestSlotMax {
        const SLOT: U256 = U256::MAX;
    }

    // Property test strategies
    fn arb_address() -> impl Strategy<Value = Address> {
        any::<[u8; 20]>().prop_map(Address::from)
    }

    fn arb_u256() -> impl Strategy<Value = U256> {
        any::<[u64; 4]>().prop_map(U256::from_limbs)
    }

    #[test]
    fn test_slot_is_zero_sized() {
        assert_eq!(std::mem::size_of::<Slot<U256, TestSlot0>>(), 0);
        assert_eq!(std::mem::size_of::<Slot<Address, TestSlot1>>(), 0);
        assert_eq!(std::mem::size_of::<Slot<bool, TestSlotMax>>(), 0);
    }

    #[test]
    fn test_slot_creation() {
        let _slot_u256: Slot<U256, TestSlot0> = Slot::new();
        let _slot_addr: Slot<Address, TestSlot1> = Slot::new();
        let _slot_default: Slot<bool, TestSlotMax> = Slot::default();
    }

    #[test]
    fn test_slot_number_extraction() {
        assert_eq!(Slot::<U256, TestSlot0>::slot(), U256::ZERO);
        assert_eq!(Slot::<Address, TestSlot1>::slot(), U256::ONE);
        assert_eq!(Slot::<bool, TestSlotMax>::slot(), U256::MAX);
    }

    #[test]
    fn test_slot_edge_case_zero() {
        // Explicit test for U256::ZERO slot
        assert_eq!(Slot::<U256, TestSlot0>::slot(), U256::ZERO);

        let mut storage = HashMapStorageProvider::new(1);
        let mut contract = setup_test_contract(&mut storage);
        type ZeroSlot = Slot<U256, TestSlot0>;
        let value = U256::random();

        _ = ZeroSlot::write(&mut contract, value);
        let loaded = ZeroSlot::read(&mut contract).unwrap();
        assert_eq!(loaded, value);
    }

    #[test]
    fn test_slot_edge_case_max() {
        // Explicit test for U256::MAX slot
        type MaxSlot = Slot<U256, TestSlotMax>;
        assert_eq!(MaxSlot::slot(), U256::MAX);

        let mut storage = HashMapStorageProvider::new(1);
        let mut contract = setup_test_contract(&mut storage);

        let value = U256::random();
        _ = MaxSlot::write(&mut contract, value);
        let loaded = MaxSlot::read(&mut contract).unwrap();
        assert_eq!(loaded, value);
    }

    #[test]
    fn test_slot_read_write_u256() {
        let mut storage = HashMapStorageProvider::new(1);
        let mut contract = setup_test_contract(&mut storage);
        type TestSlot = Slot<U256, TestSlot1>;
        let test_value = U256::random();

        // Write using new API
        _ = TestSlot::write(&mut contract, test_value);

        // Read using new API
        let loaded = TestSlot::read(&mut contract).unwrap();
        assert_eq!(loaded, test_value);

        // Verify it actually wrote to slot 1
        let raw = contract.storage.sload(contract.address, U256::ONE);
        assert_eq!(raw, Ok(test_value));
    }

    #[test]
    fn test_slot_read_write_address() {
        let mut storage = HashMapStorageProvider::new(1);
        let mut contract = setup_test_contract(&mut storage);
        let test_addr = Address::random();
        type OwnerSlot = Slot<Address, TestSlot1>;

        // Write
        _ = OwnerSlot::write(&mut contract, test_addr);

        // Read
        let loaded = OwnerSlot::read(&mut contract).unwrap();
        assert_eq!(loaded, test_addr);
    }

    #[test]
    fn test_slot_read_write_bool() {
        let mut storage = HashMapStorageProvider::new(1);
        let mut contract = setup_test_contract(&mut storage);
        type PausedSlot = Slot<bool, TestSlot1>;

        // Write true
        _ = PausedSlot::write(&mut contract, true);
        assert!(PausedSlot::read(&mut contract).unwrap());

        // Write false
        _ = PausedSlot::write(&mut contract, false);
        assert!(!PausedSlot::read(&mut contract).unwrap());
    }

    #[test]
    fn test_slot_read_write_string() {
        let mut storage = HashMapStorageProvider::new(1);
        let mut contract = setup_test_contract(&mut storage);
        type NamedSlot = Slot<String, TestSlot1>;

        let test_name = "TestToken";
        _ = NamedSlot::write(&mut contract, test_name.to_string());

        let loaded = NamedSlot::read(&mut contract).unwrap();
        assert_eq!(loaded, test_name);
    }

    #[test]
    fn test_slot_default_value_is_zero() {
        let mut storage = HashMapStorageProvider::new(1);
        let mut contract = setup_test_contract(&mut storage);
        type UninitializedSlot = Slot<U256, TestSlot1>;

        // Reading uninitialized storage should return zero
        let value = UninitializedSlot::read(&mut contract).unwrap();
        assert_eq!(value, U256::ZERO);
    }

    #[test]
    fn test_slot_overwrite() {
        let mut storage = HashMapStorageProvider::new(1);
        let mut contract = setup_test_contract(&mut storage);
        type CounterSlot = Slot<u64, TestSlot1>;

        // Write initial value
        _ = CounterSlot::write(&mut contract, 100);
        assert_eq!(CounterSlot::read(&mut contract), Ok(100));

        // Overwrite with new value
        _ = CounterSlot::write(&mut contract, 200);
        assert_eq!(CounterSlot::read(&mut contract), Ok(200));
    }

    proptest! {
        #![proptest_config(ProptestConfig::with_cases(500))]

        #[test]
        fn proptest_slot_read_write_u256(value in arb_u256()) {
            let mut storage = HashMapStorageProvider::new(1);
            let mut contract = setup_test_contract(&mut storage);
            type TestSlot = Slot<U256, TestSlot2>;

            // Write and read back
            TestSlot::write(&mut contract, value)?;
            let loaded = TestSlot::read(&mut contract)?;
            prop_assert_eq!(loaded, value, "roundtrip failed");

            // Delete and verify
            TestSlot::delete(&mut contract)?;
            let after_delete = TestSlot::read(&mut contract)?;
            prop_assert_eq!(after_delete, U256::ZERO, "not zero after delete");
        }

        #[test]
        fn proptest_slot_read_write_address(addr_value in arb_address()) {
            let mut storage = HashMapStorageProvider::new(1);
            let mut contract = setup_test_contract(&mut storage);
            type TestSlot = Slot<Address, TestSlot1>;

            // Write and read back
            TestSlot::write(&mut contract, addr_value)?;
            let loaded = TestSlot::read(&mut contract)?;
            prop_assert_eq!(loaded, addr_value, "address roundtrip failed");
        }

        #[test]
        fn proptest_slot_isolation(value1 in arb_u256(), value2 in arb_u256()) {
            let mut storage = HashMapStorageProvider::new(1);
            let mut contract = setup_test_contract(&mut storage);
            type Slot1 = Slot<U256, TestSlot1>;
            type Slot2 = Slot<U256, TestSlot2>;

            Slot1::write(&mut contract, value1)?;
            Slot2::write(&mut contract, value2)?;

            // Verify both slots retain their independent values
            let loaded1 = Slot1::read(&mut contract)?;
            let loaded2 = Slot2::read(&mut contract)?;

            prop_assert_eq!(loaded1, value1, "slot 1 value changed");
            prop_assert_eq!(loaded2, value2, "slot 2 value changed");

            // Delete slot 1, verify slot 2 unaffected
            Slot1::delete(&mut contract)?;
            let after_delete1 = Slot1::read(&mut contract)?;
            let after_delete2 = Slot2::read(&mut contract)?;

            prop_assert_eq!(after_delete1, U256::ZERO, "slot 1 not deleted");
            prop_assert_eq!(after_delete2, value2, "slot 2 affected by slot 1 delete");
        }
    }

    // -- RUNTIME SLOT OFFSET TESTS --------------------------------------------

    #[test]
    fn test_slot_at_offset() -> eyre::Result<()> {
        let mut storage = HashMapStorageProvider::new(1);
        let mut contract = setup_test_contract(&mut storage);

        let pair_key: B256 = U256::from(0xabcd).into();
        let orderbook_base_slot = mapping_slot(pair_key, DummySlot::SLOT);

        let base_address = Address::random();

        // Write to orderbook.base using runtime offset
        Slot::<Address, DummySlot>::write_at_offset(
            &mut contract,
            orderbook_base_slot,
            0, // base field at offset 0
            base_address,
        )?;

        // Read back
        let read_address =
            Slot::<Address, DummySlot>::read_at_offset(&mut contract, orderbook_base_slot, 0)?;

        assert_eq!(read_address, base_address);

        // Delete
        Slot::<Address, DummySlot>::delete_at_offset(&mut contract, orderbook_base_slot, 0)?;

        let deleted =
            Slot::<Address, DummySlot>::read_at_offset(&mut contract, orderbook_base_slot, 0)?;

        assert_eq!(deleted, Address::ZERO);

        Ok(())
    }

    #[test]
    fn test_slot_at_offset_multi_slot_value() -> eyre::Result<()> {
        let mut storage = HashMapStorageProvider::new(1);
        let mut contract = setup_test_contract(&mut storage);

        let struct_key: B256 = U256::from(0xabcd).into();
        let struct_base = mapping_slot(struct_key, DummySlot::SLOT);

        // Test writing a multi-slot value like a String at offset 2
        let test_string = "Hello, Orderbook!".to_string();

        Slot::<String, DummySlot>::write_at_offset(
            &mut contract,
            struct_base,
            2,
            test_string.clone(),
        )?;

        let read_string = Slot::<String, DummySlot>::read_at_offset(&mut contract, struct_base, 2)?;

        assert_eq!(read_string, test_string);

        Ok(())
    }

    #[test]
    fn test_multiple_primitive_fields() -> eyre::Result<()> {
        let mut storage = HashMapStorageProvider::new(1);
        let mut contract = setup_test_contract(&mut storage);

        let key: B256 = U256::from(0x9999).into();
        let base = mapping_slot(key, DummySlot::SLOT);

        // Simulate different primitive fields at different offsets
        let field_0 = Address::random();
        let field_1: u64 = 12345;
        let field_2 = U256::from(999999);

        Slot::<Address, DummySlot>::write_at_offset(&mut contract, base, 0, field_0)?;
        Slot::<u64, DummySlot>::write_at_offset(&mut contract, base, 1, field_1)?;
        Slot::<U256, DummySlot>::write_at_offset(&mut contract, base, 2, field_2)?;

        // Verify independence
        let read_0 = Slot::<Address, DummySlot>::read_at_offset(&mut contract, base, 0)?;
        let read_1 = Slot::<u64, DummySlot>::read_at_offset(&mut contract, base, 1)?;
        let read_2 = Slot::<U256, DummySlot>::read_at_offset(&mut contract, base, 2)?;

        assert_eq!(read_0, field_0);
        assert_eq!(read_1, field_1);
        assert_eq!(read_2, field_2);

        Ok(())
    }
}
