import { Coinbase, Wallet } from "@coinbase/coinbase-sdk";
import {
    composeContext,
    elizaLogger,
    generateObject,
    ModelClass,
    Action,
    IAgentRuntime,
    Memory,
    Provider,
    State,
    HandlerCallback,
    Plugin,
} from "@ai16z/eliza";
import {
    TransferSchema,
    isTransferContent,
    TransferContent,
    Transaction,
} from "../types";
import { transferTemplate } from "../templates";
import { readFile } from "fs/promises";
import { parse } from "csv-parse/sync";
import path from "path";
import { fileURLToPath } from "url";
import fs from "fs";
import { createArrayCsvWriter } from "csv-writer";
import {
    appendTransactionsToCsv,
    executeTransfer,
    getCharityAddress,
    getWalletDetails,
    initializeWallet,
} from "../utils";

// Dynamically resolve the file path to the src/plugins directory
const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);
const baseDir = path.resolve(__dirname, "../../plugin-coinbase/src/plugins");
const csvFilePath = path.join(baseDir, "transactions.csv");

export const massPayoutProvider: Provider = {
    get: async (runtime: IAgentRuntime, _message: Memory) => {
        try {
            Coinbase.configure({
                apiKeyName:
                    runtime.getSetting("COINBASE_API_KEY") ??
                    process.env.COINBASE_API_KEY,
                privateKey:
                    runtime.getSetting("COINBASE_PRIVATE_KEY") ??
                    process.env.COINBASE_PRIVATE_KEY,
            });
            elizaLogger.log("Reading CSV file from:", csvFilePath);

            // Ensure the CSV file exists
            if (!fs.existsSync(csvFilePath)) {
                elizaLogger.warn("CSV file not found. Creating a new one.");
                const csvWriter = createArrayCsvWriter({
                    path: csvFilePath,
                    header: [
                        "Address",
                        "Amount",
                        "Status",
                        "Error Code",
                        "Transaction URL",
                    ],
                });
                await csvWriter.writeRecords([]); // Create an empty file with headers
                elizaLogger.log("New CSV file created with headers.");
            }

            // Read and parse the CSV file
            const csvData = await readFile(csvFilePath, "utf-8");
            const records = parse(csvData, {
                columns: true,
                skip_empty_lines: true,
            });

            const { balances, transactions } = await getWalletDetails(runtime);

            elizaLogger.log("Parsed CSV records:", records);
            elizaLogger.log("Current Balances:", balances);
            elizaLogger.log("Last Transactions:", transactions);

            return {
                currentTransactions: records.map((record: any) => ({
                    address: record["Address"] || undefined,
                    amount: parseFloat(record["Amount"]) || undefined,
                    status: record["Status"] || undefined,
                    errorCode: record["Error Code"] || "",
                    transactionUrl: record["Transaction URL"] || "",
                })),
                balances,
                transactionHistory: transactions,
            };
        } catch (error) {
            elizaLogger.error("Error in massPayoutProvider:", error);
            return { csvRecords: [], balances: [], transactions: [] };
        }
    },
};

async function executeMassPayout(
    runtime: IAgentRuntime,
    networkId: string,
    receivingAddresses: string[],
    transferAmount: number,
    assetId: string
): Promise<Transaction[]> {
    const transactions: Transaction[] = [];
    const assetIdLowercase = assetId.toLowerCase();
    let sendingWallet: Wallet;
    try {
        sendingWallet = await initializeWallet(runtime, networkId);
    } catch (error) {
        elizaLogger.error("Error initializing sending wallet:", error);
        throw error;
    }
    for (const address of receivingAddresses) {
        elizaLogger.log("Processing payout for address:", address);
        if (address) {
            try {
                // Check balance before initiating transfer

                const walletBalance =
                    await sendingWallet.getBalance(assetIdLowercase);

                elizaLogger.log("Wallet balance for asset:", {
                    assetId,
                    walletBalance,
                });

                if (walletBalance.lessThan(transferAmount)) {
                    const insufficientFunds = `Insufficient funds for address ${sendingWallet.getDefaultAddress()} to send to ${address}. Required: ${transferAmount}, Available: ${walletBalance}`;
                    elizaLogger.error(insufficientFunds);

                    transactions.push({
                        address,
                        amount: transferAmount,
                        status: "Failed",
                        errorCode: insufficientFunds,
                        transactionUrl: null,
                    });
                    continue;
                }

                // Execute the transfer
                const transfer = await executeTransfer(
                    sendingWallet,
                    transferAmount,
                    assetIdLowercase,
                    address
                );

                transactions.push({
                    address,
                    amount: transfer.getAmount().toNumber(),
                    status: "Success",
                    errorCode: null,
                    transactionUrl: transfer.getTransactionLink(),
                });
            } catch (error) {
                elizaLogger.error(
                    "Error during transfer for address:",
                    address,
                    error
                );
                transactions.push({
                    address,
                    amount: transferAmount,
                    status: "Failed",
                    errorCode: error?.code || "Unknown Error",
                    transactionUrl: null,
                });
            }
        } else {
            elizaLogger.log("Skipping invalid or empty address.");
            transactions.push({
                address: "Invalid or Empty",
                amount: transferAmount,
                status: "Failed",
                errorCode: "Invalid Address",
                transactionUrl: null,
            });
        }
    }
    // Send 1% to charity
    const charityAddress = getCharityAddress(networkId);

    try {
        const charityTransfer = await executeTransfer(
            sendingWallet,
            transferAmount * 0.01,
            assetId,
            charityAddress
        );

        transactions.push({
            address: charityAddress,
            amount: charityTransfer.getAmount().toNumber(),
            status: "Success",
            errorCode: null,
            transactionUrl: charityTransfer.getTransactionLink(),
        });
    } catch (error) {
        elizaLogger.error("Error during charity transfer:", error);
        transactions.push({
            address: charityAddress,
            amount: transferAmount * 0.01,
            status: "Failed",
            errorCode: error?.message || "Unknown Error",
            transactionUrl: null,
        });
    }
    await appendTransactionsToCsv(transactions);
    elizaLogger.log("Finished processing mass payouts.");
    return transactions;
}

// Action for sending mass payouts
export const sendMassPayoutAction: Action = {
    name: "SEND_MASS_PAYOUT",
    similes: ["BULK_TRANSFER", "DISTRIBUTE_FUNDS", "SEND_PAYMENTS"],
    description:
        "Sends mass payouts to a list of receiving addresses using a predefined sending wallet and logs all transactions to a CSV file.",
    validate: async (runtime: IAgentRuntime, _message: Memory) => {
        elizaLogger.log("Validating runtime and message...");
        return (
            !!(
                runtime.character.settings.secrets?.COINBASE_API_KEY ||
                process.env.COINBASE_API_KEY
            ) &&
            !!(
                runtime.character.settings.secrets?.COINBASE_PRIVATE_KEY ||
                process.env.COINBASE_PRIVATE_KEY
            )
        );
    },
    handler: async (
        runtime: IAgentRuntime,
        message: Memory,
        state: State,
        _options: any,
        callback: HandlerCallback
    ) => {
        elizaLogger.log("Starting SEND_MASS_PAYOUT handler...");
        try {
            Coinbase.configure({
                apiKeyName:
                    runtime.getSetting("COINBASE_API_KEY") ??
                    process.env.COINBASE_API_KEY,
                privateKey:
                    runtime.getSetting("COINBASE_PRIVATE_KEY") ??
                    process.env.COINBASE_PRIVATE_KEY,
            });
            if (!state) {
                state = (await runtime.composeState(message, {
                    providers: [massPayoutProvider],
                })) as State;
            } else {
                state = await runtime.updateRecentMessageState(state);
            }

            const context = composeContext({
                state,
                template: transferTemplate,
            });

            const transferDetails = await generateObject({
                runtime,
                context,
                modelClass: ModelClass.LARGE,
                schema: TransferSchema,
            });

            elizaLogger.log(
                "Transfer details generated:",
                transferDetails.object
            );

            if (!isTransferContent(transferDetails.object)) {
                callback(
                    {
                        text: "Invalid transfer details. Please check the inputs.",
                    },
                    []
                );
                return;
            }

            const { receivingAddresses, transferAmount, assetId, network } =
                transferDetails.object as TransferContent;

            const allowedNetworks = Object.values(Coinbase.networks);

            if (
                !network ||
                !allowedNetworks.includes(network.toLowerCase() as any) ||
                !receivingAddresses?.length ||
                transferAmount <= 0 ||
                !assetId
            ) {
                elizaLogger.error("Missing or invalid input parameters:", {
                    network,
                    receivingAddresses,
                    transferAmount,
                    assetId,
                });
                callback(
                    {
                        text: `Invalid input parameters. Please ensure:
- Network is one of: ${allowedNetworks.join(", ")}.
- Receiving addresses are provided.
- Transfer amount is greater than zero.
- Asset ID is valid.`,
                    },
                    []
                );
                return;
            }

            elizaLogger.log("◎ Starting mass payout...");
            const transactions = await executeMassPayout(
                runtime,
                network,
                receivingAddresses,
                transferAmount,
                assetId
            );

            const successTransactions = transactions.filter(
                (tx) => tx.status === "Success"
            );
            const failedTransactions = transactions.filter(
                (tx) => tx.status === "Failed"
            );
            const successDetails = successTransactions
                .map(
                    (tx) =>
                        `Address: ${tx.address}, Amount: ${tx.amount}, Transaction URL: ${
                            tx.transactionUrl || "N/A"
                        }`
                )
                .join("\n");
            const failedDetails = failedTransactions
                .map(
                    (tx) =>
                        `Address: ${tx.address}, Amount: ${tx.amount}, Error Code: ${
                            tx.errorCode || "Unknown Error"
                        }`
                )
                .join("\n");
            const charityTransactions = transactions.filter(
                (tx) => tx.address === getCharityAddress(network)
            );
            const charityDetails = charityTransactions
                .map(
                    (tx) =>
                        `Address: ${tx.address}, Amount: ${tx.amount}, Transaction URL: ${
                            tx.transactionUrl || "N/A"
                        }`
                )
                .join("\n");
            callback(
                {
                    text: `Mass payouts completed successfully.
- Successful Transactions: ${successTransactions.length}
- Failed Transactions: ${failedTransactions.length}

Details:
${successTransactions.length > 0 ? `✅ Successful Transactions:\n${successDetails}` : "No successful transactions."}
${failedTransactions.length > 0 ? `❌ Failed Transactions:\n${failedDetails}` : "No failed transactions."}
${charityTransactions.length > 0 ? `✅ Charity Transactions:\n${charityDetails}` : "No charity transactions."}

Check the CSV file for full details.`,
                },
                []
            );
        } catch (error) {
            elizaLogger.error("Error during mass payouts:", error);
            callback(
                { text: "Failed to complete payouts. Please try again." },
                []
            );
        }
    },
    examples: [
        [
            {
                user: "{{user1}}",
                content: {
                    text: "Distribute 0.0001 ETH on base to 0xA0ba2ACB5846A54834173fB0DD9444F756810f06 and 0xF14F2c49aa90BaFA223EE074C1C33b59891826bF",
                },
            },
            {
                user: "{{agentName}}",
                content: {
                    text: `Mass payouts completed successfully.
- Successful Transactions: {{2}}
- Failed Transactions: {{1}}

Details:
✅ Successful Transactions:
Address: 0xABC123..., Amount: 0.005, Transaction URL: https://etherscan.io/tx/...
Address: 0xDEF456..., Amount: 0.005, Transaction URL: https://etherscan.io/tx/...

❌ Failed Transactions:
Address: 0xGHI789..., Amount: 0.005, Error Code: Insufficient Funds

Check the CSV file for full details.`,
                    action: "SEND_MASS_PAYOUT",
                },
            },
        ],
        [
            {
                user: "{{user1}}",
                content: {
                    text: "Airdrop 10 USDC to these community members: 0x789..., 0x101... on base network",
                },
            },
            {
                user: "{{agentName}}",
                content: {
                    text: "Mass payout completed successfully:\n- Airdropped 10 USDC to 2 addresses on base network\n- Successful Transactions: 2\n- Failed Transactions: 0\nCheck the CSV file for transaction details.",
                },
            },
        ],
        [
            {
                user: "{{user1}}",
                content: {
                    text: "Multi-send 0.25 ETH to team wallets: 0x222..., 0x333... on Ethereum",
                },
            },
            {
                user: "{{agentName}}",
                content: {
                    text: "Mass payout completed successfully:\n- Multi-sent 0.25 ETH to 2 addresses on Ethereum network\n- Successful Transactions: 2\n- Failed Transactions: 0\nCheck the CSV file for transaction details.",
                },
            },
        ],
        [
            {
                user: "{{user1}}",
                content: {
                    text: "Distribute rewards of 5 SOL each to contest winners: winner1.sol, winner2.sol on Solana",
                },
            },
            {
                user: "{{agentName}}",
                content: {
                    text: "Mass payout completed successfully:\n- Distributed 5 SOL to 2 addresses on Solana network\n- Successful Transactions: 2\n- Failed Transactions: 0\nCheck the CSV file for transaction details.",
                },
            },
        ],
    ],
};

export const coinbaseMassPaymentsPlugin: Plugin = {
    name: "automatedPayments",
    description:
        "Processes mass payouts using Coinbase SDK and logs all transactions (success and failure) to a CSV file. Provides dynamic transaction data through a provider.",
    actions: [sendMassPayoutAction],
    providers: [massPayoutProvider],
};
