// Copyright (c) Microsoft.  All Rights Reserved.  Licensed under the Apache License, Version 2.0.  See License.txt in the project root for license information.

using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.CodeAnalysis.Scripting;
using Microsoft.CodeAnalysis.Scripting.CSharp;
using Roslyn.Test.Utilities;
using Roslyn.Utilities;
using Xunit;

namespace Microsoft.CodeAnalysis.Scripting.CSharp.Test
{
    public class ScriptTests : TestBase
    {
        [Fact]
        public void TestCreateScript()
        {
            var script = CSharpScript.Create("1 + 2");
            Assert.Equal("1 + 2", script.Code);
        }

        [Fact]
        public void TestGetCompilation()
        {
            var script = CSharpScript.Create("1 + 2");
            script.RunAsync(new ScriptTests());
            var compilation = script.GetCompilation();
            Assert.Equal(script.Code, compilation.SyntaxTrees.First().GetText().ToString());
        }

        [Fact]
        public void TestCreateScriptDelegate()
        {
            // create a delegate for the entire script
            var script = CSharpScript.Create("1 + 2");
            var fn = script.CreateDelegate();
            var value = fn();
            Assert.Equal(3, value.Result);
        }

        [Fact]
        public void TestRunScript()
        {
            var result = CSharpScript.RunAsync("1 + 2");
            Assert.Equal(3, result.ReturnValue.Result);
        }

        [Fact]
        public void TestCreateAndRunScript()
        {
            var script = CSharpScript.Create("1 + 2");
            var result = script.RunAsync();
            Assert.Same(script, result.Script);
            Assert.Equal(3, result.ReturnValue.Result);
        }

        [Fact]
        public void TestEvalScript()
        {
            var value = CSharpScript.EvaluateAsync("1 + 2");
            Assert.Equal(3, value.Result);
        }

        [Fact]
        public void TestRunScriptWithSpecifiedReturnType()
        {
            var result = CSharpScript.RunAsync("1 + 2");
            Assert.Equal(3, result.ReturnValue.Result);
        }

        [Fact]
        public void TestRunVoidScript()
        {
            var result = CSharpScript.RunAsync("System.Console.WriteLine(0);");
            var task = result.ReturnValue;
            Assert.Null(task.Result);
        }

        [Fact(Skip = "https://github.com/dotnet/roslyn/issues/170")]
        public void TestRunDynamicVoidScriptWithTerminatingSemicolon()
        {
            var result = CSharpScript.RunAsync(@"
class SomeClass
{
    public void Do()
    {
    }
}
dynamic d = new SomeClass();
d.Do();"
, ScriptOptions.Default.WithReferences(MscorlibRef, SystemRef, SystemCoreRef, CSharpRef));
        }

        [Fact(Skip = "https://github.com/dotnet/roslyn/issues/170")]
        public void TestRunDynamicVoidScriptWithoutTerminatingSemicolon()
        {
            var result = CSharpScript.RunAsync(@"
class SomeClass
{
    public void Do()
    {
    }
}
dynamic d = new SomeClass();
d.Do()"
, ScriptOptions.Default.WithReferences(MscorlibRef, SystemRef, SystemCoreRef, CSharpRef));
        }

        public class Globals
        {
            public int X;
            public int Y;
        }

        [Fact]
        public void TestRunScriptWithGlobals()
        {
            var result = CSharpScript.RunAsync("X + Y", new Globals { X = 1, Y = 2 });
            Assert.Equal(3, result.ReturnValue.Result);
        }

        [Fact]
        public void TestRunCreatedScriptWithExpectedGlobals()
        {
            var script = CSharpScript.Create("X + Y").WithGlobalsType(typeof(Globals));
            var result = script.RunAsync(new Globals { X = 1, Y = 2 });
            Assert.Equal(3, result.ReturnValue.Result);
            Assert.Same(script, result.Script);
        }

        [Fact]
        public void TestRunCreatedScriptWithUnexpectedGlobals()
        {
            var script = CSharpScript.Create("X + Y");
            var result = script.RunAsync(new Globals { X = 1, Y = 2 });
            Assert.Equal(3, result.ReturnValue.Result);

            // the end state of running the script should be based on a different script instance because of the globals
            // not matching the original script definition.
            Assert.NotSame(script, result.Script);
        }

        [Fact]
        public void TestRunScriptWithScriptState()
        {
            // run a script using another scripts end state as the starting state (globals)
            var result = CSharpScript.RunAsync("int X = 100;");
            var result2 = CSharpScript.RunAsync("X + X", result);
            Assert.Equal(200, result2.ReturnValue.Result);
        }

        [Fact]
        public void TestRepl()
        {
            string[] submissions = new[]
            {
                "int x = 100;",
                "int y = x * x;",
                "x + y"
            };

            object input = null;
            ScriptState<object> result = null;
            foreach (var submission in submissions)
            {
                result = CSharpScript.RunAsync(submission, input);
                input = result;
            }

            Assert.Equal(10100, result.ReturnValue.Result);
        }

#if TODO // https://github.com/dotnet/roslyn/issues/3720
        [Fact]
        public void TestCreateMethodDelegate()
        {
            // create a delegate to a method declared in the script
            var state = CSharpScript.Run("int Times(int x) { return x * x; }");
            var fn = state.CreateDelegate<Func<int, int>>("Times");
            var result = fn(5);
            Assert.Equal(25, result);
        }
#endif

        [Fact]
        public void TestGetScriptVariableAfterRunningScript()
        {
            var result = CSharpScript.RunAsync("int x = 100;");
            var globals = result.Variables.Names.ToList();
            Assert.Equal(1, globals.Count);
            Assert.Equal(true, globals.Contains("x"));
            Assert.Equal(true, result.Variables.ContainsVariable("x"));
            Assert.Equal(100, (int)result.Variables["x"].Value);
        }

        [Fact]
        public void TestBranchingSubscripts()
        {
            // run script to create declaration of M
            var result1 = CSharpScript.RunAsync("int M(int x) { return x + x; }");

            // run second script starting from first script's end state
            // this script's new declaration should hide the old declaration
            var result2 = CSharpScript.RunAsync("int M(int x) { return x * x; } M(5)", result1);
            Assert.Equal(25, result2.ReturnValue.Result);

            // run third script also starting from first script's end state
            // it should not see any declarations made by the second script.
            var result3 = CSharpScript.RunAsync("M(5)", result1);
            Assert.Equal(10, result3.ReturnValue.Result);
        }
    }
}
