<?php

namespace Solital\Core\Security\Scanner;

use Solital\Core\Console\Output\ConsoleOutput;
use Solital\Core\Security\Scanner\Traits\{PatternTrait, PrintTrait, ScanTrait, WhiteListTrait};

class MalwareScanner
{
    use PatternTrait, PrintTrait, ScanTrait, WhiteListTrait;

    /**
     * @var mixed
     */
    private mixed $dir;

    /**
     * @var array
     */
    private array $extension = ['.php'];

    /**
     * @var array
     */
    private array $ignore = [];

    /**
     * MalwareScanner constructor.
     *
     * @param bool $cli defines its calling from commandline or using as a library, default is true
     */
    public function __construct(bool $cli = true)
    {
        if ($cli === true) {
            //Read Run Options
            $this->parseArgs();

            $dirs = [];
            if (is_array($this->dir)) {
                // allow multiple directory aka. array
                foreach ($this->dir as $path) {
                    $dirs[] = realpath($path);
                }
            } elseif ($bpos = strpos($this->dir, '{')) {
                // Check path has a "brace", expand it to subdirectories
                foreach (glob($this->dir, GLOB_BRACE) as $path) {
                    $dirs[] = realpath($path);
                }
            } else {
                // only one directory specified
                $dirs = [realpath($this->dir)];
            }

            //Make sure a directory was specified.
            if (empty($dirs)) {
                ConsoleOutput::error('No directory specified or directory doesn\'t exist');
                exit(-1);
            }

            //Initiate Scan
            if (!$this->run($dirs)) {
                exit(-1);
            }
        }
    }

    /**
     * Check if -i/--ignore flag listed this path to be omitted
     *
     * @param string $pathname
     * 
     * @return bool
     */
    private function isIgnored(string $pathname): bool
    {
        foreach ($this->ignore as $pattern) {
            $match = $this->pathMatches($pathname, $pattern);

            if ($match) {
                return true;
            }
        }

        return false;
    }

    /**
     * @param string $wp_version
     * 
     * @return void
     */
    public function addWordpressChecksums(string $wp_version): void
    {
        $apiurl = 'https://api.wordpress.org/core/checksums/1.0/?version=' . $wp_version;
        $json = json_decode(file_get_contents($apiurl));
        $checksums = $json->checksums;

        if ($checksums->$wp_version == false) { #no checksum returned
            ConsoleOutput::error('Cannot load wordpress checksums from: ' . $apiurl);
            exit(-1);
        }

        foreach ($checksums->$wp_version as $file => $checksum) {
            $this->whitelist[] = $checksum;
        }
    }

    /**
     * Handles the getopt() function call, sets attributes according to flags.
     * All flag handling stuff should be setup here.
     *
     * @return void
     */
    private function parseArgs(): void
    {
        /* $options = getopt(
            'd:e:i:o:abmcxlhkwnsptLj:E',
            array(
                'directory:',
                'extension:',
                'ignore:',
                'all-output',
                'base',
                'checksum',
                'comment',
                'extra-check',
                'follow-link',
                'help',
                'hide-ok',
                'hide-whitelist',
                'no-color',
                'no-stop',
                'pattern',
                'time',
                'line-number',
                'output-format:',
                'wordpress-version:',
                'scan-everything',
                'combined-whitelist',
                'custom-whitelist:',
                'disable-stats'
            )
        ); */

        $options = getopt(
            'd:e:i:o:abmcxlhkwnsptLj:E',
            [
                'directory:', 'extension:', 'ignore:', 'all-output', 'base', 'checksum',
                'comment', 'extra-check', 'follow-link', 'help', 'hide-ok', 'hide-whitelist',
                'no-color', 'no-stop', 'pattern', 'time', 'line-number', 'output-format:',
                'wordpress-version:', 'scan-everything', 'combined-whitelist',
                'custom-whitelist:', 'disable-stats'
            ]
        );

        //Help Option should be first
        if (isset($options['help']) || isset($options['h'])) {
            $this->showHelp();
            exit;
        }

        //Options that Require Additional Parameters
        if (isset($options['directory']) || isset($options['d'])) {
            //$this->dir = isset($options['directory']) ? $options['directory'] : $options['d'];
            $this->dir = $options['directory'] ?? $options['d'];
        }

        if (isset($options['extension']) || isset($options['e'])) {
            //$a = isset($options['extension']) ? $options['extension'] : $options['e'];
            $a = $options['extension'] ?? $options['e'];

            if (!is_array($a)) {
                $a = [$a];
            }

            $this->setExtensions($a);
        }
        if (isset($options['ignore']) || isset($options['i'])) {
            /* $tmp = isset($options['ignore']) ? $options['ignore'] : $options['i'];
            $this->setIgnore(is_array($tmp) ? $tmp : array($tmp)); */

            $tmp = $options['ignore'] ?? $options['i'];
            $this->setIgnore(is_array($tmp) ? $tmp : [$tmp]);
        }

        //Simple Flag Options
        if (isset($options['all-output']) || isset($options['a'])) {
            $this->setFlagChecksum(true);
            $this->setFlagComments(true);
            $this->setFlagPattern(true);
            $this->setFlagTime(true);
        }

        if (isset($options['base64']) || isset($options['b'])) {
            $this->setFlagBase64(true);
        }

        if (isset($options['checksum']) || isset($options['m'])) {
            $this->setFlagChecksum(true);
        }

        if (isset($options['comment']) || isset($options['c'])) {
            $this->setFlagComments(true);
        }

        if (isset($options['extra-check']) || isset($options['x'])) {
            $this->setFlagExtraCheck(true);
        }

        if (isset($options['follow-symlink']) || isset($options['l'])) {
            $this->setFlagFollowSymlink(true);
        }

        if (isset($options['hide-ok']) || isset($options['k'])) {
            $this->setFlagHideOk(true);
        }

        if (isset($options['hide-err']) || isset($options['r'])) {
            $this->setFlagHideErr(true);
        }

        if (isset($options['hide-whitelist']) || isset($options['w'])) {
            $this->setFlagHideWhitelist(true);
        }

        if (isset($options['no-color']) || isset($options['n'])) {
            $this->disableColor();
        }

        if (isset($options['no-stop']) || isset($options['s'])) {
            $this->setFlagNoStop(true);
        }

        if (isset($options['pattern']) || isset($options['p'])) {
            $this->setFlagPattern(true);
        }

        if (isset($options['time']) || isset($options['t'])) {
            $this->setFlagTime(true);
        }

        if (isset($options['line-number']) || isset($options['L'])) {
            $this->setFlagLineNumber(true);
        }

        if (isset($options['output-format']) || isset($options['o'])) {
            /* $tmp = isset($options['output-format']) ? $options['output-format'] : $options['o'];
            $this->setOutputFormat(is_array($tmp) ? $tmp : array($tmp)); */

            $tmp = $options['output-format'] ?? $options['o'];
            $this->setOutputFormat(is_array($tmp) ? $tmp : [$tmp]);
        }

        if (isset($options['wordpress-version']) || isset($options['j'])) {
            //$tmp = isset($options['wordpress-version']) ? $options['wordpress-version'] : $options['j'];
            $tmp = $options['wordpress-version'] ?? $options['j'];
            $this->addWordpressChecksums($tmp);
        }

        if (isset($options['scan-everything']) || isset($options['E'])) {
            $this->setFlagScanEverything(true);
        }

        if (isset($options['combined-whitelist'])) {
            $this->setFlagCombinedWhitelist(true);
        }

        if (isset($options['custom-whitelist'])) {
            $a = $options['custom-whitelist'];

            if (!is_array($a)) {
                $a = [$a];
            }

            $this->setCustomWhitelist(array_unique($a));
        }

        if (isset($options['disable-stats'])) {
            $this->setFlagDisableStats(true);
        }
    }

    /**
     * @param array $a
     * 
     * @return void
     */
    public function setExtensions(array $a): void
    {
        $this->extension = [];

        foreach ($a as $ext) {
            if ($ext[0] != '.') {
                $ext = '.' . $ext;
            }

            $this->extension[] = strtolower((string) $ext);
        }
    }

    /**
     * @param array $a
     * 
     * @return void
     */
    public function setIgnore(array $a): void
    {
        $this->ignore = $a;
    }

    /**
     * Recursively scales the file system.
     * Calls the scan() function for each file found.
     * 
     * @param string $dir
     * 
     * @return void
     */
    private function process(string $dir): void
    {
        $dh = opendir($dir);

        if (!$dh) {
            return;
        }

        $this->stat['directories']++;

        while (($file = readdir($dh)) !== false) {
            if ($file == '.' || $file == '..') {
                continue;
            }
            if ($this->isIgnored($dir . $file)) {
                continue;
            }
            if (!$this->flagFollowSymlink && is_link($dir . $file)) {
                continue;
            }
            if (is_dir($dir . $file)) {
                $this->process($dir . $file . '/');
            } elseif (is_file($dir . $file)) {
                $ext = strtolower(substr($file, strrpos($file, '.')));

                if ($this->flagScanEverything || in_array($ext, $this->extension)) {
                    $this->scan($dir . $file);
                }
            }
        }

        closedir($dh);
    }

    /**
     * Validates the input directory
     *
     * - Calls the load pattern and load whitelist functions
     * - Fetch and load combined whitelist
     * - Calls the process and report functions.
     *
     * @param string|array $dir A directory path or a list of paths in array
     * @return bool
     */
    public function run(string|array $dir): bool
    {
        $this->initializePatterns();
        $this->loadWhitelists();

        if ($this->flagCombinedWhitelist && !$this->updateCombinedWhitelist()) {
            return false;
        }

        $start = time();

        ConsoleOutput::info('Start time: ')->print();
        ConsoleOutput::line((string)date('Y-m-d H:i:s', $start))->print()->break();

        if (!is_array($dir)) {
            $dir = [$dir];
        }

        foreach ($dir as $path) {
            // Make sure the input is a valid directory path.
            $path = rtrim((string) $path, DIRECTORY_SEPARATOR);

            if (!is_dir($path)) {
                ConsoleOutput::error('Specified path is not a directory: ' . $path);
                return false;
            }

            $this->process($path . DIRECTORY_SEPARATOR);
        }

        if (!$this->flagDisableStats) {
            $this->report($start, implode(', ', $dir));
        }

        return true;
    }
}
