﻿using System;
using System.IO;
using System.Text.RegularExpressions;

using SimaticLib;


namespace S7Lib
{
    /// <summary>
    /// Contains methods related to Symbol Table
    /// </summary>
    public static class S7Symbols
    {
        // Default path for symbol import report file
        static string ReportFilePath = @"C:\ProgramData\Siemens\Automation\Step7\S7Tmp\sym_imp.txt";
        // Default titles for Notepad window opened on import symbols command
        // The title window may or may not include the .txt extension depending on Explorer settings
        static string[] NotepadWindowTitles = new string[]{
            "sym_imp - Notepad", "sym_imp.txt - Notepad"
        };

        public static string ReadFile(S7Context ctx, string path)
        {
            var log = ctx.Log;
            if (File.Exists(path))
                return File.ReadAllText(path);
            log.Warning($"File {path} not found");
            return "";
        }

        /// <summary>
        /// Returns counted errors, warnings and conflicts by parsing the content
        /// of the symbol importation file.
        /// </summary>
        /// <param name="errors">Number of errors during importation</param>
        /// <param name="warnings">Number of warnings during importation</param>
        /// <param name="conflicts">Number of symbol conflicts during importation</param>
        /// <returns>The total number of critical errors (sum of errors and conflicts)</returns>
        private static string GetImportReport(S7Context ctx,
            ref int errors, ref int warnings, ref int conflicts)
        {
            string report = ReadFile(ctx, ReportFilePath);
            string[] split = report.Split('\n');

            int errorIndex = Array.FindIndex<string>(split, s => Regex.IsMatch(s, "^Error:.*"));
            int warningsIndex = errorIndex + 1;
            int conflictsIndex = errorIndex + 2;

            errors = Int32.Parse(split[errorIndex].Split(' ')[1]);
            warnings = Int32.Parse(split[warningsIndex].Split(' ')[1]);
            conflicts = Int32.Parse(split[conflictsIndex].Split(' ')[1]);

            return report;
        }

        /// <summary>
        /// Close the Notepad window with importation log.
        /// </summary>
        private static void CloseSymbolImportationLogWindow(S7Context ctx)
        {
            var log = ctx.Log;
            IntPtr windowHandle = IntPtr.Zero;
            foreach (var windowTitle in NotepadWindowTitles)
            {
                windowHandle = WindowsAPI.FindWindow(null, windowTitle);
                if (windowHandle != IntPtr.Zero) break;
            }
            if (windowHandle == IntPtr.Zero)
            {
                log.Warning($"Could not find Notepad window with importation log.");
                return;
            }
            WindowsAPI.SendMessage(windowHandle, WindowsAPI.WM_CLOSE, IntPtr.Zero, IntPtr.Zero);
            log.Debug($"Closed Notepad window with importation log");
        }

        public static int ImportSymbols(S7Context ctx,
            string project, string programPath, string symbolFile, int flag = 0, bool allowConflicts = false)
        {
            var api = ctx.Api;
            var log = ctx.Log;
            var flags = (S7SymImportFlags)flag;

            S7Program programObj = Api.GetProgram(ctx, project, programPath);
            if (programObj == null) return -1;

            S7SymbolTable symbolTable;
            try
            {
                symbolTable = (S7SymbolTable)programObj.SymbolTable;
            }
            catch (Exception exc)
            {
                log.Error(exc, $"Could not access symbol table in {project}:{programObj.LogPath}");
                return -1;
            }

            int numSymbols = -1;
            try
            {
                numSymbols = symbolTable.Import(symbolFile, Flags: flags);
            }
            catch (Exception exc)
            {
                log.Error(exc, $"Could not import symbol table into {project}:{programObj.LogPath} " +
                               $"from {symbolFile}");
                return -1;
            }

            int errors = -1, warnings = -1, conflicts = -1;
            string report = GetImportReport(ctx, ref errors, ref warnings, ref conflicts);
            CloseSymbolImportationLogWindow(ctx);

            log.Debug($"Imported {numSymbols} symbols from {symbolFile} into {project}:{programObj.LogPath}\n" +
                      $"Report {errors} error(s), {warnings} warning(s) and {conflicts} conflict(s):\n" +
                      $"{report}");

            if (!allowConflicts && conflicts > 0)
                return -1;
            return 0;
        }

        public static int ExportSymbols(S7Context ctx,
            string project, string programPath, string symbolFile)
        {
            var log = ctx.Log;

            S7Program programObj = Api.GetProgram(ctx, project, programPath);
            if (programObj == null) return -1;

            S7SymbolTable symbolTable;
            try
            {
                symbolTable = (S7SymbolTable)programObj.SymbolTable;
            }
            catch (Exception exc)
            {
                log.Error(exc, $"Could not access symbol table in {project}:{programObj.LogPath}");
                return -1;
            }

            try
            {
                symbolTable.Export(symbolFile);
            }
            catch (Exception exc)
            {
                log.Error(exc, $"Could not export symbols from {project}:{programObj.LogPath} " +
                               $"to {symbolFile}");
                return -1;
            }

            log.Debug($"Exported symbols from {project}:{programObj.LogPath} to {symbolFile}");
            return 0;
        }
    }
}
