#include "../jade_assert.h"
#include "../keychain.h"
#include "../multisig.h"
#include "../process.h"
#include "../storage.h"
#include "../ui.h"
#include "../utils/cbor_rpc.h"
#include "../utils/event.h"
#include "../utils/malloc_ext.h"
#include "../utils/network.h"
#include "../wallet.h"

#include "process_utils.h"

#include <sodium/utils.h>

void make_confirm_multisig_activity(const char* multisig_name, bool sorted, size_t threshold, const signer_t* signers,
    size_t num_signers, const uint8_t* wallet_fingerprint, size_t wallet_fingerprint_len,
    const uint8_t* master_blinding_key, size_t master_blinding_key_len, bool overwriting,
    gui_activity_t** first_activity);

static void get_signers_allocate(const char* field, const CborValue* value, signer_t** data, size_t* written)
{
    JADE_ASSERT(field);
    JADE_ASSERT(value);
    JADE_INIT_OUT_PPTR(data);
    JADE_INIT_OUT_SIZE(written);

    CborValue result;
    if (!rpc_get_array(field, value, &result)) {
        return;
    }

    size_t num_array_items = 0;
    CborError cberr = cbor_value_get_array_length(&result, &num_array_items);
    if (cberr != CborNoError || !num_array_items) {
        return;
    }

    CborValue arrayItem;
    cberr = cbor_value_enter_container(&result, &arrayItem);
    if (cberr != CborNoError || !cbor_value_is_valid(&arrayItem)) {
        return;
    }

    signer_t* const signers = JADE_CALLOC(num_array_items, sizeof(signer_t));

    for (size_t i = 0; i < num_array_items; ++i) {
        JADE_ASSERT(!cbor_value_at_end(&arrayItem));
        signer_t* const signer = signers + i;

        if (!cbor_value_is_map(&arrayItem)) {
            free(signers);
            return;
        }

        size_t num_map_items = 0;
        if (cbor_value_get_map_length(&arrayItem, &num_map_items) == CborNoError && num_map_items == 0) {
            CborError err = cbor_value_advance(&arrayItem);
            JADE_ASSERT(err == CborNoError);
            continue;
        }

        if (!rpc_get_n_bytes("fingerprint", &arrayItem, sizeof(signer->fingerprint), signer->fingerprint)) {
            free(signers);
            return;
        }

        if (!rpc_get_bip32_path("derivation", &arrayItem, signer->derivation, MAX_PATH_LEN, &signer->derivation_len)) {
            free(signers);
            return;
        }

        rpc_get_string("xpub", sizeof(signer->xpub), &arrayItem, signer->xpub, &signer->xpub_len);
        if (signer->xpub_len == 0 || signer->xpub_len >= sizeof(signer->xpub)) {
            free(signers);
            return;
        }

        if (!rpc_get_bip32_path("path", &arrayItem, signer->path, MAX_PATH_LEN, &signer->path_len)) {
            free(signers);
            return;
        }

        CborError err = cbor_value_advance(&arrayItem);
        JADE_ASSERT(err == CborNoError);
    }

    cberr = cbor_value_leave_container(&result, &arrayItem);
    if (cberr != CborNoError) {
        free(signers);
        return;
    }

    *written = num_array_items;
    *data = signers;
}

void register_multisig_process(void* process_ptr)
{
    JADE_LOGI("Starting: %u", xPortGetFreeHeapSize());
    jade_process_t* process = process_ptr;

    char network[MAX_NETWORK_NAME_LEN];
    char multisig_name[MAX_MULTISIG_NAME_SIZE];
    char variant[MAX_VARIANT_LEN];

    // We expect a current message to be present
    ASSERT_CURRENT_MESSAGE(process, "register_multisig");
    ASSERT_KEYCHAIN_UNLOCKED_BY_MESSAGE_SOURCE(process);
    GET_MSG_PARAMS(process);

    // Check network is valid and consistent with prior usage
    size_t written = 0;
    rpc_get_string("network", sizeof(network), &params, network, &written);
    CHECK_NETWORK_CONSISTENT(process, network, written);

    // Get name of multisig wallet
    written = 0;
    rpc_get_string("multisig_name", sizeof(multisig_name), &params, multisig_name, &written);
    if (written == 0 || !storage_key_name_valid(multisig_name)) {
        jade_process_reject_message(
            process, CBOR_RPC_BAD_PARAMETERS, "Missing or invalid multisig name parameter", NULL);
        goto cleanup;
    }

    CborValue descriptor;
    if (!rpc_get_map("descriptor", &params, &descriptor)) {
        jade_process_reject_message(process, CBOR_RPC_BAD_PARAMETERS, "Cannot extract multisig descriptor data", NULL);
        goto cleanup;
    }

    // Handle script variants.
    written = 0;
    script_variant_t script_variant;
    rpc_get_string("variant", sizeof(variant), &descriptor, variant, &written);
    if (!get_script_variant(variant, written, &script_variant) || !is_multisig(script_variant)) {
        jade_process_reject_message(process, CBOR_RPC_BAD_PARAMETERS, "Invalid script variant parameter", NULL);
        goto cleanup;
    }

    // Handle sorted-multisig - defaults to false if not passed
    bool sorted = false;
    if (rpc_has_field_data("sorted", &descriptor)) {
        if (!rpc_get_boolean("sorted", &descriptor, &sorted)) {
            jade_process_reject_message(process, CBOR_RPC_BAD_PARAMETERS, "Invalid sorted flag value", NULL);
            goto cleanup;
        }
    }

    // Liquid master blinding key for this multisig wallet
    const uint8_t* master_blinding_key = NULL;
    size_t master_blinding_key_len = 0;
    if (rpc_has_field_data("master_blinding_key", &descriptor)) {
        rpc_get_bytes_ptr("master_blinding_key", &descriptor, &master_blinding_key, &master_blinding_key_len);
        if (!master_blinding_key || master_blinding_key_len != MULTISIG_MASTER_BLINDING_KEY_SIZE) {
            jade_process_reject_message(process, CBOR_RPC_BAD_PARAMETERS, "Invalid blinding key value", NULL);
            goto cleanup;
        }
    }

    // Threshold
    written = 0;
    rpc_get_sizet("threshold", &descriptor, &written);
    if (written == 0 || written > MAX_MULTISIG_SIGNERS) {
        jade_process_reject_message(process, CBOR_RPC_BAD_PARAMETERS, "Invalid multisig threshold value", NULL);
        goto cleanup;
    }
    const uint8_t threshold = (uint8_t)written;

    // Co-Signers
    signer_t* signers = NULL;
    size_t num_signers = 0;
    get_signers_allocate("signers", &descriptor, &signers, &num_signers);
    if (num_signers == 0) {
        jade_process_reject_message(
            process, CBOR_RPC_BAD_PARAMETERS, "Failed to extract valid co-signers from parameters", NULL);
        goto cleanup;
    }
    jade_process_free_on_exit(process, signers);

    if (num_signers > MAX_MULTISIG_SIGNERS) {
        jade_process_reject_message(
            process, CBOR_RPC_BAD_PARAMETERS, "Failed to extract valid co-signers from parameters", NULL);
        goto cleanup;
    }

    if (threshold > num_signers) {
        jade_process_reject_message(
            process, CBOR_RPC_BAD_PARAMETERS, "Invalid multisig threshold for number of co-signers", NULL);
        goto cleanup;
    }

    // Validate signers
    uint8_t wallet_fingerprint[BIP32_KEY_FINGERPRINT_LEN];
    wallet_get_fingerprint(wallet_fingerprint, sizeof(wallet_fingerprint));
    if (!multisig_validate_signers(network, signers, num_signers, wallet_fingerprint, sizeof(wallet_fingerprint))) {
        jade_process_reject_message(process, CBOR_RPC_BAD_PARAMETERS, "Failed to validate multisig co-signers", NULL);
        goto cleanup;
    }

    uint8_t registration[MAX_MULTISIG_BYTES_LEN]; // Sufficient
    const size_t registration_len = MULTISIG_BYTES_LEN(master_blinding_key_len, num_signers);
    JADE_ASSERT(registration_len <= sizeof(registration));
    if (!multisig_data_to_bytes(script_variant, sorted, threshold, signers, num_signers, master_blinding_key,
            master_blinding_key_len, registration, registration_len)) {
        jade_process_reject_message(
            process, CBOR_RPC_INTERNAL_ERROR, "Failed to serialise multisig registration", NULL);
        goto cleanup;
    }

    // See if a record for this name exists already
    const bool overwriting = storage_multisig_name_exists(multisig_name);

    // If so, see if it is identical to the record we are trying to persist
    // - if so, just return true immediately.
    if (overwriting) {
        written = 0;
        uint8_t existing[MAX_MULTISIG_BYTES_LEN]; // Sufficient
        if (storage_get_multisig_registration(multisig_name, existing, sizeof(existing), &written)
            && written == registration_len && !sodium_memcmp(existing, registration, registration_len)) {
            JADE_LOGI("Multisig %s: identical registration exists, returning immediately", multisig_name);
            goto return_ok;
        }
    } else {
        // Not overwriting an existing record - check storage slot available
        if (storage_get_multisig_registration_count() >= MAX_MULTISIG_REGISTRATIONS) {
            jade_process_reject_message(
                process, CBOR_RPC_BAD_PARAMETERS, "Already have maximum number of multisig wallets", NULL);
            goto cleanup;
        }
    }

    gui_activity_t* first_activity = NULL;
    make_confirm_multisig_activity(multisig_name, sorted, threshold, signers, num_signers, wallet_fingerprint,
        sizeof(wallet_fingerprint), master_blinding_key, master_blinding_key_len, overwriting, &first_activity);
    JADE_ASSERT(first_activity);
    gui_set_current_activity(first_activity);

    // ----------------------------------
    // wait for the last "next" (proceed with the protocol and then final confirmation)
    int32_t ev_id;
    // In a debug unattended ci build, assume buttons pressed after a short delay
#ifndef CONFIG_DEBUG_UNATTENDED_CI
    const esp_err_t gui_ret = sync_await_single_event(JADE_EVENT, ESP_EVENT_ANY_ID, NULL, &ev_id, NULL, 0);
#else
    sync_await_single_event(
        JADE_EVENT, ESP_EVENT_ANY_ID, NULL, &ev_id, NULL, CONFIG_DEBUG_UNATTENDED_CI_TIMEOUT_MS / portTICK_PERIOD_MS);
    const esp_err_t gui_ret = ESP_OK;
    ev_id = MULTISIG_ACCEPT;
#endif

    // Check to see whether user accepted or declined
    if (gui_ret != ESP_OK || ev_id != MULTISIG_ACCEPT) {
        JADE_LOGW("User declined to register multisig");
        jade_process_reject_message(process, CBOR_RPC_USER_CANCELLED, "User declined to register multisig", NULL);
        goto cleanup;
    }

    JADE_LOGD("User accepted multisig");

    // Persist multisig registration in nvs
    if (!storage_set_multisig_registration(multisig_name, registration, registration_len)) {
        jade_process_reject_message(process, CBOR_RPC_INTERNAL_ERROR, "Failed to persist multisig registration", NULL);
        await_error_activity("Error saving multisig data");
        goto cleanup;
    }

return_ok:
    // Ok, all verified and persisted
    jade_process_reply_to_message_ok(process);
    JADE_LOGI("Success");

cleanup:
    return;
}
