//
// PublicKeyTest.cs - NUnit Test Cases for 
//	System.Security.Cryptography.X509Certificates.PublicKey
//
// Author:
//	Sebastien Pouliot  <sebastien@ximian.com>
//
// Copyright (C) 2005-2006 Novell, Inc (http://www.novell.com)
//
// Permission is hereby granted, free of charge, to any person obtaining
// a copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to
// permit persons to whom the Software is furnished to do so, subject to
// the following conditions:
// 
// The above copyright notice and this permission notice shall be
// included in all copies or substantial portions of the Software.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
// EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
// NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
// LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
// OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//

#if NET_2_0

using NUnit.Framework;
using System;
using System.Security.Cryptography;
using System.Security.Cryptography.X509Certificates;

namespace MonoTests.System.Security.Cryptography.X509Certificates {

	[TestFixture]
	public class PublicKeyTest {

		static public Oid rsaOid = new Oid ("1.2.840.113549.1.1.1");
		static public Oid dsaOid = new Oid ("1.2.840.10040.4.1");
		static public Oid unknownOid = new Oid ("1.2.3.4");
		static public AsnEncodedData emptyData = new AsnEncodedData (new byte[0]);

		private static byte[] cert_a = { 0x30,0x82,0x01,0xFF,0x30,0x82,0x01,0x6C,0x02,0x05,0x02,0x72,0x00,0x06,0xE8,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x02,0x05,0x00,0x30,0x5F,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x20,0x30,0x1E,0x06,0x03,0x55,0x04,0x0A,0x13,0x17,0x52,0x53,0x41,0x20,0x44,0x61,0x74,0x61,0x20,0x53,0x65,0x63,0x75,0x72,0x69,0x74,0x79,0x2C,0x20,0x49,0x6E,0x63,0x2E,0x31,0x2E,0x30,0x2C,0x06,0x03,0x55,0x04,0x0B,0x13,0x25,0x53,0x65,0x63,0x75,0x72,0x65,0x20,0x53,0x65,0x72,0x76,
			0x65,0x72,0x20,0x43,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x69,0x6F,0x6E,0x20,0x41,0x75,0x74,0x68,0x6F,0x72,0x69,0x74,0x79,0x30,0x1E,0x17,0x0D,0x39,0x36,0x30,0x33,0x31,0x32,0x31,0x38,0x33,0x38,0x34,0x37,0x5A,0x17,0x0D,0x39,0x37,0x30,0x33,0x31,0x32,0x31,0x38,0x33,0x38,0x34,0x36,0x5A,0x30,0x61,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x13,0x30,0x11,0x06,0x03,0x55,0x04,0x08,0x13,0x0A,0x43,0x61,0x6C,0x69,0x66,0x6F,0x72,0x6E,0x69,0x61,0x31,0x14,0x30,0x12,0x06,0x03,
			0x55,0x04,0x0A,0x13,0x0B,0x43,0x6F,0x6D,0x6D,0x65,0x72,0x63,0x65,0x4E,0x65,0x74,0x31,0x27,0x30,0x25,0x06,0x03,0x55,0x04,0x0B,0x13,0x1E,0x53,0x65,0x72,0x76,0x65,0x72,0x20,0x43,0x65,0x72,0x74,0x69,0x66,0x69,0x63,0x61,0x74,0x69,0x6F,0x6E,0x20,0x41,0x75,0x74,0x68,0x6F,0x72,0x69,0x74,0x79,0x30,0x70,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x01,0x05,0x00,0x03,0x5F,0x00,0x30,0x5C,0x02,0x55,0x2D,0x58,0xE9,0xBF,0xF0,0x31,0xCD,0x79,0x06,0x50,0x5A,0xD5,0x9E,0x0E,0x2C,0xE6,0xC2,0xF7,0xF9,
			0xD2,0xCE,0x55,0x64,0x85,0xB1,0x90,0x9A,0x92,0xB3,0x36,0xC1,0xBC,0xEA,0xC8,0x23,0xB7,0xAB,0x3A,0xA7,0x64,0x63,0x77,0x5F,0x84,0x22,0x8E,0xE5,0xB6,0x45,0xDD,0x46,0xAE,0x0A,0xDD,0x00,0xC2,0x1F,0xBA,0xD9,0xAD,0xC0,0x75,0x62,0xF8,0x95,0x82,0xA2,0x80,0xB1,0x82,0x69,0xFA,0xE1,0xAF,0x7F,0xBC,0x7D,0xE2,0x7C,0x76,0xD5,0xBC,0x2A,0x80,0xFB,0x02,0x03,0x01,0x00,0x01,0x30,0x0D,0x06,0x09,0x2A,0x86,0x48,0x86,0xF7,0x0D,0x01,0x01,0x02,0x05,0x00,0x03,0x7E,0x00,0x54,0x20,0x67,0x12,0xBB,0x66,0x14,0xC3,0x26,0x6B,0x7F,
			0xDA,0x4A,0x25,0x4D,0x8B,0xE0,0xFD,0x1E,0x53,0x6D,0xAC,0xA2,0xD0,0x89,0xB8,0x2E,0x90,0xA0,0x27,0x43,0xA4,0xEE,0x4A,0x26,0x86,0x40,0xFF,0xB8,0x72,0x8D,0x1E,0xE7,0xB7,0x77,0xDC,0x7D,0xD8,0x3F,0x3A,0x6E,0x55,0x10,0xA6,0x1D,0xB5,0x58,0xF2,0xF9,0x0F,0x2E,0xB4,0x10,0x55,0x48,0xDC,0x13,0x5F,0x0D,0x08,0x26,0x88,0xC9,0xAF,0x66,0xF2,0x2C,0x9C,0x6F,0x3D,0xC3,0x2B,0x69,0x28,0x89,0x40,0x6F,0x8F,0x35,0x3B,0x9E,0xF6,0x8E,0xF1,0x11,0x17,0xFB,0x0C,0x98,0x95,0xA1,0xC2,0xBA,0x89,0x48,0xEB,0xB4,0x06,0x6A,0x22,0x54,
			0xD7,0xBA,0x18,0x3A,0x48,0xA6,0xCB,0xC2,0xFD,0x20,0x57,0xBC,0x63,0x1C };

		private static byte[] cert_b = { 0x30,0x82,0x03,0x04,0x30,0x82,0x02,0xC4,0xA0,0x03,0x02,0x01,0x02,0x02,0x01,0x03,0x30,0x09,0x06,0x07,0x2A,0x86,0x48,0xCE,0x38,0x04,0x03,0x30,0x51,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x18,0x30,0x16,0x06,0x03,0x55,0x04,0x0A,0x13,0x0F,0x55,0x2E,0x53,0x2E,0x20,0x47,0x6F,0x76,0x65,0x72,0x6E,0x6D,0x65,0x6E,0x74,0x31,0x0C,0x30,0x0A,0x06,0x03,0x55,0x04,0x0B,0x13,0x03,0x44,0x6F,0x44,0x31,0x1A,0x30,0x18,0x06,0x03,0x55,0x04,0x03,0x13,0x11,0x41,0x72,0x6D,0x65,0x64,0x20,0x46,0x6F,
			0x72,0x63,0x65,0x73,0x20,0x52,0x6F,0x6F,0x74,0x30,0x1E,0x17,0x0D,0x30,0x30,0x31,0x30,0x32,0x35,0x30,0x30,0x30,0x30,0x30,0x30,0x5A,0x17,0x0D,0x30,0x33,0x30,0x31,0x30,0x31,0x30,0x30,0x30,0x30,0x30,0x30,0x5A,0x30,0x51,0x31,0x0B,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x55,0x53,0x31,0x18,0x30,0x16,0x06,0x03,0x55,0x04,0x0A,0x13,0x0F,0x55,0x2E,0x53,0x2E,0x20,0x47,0x6F,0x76,0x65,0x72,0x6E,0x6D,0x65,0x6E,0x74,0x31,0x0C,0x30,0x0A,0x06,0x03,0x55,0x04,0x0B,0x13,0x03,0x44,0x6F,0x44,0x31,0x1A,0x30,0x18,
			0x06,0x03,0x55,0x04,0x03,0x13,0x11,0x41,0x72,0x6D,0x65,0x64,0x20,0x46,0x6F,0x72,0x63,0x65,0x73,0x20,0x52,0x6F,0x6F,0x74,0x30,0x82,0x01,0xB6,0x30,0x82,0x01,0x2B,0x06,0x07,0x2A,0x86,0x48,0xCE,0x38,0x04,0x01,0x30,0x82,0x01,0x1E,0x02,0x81,0x81,0x00,0x90,0x89,0x3E,0x18,0x1B,0xFE,0xA3,0x1D,0x16,0x89,0x00,0xB4,0xD5,0x40,0x82,0x4C,0x2E,0xEC,0x3D,0x66,0x0D,0x0D,0xB9,0x17,0x40,0x6E,0x3A,0x5C,0x03,0x7B,0x1B,0x93,0x28,0x0C,0xEF,0xB9,0x97,0xE3,0xA1,0xEB,0xE2,0xA3,0x7C,0x61,0xDD,0x6F,0xD5,0xAD,0x15,0x69,0x00,
			0x16,0xB2,0xC3,0x08,0x3D,0xC4,0x59,0xC6,0xF2,0x70,0xA5,0xB0,0xF5,0x1F,0x1D,0xF4,0xB0,0x15,0xDA,0x7E,0x28,0x39,0x24,0x99,0x36,0x5B,0xEC,0x39,0x25,0xFA,0x92,0x49,0x65,0xD2,0x43,0x05,0x6A,0x9E,0xA3,0x7B,0xF0,0xDE,0xA3,0x2F,0xD3,0x6F,0x3A,0xF9,0x35,0xC3,0x29,0xD4,0x45,0x6C,0x56,0x9A,0xDE,0x36,0x6E,0xFE,0x12,0x68,0x96,0x7B,0x45,0x1D,0x2C,0xFF,0xB9,0x2D,0xF5,0x52,0x8C,0xDF,0x3E,0x2F,0x63,0x02,0x15,0x00,0x81,0xA9,0xB5,0xD0,0x04,0xF2,0x9B,0xA7,0xD8,0x55,0x4C,0x3B,0x32,0xA1,0x45,0x32,0x4F,0xF5,0x51,0xDD,
			0x02,0x81,0x80,0x64,0x7A,0x88,0x0B,0xF2,0x3E,0x91,0x81,0x59,0x9C,0xF4,0xEA,0xC6,0x7B,0x0E,0xBE,0xEA,0x05,0xE8,0x77,0xFD,0x20,0x34,0x87,0xA1,0xC4,0x69,0xF6,0xC8,0x8B,0x19,0xDA,0xCD,0xFA,0x21,0x8A,0x57,0xA9,0x7A,0x26,0x0A,0x56,0xD4,0xED,0x4B,0x1B,0x7C,0x70,0xED,0xB4,0xE6,0x7A,0x6A,0xDE,0xD3,0x29,0xE2,0xE9,0x9A,0x33,0xED,0x09,0x8D,0x9E,0xDF,0xDA,0x2E,0x4A,0xC1,0x50,0x92,0xEE,0x2F,0xE5,0x5A,0xF3,0x85,0x62,0x6A,0x48,0xDC,0x1B,0x02,0x98,0xA6,0xB0,0xD1,0x09,0x4B,0x10,0xD1,0xF0,0xFA,0xE0,0xB1,0x1D,0x13,
			0x54,0x4B,0xC0,0xA8,0x40,0xEF,0x71,0xE8,0x56,0x6B,0xA2,0x29,0xCB,0x1E,0x09,0x7D,0x27,0x39,0x91,0x3B,0x20,0x4F,0x98,0x39,0xE8,0x39,0xCA,0x98,0xC5,0xAF,0x54,0x03,0x81,0x84,0x00,0x02,0x81,0x80,0x54,0xA8,0x88,0xB5,0x8F,0x01,0x56,0xCE,0x18,0x8F,0xA6,0xD6,0x7C,0x29,0x29,0x75,0x45,0xE8,0x31,0xA4,0x07,0x17,0xED,0x1E,0x5D,0xB2,0x7B,0xBB,0xCE,0x3C,0x97,0x67,0x1E,0x88,0x0A,0xFE,0x7D,0x00,0x22,0x27,0x1D,0x66,0xEE,0xF6,0x1B,0xB6,0x95,0x7F,0x5A,0xFF,0x06,0x34,0x02,0x43,0xC3,0x83,0xC4,0x66,0x2C,0xA1,0x05,0x0E,
			0x68,0xB3,0xCA,0xDC,0xD3,0xF9,0x0C,0xC0,0x66,0xDF,0x85,0x84,0x4B,0x20,0x5D,0x41,0xAC,0xC0,0xEC,0x37,0x92,0x0E,0x97,0x19,0xBF,0x53,0x35,0x63,0x27,0x18,0x33,0x35,0x42,0x4D,0xF0,0x2D,0x6D,0xA7,0xA4,0x98,0xAA,0x57,0xF3,0xD2,0xB8,0x6E,0x4E,0x8F,0xFF,0xBE,0x6F,0x4E,0x0F,0x0B,0x44,0x24,0xEE,0xDF,0x4C,0x22,0x5B,0x44,0x98,0x94,0xCB,0xB8,0xA3,0x2F,0x30,0x2D,0x30,0x1D,0x06,0x03,0x55,0x1D,0x0E,0x04,0x16,0x04,0x14,0x9D,0x2D,0x73,0xC3,0xB8,0xE3,0x4D,0x29,0x28,0xC3,0x65,0xBE,0xA9,0x98,0xCB,0xD6,0x8A,0x06,0x68,
			0x9C,0x30,0x0C,0x06,0x03,0x55,0x1D,0x13,0x04,0x05,0x30,0x03,0x01,0x01,0xFF,0x30,0x09,0x06,0x07,0x2A,0x86,0x48,0xCE,0x38,0x04,0x03,0x03,0x2F,0x00,0x30,0x2C,0x02,0x14,0x5A,0x1B,0x2D,0x08,0x0E,0xE6,0x99,0x38,0x8F,0xB5,0x09,0xC9,0x89,0x79,0x7E,0x01,0x30,0xBD,0xCE,0xF0,0x02,0x14,0x71,0x7B,0x08,0x51,0x97,0xCE,0x4D,0x1F,0x6A,0x84,0x47,0x3A,0xC0,0xBD,0x13,0x89,0x81,0xB9,0x01,0x97 };

		private static byte[] cert_c = {
			0x30, 0x82, 0x05, 0xA5, 0x02, 0x01, 0x03, 0x30, 0x82, 0x05, 0x5F, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07, 0x01, 0xA0, 0x82,
 			0x05, 0x50, 0x04, 0x82, 0x05, 0x4C, 0x30, 0x82, 0x05, 0x48, 0x30, 0x82, 0x02, 0x57, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x07,
			0x06, 0xA0, 0x82, 0x02, 0x48, 0x30, 0x82, 0x02, 0x44, 0x02, 0x01, 0x00, 0x30, 0x82, 0x02, 0x3D, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D,
			0x01, 0x07, 0x01, 0x30, 0x1C, 0x06, 0x0A, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x0C, 0x01, 0x03, 0x30, 0x0E, 0x04, 0x08, 0xC5, 0xB6, 0x41,
			0x76, 0xD8, 0x4F, 0x82, 0x05, 0x02, 0x02, 0x07, 0xD0, 0x80, 0x82, 0x02, 0x10, 0x9B, 0x3A, 0x6F, 0xFC, 0xBE, 0x07, 0x3F, 0x4F, 0xAD, 0xF3, 0x7B,
			0x91, 0xB4, 0x86, 0x23, 0xFE, 0x85, 0x63, 0x20, 0x37, 0x47, 0x1F, 0xD5, 0x95, 0x9C, 0x25, 0x8B, 0xE2, 0x4A, 0xA1, 0x97, 0x80, 0x8A, 0x17, 0xB3,
			0x7F, 0xD2, 0x4C, 0x7C, 0x78, 0x48, 0x82, 0x77, 0x45, 0x00, 0xC3, 0x30, 0xB9, 0xF8, 0xF2, 0x44, 0xE6, 0xEA, 0xF7, 0x62, 0xFE, 0x93, 0xEA, 0xBC,
			0xF7, 0x27, 0x51, 0x57, 0xC8, 0x64, 0x51, 0x4D, 0xA6, 0x51, 0x34, 0x74, 0x2C, 0x4B, 0x53, 0xA1, 0x24, 0x51, 0x84, 0xA1, 0xEC, 0xEE, 0x15, 0xDE,
			0xEE, 0xD0, 0x44, 0xE1, 0x10, 0xEF, 0x51, 0x03, 0xC0, 0xBF, 0x7D, 0x3A, 0xFF, 0xB2, 0x83, 0x03, 0x24, 0xC7, 0x64, 0x16, 0x65, 0x82, 0x45, 0x90,
			0x21, 0x7E, 0x5F, 0x8D, 0x45, 0x20, 0x53, 0x63, 0x4A, 0xB5, 0x7F, 0x56, 0xD2, 0x38, 0x5C, 0x1D, 0x7D, 0x4D, 0x56, 0xF2, 0xC8, 0xA2, 0x33, 0xF9,
			0x44, 0x0A, 0x74, 0xDA, 0x95, 0x1D, 0x81, 0x9F, 0xB7, 0x93, 0xE2, 0x78, 0xC6, 0xB5, 0x90, 0xCF, 0x9B, 0xE1, 0x5B, 0x3C, 0x2A, 0x2E, 0x01, 0x50,
			0x19, 0xDE, 0xD3, 0xE3, 0x99, 0xA9, 0x0D, 0xEC, 0x26, 0x30, 0x63, 0x73, 0xA9, 0xAA, 0x69, 0x91, 0xC0, 0xC0, 0x63, 0xD6, 0xF5, 0xCF, 0x64, 0xF2,
			0x9E, 0x68, 0x06, 0x7F, 0x32, 0xED, 0xFA, 0x86, 0x7B, 0xCA, 0xC4, 0x19, 0x92, 0x39, 0x26, 0x59, 0x45, 0x04, 0xAC, 0xFF, 0x88, 0x9F, 0xAB, 0xDE,
			0x9F, 0x0E, 0x4A, 0x0B, 0xFD, 0x1C, 0xBD, 0xE7, 0xD8, 0x0D, 0xFD, 0x70, 0x02, 0x0D, 0x38, 0x28, 0x5C, 0x2E, 0x94, 0x15, 0x82, 0xDA, 0xF7, 0x62,
			0x88, 0x30, 0x17, 0x5B, 0x7A, 0x77, 0x01, 0x26, 0x4F, 0x6C, 0x17, 0x6F, 0xA4, 0x1D, 0x06, 0x46, 0x94, 0x44, 0x8D, 0x84, 0x29, 0x0D, 0xCF, 0x73,
			0xD7, 0x38, 0x5C, 0x5D, 0x32, 0x11, 0x33, 0xB0, 0x2A, 0xD9, 0x89, 0x5F, 0xFC, 0x3E, 0x1A, 0x2E, 0xD5, 0xA6, 0x44, 0x3D, 0xCC, 0x13, 0xF3, 0x9E,
			0xE8, 0x5B, 0x93, 0xFB, 0xEE, 0x9D, 0xB6, 0xEB, 0xFC, 0x15, 0x36, 0xBA, 0xC8, 0x60, 0x12, 0x4A, 0x8C, 0x98, 0xC6, 0x52, 0xF9, 0xCA, 0x03, 0xAA,
			0x8F, 0x0E, 0xE8, 0xB7, 0xA9, 0x83, 0x63, 0xDA, 0x1C, 0x7E, 0xC8, 0x73, 0x07, 0x7D, 0xD9, 0x1D, 0x94, 0x74, 0xAA, 0xC9, 0xC5, 0x31, 0x69, 0x2D,
			0xA4, 0x41, 0xB0, 0x03, 0x99, 0x15, 0x27, 0x85, 0x29, 0x4A, 0xC4, 0xFC, 0x15, 0x5C, 0xC4, 0xFB, 0x7B, 0x91, 0x5C, 0x38, 0xA4, 0xE6, 0xD6, 0x32,
			0x6F, 0xC6, 0xE6, 0x01, 0x35, 0x5D, 0x98, 0xCF, 0xDD, 0x1A, 0x72, 0x3C, 0xE1, 0x00, 0x82, 0x3B, 0x8D, 0x92, 0xCB, 0x1E, 0x61, 0x24, 0x26, 0x70,
			0xE0, 0x16, 0xE0, 0xC1, 0x7D, 0x22, 0x52, 0x9F, 0x7D, 0xB3, 0xF5, 0xCC, 0xAF, 0x2C, 0x82, 0x3C, 0xB4, 0xF8, 0x3F, 0x77, 0x6C, 0x37, 0x41, 0xDD,
			0x64, 0x96, 0x91, 0x76, 0x7F, 0x60, 0xF9, 0xDB, 0xA1, 0xCC, 0xFD, 0x7A, 0x7F, 0x57, 0x45, 0x53, 0x4B, 0x55, 0xDD, 0x65, 0x09, 0xED, 0x1B, 0xD3,
			0xFC, 0xC6, 0x6F, 0x5A, 0xE7, 0xF4, 0x57, 0xE0, 0xAE, 0xF6, 0xF9, 0x80, 0xB6, 0xA4, 0x7E, 0xA5, 0xD9, 0x73, 0xA3, 0x82, 0x10, 0xFB, 0xFC, 0xEF,
			0xF0, 0x39, 0xEB, 0xC4, 0xEE, 0xCF, 0x33, 0xA9, 0x70, 0xE1, 0xBD, 0xDF, 0xF0, 0x03, 0x9E, 0x88, 0x2F, 0x25, 0x1A, 0x9A, 0x1E, 0x15, 0x34, 0x07,
			0x43, 0x27, 0xC6, 0x38, 0x00, 0x7F, 0x48, 0xC9, 0xBA, 0x6A, 0x04, 0x18, 0xA7, 0x0B, 0x27, 0x0B, 0xD2, 0x23, 0x3F, 0xD4, 0x88, 0xB3, 0x6C, 0x0C,
			0x14, 0xD0, 0x87, 0x42, 0xA4, 0x9A, 0x7A, 0x2F, 0x97, 0x86, 0xC2, 0xD9, 0x72, 0xAD, 0xCE, 0x54, 0x1A, 0xC6, 0x3C, 0x2A, 0x6F, 0x19, 0xB5, 0x08,
			0xED, 0x53, 0xB2, 0xDF, 0x8B, 0xA7, 0x4B, 0x45, 0x6E, 0x9E, 0x66, 0x87, 0x17, 0x30, 0x82, 0x02, 0xE9, 0x06, 0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7,
			0x0D, 0x01, 0x07, 0x01, 0xA0, 0x82, 0x02, 0xDA, 0x04, 0x82, 0x02, 0xD6, 0x30, 0x82, 0x02, 0xD2, 0x30, 0x82, 0x02, 0xCE, 0x06, 0x0B, 0x2A, 0x86,
			0x48, 0x86, 0xF7, 0x0D, 0x01, 0x0C, 0x0A, 0x01, 0x02, 0xA0, 0x82, 0x02, 0xA6, 0x30, 0x82, 0x02, 0xA2, 0x30, 0x1C, 0x06, 0x0A, 0x2A, 0x86, 0x48,
			0x86, 0xF7, 0x0D, 0x01, 0x0C, 0x01, 0x03, 0x30, 0x0E, 0x04, 0x08, 0x49, 0xB7, 0x9B, 0xE6, 0x18, 0xDA, 0xE1, 0x12, 0x02, 0x02, 0x07, 0xD0, 0x04,
			0x82, 0x02, 0x80, 0xE6, 0xD9, 0x5A, 0xAC, 0xA6, 0xDD, 0x9F, 0x3C, 0x24, 0x72, 0x62, 0xA6, 0xCA, 0x78, 0x16, 0xD4, 0x55, 0x67, 0xF5, 0x0C, 0xB9,
			0x29, 0x94, 0xC7, 0xD9, 0x15, 0x76, 0x8F, 0x3B, 0xA3, 0x93, 0xF3, 0x17, 0x2F, 0x0F, 0x3C, 0x82, 0x62, 0xF8, 0x51, 0xB3, 0xBC, 0xA0, 0xE6, 0x78,
			0x2D, 0xBF, 0xE0, 0xCE, 0x53, 0xCF, 0xEB, 0x8B, 0x02, 0x2E, 0x28, 0x9C, 0x1A, 0xB2, 0x7D, 0x22, 0xEB, 0x64, 0x15, 0xC9, 0xA3, 0x83, 0x5A, 0x0D,
			0xFC, 0x79, 0x43, 0x9B, 0xDE, 0xB0, 0x72, 0xEF, 0x7E, 0xCE, 0xC9, 0x0D, 0xFE, 0x16, 0xD8, 0x9D, 0x19, 0x44, 0xB6, 0x05, 0x29, 0x46, 0xE0, 0x8D,
			0xCD, 0x89, 0xAD, 0x9A, 0x3C, 0x0F, 0xCB, 0x41, 0x5A, 0x83, 0xD0, 0x60, 0x93, 0xC4, 0xF4, 0xB9, 0xD8, 0xC8, 0x71, 0x2E, 0xE1, 0x07, 0x4A, 0xE3,
			0x01, 0xFD, 0x44, 0xFF, 0x97, 0xBB, 0x9F, 0x5B, 0x0C, 0x28, 0xD5, 0x7A, 0x6A, 0xD4, 0x14, 0x56, 0xA6, 0x35, 0x02, 0x6F, 0x36, 0x8D, 0x4C, 0x0E,
			0xF4, 0x43, 0x2C, 0x35, 0x30, 0x90, 0x7B, 0x20, 0xFE, 0xC8, 0x0B, 0x88, 0xD4, 0xBD, 0xE9, 0xD9, 0xFC, 0x1C, 0xEB, 0x70, 0x97, 0x1E, 0x6E, 0xC6,
			0x87, 0x0B, 0x2B, 0xF3, 0xBC, 0x73, 0x1C, 0x8A, 0xB6, 0xEC, 0xA7, 0x35, 0x32, 0x88, 0x23, 0xAF, 0x1E, 0x48, 0x07, 0xC8, 0x8A, 0xB1, 0xCA, 0x98,
			0x97, 0x66, 0x98, 0x72, 0xDC, 0x3A, 0x6E, 0xE0, 0x6E, 0xFA, 0x07, 0xE8, 0x7F, 0x65, 0x7F, 0x20, 0xB8, 0xDC, 0x32, 0x11, 0xAF, 0xA6, 0x1E, 0x5E,
			0xC9, 0x79, 0x85, 0x53, 0x7E, 0x65, 0x63, 0xF5, 0x0B, 0x36, 0x17, 0x08, 0x1D, 0x1C, 0xD7, 0x20, 0x09, 0x7A, 0xAF, 0x8E, 0x25, 0x61, 0xE3, 0xAD,
			0x3C, 0x3D, 0x9B, 0x87, 0x0B, 0x93, 0x11, 0xB7, 0x15, 0x60, 0xDB, 0x7D, 0xBE, 0xDC, 0x7F, 0x9E, 0xE9, 0x38, 0x84, 0x53, 0x88, 0x93, 0xED, 0xD3,
			0x71, 0x15, 0x32, 0x1F, 0x1F, 0x35, 0x91, 0xF1, 0xAB, 0x2F, 0x25, 0x57, 0x6D, 0x3F, 0x6C, 0xF5, 0x1B, 0xC8, 0xED, 0x55, 0x63, 0x02, 0xF3, 0xDA,
			0x87, 0xC7, 0xF4, 0xB0, 0x45, 0x9F, 0x12, 0xC9, 0xC3, 0x38, 0x70, 0x58, 0x8F, 0x87, 0xD5, 0x53, 0xDB, 0xB5, 0x5B, 0xAF, 0x5A, 0xF6, 0xDC, 0xE6,
			0x17, 0xF6, 0xE3, 0x0C, 0x1E, 0x59, 0x6C, 0x68, 0x84, 0x09, 0xD9, 0xCD, 0x52, 0x9F, 0x6B, 0x19, 0xB1, 0xC6, 0xAC, 0xB3, 0x48, 0x2B, 0xD3, 0x9F,
			0x85, 0xAE, 0xF4, 0x2B, 0x54, 0x84, 0x39, 0x2F, 0x01, 0x5C, 0x4C, 0x68, 0xA4, 0x98, 0xF7, 0x9E, 0x73, 0x22, 0xD0, 0x76, 0x8D, 0x54, 0x1A, 0x05,
			0x34, 0xBF, 0xF7, 0x59, 0x6E, 0x8D, 0x8C, 0xA2, 0x21, 0xCB, 0xE4, 0x3F, 0x01, 0xEB, 0xF7, 0x3E, 0x62, 0x52, 0x29, 0xED, 0x45, 0xFF, 0x4C, 0xC2,
			0xA2, 0x87, 0x69, 0xE2, 0x21, 0x43, 0x10, 0xAF, 0x23, 0xB8, 0xCE, 0xD9, 0xEA, 0x51, 0xB8, 0x96, 0x17, 0x45, 0xDD, 0x07, 0x34, 0x9D, 0x0D, 0x61,
			0x42, 0x4A, 0xAB, 0xE7, 0xE7, 0x8D, 0x12, 0xC8, 0x66, 0x72, 0x09, 0x0E, 0x9B, 0x77, 0xFC, 0x6E, 0xAD, 0xE5, 0xFF, 0xE9, 0x55, 0x36, 0xB3, 0x5E,
			0xA2, 0xAA, 0x54, 0x04, 0x8E, 0x98, 0x7A, 0x89, 0xB2, 0x3A, 0x86, 0x0E, 0xCB, 0x94, 0x33, 0xCF, 0x4C, 0x71, 0xA4, 0x6E, 0x4A, 0xEA, 0xFD, 0x78,
			0xEE, 0xC1, 0xBF, 0xDA, 0x3B, 0xC1, 0x2A, 0xC2, 0x98, 0xA6, 0x61, 0xF5, 0x29, 0x4F, 0x96, 0x9F, 0xF8, 0xA9, 0xAB, 0xFB, 0x1A, 0x45, 0xEE, 0xC0,
			0x9B, 0x91, 0x0A, 0xEA, 0xF7, 0x81, 0x0C, 0xC1, 0xF4, 0xD6, 0xEF, 0x6A, 0x57, 0x27, 0xBC, 0xD5, 0xA1, 0x86, 0x41, 0x59, 0xED, 0x62, 0x13, 0x3E,
			0xB2, 0x5E, 0x85, 0x37, 0x71, 0x45, 0x5E, 0x5B, 0x88, 0xB8, 0x9A, 0xC1, 0xE9, 0x36, 0x08, 0x7A, 0x75, 0x20, 0x38, 0x3B, 0x7B, 0xA0, 0x83, 0x1F,
			0xAF, 0x68, 0xF0, 0xD5, 0x76, 0xCC, 0x68, 0xD5, 0x49, 0xA7, 0x1E, 0x45, 0xAD, 0x3E, 0x38, 0xDB, 0xE8, 0x3F, 0x19, 0xF3, 0xAA, 0x10, 0x9B, 0x42,
			0x4C, 0x3F, 0x50, 0x93, 0xD3, 0x22, 0xDC, 0x6D, 0x40, 0xE2, 0x7D, 0xBD, 0x13, 0xCD, 0xB7, 0x2D, 0xD6, 0x4D, 0x8B, 0x5B, 0x7C, 0xE8, 0x6E, 0x06,
			0xF2, 0x66, 0xB8, 0x7B, 0x97, 0x4C, 0x6F, 0x03, 0x74, 0xEE, 0x72, 0xB1, 0xFB, 0xD9, 0x14, 0xA0, 0xC2, 0x62, 0x33, 0x79, 0xB9, 0x89, 0x52, 0x72,
			0xC4, 0x13, 0x38, 0x4E, 0x2B, 0xE9, 0xC9, 0xBA, 0x53, 0xF3, 0x1E, 0xBB, 0x5A, 0x92, 0x9F, 0xF4, 0xBF, 0x9F, 0xDC, 0xD2, 0xD3, 0x18, 0xF5, 0x51,
			0x16, 0xC4, 0x75, 0x7D, 0xF1, 0x27, 0x88, 0xB3, 0x0B, 0x8A, 0xA5, 0x1C, 0x7D, 0x2E, 0x0C, 0xF5, 0xE8, 0xF1, 0x14, 0x31, 0x15, 0x30, 0x13, 0x06,
			0x09, 0x2A, 0x86, 0x48, 0x86, 0xF7, 0x0D, 0x01, 0x09, 0x15, 0x31, 0x06, 0x04, 0x04, 0x01, 0x00, 0x00, 0x00, 0x30, 0x3D, 0x30, 0x21, 0x30, 0x09,
			0x06, 0x05, 0x2B, 0x0E, 0x03, 0x02, 0x1A, 0x05, 0x00, 0x04, 0x14, 0x64, 0xCC, 0xB9, 0xBC, 0x62, 0x87, 0x49, 0x30, 0x6A, 0x0B, 0x73, 0x21, 0x4C,
			0x16, 0xE9, 0xF6, 0x01, 0x40, 0x61, 0xA0, 0x04, 0x14, 0xF4, 0xA9, 0x6C, 0x88, 0x28, 0x6B, 0x6D, 0x44, 0x1C, 0xA5, 0x29, 0xD1, 0xCB, 0x3B, 0xCF,
			0x7A, 0xF1, 0xE1, 0x50, 0x62, 0x02, 0x02, 0x07, 0xD0
		};

		private static byte[] asn_null = new byte[] { 0x05, 0x00 };
		private static byte[] rsa_public_key = new byte[] { 0x30, 0x5C, 0x02, 0x55, 0x2D, 0x58, 0xE9, 0xBF, 0xF0, 0x31, 0xCD, 0x79, 0x06, 0x50, 0x5A, 0xD5, 0x9E, 0x0E, 0x2C, 0xE6, 0xC2, 0xF7, 0xF9, 0xD2, 0xCE, 0x55, 0x64, 0x85, 0xB1, 0x90, 0x9A, 0x92, 0xB3, 0x36, 0xC1, 0xBC, 0xEA, 0xC8, 0x23, 0xB7, 0xAB, 0x3A, 0xA7, 0x64, 0x63, 0x77, 0x5F, 0x84, 0x22, 0x8E, 0xE5, 0xB6, 0x45, 0xDD, 0x46, 0xAE, 0x0A, 0xDD, 0x00, 0xC2, 0x1F, 0xBA, 0xD9, 0xAD, 0xC0, 0x75, 0x62, 0xF8, 0x95, 0x82, 0xA2, 0x80, 0xB1, 0x82, 0x69, 0xFA, 
			0xE1, 0xAF, 0x7F, 0xBC, 0x7D, 0xE2, 0x7C, 0x76, 0xD5, 0xBC, 0x2A, 0x80, 0xFB, 0x02, 0x03, 0x01, 0x00, 0x01 };
		public const string rsa_public_key_xml = "<RSAKeyValue><Modulus>LVjpv/AxzXkGUFrVng4s5sL3+dLOVWSFsZCakrM2wbzqyCO3qzqnZGN3X4QijuW2Rd1GrgrdAMIfutmtwHVi+JWCooCxgmn64a9/vH3ifHbVvCqA+w==</Modulus><Exponent>AQAB</Exponent></RSAKeyValue>";

		private static byte[] dsa_params = new byte[] { 0x30, 0x82, 0x01, 0x1E, 0x02, 0x81, 0x81, 0x00, 0x90, 0x89, 0x3E, 0x18, 0x1B, 0xFE, 0xA3, 0x1D, 0x16, 0x89, 0x00, 0xB4, 0xD5, 0x40, 0x82, 0x4C, 0x2E, 0xEC, 0x3D, 0x66, 0x0D, 0x0D, 0xB9, 0x17, 0x40, 0x6E, 0x3A, 0x5C, 0x03, 0x7B, 0x1B, 0x93, 0x28, 0x0C, 0xEF, 0xB9, 0x97, 0xE3, 0xA1, 0xEB, 0xE2, 0xA3, 0x7C, 0x61, 0xDD, 0x6F, 0xD5, 0xAD, 0x15, 0x69, 0x00, 0x16, 0xB2, 0xC3, 0x08, 0x3D, 0xC4, 0x59, 0xC6, 0xF2, 0x70, 0xA5, 0xB0, 0xF5, 0x1F, 0x1D, 0xF4, 0xB0, 0x15,
			0xDA, 0x7E, 0x28, 0x39, 0x24, 0x99, 0x36, 0x5B, 0xEC, 0x39, 0x25, 0xFA, 0x92, 0x49, 0x65, 0xD2, 0x43, 0x05, 0x6A, 0x9E, 0xA3, 0x7B, 0xF0, 0xDE, 0xA3, 0x2F, 0xD3, 0x6F, 0x3A, 0xF9, 0x35, 0xC3, 0x29, 0xD4, 0x45, 0x6C, 0x56, 0x9A, 0xDE, 0x36, 0x6E, 0xFE, 0x12, 0x68, 0x96, 0x7B, 0x45, 0x1D, 0x2C, 0xFF, 0xB9, 0x2D, 0xF5, 0x52, 0x8C, 0xDF, 0x3E, 0x2F, 0x63, 0x02, 0x15, 0x00, 0x81, 0xA9, 0xB5, 0xD0, 0x04, 0xF2, 0x9B, 0xA7, 0xD8, 0x55, 0x4C, 0x3B, 0x32, 0xA1, 0x45, 0x32, 0x4F, 0xF5, 0x51, 0xDD, 0x02, 0x81,
			0x80, 0x64, 0x7A, 0x88, 0x0B, 0xF2, 0x3E, 0x91, 0x81, 0x59, 0x9C, 0xF4, 0xEA, 0xC6, 0x7B, 0x0E, 0xBE, 0xEA, 0x05, 0xE8, 0x77, 0xFD, 0x20, 0x34, 0x87, 0xA1, 0xC4, 0x69, 0xF6, 0xC8, 0x8B, 0x19, 0xDA, 0xCD, 0xFA, 0x21, 0x8A, 0x57, 0xA9, 0x7A, 0x26, 0x0A, 0x56, 0xD4, 0xED, 0x4B, 0x1B, 0x7C, 0x70, 0xED, 0xB4, 0xE6, 0x7A, 0x6A, 0xDE, 0xD3, 0x29, 0xE2, 0xE9, 0x9A, 0x33, 0xED, 0x09, 0x8D, 0x9E, 0xDF, 0xDA, 0x2E, 0x4A, 0xC1, 0x50, 0x92, 0xEE, 0x2F, 0xE5, 0x5A, 0xF3, 0x85, 0x62, 0x6A, 0x48, 0xDC, 0x1B, 0x02,
			0x98, 0xA6, 0xB0, 0xD1, 0x09, 0x4B, 0x10, 0xD1, 0xF0, 0xFA, 0xE0, 0xB1, 0x1D, 0x13, 0x54, 0x4B, 0xC0, 0xA8, 0x40, 0xEF, 0x71, 0xE8, 0x56, 0x6B, 0xA2, 0x29, 0xCB, 0x1E, 0x09, 0x7D, 0x27, 0x39, 0x91, 0x3B, 0x20, 0x4F, 0x98, 0x39, 0xE8, 0x39, 0xCA, 0x98, 0xC5, 0xAF, 0x54 };
		private static byte[] dsa_public_key = new byte[] { 0x02, 0x81, 0x80, 0x54, 0xA8, 0x88, 0xB5, 0x8F, 0x01, 0x56, 0xCE, 0x18, 0x8F, 0xA6, 0xD6, 0x7C, 0x29, 0x29, 0x75, 0x45, 0xE8, 0x31, 0xA4, 0x07, 0x17, 0xED, 0x1E, 0x5D, 0xB2, 0x7B, 0xBB, 0xCE, 0x3C, 0x97, 0x67, 0x1E, 0x88, 0x0A, 0xFE, 0x7D, 0x00, 0x22, 0x27, 0x1D, 0x66, 0xEE, 0xF6, 0x1B, 0xB6, 0x95, 0x7F, 0x5A, 0xFF, 0x06, 0x34, 0x02, 0x43, 0xC3, 0x83, 0xC4, 0x66, 0x2C, 0xA1, 0x05, 0x0E, 0x68, 0xB3, 0xCA, 0xDC, 0xD3, 0xF9, 0x0C, 0xC0, 0x66, 0xDF, 0x85, 
			0x84, 0x4B, 0x20, 0x5D, 0x41, 0xAC, 0xC0, 0xEC, 0x37, 0x92, 0x0E, 0x97, 0x19, 0xBF, 0x53, 0x35, 0x63, 0x27, 0x18, 0x33, 0x35, 0x42, 0x4D, 0xF0, 0x2D, 0x6D, 0xA7, 0xA4, 0x98, 0xAA, 0x57, 0xF3, 0xD2, 0xB8, 0x6E, 0x4E, 0x8F, 0xFF, 0xBE, 0x6F, 0x4E, 0x0F, 0x0B, 0x44, 0x24, 0xEE, 0xDF, 0x4C, 0x22, 0x5B, 0x44, 0x98, 0x94, 0xCB, 0xB8 };
		public const string dsa_public_key_xml = "<DSAKeyValue><P>kIk+GBv+ox0WiQC01UCCTC7sPWYNDbkXQG46XAN7G5MoDO+5l+Oh6+KjfGHdb9WtFWkAFrLDCD3EWcbycKWw9R8d9LAV2n4oOSSZNlvsOSX6kkll0kMFap6je/Deoy/Tbzr5NcMp1EVsVpreNm7+EmiWe0UdLP+5LfVSjN8+L2M=</P><Q>gam10ATym6fYVUw7MqFFMk/1Ud0=</Q><G>ZHqIC/I+kYFZnPTqxnsOvuoF6Hf9IDSHocRp9siLGdrN+iGKV6l6JgpW1O1LG3xw7bTmemre0yni6Zoz7QmNnt/aLkrBUJLuL+Va84ViakjcGwKYprDRCUsQ0fD64LEdE1RLwKhA73HoVmuiKcseCX0nOZE7IE+YOeg5ypjFr1Q=</G><Y>VKiItY8BVs4Yj6bWfCkpdUXoMaQHF+0eXbJ7u848l2ceiAr+fQAiJx1m7vYbtpV/Wv8GNAJDw4PEZiyhBQ5os8rc0/kMwGbfhYRLIF1BrMDsN5IOlxm/UzVjJxgzNUJN8C1tp6SYqlfz0rhuTo//vm9ODwtEJO7fTCJbRJiUy7g=</Y></DSAKeyValue>";

		private X509Certificate2 x509a;
		private X509Certificate2 x509b;
		private X509Certificate2 x509c;

		[TestFixtureSetUp]
		public void FixtureSetUp ()
		{
			x509a = new X509Certificate2 (cert_a);
			x509b = new X509Certificate2 (cert_b);
			x509c = new X509Certificate2 (cert_c, "mono");
		}

		[Test]
		[ExpectedException (typeof (ArgumentNullException))]
		public void Constructor_NullOid ()
		{
			new PublicKey (null, emptyData, emptyData);
		}

		[Test]
		[ExpectedException (typeof (ArgumentNullException))]
		public void Constructor_NullParameters ()
		{
			new PublicKey (rsaOid, null, emptyData);
		}

		[Test]
		[ExpectedException (typeof (ArgumentNullException))]
		public void Constructor_NullKeyValue ()
		{
			new PublicKey (rsaOid, emptyData, null);
		}

		[Test]
		public void Constructor_RsaEmpty ()
		{
			PublicKey pk = new PublicKey (rsaOid, emptyData, emptyData);

			Assert.IsFalse (Object.ReferenceEquals (rsaOid, pk.Oid), "Oid-reference");
			Assert.AreEqual (rsaOid.Value, pk.Oid.Value, "Oid-Value");

			Assert.IsFalse (Object.ReferenceEquals (emptyData, pk.EncodedKeyValue), "EncodedKeyValue-reference");
			Assert.AreEqual (0, pk.EncodedKeyValue.RawData.Length, "EncodedKeyValue-RawData");
			Assert.IsNull (pk.EncodedKeyValue.Oid, "EncodedKeyValue-Oid");

			Assert.IsFalse (Object.ReferenceEquals (emptyData, pk.EncodedParameters), "EncodedParameters-reference");
			Assert.AreEqual (0, pk.EncodedParameters.RawData.Length, "EncodedParameters-RawData");
			Assert.IsNull (pk.EncodedParameters.Oid, "EncodedParameters-Oid");
		}

		[Test]
		[ExpectedException (typeof (CryptographicException))]
		public void Constructor_RsaEmpty_Key ()
		{
			PublicKey pk = new PublicKey (rsaOid, emptyData, emptyData);
			Assert.IsNull (pk.Key);
		}

		[Test]
		public void Constructor_UnknownEmpty ()
		{
			PublicKey pk = new PublicKey (unknownOid, emptyData, emptyData);
			Assert.IsFalse (Object.ReferenceEquals (unknownOid, pk.Oid), "Oid-reference");
			Assert.AreEqual (unknownOid.Value, pk.Oid.Value, "Oid-Value");
		}

		[Test]
		[ExpectedException (typeof (NotSupportedException))]
		public void Constructor_UnknownEmpty_Key ()
		{
			PublicKey pk = new PublicKey (unknownOid, emptyData, emptyData);
			Assert.IsNotNull (pk.Key);
		}

		[Test]
		public void Constructor_Rsa_FromCertificate ()
		{
			PublicKey pk1 = x509a.PublicKey;
			PublicKey pk2 = new PublicKey (pk1.Oid, pk1.EncodedParameters, pk1.EncodedKeyValue);

			Assert.AreEqual (rsaOid.Value, pk2.Oid.Value, "Oid.Value");
			Assert.AreEqual (rsaOid.Value, pk2.EncodedParameters.Oid.Value, "EncodedParameters.Oid.Value");
			Assert.AreEqual ("05-00", BitConverter.ToString (pk2.EncodedParameters.RawData), "EncodedParameters.RawData");
			Assert.AreEqual (rsaOid.Value, pk2.EncodedKeyValue.Oid.Value, "EncodedKeyValue.Oid.Value");
			Assert.AreEqual (BitConverter.ToString (rsa_public_key), BitConverter.ToString (pk2.EncodedKeyValue.RawData), "EncodedKeyValue.RawData");
			Assert.AreEqual (rsa_public_key_xml, pk2.Key.ToXmlString (false), "Key");
			Assert.IsTrue ((pk2.Key as RSACryptoServiceProvider).PublicOnly, "Key.PublicOnly");
		}

		[Test]
		public void Constructor_Rsa_FromScratch ()
		{
			// providing Oid for parameters and keyvalue isn't required
			PublicKey pk = new PublicKey (new Oid (rsaOid.Value),
				new AsnEncodedData (asn_null),
				new AsnEncodedData (rsa_public_key));

			Assert.AreEqual (rsaOid.Value, pk.Oid.Value, "Oid.Value");
			Assert.IsNull (pk.EncodedParameters.Oid, "EncodedParameters.Oid");
			Assert.AreEqual ("05-00", BitConverter.ToString (pk.EncodedParameters.RawData), "EncodedParameters.RawData");
			Assert.IsNull (pk.EncodedKeyValue.Oid, "EncodedKeyValue.Oid");
			Assert.AreEqual (BitConverter.ToString (rsa_public_key), BitConverter.ToString (pk.EncodedKeyValue.RawData), "EncodedKeyValue.RawData");
			Assert.AreEqual (rsa_public_key_xml, pk.Key.ToXmlString (false), "Key");
			Assert.IsTrue ((pk.Key as RSACryptoServiceProvider).PublicOnly, "Key.PublicOnly");
		}

		[Test]
		public void Constructor_Rsa_EmptyParameters ()
		{
			PublicKey pk = new PublicKey (new Oid (rsaOid.Value),
				new AsnEncodedData (new byte [0]), // same as NULL (0x05, 0x00)
				new AsnEncodedData (rsa_public_key));

			Assert.AreEqual (rsaOid.Value, pk.Oid.Value, "Oid.Value");
			Assert.IsNull (pk.EncodedParameters.Oid, "EncodedParameters.Oid");
			Assert.AreEqual (String.Empty, BitConverter.ToString (pk.EncodedParameters.RawData), "EncodedParameters.RawData");
			Assert.IsNull (pk.EncodedKeyValue.Oid, "EncodedKeyValue.Oid");
			Assert.AreEqual (BitConverter.ToString (rsa_public_key), BitConverter.ToString (pk.EncodedKeyValue.RawData), "EncodedKeyValue.RawData");
			Assert.AreEqual (rsa_public_key_xml, pk.Key.ToXmlString (false), "Key");
			Assert.IsTrue ((pk.Key as RSACryptoServiceProvider).PublicOnly, "Key.PublicOnly");
		}

		[Test]
		public void Constructor_Rsa_WeirdParameters ()
		{
			PublicKey pk = new PublicKey (new Oid (rsaOid.Value),
				new AsnEncodedData (new byte[16] { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F }),
				new AsnEncodedData (rsa_public_key));

			Assert.AreEqual (rsaOid.Value, pk.Oid.Value, "Oid.Value");
			Assert.IsNull (pk.EncodedParameters.Oid, "EncodedParameters.Oid");
			Assert.AreEqual ("00-01-02-03-04-05-06-07-08-09-0A-0B-0C-0D-0E-0F", BitConverter.ToString (pk.EncodedParameters.RawData), "EncodedParameters.RawData");
			Assert.IsNull (pk.EncodedKeyValue.Oid, "EncodedKeyValue.Oid");
			Assert.AreEqual (BitConverter.ToString (rsa_public_key), BitConverter.ToString (pk.EncodedKeyValue.RawData), "EncodedKeyValue.RawData");
			// weird (or any) parameters are ignored when decoding RSA public keys
			Assert.AreEqual (rsa_public_key_xml, pk.Key.ToXmlString (false), "Key");
			Assert.IsTrue ((pk.Key as RSACryptoServiceProvider).PublicOnly, "Key.PublicOnly");
		}

		[Test]
		public void Constructor_Rsa_UnknownOid ()
		{
			// providing Oid for parameters and keyvalue isn't required
			PublicKey pk = new PublicKey (new Oid (unknownOid.Value),
				new AsnEncodedData (asn_null),
				new AsnEncodedData (rsa_public_key));

			Assert.AreEqual (unknownOid.Value, pk.Oid.Value, "Oid.Value");
			Assert.IsNull (pk.EncodedParameters.Oid, "EncodedParameters.Oid");
			Assert.AreEqual ("05-00", BitConverter.ToString (pk.EncodedParameters.RawData), "EncodedParameters.RawData");
			Assert.IsNull (pk.EncodedKeyValue.Oid, "EncodedKeyValue.Oid");
			Assert.AreEqual (BitConverter.ToString (rsa_public_key), BitConverter.ToString (pk.EncodedKeyValue.RawData), "EncodedKeyValue.RawData");
		}

		[Test]
		[ExpectedException (typeof (NotSupportedException))]
		public void Constructor_Rsa_UnknownOid_Key ()
		{
			// providing Oid for parameters and keyvalue isn't required
			PublicKey pk = new PublicKey (new Oid (unknownOid.Value),
				new AsnEncodedData (asn_null),
				new AsnEncodedData (rsa_public_key));

			Assert.AreEqual (rsa_public_key_xml, pk.Key.ToXmlString (false), "Key");
		}

		[Test]
		public void Constructor_Dsa_FromCertificate ()
		{
			PublicKey pk1 = x509b.PublicKey;
			PublicKey pk2 = new PublicKey (pk1.Oid, pk1.EncodedParameters, pk1.EncodedKeyValue);

			Assert.AreEqual (dsaOid.Value, pk2.Oid.Value, "Oid.Value");
			Assert.AreEqual (dsaOid.Value, pk2.EncodedParameters.Oid.Value, "EncodedParameters.Oid.Value");
			Assert.AreEqual (BitConverter.ToString (dsa_params), BitConverter.ToString (pk2.EncodedParameters.RawData), "EncodedParameters.RawData");
			Assert.AreEqual (dsaOid.Value, pk2.EncodedKeyValue.Oid.Value, "EncodedKeyValue.Oid.Value");
			Assert.AreEqual (BitConverter.ToString (dsa_public_key), BitConverter.ToString (pk2.EncodedKeyValue.RawData), "EncodedKeyValue.RawData");
			Assert.AreEqual (dsa_public_key_xml, pk2.Key.ToXmlString (false), "Key");
			Assert.IsTrue ((pk2.Key as DSACryptoServiceProvider).PublicOnly, "Key.PublicOnly");
		}

		[Test]
		public void Constructor_Dsa_FromScratch ()
		{
			// providing Oid for parameters and keyvalue isn't required
			PublicKey pk = new PublicKey (new Oid (dsaOid.Value),
				new AsnEncodedData (dsa_params),
				new AsnEncodedData (dsa_public_key));

			Assert.AreEqual (dsaOid.Value, pk.Oid.Value, "Oid.Value");
			Assert.IsNull (pk.EncodedParameters.Oid, "EncodedParameters.Oid");
			Assert.AreEqual (BitConverter.ToString (dsa_params), BitConverter.ToString (pk.EncodedParameters.RawData), "EncodedParameters.RawData");
			Assert.IsNull (pk.EncodedKeyValue.Oid, "EncodedKeyValue.Oid");
			Assert.AreEqual (BitConverter.ToString (dsa_public_key), BitConverter.ToString (pk.EncodedKeyValue.RawData), "EncodedKeyValue.RawData");
			Assert.AreEqual (dsa_public_key_xml, pk.Key.ToXmlString (false), "Key");
			Assert.IsTrue ((pk.Key as DSACryptoServiceProvider).PublicOnly, "Key.PublicOnly");
		}

		[Test]
		public void Constructor_Dsa_EmptyParameters ()
		{
			PublicKey pk = new PublicKey (new Oid (dsaOid.Value),
				new AsnEncodedData (new byte[0]), // same as NULL (0x05, 0x00)
				new AsnEncodedData (dsa_public_key));

			Assert.AreEqual (dsaOid.Value, pk.Oid.Value, "Oid.Value");
			Assert.IsNull (pk.EncodedParameters.Oid, "EncodedParameters.Oid");
			Assert.AreEqual (String.Empty, BitConverter.ToString (pk.EncodedParameters.RawData), "EncodedParameters.RawData");
			Assert.IsNull (pk.EncodedKeyValue.Oid, "EncodedKeyValue.Oid");
			Assert.AreEqual (BitConverter.ToString (dsa_public_key), BitConverter.ToString (pk.EncodedKeyValue.RawData), "EncodedKeyValue.RawData");
		}

		[Test]
		[ExpectedException (typeof (CryptographicException))]
		public void Constructor_Dsa_EmptyParameters_Key ()
		{
			PublicKey pk = new PublicKey (new Oid (dsaOid.Value),
				new AsnEncodedData (new byte[0]), // same as NULL (0x05, 0x00)
				new AsnEncodedData (dsa_public_key));

			Assert.AreEqual (dsa_public_key_xml, pk.Key.ToXmlString (false), "Key");
		}

		[Test]
		public void Constructor_Dsa_WeirdParameters ()
		{
			PublicKey pk = new PublicKey (new Oid (dsaOid.Value),
				new AsnEncodedData (new byte[16] { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F }),
				new AsnEncodedData (dsa_public_key));

			Assert.AreEqual (dsaOid.Value, pk.Oid.Value, "Oid.Value");
			Assert.IsNull (pk.EncodedParameters.Oid, "EncodedParameters.Oid");
			Assert.AreEqual ("00-01-02-03-04-05-06-07-08-09-0A-0B-0C-0D-0E-0F", BitConverter.ToString (pk.EncodedParameters.RawData), "EncodedParameters.RawData");
			Assert.IsNull (pk.EncodedKeyValue.Oid, "EncodedKeyValue.Oid");
			Assert.AreEqual (BitConverter.ToString (dsa_public_key), BitConverter.ToString (pk.EncodedKeyValue.RawData), "EncodedKeyValue.RawData");
		}

		[Test]
		[ExpectedException (typeof (CryptographicException))]
		public void Constructor_Dsa_WeirdParameters_Key ()
		{
			PublicKey pk = new PublicKey (new Oid (dsaOid.Value),
				new AsnEncodedData (new byte[16] { 0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E, 0x0F }),
				new AsnEncodedData (dsa_public_key));

			Assert.AreEqual (dsa_public_key_xml, pk.Key.ToXmlString (false), "Key");
		}

		[Test]
		public void Constructor_Dsa_UnknownOid ()
		{
			// providing Oid for parameters and keyvalue isn't required
			PublicKey pk = new PublicKey (new Oid (unknownOid.Value),
				new AsnEncodedData (dsa_params),
				new AsnEncodedData (dsa_public_key));

			Assert.AreEqual (unknownOid.Value, pk.Oid.Value, "Oid.Value");
			Assert.IsNull (pk.EncodedParameters.Oid, "EncodedParameters.Oid");
			Assert.AreEqual (BitConverter.ToString (dsa_params), BitConverter.ToString (pk.EncodedParameters.RawData), "EncodedParameters.RawData");
			Assert.IsNull (pk.EncodedKeyValue.Oid, "EncodedKeyValue.Oid");
			Assert.AreEqual (BitConverter.ToString (dsa_public_key), BitConverter.ToString (pk.EncodedKeyValue.RawData), "EncodedKeyValue.RawData");
		}

		[Test]
		[ExpectedException (typeof (NotSupportedException))]
		public void Constructor_Dsa_UnknownOid_Key ()
		{
			// providing Oid for parameters and keyvalue isn't required
			PublicKey pk = new PublicKey (new Oid (unknownOid.Value),
				new AsnEncodedData (dsa_params),
				new AsnEncodedData (dsa_public_key));

			Assert.AreEqual (dsa_public_key_xml, pk.Key.ToXmlString (false), "Key");
		}

		[Test]
		public void Rsa_PasswordProtectedPkcs12 ()
		{
			Assert.IsTrue (x509c.HasPrivateKey, "HasPrivateKey");
			// the public key doesn't contain the private parts of the key
			Assert.IsTrue ((x509c.PublicKey.Key as RSACryptoServiceProvider).PublicOnly, "Key.PublicOnly");
		}
	}
}

#endif
