"""Tools for handling docstrings."""
# Copyright (C) TeNPy Developers, Apache license

__all__ = ['amend_parent_docstring']


def amend_parent_docstring(parent, insert_at: str = 'Parameters\n', insert_before: bool = True):
    """Return a decorator that facilitates docstring generation.

    The docstring of the decorated function is generated by amending the docstring that is
    explicitly defined to the docstring of a "parent" function/method.

    .. note ::
        No additional whitespace is created! Any desired additional spaces or linebreaks should be
        defined in the amending part, i.e. the explicit docstring of the decorated function in the code

    Parameters
    ----------
    parent
        The parent "function" (or really any object that has a ``__doc__`` attribute).
    insert_at : str | None
        A search string for ``parent.__doc__``. The amending part will be inserted before or after
        (see `insert_before` below) the first occurance of this string.
        If `insert_at` does not occur in ``parent.__doc__``, or if ``insert_at is None``, it will
        be inserted at the very end.
        Note that an empty string can be used to achieve insertion at the very beginning.
    insert_before : bool
        If the amending part will be inserted before ``insert_at``. Otherwise, after.

    """
    parent_doc = getattr(parent, '__doc__', None)
    if parent_doc is None:
        parent_name = getattr(parent, '__name__', 'unnamed')
        raise ValueError(f'Parent function {parent_name} has no docstring')

    if insert_at is None:
        idx = -1
    else:
        try:
            idx = parent_doc.index(insert_at)
        except ValueError:
            idx = -1
    if idx != -1 and not insert_before:
        idx = idx + len(insert_at)

    def decorator(func):
        amending_doc = getattr(func, '__doc__', None)
        if amending_doc is None:
            raise ValueError('Decorated function has no docstring')
        func.__doc__ = parent_doc[:idx] + amending_doc + parent_doc[idx:]
        return func

    return decorator
