Testomat functions gives you more flexibility in reporting and make your reports more powerful.

Functions can be used as ESM or CommonJS module. Use `require` instead of `import` in CommonJS.

```js
// inside TypeScript or JavaScript ESM
import { artifact, log, meta } from '@testomatio/reporter';

// inside CommonJS
const { artifact, log, meta } = require('@testomatio/reporter');
```

### Usage example

```javascript
import { artifact, log, meta } from '@testomatio/reporter';

test('my test', async () => {
  meta('ISSUE', 'MY-123');
  await page.login();
  log`I was logged in with user ${user}`;
  artifact(await saveScreenshot());
  assert(something);
});
```

After you import and invoke `testomat`, autocompletion will help you to find the right function.

### Available functions

- [artifact](#artifact)
- [log](#log)
- [step](#step)
- [meta (key:value)](#meta)
- [label](#label)
- [linkTest](#linktest)
- [linkJira](#linkjira)

### Artifact

Adds file to the test report (text, image, video, etc.)

```javascript
import { artifact } from '@testomatio/reporter';

test('my test', async () => {
  const pathToFile = await saveScreenshot();
  artifact(pathToFile);
});
```

> Artifacts, generated by testrunners (like screenshots/videos by Playwright) are uploaded automatically, you don't need to add them manually using artifact function. Just setup [S3 bucket](./artifacts.md).

### Log

Similar to [step](#step) function, intended to log any additional info to the test report (including text, arrays, complex objects).

##### Usage examples:

```javascript
import { log } from '@testomatio/reporter';

test('my test', async () => {
  log('your message');
  log(`your message ${variable}`);
  log('your message', variable1, variable2);
});
```

```javascript
import { log } from '@testomatio/reporter';

test('Your test @T12345678', async () => {
  await page.login(user);

  log`I was logged in with user ${user}`;
  assert(loggedIn);

  log`I was logged in with user ${user}`; // < shorter syntax
});
```

### Step

Adds step to the test report. Step is a human-readable description of the test action. It is used to describe the test flow. This function is similar to [log](#log) function, but looks differently in the report.

```javascript
import { step } from '@testomatio/reporter';

describe('Your suite @S12345678', () => {
  test('Your test @T12345678', async () => {
    await page.login();
    step`Login successful`;
    assert(something);
  });
});
```

### Meta

Meta information is a key:value pair(s), which is used to add additional information to the test report. E.g. browser, environment, etc.

```js
import { meta } from '@testomatio/reporter';

test('my test', () => {
  // use it inside tests as key, value
  meta('browser', 'chrome');

  // or use it as an object
  meta({
    browser: 'chrome',
    server: 'staging',
  });
});
```

Or in CommonJS style:

```javascript
const { meta } = require('@testomatio/reporter');

test('Your test @T12345678', async () => {
  await page.login();
  testomat.meta({
    browser: 'chrome',
    testType: 'smoke',
  });
  assert(something);
});
```

## Label

Adds a label to the reported test. Unlike `meta` label will be persisted to the test case itself, not just to reported run. If the label does not exist in Testomat.io, it will be automatically created and linked to the test case during the test run, or you can use existing labels in Testomat.io. You can pass also a label value, if the label was created as a custom field.

```javascript
import { label } from '@testomatio/reporter';

describe('Your suite', () => {
  test('I can login', async () => {
    label('Area', 'Auth')
    label('Severity', 'High')
    label('Browser')
    await page.login();
  });
});
```

## LinkTest

Links test IDs to the current test in the report. This allows you to associate multiple test cases with the current test execution.

```javascript
import { linkTest } from '@testomatio/reporter';

test('my test', async () => {
  // Link single test ID
  linkTest('f2916e65');
  
  // Link multiple test IDs
  linkTest('@T888ef960', '@Tf2916e65');
  
  await page.login();
  assert(something);
});
```

## LinkJira

Links JIRA issue IDs to the test report. This creates a connection between your test execution and JIRA issues.

```javascript
import { linkJira } from '@testomatio/reporter';

test('my test', async () => {
  // Link single JIRA issue
  linkJira('PROJ-123');
  
  // Link multiple JIRA issues
  linkJira('PROJ-456', 'PROJ-789');
  
  await page.login();
  assert(something);
});
```

---

Supported frameworks:

- 🟢 CodeceptJS
- 🟢 Cucumber
- 🟢 Jest
- 🟢 Mocha
- 🟢 Playwright (optional, use native functionality)
- 🟢 WDIO (everything, except artifacts)
