module Articles.GenderAndSalery.Data exposing (Datum, womenPerc, womenSalaryPerc, menSalaryPerc, groupBy, data, avgSalaryWomen, avgSalaryMen)

import Csv.Decode as Cvs
import Dict


type alias Datum =
  { year : Float
  , sector : String
  , numOfBoth : Float
  , numOfWomen : Float
  , numOfMen : Float
  , salaryBoth : Float
  , salaryWomen : Float
  , salaryMen : Float
  }


womenPerc : Datum -> Float
womenPerc datum =
  if datum.numOfBoth == 0 || datum.numOfWomen == 0 then 0
  else datum.numOfWomen / datum.numOfBoth * 100


womenSalaryPerc : Datum -> Maybe Float
womenSalaryPerc datum =
  if datum.salaryWomen == 0 || datum.salaryMen == 0 then Nothing
  else Just (datum.salaryWomen / datum.salaryMen * 100)


menSalaryPerc : Datum -> Maybe Float
menSalaryPerc datum =
  if datum.salaryWomen == 0 || datum.salaryMen == 0 then Nothing
  else Just (datum.salaryMen / datum.salaryWomen * 100)


groupBy : (Datum -> comparable) -> List Datum -> List ( comparable, List Datum )
groupBy toV =
  let fold datum =
        Dict.update (toV datum) (Maybe.map (\l -> l ++ [datum]) >> Maybe.withDefault [datum] >> Just)
  in
  List.foldl fold Dict.empty >> Dict.toList


avgSalaryWomen : Float -> Float
avgSalaryWomen year =
  let yearData =
        data
          |> List.filter (\d -> d.year == year)

      totalSalary =
        yearData
          |> List.map (\d -> d.salaryWomen * d.numOfWomen)
          |> List.sum

      totalWomen =
        yearData
          |> List.map .numOfWomen
          |> List.sum
  in
  totalSalary / totalWomen


avgSalaryMen : Float -> Float
avgSalaryMen year =
  let yearData =
        data
          |> List.filter (\d -> d.year == year)

      totalSalary =
        yearData
          |> List.map (\d -> d.salaryMen * d.numOfMen)
          |> List.sum

      totalMen =
        yearData
          |> List.map .numOfMen
          |> List.sum
  in
  totalSalary / totalMen



-- DATA


data : List Datum
data =
  Cvs.decodeCustom { fieldSeparator = ';' } Cvs.FieldNamesFromFirstRow decoder csv
    |> Result.map process
    |> Result.withDefault []


process : List Raw -> List Datum
process =
  let fold acc list =
        case list of
          a :: b :: c :: d :: e :: f :: rest ->
            case processSet a b c d e f of
              Just datum -> fold (datum :: acc) rest
              Nothing -> fold acc rest

          _ ->
            acc

      processSet a b c d e f =
        let numbers =
              [ danishNumber a.year
              , danishNumber a.contents
              , danishNumber b.contents
              , danishNumber c.contents
              , danishNumber d.contents
              , danishNumber e.contents
              , danishNumber f.contents
              ]
        in
        case numbers of
          [ Just year, Just salaryBoth, Just salaryMen, Just salaryWomen, Just numOfBoth, Just numOfMen, Just numOfWomen ] ->
            { year = year
            , sector = a.work
            , numOfBoth = numOfBoth
            , numOfWomen = numOfWomen
            , numOfMen = numOfMen
            , salaryBoth = salaryBoth
            , salaryWomen = salaryWomen
            , salaryMen = salaryMen
            } |> Just

          _ ->
            Nothing
  in
  fold [] >> List.reverse


danishNumber : String -> Maybe Float
danishNumber str =
  case str of
    ".." -> Just 0
    _ -> String.toFloat (String.replace "," "." str)


type alias Raw =
  { group : String
  , sector : String
  , year : String
  , work : String
  , salary : String
  , gender : String
  , contents : String
  }


decoder : Cvs.Decoder Raw
decoder =
  Cvs.into Raw
    |> Cvs.pipeline (Cvs.field "LONGRP" Cvs.string)
    |> Cvs.pipeline (Cvs.field "SEKTOR" Cvs.string)
    |> Cvs.pipeline (Cvs.field "TID" Cvs.string)
    |> Cvs.pipeline (Cvs.field "ARBF" Cvs.string)
    |> Cvs.pipeline (Cvs.field "LØNMÅL" Cvs.string)
    |> Cvs.pipeline (Cvs.field "KØN" Cvs.string)
    |> Cvs.pipeline (Cvs.field "INDHOLD" Cvs.string)


csv : String
csv =
  "LONGRP;SEKTOR;TID;ARBF;LØNMÅL;KØN;INDHOLD\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;011 Militært arbejde på officersniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;48109,10\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;011 Militært arbejde på officersniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;48386,91\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;011 Militært arbejde på officersniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;43360,89\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;011 Militært arbejde på officersniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4258,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;011 Militært arbejde på officersniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4012,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;011 Militært arbejde på officersniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;246,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;35577,26\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;35701,33\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33226,36\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4326,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4100,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;225,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;031 Militært arbejde, øvrige rangordener;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;30500,45\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;031 Militært arbejde, øvrige rangordener;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;30607,84\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;031 Militært arbejde, øvrige rangordener;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29060,11\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;031 Militært arbejde, øvrige rangordener;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;7859,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;031 Militært arbejde, øvrige rangordener;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;7303,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;031 Militært arbejde, øvrige rangordener;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;557,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;70542,57\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;73307,96\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;64805,58\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2100,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1431,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;670,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;112 Øverste virksomhedsledelse;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;95836,05\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;112 Øverste virksomhedsledelse;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;97971,09\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;112 Øverste virksomhedsledelse;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;80005,27\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;112 Øverste virksomhedsledelse;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;11818,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;112 Øverste virksomhedsledelse;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;10395,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;112 Øverste virksomhedsledelse;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1423,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;121 Ledelse inden for forretningsservice og administrative funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;65998,49\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;121 Ledelse inden for forretningsservice og administrative funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;69024,66\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;121 Ledelse inden for forretningsservice og administrative funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;60606,73\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;121 Ledelse inden for forretningsservice og administrative funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;17627,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;121 Ledelse inden for forretningsservice og administrative funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;11235,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;121 Ledelse inden for forretningsservice og administrative funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;6391,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;122 Ledelse af forretnings- og udviklingsorienterede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;67270,61\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;122 Ledelse af forretnings- og udviklingsorienterede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;69985,85\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;122 Ledelse af forretnings- og udviklingsorienterede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;58382,03\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;122 Ledelse af forretnings- og udviklingsorienterede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;10837,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;122 Ledelse af forretnings- og udviklingsorienterede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;8269,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;122 Ledelse af forretnings- og udviklingsorienterede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2568,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;56598,90\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;55704,42\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;174,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;148,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;26,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;59059,34\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;59117,54\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;58574,59\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;8308,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;7401,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;907,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;69501,18\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;69730,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;67921,43\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2360,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2039,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;322,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;134 Ledelse af hovedaktiviteten inden for servicefag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;53296,08\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;134 Ledelse af hovedaktiviteten inden for servicefag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;60008,52\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;134 Ledelse af hovedaktiviteten inden for servicefag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;47496,71\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;134 Ledelse af hovedaktiviteten inden for servicefag;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;14121,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;134 Ledelse af hovedaktiviteten inden for servicefag;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6313,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;134 Ledelse af hovedaktiviteten inden for servicefag;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7808,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;47045,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;47544,87\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;46303,64\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;354,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;209,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;145,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;40871,59\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;43794,97\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;34906,02\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5182,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3440,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1741,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;143 Ledelse af hovedaktiviteten inden for andre serviceområder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;53743,41\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;143 Ledelse af hovedaktiviteten inden for andre serviceområder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;54882,55\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;143 Ledelse af hovedaktiviteten inden for andre serviceområder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;50780,49\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;143 Ledelse af hovedaktiviteten inden for andre serviceområder;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1366,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;143 Ledelse af hovedaktiviteten inden for andre serviceområder;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;986,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;143 Ledelse af hovedaktiviteten inden for andre serviceområder;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;380,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;211 Arbejde inden for fysik og geologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;52607,79\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;211 Arbejde inden for fysik og geologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;54115,45\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;211 Arbejde inden for fysik og geologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;50022,77\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;211 Arbejde inden for fysik og geologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2174,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;211 Arbejde inden for fysik og geologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1365,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;211 Arbejde inden for fysik og geologi;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;809,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;59839,21\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;61604,91\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;55833,18\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1185,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;812,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;374,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;213 Arbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;47259,84\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;213 Arbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;48746,43\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;213 Arbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;45297,32\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;213 Arbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4184,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;213 Arbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2314,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;213 Arbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1871,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;214 Ingeniørarbejde (undtagen inden for elektroteknologi);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;54074,18\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;214 Ingeniørarbejde (undtagen inden for elektroteknologi);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;55039,60\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;214 Ingeniørarbejde (undtagen inden for elektroteknologi);STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;50322,87\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;214 Ingeniørarbejde (undtagen inden for elektroteknologi);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;30681,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;214 Ingeniørarbejde (undtagen inden for elektroteknologi);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;24012,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;214 Ingeniørarbejde (undtagen inden for elektroteknologi);Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;6669,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;215 Ingeniørarbejde inden for elektroteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;55752,61\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;215 Ingeniørarbejde inden for elektroteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;55905,53\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;215 Ingeniørarbejde inden for elektroteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;53589,24\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;215 Ingeniørarbejde inden for elektroteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5524,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;215 Ingeniørarbejde inden for elektroteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;5142,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;215 Ingeniørarbejde inden for elektroteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;382,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;216 Arbejde med arkitektur, infrastruktur og design;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;43051,37\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;216 Arbejde med arkitektur, infrastruktur og design;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;46130,12\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;216 Arbejde med arkitektur, infrastruktur og design;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;39803,06\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;216 Arbejde med arkitektur, infrastruktur og design;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6070,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;216 Arbejde med arkitektur, infrastruktur og design;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3078,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;216 Arbejde med arkitektur, infrastruktur og design;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2992,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;221 Lægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;67447,31\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;221 Lægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;72643,21\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;221 Lægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;61820,14\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;221 Lægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;15728,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;221 Lægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;7870,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;221 Lægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7858,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;222 Sygeplejerske- og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37516,58\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;222 Sygeplejerske- og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;40363,22\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;222 Sygeplejerske- og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;37383,47\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;222 Sygeplejerske- og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;50503,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;222 Sygeplejerske- og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2139,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;222 Sygeplejerske- og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;48364,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;223 Arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;223 Arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;223 Arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;223 Arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;9,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;223 Arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;223 Arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;6,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;224 Paramedicinsk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;40045,65\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;224 Paramedicinsk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;40016,55\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;224 Paramedicinsk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;224 Paramedicinsk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;304,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;224 Paramedicinsk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;286,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;224 Paramedicinsk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;18,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;225 Dyrlægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;48865,36\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;225 Dyrlægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;51931,36\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;225 Dyrlægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;46857,82\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;225 Dyrlægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;705,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;225 Dyrlægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;263,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;225 Dyrlægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;442,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;226 Andet sundhedsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;42802,66\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;226 Andet sundhedsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;51218,71\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;226 Andet sundhedsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;40749,90\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;226 Andet sundhedsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;20488,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;226 Andet sundhedsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3682,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;226 Andet sundhedsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;16807,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;231 Undervisning og forskning ved universiteter og højere læreanstalter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;47434,69\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;231 Undervisning og forskning ved universiteter og højere læreanstalter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;49129,78\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;231 Undervisning og forskning ved universiteter og højere læreanstalter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;45110,05\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;231 Undervisning og forskning ved universiteter og højere læreanstalter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;17526,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;231 Undervisning og forskning ved universiteter og højere læreanstalter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;10019,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;231 Undervisning og forskning ved universiteter og højere læreanstalter;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7507,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;232 Undervisning ved erhvervsuddannelser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;40218,34\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;232 Undervisning ved erhvervsuddannelser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;40518,15\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;232 Undervisning ved erhvervsuddannelser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;39821,71\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;232 Undervisning ved erhvervsuddannelser;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;11439,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;232 Undervisning ved erhvervsuddannelser;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6420,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;232 Undervisning ved erhvervsuddannelser;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;5019,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;233 Almen undervisning efter grundskoleniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;44967,83\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;233 Almen undervisning efter grundskoleniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;45644,96\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;233 Almen undervisning efter grundskoleniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;44321,09\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;233 Almen undervisning efter grundskoleniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;14143,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;233 Almen undervisning efter grundskoleniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6785,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;233 Almen undervisning efter grundskoleniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7358,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;234 Undervisning og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;35864,11\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;234 Undervisning og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;37267,19\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;234 Undervisning og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;35351,64\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;234 Undervisning og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;103697,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;234 Undervisning og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;26565,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;234 Undervisning og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;77132,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;235 Andet undervisnings- og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37173,06\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;235 Andet undervisnings- og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;38279,67\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;235 Andet undervisnings- og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;36618,75\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;235 Andet undervisnings- og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;34922,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;235 Andet undervisnings- og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;11154,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;235 Andet undervisnings- og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;23768,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;241 Arbejde inden for finans og økonomi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;52781,69\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;241 Arbejde inden for finans og økonomi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;56812,70\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;241 Arbejde inden for finans og økonomi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;47556,52\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;241 Arbejde inden for finans og økonomi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;29162,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;241 Arbejde inden for finans og økonomi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;16158,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;241 Arbejde inden for finans og økonomi;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;13005,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;242 Arbejde inden for virksomhedsadministration - privat og offentlig;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;55513,94\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;242 Arbejde inden for virksomhedsadministration - privat og offentlig;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;59082,94\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;242 Arbejde inden for virksomhedsadministration - privat og offentlig;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;52022,90\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;242 Arbejde inden for virksomhedsadministration - privat og offentlig;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;12304,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;242 Arbejde inden for virksomhedsadministration - privat og offentlig;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;5953,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;242 Arbejde inden for virksomhedsadministration - privat og offentlig;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;6351,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;243 Arbejde inden for salg, marketing og PR;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;50987,08\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;243 Arbejde inden for salg, marketing og PR;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;55362,46\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;243 Arbejde inden for salg, marketing og PR;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;45894,67\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;243 Arbejde inden for salg, marketing og PR;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;11287,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;243 Arbejde inden for salg, marketing og PR;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6002,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;243 Arbejde inden for salg, marketing og PR;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;5285,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;251 Udvikling og analyse af software og applikatoner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;53530,20\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;251 Udvikling og analyse af software og applikatoner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;54439,94\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;251 Udvikling og analyse af software og applikatoner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;50204,27\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;251 Udvikling og analyse af software og applikatoner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;31386,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;251 Udvikling og analyse af software og applikatoner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;24267,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;251 Udvikling og analyse af software og applikatoner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7119,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;252 Arbejde med databaser og netværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;48874,85\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;252 Arbejde med databaser og netværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;49584,52\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;252 Arbejde med databaser og netværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;45475,93\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;252 Arbejde med databaser og netværk;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6537,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;252 Arbejde med databaser og netværk;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;5368,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;252 Arbejde med databaser og netværk;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1169,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;261 Juridisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;54365,82\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;261 Juridisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;57832,97\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;261 Juridisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;51620,70\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;261 Juridisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;12519,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;261 Juridisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;5284,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;261 Juridisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7235,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;41614,29\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;43379,81\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;40899,92\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3477,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;969,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2508,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;263 Arbejde inden for samfundsvidenskab og religion;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;41757,62\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;263 Arbejde inden for samfundsvidenskab og religion;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;45590,73\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;263 Arbejde inden for samfundsvidenskab og religion;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;40002,63\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;263 Arbejde inden for samfundsvidenskab og religion;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;23347,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;263 Arbejde inden for samfundsvidenskab og religion;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6833,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;263 Arbejde inden for samfundsvidenskab og religion;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;16514,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;47628,57\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;49485,54\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;45447,69\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;7281,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3844,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3437,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;265 Arbejde med kunst og kreative fag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;40447,06\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;265 Arbejde med kunst og kreative fag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;41801,19\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;265 Arbejde med kunst og kreative fag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;38493,32\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;265 Arbejde med kunst og kreative fag;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2044,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;265 Arbejde med kunst og kreative fag;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1198,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;265 Arbejde med kunst og kreative fag;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;847,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;40987,60\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;43125,84\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;36183,14\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;39511,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;26599,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;12911,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;313 Teknisk kontrolarbejde inden for processtyring;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;42737,40\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;313 Teknisk kontrolarbejde inden for processtyring;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;44239,11\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;313 Teknisk kontrolarbejde inden for processtyring;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;39142,59\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;313 Teknisk kontrolarbejde inden for processtyring;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2495,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;313 Teknisk kontrolarbejde inden for processtyring;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1724,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;313 Teknisk kontrolarbejde inden for processtyring;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;771,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;314 Teknikerarbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;40810,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;314 Teknikerarbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;41797,92\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;314 Teknikerarbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;37195,55\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;314 Teknikerarbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;862,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;314 Teknikerarbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;664,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;314 Teknikerarbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;198,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;315 Teknikerarbejde inden for skibs- og luftfart;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;62393,68\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;315 Teknikerarbejde inden for skibs- og luftfart;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;63134,78\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;315 Teknikerarbejde inden for skibs- og luftfart;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;54475,98\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;315 Teknikerarbejde inden for skibs- og luftfart;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3066,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;315 Teknikerarbejde inden for skibs- og luftfart;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2786,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;315 Teknikerarbejde inden for skibs- og luftfart;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;280,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36323,19\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;38413,61\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;36044,10\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;10988,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1313,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;9675,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;44945,81\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;44485,09\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;129,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;42,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;87,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;11,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;10,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;29789,87\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29102,69\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;175,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;18,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;157,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36366,94\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;40838,88\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33429,81\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5637,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2220,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3418,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;331 Arbejde med finans, regnskab og matematik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;42941,02\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;331 Arbejde med finans, regnskab og matematik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;49738,97\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;331 Arbejde med finans, regnskab og matematik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;39085,41\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;331 Arbejde med finans, regnskab og matematik;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;26124,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;331 Arbejde med finans, regnskab og matematik;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;9514,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;331 Arbejde med finans, regnskab og matematik;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;16610,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;332 Agent- og mæglerarbejde inden for salg og indkøb;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;45133,10\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;332 Agent- og mæglerarbejde inden for salg og indkøb;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;48106,18\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;332 Agent- og mæglerarbejde inden for salg og indkøb;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;40085,92\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;332 Agent- og mæglerarbejde inden for salg og indkøb;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;30690,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;332 Agent- og mæglerarbejde inden for salg og indkøb;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;19089,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;332 Agent- og mæglerarbejde inden for salg og indkøb;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;11601,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;333 Arbejde inden for forretningsservice;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;40681,98\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;333 Arbejde inden for forretningsservice;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;44165,17\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;333 Arbejde inden for forretningsservice;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;37213,79\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;333 Arbejde inden for forretningsservice;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;9183,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;333 Arbejde inden for forretningsservice;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4429,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;333 Arbejde inden for forretningsservice;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4754,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;334 Administrativt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37377,83\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;334 Administrativt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;42575,72\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;334 Administrativt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;36398,81\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;334 Administrativt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;37963,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;334 Administrativt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6100,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;334 Administrativt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;31864,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;335 Arbejde med forvaltning og udøvelse af lovgivning;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;41324,49\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;335 Arbejde med forvaltning og udøvelse af lovgivning;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;45710,47\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;335 Arbejde med forvaltning og udøvelse af lovgivning;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;37832,30\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;335 Arbejde med forvaltning og udøvelse af lovgivning;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;10752,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;335 Arbejde med forvaltning og udøvelse af lovgivning;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4742,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;335 Arbejde med forvaltning og udøvelse af lovgivning;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;6010,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;341 Arbejde inden for jura og samfundsforhold;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37148,92\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;341 Arbejde inden for jura og samfundsforhold;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;37636,36\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;341 Arbejde inden for jura og samfundsforhold;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;36934,80\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;341 Arbejde inden for jura og samfundsforhold;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4451,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;341 Arbejde inden for jura og samfundsforhold;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1305,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;341 Arbejde inden for jura og samfundsforhold;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3146,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;342 Arbejde med sport og fitness;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;35884,52\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;342 Arbejde med sport og fitness;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;38753,89\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;342 Arbejde med sport og fitness;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29988,40\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;342 Arbejde med sport og fitness;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1134,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;342 Arbejde med sport og fitness;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;735,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;342 Arbejde med sport og fitness;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;399,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;35826,05\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;38740,88\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33380,54\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3565,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1555,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2009,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;45073,65\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;45502,24\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;43205,21\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;7464,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6030,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1434,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;41373,53\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;41262,57\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;41945,42\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3545,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2950,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;596,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;411 Almindeligt kontorarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;33994,26\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;411 Almindeligt kontorarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;37340,07\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;411 Almindeligt kontorarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33092,20\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;411 Almindeligt kontorarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;47212,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;411 Almindeligt kontorarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;9504,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;411 Almindeligt kontorarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;37709,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;412 Almindeligt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36738,80\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;412 Almindeligt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;40652,03\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;412 Almindeligt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;36436,29\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;412 Almindeligt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6455,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;412 Almindeligt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;449,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;412 Almindeligt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;6006,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;413 Indtastningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37786,13\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;413 Indtastningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;40846,94\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;413 Indtastningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;34624,88\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;413 Indtastningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2957,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;413 Indtastningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1496,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;413 Indtastningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1461,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;35640,78\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;37679,59\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;34926,57\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2883,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;693,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2190,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;422 Kundeinformationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31155,13\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;422 Kundeinformationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32204,96\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;422 Kundeinformationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;30716,36\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;422 Kundeinformationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;11375,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;422 Kundeinformationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3346,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;422 Kundeinformationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;8029,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;431 Almindeligt beregningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37857,26\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;431 Almindeligt beregningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;42403,19\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;431 Almindeligt beregningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;36670,82\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;431 Almindeligt beregningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;15938,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;431 Almindeligt beregningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3305,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;431 Almindeligt beregningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;12633,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;432 Kontorarbejde inden for materialeregistrering og transport;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32079,07\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;432 Kontorarbejde inden for materialeregistrering og transport;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32396,06\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;432 Kontorarbejde inden for materialeregistrering og transport;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31136,44\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;432 Kontorarbejde inden for materialeregistrering og transport;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;13680,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;432 Kontorarbejde inden for materialeregistrering og transport;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;10104,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;432 Kontorarbejde inden for materialeregistrering og transport;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3576,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;441 Andet almindeligt kontor- og kundeservicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32383,44\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;441 Andet almindeligt kontor- og kundeservicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32081,66\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;441 Andet almindeligt kontor- og kundeservicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32723,59\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;441 Andet almindeligt kontor- og kundeservicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;19626,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;441 Andet almindeligt kontor- og kundeservicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;10211,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;441 Andet almindeligt kontor- og kundeservicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;9415,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;511 Service- og kontrolarbejde under transport og rejser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36449,25\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;511 Service- og kontrolarbejde under transport og rejser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;37602,33\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;511 Service- og kontrolarbejde under transport og rejser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;35543,13\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;511 Service- og kontrolarbejde under transport og rejser;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4082,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;511 Service- og kontrolarbejde under transport og rejser;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1767,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;511 Service- og kontrolarbejde under transport og rejser;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2315,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;512 Kokkearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;28900,35\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;512 Kokkearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;29339,19\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;512 Kokkearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28327,42\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;512 Kokkearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4138,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;512 Kokkearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2163,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;512 Kokkearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1975,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;513 Tjenere og bartendere;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;25239,63\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;513 Tjenere og bartendere;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;26327,23\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;513 Tjenere og bartendere;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;24415,66\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;513 Tjenere og bartendere;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3450,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;513 Tjenere og bartendere;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1478,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;513 Tjenere og bartendere;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1972,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;29797,83\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;30541,99\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29713,84\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;517,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;52,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;465,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31955,55\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;31992,18\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31700,27\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;14807,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;12838,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1968,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;516 Andet servicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;30031,76\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;516 Andet servicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;31316,09\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;516 Andet servicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;27630,78\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;516 Andet servicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1513,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;516 Andet servicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;979,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;516 Andet servicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;535,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;521 Salgsarbejde på gade og marked;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;521 Salgsarbejde på gade og marked;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;521 Salgsarbejde på gade og marked;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;521 Salgsarbejde på gade og marked;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;31,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;521 Salgsarbejde på gade og marked;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;27,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;521 Salgsarbejde på gade og marked;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;522 Salgsarbejde i butik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;26876,56\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;522 Salgsarbejde i butik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;28985,09\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;522 Salgsarbejde i butik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;24854,45\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;522 Salgsarbejde i butik;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;22339,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;522 Salgsarbejde i butik;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;10829,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;522 Salgsarbejde i butik;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;11509,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;523 Kasseassistentarbejde og beslægtet kundebetjening;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;24016,68\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;523 Kasseassistentarbejde og beslægtet kundebetjening;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;24270,47\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;523 Kasseassistentarbejde og beslægtet kundebetjening;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;23892,55\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;523 Kasseassistentarbejde og beslægtet kundebetjening;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;13045,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;523 Kasseassistentarbejde og beslægtet kundebetjening;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4261,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;523 Kasseassistentarbejde og beslægtet kundebetjening;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;8784,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;524 Andet salgsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36639,23\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;524 Andet salgsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;39349,89\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;524 Andet salgsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32630,36\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;524 Andet salgsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;10273,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;524 Andet salgsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6051,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;524 Andet salgsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4222,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;531 Børneomsorgsarbejde og hjælpelærerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;26634,77\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;531 Børneomsorgsarbejde og hjælpelærerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;26025,70\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;531 Børneomsorgsarbejde og hjælpelærerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;26796,35\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;531 Børneomsorgsarbejde og hjælpelærerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;41093,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;531 Børneomsorgsarbejde og hjælpelærerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;8260,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;531 Børneomsorgsarbejde og hjælpelærerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;32833,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;532 Omsorgsarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31182,14\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;532 Omsorgsarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;30530,44\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;532 Omsorgsarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31279,43\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;532 Omsorgsarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;84130,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;532 Omsorgsarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;10435,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;532 Omsorgsarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;73694,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;541 Rednings- og overvågningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;34188,80\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;541 Rednings- og overvågningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;34549,69\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;541 Rednings- og overvågningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32321,51\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;541 Rednings- og overvågningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;19568,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;541 Rednings- og overvågningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;16329,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;541 Rednings- og overvågningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3239,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;611 Arbejde med plantevækst;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;30149,11\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;611 Arbejde med plantevækst;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;30507,87\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;611 Arbejde med plantevækst;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29223,41\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;611 Arbejde med plantevækst;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2726,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;611 Arbejde med plantevækst;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1896,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;611 Arbejde med plantevækst;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;830,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;612 Arbejde med dyreopdræt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;29420,67\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;612 Arbejde med dyreopdræt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;30095,92\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;612 Arbejde med dyreopdræt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28882,43\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;612 Arbejde med dyreopdræt;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;448,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;612 Arbejde med dyreopdræt;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;186,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;612 Arbejde med dyreopdræt;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;262,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;25398,49\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;25416,65\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;139,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;128,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;11,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;621 Skovbrugsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;29951,66\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;621 Skovbrugsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;29889,04\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;621 Skovbrugsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;621 Skovbrugsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;382,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;621 Skovbrugsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;354,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;621 Skovbrugsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;27,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;622 Arbejde med fiskeri og jagt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;622 Arbejde med fiskeri og jagt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;622 Arbejde med fiskeri og jagt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;622 Arbejde med fiskeri og jagt;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;32,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;622 Arbejde med fiskeri og jagt;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;30,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;622 Arbejde med fiskeri og jagt;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;711 Bygningsarbejde (ekskl. medhjælpere) - basis;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;33378,46\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;711 Bygningsarbejde (ekskl. medhjælpere) - basis;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;33393,79\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;711 Bygningsarbejde (ekskl. medhjælpere) - basis;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31483,62\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;711 Bygningsarbejde (ekskl. medhjælpere) - basis;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;13801,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;711 Bygningsarbejde (ekskl. medhjælpere) - basis;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;13688,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;711 Bygningsarbejde (ekskl. medhjælpere) - basis;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;113,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;712 Bygningsarbejde (ekskl. medhjælpere) - finish;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36263,09\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;712 Bygningsarbejde (ekskl. medhjælpere) - finish;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;36305,52\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;712 Bygningsarbejde (ekskl. medhjælpere) - finish;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31204,14\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;712 Bygningsarbejde (ekskl. medhjælpere) - finish;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6666,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;712 Bygningsarbejde (ekskl. medhjælpere) - finish;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6609,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;712 Bygningsarbejde (ekskl. medhjælpere) - finish;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;57,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;713 Malere og arbejde inden for rensning af bygninger;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31087,76\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;713 Malere og arbejde inden for rensning af bygninger;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;31435,45\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;713 Malere og arbejde inden for rensning af bygninger;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29589,31\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;713 Malere og arbejde inden for rensning af bygninger;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3573,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;713 Malere og arbejde inden for rensning af bygninger;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2910,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;713 Malere og arbejde inden for rensning af bygninger;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;663,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;721 Pladearbejde, svejsning og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32936,11\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;721 Pladearbejde, svejsning og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32994,19\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;721 Pladearbejde, svejsning og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29886,10\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;721 Pladearbejde, svejsning og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;8218,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;721 Pladearbejde, svejsning og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;8056,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;721 Pladearbejde, svejsning og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;163,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;722 Smede, værktøjsmagere og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;34374,87\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;722 Smede, værktøjsmagere og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;34488,20\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;722 Smede, værktøjsmagere og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;30928,36\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;722 Smede, værktøjsmagere og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;17112,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;722 Smede, værktøjsmagere og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;16525,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;722 Smede, værktøjsmagere og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;587,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;723 Mekanikerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;34440,88\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;723 Mekanikerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;34456,48\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;723 Mekanikerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32616,23\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;723 Mekanikerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;12109,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;723 Mekanikerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;11995,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;723 Mekanikerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;114,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;731 Præcisionshåndværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31424,95\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;731 Præcisionshåndværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;33204,58\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;731 Præcisionshåndværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28230,61\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;731 Præcisionshåndværk;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;884,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;731 Præcisionshåndværk;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;548,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;731 Præcisionshåndværk;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;336,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;732 Grafisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36662,27\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;732 Grafisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;37929,61\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;732 Grafisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33202,71\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;732 Grafisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2570,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;732 Grafisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1883,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;732 Grafisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;687,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;741 Installation- og reparation af elektrisk udstyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;35654,13\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;741 Installation- og reparation af elektrisk udstyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;35697,98\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;741 Installation- og reparation af elektrisk udstyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32330,33\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;741 Installation- og reparation af elektrisk udstyr;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;13738,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;741 Installation- og reparation af elektrisk udstyr;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;13558,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;741 Installation- og reparation af elektrisk udstyr;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;181,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;742 Installation og reparation af elektronik og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36275,87\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;742 Installation og reparation af elektronik og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;36742,97\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;742 Installation og reparation af elektronik og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29574,44\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;742 Installation og reparation af elektronik og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1861,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;742 Installation og reparation af elektronik og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1737,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;742 Installation og reparation af elektronik og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;125,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;751 Arbejde inden for nærings- og nydelsesmiddelområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31862,12\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;751 Arbejde inden for nærings- og nydelsesmiddelområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;33022,41\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;751 Arbejde inden for nærings- og nydelsesmiddelområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28639,65\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;751 Arbejde inden for nærings- og nydelsesmiddelområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6579,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;751 Arbejde inden for nærings- og nydelsesmiddelområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4821,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;751 Arbejde inden for nærings- og nydelsesmiddelområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1759,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;752 Håndværkspræget arbejde i træindustri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;29170,57\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;752 Håndværkspræget arbejde i træindustri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;29245,72\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;752 Håndværkspræget arbejde i træindustri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28284,49\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;752 Håndværkspræget arbejde i træindustri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1448,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;752 Håndværkspræget arbejde i træindustri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1333,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;752 Håndværkspræget arbejde i træindustri;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;114,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;753 Tekstil- og beklædningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31014,80\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;753 Tekstil- og beklædningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32012,46\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;753 Tekstil- og beklædningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;30184,09\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;753 Tekstil- og beklædningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;525,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;753 Tekstil- og beklædningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;237,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;753 Tekstil- og beklædningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;287,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;754 Andet håndværkspræget arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36935,53\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;754 Andet håndværkspræget arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;37899,66\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;754 Andet håndværkspræget arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;34655,59\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;754 Andet håndværkspræget arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1907,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;754 Andet håndværkspræget arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1322,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;754 Andet håndværkspræget arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;585,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;40231,21\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;40389,92\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;35746,26\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1768,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1703,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;65,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31079,93\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;31415,69\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29688,42\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4791,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3824,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;967,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38682,50\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;39186,34\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;36728,45\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3127,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2489,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;638,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31548,95\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32201,36\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29931,06\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5586,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3935,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1651,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;28407,75\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;31606,49\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;25304,83\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;983,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;488,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;495,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32570,17\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;33648,30\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29611,11\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;12866,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;9431,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3436,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;817 Operatørarbejde ved træforarbejdning og papirfremstilling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;29020,52\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;817 Operatørarbejde ved træforarbejdning og papirfremstilling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;29199,94\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;817 Operatørarbejde ved træforarbejdning og papirfremstilling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28040,38\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;817 Operatørarbejde ved træforarbejdning og papirfremstilling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3114,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;817 Operatørarbejde ved træforarbejdning og papirfremstilling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2616,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;817 Operatørarbejde ved træforarbejdning og papirfremstilling;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;499,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;818 Andet operatørarbejde ved stationære anlæg og maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;34870,85\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;818 Andet operatørarbejde ved stationære anlæg og maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;36068,08\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;818 Andet operatørarbejde ved stationære anlæg og maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;30893,85\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;818 Andet operatørarbejde ved stationære anlæg og maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5964,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;818 Andet operatørarbejde ved stationære anlæg og maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4559,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;818 Andet operatørarbejde ved stationære anlæg og maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1405,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;821 Monteringsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;30437,92\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;821 Monteringsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;31906,54\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;821 Monteringsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;27852,84\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;821 Monteringsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;9024,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;821 Monteringsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;5622,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;821 Monteringsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3402,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;831 Lokomotivførerarbejde og tograngering;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;41868,53\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;831 Lokomotivførerarbejde og tograngering;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;41965,43\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;831 Lokomotivførerarbejde og tograngering;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;39581,95\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;831 Lokomotivførerarbejde og tograngering;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2832,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;831 Lokomotivførerarbejde og tograngering;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2715,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;831 Lokomotivførerarbejde og tograngering;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;118,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;832 Chauffører af biler, varevogne og motorcykler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;27965,45\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;832 Chauffører af biler, varevogne og motorcykler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;28035,67\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;832 Chauffører af biler, varevogne og motorcykler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;26310,37\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;832 Chauffører af biler, varevogne og motorcykler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2801,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;832 Chauffører af biler, varevogne og motorcykler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2691,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;832 Chauffører af biler, varevogne og motorcykler;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;111,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;833 Chauffører af busser og lastbiler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;29938,79\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;833 Chauffører af busser og lastbiler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;29992,33\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;833 Chauffører af busser og lastbiler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28803,09\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;833 Chauffører af busser og lastbiler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;19709,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;833 Chauffører af busser og lastbiler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;18751,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;833 Chauffører af busser og lastbiler;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;958,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;834 Førere af mobile maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31800,72\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;834 Førere af mobile maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;31875,99\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;834 Førere af mobile maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29789,97\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;834 Førere af mobile maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5846,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;834 Førere af mobile maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;5622,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;834 Førere af mobile maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;224,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;835 Dæksarbejde og medhjælp om bord på skibe;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36518,75\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;835 Dæksarbejde og medhjælp om bord på skibe;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;36651,64\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;835 Dæksarbejde og medhjælp om bord på skibe;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;835 Dæksarbejde og medhjælp om bord på skibe;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;356,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;835 Dæksarbejde og medhjælp om bord på skibe;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;345,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;835 Dæksarbejde og medhjælp om bord på skibe;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;11,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;911 Rengøringsarbejde i private hjem, hoteller og kontorer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;25857,12\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;911 Rengøringsarbejde i private hjem, hoteller og kontorer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;27037,46\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;911 Rengøringsarbejde i private hjem, hoteller og kontorer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;25338,74\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;911 Rengøringsarbejde i private hjem, hoteller og kontorer;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;35344,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;911 Rengøringsarbejde i private hjem, hoteller og kontorer;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;10112,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;911 Rengøringsarbejde i private hjem, hoteller og kontorer;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;25232,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;26174,36\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;27029,96\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;24981,39\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2460,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1373,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1087,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;24591,32\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;25263,82\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;22772,95\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;565,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;405,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;161,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31938,17\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32058,77\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29237,68\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;22115,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;21029,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1086,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;932 Manuelt produktionsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;28447,40\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;932 Manuelt produktionsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;29379,84\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;932 Manuelt produktionsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;26597,96\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;932 Manuelt produktionsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;10831,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;932 Manuelt produktionsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;7186,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;932 Manuelt produktionsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3645,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;933 Transport- og lagerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;29312,27\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;933 Transport- og lagerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;29812,04\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;933 Transport- og lagerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;26537,06\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;933 Transport- og lagerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;20756,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;933 Transport- og lagerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;17509,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;933 Transport- og lagerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3247,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;941 Manuelt arbejde med tilberedning af mad;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;25440,11\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;941 Manuelt arbejde med tilberedning af mad;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;23896,07\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;941 Manuelt arbejde med tilberedning af mad;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;25885,67\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;941 Manuelt arbejde med tilberedning af mad;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;9645,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;941 Manuelt arbejde med tilberedning af mad;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1813,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;941 Manuelt arbejde med tilberedning af mad;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7831,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;951 Servicearbejde på gaden;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;26870,09\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;951 Servicearbejde på gaden;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;26873,92\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;951 Servicearbejde på gaden;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;951 Servicearbejde på gaden;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;54,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;951 Servicearbejde på gaden;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;54,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;951 Servicearbejde på gaden;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;0,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;952 Gadesalg (undtagen madvarer);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;952 Gadesalg (undtagen madvarer);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;952 Gadesalg (undtagen madvarer);STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;952 Gadesalg (undtagen madvarer);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;0,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;952 Gadesalg (undtagen madvarer);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;0,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;952 Gadesalg (undtagen madvarer);Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;961 Renovationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31615,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;961 Renovationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;31833,23\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;961 Renovationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;26961,93\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;961 Renovationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2052,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;961 Renovationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1966,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;961 Renovationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;86,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;962 Andet manuelt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;30175,14\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;962 Andet manuelt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;30862,61\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;962 Andet manuelt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28445,44\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;962 Andet manuelt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3758,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;962 Andet manuelt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2639,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2013;962 Andet manuelt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1119,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;011 Militært arbejde på officersniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;49926,01\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;011 Militært arbejde på officersniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;50224,57\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;011 Militært arbejde på officersniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;44586,45\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;011 Militært arbejde på officersniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3993,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;011 Militært arbejde på officersniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3777,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;011 Militært arbejde på officersniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;216,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36473,45\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;36639,48\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33327,69\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4070,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3859,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;211,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;031 Militært arbejde, øvrige rangordener;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31779,20\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;031 Militært arbejde, øvrige rangordener;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;31885,22\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;031 Militært arbejde, øvrige rangordener;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;30283,52\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;031 Militært arbejde, øvrige rangordener;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;7309,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;031 Militært arbejde, øvrige rangordener;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6812,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;031 Militært arbejde, øvrige rangordener;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;497,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;70539,18\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;73006,86\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;65338,85\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2087,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1415,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;672,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;112 Øverste virksomhedsledelse;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;99329,04\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;112 Øverste virksomhedsledelse;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;101716,76\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;112 Øverste virksomhedsledelse;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;82775,46\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;112 Øverste virksomhedsledelse;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;12131,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;112 Øverste virksomhedsledelse;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;10578,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;112 Øverste virksomhedsledelse;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1553,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;121 Ledelse inden for forretningsservice og administrative funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;67284,64\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;121 Ledelse inden for forretningsservice og administrative funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;70055,88\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;121 Ledelse inden for forretningsservice og administrative funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;62375,10\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;121 Ledelse inden for forretningsservice og administrative funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;18157,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;121 Ledelse inden for forretningsservice og administrative funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;11518,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;121 Ledelse inden for forretningsservice og administrative funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;6640,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;122 Ledelse af forretnings- og udviklingsorienterede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;68319,57\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;122 Ledelse af forretnings- og udviklingsorienterede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;71329,53\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;122 Ledelse af forretnings- og udviklingsorienterede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;59007,78\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;122 Ledelse af forretnings- og udviklingsorienterede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;11147,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;122 Ledelse af forretnings- og udviklingsorienterede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;8388,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;122 Ledelse af forretnings- og udviklingsorienterede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2760,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;56335,24\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;56490,64\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;233,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;197,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;36,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;59765,25\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;59958,55\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;58249,41\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;8558,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;7575,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;983,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;70958,67\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;71335,73\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;68497,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2393,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2058,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;335,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;134 Ledelse af hovedaktiviteten inden for servicefag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;54451,84\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;134 Ledelse af hovedaktiviteten inden for servicefag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;61215,42\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;134 Ledelse af hovedaktiviteten inden for servicefag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;48693,95\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;134 Ledelse af hovedaktiviteten inden for servicefag;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;13793,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;134 Ledelse af hovedaktiviteten inden for servicefag;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6144,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;134 Ledelse af hovedaktiviteten inden for servicefag;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7650,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;48156,95\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;49698,59\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;45681,18\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;415,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;255,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;160,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;40819,87\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;44017,18\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;34478,52\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5427,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3569,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1858,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;143 Ledelse af hovedaktiviteten inden for andre serviceområder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;53447,56\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;143 Ledelse af hovedaktiviteten inden for andre serviceområder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;54802,32\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;143 Ledelse af hovedaktiviteten inden for andre serviceområder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;50151,03\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;143 Ledelse af hovedaktiviteten inden for andre serviceområder;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1409,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;143 Ledelse af hovedaktiviteten inden for andre serviceområder;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1000,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;143 Ledelse af hovedaktiviteten inden for andre serviceområder;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;409,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;211 Arbejde inden for fysik og geologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;52982,71\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;211 Arbejde inden for fysik og geologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;54206,14\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;211 Arbejde inden for fysik og geologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;50980,84\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;211 Arbejde inden for fysik og geologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2151,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;211 Arbejde inden for fysik og geologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1326,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;211 Arbejde inden for fysik og geologi;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;825,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;61084,23\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;63353,71\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;56294,72\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1299,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;873,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;425,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;213 Arbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;47969,59\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;213 Arbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;49311,66\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;213 Arbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;46263,87\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;213 Arbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4207,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;213 Arbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2312,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;213 Arbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1895,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;214 Ingeniørarbejde (undtagen inden for elektroteknologi);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;54724,87\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;214 Ingeniørarbejde (undtagen inden for elektroteknologi);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;55750,66\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;214 Ingeniørarbejde (undtagen inden for elektroteknologi);STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;51061,59\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;214 Ingeniørarbejde (undtagen inden for elektroteknologi);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;32163,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;214 Ingeniørarbejde (undtagen inden for elektroteknologi);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;24793,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;214 Ingeniørarbejde (undtagen inden for elektroteknologi);Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7370,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;215 Ingeniørarbejde inden for elektroteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;56305,36\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;215 Ingeniørarbejde inden for elektroteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;56441,81\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;215 Ingeniørarbejde inden for elektroteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;54418,48\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;215 Ingeniørarbejde inden for elektroteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5710,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;215 Ingeniørarbejde inden for elektroteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;5309,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;215 Ingeniørarbejde inden for elektroteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;400,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;216 Arbejde med arkitektur, infrastruktur og design;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;43608,27\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;216 Arbejde med arkitektur, infrastruktur og design;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;46539,42\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;216 Arbejde med arkitektur, infrastruktur og design;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;40662,89\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;216 Arbejde med arkitektur, infrastruktur og design;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6123,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;216 Arbejde med arkitektur, infrastruktur og design;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3035,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;216 Arbejde med arkitektur, infrastruktur og design;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3088,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;221 Lægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;68558,74\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;221 Lægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;73749,70\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;221 Lægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;63166,63\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;221 Lægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;15887,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;221 Lægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;7834,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;221 Lægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;8054,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;222 Sygeplejerske- og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37375,05\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;222 Sygeplejerske- og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;40466,37\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;222 Sygeplejerske- og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;37233,75\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;222 Sygeplejerske- og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;50735,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;222 Sygeplejerske- og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2132,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;222 Sygeplejerske- og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;48602,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;223 Arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;223 Arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;223 Arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;223 Arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;10,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;223 Arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;223 Arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;224 Paramedicinsk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;41582,55\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;224 Paramedicinsk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;41647,48\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;224 Paramedicinsk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;224 Paramedicinsk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;336,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;224 Paramedicinsk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;306,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;224 Paramedicinsk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;30,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;225 Dyrlægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;49894,61\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;225 Dyrlægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;53660,35\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;225 Dyrlægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;47548,16\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;225 Dyrlægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;785,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;225 Dyrlægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;289,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;225 Dyrlægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;496,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;226 Andet sundhedsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;42709,68\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;226 Andet sundhedsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;51140,05\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;226 Andet sundhedsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;40726,82\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;226 Andet sundhedsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;21290,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;226 Andet sundhedsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3813,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;226 Andet sundhedsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;17478,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;231 Undervisning og forskning ved universiteter og højere læreanstalter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;46690,62\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;231 Undervisning og forskning ved universiteter og højere læreanstalter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;48355,36\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;231 Undervisning og forskning ved universiteter og højere læreanstalter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;44515,27\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;231 Undervisning og forskning ved universiteter og højere læreanstalter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;18864,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;231 Undervisning og forskning ved universiteter og højere læreanstalter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;10567,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;231 Undervisning og forskning ved universiteter og højere læreanstalter;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;8297,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;232 Undervisning ved erhvervsuddannelser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;40696,03\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;232 Undervisning ved erhvervsuddannelser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;40848,94\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;232 Undervisning ved erhvervsuddannelser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;40488,68\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;232 Undervisning ved erhvervsuddannelser;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;11082,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;232 Undervisning ved erhvervsuddannelser;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6304,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;232 Undervisning ved erhvervsuddannelser;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4778,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;233 Almen undervisning efter grundskoleniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;46278,54\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;233 Almen undervisning efter grundskoleniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;47047,88\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;233 Almen undervisning efter grundskoleniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;45563,83\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;233 Almen undervisning efter grundskoleniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;14438,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;233 Almen undervisning efter grundskoleniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6863,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;233 Almen undervisning efter grundskoleniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7575,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;234 Undervisning og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37239,13\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;234 Undervisning og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;38550,93\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;234 Undervisning og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;36754,75\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;234 Undervisning og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;107801,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;234 Undervisning og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;27813,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;234 Undervisning og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;79988,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;235 Andet undervisnings- og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37692,47\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;235 Andet undervisnings- og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;38735,78\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;235 Andet undervisnings- og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;37181,46\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;235 Andet undervisnings- og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;35947,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;235 Andet undervisnings- og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;11439,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;235 Andet undervisnings- og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;24508,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;241 Arbejde inden for finans og økonomi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;53497,51\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;241 Arbejde inden for finans og økonomi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;57374,82\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;241 Arbejde inden for finans og økonomi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;48517,25\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;241 Arbejde inden for finans og økonomi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;28796,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;241 Arbejde inden for finans og økonomi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;15959,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;241 Arbejde inden for finans og økonomi;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;12837,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;242 Arbejde inden for virksomhedsadministration - privat og offentlig;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;56294,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;242 Arbejde inden for virksomhedsadministration - privat og offentlig;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;59569,57\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;242 Arbejde inden for virksomhedsadministration - privat og offentlig;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;53186,94\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;242 Arbejde inden for virksomhedsadministration - privat og offentlig;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;13261,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;242 Arbejde inden for virksomhedsadministration - privat og offentlig;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6349,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;242 Arbejde inden for virksomhedsadministration - privat og offentlig;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;6912,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;243 Arbejde inden for salg, marketing og PR;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;51400,75\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;243 Arbejde inden for salg, marketing og PR;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;55787,55\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;243 Arbejde inden for salg, marketing og PR;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;46281,64\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;243 Arbejde inden for salg, marketing og PR;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;12067,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;243 Arbejde inden for salg, marketing og PR;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6433,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;243 Arbejde inden for salg, marketing og PR;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;5634,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;251 Udvikling og analyse af software og applikatoner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;54218,15\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;251 Udvikling og analyse af software og applikatoner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;55075,96\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;251 Udvikling og analyse af software og applikatoner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;51172,20\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;251 Udvikling og analyse af software og applikatoner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;31690,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;251 Udvikling og analyse af software og applikatoner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;24521,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;251 Udvikling og analyse af software og applikatoner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7169,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;252 Arbejde med databaser og netværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;48991,25\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;252 Arbejde med databaser og netværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;49657,54\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;252 Arbejde med databaser og netværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;45762,20\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;252 Arbejde med databaser og netværk;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6664,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;252 Arbejde med databaser og netværk;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;5493,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;252 Arbejde med databaser og netværk;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1171,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;261 Juridisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;54153,01\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;261 Juridisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;57649,92\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;261 Juridisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;51555,20\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;261 Juridisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;12970,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;261 Juridisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;5359,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;261 Juridisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7611,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;41228,43\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;42142,68\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;40868,44\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3251,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;893,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2358,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;263 Arbejde inden for samfundsvidenskab og religion;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;42097,50\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;263 Arbejde inden for samfundsvidenskab og religion;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;46028,91\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;263 Arbejde inden for samfundsvidenskab og religion;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;40421,38\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;263 Arbejde inden for samfundsvidenskab og religion;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;24607,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;263 Arbejde inden for samfundsvidenskab og religion;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6995,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;263 Arbejde inden for samfundsvidenskab og religion;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;17612,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;47621,39\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;49806,43\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;45130,94\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;7470,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3919,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3551,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;265 Arbejde med kunst og kreative fag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;40419,10\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;265 Arbejde med kunst og kreative fag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;41572,04\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;265 Arbejde med kunst og kreative fag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;38817,18\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;265 Arbejde med kunst og kreative fag;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2220,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;265 Arbejde med kunst og kreative fag;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1281,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;265 Arbejde med kunst og kreative fag;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;939,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;41745,25\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;43896,65\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;36929,81\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;40001,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;27128,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;12873,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;313 Teknisk kontrolarbejde inden for processtyring;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;43426,06\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;313 Teknisk kontrolarbejde inden for processtyring;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;45108,89\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;313 Teknisk kontrolarbejde inden for processtyring;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;39773,90\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;313 Teknisk kontrolarbejde inden for processtyring;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2720,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;313 Teknisk kontrolarbejde inden for processtyring;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1829,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;313 Teknisk kontrolarbejde inden for processtyring;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;891,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;314 Teknikerarbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;41528,31\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;314 Teknikerarbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;42594,49\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;314 Teknikerarbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;38066,11\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;314 Teknikerarbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;788,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;314 Teknikerarbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;592,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;314 Teknikerarbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;197,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;315 Teknikerarbejde inden for skibs- og luftfart;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;65532,87\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;315 Teknikerarbejde inden for skibs- og luftfart;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;66354,06\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;315 Teknikerarbejde inden for skibs- og luftfart;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;57220,93\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;315 Teknikerarbejde inden for skibs- og luftfart;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2932,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;315 Teknikerarbejde inden for skibs- og luftfart;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2670,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;315 Teknikerarbejde inden for skibs- og luftfart;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;261,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36542,35\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;38405,80\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;36280,55\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;10982,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1338,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;9645,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;44256,08\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;43929,20\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;120,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;41,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;79,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;10,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;9,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;30250,05\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29034,20\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;196,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;20,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;176,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36960,86\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;41366,52\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33901,42\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5755,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2261,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3494,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;331 Arbejde med finans, regnskab og matematik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;43661,92\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;331 Arbejde med finans, regnskab og matematik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;50410,90\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;331 Arbejde med finans, regnskab og matematik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;39685,19\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;331 Arbejde med finans, regnskab og matematik;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;26097,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;331 Arbejde med finans, regnskab og matematik;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;9602,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;331 Arbejde med finans, regnskab og matematik;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;16495,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;332 Agent- og mæglerarbejde inden for salg og indkøb;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;45885,72\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;332 Agent- og mæglerarbejde inden for salg og indkøb;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;48997,39\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;332 Agent- og mæglerarbejde inden for salg og indkøb;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;40659,96\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;332 Agent- og mæglerarbejde inden for salg og indkøb;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;31163,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;332 Agent- og mæglerarbejde inden for salg og indkøb;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;19332,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;332 Agent- og mæglerarbejde inden for salg og indkøb;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;11831,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;333 Arbejde inden for forretningsservice;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;41119,65\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;333 Arbejde inden for forretningsservice;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;44510,81\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;333 Arbejde inden for forretningsservice;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;37725,21\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;333 Arbejde inden for forretningsservice;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;9438,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;333 Arbejde inden for forretningsservice;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4596,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;333 Arbejde inden for forretningsservice;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4842,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;334 Administrativt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37982,08\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;334 Administrativt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;43899,68\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;334 Administrativt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;36932,49\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;334 Administrativt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;30414,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;334 Administrativt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4557,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;334 Administrativt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;25856,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;335 Arbejde med forvaltning og udøvelse af lovgivning;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;42100,49\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;335 Arbejde med forvaltning og udøvelse af lovgivning;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;46560,73\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;335 Arbejde med forvaltning og udøvelse af lovgivning;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;38786,62\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;335 Arbejde med forvaltning og udøvelse af lovgivning;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;10681,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;335 Arbejde med forvaltning og udøvelse af lovgivning;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4522,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;335 Arbejde med forvaltning og udøvelse af lovgivning;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;6159,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;341 Arbejde inden for jura og samfundsforhold;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37307,75\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;341 Arbejde inden for jura og samfundsforhold;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;37719,18\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;341 Arbejde inden for jura og samfundsforhold;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;37118,29\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;341 Arbejde inden for jura og samfundsforhold;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4272,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;341 Arbejde inden for jura og samfundsforhold;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1315,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;341 Arbejde inden for jura og samfundsforhold;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2958,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;342 Arbejde med sport og fitness;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;33343,02\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;342 Arbejde med sport og fitness;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;36569,99\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;342 Arbejde med sport og fitness;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;27402,53\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;342 Arbejde med sport og fitness;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1098,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;342 Arbejde med sport og fitness;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;693,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;342 Arbejde med sport og fitness;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;404,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36375,84\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;39199,29\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33977,61\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3677,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1634,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2044,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;45287,17\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;45786,19\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;43093,46\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;7857,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6364,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1494,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;41485,94\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;41388,51\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;41963,95\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3382,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2788,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;594,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;411 Almindeligt kontorarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;34701,49\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;411 Almindeligt kontorarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;37855,60\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;411 Almindeligt kontorarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33825,30\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;411 Almindeligt kontorarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;56190,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;411 Almindeligt kontorarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;11692,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;411 Almindeligt kontorarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;44498,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;412 Almindeligt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37031,47\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;412 Almindeligt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;40530,54\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;412 Almindeligt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;36754,20\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;412 Almindeligt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6218,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;412 Almindeligt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;439,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;412 Almindeligt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;5780,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;413 Indtastningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38210,94\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;413 Indtastningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;41527,70\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;413 Indtastningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;34768,92\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;413 Indtastningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2873,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;413 Indtastningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1454,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;413 Indtastningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1419,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36191,80\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;38129,92\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;35516,73\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2582,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;636,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1946,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;422 Kundeinformationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31305,70\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;422 Kundeinformationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32077,23\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;422 Kundeinformationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;30951,73\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;422 Kundeinformationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;11305,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;422 Kundeinformationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3476,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;422 Kundeinformationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7829,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;431 Almindeligt beregningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38325,42\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;431 Almindeligt beregningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;42502,62\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;431 Almindeligt beregningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;37197,39\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;431 Almindeligt beregningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;16207,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;431 Almindeligt beregningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3407,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;431 Almindeligt beregningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;12799,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;432 Kontorarbejde inden for materialeregistrering og transport;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32207,65\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;432 Kontorarbejde inden for materialeregistrering og transport;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32544,75\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;432 Kontorarbejde inden for materialeregistrering og transport;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31241,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;432 Kontorarbejde inden for materialeregistrering og transport;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;14364,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;432 Kontorarbejde inden for materialeregistrering og transport;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;10553,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;432 Kontorarbejde inden for materialeregistrering og transport;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3811,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;441 Andet almindeligt kontor- og kundeservicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32499,21\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;441 Andet almindeligt kontor- og kundeservicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32223,46\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;441 Andet almindeligt kontor- og kundeservicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32811,62\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;441 Andet almindeligt kontor- og kundeservicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;19101,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;441 Andet almindeligt kontor- og kundeservicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;9846,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;441 Andet almindeligt kontor- og kundeservicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;9256,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;511 Service- og kontrolarbejde under transport og rejser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37273,30\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;511 Service- og kontrolarbejde under transport og rejser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;38546,62\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;511 Service- og kontrolarbejde under transport og rejser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;36268,49\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;511 Service- og kontrolarbejde under transport og rejser;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4228,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;511 Service- og kontrolarbejde under transport og rejser;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1836,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;511 Service- og kontrolarbejde under transport og rejser;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2392,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;512 Kokkearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;29396,54\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;512 Kokkearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;29786,48\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;512 Kokkearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28921,04\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;512 Kokkearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4509,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;512 Kokkearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2377,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;512 Kokkearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2131,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;513 Tjenere og bartendere;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;25318,27\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;513 Tjenere og bartendere;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;26659,35\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;513 Tjenere og bartendere;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;24346,15\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;513 Tjenere og bartendere;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3903,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;513 Tjenere og bartendere;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1634,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;513 Tjenere og bartendere;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2269,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;30208,31\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;30862,44\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;30129,77\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;483,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;52,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;431,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32206,16\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32236,28\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31996,81\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;15440,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;13459,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1981,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;516 Andet servicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;29483,38\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;516 Andet servicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;30936,50\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;516 Andet servicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;26705,85\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;516 Andet servicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1828,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;516 Andet servicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1194,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;516 Andet servicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;634,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;521 Salgsarbejde på gade og marked;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;521 Salgsarbejde på gade og marked;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;521 Salgsarbejde på gade og marked;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;521 Salgsarbejde på gade og marked;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;47,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;521 Salgsarbejde på gade og marked;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;37,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;521 Salgsarbejde på gade og marked;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;10,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;522 Salgsarbejde i butik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;26364,71\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;522 Salgsarbejde i butik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;28182,37\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;522 Salgsarbejde i butik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;24682,03\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;522 Salgsarbejde i butik;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;27921,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;522 Salgsarbejde i butik;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;13251,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;522 Salgsarbejde i butik;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;14670,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;523 Kasseassistentarbejde og beslægtet kundebetjening;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;24539,01\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;523 Kasseassistentarbejde og beslægtet kundebetjening;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;25244,31\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;523 Kasseassistentarbejde og beslægtet kundebetjening;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;24290,48\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;523 Kasseassistentarbejde og beslægtet kundebetjening;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;9402,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;523 Kasseassistentarbejde og beslægtet kundebetjening;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2421,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;523 Kasseassistentarbejde og beslægtet kundebetjening;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;6981,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;524 Andet salgsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36739,63\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;524 Andet salgsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;39521,42\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;524 Andet salgsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32700,61\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;524 Andet salgsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;10779,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;524 Andet salgsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6310,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;524 Andet salgsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4469,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;531 Børneomsorgsarbejde og hjælpelærerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;26937,30\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;531 Børneomsorgsarbejde og hjælpelærerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;26449,59\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;531 Børneomsorgsarbejde og hjælpelærerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;27065,90\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;531 Børneomsorgsarbejde og hjælpelærerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;39415,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;531 Børneomsorgsarbejde og hjælpelærerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;8028,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;531 Børneomsorgsarbejde og hjælpelærerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;31388,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;532 Omsorgsarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31508,38\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;532 Omsorgsarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;30797,17\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;532 Omsorgsarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31618,14\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;532 Omsorgsarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;85357,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;532 Omsorgsarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;10955,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;532 Omsorgsarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;74401,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;541 Rednings- og overvågningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;34736,41\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;541 Rednings- og overvågningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;35116,45\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;541 Rednings- og overvågningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32780,54\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;541 Rednings- og overvågningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;19376,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;541 Rednings- og overvågningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;16189,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;541 Rednings- og overvågningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3187,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;611 Arbejde med plantevækst;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;29946,10\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;611 Arbejde med plantevækst;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;30491,29\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;611 Arbejde med plantevækst;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28429,22\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;611 Arbejde med plantevækst;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2534,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;611 Arbejde med plantevækst;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1840,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;611 Arbejde med plantevækst;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;694,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;612 Arbejde med dyreopdræt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;29349,67\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;612 Arbejde med dyreopdræt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;30257,13\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;612 Arbejde med dyreopdræt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28783,96\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;612 Arbejde med dyreopdræt;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;420,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;612 Arbejde med dyreopdræt;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;153,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;612 Arbejde med dyreopdræt;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;267,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;27750,09\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;27482,42\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;150,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;127,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;23,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;621 Skovbrugsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;30316,22\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;621 Skovbrugsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;30130,57\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;621 Skovbrugsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;621 Skovbrugsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;381,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;621 Skovbrugsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;344,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;621 Skovbrugsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;37,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;622 Arbejde med fiskeri og jagt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;622 Arbejde med fiskeri og jagt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;622 Arbejde med fiskeri og jagt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;622 Arbejde med fiskeri og jagt;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;35,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;622 Arbejde med fiskeri og jagt;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;31,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;622 Arbejde med fiskeri og jagt;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;711 Bygningsarbejde (ekskl. medhjælpere) - basis;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;33644,08\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;711 Bygningsarbejde (ekskl. medhjælpere) - basis;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;33658,29\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;711 Bygningsarbejde (ekskl. medhjælpere) - basis;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31782,89\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;711 Bygningsarbejde (ekskl. medhjælpere) - basis;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;15174,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;711 Bygningsarbejde (ekskl. medhjælpere) - basis;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;15058,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;711 Bygningsarbejde (ekskl. medhjælpere) - basis;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;117,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;712 Bygningsarbejde (ekskl. medhjælpere) - finish;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37205,70\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;712 Bygningsarbejde (ekskl. medhjælpere) - finish;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;37248,74\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;712 Bygningsarbejde (ekskl. medhjælpere) - finish;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31501,94\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;712 Bygningsarbejde (ekskl. medhjælpere) - finish;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;7518,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;712 Bygningsarbejde (ekskl. medhjælpere) - finish;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;7462,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;712 Bygningsarbejde (ekskl. medhjælpere) - finish;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;56,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;713 Malere og arbejde inden for rensning af bygninger;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31660,70\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;713 Malere og arbejde inden for rensning af bygninger;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32061,91\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;713 Malere og arbejde inden for rensning af bygninger;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29977,83\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;713 Malere og arbejde inden for rensning af bygninger;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4076,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;713 Malere og arbejde inden for rensning af bygninger;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3265,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;713 Malere og arbejde inden for rensning af bygninger;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;811,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;721 Pladearbejde, svejsning og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;33481,42\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;721 Pladearbejde, svejsning og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;33552,59\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;721 Pladearbejde, svejsning og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29374,16\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;721 Pladearbejde, svejsning og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;8105,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;721 Pladearbejde, svejsning og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;7963,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;721 Pladearbejde, svejsning og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;142,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;722 Smede, værktøjsmagere og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;35179,19\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;722 Smede, værktøjsmagere og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;35297,06\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;722 Smede, værktøjsmagere og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31482,43\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;722 Smede, værktøjsmagere og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;18114,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;722 Smede, værktøjsmagere og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;17528,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;722 Smede, værktøjsmagere og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;586,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;723 Mekanikerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;35096,79\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;723 Mekanikerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;35122,81\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;723 Mekanikerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32504,42\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;723 Mekanikerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;12644,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;723 Mekanikerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;12510,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;723 Mekanikerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;133,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;731 Præcisionshåndværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31624,16\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;731 Præcisionshåndværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;33517,76\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;731 Præcisionshåndværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28512,11\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;731 Præcisionshåndværk;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;912,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;731 Præcisionshåndværk;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;555,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;731 Præcisionshåndværk;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;356,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;732 Grafisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36596,90\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;732 Grafisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;38196,24\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;732 Grafisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32339,68\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;732 Grafisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2641,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;732 Grafisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1922,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;732 Grafisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;719,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;741 Installation- og reparation af elektrisk udstyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36412,64\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;741 Installation- og reparation af elektrisk udstyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;36457,67\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;741 Installation- og reparation af elektrisk udstyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32976,99\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;741 Installation- og reparation af elektrisk udstyr;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;14134,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;741 Installation- og reparation af elektrisk udstyr;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;13947,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;741 Installation- og reparation af elektrisk udstyr;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;187,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;742 Installation og reparation af elektronik og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36925,09\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;742 Installation og reparation af elektronik og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;37419,56\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;742 Installation og reparation af elektronik og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29765,99\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;742 Installation og reparation af elektronik og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1832,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;742 Installation og reparation af elektronik og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1710,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;742 Installation og reparation af elektronik og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;121,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;751 Arbejde inden for nærings- og nydelsesmiddelområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31833,11\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;751 Arbejde inden for nærings- og nydelsesmiddelområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;33043,58\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;751 Arbejde inden for nærings- og nydelsesmiddelområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28597,98\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;751 Arbejde inden for nærings- og nydelsesmiddelområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;7406,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;751 Arbejde inden for nærings- og nydelsesmiddelområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;5366,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;751 Arbejde inden for nærings- og nydelsesmiddelområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2040,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;752 Håndværkspræget arbejde i træindustri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;28847,14\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;752 Håndværkspræget arbejde i træindustri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;28992,76\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;752 Håndværkspræget arbejde i træindustri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;27121,66\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;752 Håndværkspræget arbejde i træindustri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1556,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;752 Håndværkspræget arbejde i træindustri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1435,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;752 Håndværkspræget arbejde i træindustri;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;121,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;753 Tekstil- og beklædningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;30961,93\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;753 Tekstil- og beklædningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;31740,42\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;753 Tekstil- og beklædningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;30300,38\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;753 Tekstil- og beklædningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;519,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;753 Tekstil- og beklædningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;236,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;753 Tekstil- og beklædningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;283,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;754 Andet håndværkspræget arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37047,09\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;754 Andet håndværkspræget arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;37652,96\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;754 Andet håndværkspræget arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;35542,76\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;754 Andet håndværkspræget arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2076,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;754 Andet håndværkspræget arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1474,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;754 Andet håndværkspræget arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;602,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;34425,23\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;34576,35\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;30625,76\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1352,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1299,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;53,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31424,27\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;31760,43\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29905,99\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5188,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4222,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;966,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;39804,82\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;40187,39\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;38341,03\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3211,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2557,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;654,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32145,01\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32774,66\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;30404,52\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6056,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4414,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1641,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;29102,52\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32466,65\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;25692,96\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;971,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;488,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;483,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32606,55\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;33646,23\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29786,48\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;14349,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;10461,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3888,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;817 Operatørarbejde ved træforarbejdning og papirfremstilling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;29442,26\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;817 Operatørarbejde ved træforarbejdning og papirfremstilling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;29684,73\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;817 Operatørarbejde ved træforarbejdning og papirfremstilling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28197,93\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;817 Operatørarbejde ved træforarbejdning og papirfremstilling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3036,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;817 Operatørarbejde ved træforarbejdning og papirfremstilling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2532,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;817 Operatørarbejde ved træforarbejdning og papirfremstilling;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;505,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;818 Andet operatørarbejde ved stationære anlæg og maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;35146,10\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;818 Andet operatørarbejde ved stationære anlæg og maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;36208,94\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;818 Andet operatørarbejde ved stationære anlæg og maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31569,39\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;818 Andet operatørarbejde ved stationære anlæg og maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6106,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;818 Andet operatørarbejde ved stationære anlæg og maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4695,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;818 Andet operatørarbejde ved stationære anlæg og maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1411,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;821 Monteringsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;30231,90\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;821 Monteringsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;31888,22\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;821 Monteringsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;27410,03\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;821 Monteringsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;9255,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;821 Monteringsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;5729,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;821 Monteringsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3526,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;831 Lokomotivførerarbejde og tograngering;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;42071,08\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;831 Lokomotivførerarbejde og tograngering;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;42141,45\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;831 Lokomotivførerarbejde og tograngering;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;40357,28\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;831 Lokomotivførerarbejde og tograngering;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2803,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;831 Lokomotivførerarbejde og tograngering;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2690,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;831 Lokomotivførerarbejde og tograngering;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;113,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;832 Chauffører af biler, varevogne og motorcykler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;28045,59\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;832 Chauffører af biler, varevogne og motorcykler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;28098,05\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;832 Chauffører af biler, varevogne og motorcykler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;26786,10\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;832 Chauffører af biler, varevogne og motorcykler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2864,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;832 Chauffører af biler, varevogne og motorcykler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2750,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;832 Chauffører af biler, varevogne og motorcykler;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;114,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;833 Chauffører af busser og lastbiler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;29919,76\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;833 Chauffører af busser og lastbiler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;29979,43\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;833 Chauffører af busser og lastbiler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28737,11\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;833 Chauffører af busser og lastbiler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;20717,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;833 Chauffører af busser og lastbiler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;19687,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;833 Chauffører af busser og lastbiler;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1029,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;834 Førere af mobile maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32230,63\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;834 Førere af mobile maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32299,33\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;834 Førere af mobile maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29574,16\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;834 Førere af mobile maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4432,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;834 Førere af mobile maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4315,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;834 Førere af mobile maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;117,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;835 Dæksarbejde og medhjælp om bord på skibe;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37073,89\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;835 Dæksarbejde og medhjælp om bord på skibe;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;37269,78\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;835 Dæksarbejde og medhjælp om bord på skibe;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;835 Dæksarbejde og medhjælp om bord på skibe;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;327,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;835 Dæksarbejde og medhjælp om bord på skibe;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;316,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;835 Dæksarbejde og medhjælp om bord på skibe;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;11,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;911 Rengøringsarbejde i private hjem, hoteller og kontorer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;26318,86\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;911 Rengøringsarbejde i private hjem, hoteller og kontorer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;27483,50\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;911 Rengøringsarbejde i private hjem, hoteller og kontorer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;25778,24\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;911 Rengøringsarbejde i private hjem, hoteller og kontorer;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;33346,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;911 Rengøringsarbejde i private hjem, hoteller og kontorer;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;10050,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;911 Rengøringsarbejde i private hjem, hoteller og kontorer;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;23297,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;26443,86\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;27200,56\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;25540,17\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2885,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1545,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1340,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;26565,03\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;27077,34\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;25157,36\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;432,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;315,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;117,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32394,26\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32507,39\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29735,45\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;22783,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;21760,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1024,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;932 Manuelt produktionsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;28799,43\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;932 Manuelt produktionsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;29708,58\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;932 Manuelt produktionsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;26931,16\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;932 Manuelt produktionsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;11287,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;932 Manuelt produktionsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;7587,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;932 Manuelt produktionsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3700,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;933 Transport- og lagerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;29743,40\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;933 Transport- og lagerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;30239,62\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;933 Transport- og lagerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;26750,96\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;933 Transport- og lagerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;21006,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;933 Transport- og lagerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;17965,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;933 Transport- og lagerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3041,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;941 Manuelt arbejde med tilberedning af mad;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;25717,19\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;941 Manuelt arbejde med tilberedning af mad;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;23731,13\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;941 Manuelt arbejde med tilberedning af mad;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;26260,89\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;941 Manuelt arbejde med tilberedning af mad;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;10704,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;941 Manuelt arbejde med tilberedning af mad;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1982,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;941 Manuelt arbejde med tilberedning af mad;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;8722,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;951 Servicearbejde på gaden;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;26935,24\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;951 Servicearbejde på gaden;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;27005,53\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;951 Servicearbejde på gaden;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;951 Servicearbejde på gaden;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;59,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;951 Servicearbejde på gaden;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;58,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;951 Servicearbejde på gaden;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;952 Gadesalg (undtagen madvarer);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;952 Gadesalg (undtagen madvarer);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;952 Gadesalg (undtagen madvarer);STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;952 Gadesalg (undtagen madvarer);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;952 Gadesalg (undtagen madvarer);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;952 Gadesalg (undtagen madvarer);Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;961 Renovationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32127,24\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;961 Renovationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32339,49\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;961 Renovationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28483,76\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;961 Renovationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2000,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;961 Renovationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1890,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;961 Renovationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;110,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;962 Andet manuelt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31842,07\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;962 Andet manuelt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32970,49\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;962 Andet manuelt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28342,21\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;962 Andet manuelt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4856,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;962 Andet manuelt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3647,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2014;962 Andet manuelt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1209,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;011 Militært arbejde på officersniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;51350,30\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;011 Militært arbejde på officersniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;51682,48\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;011 Militært arbejde på officersniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;45545,84\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;011 Militært arbejde på officersniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3706,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;011 Militært arbejde på officersniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3500,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;011 Militært arbejde på officersniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;206,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37399,96\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;37595,18\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33867,43\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3860,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3653,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;206,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;031 Militært arbejde, øvrige rangordener;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32023,62\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;031 Militært arbejde, øvrige rangordener;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32201,03\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;031 Militært arbejde, øvrige rangordener;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29668,47\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;031 Militært arbejde, øvrige rangordener;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;7229,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;031 Militært arbejde, øvrige rangordener;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6704,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;031 Militært arbejde, øvrige rangordener;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;525,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;69918,35\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;72284,81\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;65078,47\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2106,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1415,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;691,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;112 Øverste virksomhedsledelse;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;101332,53\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;112 Øverste virksomhedsledelse;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;103740,54\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;112 Øverste virksomhedsledelse;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;85521,06\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;112 Øverste virksomhedsledelse;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;12274,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;112 Øverste virksomhedsledelse;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;10629,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;112 Øverste virksomhedsledelse;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1644,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;121 Ledelse inden for forretningsservice og administrative funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;69406,67\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;121 Ledelse inden for forretningsservice og administrative funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;72597,66\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;121 Ledelse inden for forretningsservice og administrative funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;64067,64\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;121 Ledelse inden for forretningsservice og administrative funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;17435,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;121 Ledelse inden for forretningsservice og administrative funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;10821,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;121 Ledelse inden for forretningsservice og administrative funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;6614,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;122 Ledelse af forretnings- og udviklingsorienterede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;70288,05\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;122 Ledelse af forretnings- og udviklingsorienterede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;73235,37\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;122 Ledelse af forretnings- og udviklingsorienterede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;61190,41\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;122 Ledelse af forretnings- og udviklingsorienterede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;11466,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;122 Ledelse af forretnings- og udviklingsorienterede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;8609,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;122 Ledelse af forretnings- og udviklingsorienterede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2856,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;57417,87\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;57905,90\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;248,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;210,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;38,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;60930,14\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;61033,19\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;60134,98\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;9345,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;8256,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1089,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;72806,54\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;73310,38\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;69623,95\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2371,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2033,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;338,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;134 Ledelse af hovedaktiviteten inden for servicefag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;54712,51\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;134 Ledelse af hovedaktiviteten inden for servicefag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;61527,93\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;134 Ledelse af hovedaktiviteten inden for servicefag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;49204,04\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;134 Ledelse af hovedaktiviteten inden for servicefag;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;13506,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;134 Ledelse af hovedaktiviteten inden for servicefag;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;5858,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;134 Ledelse af hovedaktiviteten inden for servicefag;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7649,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;49593,47\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;51890,52\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;46115,68\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;411,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;246,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;164,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;42625,86\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;45986,57\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;36122,95\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5268,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3435,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1833,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;143 Ledelse af hovedaktiviteten inden for andre serviceområder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;57590,47\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;143 Ledelse af hovedaktiviteten inden for andre serviceområder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;59529,89\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;143 Ledelse af hovedaktiviteten inden for andre serviceområder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;52936,21\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;143 Ledelse af hovedaktiviteten inden for andre serviceområder;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1260,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;143 Ledelse af hovedaktiviteten inden for andre serviceområder;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;885,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;143 Ledelse af hovedaktiviteten inden for andre serviceområder;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;375,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;211 Arbejde inden for fysik og geologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;53404,33\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;211 Arbejde inden for fysik og geologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;54860,09\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;211 Arbejde inden for fysik og geologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;51203,56\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;211 Arbejde inden for fysik og geologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2244,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;211 Arbejde inden for fysik og geologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1335,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;211 Arbejde inden for fysik og geologi;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;909,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;63393,94\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;65588,38\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;59073,79\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1260,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;827,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;433,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;213 Arbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;48213,98\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;213 Arbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;49624,84\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;213 Arbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;46436,11\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;213 Arbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4259,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;213 Arbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2332,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;213 Arbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1928,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;214 Ingeniørarbejde (undtagen inden for elektroteknologi);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;55412,40\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;214 Ingeniørarbejde (undtagen inden for elektroteknologi);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;56394,61\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;214 Ingeniørarbejde (undtagen inden for elektroteknologi);STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;51991,26\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;214 Ingeniørarbejde (undtagen inden for elektroteknologi);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;34907,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;214 Ingeniørarbejde (undtagen inden for elektroteknologi);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;26805,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;214 Ingeniørarbejde (undtagen inden for elektroteknologi);Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;8101,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;215 Ingeniørarbejde inden for elektroteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;57564,68\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;215 Ingeniørarbejde inden for elektroteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;57731,80\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;215 Ingeniørarbejde inden for elektroteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;55165,91\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;215 Ingeniørarbejde inden for elektroteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5744,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;215 Ingeniørarbejde inden for elektroteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;5354,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;215 Ingeniørarbejde inden for elektroteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;390,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;216 Arbejde med arkitektur, infrastruktur og design;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;44449,82\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;216 Arbejde med arkitektur, infrastruktur og design;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;47331,25\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;216 Arbejde med arkitektur, infrastruktur og design;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;41525,73\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;216 Arbejde med arkitektur, infrastruktur og design;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6671,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;216 Arbejde med arkitektur, infrastruktur og design;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3322,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;216 Arbejde med arkitektur, infrastruktur og design;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3349,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;221 Lægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;69238,74\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;221 Lægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;74684,79\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;221 Lægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;63804,13\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;221 Lægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;16447,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;221 Lægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;7928,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;221 Lægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;8518,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;222 Sygeplejerske- og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38673,14\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;222 Sygeplejerske- og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;41537,56\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;222 Sygeplejerske- og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;38541,38\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;222 Sygeplejerske- og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;51590,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;222 Sygeplejerske- og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2158,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;222 Sygeplejerske- og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;49432,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;223 Arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;223 Arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;223 Arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;223 Arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;8,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;223 Arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;223 Arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;6,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;224 Paramedicinsk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;42621,46\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;224 Paramedicinsk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;42603,42\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;224 Paramedicinsk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;224 Paramedicinsk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;352,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;224 Paramedicinsk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;327,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;224 Paramedicinsk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;25,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;225 Dyrlægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;49526,51\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;225 Dyrlægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;53366,57\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;225 Dyrlægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;47242,65\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;225 Dyrlægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;800,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;225 Dyrlægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;285,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;225 Dyrlægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;515,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;226 Andet sundhedsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;43601,68\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;226 Andet sundhedsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;51946,24\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;226 Andet sundhedsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;41539,98\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;226 Andet sundhedsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;21894,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;226 Andet sundhedsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4061,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;226 Andet sundhedsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;17833,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;231 Undervisning og forskning ved universiteter og højere læreanstalter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;44895,64\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;231 Undervisning og forskning ved universiteter og højere læreanstalter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;46325,51\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;231 Undervisning og forskning ved universiteter og højere læreanstalter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;43092,85\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;231 Undervisning og forskning ved universiteter og højere læreanstalter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;23354,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;231 Undervisning og forskning ved universiteter og højere læreanstalter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;12825,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;231 Undervisning og forskning ved universiteter og højere læreanstalter;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;10529,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;232 Undervisning ved erhvervsuddannelser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;41151,18\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;232 Undervisning ved erhvervsuddannelser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;41316,72\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;232 Undervisning ved erhvervsuddannelser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;40934,18\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;232 Undervisning ved erhvervsuddannelser;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;10999,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;232 Undervisning ved erhvervsuddannelser;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6186,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;232 Undervisning ved erhvervsuddannelser;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4813,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;233 Almen undervisning efter grundskoleniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;46468,52\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;233 Almen undervisning efter grundskoleniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;47196,85\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;233 Almen undervisning efter grundskoleniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;45796,91\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;233 Almen undervisning efter grundskoleniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;14830,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;233 Almen undervisning efter grundskoleniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;7016,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;233 Almen undervisning efter grundskoleniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7814,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;234 Undervisning og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37651,93\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;234 Undervisning og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;38843,03\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;234 Undervisning og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;37204,24\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;234 Undervisning og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;109788,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;234 Undervisning og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;28653,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;234 Undervisning og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;81135,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;235 Andet undervisnings- og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38124,30\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;235 Andet undervisnings- og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;39065,26\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;235 Andet undervisnings- og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;37659,97\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;235 Andet undervisnings- og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;37012,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;235 Andet undervisnings- og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;11830,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;235 Andet undervisnings- og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;25182,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;241 Arbejde inden for finans og økonomi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;54958,12\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;241 Arbejde inden for finans og økonomi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;58922,97\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;241 Arbejde inden for finans og økonomi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;49845,06\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;241 Arbejde inden for finans og økonomi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;29363,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;241 Arbejde inden for finans og økonomi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;16298,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;241 Arbejde inden for finans og økonomi;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;13065,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;242 Arbejde inden for virksomhedsadministration - privat og offentlig;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;57955,22\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;242 Arbejde inden for virksomhedsadministration - privat og offentlig;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;61366,83\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;242 Arbejde inden for virksomhedsadministration - privat og offentlig;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;54820,29\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;242 Arbejde inden for virksomhedsadministration - privat og offentlig;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;14219,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;242 Arbejde inden for virksomhedsadministration - privat og offentlig;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6699,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;242 Arbejde inden for virksomhedsadministration - privat og offentlig;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7520,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;243 Arbejde inden for salg, marketing og PR;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;52671,33\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;243 Arbejde inden for salg, marketing og PR;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;57073,68\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;243 Arbejde inden for salg, marketing og PR;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;47293,91\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;243 Arbejde inden for salg, marketing og PR;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;13341,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;243 Arbejde inden for salg, marketing og PR;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;7259,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;243 Arbejde inden for salg, marketing og PR;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;6081,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;251 Udvikling og analyse af software og applikatoner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;54776,16\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;251 Udvikling og analyse af software og applikatoner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;55599,74\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;251 Udvikling og analyse af software og applikatoner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;51880,69\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;251 Udvikling og analyse af software og applikatoner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;32553,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;251 Udvikling og analyse af software og applikatoner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;25127,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;251 Udvikling og analyse af software og applikatoner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7426,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;252 Arbejde med databaser og netværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;49650,67\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;252 Arbejde med databaser og netværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;50247,27\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;252 Arbejde med databaser og netværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;46843,35\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;252 Arbejde med databaser og netværk;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6556,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;252 Arbejde med databaser og netværk;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;5381,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;252 Arbejde med databaser og netværk;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1175,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;261 Juridisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;54001,93\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;261 Juridisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;57274,69\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;261 Juridisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;51593,47\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;261 Juridisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;13963,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;261 Juridisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;5735,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;261 Juridisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;8229,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;42077,68\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;43560,59\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;41480,56\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3271,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;911,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2359,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;263 Arbejde inden for samfundsvidenskab og religion;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;42110,50\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;263 Arbejde inden for samfundsvidenskab og religion;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;45927,85\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;263 Arbejde inden for samfundsvidenskab og religion;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;40552,25\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;263 Arbejde inden for samfundsvidenskab og religion;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;25789,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;263 Arbejde inden for samfundsvidenskab og religion;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;7102,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;263 Arbejde inden for samfundsvidenskab og religion;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;18687,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;48189,33\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;50574,24\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;45577,32\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;7394,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3808,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3586,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;265 Arbejde med kunst og kreative fag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;41334,42\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;265 Arbejde med kunst og kreative fag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;42494,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;265 Arbejde med kunst og kreative fag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;39726,23\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;265 Arbejde med kunst og kreative fag;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2232,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;265 Arbejde med kunst og kreative fag;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1286,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;265 Arbejde med kunst og kreative fag;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;946,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;42569,56\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;44789,82\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;37779,31\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;39327,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;26423,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;12904,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;313 Teknisk kontrolarbejde inden for processtyring;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;44849,23\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;313 Teknisk kontrolarbejde inden for processtyring;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;46589,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;313 Teknisk kontrolarbejde inden for processtyring;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;41329,02\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;313 Teknisk kontrolarbejde inden for processtyring;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2791,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;313 Teknisk kontrolarbejde inden for processtyring;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1843,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;313 Teknisk kontrolarbejde inden for processtyring;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;948,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;314 Teknikerarbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;41673,59\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;314 Teknikerarbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;42744,29\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;314 Teknikerarbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;38558,47\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;314 Teknikerarbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;771,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;314 Teknikerarbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;567,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;314 Teknikerarbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;204,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;315 Teknikerarbejde inden for skibs- og luftfart;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;67606,51\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;315 Teknikerarbejde inden for skibs- og luftfart;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;68488,03\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;315 Teknikerarbejde inden for skibs- og luftfart;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;58497,56\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;315 Teknikerarbejde inden for skibs- og luftfart;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3259,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;315 Teknikerarbejde inden for skibs- og luftfart;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2971,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;315 Teknikerarbejde inden for skibs- og luftfart;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;288,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36575,58\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;38775,48\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;36250,68\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;11172,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1419,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;9753,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;41827,14\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;40277,07\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;42855,50\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;138,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;53,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;85,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;9,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;0,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;8,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;29595,94\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28982,32\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;222,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;23,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;200,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38247,55\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;42891,09\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;34886,24\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5958,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2380,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3577,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;331 Arbejde med finans, regnskab og matematik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;44794,25\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;331 Arbejde med finans, regnskab og matematik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;51892,47\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;331 Arbejde med finans, regnskab og matematik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;40632,26\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;331 Arbejde med finans, regnskab og matematik;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;25834,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;331 Arbejde med finans, regnskab og matematik;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;9470,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;331 Arbejde med finans, regnskab og matematik;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;16364,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;332 Agent- og mæglerarbejde inden for salg og indkøb;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;46405,49\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;332 Agent- og mæglerarbejde inden for salg og indkøb;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;49445,86\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;332 Agent- og mæglerarbejde inden for salg og indkøb;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;41212,38\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;332 Agent- og mæglerarbejde inden for salg og indkøb;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;32174,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;332 Agent- og mæglerarbejde inden for salg og indkøb;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;20082,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;332 Agent- og mæglerarbejde inden for salg og indkøb;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;12092,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;333 Arbejde inden for forretningsservice;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;41987,49\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;333 Arbejde inden for forretningsservice;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;45507,90\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;333 Arbejde inden for forretningsservice;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;38506,44\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;333 Arbejde inden for forretningsservice;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;9509,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;333 Arbejde inden for forretningsservice;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4620,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;333 Arbejde inden for forretningsservice;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4889,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;334 Administrativt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38680,47\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;334 Administrativt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;45147,54\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;334 Administrativt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;37568,45\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;334 Administrativt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;27760,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;334 Administrativt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4067,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;334 Administrativt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;23693,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;335 Arbejde med forvaltning og udøvelse af lovgivning;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;42819,61\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;335 Arbejde med forvaltning og udøvelse af lovgivning;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;47321,19\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;335 Arbejde med forvaltning og udøvelse af lovgivning;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;39537,82\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;335 Arbejde med forvaltning og udøvelse af lovgivning;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;10368,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;335 Arbejde med forvaltning og udøvelse af lovgivning;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4295,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;335 Arbejde med forvaltning og udøvelse af lovgivning;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;6073,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;341 Arbejde inden for jura og samfundsforhold;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;34273,87\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;341 Arbejde inden for jura og samfundsforhold;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;35291,98\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;341 Arbejde inden for jura og samfundsforhold;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33835,74\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;341 Arbejde inden for jura og samfundsforhold;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6236,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;341 Arbejde inden for jura og samfundsforhold;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1846,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;341 Arbejde inden for jura og samfundsforhold;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4390,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;342 Arbejde med sport og fitness;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;33004,04\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;342 Arbejde med sport og fitness;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;36253,93\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;342 Arbejde med sport og fitness;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;27608,01\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;342 Arbejde med sport og fitness;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1227,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;342 Arbejde med sport og fitness;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;745,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;342 Arbejde med sport og fitness;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;483,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36562,73\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;39096,65\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;34257,29\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3841,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1771,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2070,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;45442,33\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;46037,35\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;42901,85\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;8303,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6693,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1610,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;43519,88\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;43552,92\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;43353,49\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3252,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2691,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;560,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;411 Almindeligt kontorarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;35400,08\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;411 Almindeligt kontorarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;38703,87\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;411 Almindeligt kontorarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;34459,61\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;411 Almindeligt kontorarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;60178,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;411 Almindeligt kontorarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;12786,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;411 Almindeligt kontorarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;47391,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;412 Almindeligt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37245,18\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;412 Almindeligt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;40536,15\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;412 Almindeligt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;36982,47\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;412 Almindeligt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5718,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;412 Almindeligt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;400,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;412 Almindeligt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;5318,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;413 Indtastningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37956,89\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;413 Indtastningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;40802,79\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;413 Indtastningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;35034,89\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;413 Indtastningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2647,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;413 Indtastningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1331,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;413 Indtastningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1316,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36881,35\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;38489,23\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;36275,71\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2338,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;609,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1730,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;422 Kundeinformationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31954,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;422 Kundeinformationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;33109,39\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;422 Kundeinformationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31407,05\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;422 Kundeinformationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;11434,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;422 Kundeinformationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3618,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;422 Kundeinformationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7816,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;431 Almindeligt beregningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;39246,75\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;431 Almindeligt beregningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;43124,98\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;431 Almindeligt beregningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;38165,03\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;431 Almindeligt beregningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;16012,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;431 Almindeligt beregningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3462,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;431 Almindeligt beregningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;12550,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;432 Kontorarbejde inden for materialeregistrering og transport;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;33031,91\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;432 Kontorarbejde inden for materialeregistrering og transport;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;33431,87\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;432 Kontorarbejde inden for materialeregistrering og transport;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31882,42\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;432 Kontorarbejde inden for materialeregistrering og transport;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;15253,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;432 Kontorarbejde inden for materialeregistrering og transport;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;11206,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;432 Kontorarbejde inden for materialeregistrering og transport;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4047,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;441 Andet almindeligt kontor- og kundeservicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;33049,74\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;441 Andet almindeligt kontor- og kundeservicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32828,70\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;441 Andet almindeligt kontor- og kundeservicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33290,31\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;441 Andet almindeligt kontor- og kundeservicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;19368,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;441 Andet almindeligt kontor- og kundeservicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;9873,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;441 Andet almindeligt kontor- og kundeservicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;9495,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;511 Service- og kontrolarbejde under transport og rejser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38137,77\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;511 Service- og kontrolarbejde under transport og rejser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;39209,47\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;511 Service- og kontrolarbejde under transport og rejser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;37316,15\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;511 Service- og kontrolarbejde under transport og rejser;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4371,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;511 Service- og kontrolarbejde under transport og rejser;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1863,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;511 Service- og kontrolarbejde under transport og rejser;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2509,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;512 Kokkearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;29428,97\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;512 Kokkearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;29758,37\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;512 Kokkearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28989,87\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;512 Kokkearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4973,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;512 Kokkearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2734,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;512 Kokkearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2239,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;513 Tjenere og bartendere;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;25759,83\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;513 Tjenere og bartendere;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;26943,41\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;513 Tjenere og bartendere;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;24839,46\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;513 Tjenere og bartendere;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4309,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;513 Tjenere og bartendere;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1873,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;513 Tjenere og bartendere;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2436,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;30771,69\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;34094,85\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;30394,42\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;579,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;56,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;523,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32862,56\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32905,57\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32560,06\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;14997,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;13100,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1897,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;516 Andet servicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;29351,07\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;516 Andet servicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;30630,39\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;516 Andet servicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;26809,97\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;516 Andet servicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2174,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;516 Andet servicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1437,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;516 Andet servicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;736,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;521 Salgsarbejde på gade og marked;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;521 Salgsarbejde på gade og marked;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;521 Salgsarbejde på gade og marked;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;521 Salgsarbejde på gade og marked;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;40,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;521 Salgsarbejde på gade og marked;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;32,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;521 Salgsarbejde på gade og marked;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;8,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;522 Salgsarbejde i butik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;26888,02\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;522 Salgsarbejde i butik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;28957,04\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;522 Salgsarbejde i butik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;25094,82\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;522 Salgsarbejde i butik;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;30465,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;522 Salgsarbejde i butik;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;13957,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;522 Salgsarbejde i butik;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;16508,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;523 Kasseassistentarbejde og beslægtet kundebetjening;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;24967,82\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;523 Kasseassistentarbejde og beslægtet kundebetjening;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;25527,66\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;523 Kasseassistentarbejde og beslægtet kundebetjening;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;24761,86\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;523 Kasseassistentarbejde og beslægtet kundebetjening;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;8276,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;523 Kasseassistentarbejde og beslægtet kundebetjening;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2195,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;523 Kasseassistentarbejde og beslægtet kundebetjening;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;6081,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;524 Andet salgsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38223,82\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;524 Andet salgsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;41100,98\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;524 Andet salgsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33905,18\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;524 Andet salgsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;10970,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;524 Andet salgsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6515,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;524 Andet salgsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4455,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;531 Børneomsorgsarbejde og hjælpelærerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;27343,25\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;531 Børneomsorgsarbejde og hjælpelærerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;26873,47\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;531 Børneomsorgsarbejde og hjælpelærerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;27466,33\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;531 Børneomsorgsarbejde og hjælpelærerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;36819,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;531 Børneomsorgsarbejde og hjælpelærerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;7460,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;531 Børneomsorgsarbejde og hjælpelærerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;29359,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;532 Omsorgsarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31824,20\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;532 Omsorgsarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;31033,89\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;532 Omsorgsarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31955,10\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;532 Omsorgsarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;87439,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;532 Omsorgsarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;11909,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;532 Omsorgsarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;75530,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;541 Rednings- og overvågningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;35141,76\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;541 Rednings- og overvågningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;35548,80\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;541 Rednings- og overvågningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33132,27\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;541 Rednings- og overvågningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;19826,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;541 Rednings- og overvågningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;16441,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;541 Rednings- og overvågningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3385,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;611 Arbejde med plantevækst;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;30290,55\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;611 Arbejde med plantevækst;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;30784,78\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;611 Arbejde med plantevækst;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28872,51\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;611 Arbejde med plantevækst;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3083,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;611 Arbejde med plantevækst;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2253,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;611 Arbejde med plantevækst;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;830,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;612 Arbejde med dyreopdræt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;30874,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;612 Arbejde med dyreopdræt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;31371,60\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;612 Arbejde med dyreopdræt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;30435,47\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;612 Arbejde med dyreopdræt;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;404,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;612 Arbejde med dyreopdræt;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;181,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;612 Arbejde med dyreopdræt;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;223,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;28067,40\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;27779,11\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;165,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;134,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;31,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;621 Skovbrugsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;30513,99\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;621 Skovbrugsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;30247,74\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;621 Skovbrugsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;621 Skovbrugsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;366,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;621 Skovbrugsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;330,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;621 Skovbrugsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;36,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;622 Arbejde med fiskeri og jagt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;622 Arbejde med fiskeri og jagt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;622 Arbejde med fiskeri og jagt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;622 Arbejde med fiskeri og jagt;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;22,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;622 Arbejde med fiskeri og jagt;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;18,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;622 Arbejde med fiskeri og jagt;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;711 Bygningsarbejde (ekskl. medhjælpere) - basis;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;34213,27\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;711 Bygningsarbejde (ekskl. medhjælpere) - basis;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;34220,34\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;711 Bygningsarbejde (ekskl. medhjælpere) - basis;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33396,47\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;711 Bygningsarbejde (ekskl. medhjælpere) - basis;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;19027,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;711 Bygningsarbejde (ekskl. medhjælpere) - basis;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;18866,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;711 Bygningsarbejde (ekskl. medhjælpere) - basis;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;161,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;712 Bygningsarbejde (ekskl. medhjælpere) - finish;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37903,45\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;712 Bygningsarbejde (ekskl. medhjælpere) - finish;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;37949,33\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;712 Bygningsarbejde (ekskl. medhjælpere) - finish;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32041,43\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;712 Bygningsarbejde (ekskl. medhjælpere) - finish;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;8426,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;712 Bygningsarbejde (ekskl. medhjælpere) - finish;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;8359,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;712 Bygningsarbejde (ekskl. medhjælpere) - finish;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;66,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;713 Malere og arbejde inden for rensning af bygninger;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32150,11\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;713 Malere og arbejde inden for rensning af bygninger;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32511,06\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;713 Malere og arbejde inden for rensning af bygninger;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;30604,40\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;713 Malere og arbejde inden for rensning af bygninger;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4671,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;713 Malere og arbejde inden for rensning af bygninger;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3751,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;713 Malere og arbejde inden for rensning af bygninger;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;920,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;721 Pladearbejde, svejsning og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;34067,80\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;721 Pladearbejde, svejsning og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;34128,31\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;721 Pladearbejde, svejsning og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;30760,90\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;721 Pladearbejde, svejsning og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;8387,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;721 Pladearbejde, svejsning og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;8234,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;721 Pladearbejde, svejsning og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;153,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;722 Smede, værktøjsmagere og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;35466,50\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;722 Smede, værktøjsmagere og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;35579,43\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;722 Smede, værktøjsmagere og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32392,68\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;722 Smede, værktøjsmagere og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;19567,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;722 Smede, værktøjsmagere og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;18849,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;722 Smede, værktøjsmagere og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;718,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;723 Mekanikerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;35514,68\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;723 Mekanikerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;35546,58\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;723 Mekanikerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32358,58\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;723 Mekanikerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;13382,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;723 Mekanikerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;13238,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;723 Mekanikerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;144,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;731 Præcisionshåndværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32426,85\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;731 Præcisionshåndværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;34555,35\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;731 Præcisionshåndværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29509,80\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;731 Præcisionshåndværk;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;981,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;731 Præcisionshåndværk;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;562,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;731 Præcisionshåndværk;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;418,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;732 Grafisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36855,42\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;732 Grafisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;38378,61\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;732 Grafisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33030,61\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;732 Grafisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2611,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;732 Grafisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1874,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;732 Grafisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;737,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;741 Installation- og reparation af elektrisk udstyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37052,95\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;741 Installation- og reparation af elektrisk udstyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;37090,27\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;741 Installation- og reparation af elektrisk udstyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;34144,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;741 Installation- og reparation af elektrisk udstyr;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;15399,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;741 Installation- og reparation af elektrisk udstyr;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;15201,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;741 Installation- og reparation af elektrisk udstyr;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;198,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;742 Installation og reparation af elektronik og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38160,15\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;742 Installation og reparation af elektronik og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;38586,44\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;742 Installation og reparation af elektronik og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31390,46\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;742 Installation og reparation af elektronik og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1832,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;742 Installation og reparation af elektronik og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1721,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;742 Installation og reparation af elektronik og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;111,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;751 Arbejde inden for nærings- og nydelsesmiddelområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32527,15\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;751 Arbejde inden for nærings- og nydelsesmiddelområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;33662,99\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;751 Arbejde inden for nærings- og nydelsesmiddelområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29491,93\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;751 Arbejde inden for nærings- og nydelsesmiddelområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;7517,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;751 Arbejde inden for nærings- og nydelsesmiddelområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;5446,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;751 Arbejde inden for nærings- og nydelsesmiddelområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2071,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;752 Håndværkspræget arbejde i træindustri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;29731,71\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;752 Håndværkspræget arbejde i træindustri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;29885,32\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;752 Håndværkspræget arbejde i træindustri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;27949,84\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;752 Håndværkspræget arbejde i træindustri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1672,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;752 Håndværkspræget arbejde i træindustri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1538,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;752 Håndværkspræget arbejde i træindustri;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;134,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;753 Tekstil- og beklædningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31593,51\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;753 Tekstil- og beklædningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32536,06\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;753 Tekstil- og beklædningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;30934,09\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;753 Tekstil- og beklædningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;484,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;753 Tekstil- og beklædningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;196,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;753 Tekstil- og beklædningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;287,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;754 Andet håndværkspræget arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36961,30\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;754 Andet håndværkspræget arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;37587,05\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;754 Andet håndværkspræget arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;35666,95\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;754 Andet håndværkspræget arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1970,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;754 Andet håndværkspræget arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1328,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;754 Andet håndværkspræget arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;642,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36404,27\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;36580,74\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31593,93\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1522,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1467,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;55,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31675,66\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;31969,58\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;30290,60\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5220,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4286,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;934,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;40509,81\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;40895,98\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;39164,76\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3232,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2523,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;709,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32091,68\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32746,96\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;30384,67\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6586,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4744,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1843,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;29900,15\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32637,83\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;26401,33\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1073,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;598,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;475,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31934,83\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32794,86\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29773,48\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;11831,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;8431,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3400,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;817 Operatørarbejde ved træforarbejdning og papirfremstilling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;29815,71\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;817 Operatørarbejde ved træforarbejdning og papirfremstilling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;30002,36\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;817 Operatørarbejde ved træforarbejdning og papirfremstilling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28843,14\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;817 Operatørarbejde ved træforarbejdning og papirfremstilling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3057,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;817 Operatørarbejde ved træforarbejdning og papirfremstilling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2556,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;817 Operatørarbejde ved træforarbejdning og papirfremstilling;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;500,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;818 Andet operatørarbejde ved stationære anlæg og maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;35948,80\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;818 Andet operatørarbejde ved stationære anlæg og maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;36902,19\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;818 Andet operatørarbejde ved stationære anlæg og maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32535,61\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;818 Andet operatørarbejde ved stationære anlæg og maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6437,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;818 Andet operatørarbejde ved stationære anlæg og maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;5025,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;818 Andet operatørarbejde ved stationære anlæg og maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1411,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;821 Monteringsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31747,26\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;821 Monteringsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;33521,29\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;821 Monteringsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28559,57\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;821 Monteringsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;10995,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;821 Monteringsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;7000,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;821 Monteringsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3995,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;831 Lokomotivførerarbejde og tograngering;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;42384,94\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;831 Lokomotivførerarbejde og tograngering;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;42457,31\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;831 Lokomotivførerarbejde og tograngering;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;40689,46\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;831 Lokomotivførerarbejde og tograngering;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2771,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;831 Lokomotivførerarbejde og tograngering;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2656,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;831 Lokomotivførerarbejde og tograngering;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;115,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;832 Chauffører af biler, varevogne og motorcykler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;28038,62\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;832 Chauffører af biler, varevogne og motorcykler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;28200,08\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;832 Chauffører af biler, varevogne og motorcykler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;25094,64\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;832 Chauffører af biler, varevogne og motorcykler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3158,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;832 Chauffører af biler, varevogne og motorcykler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2995,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;832 Chauffører af biler, varevogne og motorcykler;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;163,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;833 Chauffører af busser og lastbiler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;30508,92\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;833 Chauffører af busser og lastbiler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;30590,96\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;833 Chauffører af busser og lastbiler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28903,98\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;833 Chauffører af busser og lastbiler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;21196,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;833 Chauffører af busser og lastbiler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;20123,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;833 Chauffører af busser og lastbiler;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1073,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;834 Førere af mobile maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32222,50\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;834 Førere af mobile maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32380,56\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;834 Førere af mobile maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28631,47\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;834 Førere af mobile maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5013,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;834 Førere af mobile maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4801,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;834 Førere af mobile maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;212,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;835 Dæksarbejde og medhjælp om bord på skibe;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38971,12\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;835 Dæksarbejde og medhjælp om bord på skibe;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;39068,47\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;835 Dæksarbejde og medhjælp om bord på skibe;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;835 Dæksarbejde og medhjælp om bord på skibe;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;321,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;835 Dæksarbejde og medhjælp om bord på skibe;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;311,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;835 Dæksarbejde og medhjælp om bord på skibe;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;10,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;911 Rengøringsarbejde i private hjem, hoteller og kontorer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;26892,50\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;911 Rengøringsarbejde i private hjem, hoteller og kontorer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;28161,34\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;911 Rengøringsarbejde i private hjem, hoteller og kontorer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;26259,89\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;911 Rengøringsarbejde i private hjem, hoteller og kontorer;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;31980,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;911 Rengøringsarbejde i private hjem, hoteller og kontorer;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;10145,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;911 Rengøringsarbejde i private hjem, hoteller og kontorer;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;21835,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;26967,98\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;27706,24\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;26049,05\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3097,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1680,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1417,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;28362,39\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;28534,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;27690,03\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;505,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;397,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;107,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32873,55\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32986,74\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;30061,25\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;24763,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;23724,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1040,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;932 Manuelt produktionsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;29268,41\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;932 Manuelt produktionsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;30053,04\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;932 Manuelt produktionsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;27694,42\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;932 Manuelt produktionsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;11702,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;932 Manuelt produktionsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;7771,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;932 Manuelt produktionsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3932,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;933 Transport- og lagerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;29931,16\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;933 Transport- og lagerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;30435,47\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;933 Transport- og lagerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;27002,13\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;933 Transport- og lagerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;21106,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;933 Transport- og lagerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;17947,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;933 Transport- og lagerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3159,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;941 Manuelt arbejde med tilberedning af mad;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;25936,06\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;941 Manuelt arbejde med tilberedning af mad;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;24103,46\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;941 Manuelt arbejde med tilberedning af mad;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;26505,80\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;941 Manuelt arbejde med tilberedning af mad;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;11841,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;941 Manuelt arbejde med tilberedning af mad;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2418,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;941 Manuelt arbejde med tilberedning af mad;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;9423,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;951 Servicearbejde på gaden;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;951 Servicearbejde på gaden;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;951 Servicearbejde på gaden;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;951 Servicearbejde på gaden;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;21,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;951 Servicearbejde på gaden;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;19,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;951 Servicearbejde på gaden;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;952 Gadesalg (undtagen madvarer);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;952 Gadesalg (undtagen madvarer);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;952 Gadesalg (undtagen madvarer);STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;952 Gadesalg (undtagen madvarer);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;0,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;952 Gadesalg (undtagen madvarer);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;0,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;952 Gadesalg (undtagen madvarer);Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;0,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;961 Renovationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32534,74\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;961 Renovationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32771,03\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;961 Renovationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28820,42\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;961 Renovationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2400,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;961 Renovationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2253,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;961 Renovationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;146,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;962 Andet manuelt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31390,54\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;962 Andet manuelt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32561,41\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;962 Andet manuelt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28044,83\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;962 Andet manuelt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5699,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;962 Andet manuelt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4218,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2015;962 Andet manuelt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1482,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;011 Militært arbejde på officersniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;52575,18\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;011 Militært arbejde på officersniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;52913,03\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;011 Militært arbejde på officersniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;47036,68\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;011 Militært arbejde på officersniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3584,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;011 Militært arbejde på officersniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3373,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;011 Militært arbejde på officersniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;211,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37770,37\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;37983,63\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;34085,02\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3927,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3709,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;218,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;031 Militært arbejde, øvrige rangordener;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31823,32\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;031 Militært arbejde, øvrige rangordener;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32003,01\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;031 Militært arbejde, øvrige rangordener;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29646,54\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;031 Militært arbejde, øvrige rangordener;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;7434,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;031 Militært arbejde, øvrige rangordener;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6854,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;031 Militært arbejde, øvrige rangordener;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;580,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;74262,44\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;77996,43\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;67116,46\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1735,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1137,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;598,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;112 Øverste virksomhedsledelse;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;100898,07\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;112 Øverste virksomhedsledelse;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;103098,52\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;112 Øverste virksomhedsledelse;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;86386,36\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;112 Øverste virksomhedsledelse;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;13883,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;112 Øverste virksomhedsledelse;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;12049,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;112 Øverste virksomhedsledelse;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1834,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;121 Ledelse inden for forretningsservice og administrative funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;69864,55\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;121 Ledelse inden for forretningsservice og administrative funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;73048,51\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;121 Ledelse inden for forretningsservice og administrative funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;64667,56\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;121 Ledelse inden for forretningsservice og administrative funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;18569,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;121 Ledelse inden for forretningsservice og administrative funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;11437,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;121 Ledelse inden for forretningsservice og administrative funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7133,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;122 Ledelse af forretnings- og udviklingsorienterede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;72350,61\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;122 Ledelse af forretnings- og udviklingsorienterede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;75374,26\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;122 Ledelse af forretnings- og udviklingsorienterede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;63257,82\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;122 Ledelse af forretnings- og udviklingsorienterede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;11854,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;122 Ledelse af forretnings- og udviklingsorienterede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;8848,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;122 Ledelse af forretnings- og udviklingsorienterede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3006,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;58647,99\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;58058,82\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;236,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;204,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;33,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;61599,64\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;61559,52\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;61915,52\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;10134,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;8975,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1159,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;74091,49\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;74413,05\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;72181,18\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2603,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2214,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;389,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;134 Ledelse af hovedaktiviteten inden for servicefag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;55759,01\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;134 Ledelse af hovedaktiviteten inden for servicefag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;62500,12\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;134 Ledelse af hovedaktiviteten inden for servicefag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;50366,54\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;134 Ledelse af hovedaktiviteten inden for servicefag;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;13644,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;134 Ledelse af hovedaktiviteten inden for servicefag;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;5869,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;134 Ledelse af hovedaktiviteten inden for servicefag;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7774,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;49485,45\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;50929,39\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;47091,96\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;482,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;301,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;181,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;42493,70\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;45875,11\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;36001,91\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5261,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3413,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1848,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;143 Ledelse af hovedaktiviteten inden for andre serviceområder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;60084,71\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;143 Ledelse af hovedaktiviteten inden for andre serviceområder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;61398,15\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;143 Ledelse af hovedaktiviteten inden for andre serviceområder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;56858,73\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;143 Ledelse af hovedaktiviteten inden for andre serviceområder;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1208,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;143 Ledelse af hovedaktiviteten inden for andre serviceområder;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;858,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;143 Ledelse af hovedaktiviteten inden for andre serviceområder;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;350,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;211 Arbejde inden for fysik og geologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;54424,65\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;211 Arbejde inden for fysik og geologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;55827,77\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;211 Arbejde inden for fysik og geologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;52364,84\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;211 Arbejde inden for fysik og geologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2362,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;211 Arbejde inden for fysik og geologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1386,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;211 Arbejde inden for fysik og geologi;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;976,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;63344,34\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;65554,11\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;59003,80\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1325,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;868,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;457,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;213 Arbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;49373,60\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;213 Arbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;50566,43\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;213 Arbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;47895,72\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;213 Arbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4304,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;213 Arbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2326,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;213 Arbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1978,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;214 Ingeniørarbejde (undtagen inden for elektroteknologi);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;56274,94\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;214 Ingeniørarbejde (undtagen inden for elektroteknologi);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;57145,87\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;214 Ingeniørarbejde (undtagen inden for elektroteknologi);STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;53226,67\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;214 Ingeniørarbejde (undtagen inden for elektroteknologi);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;37443,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;214 Ingeniørarbejde (undtagen inden for elektroteknologi);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;28743,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;214 Ingeniørarbejde (undtagen inden for elektroteknologi);Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;8700,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;215 Ingeniørarbejde inden for elektroteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;58384,59\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;215 Ingeniørarbejde inden for elektroteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;58590,25\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;215 Ingeniørarbejde inden for elektroteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;55479,81\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;215 Ingeniørarbejde inden for elektroteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6144,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;215 Ingeniørarbejde inden for elektroteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;5724,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;215 Ingeniørarbejde inden for elektroteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;420,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;216 Arbejde med arkitektur, infrastruktur og design;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;44957,38\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;216 Arbejde med arkitektur, infrastruktur og design;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;47794,41\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;216 Arbejde med arkitektur, infrastruktur og design;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;41962,43\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;216 Arbejde med arkitektur, infrastruktur og design;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;7523,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;216 Arbejde med arkitektur, infrastruktur og design;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3792,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;216 Arbejde med arkitektur, infrastruktur og design;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3731,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;221 Lægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;70283,25\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;221 Lægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;75462,19\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;221 Lægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;65198,45\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;221 Lægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;16699,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;221 Lægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;7967,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;221 Lægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;8732,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;222 Sygeplejerske- og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;39286,69\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;222 Sygeplejerske- og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;42291,08\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;222 Sygeplejerske- og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;39146,79\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;222 Sygeplejerske- og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;52164,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;222 Sygeplejerske- og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2214,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;222 Sygeplejerske- og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;49950,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;223 Arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;223 Arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;223 Arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;223 Arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;10,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;223 Arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;223 Arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;224 Paramedicinsk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;42355,11\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;224 Paramedicinsk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;42504,56\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;224 Paramedicinsk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;224 Paramedicinsk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;365,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;224 Paramedicinsk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;339,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;224 Paramedicinsk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;26,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;225 Dyrlægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;49399,88\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;225 Dyrlægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;52995,12\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;225 Dyrlægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;47379,59\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;225 Dyrlægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;821,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;225 Dyrlægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;283,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;225 Dyrlægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;538,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;226 Andet sundhedsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;43991,36\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;226 Andet sundhedsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;51786,19\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;226 Andet sundhedsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;42038,84\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;226 Andet sundhedsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;22311,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;226 Andet sundhedsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4145,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;226 Andet sundhedsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;18166,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;231 Undervisning og forskning ved universiteter og højere læreanstalter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;45513,63\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;231 Undervisning og forskning ved universiteter og højere læreanstalter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;47005,78\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;231 Undervisning og forskning ved universiteter og højere læreanstalter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;43665,60\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;231 Undervisning og forskning ved universiteter og højere læreanstalter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;23810,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;231 Undervisning og forskning ved universiteter og højere læreanstalter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;12995,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;231 Undervisning og forskning ved universiteter og højere læreanstalter;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;10816,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;232 Undervisning ved erhvervsuddannelser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;41427,62\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;232 Undervisning ved erhvervsuddannelser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;41586,84\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;232 Undervisning ved erhvervsuddannelser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;41220,19\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;232 Undervisning ved erhvervsuddannelser;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;10577,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;232 Undervisning ved erhvervsuddannelser;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;5935,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;232 Undervisning ved erhvervsuddannelser;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4642,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;233 Almen undervisning efter grundskoleniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;46997,45\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;233 Almen undervisning efter grundskoleniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;47670,82\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;233 Almen undervisning efter grundskoleniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;46380,45\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;233 Almen undervisning efter grundskoleniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;14739,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;233 Almen undervisning efter grundskoleniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6953,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;233 Almen undervisning efter grundskoleniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7786,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;234 Undervisning og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38214,98\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;234 Undervisning og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;39283,70\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;234 Undervisning og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;37807,67\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;234 Undervisning og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;111201,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;234 Undervisning og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;29423,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;234 Undervisning og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;81778,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;235 Andet undervisnings- og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38479,72\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;235 Andet undervisnings- og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;39307,91\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;235 Andet undervisnings- og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;38074,40\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;235 Andet undervisnings- og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;38036,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;235 Andet undervisnings- og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;12066,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;235 Andet undervisnings- og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;25970,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;241 Arbejde inden for finans og økonomi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;55792,78\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;241 Arbejde inden for finans og økonomi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;59824,28\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;241 Arbejde inden for finans og økonomi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;50626,31\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;241 Arbejde inden for finans og økonomi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;30481,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;241 Arbejde inden for finans og økonomi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;16833,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;241 Arbejde inden for finans og økonomi;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;13647,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;242 Arbejde inden for virksomhedsadministration - privat og offentlig;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;59001,24\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;242 Arbejde inden for virksomhedsadministration - privat og offentlig;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;62404,44\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;242 Arbejde inden for virksomhedsadministration - privat og offentlig;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;55902,03\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;242 Arbejde inden for virksomhedsadministration - privat og offentlig;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;15275,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;242 Arbejde inden for virksomhedsadministration - privat og offentlig;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;7163,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;242 Arbejde inden for virksomhedsadministration - privat og offentlig;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;8112,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;243 Arbejde inden for salg, marketing og PR;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;52813,38\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;243 Arbejde inden for salg, marketing og PR;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;57048,97\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;243 Arbejde inden for salg, marketing og PR;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;47568,66\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;243 Arbejde inden for salg, marketing og PR;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;14748,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;243 Arbejde inden for salg, marketing og PR;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;8068,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;243 Arbejde inden for salg, marketing og PR;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;6680,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;251 Udvikling og analyse af software og applikatoner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;55551,83\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;251 Udvikling og analyse af software og applikatoner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;56460,53\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;251 Udvikling og analyse af software og applikatoner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;52375,50\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;251 Udvikling og analyse af software og applikatoner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;36747,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;251 Udvikling og analyse af software og applikatoner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;28232,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;251 Udvikling og analyse af software og applikatoner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;8515,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;252 Arbejde med databaser og netværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;51358,34\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;252 Arbejde med databaser og netværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;52066,54\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;252 Arbejde med databaser og netværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;47999,45\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;252 Arbejde med databaser og netværk;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;7058,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;252 Arbejde med databaser og netværk;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;5790,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;252 Arbejde med databaser og netværk;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1269,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;261 Juridisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;54449,03\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;261 Juridisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;57879,40\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;261 Juridisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;51981,65\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;261 Juridisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;14774,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;261 Juridisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;5969,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;261 Juridisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;8804,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;42883,37\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;44076,02\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;42385,96\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3201,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;917,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2284,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;263 Arbejde inden for samfundsvidenskab og religion;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;42806,30\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;263 Arbejde inden for samfundsvidenskab og religion;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;47024,90\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;263 Arbejde inden for samfundsvidenskab og religion;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;41159,37\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;263 Arbejde inden for samfundsvidenskab og religion;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;26959,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;263 Arbejde inden for samfundsvidenskab og religion;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;7175,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;263 Arbejde inden for samfundsvidenskab og religion;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;19784,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;47382,17\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;49215,41\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;45359,70\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;8111,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4180,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3931,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;265 Arbejde med kunst og kreative fag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;43120,23\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;265 Arbejde med kunst og kreative fag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;44776,63\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;265 Arbejde med kunst og kreative fag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;40943,20\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;265 Arbejde med kunst og kreative fag;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2366,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;265 Arbejde med kunst og kreative fag;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1328,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;265 Arbejde med kunst og kreative fag;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1038,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;43927,54\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;46159,96\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;38650,48\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;41856,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;28931,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;12925,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;313 Teknisk kontrolarbejde inden for processtyring;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;45878,96\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;313 Teknisk kontrolarbejde inden for processtyring;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;47835,58\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;313 Teknisk kontrolarbejde inden for processtyring;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;41814,14\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;313 Teknisk kontrolarbejde inden for processtyring;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2890,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;313 Teknisk kontrolarbejde inden for processtyring;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1922,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;313 Teknisk kontrolarbejde inden for processtyring;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;969,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;314 Teknikerarbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;41923,30\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;314 Teknikerarbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;42883,86\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;314 Teknikerarbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;39190,59\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;314 Teknikerarbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;748,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;314 Teknikerarbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;547,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;314 Teknikerarbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;201,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;315 Teknikerarbejde inden for skibs- og luftfart;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;66803,32\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;315 Teknikerarbejde inden for skibs- og luftfart;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;67987,21\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;315 Teknikerarbejde inden for skibs- og luftfart;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;55232,33\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;315 Teknikerarbejde inden for skibs- og luftfart;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3217,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;315 Teknikerarbejde inden for skibs- og luftfart;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2914,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;315 Teknikerarbejde inden for skibs- og luftfart;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;302,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37414,55\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;39160,21\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;37121,38\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;10530,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1472,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;9057,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;42443,53\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;41169,01\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;43564,43\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;135,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;60,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;75,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;7,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;0,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;28265,14\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;27923,42\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;191,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;11,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;179,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38554,24\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;42898,46\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;35727,78\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6127,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2328,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3798,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;331 Arbejde med finans, regnskab og matematik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;45704,02\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;331 Arbejde med finans, regnskab og matematik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;52663,60\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;331 Arbejde med finans, regnskab og matematik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;41497,58\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;331 Arbejde med finans, regnskab og matematik;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;25662,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;331 Arbejde med finans, regnskab og matematik;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;9582,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;331 Arbejde med finans, regnskab og matematik;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;16080,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;332 Agent- og mæglerarbejde inden for salg og indkøb;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;47029,88\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;332 Agent- og mæglerarbejde inden for salg og indkøb;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;50218,99\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;332 Agent- og mæglerarbejde inden for salg og indkøb;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;41637,86\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;332 Agent- og mæglerarbejde inden for salg og indkøb;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;35215,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;332 Agent- og mæglerarbejde inden for salg og indkøb;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;21889,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;332 Agent- og mæglerarbejde inden for salg og indkøb;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;13326,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;333 Arbejde inden for forretningsservice;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;42925,06\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;333 Arbejde inden for forretningsservice;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;46468,86\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;333 Arbejde inden for forretningsservice;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;39385,82\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;333 Arbejde inden for forretningsservice;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;11537,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;333 Arbejde inden for forretningsservice;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;5637,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;333 Arbejde inden for forretningsservice;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;5899,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;334 Administrativt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;39607,61\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;334 Administrativt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;45843,80\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;334 Administrativt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;38507,96\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;334 Administrativt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;27589,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;334 Administrativt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4148,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;334 Administrativt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;23441,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;335 Arbejde med forvaltning og udøvelse af lovgivning;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;43423,03\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;335 Arbejde med forvaltning og udøvelse af lovgivning;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;47836,20\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;335 Arbejde med forvaltning og udøvelse af lovgivning;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;40161,10\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;335 Arbejde med forvaltning og udøvelse af lovgivning;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;10109,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;335 Arbejde med forvaltning og udøvelse af lovgivning;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4215,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;335 Arbejde med forvaltning og udøvelse af lovgivning;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;5894,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;341 Arbejde inden for jura og samfundsforhold;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;34829,15\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;341 Arbejde inden for jura og samfundsforhold;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;35822,79\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;341 Arbejde inden for jura og samfundsforhold;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;34390,92\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;341 Arbejde inden for jura og samfundsforhold;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6546,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;341 Arbejde inden for jura og samfundsforhold;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1958,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;341 Arbejde inden for jura og samfundsforhold;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4589,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;342 Arbejde med sport og fitness;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;34826,02\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;342 Arbejde med sport og fitness;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;37888,89\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;342 Arbejde med sport og fitness;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28595,42\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;342 Arbejde med sport og fitness;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1479,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;342 Arbejde med sport og fitness;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;982,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;342 Arbejde med sport og fitness;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;497,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37305,30\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;39910,40\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;34878,23\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3949,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1819,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2131,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;45628,89\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;46163,06\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;43389,43\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;8674,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6950,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1724,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;42549,53\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;42475,36\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;42923,23\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3421,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2831,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;591,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;411 Almindeligt kontorarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;35884,73\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;411 Almindeligt kontorarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;39268,38\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;411 Almindeligt kontorarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;34869,04\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;411 Almindeligt kontorarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;63245,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;411 Almindeligt kontorarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;13959,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;411 Almindeligt kontorarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;49287,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;412 Almindeligt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37966,81\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;412 Almindeligt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;41629,91\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;412 Almindeligt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;37652,39\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;412 Almindeligt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5907,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;412 Almindeligt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;454,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;412 Almindeligt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;5453,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;413 Indtastningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36017,40\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;413 Indtastningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;37281,46\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;413 Indtastningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;34903,74\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;413 Indtastningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2150,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;413 Indtastningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;990,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;413 Indtastningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1159,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37480,91\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;38562,81\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;37012,25\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2637,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;761,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1876,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;422 Kundeinformationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32103,49\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;422 Kundeinformationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;33062,20\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;422 Kundeinformationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31660,39\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;422 Kundeinformationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;11897,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;422 Kundeinformationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3696,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;422 Kundeinformationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;8201,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;431 Almindeligt beregningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;39591,54\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;431 Almindeligt beregningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;42820,79\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;431 Almindeligt beregningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;38637,18\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;431 Almindeligt beregningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;17051,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;431 Almindeligt beregningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3857,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;431 Almindeligt beregningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;13194,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;432 Kontorarbejde inden for materialeregistrering og transport;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;33077,33\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;432 Kontorarbejde inden for materialeregistrering og transport;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;33512,94\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;432 Kontorarbejde inden for materialeregistrering og transport;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31830,19\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;432 Kontorarbejde inden for materialeregistrering og transport;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;17198,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;432 Kontorarbejde inden for materialeregistrering og transport;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;12634,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;432 Kontorarbejde inden for materialeregistrering og transport;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4563,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;441 Andet almindeligt kontor- og kundeservicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;33805,89\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;441 Andet almindeligt kontor- og kundeservicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;33652,21\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;441 Andet almindeligt kontor- og kundeservicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33962,27\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;441 Andet almindeligt kontor- og kundeservicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;19734,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;441 Andet almindeligt kontor- og kundeservicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;9873,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;441 Andet almindeligt kontor- og kundeservicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;9862,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;511 Service- og kontrolarbejde under transport og rejser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37405,88\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;511 Service- og kontrolarbejde under transport og rejser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;39104,44\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;511 Service- og kontrolarbejde under transport og rejser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;36121,51\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;511 Service- og kontrolarbejde under transport og rejser;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4640,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;511 Service- og kontrolarbejde under transport og rejser;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1956,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;511 Service- og kontrolarbejde under transport og rejser;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2684,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;512 Kokkearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;29655,83\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;512 Kokkearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;29870,88\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;512 Kokkearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29343,40\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;512 Kokkearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5783,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;512 Kokkearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3269,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;512 Kokkearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2514,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;513 Tjenere og bartendere;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;25332,33\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;513 Tjenere og bartendere;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;26397,23\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;513 Tjenere og bartendere;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;24512,54\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;513 Tjenere og bartendere;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5221,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;513 Tjenere og bartendere;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2264,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;513 Tjenere og bartendere;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2957,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;30697,19\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32274,16\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;30512,71\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;656,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;65,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;591,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;33313,87\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;33410,30\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32639,20\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;15316,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;13379,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1937,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;516 Andet servicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;30894,67\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;516 Andet servicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32210,52\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;516 Andet servicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28007,74\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;516 Andet servicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1912,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;516 Andet servicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1309,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;516 Andet servicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;604,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;521 Salgsarbejde på gade og marked;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;521 Salgsarbejde på gade og marked;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;521 Salgsarbejde på gade og marked;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;521 Salgsarbejde på gade og marked;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;43,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;521 Salgsarbejde på gade og marked;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;26,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;521 Salgsarbejde på gade og marked;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;17,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;522 Salgsarbejde i butik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;26938,20\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;522 Salgsarbejde i butik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;29187,05\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;522 Salgsarbejde i butik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;25065,93\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;522 Salgsarbejde i butik;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;32974,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;522 Salgsarbejde i butik;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;14733,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;522 Salgsarbejde i butik;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;18241,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;523 Kasseassistentarbejde og beslægtet kundebetjening;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;24397,25\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;523 Kasseassistentarbejde og beslægtet kundebetjening;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;24724,32\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;523 Kasseassistentarbejde og beslægtet kundebetjening;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;24256,41\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;523 Kasseassistentarbejde og beslægtet kundebetjening;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;9345,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;523 Kasseassistentarbejde og beslægtet kundebetjening;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2751,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;523 Kasseassistentarbejde og beslægtet kundebetjening;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;6594,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;524 Andet salgsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37123,64\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;524 Andet salgsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;40236,06\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;524 Andet salgsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32767,89\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;524 Andet salgsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;12894,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;524 Andet salgsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;7440,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;524 Andet salgsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;5453,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;531 Børneomsorgsarbejde og hjælpelærerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;27825,64\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;531 Børneomsorgsarbejde og hjælpelærerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;27435,97\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;531 Børneomsorgsarbejde og hjælpelærerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;27932,41\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;531 Børneomsorgsarbejde og hjælpelærerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;36218,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;531 Børneomsorgsarbejde og hjælpelærerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;7575,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;531 Børneomsorgsarbejde og hjælpelærerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;28643,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;532 Omsorgsarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32186,60\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;532 Omsorgsarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;31468,45\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;532 Omsorgsarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32308,70\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;532 Omsorgsarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;88770,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;532 Omsorgsarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;12321,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;532 Omsorgsarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;76449,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;541 Rednings- og overvågningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;35568,31\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;541 Rednings- og overvågningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;36023,22\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;541 Rednings- og overvågningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33386,14\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;541 Rednings- og overvågningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;20368,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;541 Rednings- og overvågningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;16746,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;541 Rednings- og overvågningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3622,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;611 Arbejde med plantevækst;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;30887,01\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;611 Arbejde med plantevækst;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;31519,94\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;611 Arbejde med plantevækst;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29145,73\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;611 Arbejde med plantevækst;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3268,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;611 Arbejde med plantevækst;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2343,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;611 Arbejde med plantevækst;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;925,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;612 Arbejde med dyreopdræt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32274,79\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;612 Arbejde med dyreopdræt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32521,61\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;612 Arbejde med dyreopdræt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32030,21\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;612 Arbejde med dyreopdræt;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;353,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;612 Arbejde med dyreopdræt;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;169,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;612 Arbejde med dyreopdræt;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;183,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;27488,64\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;27386,65\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;161,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;137,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;24,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;621 Skovbrugsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32691,09\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;621 Skovbrugsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32604,01\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;621 Skovbrugsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;621 Skovbrugsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;363,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;621 Skovbrugsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;327,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;621 Skovbrugsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;36,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;622 Arbejde med fiskeri og jagt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;622 Arbejde med fiskeri og jagt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;622 Arbejde med fiskeri og jagt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;622 Arbejde med fiskeri og jagt;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;44,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;622 Arbejde med fiskeri og jagt;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;35,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;622 Arbejde med fiskeri og jagt;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;8,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;711 Bygningsarbejde (ekskl. medhjælpere) - basis;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;34932,24\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;711 Bygningsarbejde (ekskl. medhjælpere) - basis;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;34965,53\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;711 Bygningsarbejde (ekskl. medhjælpere) - basis;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31238,16\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;711 Bygningsarbejde (ekskl. medhjælpere) - basis;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;20816,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;711 Bygningsarbejde (ekskl. medhjælpere) - basis;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;20628,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;711 Bygningsarbejde (ekskl. medhjælpere) - basis;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;187,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;712 Bygningsarbejde (ekskl. medhjælpere) - finish;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38357,35\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;712 Bygningsarbejde (ekskl. medhjælpere) - finish;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;38415,43\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;712 Bygningsarbejde (ekskl. medhjælpere) - finish;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31207,16\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;712 Bygningsarbejde (ekskl. medhjælpere) - finish;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;9342,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;712 Bygningsarbejde (ekskl. medhjælpere) - finish;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;9265,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;712 Bygningsarbejde (ekskl. medhjælpere) - finish;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;77,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;713 Malere og arbejde inden for rensning af bygninger;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32798,28\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;713 Malere og arbejde inden for rensning af bygninger;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;33281,32\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;713 Malere og arbejde inden for rensning af bygninger;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;30801,91\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;713 Malere og arbejde inden for rensning af bygninger;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5185,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;713 Malere og arbejde inden for rensning af bygninger;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4138,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;713 Malere og arbejde inden for rensning af bygninger;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1047,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;721 Pladearbejde, svejsning og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;35143,10\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;721 Pladearbejde, svejsning og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;35233,31\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;721 Pladearbejde, svejsning og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;30813,71\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;721 Pladearbejde, svejsning og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;9247,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;721 Pladearbejde, svejsning og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;9053,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;721 Pladearbejde, svejsning og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;194,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;722 Smede, værktøjsmagere og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36002,07\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;722 Smede, værktøjsmagere og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;36106,78\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;722 Smede, værktøjsmagere og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33125,28\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;722 Smede, værktøjsmagere og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;21392,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;722 Smede, værktøjsmagere og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;20614,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;722 Smede, værktøjsmagere og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;778,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;723 Mekanikerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36213,69\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;723 Mekanikerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;36246,41\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;723 Mekanikerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33204,29\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;723 Mekanikerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;14872,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;723 Mekanikerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;14702,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;723 Mekanikerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;171,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;731 Præcisionshåndværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32922,71\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;731 Præcisionshåndværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;35081,99\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;731 Præcisionshåndværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29848,52\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;731 Præcisionshåndværk;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1085,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;731 Præcisionshåndværk;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;631,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;731 Præcisionshåndværk;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;454,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;732 Grafisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37706,77\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;732 Grafisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;39350,57\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;732 Grafisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33244,22\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;732 Grafisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2919,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;732 Grafisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2144,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;732 Grafisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;775,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;741 Installation- og reparation af elektrisk udstyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37885,56\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;741 Installation- og reparation af elektrisk udstyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;37928,23\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;741 Installation- og reparation af elektrisk udstyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;34312,15\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;741 Installation- og reparation af elektrisk udstyr;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;16559,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;741 Installation- og reparation af elektrisk udstyr;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;16356,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;741 Installation- og reparation af elektrisk udstyr;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;203,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;742 Installation og reparation af elektronik og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38726,15\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;742 Installation og reparation af elektronik og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;39102,03\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;742 Installation og reparation af elektronik og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32146,16\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;742 Installation og reparation af elektronik og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1815,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;742 Installation og reparation af elektronik og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1712,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;742 Installation og reparation af elektronik og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;103,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;751 Arbejde inden for nærings- og nydelsesmiddelområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32094,45\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;751 Arbejde inden for nærings- og nydelsesmiddelområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;33478,38\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;751 Arbejde inden for nærings- og nydelsesmiddelområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28309,69\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;751 Arbejde inden for nærings- og nydelsesmiddelområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6349,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;751 Arbejde inden for nærings- og nydelsesmiddelområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4635,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;751 Arbejde inden for nærings- og nydelsesmiddelområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1714,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;752 Håndværkspræget arbejde i træindustri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;30107,08\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;752 Håndværkspræget arbejde i træindustri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;30250,68\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;752 Håndværkspræget arbejde i træindustri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28280,85\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;752 Håndværkspræget arbejde i træindustri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1725,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;752 Håndværkspræget arbejde i træindustri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1599,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;752 Håndværkspræget arbejde i træindustri;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;127,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;753 Tekstil- og beklædningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31908,09\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;753 Tekstil- og beklædningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32839,80\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;753 Tekstil- og beklædningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31256,12\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;753 Tekstil- og beklædningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;508,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;753 Tekstil- og beklædningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;210,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;753 Tekstil- og beklædningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;299,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;754 Andet håndværkspræget arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37347,02\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;754 Andet håndværkspræget arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;37697,99\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;754 Andet håndværkspræget arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;36521,81\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;754 Andet håndværkspræget arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2058,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;754 Andet håndværkspræget arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1420,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;754 Andet håndværkspræget arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;637,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;35472,18\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;35612,42\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31142,60\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1568,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1517,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;52,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31838,52\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32107,48\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;30561,69\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5632,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4634,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;998,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;42058,13\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;42426,94\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;40607,35\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3211,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2530,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;681,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;33429,77\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;34084,16\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31688,98\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6441,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4670,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1771,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;33141,32\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;36503,26\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29034,06\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1258,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;683,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;575,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;33932,84\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;34932,65\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31092,46\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;13875,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;10228,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3647,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;817 Operatørarbejde ved træforarbejdning og papirfremstilling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;30787,24\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;817 Operatørarbejde ved træforarbejdning og papirfremstilling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;31031,34\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;817 Operatørarbejde ved træforarbejdning og papirfremstilling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29575,40\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;817 Operatørarbejde ved træforarbejdning og papirfremstilling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3366,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;817 Operatørarbejde ved træforarbejdning og papirfremstilling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2792,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;817 Operatørarbejde ved træforarbejdning og papirfremstilling;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;574,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;818 Andet operatørarbejde ved stationære anlæg og maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37437,29\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;818 Andet operatørarbejde ved stationære anlæg og maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;38350,97\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;818 Andet operatørarbejde ved stationære anlæg og maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33481,53\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;818 Andet operatørarbejde ved stationære anlæg og maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;9342,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;818 Andet operatørarbejde ved stationære anlæg og maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;7520,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;818 Andet operatørarbejde ved stationære anlæg og maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1822,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;821 Monteringsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32616,06\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;821 Monteringsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;34478,02\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;821 Monteringsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29070,75\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;821 Monteringsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;11397,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;821 Monteringsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;7381,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;821 Monteringsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4016,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;831 Lokomotivførerarbejde og tograngering;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;43184,53\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;831 Lokomotivførerarbejde og tograngering;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;43282,58\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;831 Lokomotivførerarbejde og tograngering;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;40921,42\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;831 Lokomotivførerarbejde og tograngering;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2715,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;831 Lokomotivførerarbejde og tograngering;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2600,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;831 Lokomotivførerarbejde og tograngering;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;115,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;832 Chauffører af biler, varevogne og motorcykler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;27747,06\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;832 Chauffører af biler, varevogne og motorcykler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;27883,79\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;832 Chauffører af biler, varevogne og motorcykler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;25618,80\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;832 Chauffører af biler, varevogne og motorcykler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3325,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;832 Chauffører af biler, varevogne og motorcykler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3132,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;832 Chauffører af biler, varevogne og motorcykler;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;194,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;833 Chauffører af busser og lastbiler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;30658,75\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;833 Chauffører af busser og lastbiler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;30744,52\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;833 Chauffører af busser og lastbiler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29047,67\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;833 Chauffører af busser og lastbiler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;23212,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;833 Chauffører af busser og lastbiler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;21953,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;833 Chauffører af busser og lastbiler;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1259,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;834 Førere af mobile maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32656,61\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;834 Førere af mobile maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32808,35\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;834 Førere af mobile maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29657,97\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;834 Førere af mobile maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6641,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;834 Førere af mobile maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6314,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;834 Førere af mobile maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;327,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;835 Dæksarbejde og medhjælp om bord på skibe;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38763,88\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;835 Dæksarbejde og medhjælp om bord på skibe;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;39012,96\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;835 Dæksarbejde og medhjælp om bord på skibe;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;835 Dæksarbejde og medhjælp om bord på skibe;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;330,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;835 Dæksarbejde og medhjælp om bord på skibe;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;316,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;835 Dæksarbejde og medhjælp om bord på skibe;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;14,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;911 Rengøringsarbejde i private hjem, hoteller og kontorer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;27174,14\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;911 Rengøringsarbejde i private hjem, hoteller og kontorer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;28491,19\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;911 Rengøringsarbejde i private hjem, hoteller og kontorer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;26507,52\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;911 Rengøringsarbejde i private hjem, hoteller og kontorer;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;34190,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;911 Rengøringsarbejde i private hjem, hoteller og kontorer;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;10890,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;911 Rengøringsarbejde i private hjem, hoteller og kontorer;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;23299,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;27697,55\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;28545,27\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;26365,35\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3683,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2200,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1483,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;29180,59\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;29188,77\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29148,10\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;676,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;526,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;149,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;33484,07\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;33589,93\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;30816,31\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;26370,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;25271,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1099,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;932 Manuelt produktionsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;29393,98\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;932 Manuelt produktionsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;30143,10\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;932 Manuelt produktionsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;27799,26\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;932 Manuelt produktionsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;12627,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;932 Manuelt produktionsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;8542,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;932 Manuelt produktionsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4085,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;933 Transport- og lagerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;30298,63\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;933 Transport- og lagerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;30827,81\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;933 Transport- og lagerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;27215,90\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;933 Transport- og lagerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;22691,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;933 Transport- og lagerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;19332,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;933 Transport- og lagerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3359,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;941 Manuelt arbejde med tilberedning af mad;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;26201,63\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;941 Manuelt arbejde med tilberedning af mad;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;24565,70\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;941 Manuelt arbejde med tilberedning af mad;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;26786,33\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;941 Manuelt arbejde med tilberedning af mad;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;12357,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;941 Manuelt arbejde med tilberedning af mad;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2702,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;941 Manuelt arbejde med tilberedning af mad;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;9655,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;951 Servicearbejde på gaden;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;951 Servicearbejde på gaden;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;951 Servicearbejde på gaden;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;951 Servicearbejde på gaden;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;33,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;951 Servicearbejde på gaden;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;30,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;951 Servicearbejde på gaden;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;952 Gadesalg (undtagen madvarer);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;952 Gadesalg (undtagen madvarer);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;952 Gadesalg (undtagen madvarer);STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;952 Gadesalg (undtagen madvarer);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;952 Gadesalg (undtagen madvarer);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;952 Gadesalg (undtagen madvarer);Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;961 Renovationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32919,60\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;961 Renovationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;33152,98\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;961 Renovationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29171,41\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;961 Renovationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2600,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;961 Renovationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2446,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;961 Renovationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;154,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;962 Andet manuelt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31762,60\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;962 Andet manuelt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;33063,97\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;962 Andet manuelt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28290,05\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;962 Andet manuelt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6003,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;962 Andet manuelt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4397,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2016;962 Andet manuelt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1606,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;011 Militært arbejde på officersniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;52868,10\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;011 Militært arbejde på officersniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;53358,05\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;011 Militært arbejde på officersniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;45809,63\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;011 Militært arbejde på officersniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3547,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;011 Militært arbejde på officersniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3310,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;011 Militært arbejde på officersniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;237,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37791,64\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;38001,17\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;34272,76\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4052,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3823,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;230,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;031 Militært arbejde, øvrige rangordener;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31941,04\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;031 Militært arbejde, øvrige rangordener;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32114,64\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;031 Militært arbejde, øvrige rangordener;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29931,39\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;031 Militært arbejde, øvrige rangordener;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;7316,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;031 Militært arbejde, øvrige rangordener;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6719,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;031 Militært arbejde, øvrige rangordener;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;597,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;77182,14\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;81776,46\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;69856,98\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1601,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;982,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;620,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;112 Øverste virksomhedsledelse;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;101965,44\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;112 Øverste virksomhedsledelse;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;104074,47\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;112 Øverste virksomhedsledelse;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;88442,03\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;112 Øverste virksomhedsledelse;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;14030,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;112 Øverste virksomhedsledelse;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;12134,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;112 Øverste virksomhedsledelse;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1896,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;121 Ledelse inden for forretningsservice og administrative funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;71491,52\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;121 Ledelse inden for forretningsservice og administrative funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;74892,71\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;121 Ledelse inden for forretningsservice og administrative funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;66023,84\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;121 Ledelse inden for forretningsservice og administrative funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;18722,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;121 Ledelse inden for forretningsservice og administrative funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;11461,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;121 Ledelse inden for forretningsservice og administrative funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7262,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;122 Ledelse af forretnings- og udviklingsorienterede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;74490,51\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;122 Ledelse af forretnings- og udviklingsorienterede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;77342,66\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;122 Ledelse af forretnings- og udviklingsorienterede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;65528,87\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;122 Ledelse af forretnings- og udviklingsorienterede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;11507,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;122 Ledelse af forretnings- og udviklingsorienterede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;8691,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;122 Ledelse af forretnings- og udviklingsorienterede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2817,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;58841,31\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;58623,26\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;143,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;132,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;11,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;63100,52\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;63099,99\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;63104,55\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;10250,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;9067,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1183,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;76868,28\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;77494,92\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;73333,63\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2498,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2106,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;391,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;134 Ledelse af hovedaktiviteten inden for servicefag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;56727,39\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;134 Ledelse af hovedaktiviteten inden for servicefag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;63243,82\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;134 Ledelse af hovedaktiviteten inden for servicefag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;51733,38\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;134 Ledelse af hovedaktiviteten inden for servicefag;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;13361,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;134 Ledelse af hovedaktiviteten inden for servicefag;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;5611,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;134 Ledelse af hovedaktiviteten inden for servicefag;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7749,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;48605,56\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;49787,36\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;46749,55\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;543,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;326,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;217,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;42822,21\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;45996,69\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;36743,77\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5850,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3789,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2061,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;143 Ledelse af hovedaktiviteten inden for andre serviceområder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;61246,87\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;143 Ledelse af hovedaktiviteten inden for andre serviceområder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;62620,95\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;143 Ledelse af hovedaktiviteten inden for andre serviceområder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;57618,56\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;143 Ledelse af hovedaktiviteten inden for andre serviceområder;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1210,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;143 Ledelse af hovedaktiviteten inden for andre serviceområder;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;880,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;143 Ledelse af hovedaktiviteten inden for andre serviceområder;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;330,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;211 Arbejde inden for fysik og geologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;54843,54\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;211 Arbejde inden for fysik og geologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;56435,42\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;211 Arbejde inden for fysik og geologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;52456,31\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;211 Arbejde inden for fysik og geologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2346,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;211 Arbejde inden for fysik og geologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1386,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;211 Arbejde inden for fysik og geologi;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;960,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;62722,72\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;65292,56\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;58146,66\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1700,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1074,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;625,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;213 Arbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;51422,57\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;213 Arbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;52759,68\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;213 Arbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;49771,68\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;213 Arbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4349,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;213 Arbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2348,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;213 Arbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2002,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;214 Ingeniørarbejde (undtagen inden for elektroteknologi);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;57650,81\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;214 Ingeniørarbejde (undtagen inden for elektroteknologi);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;58564,09\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;214 Ingeniørarbejde (undtagen inden for elektroteknologi);STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;54489,77\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;214 Ingeniørarbejde (undtagen inden for elektroteknologi);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;39272,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;214 Ingeniørarbejde (undtagen inden for elektroteknologi);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;30049,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;214 Ingeniørarbejde (undtagen inden for elektroteknologi);Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;9222,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;215 Ingeniørarbejde inden for elektroteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;59149,58\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;215 Ingeniørarbejde inden for elektroteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;59460,83\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;215 Ingeniørarbejde inden for elektroteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;54937,44\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;215 Ingeniørarbejde inden for elektroteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6009,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;215 Ingeniørarbejde inden for elektroteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;5578,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;215 Ingeniørarbejde inden for elektroteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;431,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;216 Arbejde med arkitektur, infrastruktur og design;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;46179,66\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;216 Arbejde med arkitektur, infrastruktur og design;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;49187,49\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;216 Arbejde med arkitektur, infrastruktur og design;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;43057,44\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;216 Arbejde med arkitektur, infrastruktur og design;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;8018,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;216 Arbejde med arkitektur, infrastruktur og design;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3988,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;216 Arbejde med arkitektur, infrastruktur og design;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4030,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;221 Lægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;70919,04\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;221 Lægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;75927,81\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;221 Lægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;66115,17\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;221 Lægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;17237,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;221 Lægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;8143,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;221 Lægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;9095,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;222 Sygeplejerske- og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;40072,30\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;222 Sygeplejerske- og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;42662,29\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;222 Sygeplejerske- og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;39950,50\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;222 Sygeplejerske- og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;53000,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;222 Sygeplejerske- og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2265,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;222 Sygeplejerske- og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;50736,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;223 Arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;223 Arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;223 Arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;223 Arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;10,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;223 Arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;223 Arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;224 Paramedicinsk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;43423,48\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;224 Paramedicinsk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;43418,18\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;224 Paramedicinsk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;224 Paramedicinsk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;382,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;224 Paramedicinsk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;358,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;224 Paramedicinsk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;23,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;225 Dyrlægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;51099,27\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;225 Dyrlægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;54458,74\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;225 Dyrlægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;49293,87\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;225 Dyrlægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;775,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;225 Dyrlægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;256,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;225 Dyrlægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;520,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;226 Andet sundhedsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;44535,39\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;226 Andet sundhedsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;51855,40\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;226 Andet sundhedsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;42751,06\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;226 Andet sundhedsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;23071,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;226 Andet sundhedsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4241,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;226 Andet sundhedsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;18829,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;231 Undervisning og forskning ved universiteter og højere læreanstalter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;46376,47\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;231 Undervisning og forskning ved universiteter og højere læreanstalter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;47909,23\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;231 Undervisning og forskning ved universiteter og højere læreanstalter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;44475,62\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;231 Undervisning og forskning ved universiteter og højere læreanstalter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;23973,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;231 Undervisning og forskning ved universiteter og højere læreanstalter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;13070,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;231 Undervisning og forskning ved universiteter og højere læreanstalter;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;10903,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;232 Undervisning ved erhvervsuddannelser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;41948,54\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;232 Undervisning ved erhvervsuddannelser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;42002,25\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;232 Undervisning ved erhvervsuddannelser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;41878,05\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;232 Undervisning ved erhvervsuddannelser;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;10015,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;232 Undervisning ved erhvervsuddannelser;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;5640,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;232 Undervisning ved erhvervsuddannelser;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4375,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;233 Almen undervisning efter grundskoleniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;47765,78\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;233 Almen undervisning efter grundskoleniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;48378,62\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;233 Almen undervisning efter grundskoleniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;47209,91\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;233 Almen undervisning efter grundskoleniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;14516,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;233 Almen undervisning efter grundskoleniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6818,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;233 Almen undervisning efter grundskoleniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7698,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;234 Undervisning og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38963,91\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;234 Undervisning og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;40019,93\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;234 Undervisning og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;38556,98\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;234 Undervisning og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;112579,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;234 Undervisning og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;30090,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;234 Undervisning og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;82489,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;235 Andet undervisnings- og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;39054,39\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;235 Andet undervisnings- og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;39835,61\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;235 Andet undervisnings- og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;38676,38\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;235 Andet undervisnings- og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;38638,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;235 Andet undervisnings- og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;12165,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;235 Andet undervisnings- og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;26473,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;241 Arbejde inden for finans og økonomi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;57115,26\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;241 Arbejde inden for finans og økonomi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;61014,42\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;241 Arbejde inden for finans og økonomi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;52113,38\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;241 Arbejde inden for finans og økonomi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;30771,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;241 Arbejde inden for finans og økonomi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;17007,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;241 Arbejde inden for finans og økonomi;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;13764,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;242 Arbejde inden for virksomhedsadministration - privat og offentlig;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;59932,08\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;242 Arbejde inden for virksomhedsadministration - privat og offentlig;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;63203,70\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;242 Arbejde inden for virksomhedsadministration - privat og offentlig;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;56923,31\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;242 Arbejde inden for virksomhedsadministration - privat og offentlig;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;17240,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;242 Arbejde inden for virksomhedsadministration - privat og offentlig;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;8104,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;242 Arbejde inden for virksomhedsadministration - privat og offentlig;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;9136,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;243 Arbejde inden for salg, marketing og PR;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;54361,13\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;243 Arbejde inden for salg, marketing og PR;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;58757,88\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;243 Arbejde inden for salg, marketing og PR;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;49084,71\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;243 Arbejde inden for salg, marketing og PR;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;15635,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;243 Arbejde inden for salg, marketing og PR;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;8438,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;243 Arbejde inden for salg, marketing og PR;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7196,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;251 Udvikling og analyse af software og applikatoner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;56784,45\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;251 Udvikling og analyse af software og applikatoner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;57737,97\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;251 Udvikling og analyse af software og applikatoner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;53466,45\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;251 Udvikling og analyse af software og applikatoner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;38786,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;251 Udvikling og analyse af software og applikatoner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;29768,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;251 Udvikling og analyse af software og applikatoner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;9018,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;252 Arbejde med databaser og netværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;52542,16\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;252 Arbejde med databaser og netværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;53239,97\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;252 Arbejde med databaser og netværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;49235,82\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;252 Arbejde med databaser og netværk;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;7171,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;252 Arbejde med databaser og netværk;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;5879,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;252 Arbejde med databaser og netværk;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1292,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;261 Juridisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;55052,70\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;261 Juridisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;58199,28\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;261 Juridisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;52735,27\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;261 Juridisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;15776,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;261 Juridisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6468,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;261 Juridisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;9307,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;43579,69\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;44652,25\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;43130,79\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3131,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;900,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2230,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;263 Arbejde inden for samfundsvidenskab og religion;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;43510,83\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;263 Arbejde inden for samfundsvidenskab og religion;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;47638,15\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;263 Arbejde inden for samfundsvidenskab og religion;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;41943,03\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;263 Arbejde inden for samfundsvidenskab og religion;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;28256,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;263 Arbejde inden for samfundsvidenskab og religion;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;7385,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;263 Arbejde inden for samfundsvidenskab og religion;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;20872,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;47842,72\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;49443,18\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;46040,06\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;8234,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4295,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3940,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;265 Arbejde med kunst og kreative fag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;43401,46\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;265 Arbejde med kunst og kreative fag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;45236,76\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;265 Arbejde med kunst og kreative fag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;41105,70\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;265 Arbejde med kunst og kreative fag;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2475,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;265 Arbejde med kunst og kreative fag;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1363,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;265 Arbejde med kunst og kreative fag;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1112,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;44952,57\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;47081,85\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;39830,51\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;42280,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;29419,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;12862,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;313 Teknisk kontrolarbejde inden for processtyring;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;46791,34\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;313 Teknisk kontrolarbejde inden for processtyring;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;48609,12\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;313 Teknisk kontrolarbejde inden for processtyring;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;42932,20\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;313 Teknisk kontrolarbejde inden for processtyring;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2720,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;313 Teknisk kontrolarbejde inden for processtyring;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1817,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;313 Teknisk kontrolarbejde inden for processtyring;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;903,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;314 Teknikerarbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;43026,27\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;314 Teknikerarbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;43959,26\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;314 Teknikerarbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;40668,79\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;314 Teknikerarbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;726,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;314 Teknikerarbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;515,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;314 Teknikerarbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;211,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;315 Teknikerarbejde inden for skibs- og luftfart;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;71250,24\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;315 Teknikerarbejde inden for skibs- og luftfart;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;72392,37\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;315 Teknikerarbejde inden for skibs- og luftfart;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;59762,20\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;315 Teknikerarbejde inden for skibs- og luftfart;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3301,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;315 Teknikerarbejde inden for skibs- og luftfart;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2996,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;315 Teknikerarbejde inden for skibs- og luftfart;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;305,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37927,75\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;39689,23\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;37643,21\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;11307,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1551,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;9756,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37499,53\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;37988,09\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;37188,46\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;366,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;140,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;226,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;7,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;0,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;28399,50\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28289,85\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;188,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;12,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;176,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;39626,18\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;44334,73\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;36592,25\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6230,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2374,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3857,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;331 Arbejde med finans, regnskab og matematik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;46189,63\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;331 Arbejde med finans, regnskab og matematik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;52669,59\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;331 Arbejde med finans, regnskab og matematik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;42169,79\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;331 Arbejde med finans, regnskab og matematik;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;26024,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;331 Arbejde med finans, regnskab og matematik;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;9838,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;331 Arbejde med finans, regnskab og matematik;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;16186,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;332 Agent- og mæglerarbejde inden for salg og indkøb;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;48404,55\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;332 Agent- og mæglerarbejde inden for salg og indkøb;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;51664,79\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;332 Agent- og mæglerarbejde inden for salg og indkøb;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;42960,10\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;332 Agent- og mæglerarbejde inden for salg og indkøb;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;35150,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;332 Agent- og mæglerarbejde inden for salg og indkøb;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;21744,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;332 Agent- og mæglerarbejde inden for salg og indkøb;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;13405,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;333 Arbejde inden for forretningsservice;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;44547,03\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;333 Arbejde inden for forretningsservice;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;48266,21\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;333 Arbejde inden for forretningsservice;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;40800,71\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;333 Arbejde inden for forretningsservice;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;12637,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;333 Arbejde inden for forretningsservice;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6188,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;333 Arbejde inden for forretningsservice;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;6448,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;334 Administrativt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;40288,21\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;334 Administrativt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;46590,94\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;334 Administrativt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;39209,93\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;334 Administrativt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;26470,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;334 Administrativt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3874,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;334 Administrativt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;22597,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;335 Arbejde med forvaltning og udøvelse af lovgivning;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;44785,95\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;335 Arbejde med forvaltning og udøvelse af lovgivning;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;49243,19\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;335 Arbejde med forvaltning og udøvelse af lovgivning;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;41505,69\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;335 Arbejde med forvaltning og udøvelse af lovgivning;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;10135,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;335 Arbejde med forvaltning og udøvelse af lovgivning;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4224,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;335 Arbejde med forvaltning og udøvelse af lovgivning;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;5912,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;341 Arbejde inden for jura og samfundsforhold;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;35498,90\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;341 Arbejde inden for jura og samfundsforhold;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;37073,51\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;341 Arbejde inden for jura og samfundsforhold;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;34818,23\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;341 Arbejde inden for jura og samfundsforhold;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6844,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;341 Arbejde inden for jura og samfundsforhold;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2028,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;341 Arbejde inden for jura og samfundsforhold;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4817,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;342 Arbejde med sport og fitness;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36259,94\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;342 Arbejde med sport og fitness;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;39658,67\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;342 Arbejde med sport og fitness;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28909,66\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;342 Arbejde med sport og fitness;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1475,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;342 Arbejde med sport og fitness;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;981,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;342 Arbejde med sport og fitness;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;493,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37960,83\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;40265,05\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;35813,19\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4094,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1892,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2202,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;46675,20\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;47199,33\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;44498,37\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;9112,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;7294,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1818,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;42849,35\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;42870,40\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;42745,89\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3330,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2742,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;588,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;411 Almindeligt kontorarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36930,54\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;411 Almindeligt kontorarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;40630,61\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;411 Almindeligt kontorarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;35780,66\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;411 Almindeligt kontorarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;64952,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;411 Almindeligt kontorarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;14774,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;411 Almindeligt kontorarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;50178,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;412 Almindeligt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38625,39\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;412 Almindeligt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;41122,86\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;412 Almindeligt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;38417,56\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;412 Almindeligt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5692,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;412 Almindeligt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;423,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;412 Almindeligt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;5269,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;413 Indtastningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37278,58\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;413 Indtastningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;38834,15\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;413 Indtastningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;36009,94\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;413 Indtastningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1929,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;413 Indtastningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;850,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;413 Indtastningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1079,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38521,73\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;39851,51\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;37933,70\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2506,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;742,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1764,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;422 Kundeinformationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32685,57\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;422 Kundeinformationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;33417,59\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;422 Kundeinformationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32335,27\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;422 Kundeinformationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;12158,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;422 Kundeinformationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3896,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;422 Kundeinformationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;8263,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;431 Almindeligt beregningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;40583,29\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;431 Almindeligt beregningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;43994,82\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;431 Almindeligt beregningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;39574,92\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;431 Almindeligt beregningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;17014,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;431 Almindeligt beregningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3843,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;431 Almindeligt beregningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;13171,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;432 Kontorarbejde inden for materialeregistrering og transport;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;33825,10\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;432 Kontorarbejde inden for materialeregistrering og transport;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;34219,89\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;432 Kontorarbejde inden for materialeregistrering og transport;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32681,67\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;432 Kontorarbejde inden for materialeregistrering og transport;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;18003,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;432 Kontorarbejde inden for materialeregistrering og transport;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;13278,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;432 Kontorarbejde inden for materialeregistrering og transport;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4725,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;441 Andet almindeligt kontor- og kundeservicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;34765,30\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;441 Andet almindeligt kontor- og kundeservicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;34563,55\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;441 Andet almindeligt kontor- og kundeservicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;34961,09\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;441 Andet almindeligt kontor- og kundeservicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;18944,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;441 Andet almindeligt kontor- og kundeservicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;9191,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;441 Andet almindeligt kontor- og kundeservicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;9753,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;511 Service- og kontrolarbejde under transport og rejser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38122,84\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;511 Service- og kontrolarbejde under transport og rejser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;39769,14\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;511 Service- og kontrolarbejde under transport og rejser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;36818,74\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;511 Service- og kontrolarbejde under transport og rejser;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4528,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;511 Service- og kontrolarbejde under transport og rejser;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1956,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;511 Service- og kontrolarbejde under transport og rejser;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2572,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;512 Kokkearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;29948,29\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;512 Kokkearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;30231,58\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;512 Kokkearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29530,81\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;512 Kokkearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6236,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;512 Kokkearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3563,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;512 Kokkearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2672,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;513 Tjenere og bartendere;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;25688,31\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;513 Tjenere og bartendere;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;26715,33\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;513 Tjenere og bartendere;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;24870,03\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;513 Tjenere og bartendere;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5482,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;513 Tjenere og bartendere;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2426,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;513 Tjenere og bartendere;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3056,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31372,10\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;31822,79\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31317,33\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;673,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;69,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;604,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;34192,19\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;34297,68\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33437,15\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;15686,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;13715,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1970,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;516 Andet servicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;29685,23\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;516 Andet servicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;31015,47\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;516 Andet servicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;26783,78\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;516 Andet servicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2471,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;516 Andet servicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1699,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;516 Andet servicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;772,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;521 Salgsarbejde på gade og marked;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;521 Salgsarbejde på gade og marked;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;521 Salgsarbejde på gade og marked;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;521 Salgsarbejde på gade og marked;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;28,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;521 Salgsarbejde på gade og marked;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;18,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;521 Salgsarbejde på gade og marked;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;10,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;522 Salgsarbejde i butik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;27448,97\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;522 Salgsarbejde i butik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;29597,61\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;522 Salgsarbejde i butik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;25640,25\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;522 Salgsarbejde i butik;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;35046,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;522 Salgsarbejde i butik;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;15812,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;522 Salgsarbejde i butik;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;19234,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;523 Kasseassistentarbejde og beslægtet kundebetjening;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;24675,65\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;523 Kasseassistentarbejde og beslægtet kundebetjening;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;24903,89\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;523 Kasseassistentarbejde og beslægtet kundebetjening;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;24572,35\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;523 Kasseassistentarbejde og beslægtet kundebetjening;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;10096,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;523 Kasseassistentarbejde og beslægtet kundebetjening;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3094,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;523 Kasseassistentarbejde og beslægtet kundebetjening;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7002,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;524 Andet salgsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36627,12\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;524 Andet salgsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;39454,08\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;524 Andet salgsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32675,20\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;524 Andet salgsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;14540,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;524 Andet salgsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;8396,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;524 Andet salgsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;6144,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;531 Børneomsorgsarbejde og hjælpelærerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;28280,28\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;531 Børneomsorgsarbejde og hjælpelærerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;27973,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;531 Børneomsorgsarbejde og hjælpelærerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28362,46\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;531 Børneomsorgsarbejde og hjælpelærerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;36147,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;531 Børneomsorgsarbejde og hjælpelærerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;7428,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;531 Børneomsorgsarbejde og hjælpelærerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;28718,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;532 Omsorgsarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32821,48\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;532 Omsorgsarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32348,31\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;532 Omsorgsarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32900,33\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;532 Omsorgsarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;90122,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;532 Omsorgsarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;12400,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;532 Omsorgsarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;77722,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;541 Rednings- og overvågningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36203,70\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;541 Rednings- og overvågningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;36654,26\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;541 Rednings- og overvågningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;34123,58\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;541 Rednings- og overvågningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;20515,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;541 Rednings- og overvågningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;16810,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;541 Rednings- og overvågningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3705,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;611 Arbejde med plantevækst;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31676,36\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;611 Arbejde med plantevækst;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32332,18\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;611 Arbejde med plantevækst;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29874,36\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;611 Arbejde med plantevækst;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3406,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;611 Arbejde med plantevækst;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2442,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;611 Arbejde med plantevækst;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;964,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;612 Arbejde med dyreopdræt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32022,54\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;612 Arbejde med dyreopdræt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32493,15\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;612 Arbejde med dyreopdræt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31616,45\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;612 Arbejde med dyreopdræt;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;365,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;612 Arbejde med dyreopdræt;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;165,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;612 Arbejde med dyreopdræt;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;201,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;27950,31\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;28033,48\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;159,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;133,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;26,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;621 Skovbrugsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;33016,50\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;621 Skovbrugsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32838,19\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;621 Skovbrugsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;621 Skovbrugsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;348,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;621 Skovbrugsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;314,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;621 Skovbrugsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;34,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;622 Arbejde med fiskeri og jagt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;622 Arbejde med fiskeri og jagt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;622 Arbejde med fiskeri og jagt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;622 Arbejde med fiskeri og jagt;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;33,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;622 Arbejde med fiskeri og jagt;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;29,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;622 Arbejde med fiskeri og jagt;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;711 Bygningsarbejde (ekskl. medhjælpere) - basis;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;35919,19\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;711 Bygningsarbejde (ekskl. medhjælpere) - basis;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;35954,76\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;711 Bygningsarbejde (ekskl. medhjælpere) - basis;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32424,56\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;711 Bygningsarbejde (ekskl. medhjælpere) - basis;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;21954,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;711 Bygningsarbejde (ekskl. medhjælpere) - basis;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;21732,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;711 Bygningsarbejde (ekskl. medhjælpere) - basis;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;221,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;712 Bygningsarbejde (ekskl. medhjælpere) - finish;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;39358,23\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;712 Bygningsarbejde (ekskl. medhjælpere) - finish;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;39409,16\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;712 Bygningsarbejde (ekskl. medhjælpere) - finish;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32307,97\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;712 Bygningsarbejde (ekskl. medhjælpere) - finish;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;9469,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;712 Bygningsarbejde (ekskl. medhjælpere) - finish;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;9398,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;712 Bygningsarbejde (ekskl. medhjælpere) - finish;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;71,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;713 Malere og arbejde inden for rensning af bygninger;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;33619,41\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;713 Malere og arbejde inden for rensning af bygninger;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;34116,57\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;713 Malere og arbejde inden for rensning af bygninger;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31535,50\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;713 Malere og arbejde inden for rensning af bygninger;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5467,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;713 Malere og arbejde inden for rensning af bygninger;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4382,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;713 Malere og arbejde inden for rensning af bygninger;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1085,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;721 Pladearbejde, svejsning og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;35713,75\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;721 Pladearbejde, svejsning og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;35786,85\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;721 Pladearbejde, svejsning og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31981,33\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;721 Pladearbejde, svejsning og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;9676,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;721 Pladearbejde, svejsning og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;9485,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;721 Pladearbejde, svejsning og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;191,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;722 Smede, værktøjsmagere og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36674,84\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;722 Smede, værktøjsmagere og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;36796,35\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;722 Smede, værktøjsmagere og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33476,49\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;722 Smede, værktøjsmagere og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;21658,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;722 Smede, værktøjsmagere og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;20832,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;722 Smede, værktøjsmagere og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;827,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;723 Mekanikerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37117,46\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;723 Mekanikerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;37142,01\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;723 Mekanikerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;34738,12\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;723 Mekanikerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;14891,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;723 Mekanikerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;14730,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;723 Mekanikerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;161,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;731 Præcisionshåndværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;33607,32\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;731 Præcisionshåndværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;36032,08\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;731 Præcisionshåndværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;30145,81\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;731 Præcisionshåndværk;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1072,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;731 Præcisionshåndværk;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;623,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;731 Præcisionshåndværk;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;449,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;732 Grafisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38314,73\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;732 Grafisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;40161,98\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;732 Grafisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33683,78\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;732 Grafisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2667,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;732 Grafisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1921,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;732 Grafisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;746,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;741 Installation- og reparation af elektrisk udstyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38974,28\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;741 Installation- og reparation af elektrisk udstyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;39021,56\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;741 Installation- og reparation af elektrisk udstyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;35369,57\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;741 Installation- og reparation af elektrisk udstyr;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;16981,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;741 Installation- og reparation af elektrisk udstyr;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;16756,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;741 Installation- og reparation af elektrisk udstyr;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;224,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;742 Installation og reparation af elektronik og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;40125,32\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;742 Installation og reparation af elektronik og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;40488,04\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;742 Installation og reparation af elektronik og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33355,78\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;742 Installation og reparation af elektronik og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1705,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;742 Installation og reparation af elektronik og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1614,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;742 Installation og reparation af elektronik og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;91,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;751 Arbejde inden for nærings- og nydelsesmiddelområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;33109,08\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;751 Arbejde inden for nærings- og nydelsesmiddelområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;34370,76\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;751 Arbejde inden for nærings- og nydelsesmiddelområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;30142,18\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;751 Arbejde inden for nærings- og nydelsesmiddelområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;7138,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;751 Arbejde inden for nærings- og nydelsesmiddelområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4992,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;751 Arbejde inden for nærings- og nydelsesmiddelområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2146,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;752 Håndværkspræget arbejde i træindustri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;30465,41\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;752 Håndværkspræget arbejde i træindustri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;30580,28\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;752 Håndværkspræget arbejde i træindustri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29104,36\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;752 Håndværkspræget arbejde i træindustri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1770,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;752 Håndværkspræget arbejde i træindustri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1632,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;752 Håndværkspræget arbejde i træindustri;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;137,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;753 Tekstil- og beklædningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31954,60\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;753 Tekstil- og beklædningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32280,09\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;753 Tekstil- og beklædningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31724,30\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;753 Tekstil- og beklædningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;544,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;753 Tekstil- og beklædningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;225,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;753 Tekstil- og beklædningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;318,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;754 Andet håndværkspræget arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38122,20\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;754 Andet håndværkspræget arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;38530,43\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;754 Andet håndværkspræget arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;37165,02\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;754 Andet håndværkspræget arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2141,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;754 Andet håndværkspræget arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1478,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;754 Andet håndværkspræget arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;664,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;35924,04\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;36024,67\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32838,26\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1743,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1687,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;57,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32416,16\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32684,24\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31160,29\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5806,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4756,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1050,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;43635,26\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;44191,54\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;41668,49\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3342,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2572,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;770,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;33841,85\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;34582,30\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31881,25\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6479,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4687,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1792,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;33844,22\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;37376,97\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29405,53\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1235,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;681,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;554,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;34281,04\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;35202,59\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31579,38\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;13493,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;10033,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3460,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;817 Operatørarbejde ved træforarbejdning og papirfremstilling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31258,57\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;817 Operatørarbejde ved træforarbejdning og papirfremstilling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;31471,96\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;817 Operatørarbejde ved træforarbejdning og papirfremstilling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;30214,17\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;817 Operatørarbejde ved træforarbejdning og papirfremstilling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3376,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;817 Operatørarbejde ved træforarbejdning og papirfremstilling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2797,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;817 Operatørarbejde ved træforarbejdning og papirfremstilling;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;580,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;818 Andet operatørarbejde ved stationære anlæg og maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38019,01\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;818 Andet operatørarbejde ved stationære anlæg og maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;39051,32\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;818 Andet operatørarbejde ved stationære anlæg og maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33594,09\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;818 Andet operatørarbejde ved stationære anlæg og maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;9478,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;818 Andet operatørarbejde ved stationære anlæg og maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;7638,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;818 Andet operatørarbejde ved stationære anlæg og maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1840,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;821 Monteringsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;33801,43\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;821 Monteringsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;35650,75\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;821 Monteringsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;30170,82\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;821 Monteringsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;11385,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;821 Monteringsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;7459,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;821 Monteringsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3926,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;831 Lokomotivførerarbejde og tograngering;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;43367,12\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;831 Lokomotivførerarbejde og tograngering;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;43472,96\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;831 Lokomotivførerarbejde og tograngering;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;40977,74\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;831 Lokomotivførerarbejde og tograngering;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2748,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;831 Lokomotivførerarbejde og tograngering;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2628,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;831 Lokomotivførerarbejde og tograngering;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;120,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;832 Chauffører af biler, varevogne og motorcykler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;28589,32\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;832 Chauffører af biler, varevogne og motorcykler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;28724,04\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;832 Chauffører af biler, varevogne og motorcykler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;26432,88\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;832 Chauffører af biler, varevogne og motorcykler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3449,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;832 Chauffører af biler, varevogne og motorcykler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3250,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;832 Chauffører af biler, varevogne og motorcykler;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;199,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;833 Chauffører af busser og lastbiler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31605,33\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;833 Chauffører af busser og lastbiler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;31721,54\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;833 Chauffører af busser og lastbiler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29674,65\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;833 Chauffører af busser og lastbiler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;25259,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;833 Chauffører af busser og lastbiler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;23754,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;833 Chauffører af busser og lastbiler;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1505,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;834 Førere af mobile maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;33388,11\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;834 Førere af mobile maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;33553,84\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;834 Førere af mobile maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;30118,01\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;834 Førere af mobile maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6796,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;834 Førere af mobile maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6464,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;834 Førere af mobile maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;332,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;835 Dæksarbejde og medhjælp om bord på skibe;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38476,01\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;835 Dæksarbejde og medhjælp om bord på skibe;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;38720,25\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;835 Dæksarbejde og medhjælp om bord på skibe;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;835 Dæksarbejde og medhjælp om bord på skibe;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;299,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;835 Dæksarbejde og medhjælp om bord på skibe;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;284,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;835 Dæksarbejde og medhjælp om bord på skibe;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;14,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;911 Rengøringsarbejde i private hjem, hoteller og kontorer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;27454,62\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;911 Rengøringsarbejde i private hjem, hoteller og kontorer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;28799,70\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;911 Rengøringsarbejde i private hjem, hoteller og kontorer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;26774,86\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;911 Rengøringsarbejde i private hjem, hoteller og kontorer;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;35304,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;911 Rengøringsarbejde i private hjem, hoteller og kontorer;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;11375,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;911 Rengøringsarbejde i private hjem, hoteller og kontorer;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;23928,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;27733,55\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;28640,08\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;26352,18\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3889,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2323,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1566,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;29410,76\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;29481,89\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29169,24\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;686,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;516,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;171,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;34086,74\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;34203,40\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31374,48\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;27824,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;26597,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1227,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;932 Manuelt produktionsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;30177,57\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;932 Manuelt produktionsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;30970,39\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;932 Manuelt produktionsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28484,90\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;932 Manuelt produktionsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;14000,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;932 Manuelt produktionsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;9505,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;932 Manuelt produktionsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4495,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;933 Transport- og lagerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31100,04\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;933 Transport- og lagerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;31711,63\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;933 Transport- og lagerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;27646,74\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;933 Transport- og lagerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;24426,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;933 Transport- og lagerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;20690,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;933 Transport- og lagerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3736,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;941 Manuelt arbejde med tilberedning af mad;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;26609,18\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;941 Manuelt arbejde med tilberedning af mad;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;24968,98\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;941 Manuelt arbejde med tilberedning af mad;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;27200,40\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;941 Manuelt arbejde med tilberedning af mad;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;12893,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;941 Manuelt arbejde med tilberedning af mad;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2953,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;941 Manuelt arbejde med tilberedning af mad;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;9940,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;951 Servicearbejde på gaden;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;30682,28\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;951 Servicearbejde på gaden;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;30730,32\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;951 Servicearbejde på gaden;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;951 Servicearbejde på gaden;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;62,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;951 Servicearbejde på gaden;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;59,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;951 Servicearbejde på gaden;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;952 Gadesalg (undtagen madvarer);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;952 Gadesalg (undtagen madvarer);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;952 Gadesalg (undtagen madvarer);STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;952 Gadesalg (undtagen madvarer);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;13,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;952 Gadesalg (undtagen madvarer);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;7,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;952 Gadesalg (undtagen madvarer);Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;961 Renovationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;33951,95\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;961 Renovationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;34199,53\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;961 Renovationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29978,19\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;961 Renovationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2761,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;961 Renovationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2599,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;961 Renovationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;162,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;962 Andet manuelt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31074,30\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;962 Andet manuelt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32602,89\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;962 Andet manuelt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;26291,84\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;962 Andet manuelt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5192,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;962 Andet manuelt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3944,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2017;962 Andet manuelt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1249,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;011 Militært arbejde på officersniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;55559,86\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;011 Militært arbejde på officersniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;56265,93\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;011 Militært arbejde på officersniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;46116,45\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;011 Militært arbejde på officersniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3603,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;011 Militært arbejde på officersniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3347,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;011 Militært arbejde på officersniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;256,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;39626,96\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;39860,69\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;35812,68\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4173,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3927,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;245,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;031 Militært arbejde, øvrige rangordener;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;33735,67\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;031 Militært arbejde, øvrige rangordener;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;33931,40\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;031 Militært arbejde, øvrige rangordener;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31485,71\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;031 Militært arbejde, øvrige rangordener;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;7254,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;031 Militært arbejde, øvrige rangordener;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6659,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;031 Militært arbejde, øvrige rangordener;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;596,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;80237,20\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;85042,99\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;72310,82\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1563,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;971,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;592,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;112 Øverste virksomhedsledelse;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;104028,18\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;112 Øverste virksomhedsledelse;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;106197,01\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;112 Øverste virksomhedsledelse;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;90814,48\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;112 Øverste virksomhedsledelse;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;13982,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;112 Øverste virksomhedsledelse;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;11999,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;112 Øverste virksomhedsledelse;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1984,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;121 Ledelse inden for forretningsservice og administrative funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;73314,07\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;121 Ledelse inden for forretningsservice og administrative funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;76427,39\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;121 Ledelse inden for forretningsservice og administrative funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;68337,15\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;121 Ledelse inden for forretningsservice og administrative funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;18832,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;121 Ledelse inden for forretningsservice og administrative funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;11500,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;121 Ledelse inden for forretningsservice og administrative funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7332,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;122 Ledelse af forretnings- og udviklingsorienterede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;75959,31\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;122 Ledelse af forretnings- og udviklingsorienterede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;78915,78\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;122 Ledelse af forretnings- og udviklingsorienterede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;66933,06\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;122 Ledelse af forretnings- og udviklingsorienterede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;11728,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;122 Ledelse af forretnings- og udviklingsorienterede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;8793,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;122 Ledelse af forretnings- og udviklingsorienterede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2936,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;57983,05\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;56635,80\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;142,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;128,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;15,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;64239,31\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;64375,01\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;63212,41\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;10275,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;9066,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1209,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;78736,18\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;79382,96\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;74793,67\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2496,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2132,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;364,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;134 Ledelse af hovedaktiviteten inden for servicefag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;58186,98\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;134 Ledelse af hovedaktiviteten inden for servicefag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;65161,29\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;134 Ledelse af hovedaktiviteten inden for servicefag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;52980,01\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;134 Ledelse af hovedaktiviteten inden for servicefag;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;13735,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;134 Ledelse af hovedaktiviteten inden for servicefag;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;5732,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;134 Ledelse af hovedaktiviteten inden for servicefag;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;8003,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;48067,54\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;49483,51\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;45871,58\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;634,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;379,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;255,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;44252,65\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;47086,04\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;38367,72\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5931,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3953,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1978,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;143 Ledelse af hovedaktiviteten inden for andre serviceområder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;59697,19\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;143 Ledelse af hovedaktiviteten inden for andre serviceområder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;62268,37\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;143 Ledelse af hovedaktiviteten inden for andre serviceområder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;53925,29\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;143 Ledelse af hovedaktiviteten inden for andre serviceområder;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1392,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;143 Ledelse af hovedaktiviteten inden for andre serviceområder;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;953,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;143 Ledelse af hovedaktiviteten inden for andre serviceområder;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;439,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;211 Arbejde inden for fysik og geologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;56515,60\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;211 Arbejde inden for fysik og geologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;58931,06\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;211 Arbejde inden for fysik og geologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;52807,99\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;211 Arbejde inden for fysik og geologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2383,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;211 Arbejde inden for fysik og geologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1421,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;211 Arbejde inden for fysik og geologi;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;962,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;62947,11\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;65208,66\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;58974,17\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1851,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1160,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;691,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;213 Arbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;53366,08\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;213 Arbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;54619,60\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;213 Arbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;51976,86\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;213 Arbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4673,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;213 Arbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2403,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;213 Arbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2270,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;214 Ingeniørarbejde (undtagen inden for elektroteknologi);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;58478,07\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;214 Ingeniørarbejde (undtagen inden for elektroteknologi);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;59497,14\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;214 Ingeniørarbejde (undtagen inden for elektroteknologi);STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;54992,13\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;214 Ingeniørarbejde (undtagen inden for elektroteknologi);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;40481,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;214 Ingeniørarbejde (undtagen inden for elektroteknologi);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;30897,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;214 Ingeniørarbejde (undtagen inden for elektroteknologi);Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;9585,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;215 Ingeniørarbejde inden for elektroteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;60841,25\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;215 Ingeniørarbejde inden for elektroteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;61142,25\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;215 Ingeniørarbejde inden for elektroteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;56653,47\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;215 Ingeniørarbejde inden for elektroteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6075,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;215 Ingeniørarbejde inden for elektroteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;5654,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;215 Ingeniørarbejde inden for elektroteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;421,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;216 Arbejde med arkitektur, infrastruktur og design;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;46746,86\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;216 Arbejde med arkitektur, infrastruktur og design;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;49718,45\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;216 Arbejde med arkitektur, infrastruktur og design;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;43746,53\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;216 Arbejde med arkitektur, infrastruktur og design;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;8483,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;216 Arbejde med arkitektur, infrastruktur og design;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4181,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;216 Arbejde med arkitektur, infrastruktur og design;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4302,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;221 Lægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;70352,87\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;221 Lægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;75493,89\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;221 Lægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;65633,75\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;221 Lægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;18220,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;221 Lægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;8415,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;221 Lægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;9805,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;222 Sygeplejerske- og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;40589,60\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;222 Sygeplejerske- og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;43214,93\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;222 Sygeplejerske- og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;40465,51\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;222 Sygeplejerske- og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;54018,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;222 Sygeplejerske- og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2326,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;222 Sygeplejerske- og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;51692,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;223 Arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;223 Arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;223 Arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;223 Arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;8,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;223 Arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;223 Arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;5,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;224 Paramedicinsk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;43516,92\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;224 Paramedicinsk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;43488,85\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;224 Paramedicinsk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;224 Paramedicinsk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;394,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;224 Paramedicinsk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;365,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;224 Paramedicinsk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;29,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;225 Dyrlægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;52132,59\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;225 Dyrlægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;56166,66\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;225 Dyrlægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;50228,29\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;225 Dyrlægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;795,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;225 Dyrlægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;244,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;225 Dyrlægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;550,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;226 Andet sundhedsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;44949,40\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;226 Andet sundhedsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;52117,24\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;226 Andet sundhedsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;43178,80\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;226 Andet sundhedsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;23806,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;226 Andet sundhedsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4456,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;226 Andet sundhedsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;19350,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;231 Undervisning og forskning ved universiteter og højere læreanstalter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;47854,57\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;231 Undervisning og forskning ved universiteter og højere læreanstalter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;49418,57\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;231 Undervisning og forskning ved universiteter og højere læreanstalter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;45977,35\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;231 Undervisning og forskning ved universiteter og højere læreanstalter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;25185,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;231 Undervisning og forskning ved universiteter og højere læreanstalter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;13548,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;231 Undervisning og forskning ved universiteter og højere læreanstalter;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;11637,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;232 Undervisning ved erhvervsuddannelser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;42815,76\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;232 Undervisning ved erhvervsuddannelser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;42915,89\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;232 Undervisning ved erhvervsuddannelser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;42683,05\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;232 Undervisning ved erhvervsuddannelser;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;9620,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;232 Undervisning ved erhvervsuddannelser;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;5433,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;232 Undervisning ved erhvervsuddannelser;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4187,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;233 Almen undervisning efter grundskoleniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;49137,09\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;233 Almen undervisning efter grundskoleniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;49707,02\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;233 Almen undervisning efter grundskoleniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;48624,16\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;233 Almen undervisning efter grundskoleniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;14353,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;233 Almen undervisning efter grundskoleniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6721,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;233 Almen undervisning efter grundskoleniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7631,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;234 Undervisning og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;39448,12\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;234 Undervisning og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;40509,43\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;234 Undervisning og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;39040,95\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;234 Undervisning og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;114266,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;234 Undervisning og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;30660,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;234 Undervisning og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;83606,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;235 Andet undervisnings- og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;39399,11\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;235 Andet undervisnings- og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;40159,95\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;235 Andet undervisnings- og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;39033,69\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;235 Andet undervisnings- og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;38948,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;235 Andet undervisnings- og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;12243,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;235 Andet undervisnings- og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;26705,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;241 Arbejde inden for finans og økonomi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;58804,54\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;241 Arbejde inden for finans og økonomi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;62664,35\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;241 Arbejde inden for finans og økonomi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;53847,69\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;241 Arbejde inden for finans og økonomi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;31890,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;241 Arbejde inden for finans og økonomi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;17638,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;241 Arbejde inden for finans og økonomi;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;14252,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;242 Arbejde inden for virksomhedsadministration - privat og offentlig;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;61321,92\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;242 Arbejde inden for virksomhedsadministration - privat og offentlig;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;64581,20\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;242 Arbejde inden for virksomhedsadministration - privat og offentlig;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;58308,31\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;242 Arbejde inden for virksomhedsadministration - privat og offentlig;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;17689,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;242 Arbejde inden for virksomhedsadministration - privat og offentlig;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;8376,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;242 Arbejde inden for virksomhedsadministration - privat og offentlig;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;9312,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;243 Arbejde inden for salg, marketing og PR;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;54930,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;243 Arbejde inden for salg, marketing og PR;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;59480,12\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;243 Arbejde inden for salg, marketing og PR;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;49621,18\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;243 Arbejde inden for salg, marketing og PR;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;16974,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;243 Arbejde inden for salg, marketing og PR;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;9029,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;243 Arbejde inden for salg, marketing og PR;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7945,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;251 Udvikling og analyse af software og applikatoner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;57397,53\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;251 Udvikling og analyse af software og applikatoner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;58358,73\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;251 Udvikling og analyse af software og applikatoner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;54075,20\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;251 Udvikling og analyse af software og applikatoner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;42218,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;251 Udvikling og analyse af software og applikatoner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;32366,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;251 Udvikling og analyse af software og applikatoner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;9852,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;252 Arbejde med databaser og netværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;53503,98\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;252 Arbejde med databaser og netværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;54309,17\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;252 Arbejde med databaser og netværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;49775,39\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;252 Arbejde med databaser og netværk;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;7641,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;252 Arbejde med databaser og netværk;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6245,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;252 Arbejde med databaser og netværk;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1396,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;261 Juridisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;56132,13\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;261 Juridisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;59109,76\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;261 Juridisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;53980,92\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;261 Juridisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;17000,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;261 Juridisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6923,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;261 Juridisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;10077,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;43527,03\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;43937,73\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;43358,05\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2954,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;840,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2114,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;263 Arbejde inden for samfundsvidenskab og religion;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;44263,98\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;263 Arbejde inden for samfundsvidenskab og religion;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;48978,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;263 Arbejde inden for samfundsvidenskab og religion;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;42497,93\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;263 Arbejde inden for samfundsvidenskab og religion;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;29700,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;263 Arbejde inden for samfundsvidenskab og religion;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;7711,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;263 Arbejde inden for samfundsvidenskab og religion;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;21988,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;48270,41\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;49993,73\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;46457,58\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;8377,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4226,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4151,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;265 Arbejde med kunst og kreative fag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;42735,20\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;265 Arbejde med kunst og kreative fag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;44857,03\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;265 Arbejde med kunst og kreative fag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;40651,62\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;265 Arbejde med kunst og kreative fag;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3905,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;265 Arbejde med kunst og kreative fag;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1905,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;265 Arbejde med kunst og kreative fag;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2000,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;45862,45\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;47957,37\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;40663,90\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;43272,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;30369,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;12903,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;313 Teknisk kontrolarbejde inden for processtyring;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;48258,38\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;313 Teknisk kontrolarbejde inden for processtyring;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;49965,54\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;313 Teknisk kontrolarbejde inden for processtyring;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;44138,99\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;313 Teknisk kontrolarbejde inden for processtyring;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2785,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;313 Teknisk kontrolarbejde inden for processtyring;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1941,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;313 Teknisk kontrolarbejde inden for processtyring;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;844,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;314 Teknikerarbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;44124,44\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;314 Teknikerarbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;45423,63\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;314 Teknikerarbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;41172,08\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;314 Teknikerarbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;738,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;314 Teknikerarbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;508,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;314 Teknikerarbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;230,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;315 Teknikerarbejde inden for skibs- og luftfart;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;73302,74\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;315 Teknikerarbejde inden for skibs- og luftfart;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;74376,51\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;315 Teknikerarbejde inden for skibs- og luftfart;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;62072,73\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;315 Teknikerarbejde inden for skibs- og luftfart;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3398,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;315 Teknikerarbejde inden for skibs- og luftfart;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3098,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;315 Teknikerarbejde inden for skibs- og luftfart;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;300,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38396,62\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;40549,37\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;38025,01\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;11505,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1648,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;9857,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38957,62\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;40238,50\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;38088,55\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;376,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;151,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;225,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;9,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;0,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;9,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;29656,95\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29470,49\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;194,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;14,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;180,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;39719,11\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;44314,38\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;36682,58\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6777,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2674,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4103,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;331 Arbejde med finans, regnskab og matematik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;47133,74\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;331 Arbejde med finans, regnskab og matematik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;53333,48\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;331 Arbejde med finans, regnskab og matematik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;43193,07\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;331 Arbejde med finans, regnskab og matematik;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;26410,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;331 Arbejde med finans, regnskab og matematik;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;10186,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;331 Arbejde med finans, regnskab og matematik;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;16224,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;332 Agent- og mæglerarbejde inden for salg og indkøb;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;49066,43\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;332 Agent- og mæglerarbejde inden for salg og indkøb;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;52389,57\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;332 Agent- og mæglerarbejde inden for salg og indkøb;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;43572,82\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;332 Agent- og mæglerarbejde inden for salg og indkøb;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;37000,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;332 Agent- og mæglerarbejde inden for salg og indkøb;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;22817,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;332 Agent- og mæglerarbejde inden for salg og indkøb;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;14183,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;333 Arbejde inden for forretningsservice;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;44702,52\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;333 Arbejde inden for forretningsservice;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;48316,47\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;333 Arbejde inden for forretningsservice;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;41185,38\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;333 Arbejde inden for forretningsservice;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;13709,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;333 Arbejde inden for forretningsservice;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6618,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;333 Arbejde inden for forretningsservice;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7091,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;334 Administrativt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;40892,64\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;334 Administrativt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;47330,85\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;334 Administrativt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;39800,18\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;334 Administrativt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;23771,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;334 Administrativt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3442,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;334 Administrativt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;20329,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;335 Arbejde med forvaltning og udøvelse af lovgivning;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;47101,85\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;335 Arbejde med forvaltning og udøvelse af lovgivning;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;51930,77\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;335 Arbejde med forvaltning og udøvelse af lovgivning;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;43595,42\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;335 Arbejde med forvaltning og udøvelse af lovgivning;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;9956,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;335 Arbejde med forvaltning og udøvelse af lovgivning;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4119,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;335 Arbejde med forvaltning og udøvelse af lovgivning;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;5837,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;341 Arbejde inden for jura og samfundsforhold;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;41799,19\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;341 Arbejde inden for jura og samfundsforhold;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;44494,46\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;341 Arbejde inden for jura og samfundsforhold;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;40738,03\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;341 Arbejde inden for jura og samfundsforhold;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4919,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;341 Arbejde inden for jura og samfundsforhold;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1340,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;341 Arbejde inden for jura og samfundsforhold;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3579,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;342 Arbejde med sport og fitness;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37076,91\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;342 Arbejde med sport og fitness;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;40244,12\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;342 Arbejde med sport og fitness;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29816,35\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;342 Arbejde med sport og fitness;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1453,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;342 Arbejde med sport og fitness;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1000,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;342 Arbejde med sport og fitness;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;453,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37867,42\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;39585,98\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;36201,03\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4444,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2094,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2349,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;47012,56\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;47617,12\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;44554,89\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;9755,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;7763,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1992,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;43115,97\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;43183,87\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;42824,86\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3308,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2659,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;649,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;411 Almindeligt kontorarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37814,61\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;411 Almindeligt kontorarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;41718,68\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;411 Almindeligt kontorarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;36533,57\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;411 Almindeligt kontorarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;68198,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;411 Almindeligt kontorarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;16239,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;411 Almindeligt kontorarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;51959,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;412 Almindeligt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;39114,75\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;412 Almindeligt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;41104,31\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;412 Almindeligt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;38955,89\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;412 Almindeligt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6115,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;412 Almindeligt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;444,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;412 Almindeligt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;5671,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;413 Indtastningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;39149,36\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;413 Indtastningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;41382,75\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;413 Indtastningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;37404,39\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;413 Indtastningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1797,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;413 Indtastningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;774,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;413 Indtastningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1023,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;39748,71\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;41560,03\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;38891,98\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2348,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;731,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1617,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;422 Kundeinformationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;33002,05\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;422 Kundeinformationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;33532,73\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;422 Kundeinformationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32728,33\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;422 Kundeinformationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;13036,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;422 Kundeinformationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4416,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;422 Kundeinformationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;8620,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;431 Almindeligt beregningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;41461,95\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;431 Almindeligt beregningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;44900,95\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;431 Almindeligt beregningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;40386,01\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;431 Almindeligt beregningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;17566,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;431 Almindeligt beregningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4136,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;431 Almindeligt beregningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;13430,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;432 Kontorarbejde inden for materialeregistrering og transport;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;34270,98\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;432 Kontorarbejde inden for materialeregistrering og transport;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;34647,07\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;432 Kontorarbejde inden for materialeregistrering og transport;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33182,73\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;432 Kontorarbejde inden for materialeregistrering og transport;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;19392,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;432 Kontorarbejde inden for materialeregistrering og transport;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;14295,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;432 Kontorarbejde inden for materialeregistrering og transport;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;5097,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;441 Andet almindeligt kontor- og kundeservicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36469,03\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;441 Andet almindeligt kontor- og kundeservicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;36540,39\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;441 Andet almindeligt kontor- og kundeservicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;36403,19\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;441 Andet almindeligt kontor- og kundeservicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;19142,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;441 Andet almindeligt kontor- og kundeservicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;9075,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;441 Andet almindeligt kontor- og kundeservicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;10067,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;511 Service- og kontrolarbejde under transport og rejser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;39637,81\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;511 Service- og kontrolarbejde under transport og rejser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;41044,33\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;511 Service- og kontrolarbejde under transport og rejser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;38469,19\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;511 Service- og kontrolarbejde under transport og rejser;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4419,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;511 Service- og kontrolarbejde under transport og rejser;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1944,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;511 Service- og kontrolarbejde under transport og rejser;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2475,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;512 Kokkearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;30182,28\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;512 Kokkearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;30467,13\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;512 Kokkearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29747,44\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;512 Kokkearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6064,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;512 Kokkearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3504,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;512 Kokkearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2560,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;513 Tjenere og bartendere;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;25990,27\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;513 Tjenere og bartendere;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;27140,75\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;513 Tjenere og bartendere;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;25067,28\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;513 Tjenere og bartendere;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5293,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;513 Tjenere og bartendere;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2349,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;513 Tjenere og bartendere;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2944,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32137,12\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;31379,53\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32245,57\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;704,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;85,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;619,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;34678,64\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;34851,67\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33484,86\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;16333,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;14237,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2096,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;516 Andet servicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;30514,09\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;516 Andet servicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;31494,83\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;516 Andet servicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28817,17\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;516 Andet servicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3392,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;516 Andet servicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2112,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;516 Andet servicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1280,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;521 Salgsarbejde på gade og marked;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;521 Salgsarbejde på gade og marked;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;521 Salgsarbejde på gade og marked;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;521 Salgsarbejde på gade og marked;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;46,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;521 Salgsarbejde på gade og marked;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;22,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;521 Salgsarbejde på gade og marked;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;23,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;522 Salgsarbejde i butik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;28118,40\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;522 Salgsarbejde i butik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;30293,04\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;522 Salgsarbejde i butik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;26291,76\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;522 Salgsarbejde i butik;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;33557,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;522 Salgsarbejde i butik;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;15139,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;522 Salgsarbejde i butik;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;18418,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;523 Kasseassistentarbejde og beslægtet kundebetjening;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;25137,63\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;523 Kasseassistentarbejde og beslægtet kundebetjening;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;25291,44\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;523 Kasseassistentarbejde og beslægtet kundebetjening;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;25067,78\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;523 Kasseassistentarbejde og beslægtet kundebetjening;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;9083,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;523 Kasseassistentarbejde og beslægtet kundebetjening;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2780,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;523 Kasseassistentarbejde og beslægtet kundebetjening;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;6304,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;524 Andet salgsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36899,10\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;524 Andet salgsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;39774,62\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;524 Andet salgsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32971,34\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;524 Andet salgsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;15183,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;524 Andet salgsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;8694,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;524 Andet salgsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;6489,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;531 Børneomsorgsarbejde og hjælpelærerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;28435,34\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;531 Børneomsorgsarbejde og hjælpelærerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;27996,17\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;531 Børneomsorgsarbejde og hjælpelærerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28543,60\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;531 Børneomsorgsarbejde og hjælpelærerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;35198,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;531 Børneomsorgsarbejde og hjælpelærerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6787,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;531 Børneomsorgsarbejde og hjælpelærerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;28411,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;532 Omsorgsarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;33254,13\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;532 Omsorgsarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32851,52\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;532 Omsorgsarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33324,01\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;532 Omsorgsarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;93502,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;532 Omsorgsarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;13400,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;532 Omsorgsarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;80102,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;541 Rednings- og overvågningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37964,56\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;541 Rednings- og overvågningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;38474,26\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;541 Rednings- og overvågningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;35702,55\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;541 Rednings- og overvågningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;21062,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;541 Rednings- og overvågningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;17121,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;541 Rednings- og overvågningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3941,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;611 Arbejde med plantevækst;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32524,98\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;611 Arbejde med plantevækst;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32971,50\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;611 Arbejde med plantevækst;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31144,55\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;611 Arbejde med plantevækst;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3102,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;611 Arbejde med plantevækst;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2287,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;611 Arbejde med plantevækst;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;815,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;612 Arbejde med dyreopdræt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32094,81\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;612 Arbejde med dyreopdræt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32302,75\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;612 Arbejde med dyreopdræt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31921,85\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;612 Arbejde med dyreopdræt;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;373,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;612 Arbejde med dyreopdræt;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;166,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;612 Arbejde med dyreopdræt;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;207,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;28176,81\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;28075,02\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;157,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;128,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;29,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;621 Skovbrugsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;34006,28\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;621 Skovbrugsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;33829,10\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;621 Skovbrugsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;621 Skovbrugsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;319,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;621 Skovbrugsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;289,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;621 Skovbrugsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;30,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;622 Arbejde med fiskeri og jagt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;622 Arbejde med fiskeri og jagt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;622 Arbejde med fiskeri og jagt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;622 Arbejde med fiskeri og jagt;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;20,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;622 Arbejde med fiskeri og jagt;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;17,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;622 Arbejde med fiskeri og jagt;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;711 Bygningsarbejde (ekskl. medhjælpere) - basis;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36696,81\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;711 Bygningsarbejde (ekskl. medhjælpere) - basis;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;36744,93\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;711 Bygningsarbejde (ekskl. medhjælpere) - basis;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32385,35\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;711 Bygningsarbejde (ekskl. medhjælpere) - basis;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;22195,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;711 Bygningsarbejde (ekskl. medhjælpere) - basis;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;21950,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;711 Bygningsarbejde (ekskl. medhjælpere) - basis;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;245,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;712 Bygningsarbejde (ekskl. medhjælpere) - finish;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;40024,78\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;712 Bygningsarbejde (ekskl. medhjælpere) - finish;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;40084,31\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;712 Bygningsarbejde (ekskl. medhjælpere) - finish;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31723,59\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;712 Bygningsarbejde (ekskl. medhjælpere) - finish;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;10234,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;712 Bygningsarbejde (ekskl. medhjælpere) - finish;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;10159,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;712 Bygningsarbejde (ekskl. medhjælpere) - finish;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;75,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;713 Malere og arbejde inden for rensning af bygninger;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;33583,39\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;713 Malere og arbejde inden for rensning af bygninger;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;34043,59\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;713 Malere og arbejde inden for rensning af bygninger;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31707,09\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;713 Malere og arbejde inden for rensning af bygninger;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5765,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;713 Malere og arbejde inden for rensning af bygninger;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4583,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;713 Malere og arbejde inden for rensning af bygninger;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1181,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;721 Pladearbejde, svejsning og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;35775,04\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;721 Pladearbejde, svejsning og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;35849,75\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;721 Pladearbejde, svejsning og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32205,80\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;721 Pladearbejde, svejsning og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;9332,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;721 Pladearbejde, svejsning og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;9131,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;721 Pladearbejde, svejsning og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;200,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;722 Smede, værktøjsmagere og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37132,12\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;722 Smede, værktøjsmagere og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;37306,56\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;722 Smede, værktøjsmagere og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32940,27\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;722 Smede, værktøjsmagere og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;22025,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;722 Smede, værktøjsmagere og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;21111,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;722 Smede, værktøjsmagere og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;914,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;723 Mekanikerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37452,89\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;723 Mekanikerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;37484,30\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;723 Mekanikerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;34955,23\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;723 Mekanikerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;15501,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;723 Mekanikerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;15298,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;723 Mekanikerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;203,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;731 Præcisionshåndværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;33700,86\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;731 Præcisionshåndværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;36535,96\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;731 Præcisionshåndværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29561,18\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;731 Præcisionshåndværk;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1099,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;731 Præcisionshåndværk;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;648,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;731 Præcisionshåndværk;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;451,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;732 Grafisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38063,34\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;732 Grafisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;39694,23\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;732 Grafisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;34021,06\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;732 Grafisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2726,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;732 Grafisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1955,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;732 Grafisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;771,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;741 Installation- og reparation af elektrisk udstyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;39575,84\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;741 Installation- og reparation af elektrisk udstyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;39620,78\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;741 Installation- og reparation af elektrisk udstyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;36275,37\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;741 Installation- og reparation af elektrisk udstyr;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;17507,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;741 Installation- og reparation af elektrisk udstyr;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;17265,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;741 Installation- og reparation af elektrisk udstyr;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;241,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;742 Installation og reparation af elektronik og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;40841,68\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;742 Installation og reparation af elektronik og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;41108,55\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;742 Installation og reparation af elektronik og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;34995,35\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;742 Installation og reparation af elektronik og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1715,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;742 Installation og reparation af elektronik og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1636,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;742 Installation og reparation af elektronik og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;80,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;751 Arbejde inden for nærings- og nydelsesmiddelområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;33889,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;751 Arbejde inden for nærings- og nydelsesmiddelområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;35152,99\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;751 Arbejde inden for nærings- og nydelsesmiddelområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;30813,92\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;751 Arbejde inden for nærings- og nydelsesmiddelområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;8197,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;751 Arbejde inden for nærings- og nydelsesmiddelområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;5775,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;751 Arbejde inden for nærings- og nydelsesmiddelområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2421,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;752 Håndværkspræget arbejde i træindustri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;30459,77\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;752 Håndværkspræget arbejde i træindustri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;30654,98\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;752 Håndværkspræget arbejde i træindustri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28491,60\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;752 Håndværkspræget arbejde i træindustri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1726,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;752 Håndværkspræget arbejde i træindustri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1570,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;752 Håndværkspræget arbejde i træindustri;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;156,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;753 Tekstil- og beklædningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32300,71\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;753 Tekstil- og beklædningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32702,01\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;753 Tekstil- og beklædningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32023,48\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;753 Tekstil- og beklædningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;546,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;753 Tekstil- og beklædningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;222,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;753 Tekstil- og beklædningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;324,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;754 Andet håndværkspræget arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38374,30\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;754 Andet håndværkspræget arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;38713,34\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;754 Andet håndværkspræget arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;37595,70\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;754 Andet håndværkspræget arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2214,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;754 Andet håndværkspræget arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1515,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;754 Andet håndværkspræget arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;699,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38457,81\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;38644,49\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33809,69\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1918,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1842,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;75,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32805,38\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;33117,13\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;30952,53\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5554,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4727,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;827,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;44295,34\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;44784,40\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;42568,66\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3354,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2585,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;770,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;34506,97\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;35220,73\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32459,64\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6417,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4746,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1671,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32828,05\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;36079,76\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28064,11\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1124,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;664,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;460,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;34522,18\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;35420,77\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31802,16\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;14108,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;10582,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3526,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;817 Operatørarbejde ved træforarbejdning og papirfremstilling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31699,04\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;817 Operatørarbejde ved træforarbejdning og papirfremstilling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;31976,36\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;817 Operatørarbejde ved træforarbejdning og papirfremstilling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;30347,63\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;817 Operatørarbejde ved træforarbejdning og papirfremstilling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3470,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;817 Operatørarbejde ved træforarbejdning og papirfremstilling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2868,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;817 Operatørarbejde ved træforarbejdning og papirfremstilling;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;602,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;818 Andet operatørarbejde ved stationære anlæg og maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38038,09\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;818 Andet operatørarbejde ved stationære anlæg og maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;39096,34\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;818 Andet operatørarbejde ved stationære anlæg og maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33766,99\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;818 Andet operatørarbejde ved stationære anlæg og maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;8564,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;818 Andet operatørarbejde ved stationære anlæg og maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6811,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;818 Andet operatørarbejde ved stationære anlæg og maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1753,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;821 Monteringsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;34215,48\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;821 Monteringsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;35955,65\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;821 Monteringsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;30572,71\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;821 Monteringsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;11618,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;821 Monteringsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;7779,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;821 Monteringsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3839,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;831 Lokomotivførerarbejde og tograngering;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;44419,66\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;831 Lokomotivførerarbejde og tograngering;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;44505,44\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;831 Lokomotivførerarbejde og tograngering;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;42524,21\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;831 Lokomotivførerarbejde og tograngering;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2851,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;831 Lokomotivførerarbejde og tograngering;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2725,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;831 Lokomotivførerarbejde og tograngering;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;126,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;832 Chauffører af biler, varevogne og motorcykler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;28733,48\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;832 Chauffører af biler, varevogne og motorcykler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;28823,51\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;832 Chauffører af biler, varevogne og motorcykler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;27075,40\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;832 Chauffører af biler, varevogne og motorcykler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3528,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;832 Chauffører af biler, varevogne og motorcykler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3348,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;832 Chauffører af biler, varevogne og motorcykler;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;180,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;833 Chauffører af busser og lastbiler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31868,77\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;833 Chauffører af busser og lastbiler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;31978,30\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;833 Chauffører af busser og lastbiler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29966,33\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;833 Chauffører af busser og lastbiler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;26157,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;833 Chauffører af busser og lastbiler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;24654,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;833 Chauffører af busser og lastbiler;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1503,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;834 Førere af mobile maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;34082,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;834 Førere af mobile maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;34186,07\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;834 Førere af mobile maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31433,12\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;834 Førere af mobile maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6593,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;834 Førere af mobile maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6336,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;834 Førere af mobile maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;257,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;835 Dæksarbejde og medhjælp om bord på skibe;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;40658,11\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;835 Dæksarbejde og medhjælp om bord på skibe;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;40888,45\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;835 Dæksarbejde og medhjælp om bord på skibe;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;835 Dæksarbejde og medhjælp om bord på skibe;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;257,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;835 Dæksarbejde og medhjælp om bord på skibe;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;248,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;835 Dæksarbejde og medhjælp om bord på skibe;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;9,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;911 Rengøringsarbejde i private hjem, hoteller og kontorer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;27658,54\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;911 Rengøringsarbejde i private hjem, hoteller og kontorer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;28921,19\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;911 Rengøringsarbejde i private hjem, hoteller og kontorer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;27043,98\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;911 Rengøringsarbejde i private hjem, hoteller og kontorer;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;34699,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;911 Rengøringsarbejde i private hjem, hoteller og kontorer;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;10950,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;911 Rengøringsarbejde i private hjem, hoteller og kontorer;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;23749,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;28200,85\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;29306,49\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;26415,65\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4053,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2473,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1580,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;28765,21\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;28672,48\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29017,19\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1029,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;728,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;301,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;34266,98\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;34569,82\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;30438,81\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;29925,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;27581,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2344,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;932 Manuelt produktionsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;30509,71\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;932 Manuelt produktionsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;31298,18\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;932 Manuelt produktionsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28765,17\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;932 Manuelt produktionsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;14535,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;932 Manuelt produktionsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;9984,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;932 Manuelt produktionsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4551,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;933 Transport- og lagerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31365,54\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;933 Transport- og lagerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32001,42\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;933 Transport- og lagerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;27942,42\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;933 Transport- og lagerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;25543,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;933 Transport- og lagerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;21488,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;933 Transport- og lagerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4056,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;941 Manuelt arbejde med tilberedning af mad;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;26957,25\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;941 Manuelt arbejde med tilberedning af mad;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;25240,75\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;941 Manuelt arbejde med tilberedning af mad;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;27596,21\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;941 Manuelt arbejde med tilberedning af mad;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;13361,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;941 Manuelt arbejde med tilberedning af mad;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3168,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;941 Manuelt arbejde med tilberedning af mad;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;10192,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;951 Servicearbejde på gaden;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32868,86\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;951 Servicearbejde på gaden;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32999,13\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;951 Servicearbejde på gaden;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;951 Servicearbejde på gaden;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;61,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;951 Servicearbejde på gaden;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;58,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;951 Servicearbejde på gaden;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;952 Gadesalg (undtagen madvarer);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;952 Gadesalg (undtagen madvarer);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;952 Gadesalg (undtagen madvarer);STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;952 Gadesalg (undtagen madvarer);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;9,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;952 Gadesalg (undtagen madvarer);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;5,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;952 Gadesalg (undtagen madvarer);Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;961 Renovationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;34635,93\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;961 Renovationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;34874,01\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;961 Renovationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31049,85\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;961 Renovationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2776,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;961 Renovationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2605,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;961 Renovationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;171,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;962 Andet manuelt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32755,47\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;962 Andet manuelt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;34212,34\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;962 Andet manuelt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;27569,47\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;962 Andet manuelt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5162,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;962 Andet manuelt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4039,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2018;962 Andet manuelt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1123,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;011 Militært arbejde på officersniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;56252,69\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;011 Militært arbejde på officersniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;57075,34\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;011 Militært arbejde på officersniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;46034,53\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;011 Militært arbejde på officersniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3667,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;011 Militært arbejde på officersniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3386,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;011 Militært arbejde på officersniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;281,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;40301,84\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;40544,60\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;36506,15\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4126,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3873,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;021 Militært arbejde på befalingsmandsniveau (underofficersarbejde);Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;252,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;031 Militært arbejde, øvrige rangordener;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;34355,94\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;031 Militært arbejde, øvrige rangordener;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;34618,87\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;031 Militært arbejde, øvrige rangordener;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31484,53\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;031 Militært arbejde, øvrige rangordener;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;7016,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;031 Militært arbejde, øvrige rangordener;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6411,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;031 Militært arbejde, øvrige rangordener;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;606,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;81949,54\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;86312,30\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;74916,58\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1614,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;993,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;111 Øverste ledelse i lovgivende myndigheder, offentlige virksomheder og organisationer;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;621,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;112 Øverste virksomhedsledelse;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;104895,89\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;112 Øverste virksomhedsledelse;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;107188,11\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;112 Øverste virksomhedsledelse;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;91244,72\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;112 Øverste virksomhedsledelse;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;14493,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;112 Øverste virksomhedsledelse;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;12391,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;112 Øverste virksomhedsledelse;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2102,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;121 Ledelse inden for forretningsservice og administrative funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;73878,70\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;121 Ledelse inden for forretningsservice og administrative funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;77164,12\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;121 Ledelse inden for forretningsservice og administrative funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;68682,35\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;121 Ledelse inden for forretningsservice og administrative funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;20549,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;121 Ledelse inden for forretningsservice og administrative funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;12481,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;121 Ledelse inden for forretningsservice og administrative funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;8068,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;122 Ledelse af forretnings- og udviklingsorienterede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;77262,81\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;122 Ledelse af forretnings- og udviklingsorienterede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;80020,49\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;122 Ledelse af forretnings- og udviklingsorienterede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;68965,99\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;122 Ledelse af forretnings- og udviklingsorienterede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;11765,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;122 Ledelse af forretnings- og udviklingsorienterede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;8800,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;122 Ledelse af forretnings- og udviklingsorienterede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2965,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;59896,64\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;59601,42\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;147,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;131,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;131 Ledelse af produktion i landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;15,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;65826,89\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;65855,41\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;65612,53\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;10709,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;9450,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;132 Ledelse af hovedaktiviteten inden for fremstillingsvirksomhed, råstofudvinding, byggeri, forsyning, distribution mv.;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1259,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;79249,16\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;80153,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;73988,73\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2718,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2313,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;133 Ledelse af hovedaktiviteten inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;405,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;134 Ledelse af hovedaktiviteten inden for servicefag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;59557,95\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;134 Ledelse af hovedaktiviteten inden for servicefag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;66745,87\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;134 Ledelse af hovedaktiviteten inden for servicefag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;54294,70\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;134 Ledelse af hovedaktiviteten inden for servicefag;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;13538,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;134 Ledelse af hovedaktiviteten inden for servicefag;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;5587,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;134 Ledelse af hovedaktiviteten inden for servicefag;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7951,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;48868,18\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;50158,11\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;46887,80\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;650,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;386,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;141 Ledelse af hovedaktiviteten inden for hoteller og restauration;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;264,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;44433,77\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;47743,21\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;38197,27\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6093,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3924,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;142 Ledelse af hovedaktiviteten inden for detail- og engroshandel;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2169,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;143 Ledelse af hovedaktiviteten inden for andre serviceområder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;58717,79\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;143 Ledelse af hovedaktiviteten inden for andre serviceområder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;61304,70\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;143 Ledelse af hovedaktiviteten inden for andre serviceområder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;53226,08\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;143 Ledelse af hovedaktiviteten inden for andre serviceområder;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1465,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;143 Ledelse af hovedaktiviteten inden for andre serviceområder;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;982,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;143 Ledelse af hovedaktiviteten inden for andre serviceområder;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;483,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;211 Arbejde inden for fysik og geologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;56661,03\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;211 Arbejde inden for fysik og geologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;58773,37\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;211 Arbejde inden for fysik og geologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;53799,55\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;211 Arbejde inden for fysik og geologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2639,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;211 Arbejde inden for fysik og geologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1493,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;211 Arbejde inden for fysik og geologi;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1146,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;63714,01\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;65582,39\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;60465,51\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1716,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1075,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;212 Arbejde med matematiske, aktuariske og statistiske metoder og teorier;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;640,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;213 Arbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;54044,48\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;213 Arbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;54752,78\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;213 Arbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;53270,37\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;213 Arbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4990,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;213 Arbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2554,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;213 Arbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2436,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;214 Ingeniørarbejde (undtagen inden for elektroteknologi);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;59089,35\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;214 Ingeniørarbejde (undtagen inden for elektroteknologi);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;60121,21\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;214 Ingeniørarbejde (undtagen inden for elektroteknologi);STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;55596,35\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;214 Ingeniørarbejde (undtagen inden for elektroteknologi);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;41952,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;214 Ingeniørarbejde (undtagen inden for elektroteknologi);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;32005,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;214 Ingeniørarbejde (undtagen inden for elektroteknologi);Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;9948,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;215 Ingeniørarbejde inden for elektroteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;61460,30\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;215 Ingeniørarbejde inden for elektroteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;61754,23\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;215 Ingeniørarbejde inden for elektroteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;57512,56\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;215 Ingeniørarbejde inden for elektroteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6091,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;215 Ingeniørarbejde inden for elektroteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;5660,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;215 Ingeniørarbejde inden for elektroteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;431,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;216 Arbejde med arkitektur, infrastruktur og design;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;47625,50\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;216 Arbejde med arkitektur, infrastruktur og design;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;50462,11\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;216 Arbejde med arkitektur, infrastruktur og design;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;44792,95\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;216 Arbejde med arkitektur, infrastruktur og design;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;8525,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;216 Arbejde med arkitektur, infrastruktur og design;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4186,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;216 Arbejde med arkitektur, infrastruktur og design;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4340,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;221 Lægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;71514,26\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;221 Lægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;76407,03\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;221 Lægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;67076,11\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;221 Lægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;18158,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;221 Lægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;8329,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;221 Lægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;9829,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;222 Sygeplejerske- og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;41573,05\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;222 Sygeplejerske- og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;44312,61\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;222 Sygeplejerske- og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;41441,99\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;222 Sygeplejerske- og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;54657,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;222 Sygeplejerske- og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2376,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;222 Sygeplejerske- og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;52281,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;223 Arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;223 Arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;223 Arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;223 Arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;15,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;223 Arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;223 Arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;10,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;224 Paramedicinsk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;43745,94\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;224 Paramedicinsk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;43636,84\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;224 Paramedicinsk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;224 Paramedicinsk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;408,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;224 Paramedicinsk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;378,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;224 Paramedicinsk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;31,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;225 Dyrlægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;53898,13\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;225 Dyrlægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;59661,82\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;225 Dyrlægearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;51365,97\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;225 Dyrlægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;848,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;225 Dyrlægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;247,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;225 Dyrlægearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;601,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;226 Andet sundhedsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;45680,42\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;226 Andet sundhedsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;52366,23\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;226 Andet sundhedsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;43986,07\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;226 Andet sundhedsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;24510,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;226 Andet sundhedsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4664,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;226 Andet sundhedsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;19847,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;231 Undervisning og forskning ved universiteter og højere læreanstalter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;49152,55\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;231 Undervisning og forskning ved universiteter og højere læreanstalter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;50779,33\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;231 Undervisning og forskning ved universiteter og højere læreanstalter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;47218,94\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;231 Undervisning og forskning ved universiteter og højere læreanstalter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;26389,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;231 Undervisning og forskning ved universiteter og højere læreanstalter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;14129,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;231 Undervisning og forskning ved universiteter og højere læreanstalter;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;12260,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;232 Undervisning ved erhvervsuddannelser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;43298,06\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;232 Undervisning ved erhvervsuddannelser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;43555,57\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;232 Undervisning ved erhvervsuddannelser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;42957,67\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;232 Undervisning ved erhvervsuddannelser;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;9537,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;232 Undervisning ved erhvervsuddannelser;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;5396,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;232 Undervisning ved erhvervsuddannelser;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4141,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;233 Almen undervisning efter grundskoleniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;49941,15\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;233 Almen undervisning efter grundskoleniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;50476,72\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;233 Almen undervisning efter grundskoleniveau;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;49462,83\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;233 Almen undervisning efter grundskoleniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;14286,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;233 Almen undervisning efter grundskoleniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6665,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;233 Almen undervisning efter grundskoleniveau;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7621,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;234 Undervisning og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;40178,68\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;234 Undervisning og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;41310,32\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;234 Undervisning og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;39742,45\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;234 Undervisning og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;114735,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;234 Undervisning og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;30925,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;234 Undervisning og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;83809,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;235 Andet undervisnings- og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;40223,12\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;235 Andet undervisnings- og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;40971,27\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;235 Andet undervisnings- og pædagogisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;39864,46\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;235 Andet undervisnings- og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;38216,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;235 Andet undervisnings- og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;12006,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;235 Andet undervisnings- og pædagogisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;26211,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;241 Arbejde inden for finans og økonomi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;59531,72\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;241 Arbejde inden for finans og økonomi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;63334,77\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;241 Arbejde inden for finans og økonomi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;54624,40\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;241 Arbejde inden for finans og økonomi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;33593,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;241 Arbejde inden for finans og økonomi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;18621,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;241 Arbejde inden for finans og økonomi;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;14972,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;242 Arbejde inden for virksomhedsadministration - privat og offentlig;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;62641,97\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;242 Arbejde inden for virksomhedsadministration - privat og offentlig;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;65960,20\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;242 Arbejde inden for virksomhedsadministration - privat og offentlig;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;59619,95\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;242 Arbejde inden for virksomhedsadministration - privat og offentlig;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;18213,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;242 Arbejde inden for virksomhedsadministration - privat og offentlig;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;8554,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;242 Arbejde inden for virksomhedsadministration - privat og offentlig;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;9658,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;243 Arbejde inden for salg, marketing og PR;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;55835,10\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;243 Arbejde inden for salg, marketing og PR;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;60521,85\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;243 Arbejde inden for salg, marketing og PR;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;50319,64\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;243 Arbejde inden for salg, marketing og PR;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;17796,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;243 Arbejde inden for salg, marketing og PR;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;9482,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;243 Arbejde inden for salg, marketing og PR;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;8314,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;251 Udvikling og analyse af software og applikatoner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;58055,83\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;251 Udvikling og analyse af software og applikatoner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;58966,52\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;251 Udvikling og analyse af software og applikatoner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;54888,25\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;251 Udvikling og analyse af software og applikatoner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;44034,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;251 Udvikling og analyse af software og applikatoner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;33883,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;251 Udvikling og analyse af software og applikatoner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;10151,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;252 Arbejde med databaser og netværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;54293,11\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;252 Arbejde med databaser og netværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;55095,59\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;252 Arbejde med databaser og netværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;50627,82\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;252 Arbejde med databaser og netværk;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;7827,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;252 Arbejde med databaser og netværk;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6369,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;252 Arbejde med databaser og netværk;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1459,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;261 Juridisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;57059,73\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;261 Juridisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;60021,83\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;261 Juridisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;54943,79\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;261 Juridisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;18298,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;261 Juridisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;7395,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;261 Juridisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;10904,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;44662,68\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;45253,22\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;44420,24\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2899,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;823,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;262 Bibliotekar-, arkivar- og museumsinspektørarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2076,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;263 Arbejde inden for samfundsvidenskab og religion;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;45033,28\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;263 Arbejde inden for samfundsvidenskab og religion;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;49735,01\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;263 Arbejde inden for samfundsvidenskab og religion;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;43272,98\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;263 Arbejde inden for samfundsvidenskab og religion;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;30891,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;263 Arbejde inden for samfundsvidenskab og religion;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;8016,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;263 Arbejde inden for samfundsvidenskab og religion;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;22875,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;49053,28\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;50856,13\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;47175,09\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;8494,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4237,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;264 Forfatterarbejde og journalistisk og sprogvidenskabeligt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4257,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;265 Arbejde med kunst og kreative fag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;43589,81\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;265 Arbejde med kunst og kreative fag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;45862,68\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;265 Arbejde med kunst og kreative fag;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;41346,88\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;265 Arbejde med kunst og kreative fag;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4109,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;265 Arbejde med kunst og kreative fag;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2010,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;265 Arbejde med kunst og kreative fag;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2100,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;46727,17\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;48790,63\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;41513,06\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;44423,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;31423,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;311 Teknikerarbejde inden for fysisk videnskab og ingeniørvirksomhed;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;13000,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;313 Teknisk kontrolarbejde inden for processtyring;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;48280,48\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;313 Teknisk kontrolarbejde inden for processtyring;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;49649,41\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;313 Teknisk kontrolarbejde inden for processtyring;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;44984,94\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;313 Teknisk kontrolarbejde inden for processtyring;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2780,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;313 Teknisk kontrolarbejde inden for processtyring;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1928,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;313 Teknisk kontrolarbejde inden for processtyring;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;852,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;314 Teknikerarbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;44599,77\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;314 Teknikerarbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;45765,19\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;314 Teknikerarbejde inden for biovidenskab;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;41854,96\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;314 Teknikerarbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;760,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;314 Teknikerarbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;529,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;314 Teknikerarbejde inden for biovidenskab;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;230,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;315 Teknikerarbejde inden for skibs- og luftfart;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;72136,24\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;315 Teknikerarbejde inden for skibs- og luftfart;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;72891,06\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;315 Teknikerarbejde inden for skibs- og luftfart;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;64221,94\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;315 Teknikerarbejde inden for skibs- og luftfart;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3294,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;315 Teknikerarbejde inden for skibs- og luftfart;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3001,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;315 Teknikerarbejde inden for skibs- og luftfart;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;294,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38949,62\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;41110,87\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;38568,62\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;11526,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1682,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;321 Teknikerarbejde inden for det medicinske og farmaceutiske område;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;9844,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38840,79\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;40282,28\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;37739,47\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;352,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;152,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;322 Assisterende arbejde inden for sygepleje og jordemoderarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;200,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;33,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;323 Assisterende arbejde inden for naturmedicin og alternativ behandling;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;30,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31843,97\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31523,82\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;228,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;16,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;324 Assisterende veterinærarbejde og veterinærteknisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;212,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;40406,11\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;44769,12\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;37494,86\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;7121,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2849,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;325 Andet tekniker- og assistentarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4272,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;331 Arbejde med finans, regnskab og matematik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;48063,55\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;331 Arbejde med finans, regnskab og matematik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;54009,58\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;331 Arbejde med finans, regnskab og matematik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;44112,28\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;331 Arbejde med finans, regnskab og matematik;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;25851,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;331 Arbejde med finans, regnskab og matematik;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;10242,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;331 Arbejde med finans, regnskab og matematik;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;15609,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;332 Agent- og mæglerarbejde inden for salg og indkøb;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;49939,89\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;332 Agent- og mæglerarbejde inden for salg og indkøb;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;53266,28\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;332 Agent- og mæglerarbejde inden for salg og indkøb;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;44371,66\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;332 Agent- og mæglerarbejde inden for salg og indkøb;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;37977,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;332 Agent- og mæglerarbejde inden for salg og indkøb;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;23530,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;332 Agent- og mæglerarbejde inden for salg og indkøb;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;14448,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;333 Arbejde inden for forretningsservice;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;45866,14\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;333 Arbejde inden for forretningsservice;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;49612,03\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;333 Arbejde inden for forretningsservice;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;42256,79\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;333 Arbejde inden for forretningsservice;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;14084,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;333 Arbejde inden for forretningsservice;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6733,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;333 Arbejde inden for forretningsservice;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;7352,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;334 Administrativt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;41818,91\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;334 Administrativt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;48340,78\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;334 Administrativt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;40677,26\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;334 Administrativt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;23646,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;334 Administrativt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3526,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;334 Administrativt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;20120,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;335 Arbejde med forvaltning og udøvelse af lovgivning;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;47145,05\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;335 Arbejde med forvaltning og udøvelse af lovgivning;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;51686,88\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;335 Arbejde med forvaltning og udøvelse af lovgivning;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;44144,86\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;335 Arbejde med forvaltning og udøvelse af lovgivning;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;8588,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;335 Arbejde med forvaltning og udøvelse af lovgivning;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3388,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;335 Arbejde med forvaltning og udøvelse af lovgivning;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;5200,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;341 Arbejde inden for jura og samfundsforhold;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;42814,65\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;341 Arbejde inden for jura og samfundsforhold;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;45614,06\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;341 Arbejde inden for jura og samfundsforhold;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;41761,97\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;341 Arbejde inden for jura og samfundsforhold;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5172,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;341 Arbejde inden for jura og samfundsforhold;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1373,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;341 Arbejde inden for jura og samfundsforhold;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3799,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;342 Arbejde med sport og fitness;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37176,62\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;342 Arbejde med sport og fitness;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;40038,36\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;342 Arbejde med sport og fitness;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31273,72\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;342 Arbejde med sport og fitness;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1742,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;342 Arbejde med sport og fitness;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1176,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;342 Arbejde med sport og fitness;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;566,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38726,83\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;40604,55\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;36892,95\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4255,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2011,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;343 Arbejde inden for kunstneriske, kulturelle og kulinariske områder;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2245,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;47524,41\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;48025,16\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;45459,52\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;10161,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;8117,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;351 Driftsteknikerarbejde og brugersupportarbejde inden for informations- og kommunikationsteknologi;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2044,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;44192,33\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;44191,62\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;44195,28\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3260,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2583,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;352 Teknikerarbejde inden for audiovisuelle medier og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;677,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;411 Almindeligt kontorarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38993,93\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;411 Almindeligt kontorarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;43276,71\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;411 Almindeligt kontorarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;37504,99\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;411 Almindeligt kontorarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;69677,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;411 Almindeligt kontorarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;17206,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;411 Almindeligt kontorarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;52471,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;412 Almindeligt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;40182,62\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;412 Almindeligt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;42812,59\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;412 Almindeligt sekretærarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;39970,93\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;412 Almindeligt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6070,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;412 Almindeligt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;444,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;412 Almindeligt sekretærarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;5627,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;413 Indtastningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;40273,74\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;413 Indtastningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;42765,82\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;413 Indtastningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;38236,92\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;413 Indtastningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1727,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;413 Indtastningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;766,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;413 Indtastningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;961,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;40969,87\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;42757,20\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;40106,71\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2597,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;824,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;421 Kasserer- og inkassatorarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1773,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;422 Kundeinformationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;33652,81\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;422 Kundeinformationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;34240,61\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;422 Kundeinformationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33356,45\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;422 Kundeinformationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;12544,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;422 Kundeinformationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4129,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;422 Kundeinformationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;8415,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;431 Almindeligt beregningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;42193,83\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;431 Almindeligt beregningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;45507,39\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;431 Almindeligt beregningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;41130,11\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;431 Almindeligt beregningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;17526,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;431 Almindeligt beregningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4226,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;431 Almindeligt beregningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;13300,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;432 Kontorarbejde inden for materialeregistrering og transport;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;34665,44\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;432 Kontorarbejde inden for materialeregistrering og transport;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;35069,73\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;432 Kontorarbejde inden for materialeregistrering og transport;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33498,47\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;432 Kontorarbejde inden for materialeregistrering og transport;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;20218,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;432 Kontorarbejde inden for materialeregistrering og transport;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;14913,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;432 Kontorarbejde inden for materialeregistrering og transport;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;5305,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;441 Andet almindeligt kontor- og kundeservicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36701,27\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;441 Andet almindeligt kontor- og kundeservicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;36578,88\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;441 Andet almindeligt kontor- og kundeservicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;36810,21\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;441 Andet almindeligt kontor- og kundeservicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;18946,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;441 Andet almindeligt kontor- og kundeservicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;8799,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;441 Andet almindeligt kontor- og kundeservicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;10147,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;511 Service- og kontrolarbejde under transport og rejser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;39830,57\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;511 Service- og kontrolarbejde under transport og rejser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;41254,50\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;511 Service- og kontrolarbejde under transport og rejser;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;38654,02\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;511 Service- og kontrolarbejde under transport og rejser;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4419,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;511 Service- og kontrolarbejde under transport og rejser;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1944,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;511 Service- og kontrolarbejde under transport og rejser;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2476,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;512 Kokkearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;30878,98\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;512 Kokkearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;31063,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;512 Kokkearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;30589,71\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;512 Kokkearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6135,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;512 Kokkearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3598,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;512 Kokkearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2537,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;513 Tjenere og bartendere;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;26673,12\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;513 Tjenere og bartendere;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;27813,25\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;513 Tjenere og bartendere;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;25765,88\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;513 Tjenere og bartendere;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5418,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;513 Tjenere og bartendere;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2412,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;513 Tjenere og bartendere;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3006,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;33403,58\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;33539,91\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33385,15\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;701,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;82,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;514 Frisørarbejde og kosmetologarbejde samt beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;620,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;35410,63\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;35589,05\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;34216,53\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;16503,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;14332,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;515 Inspektørarbejde inden for rengøring, husholdning og ejendomme;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2171,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;516 Andet servicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;30909,29\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;516 Andet servicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;31856,43\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;516 Andet servicearbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29245,56\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;516 Andet servicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3718,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;516 Andet servicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2326,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;516 Andet servicearbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1392,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;521 Salgsarbejde på gade og marked;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;521 Salgsarbejde på gade og marked;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;521 Salgsarbejde på gade og marked;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;521 Salgsarbejde på gade og marked;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;45,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;521 Salgsarbejde på gade og marked;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;26,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;521 Salgsarbejde på gade og marked;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;18,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;522 Salgsarbejde i butik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;28773,34\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;522 Salgsarbejde i butik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;31072,58\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;522 Salgsarbejde i butik;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;26800,71\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;522 Salgsarbejde i butik;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;33645,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;522 Salgsarbejde i butik;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;15301,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;522 Salgsarbejde i butik;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;18344,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;523 Kasseassistentarbejde og beslægtet kundebetjening;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;25740,24\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;523 Kasseassistentarbejde og beslægtet kundebetjening;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;25920,63\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;523 Kasseassistentarbejde og beslægtet kundebetjening;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;25655,90\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;523 Kasseassistentarbejde og beslægtet kundebetjening;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;8810,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;523 Kasseassistentarbejde og beslægtet kundebetjening;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2761,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;523 Kasseassistentarbejde og beslægtet kundebetjening;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;6049,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;524 Andet salgsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37196,16\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;524 Andet salgsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;39867,10\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;524 Andet salgsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33594,79\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;524 Andet salgsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;15036,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;524 Andet salgsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;8563,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;524 Andet salgsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;6473,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;531 Børneomsorgsarbejde og hjælpelærerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;28940,98\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;531 Børneomsorgsarbejde og hjælpelærerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;28748,67\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;531 Børneomsorgsarbejde og hjælpelærerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28988,23\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;531 Børneomsorgsarbejde og hjælpelærerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;35407,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;531 Børneomsorgsarbejde og hjælpelærerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6798,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;531 Børneomsorgsarbejde og hjælpelærerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;28609,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;532 Omsorgsarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;34071,49\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;532 Omsorgsarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;33442,83\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;532 Omsorgsarbejde inden for sundhedsområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;34184,72\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;532 Omsorgsarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;95043,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;532 Omsorgsarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;14002,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;532 Omsorgsarbejde inden for sundhedsområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;81041,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;541 Rednings- og overvågningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;39201,23\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;541 Rednings- og overvågningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;39732,51\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;541 Rednings- og overvågningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;37048,82\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;541 Rednings- og overvågningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;20614,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;541 Rednings- og overvågningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;16460,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;541 Rednings- og overvågningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4155,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;611 Arbejde med plantevækst;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32835,53\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;611 Arbejde med plantevækst;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;33329,45\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;611 Arbejde med plantevækst;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31291,26\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;611 Arbejde med plantevækst;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3176,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;611 Arbejde med plantevækst;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2342,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;611 Arbejde med plantevækst;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;835,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;612 Arbejde med dyreopdræt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;34162,86\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;612 Arbejde med dyreopdræt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;34791,23\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;612 Arbejde med dyreopdræt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33694,03\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;612 Arbejde med dyreopdræt;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;379,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;612 Arbejde med dyreopdræt;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;159,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;612 Arbejde med dyreopdræt;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;220,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;30194,41\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;30070,81\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;149,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;119,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;613 Arbejde med dyrkning af plantevækst og opdræt af dyr;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;30,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;621 Skovbrugsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;34804,72\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;621 Skovbrugsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;34624,51\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;621 Skovbrugsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;621 Skovbrugsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;271,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;621 Skovbrugsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;242,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;621 Skovbrugsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;29,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;622 Arbejde med fiskeri og jagt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;622 Arbejde med fiskeri og jagt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;622 Arbejde med fiskeri og jagt;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;622 Arbejde med fiskeri og jagt;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;23,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;622 Arbejde med fiskeri og jagt;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;19,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;622 Arbejde med fiskeri og jagt;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;711 Bygningsarbejde (ekskl. medhjælpere) - basis;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36996,38\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;711 Bygningsarbejde (ekskl. medhjælpere) - basis;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;37047,62\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;711 Bygningsarbejde (ekskl. medhjælpere) - basis;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32829,83\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;711 Bygningsarbejde (ekskl. medhjælpere) - basis;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;22195,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;711 Bygningsarbejde (ekskl. medhjælpere) - basis;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;21928,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;711 Bygningsarbejde (ekskl. medhjælpere) - basis;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;267,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;712 Bygningsarbejde (ekskl. medhjælpere) - finish;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;40335,37\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;712 Bygningsarbejde (ekskl. medhjælpere) - finish;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;40410,88\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;712 Bygningsarbejde (ekskl. medhjælpere) - finish;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31601,49\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;712 Bygningsarbejde (ekskl. medhjælpere) - finish;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;10091,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;712 Bygningsarbejde (ekskl. medhjælpere) - finish;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;10003,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;712 Bygningsarbejde (ekskl. medhjælpere) - finish;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;88,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;713 Malere og arbejde inden for rensning af bygninger;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;34089,34\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;713 Malere og arbejde inden for rensning af bygninger;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;34654,55\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;713 Malere og arbejde inden for rensning af bygninger;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31884,46\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;713 Malere og arbejde inden for rensning af bygninger;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5815,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;713 Malere og arbejde inden for rensning af bygninger;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4589,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;713 Malere og arbejde inden for rensning af bygninger;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1226,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;721 Pladearbejde, svejsning og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;36422,11\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;721 Pladearbejde, svejsning og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;36498,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;721 Pladearbejde, svejsning og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32550,28\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;721 Pladearbejde, svejsning og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;8965,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;721 Pladearbejde, svejsning og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;8789,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;721 Pladearbejde, svejsning og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;176,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;722 Smede, værktøjsmagere og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;37913,36\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;722 Smede, værktøjsmagere og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;38094,35\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;722 Smede, værktøjsmagere og beslægtede funktioner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33761,90\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;722 Smede, værktøjsmagere og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;21819,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;722 Smede, værktøjsmagere og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;20882,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;722 Smede, værktøjsmagere og beslægtede funktioner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;937,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;723 Mekanikerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38080,62\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;723 Mekanikerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;38110,39\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;723 Mekanikerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;36244,02\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;723 Mekanikerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;15155,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;723 Mekanikerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;14904,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;723 Mekanikerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;251,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;731 Præcisionshåndværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;33827,46\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;731 Præcisionshåndværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;36773,44\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;731 Præcisionshåndværk;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29573,06\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;731 Præcisionshåndværk;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1128,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;731 Præcisionshåndværk;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;666,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;731 Præcisionshåndværk;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;462,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;732 Grafisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38776,83\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;732 Grafisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;40503,35\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;732 Grafisk arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;34526,52\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;732 Grafisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2681,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;732 Grafisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1911,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;732 Grafisk arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;770,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;741 Installation- og reparation af elektrisk udstyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;40030,82\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;741 Installation- og reparation af elektrisk udstyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;40088,98\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;741 Installation- og reparation af elektrisk udstyr;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;36057,81\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;741 Installation- og reparation af elektrisk udstyr;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;17239,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;741 Installation- og reparation af elektrisk udstyr;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;16983,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;741 Installation- og reparation af elektrisk udstyr;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;256,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;742 Installation og reparation af elektronik og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;41952,26\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;742 Installation og reparation af elektronik og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;42374,70\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;742 Installation og reparation af elektronik og telekommunikation;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;34417,57\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;742 Installation og reparation af elektronik og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1719,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;742 Installation og reparation af elektronik og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1624,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;742 Installation og reparation af elektronik og telekommunikation;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;96,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;751 Arbejde inden for nærings- og nydelsesmiddelområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;34640,28\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;751 Arbejde inden for nærings- og nydelsesmiddelområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;35915,77\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;751 Arbejde inden for nærings- og nydelsesmiddelområdet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31507,56\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;751 Arbejde inden for nærings- og nydelsesmiddelområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;7864,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;751 Arbejde inden for nærings- og nydelsesmiddelområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;5554,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;751 Arbejde inden for nærings- og nydelsesmiddelområdet;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2309,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;752 Håndværkspræget arbejde i træindustri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;30830,49\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;752 Håndværkspræget arbejde i træindustri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;30994,23\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;752 Håndværkspræget arbejde i træindustri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29167,09\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;752 Håndværkspræget arbejde i træindustri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1697,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;752 Håndværkspræget arbejde i træindustri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1542,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;752 Håndværkspræget arbejde i træindustri;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;155,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;753 Tekstil- og beklædningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32811,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;753 Tekstil- og beklædningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32957,26\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;753 Tekstil- og beklædningsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32702,60\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;753 Tekstil- og beklædningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;565,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;753 Tekstil- og beklædningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;237,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;753 Tekstil- og beklædningsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;328,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;754 Andet håndværkspræget arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;39324,27\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;754 Andet håndværkspræget arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;39551,62\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;754 Andet håndværkspræget arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;38838,36\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;754 Andet håndværkspræget arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2224,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;754 Andet håndværkspræget arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1493,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;754 Andet håndværkspræget arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;731,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;39186,06\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;39302,99\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;35863,46\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1883,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1817,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;811 Operatørarbejde ved anlæg i minedrift og råstofudvinding;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;66,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;33722,58\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;34022,11\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32261,54\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5601,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4627,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;812 Operatørarbejde ved forarbejdning og færdiggørelse af metal;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;974,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;44951,47\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;45619,59\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;42751,97\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3485,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2650,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;813 Operatørarbejde ved fremstilling af kemiske og fotografiske produkter;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;835,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;35033,72\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;35717,19\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;33168,86\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;6319,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4628,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;814 Operatører af maskiner til fremstilling af gummi-, plastik- og papirprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1691,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;33899,04\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;37148,30\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29053,32\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1098,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;649,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;815 Operatørarbejde ved fremstilling af tekstil-, pels- og læderprodukter;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;449,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;35025,73\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;36045,41\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31914,08\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;12834,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;9664,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;816 Operatørarbejde ved fremstilling af nærings- og nydelsesmidler;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3170,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;817 Operatørarbejde ved træforarbejdning og papirfremstilling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32079,16\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;817 Operatørarbejde ved træforarbejdning og papirfremstilling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32428,25\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;817 Operatørarbejde ved træforarbejdning og papirfremstilling;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;30455,10\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;817 Operatørarbejde ved træforarbejdning og papirfremstilling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3676,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;817 Operatørarbejde ved træforarbejdning og papirfremstilling;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3014,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;817 Operatørarbejde ved træforarbejdning og papirfremstilling;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;662,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;818 Andet operatørarbejde ved stationære anlæg og maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;38915,94\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;818 Andet operatørarbejde ved stationære anlæg og maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;39966,17\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;818 Andet operatørarbejde ved stationære anlæg og maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;34567,61\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;818 Andet operatørarbejde ved stationære anlæg og maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;8584,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;818 Andet operatørarbejde ved stationære anlæg og maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;6861,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;818 Andet operatørarbejde ved stationære anlæg og maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1723,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;821 Monteringsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;34577,79\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;821 Monteringsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;36311,04\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;821 Monteringsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31070,48\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;821 Monteringsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;11670,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;821 Monteringsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;7714,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;821 Monteringsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;3956,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;831 Lokomotivførerarbejde og tograngering;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;45371,33\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;831 Lokomotivførerarbejde og tograngering;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;45529,55\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;831 Lokomotivførerarbejde og tograngering;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;42446,85\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;831 Lokomotivførerarbejde og tograngering;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2770,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;831 Lokomotivførerarbejde og tograngering;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2628,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;831 Lokomotivførerarbejde og tograngering;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;143,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;832 Chauffører af biler, varevogne og motorcykler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;28956,67\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;832 Chauffører af biler, varevogne og motorcykler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;29060,13\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;832 Chauffører af biler, varevogne og motorcykler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;27381,39\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;832 Chauffører af biler, varevogne og motorcykler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;3663,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;832 Chauffører af biler, varevogne og motorcykler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3437,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;832 Chauffører af biler, varevogne og motorcykler;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;227,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;833 Chauffører af busser og lastbiler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32308,41\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;833 Chauffører af busser og lastbiler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32408,56\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;833 Chauffører af busser og lastbiler;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;30664,73\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;833 Chauffører af busser og lastbiler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;23781,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;833 Chauffører af busser og lastbiler;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;22345,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;833 Chauffører af busser og lastbiler;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1435,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;834 Førere af mobile maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;35016,27\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;834 Førere af mobile maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;35105,98\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;834 Førere af mobile maskiner;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;32372,05\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;834 Førere af mobile maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5735,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;834 Førere af mobile maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;5542,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;834 Førere af mobile maskiner;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;193,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;835 Dæksarbejde og medhjælp om bord på skibe;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;40836,20\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;835 Dæksarbejde og medhjælp om bord på skibe;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;41081,76\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;835 Dæksarbejde og medhjælp om bord på skibe;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;835 Dæksarbejde og medhjælp om bord på skibe;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;285,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;835 Dæksarbejde og medhjælp om bord på skibe;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;275,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;835 Dæksarbejde og medhjælp om bord på skibe;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;11,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;911 Rengøringsarbejde i private hjem, hoteller og kontorer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;28253,88\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;911 Rengøringsarbejde i private hjem, hoteller og kontorer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;29522,44\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;911 Rengøringsarbejde i private hjem, hoteller og kontorer;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;27636,43\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;911 Rengøringsarbejde i private hjem, hoteller og kontorer;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;36535,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;911 Rengøringsarbejde i private hjem, hoteller og kontorer;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;11467,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;911 Rengøringsarbejde i private hjem, hoteller og kontorer;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;25068,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;28920,47\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;30138,60\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;26962,95\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;4281,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;2603,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;912 Manuelt rengøringsarbejde af køretøjer, vinduer, vasketøj og andet;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1678,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;29722,68\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;29709,15\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29756,24\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;1095,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;756,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;921 Manuelt arbejde inden for landbrug, skovbrug og fiskeri;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;339,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;34860,31\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;35150,98\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31237,33\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;29825,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;27473,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;931 Manuelt arbejde inden for råstofudvinding og bygge- og anlægssektoren;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;2351,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;932 Manuelt produktionsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;31177,22\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;932 Manuelt produktionsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32040,21\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;932 Manuelt produktionsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29227,60\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;932 Manuelt produktionsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;14183,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;932 Manuelt produktionsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;9777,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;932 Manuelt produktionsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4406,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;933 Transport- og lagerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32032,69\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;933 Transport- og lagerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32659,10\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;933 Transport- og lagerarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;28746,08\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;933 Transport- og lagerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;25801,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;933 Transport- og lagerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;21605,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;933 Transport- og lagerarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4196,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;941 Manuelt arbejde med tilberedning af mad;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;27290,78\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;941 Manuelt arbejde med tilberedning af mad;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;25588,36\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;941 Manuelt arbejde med tilberedning af mad;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;27935,76\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;941 Manuelt arbejde med tilberedning af mad;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;13618,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;941 Manuelt arbejde med tilberedning af mad;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3270,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;941 Manuelt arbejde med tilberedning af mad;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;10347,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;951 Servicearbejde på gaden;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;32133,93\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;951 Servicearbejde på gaden;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;32279,54\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;951 Servicearbejde på gaden;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;951 Servicearbejde på gaden;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;72,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;951 Servicearbejde på gaden;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;68,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;951 Servicearbejde på gaden;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;4,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;952 Gadesalg (undtagen madvarer);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;952 Gadesalg (undtagen madvarer);STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;952 Gadesalg (undtagen madvarer);STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;..\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;952 Gadesalg (undtagen madvarer);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;8,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;952 Gadesalg (undtagen madvarer);Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;3,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;952 Gadesalg (undtagen madvarer);Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;5,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;961 Renovationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;34710,12\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;961 Renovationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;34991,36\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;961 Renovationsarbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;31514,71\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;961 Renovationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;2151,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;961 Renovationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;1971,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;961 Renovationsarbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;180,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;962 Andet manuelt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd og kvinder i alt;34217,52\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;962 Andet manuelt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Mænd;35471,52\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;962 Andet manuelt arbejde;STANDARDBEREGNET MÅNEDSFORTJENESTE;Kvinder;29787,23\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;962 Andet manuelt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd og kvinder i alt;5381,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;962 Andet manuelt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Mænd;4200,00\nLønmodtagergrupper (eksklusiv unge og elever);Sektorer i alt;2019;962 Andet manuelt arbejde;Antal fuldtidsbeskæftigede i lønstatistikken;Kvinder;1181,00\n"