/*
 * Copyright 2010 BetaSteward_at_googlemail.com. All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without modification, are
 * permitted provided that the following conditions are met:
 *
 *    1. Redistributions of source code must retain the above copyright notice, this list of
 *       conditions and the following disclaimer.
 *
 *    2. Redistributions in binary form must reproduce the above copyright notice, this list
 *       of conditions and the following disclaimer in the documentation and/or other materials
 *       provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY BetaSteward_at_googlemail.com ``AS IS'' AND ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL BetaSteward_at_googlemail.com OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * The views and conclusions contained in the software and documentation are those of the
 * authors and should not be interpreted as representing official policies, either expressed
 * or implied, of BetaSteward_at_googlemail.com.
 */
package mage.client.dialog;

import java.io.FileNotFoundException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import javax.swing.*;
import mage.cards.decks.importer.DeckImporterUtil;
import mage.client.MageFrame;
import mage.client.SessionHandler;
import mage.client.components.MageComponents;
import mage.client.table.TablePlayerPanel;
import mage.client.util.Event;
import mage.client.util.IgnoreList;
import mage.client.util.Listener;
import mage.constants.MatchTimeLimit;
import mage.constants.MultiplayerAttackOption;
import mage.constants.RangeOfInfluence;
import mage.constants.SkillLevel;
import mage.game.match.MatchOptions;
import mage.players.PlayerType;
import mage.view.GameTypeView;
import mage.view.TableView;
import org.apache.log4j.Logger;

/**
 * @author BetaSteward_at_googlemail.com
 */
public class NewTableDialog extends MageDialog {

    private static final Logger logger = Logger.getLogger(NewTableDialog.class);

    private TableView table;
    private UUID playerId;
    private UUID roomId;
    private String lastSessionId;
    private final List<TablePlayerPanel> players = new ArrayList<>();
    private final List<PlayerType> prefPlayerTypes = new ArrayList<>();

    private static final String LIMITED = "Limited";

    /**
     * Creates new form NewTableDialog
     */
    public NewTableDialog() {
        lastSessionId = "";
        initComponents();
        player1Panel.showLevel(false);
        this.spnNumWins.setModel(new SpinnerNumberModel(1, 1, 5, 1));
        this.spnFreeMulligans.setModel(new SpinnerNumberModel(0, 0, 5, 1));
        this.spnQuitRatio.setModel(new SpinnerNumberModel(100, 0, 100, 5));
        this.spnEdhPowerLevel.setModel(new SpinnerNumberModel(100, 0, 100, 5));
        MageFrame.getUI().addButton(MageComponents.NEW_TABLE_OK_BUTTON, btnOK);
    }

    /**
     * This method is called from within the constructor to initialize the form.
     * WARNING: Do NOT modify this code. The content of this method is always
     * regenerated by the Form Editor.
     */
    @SuppressWarnings("unchecked")
    // <editor-fold defaultstate="collapsed" desc="Generated Code">//GEN-BEGIN:initComponents
    private void initComponents() {

        lblName = new javax.swing.JLabel();
        txtName = new javax.swing.JTextField();
        lblPassword = new javax.swing.JLabel();
        txtPassword = new javax.swing.JTextField();
        lbDeckType = new javax.swing.JLabel();
        cbDeckType = new javax.swing.JComboBox();
        lbTimeLimit = new javax.swing.JLabel();
        cbTimeLimit = new javax.swing.JComboBox();
        lblGameType = new javax.swing.JLabel();
        cbGameType = new javax.swing.JComboBox();
        chkRollbackTurnsAllowed = new javax.swing.JCheckBox();
        chkSpectatorsAllowed = new javax.swing.JCheckBox();
        chkPlaneChase = new javax.swing.JCheckBox();
        chkRated = new javax.swing.JCheckBox();
        lblFreeMulligans = new javax.swing.JLabel();
        spnFreeMulligans = new javax.swing.JSpinner();
        lblNumPlayers = new javax.swing.JLabel();
        spnNumPlayers = new javax.swing.JSpinner();
        lblRange = new javax.swing.JLabel();
        cbRange = new javax.swing.JComboBox();
        lblAttack = new javax.swing.JLabel();
        cbAttackOption = new javax.swing.JComboBox();
        lblSkillLevel = new javax.swing.JLabel();
        cbSkillLevel = new javax.swing.JComboBox();
        lblNumWins = new javax.swing.JLabel();
        spnNumWins = new javax.swing.JSpinner();
        jSeparator2 = new javax.swing.JSeparator();
        jLabel1 = new javax.swing.JLabel();
        player1Panel = new mage.client.table.NewPlayerPanel();
        jSeparator3 = new javax.swing.JSeparator();
        jLabel2 = new javax.swing.JLabel();
        pnlOtherPlayers = new javax.swing.JPanel();
        jSeparator1 = new javax.swing.JSeparator();
        btnOK = new javax.swing.JButton();
        btnPreviousConfiguration1 = new javax.swing.JButton();
        btnPreviousConfiguration2 = new javax.swing.JButton();
        btnCancel = new javax.swing.JButton();
        lblQuitRatio = new javax.swing.JLabel();
        lblEdhPowerLevel = new javax.swing.JLabel();
        spnQuitRatio = new javax.swing.JSpinner();
        spnEdhPowerLevel = new javax.swing.JSpinner();

        setTitle("New Table");

        lblName.setLabelFor(txtName);
        lblName.setText("Name:");

        lblPassword.setLabelFor(txtName);
        lblPassword.setText("Password:");

        lbDeckType.setText("Deck Type:");

        lbTimeLimit.setText("Time Limit:");
        lbTimeLimit.setToolTipText("The active time a player may use to finish the match. If their time runs out, the player looses the current game.");

        lblGameType.setText("Game Type:");

        cbGameType.addActionListener(evt -> cbGameTypeActionPerformed(evt));

        chkRollbackTurnsAllowed.setText("Allow rollbacks");
        chkRollbackTurnsAllowed.setToolTipText("<HTML>Allow to rollback to the start of previous turns<br>\nif all players agree.\n");

        chkSpectatorsAllowed.setText("Allow Spectators");
        chkSpectatorsAllowed.setToolTipText("<HTML>Allow spectators to watch.\n");

        chkPlaneChase.setText("Use PlaneChase");
        chkPlaneChase.setToolTipText("<HTML>Use planechase variant (suitable for all game types).\n");

        chkRated.setText("Rated");
        chkRated.setToolTipText("Indicates if matches will be rated.");

        lblFreeMulligans.setText("Free Mulligans:");
        lblFreeMulligans.setToolTipText("The number of mulligans a player can use without decreasing the number of drawn cards.");

        lblNumPlayers.setLabelFor(spnNumPlayers);
        lblNumPlayers.setText("Players");

        spnNumPlayers.addChangeListener(evt -> numPlayersChanged(evt));

        lblRange.setLabelFor(cbRange);
        lblRange.setText("Range of Influence");

        cbRange.setToolTipText("<HTML>An option for multiplayer games.\nA player's range of influence is the maximum distance from that player, measured in player seats,<br>\nthat the player can affect. Players within that many seats of the player are within that player's range<br>\nof influence. Objects controlled by players within a player's range of influence are also within that<br>\nplayer's range of influence. Range of influence covers spells, abilities, effects, damage dealing, attacking,<nr>\nmaking choices, and winning the game.");

        lblAttack.setLabelFor(cbAttackOption);
        lblAttack.setText("Attack Option");

        cbAttackOption.setToolTipText("<HTML>An option for multiplayer games that defines<br>\nwhich opponents can be attacked from a player.");

        lblSkillLevel.setLabelFor(cbAttackOption);
        lblSkillLevel.setText("Skill Level");

        cbSkillLevel.setToolTipText("<HTML>This option can be used to make it easier to find matches<br>\nwith opponents of the appropriate skill level.");

        lblNumWins.setLabelFor(spnNumWins);
        lblNumWins.setText("Wins");
        lblNumWins.setToolTipText("How many games has a player to win to win the match.");

        spnNumWins.addChangeListener(evt -> spnNumWinsnumPlayersChanged(evt));

        jLabel1.setFont(new java.awt.Font("Tahoma", 1, 11)); // NOI18N
        jLabel1.setText("Player 1 (You)");

        jLabel2.setFont(new java.awt.Font("Tahoma", 1, 11)); // NOI18N
        jLabel2.setText("Other Players");

        pnlOtherPlayers.setLayout(new java.awt.GridLayout(0, 1));

        btnOK.setText("OK");
        btnOK.addActionListener(evt -> btnOKActionPerformed(evt));

        btnPreviousConfiguration1.setText("M1");
        btnPreviousConfiguration1.setToolTipText("Load saved Match configuration #1");
        btnPreviousConfiguration1.addActionListener(evt -> btnPreviousConfigurationActionPerformed(evt, 1));
        btnPreviousConfiguration2.setText("M2");
        btnPreviousConfiguration2.setToolTipText("Load saved Match configuration #2");
        btnPreviousConfiguration2.addActionListener(evt -> btnPreviousConfigurationActionPerformed(evt, 2));

        btnCancel.setText("Cancel");
        btnCancel.addActionListener(evt -> btnCancelActionPerformed(evt));

        lblQuitRatio.setText("Allowed quit %");
        lblEdhPowerLevel.setText("EDH power level");

        spnQuitRatio.setToolTipText("Players with quit % more than this value can't join this table");
        spnEdhPowerLevel.setToolTipText("Players with decks with a higher power level can't join this table");

        javax.swing.GroupLayout layout = new javax.swing.GroupLayout(getContentPane());
        getContentPane().setLayout(layout);
        layout.setHorizontalGroup(
                layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                        .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, layout.createSequentialGroup()
                                .addContainerGap()
                                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.TRAILING)
                                        .addGroup(javax.swing.GroupLayout.Alignment.LEADING, layout.createSequentialGroup()
                                                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                                                        .addComponent(lblName)
                                                        .addComponent(lbDeckType)
                                                        .addComponent(lblGameType))
                                                .addGap(6, 6, 6)
                                                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING, false)
                                                        .addGroup(layout.createSequentialGroup()
                                                                .addComponent(cbGameType, javax.swing.GroupLayout.PREFERRED_SIZE, 270, javax.swing.GroupLayout.PREFERRED_SIZE)
                                                                .addGap(18, 18, 18)
                                                                .addComponent(chkRollbackTurnsAllowed)
                                                                .addGap(13, 13, 13)
                                                                .addComponent(lblFreeMulligans)
                                                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                                                .addComponent(spnFreeMulligans, javax.swing.GroupLayout.PREFERRED_SIZE, 50, javax.swing.GroupLayout.PREFERRED_SIZE)
                                                                .addGap(13, 13, 13)
                                                                .addComponent(chkSpectatorsAllowed)
                                                                .addGap(13, 13, 13)
                                                                .addComponent(chkPlaneChase))
                                                        .addGroup(layout.createSequentialGroup()
                                                                .addComponent(txtName, javax.swing.GroupLayout.PREFERRED_SIZE, 178, javax.swing.GroupLayout.PREFERRED_SIZE)
                                                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                                                .addComponent(lbTimeLimit)
                                                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                                                .addComponent(cbTimeLimit, javax.swing.GroupLayout.PREFERRED_SIZE, 102, javax.swing.GroupLayout.PREFERRED_SIZE)
                                                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                                                .addComponent(lblPassword)
                                                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                                                .addComponent(txtPassword, javax.swing.GroupLayout.PREFERRED_SIZE, 125, javax.swing.GroupLayout.PREFERRED_SIZE)
                                                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                                                .addComponent(btnPreviousConfiguration1, javax.swing.GroupLayout.PREFERRED_SIZE, 50, javax.swing.GroupLayout.PREFERRED_SIZE)
                                                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                                                .addComponent(btnPreviousConfiguration2, javax.swing.GroupLayout.PREFERRED_SIZE, 50, javax.swing.GroupLayout.PREFERRED_SIZE))
                                                        .addGroup(layout.createSequentialGroup()
                                                                .addComponent(cbDeckType, javax.swing.GroupLayout.PREFERRED_SIZE, 332, javax.swing.GroupLayout.PREFERRED_SIZE)
                                                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                                                                .addComponent(chkRated)
                                                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.UNRELATED)
                                                                .addComponent(lblQuitRatio)
                                                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                                                .addComponent(spnQuitRatio, javax.swing.GroupLayout.PREFERRED_SIZE, 60, javax.swing.GroupLayout.PREFERRED_SIZE)
                                                                .addComponent(lblEdhPowerLevel)
                                                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                                                .addComponent(spnEdhPowerLevel, javax.swing.GroupLayout.PREFERRED_SIZE, 60, javax.swing.GroupLayout.PREFERRED_SIZE))))
                                        .addComponent(jLabel1, javax.swing.GroupLayout.Alignment.LEADING)
                                        .addComponent(jLabel2, javax.swing.GroupLayout.Alignment.LEADING)
                                        .addGroup(layout.createSequentialGroup()
                                                .addGap(0, 0, Short.MAX_VALUE)
                                                .addComponent(btnOK)
                                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                                .addComponent(btnCancel))
                                        .addGroup(javax.swing.GroupLayout.Alignment.LEADING, layout.createSequentialGroup()
                                                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                                                        .addComponent(lblNumPlayers)
                                                        .addComponent(spnNumPlayers, javax.swing.GroupLayout.PREFERRED_SIZE, 57, javax.swing.GroupLayout.PREFERRED_SIZE))
                                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                                                        .addComponent(lblRange)
                                                        .addComponent(cbRange, javax.swing.GroupLayout.PREFERRED_SIZE, 117, javax.swing.GroupLayout.PREFERRED_SIZE))
                                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                                                        .addGroup(layout.createSequentialGroup()
                                                                .addComponent(lblAttack)
                                                                .addGap(116, 116, 116)
                                                                .addComponent(lblSkillLevel))
                                                        .addGroup(layout.createSequentialGroup()
                                                                .addComponent(cbAttackOption, javax.swing.GroupLayout.PREFERRED_SIZE, 177, javax.swing.GroupLayout.PREFERRED_SIZE)
                                                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                                                .addComponent(cbSkillLevel, javax.swing.GroupLayout.PREFERRED_SIZE, 148, javax.swing.GroupLayout.PREFERRED_SIZE)))
                                                .addGap(18, 18, 18)
                                                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                                                        .addComponent(lblNumWins)
                                                        .addComponent(spnNumWins, javax.swing.GroupLayout.PREFERRED_SIZE, 50, javax.swing.GroupLayout.PREFERRED_SIZE)))
                                        .addComponent(jSeparator2)
                                        .addComponent(player1Panel, javax.swing.GroupLayout.Alignment.LEADING, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                                        .addComponent(pnlOtherPlayers, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, Short.MAX_VALUE)
                                        .addComponent(jSeparator1, javax.swing.GroupLayout.Alignment.LEADING))
                                .addContainerGap())
                        .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                                .addGroup(layout.createSequentialGroup()
                                        .addContainerGap()
                                        .addComponent(jSeparator3, javax.swing.GroupLayout.DEFAULT_SIZE, 660, Short.MAX_VALUE)
                                        .addContainerGap()))
        );
        layout.setVerticalGroup(
                layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                        .addGroup(layout.createSequentialGroup()
                                .addGap(4, 4, 4)
                                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                                        .addComponent(txtName, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                                        .addComponent(lblName)
                                        .addComponent(txtPassword, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                                        .addComponent(btnPreviousConfiguration1, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                                        .addComponent(btnPreviousConfiguration2, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                                        .addComponent(lblPassword)
                                        .addComponent(lbTimeLimit)
                                        .addComponent(cbTimeLimit, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                                        .addComponent(cbDeckType, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                                        .addComponent(lbDeckType)
                                        .addComponent(lblQuitRatio)
                                        .addComponent(chkRated)
                                        .addComponent(spnQuitRatio, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                                        .addComponent(lblEdhPowerLevel)
                                        .addComponent(chkRated)
                                        .addComponent(spnEdhPowerLevel, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                                        .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                                                .addComponent(spnFreeMulligans, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                                                .addComponent(lblFreeMulligans)
                                                .addComponent(chkRollbackTurnsAllowed)
                                                .addComponent(chkSpectatorsAllowed)
                                                .addComponent(chkPlaneChase))
                                        .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                                                .addComponent(cbGameType, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                                                .addComponent(lblGameType)))
                                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                                        .addGroup(layout.createSequentialGroup()
                                                .addGap(6, 6, 6)
                                                .addComponent(lblNumPlayers)
                                                .addGap(0, 0, 0)
                                                .addComponent(spnNumPlayers, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))
                                        .addGroup(javax.swing.GroupLayout.Alignment.TRAILING, layout.createSequentialGroup()
                                                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                                                        .addComponent(lblSkillLevel)
                                                        .addComponent(lblNumWins)
                                                        .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                                                                .addComponent(lblRange)
                                                                .addComponent(lblAttack)))
                                                .addGap(0, 0, 0)
                                                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                                                        .addComponent(cbRange, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                                                        .addComponent(cbAttackOption, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                                                        .addComponent(cbSkillLevel, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                                                        .addComponent(spnNumWins, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE))))
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(jSeparator2, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(jLabel1)
                                .addGap(0, 0, 0)
                                .addComponent(player1Panel, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                                .addGap(16, 16, 16)
                                .addComponent(jLabel2)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addComponent(pnlOtherPlayers, javax.swing.GroupLayout.DEFAULT_SIZE, 105, Short.MAX_VALUE)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED, 7, Short.MAX_VALUE)
                                .addComponent(jSeparator1, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                                .addPreferredGap(javax.swing.LayoutStyle.ComponentPlacement.RELATED)
                                .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.BASELINE)
                                        .addComponent(btnCancel)
                                        .addComponent(btnOK))
                                .addGap(0, 0, 0))
                        .addGroup(layout.createParallelGroup(javax.swing.GroupLayout.Alignment.LEADING)
                                .addGroup(layout.createSequentialGroup()
                                        .addGap(201, 201, 201)
                                        .addComponent(jSeparator3, javax.swing.GroupLayout.PREFERRED_SIZE, javax.swing.GroupLayout.DEFAULT_SIZE, javax.swing.GroupLayout.PREFERRED_SIZE)
                                        .addContainerGap(167, Short.MAX_VALUE)))
        );

        pack();
    }// </editor-fold>//GEN-END:initComponents

    private void btnCancelActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_btnCancelActionPerformed
        this.table = null;
        this.playerId = null;
        this.hideDialog();
    }//GEN-LAST:event_btnCancelActionPerformed

    private void btnPreviousConfigurationActionPerformed(java.awt.event.ActionEvent evt, int i) {//GEN-FIRST:event_btnPreviousConfigurationActionPerformed
        currentSettingVersion = i;
        setGameSettingsFromPrefs(currentSettingVersion);
    }//GEN-LAST:event_btnPreviousConfigurationActionPerformed

    private void btnOKActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_btnOKActionPerformed
        GameTypeView gameType = (GameTypeView) cbGameType.getSelectedItem();
        MatchOptions options = new MatchOptions(this.txtName.getText(), gameType.getName(), false, 2);
        options.getPlayerTypes().add(PlayerType.HUMAN);
        for (TablePlayerPanel player : players) {
            options.getPlayerTypes().add(player.getPlayerType());
        }
        options.setDeckType((String) this.cbDeckType.getSelectedItem());
        options.setLimited(false);
        options.setMatchTimeLimit((MatchTimeLimit) this.cbTimeLimit.getSelectedItem());
        options.setAttackOption((MultiplayerAttackOption) this.cbAttackOption.getSelectedItem());
        options.setSkillLevel((SkillLevel) this.cbSkillLevel.getSelectedItem());
        options.setRange((RangeOfInfluence) this.cbRange.getSelectedItem());
        options.setWinsNeeded((Integer) this.spnNumWins.getValue());
        options.setRollbackTurnsAllowed(chkRollbackTurnsAllowed.isSelected());
        options.setSpectatorsAllowed(chkSpectatorsAllowed.isSelected());
        options.setPlaneChase(chkPlaneChase.isSelected());
        options.setRated(chkRated.isSelected());
        options.setFreeMulligans((Integer) this.spnFreeMulligans.getValue());
        options.setPassword(this.txtPassword.getText());
        options.setQuitRatio((Integer) this.spnQuitRatio.getValue());
        options.setEdhPowerLevel((Integer) this.spnEdhPowerLevel.getValue());
        String serverAddress = SessionHandler.getSession().getServerHostname().orElseGet(() -> "");
        options.setBannedUsers(IgnoreList.ignoreList(serverAddress));
        if (!checkMatchOptions(options)) {
            return;
        }
        saveGameSettingsToPrefs(options, this.player1Panel.getDeckFile());

        table = SessionHandler.createTable(roomId, options);
        if (table == null) {
            JOptionPane.showMessageDialog(MageFrame.getDesktop(), "Error creating table.", "Error", JOptionPane.ERROR_MESSAGE);
            return;
        }
        try {
            if (SessionHandler.joinTable(
                    roomId,
                    table.getTableId(),
                    this.player1Panel.getPlayerName(),
                    PlayerType.HUMAN, 1,
                    DeckImporterUtil.importDeck(this.player1Panel.getDeckFile()),
                    this.txtPassword.getText())) {
                for (TablePlayerPanel player : players) {
                    if (player.getPlayerType() != PlayerType.HUMAN) {
                        if (!player.joinTable(roomId, table.getTableId())) {
                            // error message must be send by the server
                            SessionHandler.removeTable(roomId, table.getTableId());
                            table = null;
                            return;
                        }
                    }
                }
                this.hideDialog();
                return;
            }
        } catch (FileNotFoundException ex) {
            handleError(ex);
        } catch (IOException ex) {
            handleError(ex);
        } catch (ClassNotFoundException ex) {
            handleError(ex);
        }
        // JOptionPane.showMessageDialog(MageFrame.getDesktop(), "Error joining table.", "Error", JOptionPane.ERROR_MESSAGE);
        SessionHandler.removeTable(roomId, table.getTableId());
        table = null;
    }//GEN-LAST:event_btnOKActionPerformed

    private void cbGameTypeActionPerformed(java.awt.event.ActionEvent evt) {//GEN-FIRST:event_cbGameTypeActionPerformed
        setGameOptions();
    }//GEN-LAST:event_cbGameTypeActionPerformed

    private void numPlayersChanged(javax.swing.event.ChangeEvent evt) {//GEN-FIRST:event_numPlayersChanged
        int numPlayers = (Integer) this.spnNumPlayers.getValue() - 1;
        createPlayers(numPlayers);
    }//GEN-LAST:event_numPlayersChanged

    private void spnNumWinsnumPlayersChanged(javax.swing.event.ChangeEvent evt) {//GEN-FIRST:event_spnNumWinsnumPlayersChanged
        // TODO add your handling code here:
    }//GEN-LAST:event_spnNumWinsnumPlayersChanged

    /**
     * Checks about not valid game option combinations and shows an error
     * message
     *
     * @return
     */
    private boolean checkMatchOptions(MatchOptions options) {
        switch (options.getDeckType()) {
            case "Variant Magic - Commander":
            case "Variant Magic - Duel Commander":
            case "Variant Magic - MTGO 1v1 Commander":
                if (!options.getGameType().startsWith("Commander")) {
                    JOptionPane.showMessageDialog(MageFrame.getDesktop(), "Deck type Commander needs also a Commander game type", "Error", JOptionPane.ERROR_MESSAGE);
                    return false;
                }
                break;
            case "Variant Magic - Brawl":
            case "Variant Magic - Duel Brawl":
                if (!options.getGameType().startsWith("Brawl")) {
                    JOptionPane.showMessageDialog(MageFrame.getDesktop(), "Deck type Brawl needs also a Brawl game type", "Error", JOptionPane.ERROR_MESSAGE);
                    return false;
                }
                break;
            case "Variant Magic - Tiny Leaders":
                if (!options.getGameType().startsWith("Tiny Leaders")) {
                    JOptionPane.showMessageDialog(MageFrame.getDesktop(), "Deck type Tiny Leaders needs also a Tiny Leaders game type", "Error", JOptionPane.ERROR_MESSAGE);
                    return false;
                }
                break;
            case "Variant Magic - Momir Basic":
                if (!options.getGameType().startsWith("Momir Basic")) {
                    JOptionPane.showMessageDialog(MageFrame.getDesktop(), "Deck type Momir Basic needs also a Momir Basic game type", "Error", JOptionPane.ERROR_MESSAGE);
                    return false;
                }
                break;
        }
        switch (options.getGameType()) {
            case "Commander Two Player Duel":
            case "Commander Free For All":
                if (!options.getDeckType().equals("Variant Magic - Commander")
                        && !options.getDeckType().equals("Variant Magic - Duel Commander")
                        && !options.getDeckType().equals("Variant Magic - MTGO 1v1 Commander")) {
                    JOptionPane.showMessageDialog(MageFrame.getDesktop(), "Deck type Commander needs also a Commander game type", "Error", JOptionPane.ERROR_MESSAGE);
                    return false;
                }
                break;
            case "Brawl Two Player Duel":
            case "Brawl Free For All":
                if (!options.getDeckType().equals("Variant Magic - Brawl")
                        && !options.getDeckType().equals("Variant Magic - Duel Brawl")) {
                    JOptionPane.showMessageDialog(MageFrame.getDesktop(), "Deck type Brawl needs also a Brawl game type", "Error", JOptionPane.ERROR_MESSAGE);
                    return false;
                }
                break;
            case "Tiny Leaders Two Player Duel":
                if (!options.getDeckType().equals("Variant Magic - Tiny Leaders")) {
                    JOptionPane.showMessageDialog(MageFrame.getDesktop(), "Deck type Tiny Leaders needs also a Tiny Leaders game type", "Error", JOptionPane.ERROR_MESSAGE);
                    return false;
                }
                break;
        }
        return true;
    }

    private void setGameOptions() {
        GameTypeView gameType = (GameTypeView) cbGameType.getSelectedItem();
        int oldValue = (Integer) this.spnNumPlayers.getValue();
        this.spnNumPlayers.setModel(new SpinnerNumberModel(gameType.getMinPlayers(), gameType.getMinPlayers(), gameType.getMaxPlayers(), 1));
        this.spnNumPlayers.setEnabled(gameType.getMinPlayers() != gameType.getMaxPlayers());
        if (oldValue >= gameType.getMinPlayers() && oldValue <= gameType.getMaxPlayers()) {
            this.spnNumPlayers.setValue(oldValue);
        }
        this.cbAttackOption.setEnabled(gameType.isUseAttackOption());
        this.cbRange.setEnabled(gameType.isUseRange());
        createPlayers((Integer) spnNumPlayers.getValue() - 1);
    }

    private void createPlayers(int numPlayers) {
        // add missing player panels
        if (numPlayers > players.size()) {
            while (players.size() != numPlayers) {
                TablePlayerPanel playerPanel = new TablePlayerPanel();
                PlayerType playerType = PlayerType.HUMAN;
                if (prefPlayerTypes.size() >= players.size() && !players.isEmpty()) {
                    playerType = prefPlayerTypes.get(players.size() - 1);
                }
                playerPanel.init(players.size() + 2, playerType);
                players.add(playerPanel);
                playerPanel.addPlayerTypeEventListener(
                        (Listener<Event>) event -> drawPlayers()
                );
            }

        } // remove player panels no longer needed
        else if (numPlayers < players.size()) {
            while (players.size() != numPlayers) {
                players.remove(players.size() - 1);
            }
        }
        drawPlayers();
    }

    private void drawPlayers() {
        this.pnlOtherPlayers.removeAll();
        for (TablePlayerPanel panel : players) {
            this.pnlOtherPlayers.add(panel);
        }
        this.pack();
        this.revalidate();
        this.repaint();
    }

    private void handleError(Exception ex) {
        logger.fatal("Error loading deck", ex);
        //JOptionPane.showMessageDialog(MageFrame.getDesktop(), "Error loading deck.", "Error", JOptionPane.ERROR_MESSAGE);
        MageFrame.getInstance().showErrorDialog("Error loading deck.", ex.getMessage());
    }

    public void showDialog(UUID roomId) {
        this.roomId = roomId;
        if (!lastSessionId.equals(SessionHandler.getSessionId())) {
            lastSessionId = SessionHandler.getSessionId();
            this.player1Panel.setPlayerName(SessionHandler.getUserName());
            cbGameType.setModel(new DefaultComboBoxModel(SessionHandler.getGameTypes().toArray()));
            cbDeckType.setModel(new DefaultComboBoxModel(SessionHandler.getDeckTypes()));
            selectLimitedByDefault();
            cbTimeLimit.setModel(new DefaultComboBoxModel(MatchTimeLimit.values()));
            cbRange.setModel(new DefaultComboBoxModel(RangeOfInfluence.values()));
            cbAttackOption.setModel(new DefaultComboBoxModel(MultiplayerAttackOption.values()));
            cbSkillLevel.setModel(new DefaultComboBoxModel(SkillLevel.values()));
            // Update the existing player panels (neccessary if server was changes = new session)
            int i = 2;
            for (TablePlayerPanel tablePlayerPanel : players) {
                tablePlayerPanel.init(i++, tablePlayerPanel.getPlayerType());
            }
            this.setModal(true);
            setGameOptions();
            this.setLocation(150, 100);
        }
        currentSettingVersion = 0;
        setGameSettingsFromPrefs(currentSettingVersion);
        this.setVisible(true);
    }

    public TableView getTable() {
        return table;
    }

    public UUID getPlayerId() {
        return playerId;
    }

    private void selectLimitedByDefault() {
        for (int i = 0; i < cbDeckType.getItemCount(); i++) {
            String name = (String) cbDeckType.getItemAt(i);
            if (name.equals(LIMITED)) {
                cbDeckType.setSelectedIndex(i);
                break;
            }
        }
    }

    /**
     * set the table settings from java prefs
     */
    int currentSettingVersion = 0;

    private void setGameSettingsFromPrefs(int version) {
        currentSettingVersion = version;
        String versionStr = "";
        switch (currentSettingVersion) {
            case 1:
                versionStr = "1";
                btnPreviousConfiguration1.requestFocus();
                break;
            case 2:
                versionStr = "2";
                btnPreviousConfiguration2.requestFocus();
                break;
            default:
                btnPreviousConfiguration2.getParent().requestFocus();
                break;
        }
        txtName.setText(PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_NAME + versionStr, "Game"));
        txtPassword.setText(PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_PASSWORD + versionStr, ""));

        String playerTypes = PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_PLAYER_TYPES + versionStr, "Human");
        prefPlayerTypes.clear();
        for (String pType : playerTypes.split(",")) {
            prefPlayerTypes.add(PlayerType.getByDescription(pType));
        }
        this.spnNumPlayers.setValue(Integer.parseInt(PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_NUMBER_PLAYERS + versionStr, "2")));

        String gameTypeName = PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_GAME_TYPE + versionStr, "Two Player Duel");
        for (GameTypeView gtv : SessionHandler.getGameTypes()) {
            if (gtv.getName().equals(gameTypeName)) {
                cbGameType.setSelectedItem(gtv);
                break;
            }
        }
        int timeLimit = Integer.parseInt(PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_TIME_LIMIT + versionStr, "1500"));
        for (MatchTimeLimit mtl : MatchTimeLimit.values()) {
            if (mtl.getTimeLimit() == timeLimit) {
                this.cbTimeLimit.setSelectedItem(mtl);
                break;
            }
        }
        cbDeckType.setSelectedItem(PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_DECK_TYPE + versionStr, "Limited"));
        String deckFile = PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_DECK_FILE + versionStr, null);
        if (deckFile != null) {
            this.player1Panel.setDeckFile(deckFile);
        }
        this.spnNumWins.setValue(Integer.parseInt(PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_NUMBER_OF_WINS + versionStr, "2")));
        this.chkRollbackTurnsAllowed.setSelected(PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_ROLLBACK_TURNS_ALLOWED + versionStr, "Yes").equals("Yes"));
        this.chkSpectatorsAllowed.setSelected(PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_SPECTATORS_ALLOWED + versionStr, "Yes").equals("Yes"));
        this.chkPlaneChase.setSelected(PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_PLANECHASE + versionStr, "No").equals("Yes"));
        this.chkRated.setSelected(PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_RATED + versionStr, "No").equals("Yes"));
        this.spnFreeMulligans.setValue(Integer.parseInt(PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_NUMBER_OF_FREE_MULLIGANS + versionStr, "0")));

        int range = Integer.parseInt(PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_RANGE + versionStr, "1"));
        for (RangeOfInfluence roi : RangeOfInfluence.values()) {
            if (roi.getRange() == range) {
                this.cbRange.setSelectedItem(roi);
                break;
            }
        }
        String attackOption = PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_ATTACK_OPTION + versionStr, "Attack Multiple Players");
        for (MultiplayerAttackOption mao : MultiplayerAttackOption.values()) {
            if (mao.toString().equals(attackOption)) {
                this.cbAttackOption.setSelectedItem(mao);
                break;
            }
        }
        String skillLevelDefault = PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_SKILL_LEVEL, "Casual");
        for (SkillLevel skillLevel : SkillLevel.values()) {
            if (skillLevel.toString().equals(skillLevelDefault)) {
                this.cbSkillLevel.setSelectedItem(skillLevel);
                break;
            }
        }

        this.spnQuitRatio.setValue(Integer.parseInt(PreferencesDialog.getCachedValue(PreferencesDialog.KEY_NEW_TABLE_QUIT_RATIO, "100")));
        this.spnEdhPowerLevel.setValue(0);
    }

    /**
     * Save the settings to java prefs to reload it next time the dialog will be
     * created
     *
     * @param options
     * @param deckFile
     */
    private void saveGameSettingsToPrefs(MatchOptions options, String deckFile) {
        String versionStr = "";
        if (currentSettingVersion == 1) {
            versionStr = "1";
        } else if (currentSettingVersion == 2) {
            versionStr = "2";
        }
        PreferencesDialog.saveValue(PreferencesDialog.KEY_NEW_TABLE_NAME + versionStr, options.getName());
        PreferencesDialog.saveValue(PreferencesDialog.KEY_NEW_TABLE_PASSWORD + versionStr, options.getPassword());
        PreferencesDialog.saveValue(PreferencesDialog.KEY_NEW_TABLE_DECK_TYPE + versionStr, options.getDeckType());
        PreferencesDialog.saveValue(PreferencesDialog.KEY_NEW_TABLE_TIME_LIMIT + versionStr, Integer.toString(options.getPriorityTime()));
        PreferencesDialog.saveValue(PreferencesDialog.KEY_NEW_TABLE_GAME_TYPE + versionStr, options.getGameType());
        PreferencesDialog.saveValue(PreferencesDialog.KEY_NEW_TABLE_NUMBER_OF_WINS + versionStr, Integer.toString(options.getWinsNeeded()));
        PreferencesDialog.saveValue(PreferencesDialog.KEY_NEW_TABLE_ROLLBACK_TURNS_ALLOWED + versionStr, options.isRollbackTurnsAllowed() ? "Yes" : "No");
        PreferencesDialog.saveValue(PreferencesDialog.KEY_NEW_TABLE_RATED + versionStr, options.isRated() ? "Yes" : "No");
        PreferencesDialog.saveValue(PreferencesDialog.KEY_NEW_TABLE_NUMBER_OF_FREE_MULLIGANS + versionStr, Integer.toString(options.getFreeMulligans()));
        PreferencesDialog.saveValue(PreferencesDialog.KEY_NEW_TABLE_DECK_FILE + versionStr, deckFile);
        PreferencesDialog.saveValue(PreferencesDialog.KEY_NEW_TABLE_NUMBER_PLAYERS + versionStr, spnNumPlayers.getValue().toString());
        PreferencesDialog.saveValue(PreferencesDialog.KEY_NEW_TABLE_RANGE + versionStr, Integer.toString(options.getRange().getRange()));
        PreferencesDialog.saveValue(PreferencesDialog.KEY_NEW_TABLE_ATTACK_OPTION + versionStr, options.getAttackOption().toString());
        PreferencesDialog.saveValue(PreferencesDialog.KEY_NEW_TABLE_SKILL_LEVEL + versionStr, options.getSkillLevel().toString());
        PreferencesDialog.saveValue(PreferencesDialog.KEY_NEW_TABLE_SPECTATORS_ALLOWED + versionStr, options.isSpectatorsAllowed() ? "Yes" : "No");
        PreferencesDialog.saveValue(PreferencesDialog.KEY_NEW_TABLE_PLANECHASE + versionStr, options.isPlaneChase() ? "Yes" : "No");
        PreferencesDialog.saveValue(PreferencesDialog.KEY_NEW_TABLE_QUIT_RATIO + versionStr, Integer.toString(options.getQuitRatio()));
        StringBuilder playerTypesString = new StringBuilder();
        for (Object player : players) {
            if (playerTypesString.length() > 0) {
                playerTypesString.append(',');
            }
            TablePlayerPanel tpp = (TablePlayerPanel) player;
            playerTypesString.append(tpp.getPlayerType());
        }
        PreferencesDialog.saveValue(PreferencesDialog.KEY_NEW_TABLE_PLAYER_TYPES + versionStr, playerTypesString.toString());
    }

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JButton btnCancel;
    private javax.swing.JButton btnOK;
    private javax.swing.JButton btnPreviousConfiguration1;
    private javax.swing.JButton btnPreviousConfiguration2;
    private javax.swing.JComboBox cbAttackOption;
    private javax.swing.JComboBox cbDeckType;
    private javax.swing.JComboBox cbGameType;
    private javax.swing.JComboBox cbRange;
    private javax.swing.JComboBox cbSkillLevel;
    private javax.swing.JComboBox cbTimeLimit;
    private javax.swing.JCheckBox chkRollbackTurnsAllowed;
    private javax.swing.JCheckBox chkSpectatorsAllowed;
    private javax.swing.JCheckBox chkPlaneChase;
    private javax.swing.JCheckBox chkRated;
    private javax.swing.JLabel jLabel1;
    private javax.swing.JLabel jLabel2;
    private javax.swing.JSeparator jSeparator1;
    private javax.swing.JSeparator jSeparator2;
    private javax.swing.JSeparator jSeparator3;
    private javax.swing.JLabel lbDeckType;
    private javax.swing.JLabel lbTimeLimit;
    private javax.swing.JLabel lblAttack;
    private javax.swing.JLabel lblFreeMulligans;
    private javax.swing.JLabel lblGameType;
    private javax.swing.JLabel lblName;
    private javax.swing.JLabel lblNumPlayers;
    private javax.swing.JLabel lblNumWins;
    private javax.swing.JLabel lblPassword;
    private javax.swing.JLabel lblQuitRatio;
    private javax.swing.JLabel lblEdhPowerLevel;
    private javax.swing.JLabel lblRange;
    private javax.swing.JLabel lblSkillLevel;
    private mage.client.table.NewPlayerPanel player1Panel;
    private javax.swing.JPanel pnlOtherPlayers;
    private javax.swing.JSpinner spnFreeMulligans;
    private javax.swing.JSpinner spnNumPlayers;
    private javax.swing.JSpinner spnNumWins;
    private javax.swing.JSpinner spnQuitRatio;
    private javax.swing.JSpinner spnEdhPowerLevel;
    private javax.swing.JTextField txtName;
    private javax.swing.JTextField txtPassword;
    // End of variables declaration//GEN-END:variables
}
