/*
 *  Copyright 2010 BetaSteward_at_googlemail.com. All rights reserved.
 *
 *  Redistribution and use in source and binary forms, with or without modification, are
 *  permitted provided that the following conditions are met:
 *
 *     1. Redistributions of source code must retain the above copyright notice, this list of
 *        conditions and the following disclaimer.
 *
 *     2. Redistributions in binary form must reproduce the above copyright notice, this list
 *        of conditions and the following disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 *  THIS SOFTWARE IS PROVIDED BY BetaSteward_at_googlemail.com ``AS IS'' AND ANY EXPRESS OR IMPLIED
 *  WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 *  FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL BetaSteward_at_googlemail.com OR
 *  CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 *  CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 *  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 *  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 *  ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  The views and conclusions contained in the software and documentation are those of the
 *  authors and should not be interpreted as representing official policies, either expressed
 *  or implied, of BetaSteward_at_googlemail.com.
 */
package mage.cards.c;

import java.util.UUID;
import mage.abilities.Ability;
import mage.abilities.effects.OneShotEffect;
import mage.cards.CardImpl;
import mage.cards.CardSetInfo;
import mage.cards.CardsImpl;
import mage.constants.CardType;
import mage.constants.Outcome;
import mage.constants.Zone;
import mage.game.Game;
import mage.players.Player;
import mage.target.common.TargetOpponent;

/**
 *
 * @author North
 */
public final class CovenantOfMinds extends CardImpl {

    public CovenantOfMinds(UUID ownerId, CardSetInfo setInfo) {
        super(ownerId, setInfo, new CardType[]{CardType.SORCERY}, "{4}{U}");

        // Reveal the top three cards of your library. Target opponent may choose to put those cards into your hand.
        // If he or she doesn't, put those cards into your graveyard and draw five cards.
        this.getSpellAbility().addEffect(new CovenantOfMindsEffect());
        this.getSpellAbility().addTarget(new TargetOpponent());
    }

    public CovenantOfMinds(final CovenantOfMinds card) {
        super(card);
    }

    @Override
    public CovenantOfMinds copy() {
        return new CovenantOfMinds(this);
    }
}

class CovenantOfMindsEffect extends OneShotEffect {

    public CovenantOfMindsEffect() {
        super(Outcome.DrawCard);
        this.staticText = "Reveal the top three cards of your library. Target opponent may choose to put those cards into your hand. If he or she doesn't, put those cards into your graveyard and draw five cards";
    }

    public CovenantOfMindsEffect(final CovenantOfMindsEffect effect) {
        super(effect);
    }

    @Override
    public CovenantOfMindsEffect copy() {
        return new CovenantOfMindsEffect(this);
    }

    @Override
    public boolean apply(Game game, Ability source) {
        Player player = game.getPlayer(source.getControllerId());
        Player opponent = game.getPlayer(source.getFirstTarget());
        if (player == null || opponent == null) {
            return false;
        }

        CardsImpl cards = new CardsImpl(player.getLibrary().getTopCards(game, 3));
        if (!cards.isEmpty()) {
            player.revealCards(source, cards, game);
            StringBuilder sb = new StringBuilder();
            sb.append("Put the revealed cards into ").append(player.getLogName()).append("'s hand?");
            sb.append(" If you don't, those cards are put into his graveyard and he will draw five cards.");

            if (opponent.chooseUse(Outcome.Neutral, sb.toString(), source, game)) {
                player.moveCards(cards, Zone.HAND, source, game);
            } else {
                player.moveCards(cards, Zone.GRAVEYARD, source, game);
                player.drawCards(5, game);
            }

        } else {
            if (!opponent.chooseUse(Outcome.Benefit, player.getLogName() + "'s library is empty? Do you want him to draw five cards?", source, game)) {
                player.drawCards(5, game);
            }
        }

        return true;
    }
}
